﻿using System;
using System.Collections.Specialized;

namespace Nintendo.InGameEditing.UI
{
    /// <summary>
    /// C++ ランタイムサーバーに接続し、ターゲット上の値を編集可能なユーザーインターフェースを作成する機能を提供します。
    /// C++ ランタイム上では &lt;nn/ige/ige_UiService.h&gt; を使用してください。
    /// </summary>
    public class UiService : IDisposable
    {
        private readonly EditService service;

        /// <summary>
        /// ポート名を指定してインスタンスを初期化します。
        /// </summary>
        /// <param name="portName">ポート名</param>
        public UiService(string portName) : this(portName, null)
        {
        }

        /// <summary>
        /// ポート名とピアタイプを指定してインスタンスを初期化します。
        /// </summary>
        /// <param name="portName">ポート名</param>
        /// <param name="peerType">ピアタイプ</param>
        public UiService(string portName, string peerType)
        {
            service = new EditService(portName, peerType);

            service.ConnectionChanged += (s, e) => { if (e.IsConnected) ControlModels.Refresh(); };
            service.Children.CollectionChanged += OnNodeListChanged;
        }

        private void OnNodeListChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            if (e.NewItems?.Count > 0)
            {
                foreach (Node node in e.NewItems)
                {
                    var control = node.ToControlModel(true);
                    if (control == null)
                    {
                        Nodes.Add(node);
                    }
                    else
                    {
                        ControlModels.Add(control);
                    }
                }
            }
        }

        /// <summary>
        /// 接続状態が変化したときに発行されるイベントです。
        /// </summary>
        public event EventHandler<ConnectionChangedEventArgs> ConnectionChanged
        {
            add { service.ConnectionChanged += value; }
            remove { service.ConnectionChanged -= value; }
        }

        /// <summary>
        /// ポート名を取得します。
        /// </summary>
        public string PortName => service.PortName;

        /// <summary>
        /// ピアタイプを取得します。
        /// </summary>
        public string PeerType => service.PeerType;

        /// <summary>
        /// サーバーと接続済みかどうかを取得します。
        /// </summary>
        public bool IsConnected => service.IsConnected;

        /// <summary>
        /// サーバーに接続可能になったタイミングで自動的に接続を行うかどうかを取得します。
        /// </summary>
        public bool IsAutoConnectionEnabled
        {
            get { return service.IsAutoConnectionEnabled; }
            set { service.IsAutoConnectionEnabled = value; }
        }

        /// <summary>
        /// サーバーに接続可能な状態かどうかを取得します。
        /// </summary>
        public bool Connectable => service.Connectable;

        /// <summary>
        /// コントロールのリストを取得します。
        /// </summary>
        public ControlModelList ControlModels { get; } = new ControlModelList();

        /// <summary>
        /// コントロール化されていない子要素のリストを取得します。
        /// </summary>
        public NodeList Nodes { get; } = new NodeList();

        /// <summary>
        /// サーバーに接続します。
        /// </summary>
        public void Connect() => service.Connect();

        /// <summary>
        /// サーバーから切断します。
        /// </summary>
        public void Disconnect() => service.Disconnect();

        /// <summary>
        /// 接続を停止し、サービスを破棄します。
        /// </summary>
        public void Dispose() => service.Dispose();
    }
}
