﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.IO;
using Nintendo.InGameEditing.Utilities;

namespace Nintendo.InGameEditing.UI
{
    /// <summary>
    /// int 型を扱うスライダーコントロールです。
    /// </summary>
    public class SliderModelInt : SliderModel<int>
    {
        internal static readonly string UiType = "Slider<int>";

        internal SliderModelInt(ValueNode<int> node, Dictionary<string, string> attrs)
            : base(node, attrs, int.Parse)
        {
        }
    }

    /// <summary>
    /// float 型を扱うスライダーコントロールです。
    /// </summary>
    public class SliderModelFloat : SliderModel<float>
    {
        internal static readonly string UiType = "Slider<float>";

        internal SliderModelFloat(ValueNode<float> node, Dictionary<string, string> attrs)
            : base(node, attrs, float.Parse)
        {
        }
    }

    /// <summary>
    /// スライダーコントロールです。
    /// </summary>
    /// <typeparam name="T">スライダーコントロールが扱う要素の型</typeparam>
    public abstract class SliderModel<T> : ControlModel where T : struct, IComparable, IFormattable, IConvertible, IComparable<T>, IEquatable<T>
    {
        private static readonly PropertyChangedEventArgs ValueEventArgs = new PropertyChangedEventArgs(nameof(Value));
        private readonly ValueNode<T> node;
        private T currentValue;

        protected SliderModel(ValueNode<T> node, Dictionary<string, string> attrs, Func<string, T> parse) : base(node, attrs)
        {
            if (parse == null) { throw new ArgumentNullException(nameof(parse)); }
            this.node = node;

            var min = default(string);
            if (attrs.TryGetValue(nameof(min), out min))
            {
                try
                {
                    Min = parse(min);
                }
                catch { /* ignored */ }
            }

            var max = default(string);
            if (attrs.TryGetValue(nameof(max), out max))
            {
                try
                {
                    Max = parse(max);
                }
                catch { /* ignored */ }
            }

            node.ValueReceived += (s, e) => SetPropertyValue(ref currentValue, e.Value, ValueEventArgs);
        }

        /// <summary>
        /// 値の最小値を取得します。
        /// </summary>
        public T Min { get; }

        /// <summary>
        /// 値の最大値を取得します。
        /// </summary>
        public T Max { get; }

        /// <summary>
        /// 値を取得設定します。
        /// </summary>
        public T Value
        {
            get { return currentValue; }
            set
            {
                if (SetPropertyValue(ref currentValue, value, ValueEventArgs))
                {
                    node.PushValue(value);
                }
            }
        }
    }
}
