﻿using System;
using System.Collections.Generic;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Reflection;
using Nintendo.InGameEditing.Utilities;

namespace Nintendo.InGameEditing.UI
{
    /// <summary>
    /// UI の型名からモデルの型を取得します。
    /// </summary>
    internal static class UiTypeResolver
    {
        private static readonly Dictionary<string, Type> dictionary;
        private const string UiTypeFieldName = "UiType";

        static UiTypeResolver()
        {
            var flags = BindingFlags.Static | BindingFlags.Public | BindingFlags.NonPublic;

            dictionary = Assembly.GetExecutingAssembly().GetTypes()
                .Where(t => typeof(ControlModel).IsAssignableFrom(t))
                .Where(t => !t.IsGenericTypeDefinition) // 具象型のみ
                .Select(t => new { UiType = t.GetField(UiTypeFieldName, flags), Type = t })
                .Where(t => t.UiType != null)
                .ToDictionary(pair => (string)pair.UiType.GetValue(null), pair => pair.Type);
        }

        internal static bool TryResolveUiTypeName(string uiTypeName, out Type type) => dictionary.TryGetValue(uiTypeName, out type);
    }

    internal static class ControlModelFactory
    {
        public static ControlModel ToControlModel(this Node node, bool noUiAllowed = false)
        {
            var attrs = ControlModel.ParseAttributes(node.Metadata);

            string uiType;
            if (!attrs.TryGetValue("type", out uiType))
            {
                if (noUiAllowed) return null;
                throw new NotSupportedException($"メタデータにコントロール名が含まれていません。ランタイムのメタデータを確認してください。");
            }

            Type type;
            if (!UiTypeResolver.TryResolveUiTypeName(uiType, out type))
            {
                if (noUiAllowed) return null;
                throw new NotSupportedException($"サポートされていない型名 {uiType} が指定されました。");
            }

            try
            {
                return (ControlModel)Activator.CreateInstance(type,
                    BindingFlags.Instance | BindingFlags.NonPublic | BindingFlags.Public,
                    null,
                    new object[] { node, attrs },
                    CultureInfo.CurrentCulture);
            }
            catch
            {
                if (noUiAllowed) return null;
                throw;
            }
        }
    }
}

