﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "SpyToolProfiler.h"

#if defined( ATKPLAYER_BUILD_CONFIG_ENABLE_PROFILE_WITH_SPY )

namespace{
    //  spy の通信用のバッファサイズ
    const size_t SpyControllerDataBufferLength = 1024 * 1024;
    //  spy のデータ通信スレッドのプライオリティ
    const int SpyThreadDataPriority = 17;
    //  spy の sync 通信スレッドのプライオリティ
    const int SpyThreadSyncPriority = 17;
}
namespace
{
    //  spy の初期化に必要な引数を設定します
    void SetSpyControllerInitializeArg(nn::spy::SpyController::InitializeArg& args) NN_NOEXCEPT
    {
        args.dataBufferLength = SpyControllerDataBufferLength;
    }
    //  spyAtk の初期化に必要な引数を設定します
    void SetAtkSpyModuleInitializeArg(nn::spy::atk::AtkSpyModule::InitializeArg& args) NN_NOEXCEPT
    {
        args.isPerformanceMetricsEnabled = true;
        args.isAtkProfilesEnabled = true;
        args.isWaveformEnabled = true;
    }
    //  波形を録音するバッファのサイズを取得します
    size_t GetRecordingWaveBufferSize() NN_NOEXCEPT
    {
        const auto sampleCount = nn::atk::SoundSystem::GetRendererSampleCount();
        const auto sampleSize = sizeof(int16_t);
        const auto channelCount = nn::atk::SoundSystem::GetRendererChannelCountMax();
        const auto audioFrameCount = nn::atk::SoundSystem::SoundSystemParam::DefaultRecordingAudioFrameCount;

        return sampleCount * sampleSize * channelCount * audioFrameCount;
    }
}

//  必要なバッファサイズを計算します
size_t SpyToolProfiler::GetRequiredMemorySize() NN_NOEXCEPT
{
    size_t size = 0;

    {
        nn::spy::SpyController::InitializeArg args;
        SetSpyControllerInitializeArg( args );
        size += nn::spy::SpyController::GetRequiredMemorySize( args );
    }

    size += GetRecordingWaveBufferSize();

    {
        nn::spy::atk::AtkSpyModule::InitializeArg args;
        SetAtkSpyModuleInitializeArg( args );
        size += nn::DefaultAlignment;
        size += nn::spy::atk::AtkSpyModule::GetRequiredMemorySize( args );
    }

    return size;
}
//  初期化します
void SpyToolProfiler::Initialize(void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL( buffer );
    NN_ABORT_UNLESS_GREATER_EQUAL( GetRequiredMemorySize(), bufferSize );
    nn::util::BytePtr ptr( buffer );

    {
        //  SpyController の初期化
        nn::spy::SpyController::InitializeArg args;
        SetSpyControllerInitializeArg( args );

        const size_t memSize = nn::spy::SpyController::GetRequiredMemorySize( args );
        m_SpyController.Initialize( args, ptr.Get(), memSize );
        ptr.Advance( memSize );
    }

    {
        //  録音用バッファ
        m_WaveBuffer = ptr.Get();
        m_WaveBufferSize = GetRecordingWaveBufferSize();
        ptr.Advance( m_WaveBufferSize );
    }

    {
        //  AtkSpyModule を初期化し、登録
        nn::spy::atk::AtkSpyModule::InitializeArg args;
        SetAtkSpyModuleInitializeArg( args );

        const size_t memSize = nn::spy::atk::AtkSpyModule::GetRequiredMemorySize( args );
        ptr.AlignUp( nn::DefaultAlignment );
        m_AtkSpyModule.Initialize( args, ptr.Get(), memSize );
        ptr.Advance( memSize );

        m_SpyController.RegisterModule( m_AtkSpyModule );
    }

    {
        //  Spy ツールとの通信を開始
        nn::spy::SpyController::OpenArg args;
        args.syncThreadPriority = SpyThreadSyncPriority;
        args.dataThreadPriority = SpyThreadDataPriority;

        const bool isSuccess = m_SpyController.Open( args );
        NN_ABORT_UNLESS( isSuccess );
    }
}
//  終了処理をします
void SpyToolProfiler::Finalize() NN_NOEXCEPT
{
    m_SpyController.Close();
    m_SpyController.UnregisterModule( m_AtkSpyModule );
    m_AtkSpyModule.Finalize();
    m_SpyController.Finalize();
}

//  更新処理をします
void SpyToolProfiler::Update(nn::atk::SoundArchivePlayer& soundArchivePlayer) NN_NOEXCEPT
{
    m_SpyController.SetCurrentApplicationFrame( m_SpyController.GetCurrentApplicationFrame() + 1 );

    m_AtkSpyModule.PushSoundState( soundArchivePlayer );
    m_AtkSpyModule.PushPerformanceMetrics();
    m_AtkSpyModule.PushAtkProfiles();

    const auto tick = nn::os::GetSystemTick();
    const size_t size = nn::atk::SoundSystem::ReadCircularBufferSink( m_WaveBuffer, m_WaveBufferSize );
    if( size > 0 )
    {
        m_AtkSpyModule.PushWaveform( m_WaveBuffer, size, tick );
    }
}

#else

//  必要なバッファサイズを計算します
size_t SpyToolProfiler::GetRequiredMemorySize() NN_NOEXCEPT
{
    //  0 を返り値にすると、メモリの確保に失敗してしまうため
    return sizeof(int);
}
//  初期化します
void SpyToolProfiler::Initialize(void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_UNUSED( buffer );
    NN_UNUSED( bufferSize );
}
//  終了処理をします
void SpyToolProfiler::Finalize() NN_NOEXCEPT
{
}

//  更新処理をします
void SpyToolProfiler::Update(nn::atk::SoundArchivePlayer& soundArchivePlayer) NN_NOEXCEPT
{
    NN_UNUSED( soundArchivePlayer );
}

#endif
