﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/atk.h>
#include "gfxutil/GfxContext.h"
#include "SoundSystemPanel.h"
#include "FlightRecorder.h"

namespace
{
    //  デフォルトの OutputMode
    const nn::atk::OutputMode DefaultOutputMode = nn::atk::OutputMode::OutputMode_Surround;
    //  使用できる OutputMode のしきい値
    const nn::atk::OutputMode UsableOutputModeThreshold = nn::atk::OutputMode::OutputMode_Surround;

    //  パネルのタイトル
    const char* PanelTitle = "SoundSystem";
}
namespace
{
    void SetOutputMode(int mode, gfxutil::Label& label)
    {
        NN_ABORT_UNLESS_MINMAX( mode, 0, UsableOutputModeThreshold );

        switch( mode )
        {
        case nn::atk::OutputMode_Monaural:
            FlightRecorder::GetInstance().WriteLog( "[SndSys] OutputMode : MONO" );
            nn::atk::SoundSystem::SetOutputMode( nn::atk::OutputMode_Monaural, nn::atk::OutputDevice_Main );
            label.SetText( "MONO" );
            break;
        case nn::atk::OutputMode_Stereo:
            FlightRecorder::GetInstance().WriteLog( "[SndSys] OutputMode : STEREO" );
            nn::atk::SoundSystem::SetOutputMode( nn::atk::OutputMode_Stereo, nn::atk::OutputDevice_Main );
            label.SetText( "STEREO" );
            break;
        case nn::atk::OutputMode_Surround:
            FlightRecorder::GetInstance().WriteLog( "[SndSys] OutputMode : SURROUND" );
            nn::atk::SoundSystem::SetOutputMode( nn::atk::OutputMode_Surround, nn::atk::OutputDevice_Main );
            label.SetText( "SURROUND" );
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
    }
}

//  初期化します
void SoundSystemPanel::Initialize(float positionX, float positionY, float sizeX, float sizeY) NN_NOEXCEPT
{
    m_Panel.SetTitle( PanelTitle );
    m_Panel.SetPosition( positionX, positionY );
    m_Panel.SetSize( sizeX, sizeY );

    const nn::util::Float2 panelPos = m_Panel.GetClientPositionLeftTop();
    const float itemSizeX = m_Panel.GetClientPositionRightBottom().x - panelPos.x;
    m_ItemList.Initialize( itemSizeX );
    m_ItemList.SetSelectedItemIndex( ItemType_OutputMode );
    m_ItemList.GetNameLabel( ItemType_OutputMode ).SetText( "OutputMode(Main):" );

    m_OutputMode = DefaultOutputMode;
    SetOutputMode( m_OutputMode, m_ItemList.GetValueLabel( ItemType_OutputMode ) );
}
//! @brief  入力による更新を行います
void SoundSystemPanel::UpdateByHid(const HidPad& hidPad) NN_NOEXCEPT
{
    m_ItemList.UpdateByHid( hidPad );

    if( m_Panel.IsFocused() )
    {
        switch( m_ItemList.GetSelectedItemIndex() )
        {
        case ItemType_OutputMode:
            if( hidPad.IsContinue( HidPad::Button_Left ) )
            {
                if( m_OutputMode > 0 )
                {
                    m_OutputMode--;
                    SetOutputMode( m_OutputMode, m_ItemList.GetValueLabel( ItemType_OutputMode ) );
                }
            }
            else if( hidPad.IsContinue( HidPad::Button_Right ) )
            {
                if( m_OutputMode < UsableOutputModeThreshold )
                {
                    m_OutputMode++;
                    SetOutputMode( m_OutputMode, m_ItemList.GetValueLabel( ItemType_OutputMode ) );
                }
            }
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
    }
}
//  描画します
void SoundSystemPanel::Draw(gfxutil::GfxContext& gfxContext) NN_NOEXCEPT
{
    m_Panel.Draw( gfxContext );

    const nn::util::Float2 panelPos = m_Panel.GetClientPositionLeftTop();
    m_ItemList.Draw( gfxContext, panelPos.x, panelPos.y );
}

//  フォーカスを設定します
void SoundSystemPanel::SetFocused(bool isFocused) NN_NOEXCEPT
{
    m_Panel.SetFocused( isFocused );
    m_ItemList.SetFocused( isFocused );
}
