﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nn/atk.h>
#include <nn/fs.h>
#include "ConnectionUtil.h"
#include "SoundArchiveContext.h"
#include "Application.h"
#include "Util.h"
#include "FlightRecorder.h"
#if defined( NN_BUILD_CONFIG_OS_WIN )
#include <nn/nn_Windows.h>  //  Windows の max マクロなどの置き換えの対策のために最後で include
#endif

namespace
{
    //  gfx に割り当てるヒープサイズ
    const size_t HeapMemorySizeForGfx =                       16 * 1024 * 1024;
    //  atk の SoundSystem に割り当てるヒープサイズ ( 9 * 1920 * 1024 は DelayEffect 分のバッファ )
    const size_t HeapMemorySizeForAtkSoundSystem  =           8 * 1024 * 1024 + 9 * 1920 * 1024;
    //  atk の SoundArchive に割り当てるヒープサイズ
    const size_t HeapMemorySizeForAtkSoundArchive = ( 320 + 16 ) * 1024 * 1024;

    const size_t HeapMemorySizeForOpusDecoder =                    300 * 1024;
    //  fs, htcs に割り当てるヒープサイズ
    const size_t HeapMemorySizeForConnection =                 1 * 1024 * 1024;
    //  アロケータの管理領域ヒープサイズ
    const size_t HeapMemorySizeForAllocator =                  1 * 1024 * 1024;
    //  AtkPlayer が利用するヒープサイズ
    const size_t HeapMemorySizeForAtkPlayer =                  4 * 1024 * 1024;
    //  ヒープとそのサイズ
    const size_t HeapMemorySize = HeapMemorySizeForGfx + HeapMemorySizeForAtkSoundSystem + HeapMemorySizeForAtkSoundArchive + HeapMemorySizeForOpusDecoder + HeapMemorySizeForConnection + HeapMemorySizeForAllocator + HeapMemorySizeForAtkPlayer;
    char g_HeapMemory[HeapMemorySize];
    //  アロケータ
    nn::mem::StandardAllocator g_StandardAllocator;

    //  FlightRecorder のログファイルの名前
    const char* FlightRecorderLogName = "AtkPlayer.log";

    //  メモリを確保します
    void* Allocate(std::size_t size) NN_NOEXCEPT
    {
        return g_StandardAllocator.Allocate( size );
    }
    //  メモリを確保します
    void* Allocate(std::size_t size, int alignment) NN_NOEXCEPT
    {
        return g_StandardAllocator.Allocate( size, alignment );
    }
    //  メモリを解放します
    void Free(void* pMemory) NN_NOEXCEPT
    {
        if( pMemory != NULL )
        {
            g_StandardAllocator.Free( pMemory );
        }
    }
}
namespace
{
    //  ディスプレイの幅と高さ
    const int DisplayWidth = 1280;
    const int DisplayHeight = 720;
    //  各パネル間のマージン
    const float MarginEachPanels = 10.0f;

    //  SoundArchivePlayerPanel のサイズ
    const float SoundArchivePlayerPanelSizeX = 720.0f;
    const float SoundArchivePlayerPanelSizeY = 119.0f;

    //  SoundParametersPanel のサイズ
    const float SoundParametersPanelSizeX = 360.0f;
    const float SoundParametersPanelSizeY = 435.0f;

    //  SoundSystemPanel のサイズ
    const float SoundSystemPanelSizeX = 360.0f;
    const float SoundSystemPanelSizeY = 120.0f;

    //  InformationPanel のサイズ
    const float InformationPanelSizeX = 50.0f;
    const float InformationPanelSizeY = 20.0f;

    //  EffectPanel のサイズ
    const float EffectPanelSizeX = 340.0f;
    const float EffectPanelSizeY = 350.0f;

    //  SoundOutputPanel のサイズ
    const float SoundOutputPanelSizeX = 500.0f + 24.0f;
    const float SoundOutputPanelSizeY = 350.0f;

    //  nn::atk::SoundSystem::Initialize に失敗したとき
    const char* FailedToInitializeSoundSystemText = "                   -- ERROR --\n\n This system failed to initialize \"SoundSystem\". \n      Please refer to AtkPlayer documents.\n";
    const nn::util::Unorm8x4 FailedToInitializeSoundSystemColor = GetUnorm8x4( 0, 0, 0, 255 );
    const nn::util::Uint8x4 FailedToInitializeSoundSystemBackColor = GetUint8x4( 255, 255, 255, 255 );
    const nn::util::Uint8x4 FailedToInitializeSoundSystemEdgeColor = GetUint8x4( 128, 0, 0, 255 );
    const float FailedToInitializeSoundSystemEdgeSize = 8.0f;
    const float FailedToInitializeSoundSystemScale = 2.0f;
}
namespace
{
#if defined( NN_BUILD_CONFIG_OS_WIN )
    //  Windows ならば必要な更新処理をします
    //  true を返したとき、アプリを終了します
    bool UpdateWindows() NN_NOEXCEPT
    {
        MSG  msg;
        if( PeekMessage( &msg, nullptr, 0, 0, PM_REMOVE ) )
        {
            TranslateMessage( &msg );
            if( msg.message == WM_QUIT )
            {
                return true;
            }
            DispatchMessage( &msg );
        }
        return false;
    }
#endif
}

//  初期化します
void Application::Initialize() NN_NOEXCEPT
{
    m_IsContinue = true;
    m_FocusPanelType = PanelType_SoundArchivePlayer;
#if !defined( ATKPLAYER_BUILD_CONFIG_ENABLE_GFX )
    m_NeedDrawCount = 1;
#endif

    //  Allocator の初期化
    g_StandardAllocator.Initialize( g_HeapMemory, HeapMemorySize );

    //  HidPad の初期化
    m_HidPad.Initialize();

    //  nn::fs, nn::htcs の初期化
    {
        m_pMemoryForConnection = Allocate( HeapMemorySizeForConnection );
        NN_ABORT_UNLESS_NOT_NULL( m_pMemoryForConnection );

        InitializeConnectionUtil( m_pMemoryForConnection, HeapMemorySizeForConnection );
    }

    //  コマンドライン引数を解釈します
    bool isEnabledFlightRecorder = false;
    m_StartupSoundArchivePath = nullptr;
    {
        const char* enableFlightRecorderOption = "-DebugMode:FlightRecorder";
        const int argc = nn::os::GetHostArgc();
        const char* const* argv = nn::os::GetHostArgv();

        for( int i = 1; i < argc; i++ )
        {
            if( argv[i][0] != '-' )
            {
                //  TORIAEZU: - から始まらないコマンドライン引数はサウンドアーカイブのパスであると解釈します
                m_StartupSoundArchivePath = argv[i];
            }
            else if( strcmp( argv[i], enableFlightRecorderOption ) == 0 )
            {
                isEnabledFlightRecorder = true;
            }
        }
    }

    //  FlightRecorder の初期化
    FlightRecorder::GetInstance().Initialize( isEnabledFlightRecorder, FlightRecorderLogName );

    //  GfxDeviceContext の初期化
    {
        m_pMemoryForGfxDevice = Allocate( HeapMemorySizeForGfx );
        NN_ABORT_UNLESS_NOT_NULL( m_pMemoryForGfxDevice );

        m_GfxContext.Initialize( m_pMemoryForGfxDevice, HeapMemorySizeForGfx, DisplayWidth, DisplayHeight );
    }

    {
        // Opus の初期化
        int decoderCount = 4;
        size_t memSizeForOpusDecoder = nn::atk::GetRequiredOpusDecoderBufferSize(decoderCount);
        m_pMemoryForOpusDecoder = Allocate(memSizeForOpusDecoder);
        nn::atk::InitializeOpusDecoder(m_pMemoryForOpusDecoder, memSizeForOpusDecoder, decoderCount);

        // SoundSystem の初期化
        nn::atk::SoundSystem::SoundSystemParam param;
        param.enableProfiler = true;
        param.enableRecordingFinalOutputs = true;
        param.enableCircularBufferSink = true;
        const size_t memSizeForSoundSystem = nn::atk::SoundSystem::GetRequiredMemSize( param );

        NN_ABORT_UNLESS_GREATER_EQUAL( HeapMemorySizeForAtkSoundSystem, memSizeForSoundSystem );
        //  余ったメモリはエフェクトに使われる

        m_pMemoryForSoundSystem = Allocate( memSizeForSoundSystem, nn::atk::SoundSystem::WorkMemoryAlignSize );
        NN_ABORT_UNLESS_NOT_NULL( m_pMemoryForSoundSystem );

        m_IsFailedToInitializeSoundSystem = !nn::atk::SoundSystem::Initialize( param, reinterpret_cast<uintptr_t>( m_pMemoryForSoundSystem ), memSizeForSoundSystem );
        if( m_IsFailedToInitializeSoundSystem )
        {
            //  nn::atk::SoundSystem::Initialize に失敗したため、エラー表示のための初期化
            m_FailedToInitializeSoundSystemLabel.SetText( FailedToInitializeSoundSystemText );
            m_FailedToInitializeSoundSystemLabel.SetColor( FailedToInitializeSoundSystemColor );
            m_FailedToInitializeSoundSystemLabel.SetBackColor( FailedToInitializeSoundSystemBackColor );
            m_FailedToInitializeSoundSystemLabel.SetEdgeColor( FailedToInitializeSoundSystemEdgeColor );
            m_FailedToInitializeSoundSystemLabel.SetScale( FailedToInitializeSoundSystemScale, FailedToInitializeSoundSystemScale );
            m_FailedToInitializeSoundSystemLabel.SetEdgeSize( FailedToInitializeSoundSystemScale * FailedToInitializeSoundSystemEdgeSize );

#if  defined(ATKPLAYER_BUILD_CONFIG_ENABLE_GFX)
            const float positionX = 0.5f * ( DisplayWidth  - FailedToInitializeSoundSystemScale * m_GfxContext.GetFontRenderer().CalculateWidth ( FailedToInitializeSoundSystemText ) );
            const float positionY = 0.5f * ( DisplayHeight - FailedToInitializeSoundSystemScale * m_GfxContext.GetFontRenderer().CalculateHeight( FailedToInitializeSoundSystemText ) );
            m_FailedToInitializeSoundSystemLabel.SetPosition( positionX, positionY );
#endif
            return ;
        }

        //  SoundArchiveContext の初期化
        m_pMemoryForSoundArchive = Allocate( HeapMemorySizeForAtkSoundArchive );
        NN_ABORT_UNLESS_NOT_NULL( m_pMemoryForSoundArchive );

        m_SoundArchiveContext.Initialize( m_pMemoryForSoundArchive, HeapMemorySizeForAtkSoundArchive );

    }

    //  SoundArchiveContext
    if( m_StartupSoundArchivePath != nullptr )
    {
        m_SoundArchiveContext.Open( m_StartupSoundArchivePath );
    }

    //  パネル, ランプの初期化
    {
        const float SoundArchivePlayerPanelPosisionX =  MarginEachPanels;
        const float SoundArchivePlayerPanelPosisionY =  MarginEachPanels;
        const float SoundParametersPanelPosisionX    =  MarginEachPanels;
        const float SoundParametersPanelPosisionY    =  MarginEachPanels + SoundArchivePlayerPanelSizeY + SoundArchivePlayerPanelPosisionY;
        const float SoundSystemPanelPosisionX        =  MarginEachPanels;
        const float SoundSystemPanelPosisionY        =  MarginEachPanels + SoundParametersPanelSizeY + SoundParametersPanelPosisionY;
        const float InformationPanelPosisionX        = -MarginEachPanels + DisplayWidth;
        const float InformationPanelPosisionY        =  MarginEachPanels;
        const float EffectPanelPositionX             =  MarginEachPanels + SoundSystemPanelPosisionX + SoundSystemPanelSizeX;
        const float EffectPanelPositionY             =  SoundSystemPanelPosisionY + SoundSystemPanelSizeY - EffectPanelSizeY;
        const float SoundOutputPanelPositionX        =  MarginEachPanels + EffectPanelPositionX + EffectPanelSizeX;
        const float SoundOutputPanelPositionY        =  SoundSystemPanelPosisionY + SoundSystemPanelSizeY - SoundOutputPanelSizeY;

        m_SoundArchivePlayerPanel.Initialize( SoundArchivePlayerPanelPosisionX, SoundArchivePlayerPanelPosisionY, SoundArchivePlayerPanelSizeX, SoundArchivePlayerPanelSizeY );
        m_SoundParametersPanel.Initialize( m_SoundArchiveContext, SoundParametersPanelPosisionX, SoundParametersPanelPosisionY, SoundParametersPanelSizeX, SoundParametersPanelSizeY );
        m_SoundSystemPanel.Initialize( SoundSystemPanelPosisionX, SoundSystemPanelPosisionY, SoundSystemPanelSizeX, SoundSystemPanelSizeY );
        m_InformationPanel.Initialize( InformationPanelPosisionX, InformationPanelPosisionY, InformationPanelSizeX, InformationPanelSizeY );
        m_EffectPanel.Initialize( EffectPanelPositionX, EffectPanelPositionY, EffectPanelSizeX, EffectPanelSizeY );

        const size_t memSizeForSoundOutputPanel = m_SoundOutputPanel.GetRequiredMemorySize();
        m_pMemoryForSoundOutputPanel = Allocate( memSizeForSoundOutputPanel );
        NN_ABORT_UNLESS_NOT_NULL( m_pMemoryForSoundOutputPanel );
        m_SoundOutputPanel.Initialize( m_pMemoryForSoundOutputPanel, memSizeForSoundOutputPanel, SoundOutputPanelPositionX, SoundOutputPanelPositionY, SoundOutputPanelSizeX, SoundOutputPanelSizeY );

        SetFocusedToPanel();
    }

    //  spy の初期化
    {
        const size_t memSizeForSpyModule = m_SpyToolProfiler.GetRequiredMemorySize();
        m_pMemoryForSpyModule = Allocate( memSizeForSpyModule );
        NN_ABORT_UNLESS_NOT_NULL( m_pMemoryForSpyModule );
        m_SpyToolProfiler.Initialize( m_pMemoryForSpyModule, memSizeForSpyModule );
    }
} // NOLINT(impl/function_size)
//  終了処理をします
void Application::Finalize() NN_NOEXCEPT
{
    m_SpyToolProfiler.Finalize();
    Free( m_pMemoryForSpyModule );

    if( !m_IsFailedToInitializeSoundSystem )
    {
        //  SoundSystem の初期化に失敗していなければ終了処理
        m_SoundOutputPanel.Finalize();
        Free( m_pMemoryForSoundOutputPanel );

        m_EffectPanel.Finalize( &g_StandardAllocator );

        m_SoundArchiveContext.Finalize();
        Free(m_pMemoryForSoundArchive);

        nn::atk::SoundSystem::Finalize();
        Free( m_pMemoryForSoundSystem );

        nn::atk::FinalizeOpusDecoder();
        Free(m_pMemoryForOpusDecoder);
    }

    m_GfxContext.Finalize();
    Free( m_pMemoryForGfxDevice );

    FlightRecorder::GetInstance().Finalize();
    FinalizeConnectionUtil();
    g_StandardAllocator.Finalize();
}
//  更新処理をします
void Application::Update() NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_WIN )
    if( UpdateWindows() )
    {
        m_IsContinue = false;
        return ;
    }
#endif

    m_HidPad.Update();
    if( m_IsFailedToInitializeSoundSystem )
    {
        //  SoundSystem の初期化に失敗していた場合、処理をここで中断する
        return ;
    }

    m_SoundArchiveContext.Update();
    {
        WrappedSoundHandle& currentHandle = m_SoundArchivePlayerPanel.GetCurrentHandle( m_SoundArchiveContext );
        m_SoundArchivePlayerPanel.Update( m_SoundArchiveContext );
        m_SoundParametersPanel.Update( currentHandle );
        m_InformationPanel.Update( m_SoundArchiveContext );
    }
    m_EffectPanel.Update( &g_StandardAllocator );
    m_SoundOutputPanel.Update();

    //  Z の入力があるときはパネルの移動
    if( m_HidPad.IsHold( HidPad::Button_Z ) )
    {
        if( m_HidPad.IsTrigger( HidPad::Button_Up ) )
        {
            m_FocusPanelType = GetNextValueOnLoopSelection( m_FocusPanelType - 1, PanelType_Max );
            SetFocusedToPanel();
        }
        if( m_HidPad.IsTrigger( HidPad::Button_Down ) )
        {
            m_FocusPanelType = GetNextValueOnLoopSelection( m_FocusPanelType + 1, PanelType_Max );
            SetFocusedToPanel();
        }
    }
    //  R の入力があるときは、どのパネルにフォーカスがあるときでも行われる処理
    else if( m_HidPad.IsHold( HidPad::Button_R ) )
    {
        //  再生, 停止機能
        WrappedSoundHandle& handle = m_SoundArchivePlayerPanel.GetCurrentHandle( m_SoundArchiveContext );
        if( m_HidPad.IsTrigger( HidPad::Button_A ) )
        {
            handle.Start();
        }
        else if( m_HidPad.IsTrigger( HidPad::Button_B ) )
        {
            handle.Stop();
        }

        //  RELOAD_SOUND_ARCHIVE_TEMP: サウンドアーカイブの再読み込みのための暫定処理
        //  サウンドアーカイブを再度読み込み、InformationPanel に通知します
        else if( m_HidPad.IsTrigger( HidPad::Button_X ) )
        {
            if( m_StartupSoundArchivePath != nullptr )
            {
                m_SoundArchiveContext.Open( m_StartupSoundArchivePath );
                m_InformationPanel.OnReloadSoundArchive();
            }
        }
    }
    else
    {
        m_SoundArchivePlayerPanel.UpdateByHid( m_SoundArchiveContext, m_HidPad );
        m_SoundParametersPanel.UpdateByHid( m_SoundArchivePlayerPanel.GetCurrentHandle( m_SoundArchiveContext ), m_HidPad );
        m_SoundSystemPanel.UpdateByHid( m_HidPad );
        m_EffectPanel.UpdateByHid( m_HidPad );
        m_SoundOutputPanel.UpdateByHid( m_HidPad );
    }

#if !defined( ATKPLAYER_BUILD_CONFIG_ENABLE_GFX )
    //  スタートボタンで終了する
    if( m_HidPad.IsTrigger( HidPad::Button_Start ) )
    {
        m_IsContinue = false;
        return ;
    }

    if( m_HidPad.IsTriggerAnyKey() || m_HidPad.IsContinueAnyKey() )
    {
        //  1 回目の Draw() では値が反映されていないため、
        //  2 回 Draw() を行う必要がある
        m_NeedDrawCount = 2;
    }
#endif

    {
        auto pSoundArchivePlayer = m_SoundArchiveContext.GetSoundArchivePlayer();
        if( pSoundArchivePlayer != nullptr )
        {
            m_SpyToolProfiler.Update( *pSoundArchivePlayer );
        }
    }
}
//  描画処理をします
void Application::Draw() NN_NOEXCEPT
{
#if !defined( ATKPLAYER_BUILD_CONFIG_ENABLE_GFX )
    // Vsync の代わり
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( 16 ) );

    if( m_NeedDrawCount == 0 )
    {
        return ;
    }
    m_NeedDrawCount--;

    const char* pConnect = "";
    if( m_InformationPanel.IsConnected() )
    {
        pConnect = "connected";
    }
    else
    {
        pConnect = "no connection";
    }

    const char* pReload = "";
    if( m_InformationPanel.IsDrawReloadLabel() )
    {
        pReload = ", reloaded.";
    }
    else
    {
        pReload = ".";
    }

    NN_LOG( "    [info]: %14s%s\n", pConnect, pReload );
#endif

    m_GfxContext.Begin();

    if( m_IsFailedToInitializeSoundSystem )
    {
        //  SoundSystem の初期化に失敗していた場合、他のパネルの表示は行わない
        m_FailedToInitializeSoundSystemLabel.Draw( m_GfxContext );
    }
    else
    {
        m_SoundArchivePlayerPanel.Draw( m_GfxContext );
        m_SoundParametersPanel.Draw( m_GfxContext );
        m_SoundSystemPanel.Draw( m_GfxContext );
        m_InformationPanel.Draw( m_GfxContext );
        m_EffectPanel.Draw( m_GfxContext );
        m_SoundOutputPanel.Draw( m_GfxContext );
    }

    m_GfxContext.End();
}

//  アプリケーションが続いているかを返します
bool Application::IsContinue() const NN_NOEXCEPT
{
    return m_IsContinue;
}

//  パネルにフォーカスを設定します
void Application::SetFocusedToPanel() NN_NOEXCEPT
{
    m_SoundArchivePlayerPanel.SetFocused( m_FocusPanelType == PanelType_SoundArchivePlayer );
    m_SoundParametersPanel.SetFocused( m_FocusPanelType == PanelType_SoundParameters );
    m_SoundSystemPanel.SetFocused( m_FocusPanelType == PanelType_SoundSystem );
    m_EffectPanel.SetFocused( m_FocusPanelType == PanelType_Effect );
    m_SoundOutputPanel.SetFocused( m_FocusPanelType == PanelType_SoundOutput );
}
