﻿/*---------------------------------------------------------------------------*
  Copyright (C)Nintendo/HAL Laboratory, Inc.  All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *---------------------------------------------------------------------------*/

#if NEED_330_EXTENSION
#extension GL_ARB_separate_shader_objects : enable
#extension GL_ARB_enhanced_layouts : enable
#endif

#ifdef NN_GFX_VULKAN
#define FONT_BINDING_LAYOUT(n) layout( binding = n )
#define FONT_LOCATION_LAYOUT(n) layout( location = n )
#define FONT_BINDING(n) binding = n,
#else
#define FONT_BINDING_LAYOUT(n)
#define FONT_LOCATION_LAYOUT(n)
#define FONT_BINDING(n)
#endif

layout(std140, binding=0) uniform ShaderParam
{
    mat4 uUser;
    vec4 uInterpolateWidth;
    vec4 uInterpolateOffset;
    vec4 uShadowInterpolateWidth;
    vec4 uShadowInterpolateOffset;
};

struct PerCharacterParam
{
    vec4    uPosAndSize;    // width, height, x, y
    vec4    uTexCoord;      //
    uvec2   uColor;         // upperColor と lowerColor が uint にエンコードされている。
    int     uSheetIndex;    // 参照するテクスチャのインデックス
    float	uItalicOffset;  // イタリック体のオフセット値
    vec4    uTranslate;     // Translate の値(w には RectangleTopXOffset が入ります)

#if NW_PER_CHARACTER_TRANSFORM_ENABLED
    vec4    uRotateMatrixAndCenterX;    // 各軸回転ごとの sin の値と w には中心座標の X 値が入っている。
    vec4    uRotateMatrixAndCenterY;    // 各軸回転ごとの cos の値と w には中心座標の Y 値が入っている。
#endif
};

// 可変長配列としてアクセスしたいため Shader Storage Buffer としてパラメータを受け取る。
#ifdef NN_GFX_VULKAN
layout(std140, binding=1) buffer PerCharacterParamBlock
#else
layout(std140, binding=0) buffer PerCharacterParamBlock
#endif
{
    readonly PerCharacterParam uPerCharacterParams[];
};

FONT_LOCATION_LAYOUT(0) in vec3 aVertex;

// This is tentative workaround.
// NN_GFX_VULKAN is defined in preprocessor definition on compilation with ShaderConverter,
// and currently gfxVk library is using the reflection generated with ShaderConverter
// that doesn't define VULKAN. If we use gl_VertexIndex in NN_GFX_VULKAN enabled,
// ShaderConverter generates error.
// This will be resolved when ShaderConverter supports SPIR-V for gfxVulkan.
#ifdef VULKAN
#define gl_VertexID gl_VertexIndex
#define gl_InstanceID gl_InstanceIndex
#else
in int gl_VertexID;
in int gl_InstanceID;
#endif

out gl_PerVertex
{
    vec4 gl_Position;
};
layout(location = 0) out vec4 vColor;
layout(location = 1) out vec4 vTexCoord;

void main(void)
{
    const int TexCoordIndex[8] =
    {
        0, 1, 2, 1, 0, 3, 2, 3
    };
    const int UpperLowerTable[4] =
    {
        0, 0, 1, 1
    };

    vec2 size = aVertex.xy * uPerCharacterParams[gl_InstanceID].uPosAndSize.xy;

#if NW_PER_CHARACTER_TRANSFORM_ENABLED
    // イタリックと回転中心分オフセットをかけている。
    vec2 posOffset;
    posOffset.x = uPerCharacterParams[gl_InstanceID].uTranslate.w -
                    UpperLowerTable[gl_VertexID] * uPerCharacterParams[gl_InstanceID].uItalicOffset -
                    uPerCharacterParams[gl_InstanceID].uRotateMatrixAndCenterX.w;
    posOffset.y = -uPerCharacterParams[gl_InstanceID].uRotateMatrixAndCenterY.w;

    // 回転と回転後の中心座標戻し分＋移動を行う行列を作成する。
    mat4 mRot = mat4(
                    uPerCharacterParams[gl_InstanceID].uRotateMatrixAndCenterX.xyz, 0.0f,
                    uPerCharacterParams[gl_InstanceID].uRotateMatrixAndCenterY.xyz, 0.0f,
                    vec3(0.0f, 0.0f, 1.0f), 0.0f,
                    vec3(
                        uPerCharacterParams[gl_InstanceID].uRotateMatrixAndCenterX.w + uPerCharacterParams[gl_InstanceID].uTranslate.x,
                        uPerCharacterParams[gl_InstanceID].uRotateMatrixAndCenterY.w + uPerCharacterParams[gl_InstanceID].uTranslate.y,
                        uPerCharacterParams[gl_InstanceID].uTranslate.z),
                        1.0f
                    );

    vec4 pos = mRot * vec4(size + uPerCharacterParams[gl_InstanceID].uPosAndSize.zw + posOffset, 0.0f, 1.0f);

    gl_Position = pos * uUser;
#else
    // イタリックを実現するために、ポリゴンの上の頂点だけオフセットをかける。
    vec2 italicOffset;
    italicOffset.x = uPerCharacterParams[gl_InstanceID].uTranslate.w -
                        UpperLowerTable[gl_VertexID] * uPerCharacterParams[gl_InstanceID].uItalicOffset;
    italicOffset.y = 0.0f;

    gl_Position = (vec4(size + uPerCharacterParams[gl_InstanceID].uPosAndSize.zw + italicOffset, 0, 1)) * uUser;
#endif
    // カラー設定
    // 上下で色が違う設定ができるためテーブルで上下に振り分けて処理を分岐。
    uint color = uPerCharacterParams[gl_InstanceID].uColor[UpperLowerTable[gl_VertexID]];

    // uint にエンコードされた色を取り出す。
    vColor = unpackUnorm4x8(color);

    // TexCoord には CharAttribute.tex の内容が入っています。
    // 各頂点に対応した UV 値をテーブルを参照しつつ設定する。
    int vertexIndex = gl_VertexID - (gl_VertexID / 4) * 4;
    vTexCoord.x = uPerCharacterParams[gl_InstanceID].uTexCoord[TexCoordIndex[vertexIndex * 2]];
    vTexCoord.y = uPerCharacterParams[gl_InstanceID].uTexCoord[TexCoordIndex[vertexIndex * 2 + 1]];
    vTexCoord.z = uPerCharacterParams[gl_InstanceID].uSheetIndex;
    vTexCoord.w = 0.0f;

    // aTexCoord0.z にはシートインデックスが格納されていますが、
    // この値がマイナスの場合、影用の文字描画であることを表します。
    if (vTexCoord.z < 0)
    {
        // ピクセルシェーダでは頂点カラーのアルファ値がマイナスで
        // あることが影用の文字であることを表すので、符号を反転します。
        // aColor.a は符号なし１バイト整数値で送られてくるので、
        // 頂点シェーダでマイナスにします。
        // この判定を頂点シェーダで行っているのは、ピクセルシェーダの
        // 処理負荷を少しでも重くならないようにするためです。
        vColor.a = -vColor.a;
        vTexCoord.z = -vTexCoord.z - 1;
    }

#ifdef NN_GFX_VULKAN
    gl_Position.y = -gl_Position.y;
    gl_Position.z = (gl_Position.z + gl_Position.w) / 2.0;
#endif
}

