﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vfx/vfx_Misc.h>
#include <nn/vfx/vfx_System.h>
#include <nn/vfx/viewer/vfx_ModelPreview.h>
#include <nn/vfx/viewer/vfx_Viewer.h>
#include <nn/vfx/viewer/vfx_UsrModelPreview.h>
#include <nn/vfx/viewer/vfx_ToolConnector.h>
#include <nn/util/util_StringUtil.h>

#if defined( NN_BUILD_CONFIG_OS_WIN )
#include <nn/nn_Windows.h>
#include <cstdio>
#endif

//--------------------------------------------------------------------------------------------------
// ミドルウェア情報
#include <nn/nn_Middleware.h>
#include <nn/nn_Version.h>

#define NN_DETAIL_XX_MACRO_TO_STRING(x) NN_DETAIL_XX_MACRO_TO_STRING_(x)
#define NN_DETAIL_XX_MACRO_TO_STRING_(x) #x

#define NW_VFX_VIEWER_MIDDLEWARE_SYMBOL(buildOption) "NintendoWare_Vfx_Viewer_For_Develop-" NN_DETAIL_XX_MACRO_TO_STRING(NN_NX_ADDON_VERSION_MAJOR) "_" \
NN_DETAIL_XX_MACRO_TO_STRING(NN_NX_ADDON_VERSION_MINOR) "_" NN_DETAIL_XX_MACRO_TO_STRING(NN_NX_ADDON_VERSION_MICRO) "-" #buildOption

namespace {
#if defined(NN_SDK_BUILD_DEBUG)
    NN_DEFINE_MIDDLEWARE(g_MiddlewareInfo, "Nintendo", NW_VFX_VIEWER_MIDDLEWARE_SYMBOL(Debug));
#elif defined(NN_SDK_BUILD_DEVELOP)
    NN_DEFINE_MIDDLEWARE(g_MiddlewareInfo, "Nintendo", NW_VFX_VIEWER_MIDDLEWARE_SYMBOL(Develop));
#elif defined(NN_SDK_BUILD_RELEASE)
    NN_DEFINE_MIDDLEWARE(g_MiddlewareInfo, "Nintendo", NW_VFX_VIEWER_MIDDLEWARE_SYMBOL(Release));
#endif
}

//--------------------------------------------------------------------------------------------------

namespace nn {
namespace vfx {
namespace viewer {
namespace detail {

static const int MaxResourceEntryCount = 32;                   //!< EffectMaker から受け取るリソース最大数

} // namespace detail

//---------------------------------------------------------------------------
//  コンストラクタの内部処理
//---------------------------------------------------------------------------
void ViewerSystem::Internal_Constructor( nn::vfx::Heap* pHeap,
                                         nn::vfx::System* pSystem,
                                         nn::vfx::viewer::detail::ToolConnector* pToolConnector,
                                         size_t transmitBufferSize,
                                         bool noUseInternalThread,
                                         int priority ) NN_NOEXCEPT
{
    m_pTransmitBuffer = pHeap->Alloc( transmitBufferSize, 16 );
    NN_SDK_ASSERT_NOT_NULL( m_pTransmitBuffer );
    m_TransmitHeap.Initialize( m_pTransmitBuffer, transmitBufferSize );
    Initialize( pHeap, &m_TransmitHeap, pSystem, pToolConnector, noUseInternalThread, priority );
}

//---------------------------------------------------------------------------
//! @brief  デストラクタです。
//---------------------------------------------------------------------------
ViewerSystem::~ViewerSystem() NN_NOEXCEPT
{
    NN_SDK_ASSERT( m_IsFinalized, "The ViewerSystem Finalized function has not be called yet.\n" );
}

//---------------------------------------------------------------------------
//! @brief  初期化処理です。
//---------------------------------------------------------------------------
void ViewerSystem::Initialize( nn::vfx::Heap* pHeap,
                               nn::vfx::Heap* pConnectionHeap,
                               nn::vfx::System* pSystem,
                               nn::vfx::viewer::detail::ToolConnector* pToolConnector,
                               bool noUseInternalThread,
                               int priority ) NN_NOEXCEPT
{
    NN_USING_MIDDLEWARE(g_MiddlewareInfo);

    m_pViewerHeap = pHeap;
    m_pConnectionHeap = pConnectionHeap;
    m_Time = 0.0f;
    m_FrameRate = 1.0f;
    m_EndTime = 120.f;
    m_IsLoop = false;
    m_IsPaused = false;
    m_IsStepMode = false;
    m_IsAppInitialized = false;
    m_IsConnected = false;
    m_IsResetTime = false;
    m_IsReloadBinary = false;
    m_CreateModelPreviewCallback = NULL;
    m_DestroyModelPreviewCallback = NULL;
    m_IsFinalized = false;
    m_NoUseInternalThread = noUseInternalThread;
    m_StackBase = NULL;
    m_SentModelInfoCount = 0;
    nn::util::MatrixIdentity( &m_ViewerCenterMatrix );

    // エフェクトシステムのポインタを設定する
    m_pSystem = pSystem;

    m_ToolConnector = pToolConnector;

    // ツールコネクタの初期化
    m_ToolConnector->Initialize( m_pConnectionHeap );

    // 通信スレッドの初期化
    if ( ( m_ToolConnector->IsConnectorEmpty() == false ) && ( !m_NoUseInternalThread ) )
    {
        m_StackBase = reinterpret_cast< uint8_t* >( m_pConnectionHeap->Alloc( nn::vfx::viewer::detail::ToolConnector::taskMemsize,
                                                                              nn::vfx::viewer::detail::ToolConnector::taskAlignSize ) );
        nn::Result result = nn::os::CreateThread( &m_Thread,
                                                  ReadingPacketProcess,
                                                  reinterpret_cast< void* >( this ),
                                                  reinterpret_cast< void* >( m_StackBase ),
                                                  nn::vfx::viewer::detail::ToolConnector::taskMemsize,
                                                  priority );
        if ( result.IsSuccess() == true )
        {
            nn::os::SetThreadName( &m_Thread, "NnVfxViewerToolConnector" );
            nn::os::ChangeThreadPriority( &m_Thread, priority );
            nn::os::StartThread( &m_Thread );
        }
    }


    // パケットプロシジャーの初期化
    m_PacketProcedure.Initialize( this, m_pConnectionHeap );

    // リソースマネージャーの初期化
    if ( m_ToolConnector->IsConnectorEmpty() == false )
    {
        m_ResManager.Initialize( pHeap, detail::MaxResourceEntryCount, pSystem->GetResourceCountMax() - nn::vfx::Config::DefaultConfigSettings_ResourceCountForViewer );
    }

    // プレビュー先頭を初期化
    nn::vfx::viewer::detail::Guid guid;
    guid.Clear();
    m_PreviewHead.Initialize( pHeap, detail::Preview::VfxPreviewType_TopPreview, guid );
}

//---------------------------------------------------------------------------
//! @brief  パケットの受信処理の終了
//---------------------------------------------------------------------------
void ViewerSystem::StopReadingPacketProcessThread() NN_NOEXCEPT
{
    m_ToolConnector->StopThread();
}

//---------------------------------------------------------------------------
//  終了処理です。
//---------------------------------------------------------------------------
void ViewerSystem::Finalize( nn::vfx::UnregisterTextureViewSlot pUnregisterTextureSlotCallback, void* pUserData ) NN_NOEXCEPT
{
    // プレビューの破棄
    DestroyAllPreview();

    // リソース破棄
    DestroyAllResource( pUnregisterTextureSlotCallback, pUserData );
    m_ResManager.Finalize();

    // パケットプロシジャーの終了処理
    m_PacketProcedure.Finalize();

    // 通信スレッドの終了処理
    if ( ( m_ToolConnector->IsConnectorEmpty() == false ) && ( m_NoUseInternalThread == false ) )
    {
        StopReadingPacketProcessThread();
        nn::os::DestroyThread( &m_Thread );
        if ( m_StackBase != NULL )
        {
            m_pConnectionHeap->Free( m_StackBase );
            m_StackBase = NULL;
        }
    }
    m_ToolConnector->Finalize();
    m_ToolConnector->~ToolConnector();
    m_pViewerHeap->Free( m_pToolConnectorPtr );

    // 通信用ヒープバッファの終了処理
    if ( m_pTransmitBuffer != nullptr )
    {
        m_TransmitHeap.Finalize();
        m_pViewerHeap->Free( m_pTransmitBuffer );
        m_pTransmitBuffer = nullptr;
    }

    m_IsFinalized = true;
}

//---------------------------------------------------------------------------
//  アプリケーションの初期化完了を通知します。
//---------------------------------------------------------------------------
void ViewerSystem::SetAppIsInitialized() NN_NOEXCEPT
{
    // 接続中のとき初期化完了通知メッセージを送る
    if( m_IsConnected )
    {
        SendAppIsInitialized();
    }

    m_IsAppInitialized = true;
}

//---------------------------------------------------------------------------
// コマンド処理を実行します。
//---------------------------------------------------------------------------
void ViewerSystem::ExecuteCommand( nn::vfx::RegisterTextureViewSlot pRegisterTextureSlotCallback, void* pUserData ) NN_NOEXCEPT
{
    m_IsReloadBinary = false;

    // ToolConnectorが空の場合な何もしない
    detail::CommandReceiver* pCommandReceiver = m_ToolConnector->GetCommandReceiver();
    if ( pCommandReceiver== NULL ) return;

    // EffectMakerからのコマンド要求
    pCommandReceiver->BeginProc();
    detail::Command* pCommand = pCommandReceiver->GetEffectCommand();
    bool ret = false;
    while( pCommand )
    {
        switch( pCommand->GetCommand() )
        {
        // パケットを処理します
        case 101:

            ret = m_PacketProcedure.DoPacket( reinterpret_cast< char* >( pCommand->GetCommandBuffer() ),
                                                           static_cast< size_t >( pCommand->GetCommandBufferSize() ),
                                                           pRegisterTextureSlotCallback, pUserData );
            if ( ret ) m_IsReloadBinary = true;
            break;
        default:
            break;
        }

        // コマンドをpop
        pCommandReceiver->PopEffectCommand();
        pCommand = pCommandReceiver->GetEffectCommand();
    }
    pCommandReceiver->EndProc();
}

//---------------------------------------------------------------------------
//  全てのプレビューを破棄します。
//---------------------------------------------------------------------------
void ViewerSystem::DestroyAllPreview() NN_NOEXCEPT
{
    detail::Preview* pPreview = m_PreviewHead.GetNextPreview();
    while( pPreview )
    {
        detail::Preview* pTemp = pPreview;
        pPreview = pPreview->GetNextPreview();
        {
            // リストから削除
            pTemp->RemovePreview();

            // 破棄コールバック呼び出し
            pTemp->GetDestroyPreviewCallback()( m_pViewerHeap, pTemp );
        }
    }
}

//---------------------------------------------------------------------------
//  全てのリソースを破棄します。
//---------------------------------------------------------------------------
void ViewerSystem::DestroyAllResource( nn::vfx::UnregisterTextureViewSlot pUnregisterTextureSlotCallback, void* pUserData) NN_NOEXCEPT
{
    for( int i = 0; i < nn::vfx::Config::DefaultConfigSettings_ResourceCountForViewer; i++ )
    {
        int resId = i + ( m_pSystem->GetResourceCountMax() - nn::vfx::Config::DefaultConfigSettings_ResourceCountForViewer );

        if( m_pSystem->GetResource( resId ) )
        {
            m_pSystem->GetResource(resId)->UnregisterTextureViewFromDescriptorPool( pUnregisterTextureSlotCallback, pUserData );
            m_pSystem->ClearResource( m_pViewerHeap, resId );
        }
    }
}

//---------------------------------------------------------------------------
//  システムの定期処理
//---------------------------------------------------------------------------
bool ViewerSystem::ProcessCalculation( float frameRate, const nn::util::Matrix4x3fType& viewMatrix, nn::vfx::RegisterTextureViewSlot pRegisterTextureSlotCallback, void* pUserData ) NN_NOEXCEPT
{
    // 切断された。
    if( m_IsConnected && !m_ToolConnector->IsConnected() )
    {
        int resCount = m_ResManager.GetResourceCount();
        for ( int i = 0; i < resCount; i++ )
        {
            UnLoadEffectData( m_ResManager.GetResource(i)->guId );
        }

        m_IsConnected = m_ToolConnector->IsConnected();

        return true;
    }

    // 接続時にアプリケーションが初期化済みだったとき初期化完了メッセージを送信
    if( !m_IsConnected && m_ToolConnector->IsConnected() && m_IsAppInitialized )
    {
        SendAppIsInitialized();
    }

    // 接続状態を保持
    m_IsConnected = m_ToolConnector->IsConnected();
    if( !m_IsConnected )
    {
        detail::Preview* pPreview = &m_PreviewHead;
        while( NULL != pPreview )
        {
            pPreview->Calculate( m_IsPaused, m_FrameRate, m_ViewerCenterMatrix, viewMatrix );
            pPreview = pPreview->GetNextPreview();
        }
        return true;
    }

    // 今フレームに受信されたコマンドの実行
    ExecuteCommand( pRegisterTextureSlotCallback, pUserData );

    bool addFrame = false;
    if( !m_IsPaused )
    {
        addFrame = true;
    }
    else
    {
        if( m_IsStepMode )
        {
            addFrame = true;
        }
    }

    m_FrameRate = frameRate;
    bool isPausedPreviewCalculation = m_IsPaused && !m_IsStepMode;
    m_IsStepMode = false;

    if( addFrame )
    {
        if( m_IsResetTime )
        {
            ResetTime( true );
        }

        // ループ処理を行います
        if( m_IsLoop && m_Time >= m_EndTime - std::numeric_limits< float >::epsilon() )
        {
            m_Time = 0.0f;
            m_IsResetTime = true;
        }
        else
        {
            m_Time += m_FrameRate;
            m_IsResetTime = false;
        }
    }

    // プレビュー定期処理を行います。
    // プレビューのリセットをコール
    detail::Preview* pPreview = &m_PreviewHead;
    detail::Preview* pPreviewNext = NULL;

    // モデルの計算処理
    while( NULL != pPreview )
    {
        pPreviewNext = pPreview->GetNextPreview();
        if ( pPreview->GetPreviewType() == detail::Preview::VfxPreviewType_ModelPreview )
        {
            bool ret = pPreview->Calculate( isPausedPreviewCalculation, m_FrameRate, m_ViewerCenterMatrix, viewMatrix );
            if ( !ret )
            {
                pPreview->RemovePreview();
                pPreview->GetDestroyPreviewCallback()( m_pViewerHeap, pPreview );
            }
        }
        pPreview = pPreviewNext;
    }

    // エフェクトの計算処理
    pPreview = &m_PreviewHead;
    pPreviewNext = NULL;
    while( NULL != pPreview )
    {
        pPreviewNext = pPreview->GetNextPreview();
        if ( pPreview->GetPreviewType() == detail::Preview::VfxPreviewType_EffectPreview )
        {
            bool ret = pPreview->Calculate( isPausedPreviewCalculation, m_FrameRate, m_ViewerCenterMatrix, viewMatrix );
            if ( !ret )
            {
                pPreview->RemovePreview();
            }
        }
        pPreview = pPreviewNext;
    }

    // リソースのフラッシュ
    m_ResManager.Flush( m_pSystem, m_pViewerHeap );

    return ( !addFrame );
}

//------------------------------------------------------------------------------
//  タイムをリセットします。
//------------------------------------------------------------------------------
void ViewerSystem::ResetTime( bool doFade ) NN_NOEXCEPT
{
    // フェードで追えないエフェクトを削除
    if( !doFade )
    {
        m_pSystem->KillEmitterGroup( detail::EffectPreviewGroupId );
    }

    // プレビューのリセットをコール
    detail::Preview* pPreview = &m_PreviewHead;
    while( NULL != pPreview )
    {
        pPreview->ResetPreview( doFade );
        pPreview = pPreview->GetNextPreview();
    }

    m_Time = 0.0f;
}


//------------------------------------------------------------------------------
//  強制的にフェードさせます。
//------------------------------------------------------------------------------
void ViewerSystem::ForceFade() NN_NOEXCEPT
{
    detail::Preview* preview = &m_PreviewHead;

    while( preview )
    {
        if( preview->GetPreviewType() == detail::Preview::VfxPreviewType_EffectPreview )
        {
            detail::EffectPreview* effectPreview = reinterpret_cast< detail::EffectPreview* >( preview );
            effectPreview->Fade();
        }

        preview = preview->GetNextPreview();
    }
}


//---------------------------------------------------------------------------
//  指定リソースIDのエフェクトプレビューを削除。
//---------------------------------------------------------------------------
bool ViewerSystem::RemoveEffectPreviewFromResId( int resId ) NN_NOEXCEPT
{
    bool ret = false;
    detail::Preview* pPreview = &m_PreviewHead;

    while( pPreview )
    {
        detail::Preview* pNext = pPreview->GetNextPreview();
        if( pPreview->GetPreviewType() == detail::Preview::VfxPreviewType_EffectPreview )
        {
            detail::EffectPreview* pEffectPreview = reinterpret_cast< detail::EffectPreview* >( pPreview );

            if( pEffectPreview->GetResourceId() == resId )
            {
                pPreview->RemovePreview();
                detail::EffectPreview::DestroyEffectPreview( m_pViewerHeap, pPreview );
                ret = true;
            }
        }
        pPreview = pNext;
    }

    return ret;
}


//---------------------------------------------------------------------------
//  エミッタセットバイナリの読み込みを行います。
//---------------------------------------------------------------------------
bool ViewerSystem::LoadEffectData( const char* pPath, nn::vfx::viewer::detail::Guid guid, nn::vfx::RegisterTextureViewSlot pRegisterTextureSlotCallback, void* pUserData ) NN_NOEXCEPT
{
    detail::ResourceManager::EffectResource* pEffectResource = NULL;

    int oldResId = -1;
    int newResId = -1;

    // リソースが既登録であれば、それに伴うプレビューを削除する。
    if( m_ResManager.GetResource( guid ) )
    {
        oldResId = m_ResManager.GetResource( guid )->resourceId;
        if( oldResId != -1 )
        {
            UnLoadEffectData( m_ResManager.GetResource( guid )->guId );
        }
    }

    // リソースを登録する。
    if( m_ResManager.Entry( pPath, guid ) )
    {
        pEffectResource = m_ResManager.GetResource( guid );
        if( !pEffectResource )
        {
            return false;
        }
        newResId = pEffectResource->resourceId;
        if( newResId == -1 )
        {
            return false;
        }
    }
    else
    {
        return false;
    }

    // リソース登録
    m_pSystem->EntryResource( m_pViewerHeap, pEffectResource->binary, newResId, false );
    m_pSystem->GetResource(newResId)->RegisterTextureViewToDescriptorPool( pRegisterTextureSlotCallback, pUserData );

    char text[ nn::vfx::viewer::detail::Guid::GuidSettings_StringBufferSize ];
    NN_UNUSED( text );
    nn::vfx::detail::OutputLog( "Entry Resource %x : %d / guid = %s\n", pEffectResource->binary, newResId, guid.GetString( text, nn::vfx::viewer::detail::Guid::GuidSettings_StringBufferSize ) );
    m_pSystem->OutputResourceInfo( newResId );

    // バインド対象のリソースがあればバインドする。
    BindResource( guid );

    // oldResIdに紐づいたエミッタセットは、
    // newResIdで再生成する
    if( oldResId != -1 )
    {
        const char* name = m_pSystem->GetResource( newResId )->GetEmitterSetName( 0 );
        m_pSystem->RecreateEmitterSet2( name, oldResId, newResId );
    }

    return true;
}

//---------------------------------------------------------------------------
//  エミッタセットバイナリの破棄を行います。
//---------------------------------------------------------------------------
bool ViewerSystem::UnLoadEffectData( nn::vfx::viewer::detail::Guid guid ) NN_NOEXCEPT
{
    detail::ResourceManager::EffectResource* resource = m_ResManager.GetResource( guid );

    if( resource )
    {
        if( resource->resourceId == -1 )
        {
            return false;
        }

        // リソースに紐づいたプレビューを削除
        RemoveEffectPreviewFromResId( resource->resourceId );

        // 対象のリソースがあればunbindする。
        UnbindResource( guid );

        // それ以外にも紐づいたエフェクトがあれば削除
        const char* name = m_pSystem->GetResource( resource->resourceId )->GetEmitterSetName( 0 );
        m_pSystem->KillEmitterSet( name, resource->resourceId );

        // リソースの利用を不可に。
        m_pSystem->GetResource( resource->resourceId )->SetKillMark();

        // リソースの削除
        m_ResManager.Release( guid );

        char text[ nn::vfx::viewer::detail::Guid::GuidSettings_StringBufferSize ];
        NN_UNUSED( text );
        nn::vfx::detail::OutputLog( "Clear Resource guid = %s\n", guid.GetString( text, nn::vfx::viewer::detail::Guid::GuidSettings_StringBufferSize ) );
    }

    return true;
}

//---------------------------------------------------------------------------
// エフェクトプレビューを生成します。
//---------------------------------------------------------------------------
bool ViewerSystem::CreateEffectPreview( nn::vfx::viewer::detail::Guid resGuid, nn::vfx::viewer::detail::Guid prevGuid ) NN_NOEXCEPT
{
    // TODO:ビューアではVisible機能の為、0Emitterのデータの再生要求があるのでリソースのエミッタ数を確認
    //      エミッタセットを探す方法は後程考える。

    //再生対象となるリソースを取得
    detail::ResourceManager::EffectResource* res = m_ResManager.GetResource( resGuid );
    if( !res )
    {
        return false;
    }

    // プレビューを生成
    detail::EffectPreview* vfxPreview = detail::EffectPreview::CreateEffectPreview( m_pViewerHeap, m_pSystem, prevGuid );
    if( !vfxPreview )
    {
        return false;
    }

    // エミッタセット生成する為のリソースIDを設定
    vfxPreview->SetResourceId( res->resourceId );

    // 生成したプレビューをリストに追加
    m_PreviewHead.AddPreview( vfxPreview );

    return true;
}

//---------------------------------------------------------------------------
// モデルプレビューを生成します。
//---------------------------------------------------------------------------
bool ViewerSystem::CreateModelPreview( nn::vfx::viewer::detail::Guid guid, void* pData, size_t dataSize ) NN_NOEXCEPT
{
    if( m_CreateModelPreviewCallback )
    {
        detail::Preview* pModelPreview = m_CreateModelPreviewCallback( m_pViewerHeap, this, pData, dataSize, guid );
        if( !pModelPreview )
        {
            return false;
        }

        // 生成したプレビューをリストに追加
        m_PreviewHead.AddPreview( pModelPreview );
    }
    return true;
}

//---------------------------------------------------------------------------
// モデルプレビューを破棄します。
//---------------------------------------------------------------------------
bool ViewerSystem::DestroyModelPreview( nn::vfx::viewer::detail::Guid guid ) NN_NOEXCEPT
{
    if( m_DestroyModelPreviewCallback )
    {
        // 破棄するプレビューを取得
        detail::Preview* pModelPreview = m_PreviewHead.GetPreview( guid );
        if( !pModelPreview )
        {
            return false;
        }

        // 削除カウンタを設定
        pModelPreview->SetReleaseCounter( 3 );
    }
    return true;
}

//---------------------------------------------------------------------------
//! @brief        指定エミッタセットをバインドします。
//---------------------------------------------------------------------------
bool ViewerSystem::BindResource( nn::vfx::viewer::detail::Guid resGuid ) NN_NOEXCEPT
{
    // 変更元リソース と 変更先リソース の特定
    nn::vfx::Resource* pSrcResource = NULL;
    nn::vfx::Resource* pDstResource = NULL;
    int dstEmitterSetId = nn::vfx::InvalidValueId_EmitterSetId;

    // 変更元リソース
    {
        detail::ResourceManager::EffectResource* res = m_ResManager.GetResource( resGuid );
        if ( !res )
        {
            return false;
        }
        if ( res->resourceId == -1 )
        {
            return false;
        }

        // 変更元リソースは、EffectMakerから送られてきたリソースなので、
        // EmitterSetは１つしか含まない。
        pSrcResource = m_pSystem->GetResource( res->resourceId );
        if ( !pSrcResource )
        {
            return false;
        }
    }

    // バインドするエミッタセットのファイルパス
    const char* emitterSetFilePath = pSrcResource->GetEmitterSetFilePath( 0 );
    // バインドするエミッタセット名
    const char* emitterSetName = pSrcResource->GetEmitterSetName( 0 );

    // 全リソースの中から同名のエミッタセットを検索する
    bool ret = false;
    int  resCount = m_pSystem->GetResourceCountMax() - nn::vfx::Config::DefaultConfigSettings_ResourceCountForViewer;
    for ( int i = 0; i < resCount; ++i )
    {
        pDstResource = m_pSystem->GetResource( i );
        if ( !pDstResource )
        {
            continue;
        }

        // 差し替え先リソースがファイルパス付かどうか
        bool isFilePathSearch = false;
        if ( pDstResource->GetEmitterSetResource( 0 )->pFilePath )
        {
            isFilePathSearch = true;

            if ( emitterSetFilePath == nullptr )
            {
                continue;
            }
        }

        // ファイルパスでチェック
        if ( isFilePathSearch )
        {
            dstEmitterSetId = pDstResource->SearchEmitterSetIdWithFilePath( emitterSetFilePath );
        }
        else
            // エミッタセット名でチェック
        {
            dstEmitterSetId = pDstResource->SearchEmitterSetId( emitterSetName );
        }

        // 差し替え対象が検索できなかった
        if ( dstEmitterSetId == static_cast< int >( nn::vfx::InvalidValueId_EmitterSetId ) )
        {
            continue;
        }

        // 差し替え対象が見つかればリソースを差し替える
        if ( pDstResource->BindResource( dstEmitterSetId, pSrcResource->GetEmitterSetResource( 0 ) ) )
        {
            // 差し替えたことをEffectMakerに送信( 鎖マークアイコン )
            SendLinkedEmitterSetV2E( emitterSetName );

            ret = true;
            break;
        }
        else
        {
            // 差し替え失敗
            nn::vfx::detail::OutputError( "EmitterSet Resource Bind Failed.\n" );
            break;
        }
    }

    return ret;
}

//---------------------------------------------------------------------------
//! @brief        指定リソースをアンバインドします。
//---------------------------------------------------------------------------
bool ViewerSystem::UnbindResource( nn::vfx::viewer::detail::Guid resGuid ) NN_NOEXCEPT
{
    // 変更元リソース と 変更先リソース の特定
    nn::vfx::Resource* pSrcResource = NULL;
    nn::vfx::Resource* pDstResource = NULL;
    int dstEmitterSetId = nn::vfx::InvalidValueId_EmitterSetId;

    // 変更元リソース
    {
        detail::ResourceManager::EffectResource* res = m_ResManager.GetResource( resGuid );
        if( !res )
        {
            return false;
        }
        if( res->resourceId == -1 )
        {
            return false;
        }

        // 変更元リソースは、EffectMakerから送られてきたリソースなので、
        // EmitterSetは１つしか含まない。
        pSrcResource = m_pSystem->GetResource( res->resourceId );
        if( !pSrcResource )
        {
            return false;
        }
    }

    // アンバインドするエミッタセットのファイルパス
    const char* emitterSetFilePath = pSrcResource->GetEmitterSetFilePath( 0 );
    // アンバインドするエミッタセット名
    const char* emitterSetName = pSrcResource->GetEmitterSetName( 0 );

    // 変更先リソース
    {
        // 全リソースの中から同名のエミッタセットを検索する
        for( int i = 0; i < m_pSystem->GetResourceCountMax() - nn::vfx::Config::DefaultConfigSettings_ResourceCountForViewer; ++i )
        {
            pDstResource = m_pSystem->GetResource( i );
            if( !pDstResource )
            {
                continue;
            }

            // 差し替え先リソースがファイルパス付かどうか
            bool isFilePathSearch = false;
            if ( pDstResource->GetEmitterSetResource( 0 )->pFilePath )
            {
                isFilePathSearch = true;

                if ( emitterSetFilePath == nullptr )
                {
                    continue;
                }
            }


            // ファイルパスでチェック
            if ( isFilePathSearch )
            {
                dstEmitterSetId = pDstResource->SearchEmitterSetIdWithFilePath( emitterSetFilePath );
            }
            else
                // エミッタセット名でチェック
            {
                dstEmitterSetId = pDstResource->SearchEmitterSetId( emitterSetName );
            }

            // 差し替え対象が検索できなかった
            if ( dstEmitterSetId == static_cast< int >( nn::vfx::InvalidValueId_EmitterSetId ) )
            {
                continue;
            }

            // リソースを差し替え
            if( pDstResource->UnbindResource( dstEmitterSetId ) )
            {
                // EM4Fに指定エミッタセットがユーザ再生済みエフェクトがバインドが解消されたことを送信する。
                SendUnLinkedEmitterSetV2E( emitterSetName );
                break;
            }
            else
            {
                // 差し替え失敗
                break;
            }
        }
    }

    return true;
}

//---------------------------------------------------------------------------
//! @brief  指定リソースをアンバインドします。
//---------------------------------------------------------------------------
bool ViewerSystem::UnbindResource( int resId ) NN_NOEXCEPT
{
    // 変更元リソース と 変更先リソース の特定
    nn::vfx::Resource* pSrcResource = NULL;
    nn::vfx::Resource* pDstResource = NULL;
    int dstEmitterSetId = nn::vfx::InvalidValueId_EmitterSetId;

    // 変更元リソース
    {
        // 変更元リソースは、EffectMakerから送られてきたリソースなので、
        // EmitterSetは１つしか含まない。
        pSrcResource = m_pSystem->GetResource( resId );
        if( !pSrcResource )
        {
            return false;
        }
    }

    // アンバインドするエミッタセット名
    const char* emitterSetName = pSrcResource->GetBinaryName();

    // 変更先リソース
    {
        // 全リソースの中から同名のエミッタセットを検索する
        for( int i = 0; i < m_pSystem->GetResourceCountMax() - nn::vfx::Config::DefaultConfigSettings_ResourceCountForViewer; ++i )
        {
            pDstResource = m_pSystem->GetResource( i );
            if( !pDstResource )
            {
                continue;
            }

            dstEmitterSetId = pDstResource->SearchEmitterSetId( emitterSetName );
            if( dstEmitterSetId == static_cast< int >( nn::vfx::InvalidValueId_EmitterSetId ) )
            {
                continue;
            }

            // リソースを差し替え
            if( pDstResource->UnbindResource( dstEmitterSetId ) )
            {
                // TODO:EM4Fに指定エミッタセットがユーザ再生済みエフェクトがバインドが解消されたことを送信する。
                SendUnLinkedEmitterSetV2E( emitterSetName );
                break;
            }
            else
            {
                // 差し替え失敗
                break;
            }
        }
    }

    return true;
}


//---------------------------------------------------------------------------
//  エフェクトプレビューリソースをエフェクトプレビューに設定します。
//---------------------------------------------------------------------------
bool ViewerSystem::SetResEffectPreview( nn::vfx::viewer::detail::Guid prevGuid, detail::ResEffectPreviewPack* pResource ) NN_NOEXCEPT
{
    detail::Preview* pPreview = m_PreviewHead.GetPreview( prevGuid );
    if( !pPreview )
    {
        return false;
    }
    if( pPreview->GetPreviewType() != detail::Preview::VfxPreviewType_EffectPreview )
    {
        return false;
    }

    detail::EffectPreview* pEffectPreview = reinterpret_cast< detail::EffectPreview* >( pPreview );
    pEffectPreview->SetResPreview( &pResource->preview );
    pEffectPreview->SetResEffectPreview( &pResource->effectPreview );

    pEffectPreview->ResetPreview( false );

    return true;
}


//---------------------------------------------------------------------------
//! @brief モデルプレビューリソースをモデルプレビューに設定します。
//---------------------------------------------------------------------------
bool ViewerSystem::SetResModelPreview( nn::vfx::viewer::detail::Guid prevGuid, detail::ResModelPreviewPack* pResource ) NN_NOEXCEPT
{
    detail::Preview* pPreview = m_PreviewHead.GetPreview( prevGuid );
    if( !pPreview )
    {
        return false;
    }
    if( pPreview->GetPreviewType() != detail::Preview::VfxPreviewType_ModelPreview )
    {
        return false;
    }

    detail::ModelPreview* pModelPreview = reinterpret_cast< detail::ModelPreview* >( pPreview );
    pModelPreview->SetResPreview( &pResource->preview );

    return true;
}


//---------------------------------------------------------------------------
//  エフェクトプレビューリソースを更新します。
//---------------------------------------------------------------------------
bool ViewerSystem::UpdateResEffectPreview( nn::vfx::viewer::detail::Guid prevGuid,
    size_t              offset,
    void*               pOverwriteData,
    size_t              overwriteDataSize,
    uint8_t             endian ) NN_NOEXCEPT
{
#if !defined( VFX_BI_ENDIAN )
    NN_UNUSED( endian );
#endif

    detail::Preview* pPreview = m_PreviewHead.GetPreview( prevGuid );
    if( !pPreview )
    {
        return false;
    }
    if( pPreview->GetPreviewType() != detail::Preview::VfxPreviewType_EffectPreview )
    {
        return false;
    }

    // ResEffectPreviewを書き換える
    if( offset < sizeof( detail::ResEffectPreview ) )
    {
        detail::EffectPreview* pEffectPreview = reinterpret_cast< detail::EffectPreview* >( pPreview );
        nn::util::BytePtr ptr( pEffectPreview->GetResEffectPreview() );
        ptr.Advance( offset );
        uint8_t* pDst = reinterpret_cast< uint8_t* >( ptr.Get() );

        // フリップ専用のデータを用意.
        detail::ResEffectPreview temp;
        uint8_t* pSrc = reinterpret_cast< uint8_t* >( &temp ) + offset;

        // 一時データに書き込み
        memcpy( pSrc, pOverwriteData, overwriteDataSize );

#if defined( VFX_BI_ENDIAN )
        if( endian != SystemEndianType )
        {
            // 一時データをフリップ.
            temp.FlipEndian();
        }
#endif

        // フリップしたデータのみをコピる.
        memcpy( pDst, pSrc, overwriteDataSize );
    }
    // ResPreviewを書き換える
    else
    {
        offset -= sizeof( detail::ResEffectPreview );
        nn::util::BytePtr dstPtr( pPreview->GetResPreview() );
        dstPtr.Advance( offset );
        uint8_t* pDst = reinterpret_cast< uint8_t* >( dstPtr.Get() );

        // フリップ専用のデータを用意.
        detail::ResPreview temp;
        nn::util::BytePtr srcPtr( &temp );
        srcPtr.Advance( offset );
        uint8_t* pSrc = reinterpret_cast< uint8_t* >( srcPtr.Get() );

        // 一時データに書き込み
        memcpy( pSrc, pOverwriteData, overwriteDataSize );

#if defined( VFX_BI_ENDIAN )
        if( endian != SystemEndianType )
        {
            // 一時データをフリップ.
            temp.FlipEndian();
        }
#endif

        // フリップしたデータのみをコピる.
        memcpy( pDst, pSrc, overwriteDataSize );
    }

    return true;
}

//---------------------------------------------------------------------------
//  モデルプレビューリソースを更新します。
//---------------------------------------------------------------------------
bool ViewerSystem::UpdateResModelPreview( nn::vfx::viewer::detail::Guid prevGuid,
    size_t          offset,
    void*           pOverwriteData,
    size_t          overwriteDataSize,
    uint8_t         endian ) NN_NOEXCEPT
{
#if !defined( VFX_BI_ENDIAN )
    NN_UNUSED( endian );
#endif

    detail::Preview* pPreview = m_PreviewHead.GetPreview( prevGuid );
    if( !pPreview )
    {
        return false;
    }
    if( pPreview->GetPreviewType() != detail::Preview::VfxPreviewType_ModelPreview )
    {
        return false;
    }

    // ResPreviewを書き換える
    nn::util::BytePtr dstPtr( pPreview->GetResPreview() );
    dstPtr.Advance( offset );
    uint8_t* pDst = reinterpret_cast< uint8_t* >( dstPtr.Get() );

    // フリップ専用のデータを用意.
    detail::ResPreview temp;
    nn::util::BytePtr srcPtr( &temp );
    srcPtr.Advance( offset );
    uint8_t* pSrc = reinterpret_cast< uint8_t* >( srcPtr.Get() );

    // 一時データに書き込み
    memcpy( pSrc, pOverwriteData, overwriteDataSize );

#if defined( VFX_BI_ENDIAN )
    if( endian != SystemEndianType )
    {
        // 一時データをフリップ.
        temp.FlipEndian();
    }
#endif

    // フリップしたデータのみをコピる.
    memcpy( pDst, pSrc, overwriteDataSize );

    return true;
}

//------------------------------------------------
// 指定されたEmitterSetの変数を更新します。
//------------------------------------------------
void ViewerSystem::UpdateAssetEmitterSet( nn::vfx::viewer::detail::Guid resGuid,
    size_t offset, void* pOverwriteData,
    size_t overwriteDataSize, uint8_t endian ) NN_NOEXCEPT
{
    if( endian != SystemEndianType )
    {
        // 通信メッセージとランタイムのエンディアンが異なる
        nn::vfx::detail::OutputError( "EffectMaker Connection Massage Endian Error. \n" );
    }

    //更新対象となるリソースを取得
    detail::ResourceManager::EffectResource* pEffectResource = m_ResManager.GetResource( resGuid );
    if( !pEffectResource )
    {
        return;
    }

    nn::vfx::Resource* pVfxResource = m_pSystem->GetResource( pEffectResource->resourceId );
    nn::vfx::EmitterSetResource* pEmitterSetResource = pVfxResource->GetEmitterSetResource( 0 );

    nn::util::BytePtr ptr( pEmitterSetResource->pResEmitterSet );
    ptr.Advance( offset );
    uint8_t* pDst = reinterpret_cast< uint8_t* >( ptr.Get() );
    memcpy( pDst, pOverwriteData, overwriteDataSize );
}

//------------------------------------------------
// 指定されたemitterの変数を更新します。
//------------------------------------------------
void ViewerSystem::UpdateAssetEmitter( nn::vfx::viewer::detail::Guid resGuid,
    const char* emitterName, size_t offset,
    void* pOverwriteData, size_t overwriteDataSize,
    const bool withReset, uint8_t endian ) NN_NOEXCEPT
{
    if( endian != SystemEndianType )
    {
        // 通信メッセージとランタイムのエンディアンが異なる
        nn::vfx::detail::OutputError( "EffectMaker Connection Massage Endian Error. \n" );
    }

    //再生対象となるリソースを取得
    detail::ResourceManager::EffectResource* pEffectResource = m_ResManager.GetResource( resGuid );
    if( !pEffectResource )
    {
        return;
    }

    nn::vfx::Resource* pVfxResource = m_pSystem->GetResource( pEffectResource->resourceId );
    nn::vfx::EmitterSetResource* pEmitterSetResource = pVfxResource->GetEmitterSetResource( 0 );

    nn::vfx::EmitterResource* pEmitterResource = pEmitterSetResource->pEmitterResource;
    while( pEmitterResource )
    {
        // 同名エミッタがあれば上書き
        if( strcmp( pEmitterResource->m_pResEmitter->name, emitterName ) == 0 )
        {
            nn::util::BytePtr ptr( pEmitterResource->m_pResEmitter );
            ptr.Advance( offset );
            uint8_t* pDst = reinterpret_cast< uint8_t* >( ptr.Get() );
            memcpy( pDst, pOverwriteData, overwriteDataSize );
            m_pSystem->UpdateFromResource( pEmitterResource, withReset );
            pEmitterResource->UpdateParams( pVfxResource );
        }

        // 子を検索して同名エミッタがあれば上書き
        for( int i = 0; i < nn::vfx::SystemParameters_MaxEmitterInclusionCount; i++ )
        {
            if( pEmitterResource->m_ChildEmitterResSet[ i ] )
            {
                if( strcmp( pEmitterResource->m_ChildEmitterResSet[ i ]->m_pResEmitter->name, emitterName ) == 0 )
                {
                    nn::util::BytePtr ptr( pEmitterResource->m_ChildEmitterResSet[ i ]->m_pResEmitter );
                    ptr.Advance( offset );
                    uint8_t* pDst = reinterpret_cast< uint8_t* >( ptr.Get() );
                    memcpy( pDst, pOverwriteData, overwriteDataSize );
                    m_pSystem->UpdateFromResource( pEmitterResource->m_ChildEmitterResSet[ i ], withReset );
                    pEmitterResource->m_ChildEmitterResSet[ i ]->UpdateParams( pVfxResource );
                }
            }
        }

        pEmitterResource = pEmitterResource->m_NextEmitterResource;
    }
}// NOLINT(readability/fn_size)

//------------------------------------------------
//  指定されたエミッタセットのビジビリティを設定します。
//------------------------------------------------
void ViewerSystem::SetEmitterSetVisibility( nn::vfx::viewer::detail::Guid resGuid, bool visibility ) NN_NOEXCEPT
{
    //再生対象となるリソースを取得
    detail::ResourceManager::EffectResource* pEffectResource = m_ResManager.GetResource( resGuid );
    if( !pEffectResource )
    {
        return;
    }
    nn::vfx::Resource* pVfxResource = m_pSystem->GetResource( pEffectResource->resourceId );

    nn::vfx::EmitterSetResource* pEmitterSetResource = pVfxResource->GetEmitterSetResource( 0 );
    pEmitterSetResource->isVisible = visibility;
}

//------------------------------------------------
//  指定されたエミッタのビジビリティを設定します。
//------------------------------------------------
void ViewerSystem::SetEmitterVisibility( nn::vfx::viewer::detail::Guid resGuid, const char* emitterName, bool visibility ) NN_NOEXCEPT
{
    //再生対象となるリソースを取得
    detail::ResourceManager::EffectResource* res = m_ResManager.GetResource( resGuid );
    if( !res )
    {
        return;
    }
    nn::vfx::Resource* vfxRes = m_pSystem->GetResource( res->resourceId );

    nn::vfx::EmitterSetResource* emSetRes = vfxRes->GetEmitterSetResource( 0 );

    for( int i = 0; i < emSetRes->emitterAllCount; i++ )
    {
        if( strcmp( emitterName, emSetRes->pEmitterResource[ i ].m_pResEmitter->name ) == 0 )
        {
            emSetRes->pEmitterResource[ i ].m_IsVisible = visibility;
        }

        // 子エミッタも検索
        for( int j = 0; j < nn::vfx::SystemParameters_MaxEmitterInclusionCount; j++ )
        {
            if( emSetRes->pEmitterResource[ i ].m_ChildEmitterResSet[ j ] )
            {
                if( strcmp( emitterName, emSetRes->pEmitterResource[ i ].m_ChildEmitterResSet[ j ]->m_pResEmitter->name ) == 0 )
                {
                    emSetRes->pEmitterResource[ i ].m_ChildEmitterResSet[ j ]->m_IsVisible = visibility;
                }
            }
        }
    }
}

//---------------------------------------------------------------------------
// エミッタセットを作成します。
//---------------------------------------------------------------------------
void ViewerSystem::CreateViewSysEmitterSet( const char* emitterSetName ) NN_NOEXCEPT
{
    NN_UNUSED( emitterSetName );

    nn::vfx::Handle handle;
    m_pSystem->CreateEmitterSetId( &handle, 0, 0, 0 );
    if( handle.IsValid() )
    {
        nn::vfx::detail::OutputLog( "Succes New EmitterSet.\n" );
    }
}

//------------------------------------------------
//  指定されたプレビューのビジビリティを設定します。
//------------------------------------------------
void ViewerSystem::SetPreviewVisibility( nn::vfx::viewer::detail::Guid prevGuid, bool visibility ) NN_NOEXCEPT
{
    detail::Preview* preview = &m_PreviewHead;

    while( preview )
    {
        if( preview->GetGuid() == prevGuid )
        {
            preview->SetVisible( visibility );
        }
        preview = preview->GetNextPreview();
    }
}

//------------------------------------------------
//  パケットの受信処理。
//------------------------------------------------
void ViewerSystem::ReadingPacketProcess( void* pArg ) NN_NOEXCEPT
{
    static_cast<ViewerSystem*>(pArg)->m_ToolConnector->ThreadToolReceiver( 0, NULL );
}

//---------------------------------------------------------------------------
//  モデル情報更新
//---------------------------------------------------------------------------
// ボーンバッファのサイズを計算
size_t CalculateSizeOfBoneNameBuff( const char* modelName, ModelEnumerator* pVfxModel ) NN_NOEXCEPT
{
    int boneCount = pVfxModel->GetBoneCount( modelName );
    size_t dataBufferSize = 0;
    for( int j = 0; j < boneCount; ++j )
    {
        // ボーン名
        const char* boneName = pVfxModel->GetBoneName( modelName, j );
        size_t nameSize = strlen( boneName );
        // ボーン名サイズ
        dataBufferSize += nameSize;
        dataBufferSize += 1; // "," 分＋１
    }
    dataBufferSize += 1; // "\0" 分＋１
    dataBufferSize += ( 4 - ( dataBufferSize & 0x03 ) ) & 0x03;   // ４バイトのパディング

    return dataBufferSize;
}

// SendModelInfoV2E パケットを生成
void MakePacket_SendModelInfoV2E(
    nn::vfx::viewer::detail::ModelDataMessage* packetBuff,
    size_t boneNameBufSize,
    ModelEnumerator* pModelEnumerator,
    const char* modelName,
    uint8_t* pModelGuid,
    bool isGame ) NN_NOEXCEPT
{
    int boneCount = pModelEnumerator->GetBoneCount( modelName );

    // モデル名を設定
    nn::util::Strlcpy( packetBuff->modelName, modelName, nn::vfx::viewer::detail::MaxModelName );

    // モデルＩＤ(EM4Fから渡されたID)
    memcpy( static_cast< void* >( &packetBuff->modelGuid ), pModelGuid, 16 );

    packetBuff->isGame = ( isGame == true ) ? 1 : 0;

    // ボーン数
    packetBuff->boneCount = boneCount;

    // ボーン名
    packetBuff->boneDataSize = static_cast< uint32_t >( boneNameBufSize );
    char* pBoneName = &packetBuff->boneData;

    size_t bufferSize = boneNameBufSize;

    for( int i = 0; i < boneCount; ++i )
    {
        // ボーン名を追加します。
        const char* boneName = pModelEnumerator->GetBoneName( modelName, i );
        size_t nameSize = strlen( boneName );
        memcpy( pBoneName, boneName, nameSize );
        // 区切り
        pBoneName += nameSize;
        *pBoneName++ = ',';
        bufferSize -= nameSize + 1;
        NN_SDK_ASSERT( bufferSize > 0 );
    }
    *( pBoneName - 1 ) = '\0';
}

//---------------------------------------------------------------------------
//  モデル情報更新
//---------------------------------------------------------------------------
void ViewerSystem::UpdateModelInfo( ModelEnumerator* pModelEnumerator ) NN_NOEXCEPT
{
    // 未接続時は送信しない。
    if( !m_ToolConnector->IsConnected() )
    {
        return;
    }

    if( !pModelEnumerator )
    {
        return;
    }

    // すべてのユーザモデルプレビューを一旦破棄
    {
        detail::Preview* pPreview = &m_PreviewHead;

        while( pPreview )
        {
            if( pPreview->GetPreviewType() == detail::Preview::VfxPreviewType_UserModel )
            {
                detail::Preview* modelPreview = pPreview;
                pPreview = pPreview->GetNextPreview();

                modelPreview->RemovePreview();
                detail::UserModelPreview::DestroyUserModelPreview( m_pViewerHeap, ( detail::UserModelPreview * )modelPreview );
            }
            else
            {
                pPreview = pPreview->GetNextPreview();
            }
        }
    }

    // EffectMaker側をリセット、モデル連携UIをリセット
    SendClearPreviewModelV2E();
    nn::vfx::detail::OutputLog( "Clear Model List \n" );

    // モデルが無い場合はリターンする。
    if ( pModelEnumerator->GetModelCount() == 0)
    {
        return;
    }

    // TODO:バッファサイズの算出方法はこのままかな？
    const size_t BufferSize = 1024 * 1024;
    char* packetBuffer = reinterpret_cast< char* >( m_pViewerHeap->Alloc( BufferSize ) );
    NN_SDK_ASSERT_NOT_NULL( packetBuffer );
    if( !packetBuffer )
    {
        return;
    }

    nn::vfx::viewer::detail::ModelDataMessage*  pPacket = NULL;

    // モデル情報開始メッセージを送信
    SendBeginPreviewModelV2E();

    // モデル数分、モデル情報パケットを生成して送信する
    for( int i = 0; i < pModelEnumerator->GetModelCount(); ++i )
    {
        // モデル名を取得
        const char* modelName = pModelEnumerator->GetModelName( i );
        if( strlen( modelName ) == 0 )
        {
            continue;
        }

        // ボーン数を取得
        int boneCount = pModelEnumerator->GetBoneCount( modelName );
        if( boneCount == 0 )
        {
            continue;
        }

        // ボーン名バッファのサイズを求める
        size_t packetSize = sizeof( nn::vfx::viewer::detail::ModelDataMessage ) + sizeof( nn::vfx::viewer::detail::ModelHeaderMessage );
        size_t boneNameBufSize = CalculateSizeOfBoneNameBuff( modelName, pModelEnumerator );
        packetSize += boneNameBufSize;

        // バッファオーバーチェック
         NN_SDK_ASSERT( packetSize < BufferSize );

        // バッファ確保します。
        void* buf = packetBuffer;
        memset( buf, 0, packetSize );

        // パケットを生成します。
        nn::vfx::viewer::detail::ModelHeaderMessage* header = ( nn::vfx::viewer::detail::ModelHeaderMessage* )packetBuffer;
        header->type = nn::vfx::viewer::detail::ModelMessageType_SendModelInfo;
        header->size = static_cast< uint32_t >( boneNameBufSize + sizeof( nn::vfx::viewer::detail::ModelDataMessage ) );

        pPacket = ( nn::vfx::viewer::detail::ModelDataMessage* )( packetBuffer + sizeof( nn::vfx::viewer::detail::ModelHeaderMessage ) );

        uint8_t modelGuid[ 16 ];
        memset( modelGuid, 0, 16 );
        int modelCountGuid = m_SentModelInfoCount + 16;
        modelGuid[ 0 ] = static_cast< uint8_t >( modelCountGuid );
        modelGuid[ 1 ] = static_cast< uint8_t >( modelCountGuid >> 8 );
        MakePacket_SendModelInfoV2E( pPacket, boneNameBufSize, pModelEnumerator, modelName, modelGuid, true );
        m_SentModelInfoCount++;

        // モデル内の情報をEM4Fに送信
        detail::CommandSender* pCommandSender = m_ToolConnector->GetCommandSender();
        if ( pCommandSender != NULL )
        {
            pCommandSender->SendCommand( nn::vfx::viewer::detail::MessageType_ModelInfo, static_cast< void* >( packetBuffer ), packetSize );
        }

        // モデルプレビューを作成する
        nn::vfx::viewer::detail::UserModelPreview* usrModel =
            nn::vfx::viewer::detail::UserModelPreview::CreateUserModelPreview(m_pViewerHeap, pPacket->modelGuid, modelName, pModelEnumerator);
        if ( usrModel )
        {
            AddPreview( usrModel );
        }
    }

    // モデル情報終了メッセージを送信
    SendEndPreviewModelV2E();

    m_pViewerHeap->Free( packetBuffer );
}


//---------------------------------------------------------------------------
//  モデル情報を追加する
//---------------------------------------------------------------------------
bool ViewerSystem::AddModelInfo( ModelEnumerator* pModelEnumerator, const char* pAddModelName ) NN_NOEXCEPT
{
    uint8_t modelGuid[16];
    memset( modelGuid, 0, 16 );
    int modelCountGuid = m_SentModelInfoCount + 16;
    modelGuid[ 0 ] = static_cast< uint8_t >( modelCountGuid );
    modelGuid[ 1 ] = static_cast< uint8_t >( modelCountGuid >> 8 );
    m_SentModelInfoCount++;

    detail::Guid guid;
    memcpy( static_cast< void* >( &guid ), modelGuid, 16 );

    // モデル情報を送信
    if ( AddModelInfoInternal( pModelEnumerator, pAddModelName, &guid ) == false )
    {
        return false; // 送信がうまくいかなかった場合はfalseで帰る。
    }

    // モデルプレビューを作成する
    nn::vfx::viewer::detail::UserModelPreview* usrModel =
        nn::vfx::viewer::detail::UserModelPreview::CreateUserModelPreview( m_pViewerHeap, guid, pAddModelName, pModelEnumerator );
    if ( usrModel )
    {
        AddPreview( usrModel );
    }

    return true;
}

//---------------------------------------------------------------------------
//  モデル情報を追加する(内部使用)
//---------------------------------------------------------------------------
bool ViewerSystem::AddModelInfoInternal( ModelEnumerator* pModelEnumerator, const char* pAddModelName, detail::Guid* pGuid ) NN_NOEXCEPT
{
    // 未接続時は送信しない。
    if ( !m_ToolConnector->IsConnected() )
    {
        return false;
    }

    // 対象のモデルが存在するか
    int modelIndex = -1;
    for ( int i = 0; i < pModelEnumerator->GetModelCount(); ++i )
    {
        if ( !strcmp( pAddModelName, pModelEnumerator->GetModelName( i ) ) )
        {
            modelIndex = i;
            break;
        }
    }
    if ( modelIndex == -1 ) return false;


    // TODO:バッファサイズの算出方法はこのままかな？
    const size_t BufferSize = 1024 * 1024;
    char* packetBuffer = reinterpret_cast< char* >( m_pViewerHeap->Alloc( BufferSize ) );
    if ( packetBuffer == nullptr )
    {
        return false;
    }
    memset( packetBuffer, 0, BufferSize );

    nn::vfx::viewer::detail::ModelDataMessage*  pPacket = NULL;

    // モデル情報開始メッセージを送信
    SendBeginPreviewModelV2E();

    {
        // ボーン数を取得
        int boneCount = pModelEnumerator->GetBoneCount( pAddModelName );
        if ( boneCount == 0 )
        {
            return false;
        }

        // ボーン名バッファのサイズを求める
        size_t packetSize = sizeof( nn::vfx::viewer::detail::ModelDataMessage ) + sizeof( nn::vfx::viewer::detail::ModelHeaderMessage );
        size_t boneNameBufSize = CalculateSizeOfBoneNameBuff( pAddModelName, pModelEnumerator );
        packetSize += boneNameBufSize;

        // バッファオーバーチェック
        NN_SDK_ASSERT( packetSize < BufferSize );

        // パケットを生成します。
        nn::vfx::viewer::detail::ModelHeaderMessage* header = ( nn::vfx::viewer::detail::ModelHeaderMessage* )packetBuffer;
        header->type = nn::vfx::viewer::detail::ModelMessageType_SendModelInfo;
        header->size = static_cast< uint32_t >( boneNameBufSize + sizeof( nn::vfx::viewer::detail::ModelDataMessage ) );

        pPacket = ( nn::vfx::viewer::detail::ModelDataMessage* )( packetBuffer + sizeof( nn::vfx::viewer::detail::ModelHeaderMessage ) );

        MakePacket_SendModelInfoV2E( pPacket, boneNameBufSize, pModelEnumerator, pAddModelName, reinterpret_cast<uint8_t*>( pGuid ), true );

        // モデル内の情報をEM4Fに送信
        detail::CommandSender* pCommandSender = m_ToolConnector->GetCommandSender();
        if ( pCommandSender != NULL )
        {
            pCommandSender->SendCommand( nn::vfx::viewer::detail::MessageType_ModelInfo,
                static_cast< void* >( packetBuffer ), packetSize );
        }
    }

    // モデル情報終了メッセージを送信
    SendEndPreviewModelV2E();

    m_pViewerHeap->Free( packetBuffer );

    return true;
}

//---------------------------------------------------------------------------
//  モデル情報を削除する
//---------------------------------------------------------------------------
bool ViewerSystem::RemoveModelInfo( const char* pModelName ) NN_NOEXCEPT
{
    // 未接続時は送信しない。
    if ( !m_ToolConnector->IsConnected() )
    {
        return false;
    }

    // 対象のモデルをプレビューから探す。
    detail::Guid guid;
    bool findModel = false;
    detail::Preview* pPreview = &m_PreviewHead;
    while( NULL != pPreview )
    {
        if ( pPreview->GetPreviewType() == nn::vfx::viewer::detail::Preview::VfxPreviewType_UserModel )
        {
            if ( !strcmp( ((nn::vfx::viewer::detail::UserModelPreview*)pPreview)->GetPreivewName().CStr(),  pModelName ) )
            {
                guid = pPreview->GetGuid();
                findModel = true;
            }
        }
        pPreview = pPreview->GetNextPreview();
    }

    if ( findModel == false ) return false;

    // TODO:バッファサイズの算出方法はこのままかな？
    const size_t BufferSize = 1024;
    char* packetBuffer = reinterpret_cast< char* >( m_pViewerHeap->Alloc( BufferSize ) );
    if ( packetBuffer == nullptr )
    {
        return false;
    }
    memset( packetBuffer, 0, BufferSize );

    nn::vfx::viewer::detail::ModelDataMessage*  pPacket = NULL;

    // モデル情報開始メッセージを送信
    SendBeginPreviewModelV2E();

    {
         // ボーン名バッファのサイズを求める
        size_t packetSize = sizeof( nn::vfx::viewer::detail::ModelDataMessage ) + sizeof( nn::vfx::viewer::detail::ModelHeaderMessage );

        // パケットを生成します。
        nn::vfx::viewer::detail::ModelHeaderMessage* header = ( nn::vfx::viewer::detail::ModelHeaderMessage* )packetBuffer;
        header->type = nn::vfx::viewer::detail::ModelMessageType_RemovePreviewModel;
        header->size = static_cast< uint32_t >( sizeof( nn::vfx::viewer::detail::ModelDataMessage ) );

        pPacket = ( nn::vfx::viewer::detail::ModelDataMessage* )( packetBuffer + sizeof( nn::vfx::viewer::detail::ModelHeaderMessage ) );

    // モデル名を設定
        nn::util::Strlcpy( pPacket->modelName, pModelName, nn::vfx::viewer::detail::MaxModelName );

        // モデルＩＤ(EM4Fから渡されたID)
        memcpy( static_cast< void* >( &pPacket->modelGuid ), &guid, 16 );

        pPacket->isGame = true;

        // モデル内の情報をEM4Fに送信
        detail::CommandSender* pCommandSender = m_ToolConnector->GetCommandSender();
        if ( pCommandSender != NULL )
        {
            pCommandSender->SendCommand( nn::vfx::viewer::detail::MessageType_ModelInfo,
                static_cast< void* >( packetBuffer ), packetSize );
        }
    }

    // モデル情報終了メッセージを送信
    SendEndPreviewModelV2E();

    m_pViewerHeap->Free( packetBuffer );

    return true;
}

//-----------------------------------------------------------------------------------------------------
//  バイナリをEffectMakerへ送信します。
//-----------------------------------------------------------------------------------------------------
void ViewerSystem::SendBinaryData(void* pBinaryData, size_t binarySize)
{
    // ヒープからメモリを切り出す
    const size_t BufferSize = sizeof(nn::vfx::viewer::detail::BinaryDataHeaderMessage) + binarySize;
    char* packetBuffer = reinterpret_cast< char* >( m_pViewerHeap->Alloc( BufferSize ) );
    NN_SDK_ASSERT_NOT_NULL( packetBuffer );
    if( !packetBuffer )
    {
        return;
    }

    // パケットを生成します。
    nn::vfx::viewer::detail::BinaryDataHeaderMessage* header = ( nn::vfx::viewer::detail::BinaryDataHeaderMessage* )packetBuffer;
    header->type = 0;
    header->size = static_cast<uint32_t>(binarySize);

    void* pPacket = nullptr;
    pPacket = packetBuffer + sizeof( nn::vfx::viewer::detail::BinaryDataHeaderMessage );
    std::memcpy(pPacket, pBinaryData, binarySize);

    detail::CommandSender* pCommandSender = m_ToolConnector->GetCommandSender();
    if ( pCommandSender != NULL )
    {
        pCommandSender->SendCommand(
            nn::vfx::viewer::detail::MessageType_BinaryData,
            static_cast< void* >( packetBuffer ), sizeof( nn::vfx::viewer::detail::BinaryDataHeaderMessage ) + binarySize);
    }

    m_pViewerHeap->Free( packetBuffer );
}

//-----------------------------------------------------------------------------------------------------
//  エフェクトメーカー操作 共通処理
//-----------------------------------------------------------------------------------------------------
void ViewerSystem::RequestEset( uint32_t type, const char* emitterSetName, const char* duplicateEmitterSetName ) NN_NOEXCEPT
{
    size_t packetSize = sizeof( nn::vfx::viewer::detail::EmitterSetRequestHeaderMessage ) +
        sizeof( nn::vfx::viewer::detail::EmitterSetRequestMessage );
    char* packetBuffer = reinterpret_cast< char* >( m_pViewerHeap->Alloc( packetSize ) );
    NN_SDK_ASSERT_NOT_NULL( packetBuffer );
    if( !packetBuffer )
    {
        return;
    }

    memset( packetBuffer, 0, packetSize );
    nn::vfx::viewer::detail::EmitterSetRequestHeaderMessage* header = ( nn::vfx::viewer::detail::EmitterSetRequestHeaderMessage* )packetBuffer;
    header->type = type;
    header->size = sizeof( nn::vfx::viewer::detail::EmitterSetRequestMessage );

    nn::vfx::viewer::detail::EmitterSetRequestMessage* eset = ( nn::vfx::viewer::detail::EmitterSetRequestMessage* )( reinterpret_cast< uint8_t* >( packetBuffer ) + sizeof( nn::vfx::viewer::detail::EmitterSetRequestHeaderMessage ) );
    memcpy( reinterpret_cast< void* >( eset->emitterSetName ), emitterSetName, strlen( emitterSetName ) );
    if( type == nn::vfx::viewer::detail::RequestEmitterSetMessageType_Duplicate )
    {
        memcpy( reinterpret_cast< void* >( eset->dupEmitterSetName ), duplicateEmitterSetName, strlen( duplicateEmitterSetName ) );
    }

    detail::CommandSender* pCommandSender = m_ToolConnector->GetCommandSender();
    if ( pCommandSender != NULL )
    {
        pCommandSender->SendCommand( nn::vfx::viewer::detail::MessageType_EmitterSetRequest,
            static_cast< void* >( packetBuffer ), packetSize );
    }

    m_pViewerHeap->Free( packetBuffer );
}

//-----------------------------------------------------------------------------------------------------
//  エフェクトメーカー操作 エミッタセットファイルオープン
//-----------------------------------------------------------------------------------------------------
void ViewerSystem::RequestEsetFileOpen( const char* emitterSetFilePath ) NN_NOEXCEPT
{
    RequestEset( nn::vfx::viewer::detail::RequestEmitterSetMessageType_FileOpen, emitterSetFilePath );
}

//-----------------------------------------------------------------------------------------------------
//  エフェクトメーカー操作 エミッタセットファイルクローズ
//-----------------------------------------------------------------------------------------------------
void ViewerSystem::RequestEsetFileClose( const char* emitterSetFilePath ) NN_NOEXCEPT
{
    RequestEset( nn::vfx::viewer::detail::RequestEmitterSetMessageType_Close, emitterSetFilePath );
}

//-----------------------------------------------------------------------------------------------------
//  エフェクトメーカー操作 新規エミッタセット作成
//-----------------------------------------------------------------------------------------------------
void ViewerSystem::RequestEsetFileCreate( const char* emitterSetFilePath ) NN_NOEXCEPT
{
    RequestEset( nn::vfx::viewer::detail::RequestEmitterSetMessageType_Create, emitterSetFilePath );
}

//-----------------------------------------------------------------------------------------------------
//  エフェクトメーカー操作 エミッタセット複製
//-----------------------------------------------------------------------------------------------------
void ViewerSystem::RequestEsetFileDuplicate( const char* srcEmitterSetName, const char* duplicateEmitterSetName ) NN_NOEXCEPT
{
    RequestEset( nn::vfx::viewer::detail::RequestEmitterSetMessageType_Duplicate, srcEmitterSetName, duplicateEmitterSetName );
}

//-----------------------------------------------------------------------------------------------------
//  リンクエミッタセットメッセージを送信する。
//-----------------------------------------------------------------------------------------------------
void ViewerSystem::SendLinkedEmitterSetV2E( const char* emitterSetName ) NN_NOEXCEPT
{
    RequestEset( nn::vfx::viewer::detail::RequestEmitterSetMessageType_LinkEmitterSet, emitterSetName );
}

//-----------------------------------------------------------------------------------------------------
//  アンリンクエミッタセットメッセージを送信する。
//-----------------------------------------------------------------------------------------------------
void ViewerSystem::SendUnLinkedEmitterSetV2E( const char* emitterSetName ) NN_NOEXCEPT
{
    RequestEset( nn::vfx::viewer::detail::RequestEmitterSetMessageType_UnlinkEmitterSet, emitterSetName );
}


} // namespace viewer
} // namespace vfx
} // namespace nn
