﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vfx/vfx_TargetDef.h>
#include <nn/vfx/viewer/vfx_FileSystem.h>
#include <nn/fs.h>

namespace nn {
namespace vfx {
namespace viewer {
namespace detail {

#if defined( NN_BUILD_CONFIG_OS_WIN )
bool IsFullPath( const char* name ) NN_NOEXCEPT
{
    return
        ( ( ( 'a' <= name[ 0 ] && name[ 0 ] <= 'z' ) ||
        ( 'A' <= name[ 0 ] && name[ 0 ] <= 'Z' ) ) &&
        ( name[ 1 ] == ':' ) );
}
#endif

//------------------------------------------------------------------------------
//  ファイル読み込み処理です。
//------------------------------------------------------------------------------
bool FileSystem::Load(
    void**      pOutReturnBuffer,
    size_t*     pOutReturnBufferSize,
    const char* fileName,
    nn::vfx::Heap* pHeap,
    size_t      alignment ) NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_WIN )
    // フルパスかをチェックする
    if( !IsFullPath( fileName ) )
    {
        return false;
    }
#endif

    nn::fs::FileHandle fsHandle;

    // ファイルオープン
    nn::Result ret = nn::fs::OpenFile( &fsHandle, fileName, nn::fs::OpenMode_Read );
    if ( ret.IsFailure() )
    {
        return false;
    }

    // ファイルサイズの取得
    int64_t fsSize;
    nn::fs::GetFileSize( &fsSize, fsHandle );

    // MemoryPool アライン合わせ処理 64Kb Align
    *pOutReturnBufferSize = static_cast<uint32_t>(nn::util::align_up( fsSize, 64 * 1024 ));

    // 読み込みバッファの確保
    *pOutReturnBuffer = pHeap->Alloc( *pOutReturnBufferSize + 4, alignment );

    if ( !(*pOutReturnBuffer) )
    {
        nn::fs::CloseFile( fsHandle );
        NN_SDK_LOG( "[Vfx] File loading failed: %s\n", fileName );
        return false;
    }

    memset( *pOutReturnBuffer, 0, *pOutReturnBufferSize + 4 );
    // ファイル読み込み
    ret = nn::fs::ReadFile( fsHandle, 0, *pOutReturnBuffer, static_cast<size_t>(fsSize) );
    nn::fs::CloseFile( fsHandle );

    if ( ret.IsFailure() )
    {
        return false;
    }

    return true;
}

//---------------------------------------------------------------------------
//! @brief        ファイル書き出し処理です。
//---------------------------------------------------------------------------
void FileSystem::Write( const char* fileName, void* buffer, size_t size, int count, bool isBinary ) NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_WIN )

    // TODO : なるべく : nn対応

    FILE* fp = NULL;
    errno_t error;

    error = fopen_s( &fp, fileName, isBinary ? "wb" : "w" );
    if( error != 0 )
    {
        return;
    }
    else
    {
        fwrite( buffer, size, count, fp );
    }

    fclose( fp );

    return;
#else
    // Windows以外では書き出さない
    return;
#endif
}


} // namespace detail
} // namespace viewer
} // namespace vfx
} // namespace nn
