﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vfx/vfx_Misc.h>
#include <nn/vfx/vfx_MemUtil.h>
#include <nn/vfx/vfx_TemporaryBuffer.h>

namespace nn {
namespace vfx {

//---------------------------------------------------------------------------
//  初期化処理を行います。
//---------------------------------------------------------------------------
bool TemporaryBuffer::Initialize( nn::gfx::Device* pGfxDevice, size_t bufferSize, BufferingMode bufferMode ) NN_NOEXCEPT
{
    m_BufferSide    = BufferSide_FrontBuffer;
    m_BufferMode = bufferMode;

    detail::BufferSizeCalculator bufferSizeCalculator( pGfxDevice, nn::gfx::GpuAccess_ConstantBuffer );
    bufferSizeCalculator.AddBufferSize( bufferSize );

    m_BufferSize = bufferSizeCalculator.GetBufferSize();

    if ( !m_ConstantBuffer[ BufferSide_FrontBuffer ].Initialize( pGfxDevice, detail::GetStaticHeap(), nn::gfx::GpuAccess_ConstantBuffer, m_BufferSize ) )
    {
        return false;
    }

    if ( !m_ConstantBuffer[ BufferSide_BackBuffer  ].Initialize( pGfxDevice, detail::GetStaticHeap(), nn::gfx::GpuAccess_ConstantBuffer, m_BufferSize ) )
    {
        return false;
    }

    // トリプルバッファリングの時は三本目を有効化
    if( m_BufferMode == BufferingMode_TripleBuffering )
    {
        if( !m_ConstantBuffer[ BufferSide_TripleBuffer ].Initialize( pGfxDevice, detail::GetStaticHeap(), nn::gfx::GpuAccess_ConstantBuffer, m_BufferSize ) )
        {
            return false;
        }
    }

    m_IsInitialized = true;

    return true;
}

//---------------------------------------------------------------------------
//  無効化処理を行います。
//---------------------------------------------------------------------------
void TemporaryBuffer::Invalidate() NN_NOEXCEPT
{
    m_BufferSide = BufferSide_FrontBuffer;
    m_AllocatedSize[ BufferSide_FrontBuffer ]  = 0;
    m_AllocatedSize[ BufferSide_BackBuffer ]   = 0;
    m_AllocatedSize[ BufferSide_TripleBuffer ] = 0;
    m_MaxAllocatedSize = 0;
    m_IsInitialized = false;
}

//---------------------------------------------------------------------------
//  終了処理を行います。
//---------------------------------------------------------------------------
void TemporaryBuffer::Finalize( nn::gfx::Device* pGfxDevice ) NN_NOEXCEPT
{
    m_ConstantBuffer[ BufferSide_FrontBuffer ].Finalize( pGfxDevice, detail::GetStaticHeap() );
    m_ConstantBuffer[ BufferSide_BackBuffer  ].Finalize( pGfxDevice, detail::GetStaticHeap() );
    if( m_BufferMode == BufferingMode_TripleBuffering )
    {
        m_ConstantBuffer[ BufferSide_TripleBuffer ].Finalize( pGfxDevice, detail::GetStaticHeap() );
    }
}

//---------------------------------------------------------------------------
//  テンポラリをスワップします。
//---------------------------------------------------------------------------
void TemporaryBuffer::Swap() NN_NOEXCEPT
{
    if( !m_IsInitialized )
    {
        return;
    }

    if( m_MaxAllocatedSize < m_AllocatedSize[ m_BufferSide ] )
    {
        m_MaxAllocatedSize = m_AllocatedSize[ m_BufferSide ];
    }

    m_ConstantBuffer[ m_BufferSide ].End();

    const int maxBufferCount = ( m_BufferMode == BufferingMode_TripleBuffering ) ? 3 : 2;
    m_BufferSide = ( m_BufferSide + 1 ) % maxBufferCount;

    m_ConstantBuffer[ m_BufferSide ].Begin();
    m_AllocatedSize[ m_BufferSide ]    = 0;
}

//---------------------------------------------------------------------------
//  テンポラリバッファからメモリを確保する。
//---------------------------------------------------------------------------
void* TemporaryBuffer::Map( nn::gfx::GpuAddress* gpuAddress, size_t size ) NN_NOEXCEPT
{
    if ( !m_IsInitialized )
    {
        return NULL;
    }

    if ( size == 0 )
    {
        return NULL;
    }

    void* addr = m_ConstantBuffer[ m_BufferSide ].Cut( size );
    if ( !addr )
    {
        Warning( nullptr, detail::RuntimeWarningId_TemporaryBufferAllocationFailed );
        detail::OutputWarning( "TemporaryBuffer Allocation Failed. Allocation size : %d Byte.\n", size );
        detail::OutputWarning( "Current TemporaryBuffer Allocation : %d/%d.\n", m_AllocatedSize[ m_BufferSide ], m_BufferSize );
        return NULL;
    }

    m_ConstantBuffer[ m_BufferSide ].GetGpuAddress( gpuAddress, addr );
    m_AllocatedSize[ m_BufferSide ] += nn::util::align_up( size, m_ConstantBuffer[ m_BufferSide ].GetBufferAlign() );

    return addr;
}

//---------------------------------------------------------------------------
//  テンポラリバッファからメモリを確保されたワークを確定する。
//---------------------------------------------------------------------------
void TemporaryBuffer::Unmap() NN_NOEXCEPT
{
    if( !m_IsInitialized )
    {
        return;
    }
}


} // namespace vfx
} // namespace nn

