﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_BytePtr.h>
#include <nn/vfx/vfx_TargetDef.h>
#include <nn/vfx/vfx_System.h>
#include <nn/vfx/vfx_Misc.h>
#include <nn/vfx/vfx_Random.h>
#include <nn/vfx/vfx_CurlNoiseData.h>
#include <nn/vfx/vfx_MemUtil.h>
#include <nn/vfx/vfx_AreaLoop.h>
#include <nn/vfx/vfx_Callback.h>
#include <nn/vfx/vfx_Stripe.h>
#include <nn/vfx/vfx_SuperStripe.h>
#include <nn/vfx/vfx_StripeConnection.h>

#if !defined( NN_BUILD_CONFIG_OS_COS )
#include <nn/lmem/lmem_FrameHeap.h>
#endif

#include <nn/nn_Middleware.h>
#include <nn/nn_Version.h>

#define NW_MIDDLEWARE_SYMBOL(buildOption) "NintendoWare_Vfx-" NN_MACRO_STRINGIZE(NN_NX_ADDON_VERSION_MAJOR) "_" NN_MACRO_STRINGIZE(NN_NX_ADDON_VERSION_MINOR) "_" NN_MACRO_STRINGIZE(NN_NX_ADDON_VERSION_MICRO) "-" #buildOption

#if defined(NN_SDK_BUILD_DEBUG)
NN_DEFINE_MIDDLEWARE( g_MiddlewareInfo, "Nintendo", NW_MIDDLEWARE_SYMBOL( Debug ) );
#elif defined(NN_SDK_BUILD_DEVELOP)
NN_DEFINE_MIDDLEWARE( g_MiddlewareInfo, "Nintendo", NW_MIDDLEWARE_SYMBOL( Develop ) );
#elif defined(NN_SDK_BUILD_RELEASE)
NN_DEFINE_MIDDLEWARE( g_MiddlewareInfo, "Nintendo", NW_MIDDLEWARE_SYMBOL( Release ) );
#endif

namespace nn {
namespace vfx {

const float ViewParam::DefaultZOffset = 0.01f;  // Z-Fighting抑制のための深度オフセットの規定値

// 32bit float から 24bit float を生成します。
static uint32_t Float32ToBits24( float value ) NN_NOEXCEPT;

bool System::g_IsInitialized = false;

//------------------------------------------------------------------------------
//  エフェクトシステムの生成を行います。
//------------------------------------------------------------------------------
System::System( const Config& config ) NN_NOEXCEPT
{
    // ミドルウェア情報埋め込み
    NN_USING_MIDDLEWARE( g_MiddlewareInfo );

    if( g_IsInitialized )
    {
        VFX_ERROR( "[VFX] Error!! System is Initialized.\n" );
    }
    if( !config.GetEffectHeap() )
    {
        VFX_ERROR( "[VFX] Error!! EffectHeap is NULL.\n" );
    }
    if( !config.GetEffectDynamicHeap() )
    {
        VFX_ERROR( "[VFX] Error!! EffectDynamicHeap is NULL.\n" );
    }

    m_EmitterBuffer = NULL;
    m_EmitterSetArray = NULL;
    m_ResourceArray = NULL;
    m_pComputeShaderEmitterList = NULL;
    m_pComputeShaderEmitterTail = NULL;

    // システム初期化
    Initialize( config.GetEffectHeap(), config.GetEffectDynamicHeap(), config );
}

//---------------------------------------------------------------------------
//  システムの初期化をします。
//---------------------------------------------------------------------------
void System::Initialize( Heap* pStaticHeap, Heap* pDynamicHeap, const Config& config ) NN_NOEXCEPT
{
    m_pDevice = config.GetGfxDevice();
    NN_SDK_ASSERT_NOT_NULL( m_pDevice );

    // パラメータ初期化
    m_UserResourceCount = config.GetResourceCount();
    m_ResourceCount = config.GetResourceCount() + Config::DefaultConfigSettings_ResourceCountForViewer;
    m_EmitterSetCount = config.GetEmitterSetCount();
    m_EmitterCount = config.GetEmitterCount();
    m_ProcessingCountMax = config.GetProcessingCount();
    m_StripeCountMax = config.GetStripeCount();
    m_SuperStripeCountMax = config.GetSuperStripeCount();

    m_StaticHeap.SetHeap( pStaticHeap );
    m_IsEnableTripleBuffer = config.IsEnableTripleBuffer();

    // Misc.cpp の初期化
    detail::SetStaticHeap( &m_StaticHeap );
    detail::SetDynamicHeap( pDynamicHeap );

    detail::BufferAllocator::InitializeArg arg;

    arg.m_pGfxDevice            = m_pDevice;
    arg.m_MemoryPoolProperty    = nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuUncached;
    arg.m_GpuAccessFlag         = nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_UnorderedAccessBuffer | nn::gfx::GpuAccess_IndexBuffer;
    arg.m_ManagementBufferSize  = detail::BufferAllocator::InitializeArg::CalculateManagementBufferSize( config.GetEmitterCount() * 2 );
    m_BufferAllocatorManagementBuffer = pDynamicHeap->Alloc( arg.m_ManagementBufferSize );
    NN_SDK_ASSERT_NOT_NULL( m_BufferAllocatorManagementBuffer );
    arg.m_pManagementBuffer     = m_BufferAllocatorManagementBuffer;

    size_t poolAlign            = detail::BufferAllocator::InitializeArg::CalculateMemoryPoolBufferAlignment( m_pDevice, arg.m_MemoryPoolProperty );
    arg.m_MemoryPoolBufferSize  = detail::BufferAllocator::InitializeArg::CalculateMemoryPoolBufferSize( m_pDevice, arg.m_MemoryPoolProperty, config.GetGpuBufferSize() );
    m_BufferAllocatorMemoryPoolBuffer = pDynamicHeap->Alloc( arg.m_MemoryPoolBufferSize, poolAlign );
    NN_SDK_ASSERT_NOT_NULL( m_BufferAllocatorMemoryPoolBuffer );
    arg.m_pMemoryPoolBuffer     = m_BufferAllocatorMemoryPoolBuffer;

    m_BufferAllocator.Initialize( arg );

    m_BufferMode = m_IsEnableTripleBuffer ? BufferingMode_TripleBuffering : BufferingMode_DoubleBuffering;

    // テクスチャサンプラを初期化
    detail::TextureSampler::InitializeSamplerTable( m_pDevice, &m_StaticHeap );

    {
        uint32_t temp = static_cast< uint32_t >( detail::GetAllocatedSizeFromStaticHeap() );
        detail::InitializeDelayFreeList( m_EmitterCount * 2 ); // 1Gpuエミッタで最大2回動的確保が発生する
        m_DelayFreeWorkSize = detail::GetAllocatedSizeFromStaticHeap() - temp;
    }

    detail::SetSuppressOutputLog( config.IsSuppressionLog() );

    m_EnableCalculation = true;
    m_IsEnableDrawProcess = true;
    m_IsEnableComputeShaderProcess = true;
    m_IsBatchProcessComputeShaderEmitter = config.IsEnableComputeShaderBatchProcess();
    m_EmitterSetIndex = 0;
    m_EmitterIndex = 0;
    m_EmitterSetCreateId = 0;
    m_FreeEmitterCount = m_EmitterCount;
    m_GlobalCounter = 0;
    m_ProcessingGroupId = 0;
    m_ProcessingInfo.Clear();
    m_ParticleSortBufferCount = static_cast< uint32_t >( config.GetParticleSortBufferCount() );
    m_IsEnabledPreciseGpuCounterMode = false;

    memset( m_EmitterSetHead, NULL, sizeof( EmitterSet* ) * SystemParameters_MaxGroupCount );
    memset( m_EmitterSetTail, NULL, sizeof( EmitterSet* ) * SystemParameters_MaxGroupCount );

    memset( m_DelayCreateEmitterSetHead, NULL, sizeof( EmitterSet* ) * SystemParameters_MaxGroupCount );
    memset( m_DelayCreateEmitterSetTail, NULL, sizeof( EmitterSet* ) * SystemParameters_MaxGroupCount );

    m_DelayKillEmitterSetArray      = NULL;
    m_DelayKillEmitterSetAddCount   = 0;

    m_EmitterSetInitializeCallback = NULL;
    m_EmitterSetFinalizeCallback = NULL;
    m_EmitterDrawProfilerCallback = NULL;
    m_EmitterCalculateLodCallback = NULL;
    m_EmitterDrawCullingCallback = NULL;
    m_CustomFieldCallback = NULL;

    for( int i = 0; i < CustomShaderConstantBufferIndex_MaxIndexCount; i++ )
    {
        m_CommonCustomShaderConstantBuffer[ i ] = NULL;
        m_CommonCustomShaderConstantBufferSize[ i ] = 0;
    }

    //---------------------------------------------------------
    // 各種ワーク生成
    //---------------------------------------------------------

    //----------------------------------
    // ワークサイズを計算
    nn::lmem::HeapHandle    frameHeapHandle;
    static const int frameHeapAlignment = 16;

    {
        m_ResourceWorkSize                  = nn::util::align_up( sizeof(Resource*) * m_ResourceCount, frameHeapAlignment );
        m_EmitterSetWorkSize                = nn::util::align_up( sizeof(EmitterSet) * m_EmitterSetCount, frameHeapAlignment );
        m_EmitterSetDelayFreeWorkSize       = nn::util::align_up( sizeof(EmitterSet*) * m_EmitterSetCount, frameHeapAlignment );
        m_EmitterWorkSize                   = nn::util::align_up( sizeof(Emitter) * m_EmitterCount, frameHeapAlignment );
        m_EmitterPtrArrayWorkSize           = nn::util::align_up( sizeof(Emitter*) * m_EmitterCount, frameHeapAlignment );
        m_EmitterCalculationWorkSize        = nn::util::align_up( sizeof(detail::EmitterCalculator), frameHeapAlignment );
        m_ParticleSetSortWorkSize           = nn::util::align_up( sizeof(detail::SortData) * m_ParticleSortBufferCount, frameHeapAlignment ) * m_ProcessingCountMax;
        m_EmitterSetSortWorkSize            = nn::util::align_up( sizeof(SortEmitterSetData) * m_EmitterSetCount, frameHeapAlignment ) * m_ProcessingCountMax;
        m_ProcessingDrawParameterWorkSize   = nn::util::align_up( sizeof( DrawParameter ), frameHeapAlignment ) * m_ProcessingCountMax;
        m_DrawParameterArgWorkSize          = nn::util::align_up( sizeof( DrawParameterArg ), frameHeapAlignment ) * m_ProcessingCountMax;


        size_t totalBufferSize = m_ResourceWorkSize + m_EmitterSetWorkSize + m_EmitterSetDelayFreeWorkSize +
                                 m_EmitterWorkSize + m_EmitterPtrArrayWorkSize + m_EmitterCalculationWorkSize +
                                 m_ParticleSetSortWorkSize + m_EmitterSetSortWorkSize + m_ProcessingDrawParameterWorkSize + m_DrawParameterArgWorkSize;

        totalBufferSize = nn::util::align_up( totalBufferSize, frameHeapAlignment ) + 256;
        m_StaticFrameHeapBuffer = m_StaticHeap.Alloc( totalBufferSize, 256 );
        NN_SDK_ASSERT_NOT_NULL( m_StaticFrameHeapBuffer );
        frameHeapHandle = nn::lmem::CreateFrameHeap( m_StaticFrameHeapBuffer, totalBufferSize, 1 );
    }

    //----------------------------------
    // リソースワーク生成
    {
        m_ResourceWorkSize = sizeof( Resource* ) * ( m_ResourceCount );
        m_ResourceArray = reinterpret_cast< nn::vfx::Resource ** >( nn::lmem::AllocateFromFrameHeap( frameHeapHandle, m_ResourceWorkSize, frameHeapAlignment ) );
        NN_SDK_ASSERT_NOT_NULL( m_ResourceArray );
        memset( m_ResourceArray, 0, m_ResourceWorkSize );
    }

    //----------------------------------
    // エミッタセットワーク生成
    {
        m_EmitterSetArray = reinterpret_cast< nn::vfx::EmitterSet * >( nn::lmem::AllocateFromFrameHeap( frameHeapHandle, m_EmitterSetWorkSize, frameHeapAlignment ) );
        NN_SDK_ASSERT_NOT_NULL( m_EmitterSetArray );
        memset( m_EmitterSetArray, 0, m_EmitterSetWorkSize );

        for( int i = 0; i < m_EmitterSetCount; i++ )
        {
            m_EmitterSetArray[ i ].m_pSystem = this;
            m_EmitterSetArray[ i ].m_IsUsage = false;
        }

        // 遅延エミッタセット削除配列を確保
        size_t delayWorkSize = sizeof( EmitterSet* ) * m_EmitterSetCount;
        m_DelayKillEmitterSetArray = reinterpret_cast< nn::vfx::EmitterSet ** >( nn::lmem::AllocateFromFrameHeap( frameHeapHandle, delayWorkSize, frameHeapAlignment ) );
        NN_SDK_ASSERT_NOT_NULL( m_DelayKillEmitterSetArray );
    }

    //----------------------------------
    // エミッタワーク生成
    {
        const uint32_t blockCount = GetBufferingCount();
        m_EmitterBuffer = reinterpret_cast< nn::vfx::Emitter * >( nn::lmem::AllocateFromFrameHeap( frameHeapHandle, m_EmitterWorkSize, frameHeapAlignment ) );
        NN_SDK_ASSERT_NOT_NULL( m_EmitterBuffer );
        detail::MemUtil::FillZero( m_EmitterBuffer, m_EmitterWorkSize );
        uint8_t* ptrEmrWork = reinterpret_cast< uint8_t * >( m_EmitterBuffer );

        detail::BufferSizeCalculator bufferSizeCalculator( m_pDevice, nn::gfx::GpuAccess_ConstantBuffer );
        bufferSizeCalculator.AddBufferSize( sizeof( detail::EmitterDynamicConstantBuffer ), m_EmitterCount * blockCount );

        bool result = m_EmitterDynamicConstantBuffer.Initialize( m_pDevice, &m_StaticHeap, nn::gfx::GpuAccess_ConstantBuffer, bufferSizeCalculator.GetBufferSize() );
        NN_SDK_ASSERT( result );
        NN_UNUSED( result );
        m_EmitterConstantBufferWorkSize = bufferSizeCalculator.GetBufferSize();

        m_EmitterDynamicConstantBuffer.Begin();
        for( int i = 0; i < m_EmitterCount; i++ )
        {
            Emitter* pEmitter = new ( ptrEmrWork ) Emitter();
            NN_SDK_ASSERT_NOT_NULL( pEmitter );
            pEmitter->GetGfxObjects()->m_DynamicConstantBuffer  = &m_EmitterDynamicConstantBuffer;
            pEmitter->GetGfxObjects()->m_pBufferAllocator       = &m_BufferAllocator;
            pEmitter->GetGfxObjects()->m_DynamicConstantBufferAddr[0] = m_EmitterDynamicConstantBuffer.Cut( sizeof( detail::EmitterDynamicConstantBuffer ) );
            pEmitter->GetGfxObjects()->m_DynamicConstantBufferAddr[1] = m_EmitterDynamicConstantBuffer.Cut( sizeof( detail::EmitterDynamicConstantBuffer ) );
            if( IsEnableTripleBuffer() )
            {
                pEmitter->GetGfxObjects()->m_DynamicConstantBufferAddr[ 2 ] = m_EmitterDynamicConstantBuffer.Cut( sizeof( detail::EmitterDynamicConstantBuffer ) );
            }
            ptrEmrWork += sizeof( Emitter );
        }
        m_EmitterDynamicConstantBuffer.End();
    }

    //----------------------------------
    // エミッタ挙動計算クラス
    void* pEmitterCalculationWork = nn::lmem::AllocateFromFrameHeap( frameHeapHandle, m_EmitterCalculationWorkSize, frameHeapAlignment );
    NN_SDK_ASSERT_NOT_NULL( pEmitterCalculationWork );
    m_pEmitterCalculator = ( detail::EmitterCalculator* ) new ( pEmitterCalculationWork )detail::EmitterCalculator( this );
    NN_SDK_ASSERT_NOT_NULL( m_pEmitterCalculator );

    //----------------------------------
    // 乱数初期化
    {
        size_t temp = static_cast< uint32_t >( detail::GetAllocatedSizeFromStaticHeap() );
        detail::Random::Initialize();
        m_RandomWorkSize = detail::GetAllocatedSizeFromStaticHeap() - temp;
    }

    //----------------------------------
    // カールノイズ初期化
    {
        detail::InitializeCurlNoise( m_pDevice, &m_StaticHeap );
    }


    //----------------------------------
    // コールバックの初期化
    for( int i = 0; i < CallbackId_MaxCallbackIdCount; i++ )
    {
        m_IsEnableCallback[ i ] = false;

        // 現状はエンディアン反転コールバックを強制的に付加。
        m_Callback[ i ].endianFlip = EndianFlipCallbackImpl;

        // 描画設定後コールバックを強制的に付加
        m_Callback[ i ].renderStateSet = BindReservedCustomShaderConstantBuffer;
    }

    //----------------------------------
    // エミッタプラグインコールバックの初期化
    // ストライプ初期化
    size_t stripeBufferSize =  m_StaticHeap.GetAllocatedSize();
    detail::StripeSystem::InitializeSystem( &m_StaticHeap, this, m_BufferMode, m_StripeCountMax );
    detail::ConnectionStripeSystem::InitializeSystem( &m_StaticHeap, this, m_BufferMode );
    detail::SuperStripeSystem::InitializeSystem( &m_StaticHeap, this, m_BufferMode, m_SuperStripeCountMax );
    stripeBufferSize =  m_StaticHeap.GetAllocatedSize() - stripeBufferSize;

    // 範囲内ループ初期化
    detail::AreaLoopSystem::Initialize( this );

    //----------------------------------
    // コアごとの描画パラメータ初期化
    m_ProcessingDrawParameter = reinterpret_cast< DrawParameter* >( nn::lmem::AllocateFromFrameHeap( frameHeapHandle, m_ProcessingDrawParameterWorkSize, frameHeapAlignment ) );
    m_DrawParameterArg = reinterpret_cast< DrawParameterArg* >( nn::lmem::AllocateFromFrameHeap( frameHeapHandle, m_DrawParameterArgWorkSize, frameHeapAlignment ) );

    size_t emitterSetSortBufferSize = sizeof( SortEmitterSetData ) * m_EmitterSetCount;
    size_t tempBufferSize           = config.GetTemporaryBufferSize();
    size_t ptclSortBufferSize       = sizeof( detail::SortData ) * m_ParticleSortBufferCount;

    for ( int i = 0; i < m_ProcessingCountMax; i++ )
    {
        m_ProcessingDrawParameter[i].m_DrawViewFlag                     = DrawViewFlag_All;
        m_ProcessingDrawParameter[i].m_DrawPathFlag                     = 0;
        detail::MemUtil::FillZero( m_ProcessingDrawParameter[i].m_EnableDrawPath, SystemParameters_MaxGroupCount * sizeof( uint32_t ) );
        m_ProcessingDrawParameter[i].m_CurrentShaderType                = ShaderType_Normal;
        // コンストラクタ呼び出し
        new (&m_ProcessingDrawParameter[i].m_DrawTempBuffer) nn::vfx::TemporaryBuffer();
        m_ProcessingDrawParameter[i].m_DrawTempBuffer.Initialize( m_pDevice, tempBufferSize, m_BufferMode );
        NN_SDK_ASSERT( m_ProcessingDrawParameter[i].m_DrawTempBuffer.IsValidate() );
        // m_ProcessingDrawParameter[i].m_ViewParam
        // m_ProcessingDrawParameter[i].m_ViewGpuAddress
        m_ProcessingDrawParameter[i].m_FrameBufferTexture.Invalidate();
        m_ProcessingDrawParameter[i].m_DepthBufferTexture.Invalidate();
        detail::MemUtil::FillZero( m_ProcessingDrawParameter[i].m_RequestFrameBufferTexturePath, SystemParameters_MaxGroupCount * sizeof( uint32_t ) );
        detail::MemUtil::FillZero( m_ProcessingDrawParameter[i].m_RequestDepthBufferTexturePath, SystemParameters_MaxGroupCount * sizeof( uint32_t ) );
        m_ProcessingDrawParameter[i].m_pCurrentCustomShader             = NULL;
        m_ProcessingDrawParameter[i].m_CurrentCustomShaderTextureSlot   = CustomShaderTextureSlotId_0;
        m_ProcessingDrawParameter[i].m_AddedEmittetSetCount             = 0;
        m_ProcessingDrawParameter[ i ].m_SortEmittetSet                 = reinterpret_cast< SortEmitterSetData* >( nn::lmem::AllocateFromFrameHeap( frameHeapHandle, emitterSetSortBufferSize, frameHeapAlignment ) );
        m_ProcessingDrawParameter[ i ].m_pParticleSortBuffer            = reinterpret_cast< detail::SortData* >( nn::lmem::AllocateFromFrameHeap( frameHeapHandle, ptclSortBufferSize, frameHeapAlignment ) );
        m_DrawParameterArg[i].m_ProccesingIndex                         = i;
        m_DrawParameterArg[i].m_pViewParam                              = &m_ProcessingDrawParameter[i].m_ViewParam;
        m_DrawParameterArg[i].m_DrawViewFlag                            = DrawViewFlag_All;
        m_DrawParameterArg[i].m_ShaderType                              = ShaderType_Normal;
        m_DrawParameterArg[i].m_pDrawTempBuffer                         = &m_ProcessingDrawParameter[i].m_DrawTempBuffer;
        m_DrawParameterArg[i].m_pViewGpuAddress                         = &m_ProcessingDrawParameter[i].m_ViewGpuAddress;
        m_DrawParameterArg[i].m_FrameBufferTexture.Invalidate();
        m_DrawParameterArg[i].m_DepthBufferTexture.Invalidate();
        m_DrawParameterArg[i].m_pParticleSortBuffer                     = m_ProcessingDrawParameter[i].m_pParticleSortBuffer;
    }

    nn::lmem::DestroyFrameHeap( frameHeapHandle );

    // 描画パスコールバック初期化
    for( int i = 0; i < DrawPathCallbackId_MaxCallbackId; ++i )
    {
        m_DrawPathCallbackFlag[ i ] = 0;
        m_DrawPathRenderStateSetCallback[ i ] = NULL;
    }

    // メモリの使用状況をダンプします。
    {
        detail::OutputLog( "\n" );
        detail::OutputLog( "System Static     WorkSize : %d \n", m_StaticHeap.GetAllocatedSize() );
        detail::OutputLog( "System Static     AlcCount : %d \n", m_StaticHeap.GetAllocatedCount() );
        detail::OutputLog( "  DelayFree       WorkSize : %d \n", m_DelayFreeWorkSize );
        detail::OutputLog( "  Random          WorkSize : %d \n", m_RandomWorkSize );
        detail::OutputLog( "  Resource        WorkSize : %d \n", m_ResourceWorkSize );
        detail::OutputLog( "  EmitterSet      WorkSize : %d ( EmitterSetNum:%d )\n", m_EmitterSetWorkSize, m_EmitterSetCount );
        detail::OutputLog( "  Emitter         WorkSize : %d ( EmitterNum:%d )\n", m_EmitterWorkSize, m_EmitterCount );
        detail::OutputLog( "  EmitterCalc     WorkSize : %d \n", m_EmitterCalculationWorkSize );
        detail::OutputLog( "  EmitterConstBuf WorkSize : %d \n", m_EmitterConstantBufferWorkSize );
        detail::OutputLog( "  EmitterSetSort  WorkSize : %d \n", m_EmitterSetSortWorkSize );
        detail::OutputLog( "  ParticleSort    WorkSize : %d \n", m_ParticleSetSortWorkSize );
        detail::OutputLog( "  NoiseTexture    WorkSize : %d \n", detail::GetCurlNoiseTextureAllocatedSize() );
        detail::OutputLog( "  TempBuffer      WorkSize : %d \n", tempBufferSize );
        detail::OutputLog( "-------------------------------\n" );
        detail::OutputLog( "Stripe            WorkSize : %d \n", stripeBufferSize );
        //detail::OutputLog( "SuperStripe       WorkSize : %d \n", SuperStripeSystem::GetWorkSize() );
        //detail::OutputLog( "ConnectStripe     WorkSize : %d \n", ConnectionStripeSystem::GetWorkSize() );
        detail::OutputLog("-------------------------------\n" );
    }

}// NOLINT(readability/fn_size)

//------------------------------------------------------------------------------
//  エフェクトシステムの解放を行います。
//------------------------------------------------------------------------------
System::~System() NN_NOEXCEPT
{
    for( int i = 0; i < m_EmitterSetCount; i++ )
    {
        m_EmitterSetArray[ i ].Finalize();
    }
    for( int i = 0; i < m_ResourceCount; i++ )
    {
        if( m_ResourceArray[ i ] )
        {
            m_ResourceArray[ i ]->Finalize( NULL );
            detail::OutputWarning( "vfx system deleted the registered binary. resource id : %d.", i );
        }
    }

    m_BufferAllocator.Finalize( m_pDevice );
    detail::GetDynamicHeap()->Free( m_BufferAllocatorManagementBuffer );
    detail::GetDynamicHeap()->Free( m_BufferAllocatorMemoryPoolBuffer );

    m_EmitterDynamicConstantBuffer.Finalize( m_pDevice, &m_StaticHeap );

    for( int i = 0; i < m_EmitterCount; i++ )
    {
        m_EmitterBuffer[ i ].~Emitter();
    }

    if( m_pEmitterCalculator )
    {
        m_pEmitterCalculator->~EmitterCalculator();
    }

    for ( int i = 0; i < m_ProcessingCountMax; i++ )
    {
        m_ProcessingDrawParameter[i].m_DrawTempBuffer.Finalize( m_pDevice );
    }

    // フレームヒープ用バッファを破棄
    m_StaticHeap.Free( m_StaticFrameHeapBuffer );

    // カールノイズ終了処理
    detail::FinalizeCurlNoise( m_pDevice );

    // ランダム終了処理
    detail::Random::Finalize();
    detail::FinalizeDelayFreeList();

    // テクスチャサンプラを破棄
    detail::TextureSampler::FinalizeSamplerTable( m_pDevice, &m_StaticHeap );

    detail::SetStaticHeap( NULL );
    detail::SetDynamicHeap( NULL );

    // ストライプ終了処理
    detail::StripeSystem::FinalizeSystem( &m_StaticHeap );
    detail::SuperStripeSystem::FinalizeSystem( &m_StaticHeap );
    detail::ConnectionStripeSystem::FinalizeSystem( &m_StaticHeap );
}


//------------------------------------------------------------------------------
//  リソースを登録する
//------------------------------------------------------------------------------
bool System::EntryResource( Heap* pHeap, void* pResource, nn::gfx::MemoryPool* pMemoryPool, size_t memoryPoolSize, size_t memoryPoolOffset, int resourceId, bool shaderDelaySetup, Resource* residentResource ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( pHeap, "[VFX] heap is null." );
    NN_SDK_ASSERT( pResource, "[VFX] bin is null." );
    NN_SDK_ASSERT( static_cast< bool >( resourceId < m_ResourceCount ), "[VFX] resId is over." );

    if ( m_ResourceArray[resourceId] != NULL )
    {
        ClearResource( pHeap, resourceId );
        detail::OutputWarning( "vfx system deleted the registered binary. resource id : %d.\n", resourceId );
    }

    void* ptr = pHeap->Alloc( sizeof( Resource ) );
    if ( !ptr )
    {
        return false;
    }

    m_ResourceArray[resourceId] = new ( ptr )Resource( m_pDevice, pHeap, pResource, pMemoryPool, memoryPoolSize, memoryPoolOffset, resourceId, this, shaderDelaySetup, residentResource );

    return true;
}


bool System::EntryResource( Heap* pHeap, void* pResource, int resourceId, bool shaderDelaySetup, Resource* residentResource ) NN_NOEXCEPT
{
    return EntryResource( pHeap, pResource, nullptr, 0, 0, resourceId, shaderDelaySetup, residentResource );
}


//---------------------------------------------------------------------------
//  指定IDのリソースをエフェクトシステムから破棄します。
//---------------------------------------------------------------------------
void System::ClearResource( Heap* pHeap, int resourceId ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pHeap );

    NN_SDK_ASSERT( ( resourceId < m_ResourceCount ), "[VFX] resId is over." );
    if( !m_ResourceArray[ resourceId ] )
    {
        detail::OutputWarning( "The Resource to be cleared does not exist. ResourceId : %d.\n", resourceId );
        return;
    }

    m_ResourceArray[ resourceId ]->Finalize( pHeap );
    pHeap->Free( m_ResourceArray[ resourceId ] );

    m_ResourceArray[ resourceId ] = NULL;
}

//---------------------------------------------------------------------------
//  エミッタセットを確保します。
//---------------------------------------------------------------------------
EmitterSet* System::AllocEmitterSet() NN_NOEXCEPT
{
    EmitterSet* pEmitterSet = NULL;
    int         i = 0;

    do
    {
        m_EmitterSetIndex++;
        if( m_EmitterSetIndex >= m_EmitterSetCount )
        {
            m_EmitterSetIndex = 0;
        }
        if( !m_EmitterSetArray[ m_EmitterSetIndex ].m_IsUsage )
        {
            pEmitterSet = &m_EmitterSetArray[ m_EmitterSetIndex ];
            break;
        }
    } while( ( ++i ) < m_EmitterSetCount );

    if( pEmitterSet == NULL )
    {
        detail::OutputWarning( "There is no available EmitterSet instance.\n" );
    }

    return pEmitterSet;
}

//---------------------------------------------------------------------------
//  エミッタを確保します。
//---------------------------------------------------------------------------
Emitter* System::AllocEmitter() NN_NOEXCEPT
{
    Emitter* pEmitter = NULL;
    int i = 0;

    do
    {
        m_EmitterIndex++;
        if( m_EmitterIndex >= m_EmitterCount )
        {
            m_EmitterIndex = 0;
        }
        if( !m_EmitterBuffer[ m_EmitterIndex ].IsAlive() )
        {
            pEmitter = &m_EmitterBuffer[ m_EmitterIndex ];
            break;
        }
    } while( ( ++i ) < m_EmitterCount );

    if( pEmitter == NULL )
    {
        detail::OutputWarning( "There is no available Emitter instance.\n" );
        return NULL;
    }

    m_FreeEmitterCount--;

    return pEmitter;
}

//---------------------------------------------------------------------------
//  エミッタを初期化します。
//---------------------------------------------------------------------------
void System::InitializeEmitter( Emitter* pEmitter ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pEmitter );
    pEmitter->m_pEmitterCalculator = m_pEmitterCalculator;
}


//---------------------------------------------------------------------------
//! @brief      エミッタの終了処理をします。
//---------------------------------------------------------------------------
void System::FinalizeEmitter( Emitter* pEmitter ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pEmitter );
    pEmitter->Reset();
    m_FreeEmitterCount++;
}


//---------------------------------------------------------------------------
//  指定IDのエミッタセットを放出します。
//---------------------------------------------------------------------------
bool System::CreateEmitterSetId( Handle* pOutHandle, int emitterSetId, int resourceId, int groupId, int maxParticleCount, Heap* pHeap, bool delay ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pOutHandle );
    Resource* pResource = GetResource( resourceId );
    if( !pResource )
    {
        return false;
    }

    // 生成するエミッタ数
    int emitterCount = pResource->GetEmitterCount( emitterSetId );

    // エミッタ数チェック
    if( emitterCount > m_FreeEmitterCount )
    {
        pOutHandle->Invalidate();
        detail::OutputWarning( "There is no available Emitter instance.\n" );
        return false;
    }

    // エミッタセット確保
    EmitterSet* pEmitterSet = AllocEmitterSet();
    if( !pEmitterSet )
    {
        return false;
    }

    pOutHandle->m_pEmitterSet = pEmitterSet;
    pOutHandle->m_CreateId = m_EmitterSetCreateId++;

    // エミッタセットの初期化
    if( !pEmitterSet->Initialize( emitterSetId, pOutHandle->m_CreateId, resourceId, groupId, maxParticleCount, pHeap ) )
    {
        pEmitterSet->Finalize();
        return false;
    }

    if( !delay )
    {
        // 管理リストへ追加
        AddEmitterSetList( pEmitterSet, groupId );
    }
    else
    {
        // 遅延管理リストへ追加
        m_CreateEmitterSetListMutex.Lock();
        {
            AddDelayCreateEmitterSetList( pEmitterSet, groupId );
        }
        m_CreateEmitterSetListMutex.Unlock();

        // 遅延生成フラグを立てる
        pEmitterSet->m_IsDelayCreate = true;
    }

    return true;
}

//---------------------------------------------------------------------------
//  指定IDのエミッタセットを生成します。
//---------------------------------------------------------------------------
bool System::CreateEmitterSetId( Handle* pOutHandle, int emitterSetId, int resourceId, int groupId, Heap* pHeap, bool delay ) NN_NOEXCEPT
{
    return CreateEmitterSetId( pOutHandle, emitterSetId, resourceId, groupId, 0, pHeap, delay );
}

//---------------------------------------------------------------------------
//  指定IDのエミッタセットをマニュアル放出状態で生成します。
//---------------------------------------------------------------------------
bool System::CreateManualEmitterSetId( Handle* pOutHandle, int emitterSetId, int resourceId, int groupId,
    int maxParticleCount, int maxEmitCountPerFrame, EmitReservationInfo* pEmitReservationListHead,
    Heap* pHeap, CallbackSet* pCustomActionCallbackSet, int residentEmitterTime ) NN_NOEXCEPT
{
    // 事前条件チェック
    NN_SDK_ASSERT_NOT_NULL( pOutHandle );
    NN_SDK_ASSERT_NOT_NULL( pEmitReservationListHead );
    NN_SDK_ASSERT_GREATER_EQUAL( emitterSetId, 0 );
    NN_SDK_ASSERT_GREATER_EQUAL( resourceId, 0 );
    NN_SDK_ASSERT_RANGE( groupId, 0, SystemParameters_MaxGroupCount );
    NN_SDK_ASSERT_GREATER_EQUAL( maxParticleCount, 0 );
    NN_SDK_ASSERT_GREATER_EQUAL( maxEmitCountPerFrame, 0 );
    NN_SDK_ASSERT( util::BytePtr( pEmitReservationListHead ).IsAligned( NN_ALIGNOF( EmitReservationInfo ) ) );

    if ( maxParticleCount < maxEmitCountPerFrame )
    {
        detail::OutputWarning( "maxParticleCount must be larger than maxEmitCountPerFrame.\n" );
        return false;
    }

    // チャイルド付きエミッタセットはサポートしない
    Resource* pResource = GetResource( resourceId );
    if( !pResource )
    {
        detail::OutputWarning( "ResourceID: %d does not exist.\n", resourceId );
        return false;
    }
    if( pResource->IsExistChildEmitter( emitterSetId ) )
    {
        detail::OutputWarning( "Manual EmitterSet must not to have Child Emitter.\n" );
        return false;
    }

    // エミッタセット生成
    bool result = CreateEmitterSetId( pOutHandle, emitterSetId, resourceId, groupId, maxParticleCount, pHeap );
    if( !result || !pOutHandle->IsValid() )
    {
        detail::OutputWarning( "CreateEmitterSetId has failed.\n" );
        return false;
    }

    // EmitterSet が正しく生成できたら、マニュアル放出モードへ移行
    pOutHandle->GetEmitterSet()->m_IsManualEmission = true;
    pOutHandle->GetEmitterSet()->m_pEmitReservationListHead = pEmitReservationListHead;
    pOutHandle->GetEmitterSet()->m_MaxEmitCountPerFrame = maxEmitCountPerFrame;
    pOutHandle->GetEmitterSet()->m_ResidentEmitterTime = residentEmitterTime;

    // コールバックが指定されていればそれに差し替え
    if( pCustomActionCallbackSet )
    {
        pOutHandle->GetEmitterSet()->OverwriteCustomActionCallbackSet( pCustomActionCallbackSet );
    }

    return true;
}

//---------------------------------------------------------------------------
//  指定エミッタセットを再生成します。
//---------------------------------------------------------------------------
bool System::ReCreateEmitterSet( int resourceID, int emitterSetId ) NN_NOEXCEPT
{
    bool ret = false;

    // 指定エミッタセットを検索します。
    for( int groupId = 0; groupId < SystemParameters_MaxGroupCount; groupId++ )
    {
        if( !m_EmitterSetHead[ groupId ] )
        {
            continue;
        }
        EmitterSet* emitterSet = m_EmitterSetHead[ groupId ];

        while( emitterSet )
        {
            EmitterSet* pNext = emitterSet->m_pNextEmitterSet;

            // 同IDのエミッタセットがある場合はResetする。
            if( emitterSet->GetResourceId() == resourceID && emitterSet->GetEmitterSetId() == emitterSetId )
            {
                if( emitterSet->IsFadeRequest() )
                {
                    emitterSet->Kill();
                }
                else
                {
                    emitterSet->Reset();
                    ret = true;
                }
            }

            emitterSet = pNext;
        }
    }

    return ret;
}

//---------------------------------------------------------------------------
//  引数名のエミッタセットを削除します。
//---------------------------------------------------------------------------
void System::RecreateEmitterSet2( const char* emitterSetName, int oldResId, int newResId ) NN_NOEXCEPT
{
    for( int i = 0; i < SystemParameters_MaxGroupCount; i++ )
    {
        EmitterSet* pEmitterSet = m_EmitterSetHead[ i ];

        while( pEmitterSet )
        {
            EmitterSet* pNext = pEmitterSet->GetNext();

            if( strcmp( pEmitterSet->GetName(), emitterSetName ) == 0 && pEmitterSet->m_ResourceId == oldResId )
            {
                pEmitterSet->m_ResourceId = newResId;
                pEmitterSet->Reset();
            }

            pEmitterSet = pNext;
        }
    }
}


//---------------------------------------------------------------------------
//  引数のエミッタセットを削除します。
//---------------------------------------------------------------------------
void System::KillEmitterSet( EmitterSet* pEmitterSet, bool immediate ) NN_NOEXCEPT
{
    if( !pEmitterSet->m_IsUsage )
    {
        detail::OutputWarning( "EmitterSet has been already deleted.\n" );
        return;
    }

    m_KillEmitterSetListMutex.Lock();
    {
        // 既に遅延削除リスト登録済みの場合は削除しない
        for( int i = 0; i < m_DelayKillEmitterSetAddCount; i++ )
        {
            if ( m_DelayKillEmitterSetArray[ i ] == pEmitterSet )
            {
                goto exit;
            }
        }

        // 現状、必ず遅延リストに積まれ、BeginFrame() の時点で削除される。
        NN_UNUSED( immediate );
        //if( immediate )
        //{
        //    pEmitterSet->Finalize();
        //    RemoveEmitterSetList( pEmitterSet );
        //}
        //else
        {
            NN_SDK_ASSERT( m_DelayKillEmitterSetAddCount < m_EmitterSetCount );
            m_DelayKillEmitterSetArray[ m_DelayKillEmitterSetAddCount ] = pEmitterSet;
            m_DelayKillEmitterSetAddCount++;
        }
    }

exit:
    m_KillEmitterSetListMutex.Unlock();
}


//---------------------------------------------------------------------------
//  引数名のエミッタセットを削除します。
//---------------------------------------------------------------------------
void System::KillEmitterSet( const char* emitterSetName, int resId ) NN_NOEXCEPT
{
    for( uint32_t i = 0; i < SystemParameters_MaxGroupCount; i++ )
    {
        EmitterSet* pEmitterSet = m_EmitterSetHead[ i ];

        while( pEmitterSet )
        {
            EmitterSet* pNext = pEmitterSet->GetNext();

            if( strcmp( pEmitterSet->GetName(), emitterSetName ) == 0 && pEmitterSet->GetResourceId() == resId )
            {
                KillEmitterSet( pEmitterSet );
            }

            pEmitterSet = pNext;
        }

        if( m_DelayCreateEmitterSetHead[ i ] )
        {
            pEmitterSet = m_DelayCreateEmitterSetHead[ i ];

            while( pEmitterSet )
            {
                EmitterSet* next = pEmitterSet->GetNext();
                if( strcmp( pEmitterSet->GetName(), emitterSetName ) == 0 && pEmitterSet->GetResourceId() == resId )
                {
                    RemoveDelayCreateEmitterSetList( pEmitterSet );
                    KillEmitterSet( pEmitterSet );
                }
                pEmitterSet = next;
            }
        }
    }
}

//---------------------------------------------------------------------------
//  指定グループに所属するエミッタを削除します。
//---------------------------------------------------------------------------
void System::KillEmitterSetGroup( int groupID ) NN_NOEXCEPT
{
    if( m_EmitterSetHead[ groupID ] )
    {
        EmitterSet* pEmitterSet = m_EmitterSetHead[ groupID ];
        EmitterSet* pNext = NULL;

        while( pEmitterSet )
        {
            pNext = pEmitterSet->GetNext();
            KillEmitterSet( pEmitterSet );
            pEmitterSet = pNext;
        }
    }

    if( m_DelayCreateEmitterSetHead[ groupID ] )
    {
        EmitterSet* pEmitterSet = m_DelayCreateEmitterSetHead[ groupID ];

        while( pEmitterSet )
        {
            EmitterSet* pNext = pEmitterSet->GetNext();
            KillEmitterSet( pEmitterSet );
            pEmitterSet->m_IsDelayCreate = false;
            pEmitterSet = pNext;
        }
    }
}

//---------------------------------------------------------------------------
//  再生中の全てのエミッタセットを削除します。
//---------------------------------------------------------------------------
void System::KillAllEmitterSet() NN_NOEXCEPT
{
    for( int i = 0; i < SystemParameters_MaxGroupCount; i++ )
    {
        KillEmitterSetGroup( i );
    }

    // ストリームアウトエミッタのリストを空に。
    m_pComputeShaderEmitterList = NULL;
    m_pComputeShaderEmitterTail = NULL;
}

//---------------------------------------------------------------------------
//  生成されたエミッタセットを管理リストへ登録します。
//---------------------------------------------------------------------------
void System::AddEmitterSetList( EmitterSet* pEmitterSet, int groupId ) NN_NOEXCEPT
{
    if( m_EmitterSetHead[ groupId ] == NULL )
    {
        m_EmitterSetHead[ groupId ] = pEmitterSet;
        pEmitterSet->m_pPrevEmitterSet = NULL;
        pEmitterSet->m_pNextEmitterSet = NULL;
    }
    else
    {
        m_EmitterSetTail[ groupId ]->m_pNextEmitterSet = pEmitterSet;
        pEmitterSet->m_pPrevEmitterSet = m_EmitterSetTail[ groupId ];
        pEmitterSet->m_pNextEmitterSet = NULL;
    }

    m_EmitterSetTail[ groupId ] = pEmitterSet;
}

//---------------------------------------------------------------------------
//  生成されたエミッタセットを遅延生成管理リストへ登録します。
//---------------------------------------------------------------------------
void System::AddDelayCreateEmitterSetList( EmitterSet* pEmitterSet, int groupId ) NN_NOEXCEPT
{
    m_CreateEmitterSetListMutex.Lock();

    if( m_DelayCreateEmitterSetHead[ groupId ] == NULL )
    {
        m_DelayCreateEmitterSetHead[ groupId ] = pEmitterSet;
        pEmitterSet->m_pPrevEmitterSet = NULL;
        pEmitterSet->m_pNextEmitterSet = NULL;
    }
    else
    {
        m_DelayCreateEmitterSetTail[ groupId ]->m_pNextEmitterSet = pEmitterSet;
        pEmitterSet->m_pPrevEmitterSet = m_DelayCreateEmitterSetTail[ groupId ];
        pEmitterSet->m_pNextEmitterSet = NULL;
    }

    m_DelayCreateEmitterSetTail[ groupId ] = pEmitterSet;

    m_CreateEmitterSetListMutex.Unlock();
}

//---------------------------------------------------------------------------
//  指定エミッタセットを管理リストから削除します。
//---------------------------------------------------------------------------
void System::RemoveEmitterSetList( EmitterSet* pEmitterSet ) NN_NOEXCEPT
{
    if( pEmitterSet->m_pNextEmitterSet == NULL && pEmitterSet->m_pPrevEmitterSet == NULL && pEmitterSet->m_IsDelayCreate )
    {
        // 遅延生成リストから削除
        RemoveDelayCreateEmitterSetList( pEmitterSet );
        return;
    }

    if( pEmitterSet == m_EmitterSetHead[ pEmitterSet->m_GroupId ] )
    {
        m_EmitterSetHead[ pEmitterSet->m_GroupId ] = pEmitterSet->m_pNextEmitterSet;

        if( m_EmitterSetHead[ pEmitterSet->m_GroupId ] != NULL )
        {
            m_EmitterSetHead[ pEmitterSet->m_GroupId ]->m_pPrevEmitterSet = NULL;
        }

        if( pEmitterSet == m_EmitterSetTail[ pEmitterSet->m_GroupId ] )
        {
            m_EmitterSetTail[ pEmitterSet->m_GroupId ] = NULL;
        }
    }
    else
    {
        if( pEmitterSet == m_EmitterSetTail[ pEmitterSet->m_GroupId ] )
        {
            m_EmitterSetTail[ pEmitterSet->m_GroupId ] = pEmitterSet->m_pPrevEmitterSet;
        }

        if( pEmitterSet->m_pNextEmitterSet != NULL )
        {
            pEmitterSet->m_pNextEmitterSet->m_pPrevEmitterSet = pEmitterSet->m_pPrevEmitterSet;
        }

        if( pEmitterSet->m_pPrevEmitterSet != NULL )
        {
            pEmitterSet->m_pPrevEmitterSet->m_pNextEmitterSet = pEmitterSet->m_pNextEmitterSet;
        }
        else
        {
            detail::OutputError( "EmitterSet Remove Failed.\n" );
        }
    }

    pEmitterSet->m_pNextEmitterSet = NULL;
    pEmitterSet->m_pPrevEmitterSet = NULL;
}

//---------------------------------------------------------------------------
//  指定エミッタセットを遅延生成管理リストから削除します。
//---------------------------------------------------------------------------
void System::RemoveDelayCreateEmitterSetList( EmitterSet* pEmitterSet ) NN_NOEXCEPT
{
    if( pEmitterSet == m_DelayCreateEmitterSetHead[ pEmitterSet->m_GroupId ] )
    {
        m_DelayCreateEmitterSetHead[ pEmitterSet->m_GroupId ] = pEmitterSet->m_pNextEmitterSet;

        if( m_DelayCreateEmitterSetHead[ pEmitterSet->m_GroupId ] != NULL )
        {
            m_DelayCreateEmitterSetHead[ pEmitterSet->m_GroupId ]->m_pPrevEmitterSet = NULL;
        }

        if( pEmitterSet == m_DelayCreateEmitterSetTail[ pEmitterSet->m_GroupId ] )
        {
            m_DelayCreateEmitterSetTail[ pEmitterSet->m_GroupId ] = NULL;
        }
    }
    else
    {
        if( pEmitterSet == m_DelayCreateEmitterSetTail[ pEmitterSet->m_GroupId ] )
        {
            m_DelayCreateEmitterSetTail[ pEmitterSet->m_GroupId ] = pEmitterSet->m_pPrevEmitterSet;
        }

        if( pEmitterSet->m_pNextEmitterSet != NULL )
        {
            pEmitterSet->m_pNextEmitterSet->m_pPrevEmitterSet = pEmitterSet->m_pPrevEmitterSet;
        }

        if( pEmitterSet->m_pPrevEmitterSet != NULL )
        {
            pEmitterSet->m_pPrevEmitterSet->m_pNextEmitterSet = pEmitterSet->m_pNextEmitterSet;
        }
        else
        {
            detail::OutputError( "EmitterSet Remove Failed.\n" );
        }
    }

    pEmitterSet->m_IsDelayCreate = false;
    pEmitterSet->m_pNextEmitterSet = NULL;
    pEmitterSet->m_pPrevEmitterSet = NULL;
}


//---------------------------------------------------------------------------
//  内部で確保されたメモリの遅延解放やカウンタ等のリセットを行います。
//---------------------------------------------------------------------------
void System::BeginFrame() NN_NOEXCEPT
{
    if( !m_EnableCalculation )
    {
        return;
    }

    m_GlobalCounter++;

    m_ProcessingInfo.Clear();
    m_ProcessingGroupId = 0;
    m_pComputeShaderEmitterList = NULL;
    m_pComputeShaderEmitterTail = NULL;

    // 遅延解放されるワークの解放を行います。
    detail::FlushDelayFreeList();
    m_BufferAllocator.FlushFreeList();

    for ( int i = 0; i < m_ProcessingCountMax; i++ )
    {
        detail::MemUtil::FillZero( m_ProcessingDrawParameter[ i ].m_EnableDrawPath, SystemParameters_MaxGroupCount * sizeof( uint32_t ) );
        detail::MemUtil::FillZero( m_ProcessingDrawParameter[ i ].m_RequestFrameBufferTexturePath, SystemParameters_MaxGroupCount * sizeof( uint32_t ) );
        detail::MemUtil::FillZero( m_ProcessingDrawParameter[ i ].m_RequestDepthBufferTexturePath, SystemParameters_MaxGroupCount * sizeof( uint32_t ) );
    }

    // 遅延削除指定されたエミッタセットを削除
    m_KillEmitterSetListMutex.Lock();
    {
        if ( m_DelayKillEmitterSetAddCount > 0 )
        {
            for ( int i = 0; i < m_DelayKillEmitterSetAddCount; i++ )
            {
                NN_SDK_ASSERT_NOT_NULL( m_DelayKillEmitterSetArray[ i ] );
                m_DelayKillEmitterSetArray[ i ]->Finalize();
                RemoveEmitterSetList( m_DelayKillEmitterSetArray[ i ] );
                m_DelayKillEmitterSetArray[ i ] = NULL;
            }
        }
        m_DelayKillEmitterSetAddCount = 0;
    }
    m_KillEmitterSetListMutex.Unlock();

    // 遅延生成されたエミッタセットをリストに追加
    bool doClear = false;
    m_CreateEmitterSetListMutex.Lock();
    {
        for( int group = 0; group < SystemParameters_MaxGroupCount; group++ )
        {
            if( m_DelayCreateEmitterSetHead[ group ] )
            {
                EmitterSet* pEmitterSet = m_DelayCreateEmitterSetHead[ group ];

                while( pEmitterSet )
                {
                    EmitterSet* next = pEmitterSet->GetNext();
                    AddEmitterSetList( pEmitterSet, group );
                    pEmitterSet->m_IsDelayCreate = false;
                    pEmitterSet = next;
                }

                doClear = true;
            }
        }
    }
    m_CreateEmitterSetListMutex.Unlock();

    if( doClear )
    {
        detail::MemUtil::FillZero( m_DelayCreateEmitterSetHead, sizeof( EmitterSet* ) * SystemParameters_MaxGroupCount );
        detail::MemUtil::FillZero( m_DelayCreateEmitterSetTail, sizeof( EmitterSet* ) * SystemParameters_MaxGroupCount );
    }
}


//---------------------------------------------------------------------------
//  計算処理を行います。
//---------------------------------------------------------------------------
bool System::Calculate( EmitterSet* pEmitterSet, float frameRate, BufferSwapMode bufferSwapMode ) NN_NOEXCEPT
{
    int processingIndex = 0;
    int groupId         = pEmitterSet->GetGroupId();

    if ( frameRate != 0.0f )
    {
        m_ProcessingGroupId |= static_cast< uint64_t >( static_cast< uint64_t >( 0x1 ) << static_cast< uint64_t >( groupId ) );
    }

    if ( pEmitterSet->IsCalculationEnabled() )
    {
        pEmitterSet->Calculate( frameRate, bufferSwapMode, false, m_EmitterCalculateLodCallback );
    }
    else
    {
        pEmitterSet->Calculate( 0.0f, bufferSwapMode, false, m_EmitterCalculateLodCallback );
    }

    if ( pEmitterSet->IsAlive() )
    {
        m_ProcessingInfo.emitterSetCount++;
        m_ProcessingInfo.emitterCount += pEmitterSet->GetProcessingEmitterCount();
        m_ProcessingInfo.emitterCountSkippedCalculation += pEmitterSet->GetEmitterCountSkippedCalculation();
        m_ProcessingInfo.cpuParticleCount += pEmitterSet->GetProcessingCpuParticleCount();
        m_ProcessingInfo.gpuParticleCount += pEmitterSet->GetProcessingGpuParticleCount();
        m_ProcessingInfo.gpusoParticleCount += pEmitterSet->GetProcessingGpuSoParticleCount();
        m_ProcessingInfo.emitterAnimCount += pEmitterSet->GetProcessingEmitterAnimCount();
        m_ProcessingInfo.cpuEmitterCount += pEmitterSet->GetProcessingCpuEmitterCount();
        m_ProcessingInfo.gpuEmitterCount += pEmitterSet->GetProcessingGpuEmitterCount();
        m_ProcessingInfo.computeEmitterCount += pEmitterSet->GetProcessingComputeEmitterCount();
        m_ProcessingInfo.stripeCount += pEmitterSet->GetProcessingStripeCount();
        m_ProcessingInfo.superStripeCount += pEmitterSet->GetProcessingSuperStripeCount();
        m_ProcessingInfo.allocatedDynamicHeapSize += pEmitterSet->GetAllocatedFromDynamicHeapSize();
        m_ProcessingInfo.connectionStripeCount += pEmitterSet->GetProcessingConnectionStripeCount();
        m_ProcessingInfo.allocatedDynamicHeapSize += pEmitterSet->GetAllocatedFromDynamicHeapSize();

        m_ProcessingDrawParameter[ processingIndex ].m_EnableDrawPath[ groupId ] |= pEmitterSet->GetDrawPath();
        m_ProcessingDrawParameter[ processingIndex ].m_RequestFrameBufferTexturePath[ groupId ] |= pEmitterSet->GetRequestFrameBufferTextureDrawPath();
        m_ProcessingDrawParameter[ processingIndex ].m_RequestDepthBufferTexturePath[ groupId ] |= pEmitterSet->GetRequestDepthBufferTextureDrawPath();
    }
    else
    {
        if ( pEmitterSet->m_IsUsage )
        {
            KillEmitterSet( pEmitterSet );
        }
        else
        {
            // MEMO: 中途半端に消されてしまったエミッタ。ハンドルを確実に無効にするために暫定で CreateId を無効化する。
            detail::OutputWarning( "Invalid EmitterSet has been detected. Invalidates handle." );
            pEmitterSet->m_EmitterSetCreateId = 0xFFFFFFFF;
        }

        return false;
    }

    return true;
}


//---------------------------------------------------------------------------
//  計算処理を行います。
//---------------------------------------------------------------------------
void System::Calculate( int groupId, float frameRate, BufferSwapMode bufferSwapMode ) NN_NOEXCEPT
{
    if( !m_EnableCalculation )
    {
        return;
    }

    if( !m_EmitterSetHead[ groupId ] )
    {
        return;
    }

    EmitterSet* pEmitterSet = m_EmitterSetHead[ groupId ];
    while( pEmitterSet )
    {
        EmitterSet* next = pEmitterSet->GetNext();
        Calculate( pEmitterSet, frameRate, bufferSwapMode );
        pEmitterSet = next;
    }
}

//---------------------------------------------------------------------------
//  GPUキャッシュのフラッシュを行います。
//---------------------------------------------------------------------------
void System::FlushGpuCache() NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_COS )
    GX2Invalidate( static_cast< GX2InvalidateType >(
        GX2_INVALIDATE_ATTRIB_BUFFER | GX2_INVALIDATE_TEXTURE |
        GX2_INVALIDATE_UNIFORM_BLOCK | GX2_INVALIDATE_SHADER ),
        NULL, 0xFFFFFFFF );
#endif
}

//---------------------------------------------------------------------------
//  ビュービューパラメータの設定を行います。
//---------------------------------------------------------------------------
void System::SetViewParam( int processingIndex, ViewParam* pViewParam ) NN_NOEXCEPT
{
    m_ProcessingDrawParameter[ processingIndex ].m_ViewParam = *pViewParam;

    void* viewBuffer = m_ProcessingDrawParameter[ processingIndex ].m_DrawTempBuffer.Map( &m_ProcessingDrawParameter[ processingIndex ].m_ViewGpuAddress, sizeof( ViewParam ) );
    if ( viewBuffer )
    {
        memcpy( viewBuffer, pViewParam, sizeof( ViewParam ) );
        m_ProcessingDrawParameter[ processingIndex ].m_DrawTempBuffer.Unmap();
    }

    m_ProcessingDrawParameter[ processingIndex ].m_IsViewConstantBufferSetuped = false;
}

//---------------------------------------------------------------------------
//  ビュー情報の設定を行います。
//---------------------------------------------------------------------------
void System::SetViewParam( int processingIndex, nn::gfx::CommandBuffer* pCommandBuffer, ViewParam* pViewParam ) NN_NOEXCEPT
{
    m_ProcessingDrawParameter[ processingIndex ].m_ViewParam = *pViewParam;

    void* viewBuffer = m_ProcessingDrawParameter[ processingIndex ].m_DrawTempBuffer.Map( &m_ProcessingDrawParameter[ processingIndex ].m_ViewGpuAddress, sizeof( ViewParam ) );
    if ( viewBuffer )
    {
        memcpy( viewBuffer, pViewParam, sizeof( ViewParam ) );

        // ビューのコンスタントバッファを設定
        pCommandBuffer->SetConstantBuffer( detail::ConstantBufferIndex_View, nn::gfx::ShaderStage_Vertex,  m_ProcessingDrawParameter[ processingIndex ].m_ViewGpuAddress, sizeof( ViewParam ) );
        pCommandBuffer->SetConstantBuffer( detail::ConstantBufferIndex_View, nn::gfx::ShaderStage_Pixel,   m_ProcessingDrawParameter[ processingIndex ].m_ViewGpuAddress, sizeof( ViewParam ) );
        pCommandBuffer->SetConstantBuffer( detail::ConstantBufferIndex_View, nn::gfx::ShaderStage_Compute, m_ProcessingDrawParameter[ processingIndex ].m_ViewGpuAddress, sizeof( ViewParam ) );

        m_ProcessingDrawParameter[ processingIndex ].m_DrawTempBuffer.Unmap();
    }

    m_ProcessingDrawParameter[ processingIndex ].m_IsViewConstantBufferSetuped = true;

    // 共通コンスタントバッファを設定する。
    for ( int i = 0; i < CustomShaderConstantBufferIndex_MaxIndexCount; i++ )
    {
        if ( m_CommonCustomShaderConstantBuffer[ i ] )
        {
            nn::gfx::GpuAddress gpuAddress;

            void* tempBuffer = m_ProcessingDrawParameter[ processingIndex ].m_DrawTempBuffer.Map( &gpuAddress, m_CommonCustomShaderConstantBufferSize[ i ] );
            if ( tempBuffer )
            {
                memcpy( tempBuffer, m_CommonCustomShaderConstantBuffer[ i ], m_CommonCustomShaderConstantBufferSize[ i ] );

                pCommandBuffer->SetConstantBuffer( i + detail::ConstantBufferIndex_CustomShader0, nn::gfx::ShaderStage_Vertex,  gpuAddress, m_CommonCustomShaderConstantBufferSize[ i ] );
                pCommandBuffer->SetConstantBuffer( i + detail::ConstantBufferIndex_CustomShader0, nn::gfx::ShaderStage_Pixel,   gpuAddress, m_CommonCustomShaderConstantBufferSize[ i ] );
                pCommandBuffer->SetConstantBuffer( i + detail::ConstantBufferIndex_CustomShader0, nn::gfx::ShaderStage_Compute, gpuAddress, m_CommonCustomShaderConstantBufferSize[ i ] );

                m_ProcessingDrawParameter[ processingIndex ].m_DrawTempBuffer.Unmap();
            }
        }
    }
}


//---------------------------------------------------------------------------
//  描画処理を行います。
//---------------------------------------------------------------------------
void System::Draw( int processingIndex, nn::gfx::CommandBuffer* pCommandBuffer, int groupId, uint32_t drawPathFlag, bool sort, bool doComputeShaderProcess, void* pUserParam ) NN_NOEXCEPT
{
    if( !m_EmitterSetHead[ groupId ] )
    {
        return;
    }

    EmitterSet* pEmitterSet = m_EmitterSetHead[ groupId ];
    if( !sort )
    {
        // 描画パラメータの取得
        DrawParameterArg* pDrawParameter = GetDrawParameterArg( processingIndex );

        while( pEmitterSet )
        {
            // 描画ビューフラグ判定
            if( pEmitterSet->GetDrawViewFlag() & m_ProcessingDrawParameter[ processingIndex ].m_DrawViewFlag && pEmitterSet->GetDrawPath() & drawPathFlag )
            {
                if( pEmitterSet->IsVisible() )
                {
                    pEmitterSet->Draw( pCommandBuffer,
                                       drawPathFlag,
                                       doComputeShaderProcess,
                                       pUserParam,
                                       pDrawParameter,
                                       m_EmitterDrawCullingCallback,
                                       m_EmitterDrawProfilerCallback );
                }
            }
            pEmitterSet = pEmitterSet->GetNext();
        }
    }
    else
    {
        AddSortBuffer( processingIndex, groupId, drawPathFlag );
        DrawSortBuffer( processingIndex, pCommandBuffer, doComputeShaderProcess, pUserParam );
    }
}

//---------------------------------------------------------------------------
//  エミッタを描画します。
//---------------------------------------------------------------------------
void System::DrawEmitter( nn::gfx::CommandBuffer* pCommandBuffer, Emitter* pEmitter, bool doComputeShaderProcess, void* pUserParam, DrawParameterArg* pDrawParameterArg ) NN_NOEXCEPT
{
    // コンピュートシェーダエミッタで、コンピュートシェーダ一括処理モードではなく、コンピュートシェーダ処理が有効な場合
    if ( ( pEmitter->GetCalculationType() == detail::EmitterCalculationMode_GpuStreamOut ) && !IsBatchProcessComputeShaderEmitter() && m_IsEnableComputeShaderProcess )
    {
        int globalCounter = pEmitter->GetEmitterSet()->m_GlobalCounter;
        NN_SDK_ASSERT_NOT_NULL( pEmitter->GetEmitterResource()->m_pComputeShader );
        pEmitter->m_pEmitterCalculator->CalculateComputeShader( pCommandBuffer, pEmitter, pEmitter->GetEmitterResource()->m_pComputeShader,
                                                                globalCounter, doComputeShaderProcess, pUserParam, true );
    }
    if (m_IsEnableDrawProcess)
    {
        pEmitter->m_pEmitterCalculator->Draw(pCommandBuffer, pEmitter, pUserParam, pDrawParameterArg);
    }
}

//------------------------------------------------------------------------------
//  描画を行うIDグループをソートバッファに追加します。
//------------------------------------------------------------------------------
void System::AddSortBuffer( int processingIndex, int groupId, uint32_t drawPathFlag ) NN_NOEXCEPT
{
    EmitterSet* emitterSet = m_EmitterSetHead[ groupId ];
    if( !emitterSet )
    {
        return;
    }

    DrawParameter* pDrawParameter = &m_ProcessingDrawParameter[ processingIndex ];
    const nn::util::Float4x4& viewMatrix = pDrawParameter->m_ViewParam.viewMatrix;

    while( emitterSet )
    {
        // 描画ビューフラグ判定
        if( ( emitterSet->GetDrawViewFlag() & pDrawParameter->m_DrawViewFlag ) &&
            ( emitterSet->GetDrawPath() & drawPathFlag ) )
        {
            if( emitterSet->IsVisible() )
            {
                nn::util::Vector3fType ePos;
                emitterSet->GetPos( &ePos );

                pDrawParameter->m_SortEmittetSet[ pDrawParameter->m_AddedEmittetSetCount ].pEmitterSet = emitterSet;
                const nn::util::Vector3fType z = NN_UTIL_VECTOR_3F_INITIALIZER( viewMatrix.m[ 2 ][ 0 ], viewMatrix.m[ 2 ][ 1 ], viewMatrix.m[ 2 ][ 2 ] );
                const float viewZ = nn::util::VectorDot( z, ePos ) + viewMatrix.m[ 2 ][ 3 ];
                pDrawParameter->m_SortEmittetSet[ pDrawParameter->m_AddedEmittetSetCount ].param = ( ( ( emitterSet->GetDrawPriority() ) << 24 ) | Float32ToBits24( viewZ ) );
                pDrawParameter->m_AddedEmittetSetCount++;
            }
        }

        emitterSet = emitterSet->GetNext();
    }

    pDrawParameter->m_DrawPathFlag |= drawPathFlag;
}

//------------------------------------------------------------------------------
//  ソートバッファの描画処理を行います。
//------------------------------------------------------------------------------
void System::DrawSortBuffer( int processingIndex, nn::gfx::CommandBuffer* pCommandBuffer, bool doComputeShaderProcess, void* pUserParam ) NN_NOEXCEPT
{
    DrawParameter* drawParameter = &m_ProcessingDrawParameter[ processingIndex ];

    if( drawParameter->m_AddedEmittetSetCount == 0 )
    {
        drawParameter->m_DrawPathFlag = 0;
        return;
    }

    // エミッタセットソート
    std::sort( drawParameter->m_SortEmittetSet, drawParameter->m_SortEmittetSet + drawParameter->m_AddedEmittetSetCount, detail::SortCompareLessUInt<SortEmitterSetData>() );

    // 描画パラメータの取得
    DrawParameterArg* pDrawParameter = GetDrawParameterArg( processingIndex );

    // ソートした結果を描画
    for( int i = 0; i < drawParameter->m_AddedEmittetSetCount; i++ )
    {
        EmitterSet* emitterSet = drawParameter->m_SortEmittetSet[ i ].pEmitterSet;
        emitterSet->Draw( pCommandBuffer,
                          drawParameter->m_DrawPathFlag,
                          doComputeShaderProcess,
                          pUserParam,
                          pDrawParameter,
                          m_EmitterDrawCullingCallback,
                          m_EmitterDrawProfilerCallback );
    }

    // ソートバッファをリセット
    drawParameter->m_AddedEmittetSetCount = 0;
    drawParameter->m_DrawPathFlag         = 0;
}

//---------------------------------------------------------------------------
//  描画処理用テンポラリバッファをスワップします。
//---------------------------------------------------------------------------
void System::SwapBuffer() NN_NOEXCEPT
{
    for ( int i = 0; i < m_ProcessingCountMax; i++ )
    {
        m_ProcessingDrawParameter[ i ].m_DrawTempBuffer.Swap();
    }
}

//---------------------------------------------------------------------------
//  描画処理用テンポラリバッファからメモリを確保します。
//---------------------------------------------------------------------------
void* System::AllocFromTempBuffer( int processingIndex, nn::gfx::GpuAddress* address, size_t size ) NN_NOEXCEPT
{
    if ( processingIndex >= m_ProcessingCountMax ) return NULL;
    return m_ProcessingDrawParameter[ processingIndex ].m_DrawTempBuffer.Map( address, size );
}

//---------------------------------------------------------------------------
//  描画処理用テンポラリバッファのキャッシュフラッシュを行います。
//---------------------------------------------------------------------------
void System::FlushTempBuffer() NN_NOEXCEPT {}

//---------------------------------------------------------------------------
//  リソース更新に伴うアップデートを行います。
//---------------------------------------------------------------------------
void System::UpdateFromResource( EmitterResource* pEmitterResource, const bool withReset ) NN_NOEXCEPT
{
    // RenderStateオブジェクトの更新
    if (pEmitterResource)
    {
        // Gfxオブジェクトの再構築
        pEmitterResource->FinalizeRenderState(m_pDevice);
        pEmitterResource->InitializeRenderState(m_pDevice);

        // チャイルドエミッタもトラバースする
        for (int i = 0; i < pEmitterResource->m_ChildEmitterResCount; i++)
        {
            if (pEmitterResource->m_ChildEmitterResSet[i])
            {
                pEmitterResource->m_ChildEmitterResSet[i]->FinalizeRenderState(m_pDevice);
                pEmitterResource->m_ChildEmitterResSet[i]->InitializeRenderState(m_pDevice);
            }
        }
    }

    for( int i = 0; i < SystemParameters_MaxGroupCount; i++ )
    {
        if( m_EmitterSetHead[ i ] )
        {
            EmitterSet* pEmitterSet = m_EmitterSetHead[ i ];

            while( pEmitterSet )
            {
                if( withReset )
                {
                    // リセット指定の場合、リセットしてメモリサイズなどの再計算。
                    pEmitterSet->Reset();
                }
                else
                {
                    // そうでない場合、リソースのアップデート。
                    bool result = pEmitterSet->UpdateFromResource( pEmitterResource );
                    if( !result )
                    {
                        // リソース更新に失敗した場合、エミッタセットを消しておく（即時）
                        EmitterSet* pFailedEmitterSet = pEmitterSet;
                        pEmitterSet = pEmitterSet->m_pNextEmitterSet;
                        KillEmitterSet( pFailedEmitterSet, true );
                        continue;
                    }
                }

                pEmitterSet = pEmitterSet->m_pNextEmitterSet;
            }
        }
    }
}

//---------------------------------------------------------------------------
//! @brief      エミッタ情報をダンプする
//---------------------------------------------------------------------------
void DumpEmitterInformation( Emitter* pEmitter, const int debugLevel ) NN_NOEXCEPT
{
    // デバッグレベルに応じて、出す情報量を調整する。
    switch( debugLevel )
    {
    case 0:
        detail::OutputWarning( "  %s - %s\n", pEmitter->GetEmitterSet()->GetName(), pEmitter->GetName() );
        break;
    case 1:
    default:
        detail::OutputWarning( "  %s-%s-GroupId:%d\n", pEmitter->GetEmitterSet()->GetName(), pEmitter->GetName(), pEmitter->GetEmitterSet()->GetGroupId() );
        break;
    }
}

//------------------------------------------------------------------------------
//  32bit float から 24bit float を生成します。
//------------------------------------------------------------------------------
uint32_t Float32ToBits24( float value ) NN_NOEXCEPT
{
    enum
    {
        SIGN32 = 0x80000000,
        SIGN24 = 0x00800000,

        EXP_BIAS32 = 127,
        EXP_BIAS24 = 63,
        EXP_MASK32 = 0x7F800000,
        EXP_MASK24 = 0x007F0000,

        FRACTION_WIDTH32 = 23,
        FRACTION_MASK32 = 0x007FFFFF,
        FRACTION_WIDTH24 = 16,
        FRACTION_MASK24 = 0x0000FFFF
    };

    uint32_t bits32 = *reinterpret_cast< uint32_t* >( &value );

    uint32_t sign = bits32 & SIGN32;
    int exp = static_cast< int >( ( bits32 & EXP_MASK32 ) >> FRACTION_WIDTH32 );
    uint32_t fraction = bits32 & FRACTION_MASK32;

    uint32_t bits24 = 0;
    bits24 |= ( sign != 0 ) ? SIGN24 : 0;

    if( ( bits32 & ~SIGN32 ) == 0 )
    {
        exp = 0;
    }
    else
    {
        exp = exp - EXP_BIAS32 + EXP_BIAS24;
    }

    fraction = fraction >> ( FRACTION_WIDTH32 - FRACTION_WIDTH24 );

    if( exp < 0 )
    {
        // +0 もしくは -0 なのでそのまま。
    }
    else if( exp > 127 )
    {
        // 無限大の処理
        // TODO: IEEE float の無限大の表現がGPU上で有効なのかどうかを要確認。
        bits24 = static_cast< uint32_t >( 0x7F ) << FRACTION_WIDTH24;
    }
    else
    {
        bits24 |= fraction & FRACTION_MASK24;
        bits24 |= ( static_cast< uint32_t >( exp )& 0x7F ) << FRACTION_WIDTH24;
    }

    return bits24;
}

//---------------------------------------------------------------------------
//  エミッタセットIDを検索します。
//---------------------------------------------------------------------------
int System::SearchEmitterSetId( const char* emitterSetName, int resId ) const NN_NOEXCEPT
{
    const Resource* pResource = GetResource( resId );
    if( pResource )
    {
        return pResource->SearchEmitterSetId( emitterSetName );
    }

    return InvalidValueId_EmitterSetId;
}

//---------------------------------------------------------------------------
//  エミッタセット名を検索します。
//---------------------------------------------------------------------------
const char* System::SearchEmitterSetName( int emitterSetId, int resId ) const NN_NOEXCEPT
{
    const Resource* pResource = GetResource( resId );
    if( pResource )
    {
        return pResource->GetEmitterSetName( emitterSetId );
    }

    return NULL;
}


//---------------------------------------------------------------------------
//  描画パス 描画設定コールバックを設定します。
//---------------------------------------------------------------------------
void System::SetDrawPathRenderStateSetCallback( DrawPathCallbackId id, DrawPathFlag drawPathFlag, DrawPathRenderStateSetCallback callback ) NN_NOEXCEPT
{
    if( id < DrawPathCallbackId_MaxCallbackId )
    {
        m_DrawPathCallbackFlag[ id ] = drawPathFlag;
        m_DrawPathRenderStateSetCallback[ id ] = callback;
    }
}

//---------------------------------------------------------------------------
//  描画パス 描画設定コールバックを設定します。
//---------------------------------------------------------------------------
void System::SetDrawPathRenderStateSetCallback( DrawPathFlag drawPathFlag, DrawPathRenderStateSetCallback callback ) NN_NOEXCEPT
{
    m_DrawPathCallbackFlag[ DrawPathCallbackId_0 ] = drawPathFlag;
    m_DrawPathRenderStateSetCallback[ DrawPathCallbackId_0 ] = callback;
}

//---------------------------------------------------------------------------
//  描画パス 描画設定コールバックを取得します。
//---------------------------------------------------------------------------
DrawPathRenderStateSetCallback System::GetDrawPathRenderStateSetCallback( DrawPathFlag drawPathFlag ) NN_NOEXCEPT
{
    if( drawPathFlag == 0 )
    {
        return NULL;
    }

    for( int i = 0; i < DrawPathCallbackId_MaxCallbackId; ++i )
    {
        if( drawPathFlag & m_DrawPathCallbackFlag[ i ] )
        {
            return m_DrawPathRenderStateSetCallback[ i ];
        }
    }

    return NULL;
}

//---------------------------------------------------------------------------
//  ストリームアウトエミッタを一時リストに追加します。
//---------------------------------------------------------------------------
void System::AddComputeShaderEmitterList( Emitter* pEmitter ) NN_NOEXCEPT
{
    Emitter* p = m_pComputeShaderEmitterList;
    while ( p )
    {
        if ( p == pEmitter )
        {
            // 同じエミッタがすでに登録済みの場合は追加しない（リストが壊れる）
            return;
        }
        p = p->m_pNextUseComputeShaderEmitter;
    }

    if ( !m_pComputeShaderEmitterList )
    {
        m_pComputeShaderEmitterList = pEmitter;
        m_pComputeShaderEmitterTail = m_pComputeShaderEmitterList;
        m_pComputeShaderEmitterTail->m_pNextUseComputeShaderEmitter = NULL;
    }
    else
    {
        m_pComputeShaderEmitterTail->m_pNextUseComputeShaderEmitter = pEmitter;
        m_pComputeShaderEmitterTail = pEmitter;
        m_pComputeShaderEmitterTail->m_pNextUseComputeShaderEmitter = NULL;
    }
}

//---------------------------------------------------------------------------
//  ストリームアウトエミッタリストを処理します。
//---------------------------------------------------------------------------
void System::BatchCalculationComputeShaderEmitter( nn::gfx::CommandBuffer* pCommandBuffer, void* pUserParam, uint32_t processEmitterFlag ) NN_NOEXCEPT
{
    if( !m_pComputeShaderEmitterList )
    {
        return;
    }

    if ( !m_IsEnableComputeShaderProcess )
    {
        return;
    }

    Emitter* pComputeShaderEmitter = m_pComputeShaderEmitterList;

    while( pComputeShaderEmitter )
    {
        if ( pComputeShaderEmitter->GetEmitterSet() &&
             ( pComputeShaderEmitter->GetProcessEmitterFlag() & processEmitterFlag ) != 0 &&
             pComputeShaderEmitter->m_pEmitterData->emitter.calcType == detail::EmitterCalculationMode_GpuStreamOut )
        {
            int globalCounter = pComputeShaderEmitter->GetEmitterSet()->m_GlobalCounter;
            NN_SDK_ASSERT_NOT_NULL( pComputeShaderEmitter->GetEmitterResource()->m_pComputeShader );
            pComputeShaderEmitter->m_pEmitterCalculator->CalculateComputeShader( pCommandBuffer, pComputeShaderEmitter,
                        pComputeShaderEmitter->GetEmitterResource()->m_pComputeShader, globalCounter, true, pUserParam, false );
        }
        pComputeShaderEmitter = pComputeShaderEmitter->m_pNextUseComputeShaderEmitter;
    }

    pCommandBuffer->FlushMemory( nn::gfx::GpuAccess_UnorderedAccessBuffer );
}

//---------------------------------------------------------------------------
//  パーティクルのソートを行い、その結果の配列の先頭を返します。
//---------------------------------------------------------------------------
bool System::GetSortedParticleList( detail::SortData** pOutSortedList, int* pOutParticleCount, Emitter* pEmitter, detail::ParticleSortType sortType, float baseEmitterTime, int processingIndex /*= 0 */ ) const NN_NOEXCEPT
{
    // 放出中のパーティクル数がバッファサイズ（コンフィグ指定）を超えた場合
    if( pEmitter->m_ParticleCount > m_ParticleSortBufferCount )
    {
        // 警告を出して失敗させる
        detail::OutputWarning( "Particle sort has failed. More buffer size is needed. \n" );
        *pOutSortedList = NULL;
        *pOutParticleCount = 0;
        return false;
    }

    *pOutSortedList     = m_ProcessingDrawParameter[ processingIndex ].m_pParticleSortBuffer;  // 先頭アドレスを取得。前回結果が入っている可能性がある。
    *pOutParticleCount  = 0;

    // ソートタイプで分岐。現状は昇順／降順／Zソートに対応
    // 現状、m_ParticlePropertyFront を参照しつつソートを行うので、
    // GPUパーティクルの場合、unchaced 領域を利用する為CPUコストが高くなる
    detail::ParticleProperty*   pParticleProperty = pEmitter->m_ParticlePropertyFront;
    if( pEmitter->m_pEmitterData->emitter.calcType == detail::EmitterCalculationMode_Cpu )
    {
        pParticleProperty   = &pEmitter->m_CpuParticleProperty;
    }

    switch( sortType )
    {
    case detail::ParticleSortType_AscendingOrder:  // 昇順（年寄り順）
    {
        for( int i = 0; i < pEmitter->m_ParticleCount; i++ )
        {
            if( pEmitter->m_pParticleData[ i ].IsAlive( baseEmitterTime ) )
            {
                ( *pOutSortedList )[ *pOutParticleCount ].param = pParticleProperty->localVec[ i ].w; // 生成時刻
                ( *pOutSortedList )[ *pOutParticleCount ].index = i;
                ( *pOutParticleCount )++;
            }
        }
        std::sort( *pOutSortedList, *pOutSortedList + *pOutParticleCount, detail::SortCompareGreaterIndexStable<detail::SortData>() );
    }
        break;
    case detail::ParticleSortType_DescendingOrder: // 降順（若い順）
    {
        for( int i = 0; i < pEmitter->m_ParticleCount; i++ )
        {
            if( pEmitter->m_pParticleData[ i ].IsAlive( baseEmitterTime ) )
            {
                ( *pOutSortedList )[ *pOutParticleCount ].param = pParticleProperty->localVec[ i ].w; // 生成時刻
                ( *pOutSortedList )[ *pOutParticleCount ].index = i;
                ( *pOutParticleCount )++;
            }
        }
        std::sort( *pOutSortedList, *pOutSortedList + *pOutParticleCount, detail::SortCompareLessIndexStable<detail::SortData>() );
    }
        break;
    case detail::ParticleSortType_ZSort:          // Zソート
    {
        ViewParam* pViewParam = &m_ProcessingDrawParameter[ processingIndex ].m_ViewParam;

        for( int i = 0; i < pEmitter->m_ParticleCount; i++ )
        {
            if( pEmitter->m_pParticleData[ i ].IsAlive( baseEmitterTime ) )
            {
                nn::util::Vector3fType worldPos;
                nn::util::Vector3fType localPos;
                nn::util::VectorLoad( &localPos, pParticleProperty->localPos[ i ].v );
                pEmitter->TransformToWorldPos( &worldPos, localPos, i );

                ( *pOutSortedList )[ *pOutParticleCount ].param =
                    pViewParam->viewMatrix.m[ 2 ][ 0 ] * nn::util::VectorGetX( worldPos ) +
                    pViewParam->viewMatrix.m[ 2 ][ 1 ] * nn::util::VectorGetY( worldPos ) +
                    pViewParam->viewMatrix.m[ 2 ][ 2 ] * nn::util::VectorGetZ( worldPos ) +
                    pViewParam->viewMatrix.m[ 2 ][ 3 ];
                ( *pOutSortedList )[ *pOutParticleCount ].index = i;
                ( *pOutParticleCount )++;
            }
        }

        if (pEmitter->GetEmitterResource()->m_pResEmitter->renderState.isDepthMask)
        {
            std::sort(*pOutSortedList, *pOutSortedList + *pOutParticleCount, detail::SortCompareViewInvZ<detail::SortData>());
        }
        else
        {
            std::sort(*pOutSortedList, *pOutSortedList + *pOutParticleCount, detail::SortCompareViewZ<detail::SortData>());
        }
    }
        break;
    default:
        // 不明なソートタイプ
        NN_SDK_ASSERT( 0 );
        break;
    }

    return true;
}

//---------------------------------------------------------------------------
//  描画前コールバック（RenderStateCallback、DrawPathCallback）を呼び出します。
//---------------------------------------------------------------------------
bool System::InvokeBeforeRenderCallbacks( nn::gfx::CommandBuffer* pCommandBuffer, Emitter* pEmitter, const ShaderType shaderType, void* pUserParam, DrawParameterArg* pDrawParameterArg ) NN_NOEXCEPT
{
    RenderStateSetArg rssArg;
    rssArg.pCommandBuffer = pCommandBuffer;
    rssArg.pEmitter = pEmitter;
    rssArg.pUserParam = pUserParam;
    rssArg.shaderType = shaderType;
    rssArg.isComputeShaderMode = ( shaderType == nn::vfx::ShaderType_MaxShaderType ? true : false );    // ComputeShader の時は shaderType == nn::vfx::ShaderType_MaxShaderType
    rssArg.pDrawParameterArg = pDrawParameterArg;

    ResetCustomShaderSetting( pDrawParameterArg->m_ProccesingIndex, pEmitter->GetShader( shaderType ) );
    {
        // 描画設定後コールバックの呼び出し
        bool ret = InvokeRenderStateSetCallback( rssArg );
        if( !ret )
        {
            return false;
        }

        // DrawPath コールバックの呼び出し
        if( pEmitter->m_DrawPathCallback )
        {
            pEmitter->m_DrawPathCallback( rssArg );
        }
    }
    ResetCustomShaderSetting( pDrawParameterArg->m_ProccesingIndex, NULL );

    return true;
}

//---------------------------------------------------------------------------
//  カスタムシェーダテクスチャをバインドします。
//---------------------------------------------------------------------------
bool System::BindCustomShaderTexture( int processingIndex, nn::gfx::CommandBuffer* pCommandBuffer, CustomShaderTextureType customShaderType, nn::gfx::DescriptorSlot textureSlot ) NN_NOEXCEPT
{
    if( !m_ProcessingDrawParameter[ processingIndex ].m_pCurrentCustomShader )
    {
        return false;
    }

    int vertexLoc = m_ProcessingDrawParameter[ processingIndex ].m_pCurrentCustomShader->GetCustomTextureVertexSamplerLocation( customShaderType );
    int pixelLoc  = m_ProcessingDrawParameter[ processingIndex ].m_pCurrentCustomShader->GetCustomTexturePixelSamplerLocation( customShaderType );

    // 標準的なサンプラを取得して設定する
    vfx::detail::ResTextureSampler resSampler;
    resSampler.filter = detail::TextureFilterMode_Linear;
    resSampler.wrapU  = detail::TextureWrapMode_Mirror;
    resSampler.wrapV  = detail::TextureWrapMode_Mirror;

    nn::vfx::detail::TextureSampler* sampler = detail::TextureSampler::GetSamplerFromTable( &resSampler );
    if ( !sampler ) return false;

    bool ret = false;

    if( vertexLoc != InvalidValueId_TextureSamplerSlotId )
    {
        pCommandBuffer->SetTextureAndSampler( vertexLoc, nn::gfx::ShaderStage_Vertex, textureSlot, sampler->GetDescriptorSlot() );
        ret = true;
    }
    if( pixelLoc != InvalidValueId_TextureSamplerSlotId )
    {
        pCommandBuffer->SetTextureAndSampler( pixelLoc, nn::gfx::ShaderStage_Pixel, textureSlot, sampler->GetDescriptorSlot() );
        ret = true;
    }

    return ret;
}

//---------------------------------------------------------------------------
// vfxシステムが用意するテクスチャを登録します。
//---------------------------------------------------------------------------
void System::RegisterTextureViewToDescriptorPool( nn::vfx::RegisterTextureViewSlot pRegisterTextureSlotCallback, void* pUserData ) NN_NOEXCEPT
{
    // カールノイズのテクスチャービューをディスクリプタプールに登録します。
    detail::RegisterCurlNoiseTextureViewToDescriptorPool( pRegisterTextureSlotCallback, pUserData );
}

//---------------------------------------------------------------------------
// vfxシステムが用意するテクスチャを登録解除します。
//---------------------------------------------------------------------------
void System::UnregisterTextureViewFromDescriptorPool( nn::vfx::UnregisterTextureViewSlot pUnregisterTextureSlotCallback, void* pUserData) NN_NOEXCEPT
{
    // カールノイズのテクスチャービューをディスクリプタプールから解放します。
    detail::UnRegisterCurlNoiseTextureViewToDescriptorPool( pUnregisterTextureSlotCallback, pUserData );
}

//---------------------------------------------------------------------------
//  vfxシステムが用意するテクスチャーサンプラをディスクリプタプールに登録します。
//---------------------------------------------------------------------------
void System::RegisterSamplerToDescriptorPool( nn::vfx::RegisterSamplerSlot pRegisterSamplerSlotCallback, void* pUserData) NN_NOEXCEPT
{
    detail::TextureSampler::RegisterSamplerToDescriptorPool(reinterpret_cast<void *>( pRegisterSamplerSlotCallback), pUserData );
}

//---------------------------------------------------------------------------
//  vfxシステムが用意するテクスチャーサンプラを登録解除します。
//---------------------------------------------------------------------------
void System::UnregisterSamplerFromDescriptorPool( nn::vfx::UnregisterSamplerSlot pUnregisterSamplerSlotCallback, void* pUserData) NN_NOEXCEPT
{
    detail::TextureSampler::UnregisterSamplerFromDescriptorPool( reinterpret_cast<void *>(pUnregisterSamplerSlotCallback), pUserData );
}

} // namespace vfx
} // namespace nn

