﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vfx/vfx_SuperStripe.h>
#include <nn/vfx/vfx_Heap.h>
#include <nn/vfx/vfx_Emitter.h>
#include <nn/vfx/vfx_System.h>
#include <nn/vfx/vfx_Misc.h>
#include <nn/vfx/vfx_MemUtil.h>

namespace nn {
namespace vfx {
namespace detail {

namespace {
//---------------------------------------------------------------------------
//  テクスチャ座標計算：「履歴毎に固定割合」x「全て貼る」
//---------------------------------------------------------------------------
float CalculateTextureOffsetUniformFill( int index, int maxIndex, float indexOffset ) NN_NOEXCEPT
{
    NN_UNUSED( indexOffset );
    // MEMO: CalculateTextureOffsetUniformTile() とは実装を分けておく。
    //       UV補正処理で分岐できそう。
    // (index == maxIndex-1) の時に 1.0 を返すので、分母を 1引いて計算する
    const float fIndex = static_cast< float >( index );
    return static_cast< float >( fIndex ) / static_cast< float >( maxIndex - 1 );
}
//---------------------------------------------------------------------------
//  テクスチャ座標計算：「履歴距離に依存」x「全て貼る」
//---------------------------------------------------------------------------
float CalculateTextureOffsetDistanceBasedFill( int index, float prevOffset, float delta, float invStripeLength ) NN_NOEXCEPT
{
    // 先頭を揃える
    if( index == 0 )
    {
        return 0.0f;
    }

    // 直前の UV 差分をに加算
    const float offset = prevOffset + ( delta * invStripeLength );
    return offset;
}

//---------------------------------------------------------------------------
//  テクスチャ座標計算：「履歴毎に固定割合」x「全て貼る」
//---------------------------------------------------------------------------
float CalculateTextureOffsetUniformTile( int index, int maxIndex, float indexOffset ) NN_NOEXCEPT
{
    // (index == maxIndex-1) の時に 1.0 を返すので、分母を 1引いて計算する
    const float fIndex = ( index + indexOffset );
    float offset = static_cast< float >( fIndex ) / static_cast< float >( maxIndex - 1 );
    if( offset > 1.0f )
    {
        // 1.0 を超えた分を Clamp
        offset = 1.0f;
    }
    return offset;
}

//---------------------------------------------------------------------------
//  テクスチャ座標計算：「履歴距離に依存」x「描画範囲に合わせて貼る」
//---------------------------------------------------------------------------
float CalculateTextureOffsetDistanceBasedTile( int index, float prevOffset, float delta, float length ) NN_NOEXCEPT
{
    const float UvUnitLengthOnWorld = 0.1f;
    if( index == 0 )
    {
        return length * UvUnitLengthOnWorld;
    }

    const float offset = prevOffset + ( -delta * UvUnitLengthOnWorld );
    return offset;
}

//---------------------------------------------------------------------------
//  ストライプの履歴点のなす折れ線の長さを計算する
//---------------------------------------------------------------------------
float CalculateStripeLength( const nn::vfx::detail::SuperStripeSystem::History* pHistory, int historyCount ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pHistory );

    float stripeLength = 0.0f;
    for( int i = 1; i < historyCount; ++i, pHistory = pHistory->pNext )
    {
        nn::util::Vector3fType temp;
        nn::util::VectorSubtract( &temp, pHistory->pos, pHistory->pPrev->pos );
        stripeLength += nn::util::VectorLength( temp );
    }
    return stripeLength;
}
}

static const CustomShaderConstantBufferIndex SuperStripeConstantBufferIndex = CustomShaderConstantBufferIndex_4;               //!< カスタム定数バッファIndex
static const EmitterPluginCallbackIndex SuperStripeCallbackIndex = EmitterPluginCallbackIndex_3; //!< エミッタプラグイン用コールバックID

SuperStripeSystem* SuperStripeSystem::g_pStripeSystem = NULL;

//---------------------------------------------------------------------------
//  コンストラクタ
//---------------------------------------------------------------------------
SuperStripeSystem::SuperStripeSystem( nn::vfx::Heap* pHeap, nn::vfx::System* pSystem, BufferingMode bufferMode, int stripeCount ) NN_NOEXCEPT
    : m_pSystem( pSystem )
    , m_pHeap( pHeap )
    , m_BufferMode( bufferMode )
    , m_StripeArrayCount( stripeCount )
    , m_StripeWorkSize( sizeof( StripeInstance ) * stripeCount )
    , m_StripeArrayIndex( 0 )
    , m_StripeCount( 0 )
    , m_StripeArray( NULL )
{
    m_StripeArray = static_cast< StripeInstance* >( m_pHeap->Alloc( sizeof( StripeInstance ) * m_StripeArrayCount ) );
    memset( m_StripeArray, 0, sizeof( StripeInstance ) * m_StripeArrayCount );

    // ストライプコールバックを設定する
    CallbackSet callbackSet;
    callbackSet.emitterInitialize    = SuperStripeSystem::InitializeStripeEmitter;
    callbackSet.particleEmit         = SuperStripeSystem::EmitStripe;
    callbackSet.particleRemove       = SuperStripeSystem::KillStripe;
    callbackSet.particleCalculate    = SuperStripeSystem::ParticleCalculateCallback;
    callbackSet.emitterPreCalculate  = SuperStripeSystem::EmitterPreCalculateCallback;
    callbackSet.emitterPostCalculate = SuperStripeSystem::EmitterPostCalculateCallback;
    callbackSet.emitterDraw          = SuperStripeSystem::EmitterDrawCallback;
    callbackSet.emitterFinalize      = SuperStripeSystem::FinalizeStripeEmitter;
    callbackSet.renderStateSet       = detail::DummyRenderStateSetCallback;
    m_pSystem->SetEmitterPluginCallbackSet( detail::SuperStripeCallbackIndex, callbackSet );
}

//---------------------------------------------------------------------------
//  デストラクタ
//---------------------------------------------------------------------------
SuperStripeSystem::~SuperStripeSystem() NN_NOEXCEPT
{
    m_pHeap->Free( m_StripeArray );
    m_StripeArray = NULL;
}

//------------------------------------------------------------------------------
//  ストライプシステムの初期化処理を行います。
//------------------------------------------------------------------------------
void SuperStripeSystem::InitializeSystem( nn::vfx::Heap* pHeap, nn::vfx::System* pSystem, BufferingMode bufferMode, int stripeNum ) NN_NOEXCEPT
{
    if( g_pStripeSystem )
    {
        OutputError( "SuperStripeSystem is already initialized.\n" );
    }

    // プレビュー生成用のメモリを確保します。
    void* pBuffer = pHeap->Alloc( sizeof( SuperStripeSystem ) );
    if( pBuffer == NULL )
    {
        nn::vfx::detail::OutputWarning( "[SuperStripe] Memory Allocate Error!! : %d\n", sizeof( SuperStripeSystem ) );
        return;
    }

    // システムのコンストラクタを走らせる
    g_pStripeSystem = new ( pBuffer )SuperStripeSystem( pHeap, pSystem, bufferMode, stripeNum );
}

//------------------------------------------------------------------------------
//  ストライプシステムの終了処理を行います。
//------------------------------------------------------------------------------
void SuperStripeSystem::FinalizeSystem( nn::vfx::Heap* pHeap ) NN_NOEXCEPT
{
    g_pStripeSystem->~SuperStripeSystem();
    pHeap->Free( g_pStripeSystem );
    g_pStripeSystem = NULL;
}

//------------------------------------------------------------------------------
//  エミッタ生成後コールバック
//------------------------------------------------------------------------------
bool SuperStripeSystem::InitializeStripeEmitter( EmitterInitializeArg& arg ) NN_NOEXCEPT
{
    //------------------------------------------------------------------------------
    // シェーダアトリビュートの方は ShaderManager 側で初期化済みという前提。
    // （代わりにアトリビュート名が決め打ちかつ VEC4 縛りになっている）
    //------------------------------------------------------------------------------
    if( arg.pEmitter->GetCalculationType() != EmitterCalculationMode_Cpu )
    {
        return false;
    }

    // エミッタ単位での記憶領域を確保
    EmitterPluginUserData* pEPUserData = reinterpret_cast< EmitterPluginUserData* >( arg.pEmitter->GetDynamicHeap().Alloc( sizeof( EmitterPluginUserData ) ) );
    if( pEPUserData == NULL )
    {
        return false;
    }
    MemUtil::FillZero( pEPUserData, sizeof( EmitterPluginUserData ) ); // ゼロ埋めで初期化
    pEPUserData->bufferSide = BufferSide_FrontBuffer;
    arg.pEmitter->SetEmitterPluginUserData( pEPUserData );

    //------------------------------------------------------------------------------
    // エミッタ単位で記憶する情報を保存
    //------------------------------------------------------------------------------
    {
        pEPUserData->maxEmitRatio = static_cast< uint32_t >( ::std::ceilf( arg.pEmitter->GetEmissionRateMax() ) );
        pEPUserData->maxParticleLife = arg.pEmitter->GetParticleLifeMax();
        {
            //------------------------------------------------------------------------------
            // 静的な定数バッファの内容をあらかじめ作っておく
            //------------------------------------------------------------------------------
            ResStripeSuper* pStripeData = GetStripeData( arg.pEmitter );
            pEPUserData->staticParam0.x = pStripeData->headAlpha;
            pEPUserData->staticParam0.y = pStripeData->tailAlpha;
            pEPUserData->staticParam0.z = pStripeData->headScale;
            pEPUserData->staticParam0.w = pStripeData->tailScale;
        }
    }

    bool ret = false;

    // 定数バッファ を初期化
    ret = arg.pEmitter->InitializeCustomConstantBuffer( detail::SuperStripeConstantBufferIndex, sizeof( ConstantBufferObject ) );
    if ( !ret )
    {
        return ret;
    }

    // ポリゴン用バッファを確保
    ret = g_pStripeSystem->AllocStripeSystemVertexBuffer( arg.pEmitter );
    return ret;
}

//---------------------------------------------------------------------------
//! @brief        ポリゴン用ワークを確保します。
//---------------------------------------------------------------------------
bool SuperStripeSystem::AllocStripeSystemVertexBuffer( Emitter* pEmitter ) NN_NOEXCEPT
{
    //---------------------------------------------------------------------------
    // ここでエミッタ全体のストライプ用頂点バッファを初期化。
    // 描画はストライプ1本ごとにオフセットと範囲を指定して描画していく。
    //---------------------------------------------------------------------------
    ResStripeSuper* pStripeData = GetStripeData( pEmitter );
    EmitterPluginUserData* pEPUserData = SuperStripeSystem::GetEmitterUserData( pEmitter );

    const int historyCount = static_cast< int >( pStripeData->numHistory );
    const int maxDelayedStripeCount = StripeSystemUtility::CalculateDelayStripeCount( pEmitter, historyCount );

    // 最大のパーティクル存在数＋遅延ストライプの本数（上記）が最大数
    pEPUserData->maxStripeCount = pEmitter->GetParticleAttrFillMax() + maxDelayedStripeCount;

    const int divCount = static_cast< int >( pStripeData->numDivide );
    pEPUserData->vertexCountPerStripe = ( historyCount + ( ( historyCount - 1 ) * divCount ) ) * 2;

    const int vertexBufferCount = pEPUserData->vertexCountPerStripe * pEPUserData->maxStripeCount;  // 背骨の数×左右（2個）×ストライプ本数
    pEPUserData->vertexBufferCount = vertexBufferCount * m_BufferMode;                       // バッファリングの分倍加する

    // 頂点バッファサイズ（シングルバッファの分）
    const size_t bufferSize = vertexBufferCount * sizeof( VertexAttribute );

    //---------------------------------------------------------------------------
    // Attribute の初期化
    // 確保した領域は後に自分で開放する必要があるので記憶する
    //---------------------------------------------------------------------------
    pEPUserData->vertexBufferInitialized = false;
    if( bufferSize != 0 )
    {
        if ( pEmitter->InitializeCustomAttribute( m_BufferMode, bufferSize ) )
        {
            pEPUserData->vertexBufferInitialized = true;
        }
    }

    return pEPUserData->vertexBufferInitialized;
}

//------------------------------------------------------------------------------
//  エミッタ破棄後コールバック
//------------------------------------------------------------------------------
void SuperStripeSystem::FinalizeStripeEmitter( EmitterFinalizeArg& arg ) NN_NOEXCEPT
{
    if( arg.pEmitter->GetCalculationType() != EmitterCalculationMode_Cpu )
    {
        return;
    }

    EmitterPluginUserData* pEPUserData = SuperStripeSystem::GetEmitterUserData( arg.pEmitter );
    if ( pEPUserData )
    {
        pEPUserData->vertexBufferInitialized = false;

        // 遅延ストライプの解放
        StripeSystemUtility::FreeAllDelayedStripe( g_pStripeSystem, &pEPUserData, arg.pEmitter );

        // エミッタユーザーデータの領域を解放
        detail::FreeFromDynamicHeapSafely( reinterpret_cast< void** >( &pEPUserData ), arg.pEmitter );
        arg.pEmitter->SetEmitterPluginUserData( NULL );
    }
}

//------------------------------------------------------------------------------
//  パーティクル生成コールバック
//------------------------------------------------------------------------------
bool SuperStripeSystem::EmitStripe( ParticleEmitArg& arg ) NN_NOEXCEPT
{
    if( arg.pEmitter->GetCalculationType() != EmitterCalculationMode_Cpu )
    {
        return false;
    }

    StripeInstance* pStripe = StripeSystemUtility::AllocStripe( g_pStripeSystem, arg.pEmitter );
    if( pStripe == NULL )
    {
        return false;
    }
    pStripe->journey = 0;
    pStripe->prevFrame = 0;
    pStripe->delayedHistoryCount = 0;

    // ユーザーデータの割り当て
    arg.pEmitterPluginData = pStripe;

    // 頂点バッファの開始位置を設定
    // 最大履歴数でアラインしてラウンドロビンで割り当て。
    // かならず隙間ができるので、確定で詰められるならその方がベター。
    EmitterPluginUserData* pEPUserData = SuperStripeSystem::GetEmitterUserData( arg.pEmitter );
    pStripe->indexOffset = pEPUserData->attributeIndex % pEPUserData->maxStripeCount;
    if( ++pEPUserData->attributeIndex >= pEPUserData->maxStripeCount )
    {
        pEPUserData->attributeIndex = 0;
    }

    // ランダム値の割り当て
    arg.GetRandom( &pStripe->random );

    return true;
}

//------------------------------------------------------------------------------
// ストライプのカラーを更新（ストライプ一本で一つのカラーを共有）
//------------------------------------------------------------------------------
void SuperStripeSystem::UpdateStripeColor( ParticleCalculateArgImpl& arg, StripeInstance* pStripe ) NN_NOEXCEPT
{
    nn::util::Vector4fType random;
    arg.GetRandom( &random );
    arg.pEmitter->GetEmitterCalculator()->CalculateParticleColor0RawValue(
        &pStripe->color0,
        arg.pEmitter->GetEmitterResource(),
        random,
        arg.GetLife(),
        arg.GetTime() );
    arg.pEmitter->GetEmitterCalculator()->CalculateParticleColor1RawValue(
        &pStripe->color1,
        arg.pEmitter->GetEmitterResource(),
        random,
        arg.GetLife(),
        arg.GetTime() );
    // カラースケールを乗算
    // MEMO: 本来はシェーダ側で乗算するべきだが、シェーダー側がまだ追いついていないのでこちらで乗算しておく
    // エミッタカラー / カラースケール
    const float colorScale = arg.pEmitter->GetEmitterResource()->m_ResEmitterStaticConstantBuffer->colorScale;
    nn::util::VectorSetX( &pStripe->color0, nn::util::VectorGetX( pStripe->color0 ) * colorScale );
    nn::util::VectorSetY( &pStripe->color0, nn::util::VectorGetY( pStripe->color0 ) * colorScale );
    nn::util::VectorSetZ( &pStripe->color0, nn::util::VectorGetZ( pStripe->color0 ) * colorScale );

    nn::util::VectorSetX( &pStripe->color1, nn::util::VectorGetX( pStripe->color1 ) * colorScale );
    nn::util::VectorSetY( &pStripe->color1, nn::util::VectorGetY( pStripe->color1 ) * colorScale );
    nn::util::VectorSetZ( &pStripe->color1, nn::util::VectorGetZ( pStripe->color1 ) * colorScale );
}

//------------------------------------------------------------------------------
//  パーティクル削除コールバック
//------------------------------------------------------------------------------
bool SuperStripeSystem::KillStripe( ParticleRemoveArg& arg ) NN_NOEXCEPT
{
    if( arg.pEmitter->GetCalculationType() != EmitterCalculationMode_Cpu )
    {
        return false;
    }

    if( arg.pEmitterPluginData )
    {
        StripeInstance* pStripe = reinterpret_cast< StripeInstance* >( arg.pEmitterPluginData );
        NN_SDK_ASSERT_NOT_NULL( pStripe );

        pStripe->delayedHistoryCount = pStripe->historyCount;   //現時点でのストライプ数を保存
        // ストライプのカラーを更新（ストライプ一本で一つのカラーを共有）
#ifdef _VFX_NSDK01215_COMPATIBLE
        arg.CalculateLocalColor0( &pStripe->color0 );
        arg.CalculateLocalColor1( &pStripe->color1 );
#else
        UpdateStripeColor( arg, pStripe );
#endif
        EmitterPluginUserData* pEPUserData = SuperStripeSystem::GetEmitterUserData( arg.pEmitter );
        StripeSystemUtility::MoveStripeToDelayList< SuperStripeSystem >( &pEPUserData, pStripe );
    }

    return false;
}

//---------------------------------------------------------------------------
//  ストライプ計算処理を行います。
//---------------------------------------------------------------------------
void SuperStripeSystem::CalculateStripe( ParticleCalculateArg& arg, StripeInstance* pStripe ) NN_NOEXCEPT
{
    //---------------------------------------------------------------------------
    // 事前条件チェックは呼び出し元のコールバック側で確認済み。
    //---------------------------------------------------------------------------
    Emitter* pEmitter = arg.pEmitter;
    ResStripeSuper* pStripeData = GetStripeData( pEmitter );
    EmitterPluginUserData* pEPUserData = SuperStripeSystem::GetEmitterUserData( pEmitter );
    VertexAttribute* pVertexBufferHead = reinterpret_cast< VertexAttribute* >( arg.pEmitter->GetGfxObjects()->m_ParticleEmitterPluginAttribute.Map( pEPUserData->bufferSide ) );

    // 現在時刻をコピー
    pStripe->time = arg.GetTime();
    pStripe->life = arg.GetLife();
    pStripe->renderVertexCount = 0;

    // ストライプのカラーを更新（ストライプ一本で一つのカラーを共有）
#ifdef _VFX_NSDK01215_COMPATIBLE
    arg.CalculateLocalColor0( &pStripe->color0 );
    arg.CalculateLocalColor1( &pStripe->color1 );
#else
    UpdateStripeColor( arg, pStripe );
#endif

    //---------------------------------------------------------------------------
    // 履歴を積むか積まないかのチェック
    //---------------------------------------------------------------------------
    const float testPrevFrame = ::std::floorf( pStripe->prevFrame );
    const float testNowFrame = ::std::floorf( pStripe->time );
    const bool doPushHistory = ( testNowFrame - testPrevFrame > 0 ) || ( pStripe->time == 0.0f );
    pStripe->prevFrame = arg.GetTime(); // ストライプ時間（前回更新）を記憶

    //---------------------------------------------------------------------------
    // 履歴の更新
    //---------------------------------------------------------------------------
    UpdateHistory( arg, pStripe, pStripeData, doPushHistory );

    //---------------------------------------------------------------------------
    // ポリゴンバッファを生成
    //---------------------------------------------------------------------------
    {
        // 今更新中のストライプが書くべき頂点バッファの先頭を取得
        VertexAttribute* pVertexAttribute = &pVertexBufferHead[ pStripe->indexOffset * pEPUserData->vertexCountPerStripe ];

        // 可変フレームレートでも滑らかに動かすためのテクスチャオフセット量: [0,1)
        if( pStripeData->numDivide == 0 )
        {
            // 分割無し
            MakeStripePolygon( pVertexAttribute, pStripe, pEmitter, pStripeData );
        }
        else
        {
            // 分割あり
            MakeStripePolygonWithDivision( pVertexAttribute, pStripe, pEmitter, pStripeData );
        }
    }
    arg.pEmitter->GetGfxObjects()->m_ParticleEmitterPluginAttribute.Unmap();
}

//---------------------------------------------------------------------------
//  ストライプの履歴の更新を行います（複雑になったので分離）
//---------------------------------------------------------------------------
void SuperStripeSystem::UpdateHistory(
    ParticleCalculateArg& arg,
    StripeInstance* pStripe,
    ResStripeSuper* pStripeData,
    const bool doPushHistory ) NN_NOEXCEPT
{
    const float frameRate = arg.pEmitter->GetFrameRate();
    const float histAirRegist = pStripeData->historyAirRegist;
    const float invRate = 1.0f - frameRate;
    const float airRegist = histAirRegist + ( 1.0f - histAirRegist ) * invRate;

    nn::util::Vector4fType color0, color1;
    arg.CalculateLocalColor0( &color0 );
    arg.CalculateLocalColor1( &color1 );
    if( doPushHistory == false )
    {
        History* const pNowHistory = pStripe->pHistoryHead->pNext;    // 先頭は空なのでその次から。

        // 先頭の履歴を合わせる
        nn::util::Vector3fType scale;
        arg.CalculateWorldScale( &scale );

        nn::util::Vector3fType prevPos;
        prevPos = pNowHistory->pos;
        if( pStripeData->emitterFollow )
        {
            // エミッタに完全追従：ローカル系で保存
            arg.GetLocalPos( &pNowHistory->pos );
        }
        else
        {
            // エミッタに完全追従しない：ワールド系で保存
            arg.GetWorldPos( &pNowHistory->pos );
        }

#ifndef _VFX_NSDK01215_COMPATIBLE
        // 先頭位置を記憶
        pStripe->headPos = pNowHistory->pos;
#endif

        nn::util::Vector3fType temp;
        nn::util::VectorSubtract( &temp, pNowHistory->pos, prevPos );
        pStripe->journey += nn::util::VectorLength( temp );

        pNowHistory->scale = nn::util::VectorGetX( scale );
        const nn::util::Matrix4x3fType& matrixSrt = arg.pEmitter->GetMatrixSrt();
        nn::util::MatrixGetAxisY( &pNowHistory->emat, matrixSrt );

        // 履歴を動かす
        {
            History* pHistory = pStripe->pHistoryHead->pNext->pNext;

            // 履歴速度を更新（加速度＆空気抵抗）
            const float temp2 = frameRate * airRegist;
            nn::util::Vector3fType accel = NN_UTIL_VECTOR_3F_INITIALIZER(
                pStripeData->historyAcceleration.x * temp2,
                pStripeData->historyAcceleration.y * temp2,
                pStripeData->historyAcceleration.z * temp2 );
            for( int i = 1; i < pStripe->historyCount; ++i, pHistory = pHistory->pNext )
            {
                nn::util::Vector3fType temp3;
                nn::util::VectorMultiply( &temp3, pHistory->vec, airRegist );
                nn::util::VectorAdd( &pHistory->vec, temp3, accel );

                nn::util::VectorMultiply( &temp3, pHistory->vec, frameRate );
                nn::util::VectorAdd( &pHistory->pos, pHistory->pos, temp3 );
            }
        }
    }
    else
    {
        if( pStripeData->calcType != StripeOrientationType_Ribbon )
        {
            //------------------------------------------------------------------------------
            // リボン以外
            //------------------------------------------------------------------------------
            // 履歴を積む（位置差分が0以上であることが確定）
            History* const pNowHistory = pStripe->pHistoryHead;          // 新しい履歴（最後尾）
            History* const pPrevHistory = pNowHistory->pNext;
            pStripe->pHistoryHead = pStripe->pHistoryHead->pPrev;    // 最後尾を指すポインタを更新

            // 履歴を積む
            nn::util::Vector3fType scale;
            arg.CalculateWorldScale( &scale );

            if( pStripeData->emitterFollow )
            {
                // エミッタに完全追従：ローカル系で保存
                arg.GetLocalPos( &pNowHistory->pos );
            }
            else
            {
                // エミッタに完全追従しない：ワールド系で保存
                arg.GetWorldPos( &pNowHistory->pos );
            }

#ifndef _VFX_NSDK01215_COMPATIBLE
            // 先頭位置を記憶
            pStripe->headPos = pNowHistory->pos;
#endif

            if( pStripe->historyCount == 0 )
            {
                pStripe->journey += nn::util::VectorLength( pNowHistory->pos );
            }
            else
            {
                nn::util::Vector3fType prevPos;
                prevPos = pPrevHistory->pos;
                nn::util::Vector3fType temp;
                nn::util::VectorSubtract( &temp, pNowHistory->pos, prevPos );
                pStripe->journey += nn::util::VectorLength( temp );
            }

            {
                pNowHistory->scale = nn::util::VectorGetX( scale );
            }

            if( pStripeData->emitterFollow )
            {
                // エミッタに完全追従：ローカル系で位置を覚えるので上方向はあくまでY軸方向
                nn::util::VectorSet( &pNowHistory->emat, 0, 1, 0 );
            }
            else
            {
                // エミッタに完全追従しない：ワールド系で位置を覚えるので上方向はエミッタ行列のY軸方向
                const nn::util::Matrix4x3fType& matrixSrt = arg.pEmitter->GetMatrixSrt();
                nn::util::Vector3f ay;
                nn::util::MatrixGetAxisY( &ay, matrixSrt );
                nn::util::VectorSetX( &pNowHistory->emat, nn::util::VectorGetX( ay ) );
                nn::util::VectorSetY( &pNowHistory->emat, nn::util::VectorGetY( ay ) );
                nn::util::VectorSetZ( &pNowHistory->emat, nn::util::VectorGetZ( ay ) );
            }

#ifndef _VFX_NSDK01215_COMPATIBLE
            // ストライプ Z ロール: ematを使うビルボードタイプ（エミッタ行列ストライプ＆エミッタ上下に伸ばす）の時、emat を回転させてストライプを曲げる
            if( pStripeData->calcType == StripeOrientationType_EmitterMatrix || pStripeData->calcType == StripeOrientationType_EmitterUpright )
            {
                if( arg.pEmitter->GetResEmitter()->ptcl.isRotateZ )
                {
                    const float Epsilon = 0.00001f; // 「ほぼ」0とみなす値

                    nn::util::Vector3fType rotateAxis = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 0, 0 );
                    if( pStripe->historyCount >= 1 )
                    {
                        // 2つ目以降の履歴点は、移動差分も進行方向とみなす
                        nn::util::Vector3fType temp;
                        nn::util::VectorSubtract( &temp, pNowHistory->pos, pNowHistory->pNext->pos );
                        nn::util::Vector3fType diff;
                        detail::Vector3fCopy( &diff, temp );
                        nn::util::VectorMultiply( &diff, diff, 1.0f / arg.pEmitter->GetFrameRate() ); // フレームレートから速度換算
                        nn::util::VectorAdd( &rotateAxis, rotateAxis, diff );
                    }
                    else
                    {
                        // 一旦速度で代用（次の更新で2回目の値で上書きされる）
                        rotateAxis = pNowHistory->dir;
                    }

                    if( nn::util::VectorLengthSquared( rotateAxis ) < Epsilon )
                    {
                        // 進行方向＋移動差分が 0 に近い場合、エミッタ行列を採用
                        const nn::util::Matrix4x3fType& matrixSrt = arg.pEmitter->GetMatrixSrt();
                        nn::util::MatrixGetAxisZ( &rotateAxis, matrixSrt );
                    }
                    nn::util::VectorNormalize( &rotateAxis, rotateAxis );

                    // 進行方向について、 emat を rotateZ だけ回転させる
                    nn::util::Vector3fType rotate;
                    arg.CalculateCurrentRotateRadian( &rotate );    // Z 要素だけ使う
                    const float rotateZ = nn::util::VectorGetZ( rotate );
                    detail::VectorRotateArbitraryAxis( &pNowHistory->emat, pNowHistory->emat, rotateAxis, rotateZ );
                    nn::util::VectorNormalize( &pNowHistory->emat, pNowHistory->emat );

                    if( pStripe->historyCount == 1 )
                    {
                        // 2つ目の履歴点ができる瞬間、初回の履歴点を2回目と揃えておく。
                        pNowHistory->pNext->emat = pNowHistory->emat;
                    }
                }
            }
#endif

            // 履歴に与える初速（仮）
            {
                nn::util::Vector4fType random;
                arg.GetRandom( &random );
                const float hVecD = pStripeData->historyVecInitSpeed;
                const float t = pStripe->time;

                nn::util::Vector3fType hVecCycRatio = NN_UTIL_VECTOR_3F_INITIALIZER(
                    pStripeData->historyInitVecRotateCycle.x * pStripeData->historyVecRegulation,
                    pStripeData->historyInitVecRotateCycle.y * pStripeData->historyVecRegulation,
                    pStripeData->historyInitVecRotateCycle.z * pStripeData->historyVecRegulation );

                nn::util::VectorSetX( &pNowHistory->vec, nn::util::SinTable( nn::util::DegreeToAngleIndex( t * ( nn::util::VectorGetX( hVecCycRatio ) ) + 360.0f * nn::util::VectorGetX( random ) ) ) + nn::util::SinTable( nn::util::DegreeToAngleIndex( t * 0.5f + 360.0f * nn::util::VectorGetY( random ) ) ) );
                nn::util::VectorSetY( &pNowHistory->vec, nn::util::SinTable( nn::util::DegreeToAngleIndex( t * ( nn::util::VectorGetY( hVecCycRatio ) ) + 360.0f * nn::util::VectorGetY( random ) ) ) + nn::util::SinTable( nn::util::DegreeToAngleIndex( t * 0.5f + 360.0f * nn::util::VectorGetZ( random ) ) ) );
                nn::util::VectorSetZ( &pNowHistory->vec, nn::util::SinTable( nn::util::DegreeToAngleIndex( t * ( nn::util::VectorGetZ( hVecCycRatio ) ) + 360.0f * nn::util::VectorGetZ( random ) ) ) + nn::util::SinTable( nn::util::DegreeToAngleIndex( t * 0.5f + 360.0f * nn::util::VectorGetW( random ) ) ) );
                nn::util::VectorMultiply( &pNowHistory->vec, pNowHistory->vec, hVecD );
            }

            if( pStripe->historyCount < pStripeData->numHistory - 1 )
            {
                pStripe->historyCount++;
            }

            //-------------------------------
            // 背骨の方向（hist.dir）・羽根の計算のための方向（hist.wing）を設定
            //-------------------------------
            // TODO: まだまだ改良の余地あり
            {
                if( pStripe->historyCount <= 1 )
                {
                    if( pStripeData->emitterFollow )
                    {
                        // エミッタに完全追従：ローカル系で保存
                        arg.GetLocalVec( &pNowHistory->dir );   // 最初の進行方向はパーティクルの初速
                    }
                    else
                    {
                        // エミッタに完全追従しない：ワールド系で保存
                        arg.GetWorldVec( &pNowHistory->dir );   // 最初の進行方向はパーティクルの初速
                    }

                    if( nn::util::VectorLengthSquared( pNowHistory->dir ) > 0 )
                    {
                        nn::util::VectorNormalize( &pNowHistory->dir, pNowHistory->dir );
                        detail::Vector3fCopy( &pNowHistory->wing, pNowHistory->dir );
                    }
                    else
                    {
                        // 初回かつ速度0（速度0＋自動移動などのケース）
                        const nn::util::Matrix4x3fType& matrixSrt = arg.pEmitter->GetMatrixSrt();
                        nn::util::Vector3fType emitterPos;
                        nn::util::MatrixGetAxisW( &emitterPos, matrixSrt );

                        nn::util::Vector3fType worldDiff;
                        nn::util::VectorSubtract( &worldDiff, emitterPos, arg.pEmitter->GetEmitterPrevPos() );

                        if( nn::util::VectorLengthSquared( worldDiff ) > 0 )
                        {
                            // エミッタが動いている場合、位置差分を方向にする。
                            pNowHistory->dir = worldDiff;
                            nn::util::VectorNormalize( &pNowHistory->dir, pNowHistory->dir );
                            detail::Vector3fCopy( &pNowHistory->wing, pNowHistory->dir );
                        }
                        else
                        {
                            // まったく動いていない場合。仮の向きを入れておく。
                            nn::util::VectorSet( &pNowHistory->dir, 0, 0.001f, 0 );
                            detail::Vector3fCopy( &pNowHistory->wing, pNowHistory->dir );
                        }
                    }
                }
                else
                {
                    nn::util::VectorSubtract( &pNowHistory->dir, pNowHistory->pos, pPrevHistory->pos );

                    if( pStripe->historyCount == 2 )
                    {
                        // 履歴点が二つになった時、最初の方を今できた向きで上書き
                        pPrevHistory->dir = pNowHistory->dir;
                    }

                    // 平均をとって全体に適用（テスト）
                    nn::util::Vector3fType average = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 0, 0 );
                    History* pHistory = pNowHistory;
                    for( int i = 0; i < pStripe->historyCount; ++i, pHistory = pHistory->pNext )
                    {
                        nn::util::Vector3fType temp;
                        nn::util::VectorMultiply( &temp, pHistory->dir, static_cast< float >( ( pStripe->historyCount - i ) ) );
                        nn::util::VectorAdd( &average, average, temp );
                    }
                    if( nn::util::VectorLengthSquared( average ) > 0 )
                    {
                        nn::util::VectorNormalize( &average, average );
                    }

                    detail::Vector3fCopy( &pNowHistory->wing, average );
                    if( pStripe->historyCount == 2 )
                    {
                        // 履歴点が二つになった時、最初の方を今できた向きで上書き
                        detail::Vector3fCopy( &pPrevHistory->wing, pNowHistory->wing );
                    }
                }

                {
                    nn::util::Vector3fType wingDir; // VEC4からVEC3だけ抜き出す
                    detail::Vector3fCopy( &wingDir, pNowHistory->wing );
                    if( nn::util::VectorLengthSquared( wingDir ) > 0 )
                    {
                        nn::util::VectorNormalize( &wingDir, wingDir );
                    }
                    else
                    {
                        // 向きがどうしても決定しなかった場合
                        nn::util::VectorSet( &wingDir, 0, 1, 0 );
                    }
                    detail::Vector3fCopy( &pNowHistory->wing, wingDir );
                }
            }

            // 履歴を動かす
            {
                // 履歴速度を更新（加速度＆空気抵抗）
                const float temp = frameRate * airRegist;
                const nn::util::Vector3fType accel = NN_UTIL_VECTOR_3F_INITIALIZER(
                    pStripeData->historyAcceleration.x * temp,
                    pStripeData->historyAcceleration.y * temp,
                    pStripeData->historyAcceleration.z * temp );
                History* pHistory = pNowHistory->pNext;
                for( int i = 1; i < pStripe->historyCount; ++i, pHistory = pHistory->pNext )
                {
                    nn::util::Vector3fType temp2;
                    nn::util::VectorMultiply( &temp2, pHistory->vec, airRegist );
                    nn::util::VectorAdd( &pHistory->vec, temp2, accel );

                    nn::util::VectorMultiply( &temp2, pHistory->vec, frameRate );
                    nn::util::VectorAdd( &pHistory->pos, pHistory->pos, temp2 );
                }
            }
        }
        else
        {
            //------------------------------------------------------------------------------
            // リボンストライプ
            //------------------------------------------------------------------------------

            //------------------------------------------------------------------------------
            // 履歴を積む（位置差分が0以上であることが確定）
            //------------------------------------------------------------------------------
            History* const pNowHistory = pStripe->pHistoryHead;     // 新しい履歴（最後尾）
            History* const pPrevHistory = pNowHistory->pNext;
            pStripe->pHistoryHead = pStripe->pHistoryHead->pPrev;   // 最後尾を指すポインタを更新

            //------------------------------------------------------------------------------
            // 履歴カウンタの更新
            //------------------------------------------------------------------------------
            if( pStripe->historyCount < pStripeData->numHistory - 1 )
            {
                pStripe->historyCount++;
            }

            //------------------------------------------------------------------------------
            // 位置を決定：既存の履歴点の更新より前に行う必要がある
            //------------------------------------------------------------------------------
            if( pStripeData->emitterFollow )
            {
                // エミッタに完全追従：ローカル系で保存
                arg.GetLocalPos( &pNowHistory->pos );
            }
            else
            {
                // エミッタに完全追従しない：ワールド系で保存
                arg.GetWorldPos( &pNowHistory->pos );
            }

#ifndef _VFX_NSDK01215_COMPATIBLE
            // 先頭位置を記憶
            pStripe->headPos = pNowHistory->pos;
#endif

            //------------------------------------------------------------------------------
            // 背骨方向の計算の前に、既存の履歴点の位置速度を更新する必要があるのでこのタイミングで行う。
            //------------------------------------------------------------------------------
            {
                // 履歴速度を更新（加速度＆空気抵抗）
                const float temp = frameRate * airRegist;
                const nn::util::Vector3fType accel = NN_UTIL_VECTOR_3F_INITIALIZER(
                    pStripeData->historyAcceleration.x * temp,
                    pStripeData->historyAcceleration.y * temp,
                    pStripeData->historyAcceleration.z * temp );
                History* pHistory = pNowHistory->pNext;
                for( int i = 1; i < pStripe->historyCount; ++i, pHistory = pHistory->pNext )
                {
                    nn::util::Vector3fType temp2;
                    nn::util::VectorMultiply( &temp2, pHistory->vec, airRegist );
                    nn::util::VectorAdd( &pHistory->vec, temp2, accel );

                    nn::util::VectorMultiply( &temp2, pHistory->vec, frameRate );
                    nn::util::VectorAdd( &pHistory->pos, pHistory->pos, temp2 );
                }
            }

            //------------------------------------------------------------------------------
            // 履歴点の初速の決定
            //------------------------------------------------------------------------------
            {
                nn::util::Vector4fType random;
                arg.GetRandom( &random );
                const float hVecD = pStripeData->historyVecInitSpeed;
                const float t = pStripe->time;

                nn::util::Vector3fType hVecCycRatio = NN_UTIL_VECTOR_3F_INITIALIZER(
                    pStripeData->historyInitVecRotateCycle.x * pStripeData->historyVecRegulation,
                    pStripeData->historyInitVecRotateCycle.y * pStripeData->historyVecRegulation,
                    pStripeData->historyInitVecRotateCycle.z * pStripeData->historyVecRegulation );

                nn::util::VectorSetX( &pNowHistory->vec, nn::util::SinTable( nn::util::DegreeToAngleIndex( t * ( nn::util::VectorGetX( hVecCycRatio ) ) + 360.0f * nn::util::VectorGetX( random ) ) ) + nn::util::SinTable( nn::util::DegreeToAngleIndex( t * 0.5f + 360.0f * nn::util::VectorGetY( random ) ) ) );
                nn::util::VectorSetY( &pNowHistory->vec, nn::util::SinTable( nn::util::DegreeToAngleIndex( t * ( nn::util::VectorGetY( hVecCycRatio ) ) + 360.0f * nn::util::VectorGetY( random ) ) ) + nn::util::SinTable( nn::util::DegreeToAngleIndex( t * 0.5f + 360.0f * nn::util::VectorGetZ( random ) ) ) );
                nn::util::VectorSetZ( &pNowHistory->vec, nn::util::SinTable( nn::util::DegreeToAngleIndex( t * ( nn::util::VectorGetZ( hVecCycRatio ) ) + 360.0f * nn::util::VectorGetZ( random ) ) ) + nn::util::SinTable( nn::util::DegreeToAngleIndex( t * 0.5f + 360.0f * nn::util::VectorGetW( random ) ) ) );
                nn::util::VectorMultiply( &pNowHistory->vec, pNowHistory->vec, hVecD );
            }

            //------------------------------------------------------------------------------
            // 履歴点の emat 方向の決定
            //------------------------------------------------------------------------------
            if( pStripeData->emitterFollow )
            {
                // エミッタに完全追従：ローカル系で位置を覚えるので上方向はあくまでY軸方向
                nn::util::VectorSet( &pNowHistory->emat, 0, 1, 0 );
            }
            else
            {
                // エミッタに完全追従しない：ワールド系で位置を覚えるので上方向はエミッタ行列のY軸方向
                const nn::util::Matrix4x3fType& matrixSrt = arg.pEmitter->GetMatrixSrt();
                nn::util::Vector3f ay;
                nn::util::MatrixGetAxisY( &ay, matrixSrt );
                nn::util::VectorSetX( &pNowHistory->emat, nn::util::VectorGetX( ay ) );
                nn::util::VectorSetY( &pNowHistory->emat, nn::util::VectorGetY( ay ) );
                nn::util::VectorSetZ( &pNowHistory->emat, nn::util::VectorGetZ( ay ) );
            }

            //------------------------------------------------------------------------------
            // 履歴点のスケール値の設定
            //------------------------------------------------------------------------------
            {
                nn::util::Vector3fType scale;
                arg.CalculateWorldScale( &scale );
                pNowHistory->scale = nn::util::VectorGetX( scale );
            }

            //------------------------------------------------------------------------------
            // 履歴点の背骨方向・羽根の展開方向の決定
            // 直前の履歴点が上段の処理で加速度による更新を受けているのに依存している
            //------------------------------------------------------------------------------
            {
                const float Epsilon = 0.00001f; // 「ほぼ」0とみなす値
                if( pStripe->historyCount == 1 )
                {
                    if( pStripeData->emitterFollow )
                    {
                        // エミッタに完全追従：ローカル系で保存
                        arg.GetLocalVec( &pNowHistory->dir );   // 最初の進行方向はパーティクルの初速
                    }
                    else
                    {
                        // エミッタに完全追従しない：ワールド系で保存
                        arg.GetWorldVec( &pNowHistory->dir );   // 最初の進行方向はパーティクルの初速
                    }

                    // 加速度の向きを加算しておく
                    // MEMO: 加速度＞＞速度のケースで初期のねじれを抑制できる
                    {
                        const float ratio = arg.GetLife();
                        const nn::util::Vector3fType accel = NN_UTIL_VECTOR_3F_INITIALIZER(
                            pStripeData->historyAcceleration.x * ratio,
                            pStripeData->historyAcceleration.y * ratio,
                            pStripeData->historyAcceleration.z * ratio );
                        nn::util::VectorAdd( &pNowHistory->dir, pNowHistory->dir, accel );
                    }

                    // 最初の履歴点（※前の履歴が無いので単品で向きを決める必要がある）
                    if( nn::util::VectorLengthSquared( pNowHistory->dir ) < Epsilon )
                    {
                        nn::util::Vector3fType dir = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 1, 0 );
                        nn::util::Vector3fType wing = NN_UTIL_VECTOR_3F_INITIALIZER( 1, 0, 0 );
                        detail::Vector3fCopy( &pNowHistory->dir, dir );
                        detail::Vector3fCopy( &pNowHistory->wing, wing );
                    }
                    else
                    {
                        const nn::util::Vector3fType axisY = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 1, 0 );
                        nn::util::Vector3fType dir = pNowHistory->dir;
                        nn::util::VectorNormalize( &dir, dir );
                        const float d = nn::util::VectorDot( dir, axisY );
                        if( ::std::abs( ::std::abs( d ) - 1.0f ) < Epsilon )
                        {
                            nn::util::Vector3fType wing = NN_UTIL_VECTOR_3F_INITIALIZER( 1, 0, 0 );
                            nn::util::VectorMultiply( &wing, wing, d );
                            nn::util::VectorNormalize( &wing, wing );
                            detail::Vector3fCopy( &pNowHistory->wing, wing );
                        }
                        else
                        {
                            nn::util::Vector3fType wing;
                            nn::util::VectorCross( &wing, dir, axisY );
                            nn::util::VectorNormalize( &wing, wing );
                            detail::Vector3fCopy( &pNowHistory->wing, wing );
                        }
                    }
                }
                else
                {
                    nn::util::VectorSubtract( &pNowHistory->dir, pNowHistory->pos, pNowHistory->pNext->pos );

                    // 二つ目以降の履歴点
                    if( nn::util::VectorLengthSquared( pNowHistory->dir ) < Epsilon )
                    {
                        // 前回からの位置差分を速度とみなす
                        nn::util::Vector3fType prevVec;
                        nn::util::VectorSubtract( &prevVec, pNowHistory->pos, pNowHistory->pNext->pos );

                        if( nn::util::VectorLengthSquared( prevVec ) < Epsilon )
                        {
                            // 前回の点も動いてないなら引継ぎ
                            detail::Vector3fCopy( &pNowHistory->wing, pNowHistory->pNext->wing );
                            detail::Vector3fCopy( &pNowHistory->dir, pNowHistory->pNext->dir );
                        }
                        else
                        {
                            // ある程度動いている場合はその速度を向きとして展開方向を計算。
                            const nn::util::Vector3fType axisY = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 1, 0 );
                            nn::util::Vector3fType dir = prevVec;
                            nn::util::VectorNormalize( &dir, dir );
                            const float d = nn::util::VectorDot( dir, axisY );
                            if( ::std::abs( ::std::abs( d ) - 1.0f ) < Epsilon )
                            {
                                nn::util::Vector3fType wing = NN_UTIL_VECTOR_3F_INITIALIZER( 1, 0, 0 );
                                nn::util::VectorMultiply( &wing, wing, d );
                                nn::util::VectorNormalize( &wing, wing );
                                detail::Vector3fCopy( &pNowHistory->wing, wing );
                            }
                            else
                            {
                                nn::util::Vector3fType wing;
                                nn::util::VectorCross( &wing, dir, axisY );
                                nn::util::VectorNormalize( &wing, wing );
                                detail::Vector3fCopy( &pNowHistory->wing, wing );
                            }
                        }
                    }
                    else
                    {
                        // pNowHistory->dir が 非零の場合
                        nn::util::Vector3fType nowDir = pNowHistory->dir;
                        nn::util::Vector3fType prevDir = pNowHistory->pNext->dir;
                        nn::util::Vector3fType prevWingDir = pNowHistory->pNext->wing;
                        nn::util::VectorNormalize( &nowDir, nowDir );
                        nn::util::VectorNormalize( &prevDir, prevDir );
                        nn::util::VectorNormalize( &prevWingDir, prevWingDir );

                        // 前回の法線方向
                        nn::util::Vector3fType prevNormal;
                        nn::util::VectorCross( &prevNormal, prevWingDir, prevDir );
                        nn::util::VectorNormalize( &prevNormal, prevNormal );

                        // 法線→進行方向から展開方向を決定
                        nn::util::Vector3fType wing;
                        nn::util::VectorCross( &wing, nowDir, prevNormal );
                        nn::util::VectorNormalize( &wing, wing );
                        detail::Vector3fCopy( &pNowHistory->wing, wing );
                    }

                    if( pStripe->historyCount == 2 )
                    {
                        if( arg.pEmitter->GetResEmitter()->ptcl.isRotateZ )
                        {
                            // Zロールで回す。
                            // pNowHistory->dir を軸に 一定量回転させる。
                            // 必ず背骨の向きが必要なので、ストライプが描画され始める直前であるこのタイミングで行う。
                            nn::util::VectorNormalize( &pNowHistory->dir, pNowHistory->dir );
                            nn::util::VectorNormalize( &pNowHistory->wing, pNowHistory->wing );
                            nn::util::Vector4f random;
                            arg.GetRandom( &random );

                            const float randomInitZ = arg.pEmitter->GetEmitterResource()->m_ResEmitterStaticConstantBuffer->rotateInitRand.z;
                            const float rz = arg.pEmitter->GetEmitterResource()->m_ResEmitterStaticConstantBuffer->rotateInit.z + nn::util::VectorGetX( random ) * randomInitZ;
                            const float ax = nn::util::VectorGetX( pNowHistory->dir );
                            const float ay = nn::util::VectorGetY( pNowHistory->dir );
                            const float az = nn::util::VectorGetZ( pNowHistory->dir );

                            const float theta = rz;

                            const float c = nn::util::CosEst( theta );
                            const float s = nn::util::SinEst( theta );
                            nn::util::Matrix4x3fType mat = NN_UTIL_MATRIX_4X3F_INITIALIZER(
                                ax*ax*( 1 - c ) + c, ax*ay*( 1 - c ) - az*s, az*ax*( 1 - c ) + ay*s,
                                ax*ay*( 1 - c ) + az*s, ay*ay*( 1 - c ) + c, ay*az*( 1 - c ) - ax*s,
                                az*ax*( 1 - c ) - ay*s, ay*az*( 1 - c ) + ax*s, az*az*( 1 - c ) + c,
                                0, 0, 0
                            );
                            nn::util::Vector3fType temp;
                            detail::Vector3fCopy( &temp, pNowHistory->wing );
                            nn::util::VectorTransform( &temp, temp, mat );
                            detail::Vector3fCopy( &pNowHistory->wing, temp );
                        }

                        // 二つ目の履歴点（初めてストライプが見えるタイミング）の時、
                        // 初回の履歴点の wing を今回のモノと揃えておく。
                        pNowHistory->pNext->wing = pNowHistory->wing;
                        pNowHistory->pNext->dir = pNowHistory->dir;
                        nn::util::VectorNormalize( &pNowHistory->pNext->wing, pNowHistory->pNext->wing );
                        nn::util::VectorNormalize( &pNowHistory->pNext->dir, pNowHistory->pNext->dir );
                    }
                }

                // 最終的に決まったベクトルをここで正規化
                nn::util::VectorNormalize( &pNowHistory->dir, pNowHistory->dir );
                nn::util::VectorNormalize( &pNowHistory->wing, pNowHistory->wing );
            }

            //------------------------------------------------------------------------------
            // 移動距離の更新
            //------------------------------------------------------------------------------
            {
                if( pStripe->historyCount == 1 )
                {
                    pStripe->journey += nn::util::VectorLength( pNowHistory->pos );
                }
                else
                {
                    nn::util::Vector3fType prevPos;
                    prevPos = pPrevHistory->pos;
                    nn::util::Vector3fType temp;
                    nn::util::VectorSubtract( &temp, pNowHistory->pos, prevPos );
                    pStripe->journey += nn::util::VectorLength( temp );
                }
            }
        }
    }
} // NOLINT(readability/fn_size)

//---------------------------------------------------------------------------
//  遅延ストライプ計算処理を行います。
//---------------------------------------------------------------------------
void SuperStripeSystem::CalculateDelayedStripe( Emitter* pEmitter ) NN_NOEXCEPT
{
    EmitterPluginUserData* pEPUserData = SuperStripeSystem::GetEmitterUserData( pEmitter );
    const float frameRate = pEmitter->GetFrameRate();
    if( !pEPUserData->pDelayedStripeListHead )
    {
        return;
    }
    if( pEPUserData->vertexBufferInitialized == false )
    {
        return;
    }

    StripeInstance* pPreStripe = NULL;
    StripeInstance* pStripe = pEPUserData->pDelayedStripeListHead;
    ResStripeSuper* pStripeData = GetStripeData( pEmitter );
    VertexAttribute* pVertexBufferHead = reinterpret_cast< VertexAttribute* >( pEmitter->GetGfxObjects()->m_ParticleEmitterPluginAttribute.Map( pEPUserData->bufferSide ) );

    while( pStripe )
    {
        VertexAttribute* pVertexAttribute = &pVertexBufferHead[ pStripe->indexOffset * pEPUserData->vertexCountPerStripe ];
        StripeInstance* pNextStripe = pStripe->pNextDelayedStripe;

        //---------------------------------------------------------------------
        // 履歴を動かす
        //---------------------------------------------------------------------
        {
            const float histAirRegist = pStripeData->historyAirRegist;
            const float invRate = 1.0f - frameRate;
            const float airRegist = histAirRegist + ( 1.0f - histAirRegist ) * invRate;

            // 履歴速度を更新（加速度）
            History* pHistory = pStripe->pHistoryHead->pNext;
            for( int i = 0; i < pStripe->historyCount; ++i, pHistory = pHistory->pNext )
            {
                nn::util::Vector3fType temp;
                nn::util::VectorLoad( &temp, pStripeData->historyAcceleration.v );
                nn::util::VectorMultiply( &temp, temp, frameRate );
                nn::util::VectorAdd( &pHistory->vec, pHistory->vec, temp );
                nn::util::VectorMultiply( &pHistory->vec, pHistory->vec, airRegist );
            }

            // 履歴位置を更新
            pHistory = pStripe->pHistoryHead->pNext;
            for( int i = 0; i < pStripe->historyCount; ++i, pHistory = pHistory->pNext )
            {
                nn::util::Vector3fType temp;
                nn::util::VectorMultiply( &temp, pHistory->vec, frameRate );
                nn::util::VectorAdd( &pHistory->pos, pHistory->pos, temp );
            }
        }

        //---------------------------------------------------------------------
        // 履歴を取り除くかどうかチェック
        //---------------------------------------------------------------------
        pStripe->time += pEmitter->GetFrameRate();
        const float testPrevFrame = ::std::floorf( pStripe->prevFrame );
        const float testNowFrame = ::std::floorf( pStripe->time );
        const bool doRemoveHistory = ( testNowFrame - testPrevFrame > 0 );
        pStripe->prevFrame = pStripe->time; // ストライプ時間（前回更新）を記憶

        //---------------------------------------------------------------------------
        // 死亡判定
        //---------------------------------------------------------------------------
        if( pStripe->historyCount > 1 )
        {
            if( doRemoveHistory )
            {
                pStripe->historyCount--;
            }

            float uvIndexOffset = static_cast< float >( pStripe->delayedHistoryCount - pStripe->historyCount );
            if( pStripeData->numDivide == 0 )
            {
                // 分割無し
                MakeStripePolygon( pVertexAttribute, pStripe, pEmitter, pStripeData, uvIndexOffset );
            }
            else
            {
                // 分割あり
                uvIndexOffset *= ( pStripeData->numDivide + 1 );
                MakeStripePolygonWithDivision( pVertexAttribute, pStripe, pEmitter, pStripeData, uvIndexOffset );
            }

            pPreStripe = pStripe;
            pStripe = pNextStripe;
        }
        else
        {
            if( pPreStripe )
            {
                pPreStripe->pNextDelayedStripe = pNextStripe;
            }
            else
            {
                pEPUserData->pDelayedStripeListHead = pNextStripe;
            }
            StripeSystemUtility::FreeStripe( g_pStripeSystem, pStripe, pEmitter );
            pStripe = pNextStripe;
        }
    }
    pEmitter->GetGfxObjects()->m_ParticleEmitterPluginAttribute.Unmap();
}

//---------------------------------------------------------------------------
//  ストライプのポリゴン生成を行います。
//---------------------------------------------------------------------------
void SuperStripeSystem::MakeStripePolygon(
    VertexAttribute* pOutVertexAttribute,
    StripeInstance* pStripe,
    const Emitter* pEmitter,
    const ResStripeSuper* pStripeData,
    float indexOffset ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pOutVertexAttribute );
    NN_SDK_REQUIRES_NOT_NULL( pStripe );
    NN_SDK_REQUIRES_NOT_NULL( pEmitter );
    NN_SDK_REQUIRES_NOT_NULL( pStripeData );

    //---------------------------------------------------------------------------
    // 履歴ポイントが2つは無いとポリゴンにならないので処理を返す。
    //---------------------------------------------------------------------------
    const int historyCount = pStripe->historyCount;
    if( historyCount <= 1 )
    {
        pStripe->renderVertexCount = 0;
        return;
    }
    const History* pHistory = pStripe->pHistoryHead->pNext;    // 先頭は空なのでその次から。

    //---------------------------------------------------------------------------
    // ストライプの全長を求める
    //---------------------------------------------------------------------------
    float stripeLengthInv = 0.0f;
    if( pStripeData->textureUvMapType == TextureMappingType_DistanceBased )
    {
        stripeLengthInv = 1.0f / CalculateStripeLength( pHistory->pNext, historyCount );
    }

    //---------------------------------------------------------------------------
    // スケールフェード値を計算
    //---------------------------------------------------------------------------
    const float scaleFadeValue = pEmitter->CalculateCurrentScaleFadeValue();

    //---------------------------------------------------------------------------
    // テクスチャオフセット値の計算
    //---------------------------------------------------------------------------
    const float texOffset = Fract( pStripe->time ) + indexOffset;

    //---------------------------------------------------------------------------
    // 頂点バッファの必須の要素を埋める。
    //---------------------------------------------------------------------------
    int index = 0;
    for( index = 0; index < historyCount; ++index, pHistory = pHistory->pNext )
    {
        nn::util::Vector3fType textureUv = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 0, 0 );
        switch( pStripeData->textureUvMapType )
        {
        case TextureMappingType_Uniform:
            {
                const int maxVertexCount = static_cast< int >( pStripeData->numHistory );
                CalculateTextureOffsetUniform( &textureUv, pStripeData, index, historyCount, maxVertexCount, texOffset );
                break;
            }
        case TextureMappingType_DistanceBased:
            {
                nn::util::Vector3fType sub;
                nn::util::VectorSubtract( &sub, pHistory->pPrev->pos, pHistory->pos );

                int prevVertexIndex = ( index * 2 ) - 2;
                if( prevVertexIndex < 0 )
                {
                    prevVertexIndex = 0;
                }
                const float delta = nn::util::VectorLength( sub );
                const nn::util::Float4& preTex = pOutVertexAttribute[ prevVertexIndex ].tex;

                CalculateTextureOffsetDistanceBased( &textureUv, pStripeData, index, delta, pStripe->journey, stripeLengthInv, preTex );
                break;
            }
        default:
            NN_SDK_ASSERT( 0 ); // ここには来ない！
            break;
        }

        // 新アトリビュートに全部まとめる
        MakeDefaultVertexAttribute( pOutVertexAttribute, index, pHistory->pos, pHistory->dir, pHistory->wing, textureUv, pHistory->scale * scaleFadeValue );
    }
    pStripe->renderVertexCount = index * 2;

    //---------------------------------------------------------------------------
    // Ematが必要なストライプタイプの場合、emat要素も埋める。
    //---------------------------------------------------------------------------
    if( pStripeData->calcType != StripeOrientationType_Billboard && pStripeData->calcType != StripeOrientationType_Ribbon )
    {
        pHistory = pStripe->pHistoryHead->pNext;
        for( index = 0; index < historyCount; ++index, pHistory = pHistory->pNext )
        {
            const int iLeft = index * 2;
            const int iRight = iLeft + 1;
            nn::util::VectorStore( pOutVertexAttribute[ iLeft ].emat.v, pHistory->emat );
            nn::util::VectorStore( pOutVertexAttribute[ iRight ].emat.v, pHistory->emat );
        }
    }

    return;
}

//---------------------------------------------------------------------------
//  ストライプのポリゴン生成を行います。（ストライプ分割版）
//---------------------------------------------------------------------------
void SuperStripeSystem::MakeStripePolygonWithDivision(
    VertexAttribute* pOutVertexAttribute,
    StripeInstance* pStripe,
    const Emitter* pEmitter,
    const ResStripeSuper* pStripeData,
    float indexOffset ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pOutVertexAttribute );
    NN_SDK_REQUIRES_NOT_NULL( pStripe );
    NN_SDK_REQUIRES_NOT_NULL( pEmitter );
    NN_SDK_REQUIRES_NOT_NULL( pStripeData );

    //---------------------------------------------------------------------------
    // 履歴ポイントが3つは無いとポリゴンにならないので処理を返す。
    //---------------------------------------------------------------------------
    const int historyCount = pStripe->historyCount;
    if( historyCount <= 2 )
    {
        pStripe->renderVertexCount = 0;
        return;
    }
    const History* pHistory = pStripe->pHistoryHead->pNext;                         // 先頭は空なのでその次から。

    //---------------------------------------------------------------------------
    // ストライプの全長を求める
    //---------------------------------------------------------------------------
    float stripeLengthInv = 0.0f;
    if( pStripeData->textureUvMapType == TextureMappingType_DistanceBased )
    {
        stripeLengthInv = 1.0f / CalculateStripeLength( pHistory->pNext, historyCount );
    }

    //---------------------------------------------------------------------------
    // スケールフェード値を計算
    //---------------------------------------------------------------------------
    const float scaleFadeValue = pEmitter->CalculateCurrentScaleFadeValue();

    //---------------------------------------------------------------------------
    // テクスチャオフセット値の計算
    //---------------------------------------------------------------------------
    const float texOffset = Fract( pStripe->time ) + indexOffset;

    //---------------------------------------------------------------------------
    // 頂点バッファの必須の要素を埋める。
    //---------------------------------------------------------------------------
    const int divCount          = static_cast< int >( pStripeData->numDivide );         // 分割数。
    const int drawHistoryCount  = historyCount + ( ( historyCount - 1 ) * divCount );   // 分割分の頂点を追加。
    const float fIndexDiv       = 1.0f / ( 1 + divCount );                              // 小数部分のインデックス差分

    //---------------------------------------------------------------------------
    // 頂点バッファの必須の要素を埋める。
    //---------------------------------------------------------------------------
    int index = 0;
    int prevPos = 0;
    nn::util::Vector3fType prevVertexPos = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 0, 0 );
    for( index = 0; index < drawHistoryCount; ++index )
    {
        const float fIndex = index * fIndexDiv;                             // ストライプ状の位置（履歴ポイントが整数になる）
        const int histPos = static_cast< int >( ::std::floorf( fIndex ) );  // 所属する履歴ポイント
        if( histPos > prevPos )
        {
            pHistory = pHistory->pNext;
            prevPos = histPos;
        }
        const float innerPos = fIndex - histPos;    // 履歴ポイント内の位置 [0.0f, 1.0f]

        //-----------------------------------------------------------------
        // 履歴点の間をエルミート補間
        //-----------------------------------------------------------------
        nn::util::Vector3fType curvePos;    // 履歴ポイントの位置（エルミート補間後）
        nn::util::Vector3fType curveDir;    // 履歴ポイントの向き（線形補間後）
        StripeSystemUtility::CalculateHermiteInterpolatedCurveVec<SuperStripeSystem>( &curvePos, &curveDir, pHistory, histPos, historyCount, innerPos );

        // テクスチャ座標
        nn::util::Vector3fType textureUv = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 0, 0 );
        {
            switch( pStripeData->textureUvMapType )
            {
            case TextureMappingType_Uniform:
                {
                    const int numHistory = static_cast< int >( pStripeData->numHistory );
                    const int maxVertexCount = ( numHistory + ( ( numHistory - 1 ) * divCount ) );
                    CalculateTextureOffsetUniform( &textureUv, pStripeData, index, drawHistoryCount, maxVertexCount, texOffset );
                    break;
                }
            case TextureMappingType_DistanceBased:
                {
                    float delta = 0.0f;
                    if( index > 0 )
                    {
                        nn::util::Vector3fType sub;
                        nn::util::VectorSubtract( &sub, curvePos, prevVertexPos );
                        delta = nn::util::VectorLength( sub );
                    }

                    int prevVertexIndex = ( index * 2 ) - 2;
                    if( prevVertexIndex < 0 )
                    {
                        prevVertexIndex = 0;
                    }
                    const nn::util::Float4& preTex = pOutVertexAttribute[ prevVertexIndex ].tex;

                    CalculateTextureOffsetDistanceBased( &textureUv, pStripeData, index, delta, pStripe->journey, stripeLengthInv, preTex );

                    // ひとつ前の頂点位置を記憶
                    prevVertexPos = curvePos;
                    break;
                }
            default:
                NN_SDK_ASSERT( 0 ); // ここには来ない！
                break;
            }
        }

        // 新アトリビュートに全部まとめる
        MakeDefaultVertexAttribute( pOutVertexAttribute, index, curvePos, curveDir, pHistory->wing, textureUv, pHistory->scale * scaleFadeValue );
    }
    pStripe->renderVertexCount = index * 2;

    //---------------------------------------------------------------------------
    // Ematが必要なストライプタイプの場合、emat要素も埋める。
    //---------------------------------------------------------------------------
    if( pStripeData->calcType != StripeOrientationType_Billboard && pStripeData->calcType != StripeOrientationType_Ribbon )
    {
        pHistory = pStripe->pHistoryHead->pNext;
        prevPos = 0;
        for( index = 0; index < drawHistoryCount; ++index )
        {
            const int iLeft = index * 2;
            const int iRight = iLeft + 1;

            const float fIndex = index * fIndexDiv;                             // ストライプ状の位置（履歴ポイント場が整数になる）
            const int histPos = static_cast< int >( ::std::floorf( fIndex ) );  // 所属する履歴ポイント
            if( histPos > prevPos )
            {
                pHistory = pHistory->pNext;
                prevPos = histPos;
            }

#ifndef _VFX_NSDK01215_COMPATIBLE
            // emat 要素は線形補間＋ Normalize する
            const float innerPos = fIndex - histPos;
            nn::util::Vector3fType emat;
            nn::util::VectorLerp( &emat, pHistory->emat, pHistory->pNext->emat, innerPos );
            nn::util::VectorNormalize( &emat, emat );

            nn::util::VectorStore( pOutVertexAttribute[ iLeft ].emat.v, emat );
            nn::util::VectorStore( pOutVertexAttribute[ iRight ].emat.v, emat );
#else
            nn::util::VectorStore( pOutVertexAttribute[ iLeft ].emat.v, pHistory->emat );
            nn::util::VectorStore( pOutVertexAttribute[ iRight ].emat.v, pHistory->emat );
#endif
        }
    }

    return;
} // NOLINT(impl/function_size)

//------------------------------------------------------------------------------
//  ワンタイムエミッタ用の追加のエミッタ寿命を取得
//------------------------------------------------------------------------------
int SuperStripeSystem::GetExtendedEndTimeForOneTimeEmitter( nn::vfx::Emitter* pEmitter ) NN_NOEXCEPT
{
    ResStripeSuper* pStripeData = GetStripeData( pEmitter );
    float extendLife = pStripeData->numHistory;
    if( pEmitter->GetFrameRate() > 1.0 )
    {
        extendLife *= pEmitter->GetFrameRate();
    }
    return static_cast< int >( extendLife );
}

//------------------------------------------------------------------------------
//  ストライプの正確な再生本数を取得
//------------------------------------------------------------------------------
int SuperStripeSystem::GetActualSuperStripeCalclationCount( const nn::vfx::Emitter* pEmitter ) NN_NOEXCEPT
{
    EmitterPluginUserData* pEPUserData = SuperStripeSystem::GetEmitterUserData( pEmitter );
    if ( pEPUserData->vertexBufferInitialized == false )
    {
        return 0;
    }

    int actualSuperStripeCount = pEmitter->GetProcessingParticleCount();

    StripeInstance* pStripe = pEPUserData->pDelayedStripeListHead;
    while ( pStripe )
    {
        actualSuperStripeCount++;
        pStripe = pStripe->pNextDelayedStripe;
    }

    return actualSuperStripeCount;
}

//------------------------------------------------------------------------------
//! @brief パーティクル計算コールバック
//------------------------------------------------------------------------------
void SuperStripeSystem::ParticleCalculateCallback( ParticleCalculateArg& arg ) NN_NOEXCEPT
{
    StripeInstance* stripe = reinterpret_cast< StripeInstance* >( arg.pEmitterPluginData );
    EmitterPluginUserData* epData = SuperStripeSystem::GetEmitterUserData( arg.pEmitter );
    if( stripe == NULL )
    {
        OutputWarning( "SuperStripe instance is null\n" );
        return;
    }
    else if( stripe->used == false )
    {
        OutputWarning( "SuperStripe instance is not used\n" );
        return;
    }
    if( epData == NULL )
    {
        OutputWarning( "EmitterPluginUserData is empty\n" );
        return;
    }
    else if( epData->vertexBufferInitialized == false )
    {
        OutputWarning( "Vertex buffer for stripe has not allocated\n" );
        return;
    }
    g_pStripeSystem->CalculateStripe( arg, stripe );
}

//------------------------------------------------------------------------------
//! @brief エミッタ計算処理前コールバック
//------------------------------------------------------------------------------
void SuperStripeSystem::EmitterPreCalculateCallback( EmitterPreCalculateArg& arg ) NN_NOEXCEPT
{
    if( arg.pEmitter->GetCalculationType() == EmitterCalculationMode_Cpu )
    {
        EmitterPluginUserData* epData = SuperStripeSystem::GetEmitterUserData( arg.pEmitter );
        StripeSystemUtility::SwapBufferSide( &epData->bufferSide, arg.pEmitter );
    }
}

//------------------------------------------------------------------------------
//! @brief エミッタ計算処理後コールバック
//------------------------------------------------------------------------------
void SuperStripeSystem::EmitterPostCalculateCallback( EmitterPostCalculateArg& arg ) NN_NOEXCEPT
{
    if( arg.pEmitter->GetCalculationType() == EmitterCalculationMode_Cpu )
    {
        g_pStripeSystem->CalculateDelayedStripe( arg.pEmitter );
    }
}

//------------------------------------------------------------------------------
//! @brief エミッタ描画コールバック
//------------------------------------------------------------------------------
bool SuperStripeSystem::EmitterDrawCallback( EmitterDrawArg& arg ) NN_NOEXCEPT
{
    if( arg.pEmitter->GetCalculationType() == EmitterCalculationMode_Cpu )
    {
        StripeSystemUtility::DrawParticleStripeEmitter< SuperStripeSystem >(
            arg.pCommandBuffer, g_pStripeSystem->m_pSystem, arg.pEmitter, arg.shaderType, arg.pUserParam, arg.pDrawParameterArg );
        return true;
    }
    return false;
}

//---------------------------------------------------------------------------
//! @brief  定数バッファ を作成します。
//---------------------------------------------------------------------------
void SuperStripeSystem::MakeConstantBufferObject(
    ConstantBufferObject* const pConstantBuffer,
    const ResStripeSuper* pStripeData,
    const EmitterPluginUserData* pEPUserData,
    const StripeInstance* pStripe,
    const Emitter* pEmitter,
    StripeMeshType meshType ) NN_NOEXCEPT
{
    NN_UNUSED( pEmitter );

    // 静的な要素
    nn::util::VectorStore( &pConstantBuffer->random, pStripe->random );
    pConstantBuffer->param0 = pEPUserData->staticParam0;

    // 動的な要素
    nn::util::VectorStore( &pConstantBuffer->color0, pStripe->color0 );
    nn::util::VectorStore( &pConstantBuffer->color1, pStripe->color1 );
    pConstantBuffer->param1.x = pStripe->time;
    pConstantBuffer->param1.y = static_cast< float >( pStripe->historyCount + pStripeData->numDivide * ( pStripe->historyCount - 1 ) );
    pConstantBuffer->param1.z = static_cast< float >( meshType );
    pConstantBuffer->param1.w = pStripe->life;
#ifndef _VFX_NSDK01215_COMPATIBLE
    pConstantBuffer->param2.x = nn::util::VectorGetX( pStripe->headPos );
    pConstantBuffer->param2.y = nn::util::VectorGetY( pStripe->headPos );
    pConstantBuffer->param2.z = nn::util::VectorGetZ( pStripe->headPos );
    pConstantBuffer->param2.w = 0.0f;
#else
    pConstantBuffer->param2.x = 0.0f;
    pConstantBuffer->param2.y = 0.0f;
    pConstantBuffer->param2.z = 0.0f;
    pConstantBuffer->param2.w = 0.0f;
#endif
}

//------------------------------------------------------------------------------
//! @brief  ストライプシステムで確保したワークサイズを取得します。
//------------------------------------------------------------------------------
size_t SuperStripeSystem::GetWorkSize() NN_NOEXCEPT
{
    if( g_pStripeSystem == NULL )
    {
        return 0;
    }
    return g_pStripeSystem->m_StripeWorkSize;
}

//---------------------------------------------------------------------------
//! @brief  現在のストライプ数取得を行います。
//---------------------------------------------------------------------------
int SuperStripeSystem::GetProcessingStripeCount() NN_NOEXCEPT
{
    if( g_pStripeSystem == NULL )
    {
        return 0;
    }
    return g_pStripeSystem->m_StripeCount;
}

//---------------------------------------------------------------------------
//  UV割り当て: 「履歴毎に固定割合」
//
//  【例】
//  履歴点:   (P0)----(P1)------------(P2)----(P3)----(P4)
//  UV 座標:  0.00    0.25            0.50    0.75    1.00
//
//  要するに従来通りの仕様。
//---------------------------------------------------------------------------
void SuperStripeSystem::CalculateTextureOffsetUniform(
    nn::util::Vector3fType* pOutTexOffset,  // 【共通】出力用 テクスチャオフセット（0/1/2）
    const ResStripeSuper* pStripeData,      // 【共通】ストライプデータ
    int index,                              // 【共通】頂点インデックス
    int vertexCount,                        // 【固定割当】頂点数
    int maxVertexCount,                     // 【固定割当】頂点の最大数
    float indexOffset                       // 【固定割当】インデックスのオフセット（含む小数）
) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pOutTexOffset );
    NN_SDK_REQUIRES_NOT_NULL( pStripeData );

    nn::util::Float3 offsetBase[ StripeTexturingOption_MaxType ];

    offsetBase[ StripeTexturingOption_Fill ].x =
    offsetBase[ StripeTexturingOption_Fill ].y =
    offsetBase[ StripeTexturingOption_Fill ].z = CalculateTextureOffsetUniformFill( index, vertexCount, indexOffset );
    offsetBase[ StripeTexturingOption_Tile ].x =
    offsetBase[ StripeTexturingOption_Tile ].y =
    offsetBase[ StripeTexturingOption_Tile ].z = CalculateTextureOffsetUniformTile( index, maxVertexCount, indexOffset );

    // テクスチャリング設定を見て必要な方を取る
    nn::util::Float3 offset = NN_UTIL_FLOAT_3_INITIALIZER(
        offsetBase[ pStripeData->texturing0 ].x,
        offsetBase[ pStripeData->texturing1 ].y,
        offsetBase[ pStripeData->texturing2 ].z );

    nn::util::VectorLoad( pOutTexOffset, offset );
}

//---------------------------------------------------------------------------
//  UV割り当て: 「履歴距離に依存」
//
//  【例】
//  履歴点:   (P0)----(P1)------------(P2)----(P3)----(P4)
//  UV 座標:  0.00    0.20            0.60    0.80    1.00
//
//  距離に応じてUV加算を調整する。
//  描画範囲に合わせて貼る場合は、ワールド空間に貼られたテクスチャをストライプがマスクするイメージ
//---------------------------------------------------------------------------
void SuperStripeSystem::CalculateTextureOffsetDistanceBased(
    nn::util::Vector3fType* pOutTexOffset,  // 【共通】出力用 テクスチャオフセット（0/1/2）
    const ResStripeSuper* pStripeData,      // 【共通】ストライプデータ
    int index,                              // 【共通】頂点インデックス
    float delta,                            // 【距離依存】履歴点の差分距離
    float journey,                          // 【距離依存】ストライプの通算移動距離
    float invStripeLength,                  // 【距離依存】ストライプの折れ線の長さの逆数
    const nn::util::Float4& prevOffset      // 【距離依存】ひとつ前の頂点のオフセット
) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pOutTexOffset );
    NN_SDK_REQUIRES_NOT_NULL( pStripeData );

    nn::util::Float3 offsetBase[ StripeTexturingOption_MaxType ];

    // 直前のUVに加算
    offsetBase[ StripeTexturingOption_Fill ].x = CalculateTextureOffsetDistanceBasedFill( index, prevOffset.x, delta, invStripeLength );
    offsetBase[ StripeTexturingOption_Fill ].y = CalculateTextureOffsetDistanceBasedFill( index, prevOffset.y, delta, invStripeLength );
    offsetBase[ StripeTexturingOption_Fill ].z = CalculateTextureOffsetDistanceBasedFill( index, prevOffset.z, delta, invStripeLength );
    offsetBase[ StripeTexturingOption_Tile ].x = CalculateTextureOffsetDistanceBasedTile( index, prevOffset.x, delta, journey );
    offsetBase[ StripeTexturingOption_Tile ].y = CalculateTextureOffsetDistanceBasedTile( index, prevOffset.y, delta, journey );
    offsetBase[ StripeTexturingOption_Tile ].z = CalculateTextureOffsetDistanceBasedTile( index, prevOffset.z, delta, journey );

    // テクスチャリング設定を見て必要な方を取る
    nn::util::Float3 offset = NN_UTIL_FLOAT_3_INITIALIZER(
        offsetBase[ pStripeData->texturing0 ].x,
        offsetBase[ pStripeData->texturing1 ].y,
        offsetBase[ pStripeData->texturing2 ].z );

    nn::util::VectorLoad( pOutTexOffset, offset );
}

//---------------------------------------------------------------------------
//! @brief  ストライプの頂点バッファの共通部分を作成
//---------------------------------------------------------------------------
void SuperStripeSystem::MakeDefaultVertexAttribute(
    SuperStripeSystem::VertexAttribute* pOutVertexAttribute,
    int index,
    const nn::util::Vector3fType& pos,
    const nn::util::Vector3fType& dir,
    const nn::util::Vector3fType& wing,
    const nn::util::Vector3fType& textureOffset,
    float scaleFadeValue ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pOutVertexAttribute );

    const int iLeft = index * 2;    // 頂点（右側）Index
    const int iRight = iLeft + 1;   // 頂点（左側）Index

    // 履歴の位置xyz / 羽根の長さ
    nn::util::VectorStore( pOutVertexAttribute[ iLeft ].pos.v, pos );
    nn::util::VectorStore( pOutVertexAttribute[ iRight ].pos.v, pos );
    pOutVertexAttribute[ iLeft ].pos.w = scaleFadeValue;
    pOutVertexAttribute[ iRight ].pos.w = -scaleFadeValue;

    // 背骨の方向（前の履歴との位置差分）
    nn::util::VectorStore( pOutVertexAttribute[ iLeft ].dir.v, dir );
    nn::util::VectorStore( pOutVertexAttribute[ iRight ].dir.v, dir );

    // 羽根の方向
    nn::util::VectorStore( pOutVertexAttribute[ iLeft ].wing.v, wing );
    nn::util::VectorStore( pOutVertexAttribute[ iRight ].wing.v, wing );

    // テクスチャリング
    nn::util::VectorStore( pOutVertexAttribute[ iLeft ].tex.v, textureOffset );
    nn::util::VectorStore( pOutVertexAttribute[ iRight ].tex.v, textureOffset );

    // 頂点インデックス
    pOutVertexAttribute[ iRight ].dir.w = pOutVertexAttribute[ iLeft ].dir.w = static_cast< float >( index );
}

} // namespace detail
} // namespace vfx
} // namespace nn
