﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vfx/vfx_ShaderManager.h>

namespace nn  {
namespace vfx {
namespace detail {


//---------------------------------------------------------------------------
//  コンストラクタ
//---------------------------------------------------------------------------
ShaderManager::ShaderManager() NN_NOEXCEPT
{
    m_IsInitialized         = false;
    m_ShaderArray           = NULL;
    m_ShaderArrayPtr        = NULL;
    m_ShaderCount           = 0;
    m_pResShaderContainer   = NULL;
    m_GfxDevice             = nullptr;
    m_CodeType              = nn::gfx::ShaderCodeType_Binary;
}

//---------------------------------------------------------------------------
//  初期化処理
//---------------------------------------------------------------------------
void ShaderManager::Initialize( nn::gfx::Device* pDevice, Heap* pHeap, nn::gfx::ResShaderContainer* shaderContainer, bool delaySetup ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( shaderContainer );
    m_pResShaderContainer = shaderContainer;

    // シェーダ総数
    m_ShaderCount = m_pResShaderContainer->GetShaderVariationCount();

    // 管理用バッファを確保
    size_t shaderArraySize = sizeof( Shader ) * m_ShaderCount + 32;   // +32は配列でのnewの為
    m_ShaderArrayPtr = pHeap->Alloc( shaderArraySize );
    m_ShaderArray = new ( m_ShaderArrayPtr )Shader[ m_ShaderCount ];

    if ( delaySetup )
    {
        m_GfxDevice = pDevice;

        for ( int i = 0; i < m_ShaderCount; ++i )
        {
            m_ShaderArray[ i ].Invalidate();
        }
    }
    else
    {
        m_GfxDevice = nullptr;

        // 各シェーダを初期化
        for ( int i = 0; i < m_ShaderCount; ++i )
        {
            nn::gfx::ShaderInitializeResult shaderResult = nn::gfx::ShaderInitializeResult_SetupFailed;
            NN_UNUSED( shaderResult ); // m_ShaderCountが0になるプラットフォームでの警告対応。

            nn::gfx::ResShaderVariation* pVariation = m_pResShaderContainer->GetResShaderVariation( i );
            NN_SDK_ASSERT_NOT_NULL( pVariation );

            nn::gfx::ShaderCodeType codeType = nn::gfx::ShaderCodeType_Binary;
            nn::gfx::ResShaderProgram* binaryProgram = pVariation->GetResShaderProgram( nn::gfx::ShaderCodeType_Binary );
            nn::gfx::ResShaderProgram* irProgram = pVariation->GetResShaderProgram( nn::gfx::ShaderCodeType_Ir );
            nn::gfx::ResShaderProgram* codeProgram = pVariation->GetResShaderProgram( nn::gfx::ShaderCodeType_Source );
            NN_SDK_ASSERT( binaryProgram || irProgram || codeProgram );

            if ( binaryProgram )
            {
                shaderResult = binaryProgram->Initialize( pDevice );
            }

            if ( shaderResult != nn::gfx::ShaderInitializeResult_Success )
            {
                if ( irProgram )
                {
                    shaderResult = irProgram->Initialize( pDevice );
                    codeType = nn::gfx::ShaderCodeType_Ir;
                }
            }

            if ( shaderResult != nn::gfx::ShaderInitializeResult_Success )
            {
                if ( codeProgram )
                {
                    shaderResult = codeProgram->Initialize( pDevice );
                    codeType = nn::gfx::ShaderCodeType_Source;
                }
            }
            NN_SDK_ASSERT_EQUAL( shaderResult, nn::gfx::ShaderInitializeResult_Success );

            m_ShaderArray[ i ].Invalidate();
            m_ShaderArray[ i ].InitializeShader( pVariation, codeType );
            m_ShaderArray[ i ].InitializeSamplerSlot();
            m_ShaderArray[ i ].InitializeConstantBufferSlot();

            m_CodeType = codeType;
        }
    }

    m_IsInitialized = true;
}

//---------------------------------------------------------------------------
//  終了処理
//---------------------------------------------------------------------------
void ShaderManager::Finalize( nn::gfx::Device* pDevice, Heap* pHeap ) NN_NOEXCEPT
{
    if( m_ShaderArray )
    {
        for( int i = 0; i < m_ShaderCount; i++ )
        {
            nn::gfx::ResShaderVariation* pVariation = m_pResShaderContainer->GetResShaderVariation( i );
            NN_SDK_ASSERT_NOT_NULL( pVariation );

            if ( m_ShaderArray[ i ].IsInitialized() )
            {
                pVariation->GetResShaderProgram( m_CodeType )->Finalize( pDevice );
                m_ShaderArray[ i ].Finalize();
            }
        }
        pHeap->Free( m_ShaderArrayPtr );

        m_ShaderArray    = NULL;
        m_ShaderArrayPtr = NULL;
    }
}

//---------------------------------------------------------------------------
//  シェーダを取得します。
//---------------------------------------------------------------------------
Shader* ShaderManager::GetShader( int index ) NN_NOEXCEPT
{
    if ( !m_IsInitialized )
    {
        return NULL;
    }

    NN_SDK_ASSERT( index < m_ShaderCount );
    NN_SDK_ASSERT_NOT_NULL( m_ShaderArray );

    if ( !m_ShaderArray[ index ].IsInitialized() )
    {
        NN_SDK_ASSERT_NOT_NULL( m_GfxDevice );

        nn::gfx::ShaderInitializeResult shaderResult = nn::gfx::ShaderInitializeResult_SetupFailed;
        NN_UNUSED( shaderResult ); // m_ShaderCountが0になるプラットフォームでの警告対応。

        nn::gfx::ResShaderVariation* pVariation = m_pResShaderContainer->GetResShaderVariation( index );
        NN_SDK_ASSERT_NOT_NULL( pVariation );

        nn::gfx::ShaderCodeType codeType = nn::gfx::ShaderCodeType_Binary;
        nn::gfx::ResShaderProgram* binaryProgram = pVariation->GetResShaderProgram( nn::gfx::ShaderCodeType_Binary );
        nn::gfx::ResShaderProgram* irProgram = pVariation->GetResShaderProgram( nn::gfx::ShaderCodeType_Ir );
        nn::gfx::ResShaderProgram* codeProgram = pVariation->GetResShaderProgram( nn::gfx::ShaderCodeType_Source );
        NN_SDK_ASSERT( binaryProgram || irProgram || codeProgram );

        if ( binaryProgram )
        {
            shaderResult = binaryProgram->Initialize( m_GfxDevice );
        }

        if ( shaderResult != nn::gfx::ShaderInitializeResult_Success )
        {
            if ( irProgram )
            {
                shaderResult = irProgram->Initialize( m_GfxDevice );
                codeType = nn::gfx::ShaderCodeType_Ir;
            }
        }

        if ( shaderResult != nn::gfx::ShaderInitializeResult_Success )
        {
            if ( codeProgram )
            {
                shaderResult = codeProgram->Initialize( m_GfxDevice );
                codeType = nn::gfx::ShaderCodeType_Source;
            }
        }
        NN_SDK_ASSERT_EQUAL( shaderResult, nn::gfx::ShaderInitializeResult_Success );

        m_ShaderArray[ index ].Invalidate();
        m_ShaderArray[ index ].InitializeShader( pVariation, codeType );
        m_ShaderArray[ index ].InitializeSamplerSlot();
        m_ShaderArray[ index ].InitializeConstantBufferSlot();

        m_CodeType = codeType;
    }

    return &m_ShaderArray[ index ];
}


//---------------------------------------------------------------------------
//  コンストラクタ
//---------------------------------------------------------------------------
ComputeShaderManager::ComputeShaderManager() NN_NOEXCEPT
{
    m_IsInitialized     = false;
    m_ShaderArray       = NULL;
    m_ShaderArrayPtr    = NULL;
    m_ShaderCount       = 0;
}

//---------------------------------------------------------------------------
//  初期化処理
//---------------------------------------------------------------------------
void ComputeShaderManager::Initialize( nn::gfx::Device* pDevice, Heap* pHeap, nn::gfx::ResShaderContainer* shaderContainer ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( shaderContainer );
    m_pResShaderContainer = shaderContainer;

    // シェーダ総数
    m_ShaderCount = m_pResShaderContainer->GetShaderVariationCount();

    // 管理用バッファを確保
    size_t shaderArraySize = sizeof( ComputeShader ) * m_ShaderCount + 32;   // +32は配列でのnewの為
    m_ShaderArrayPtr = pHeap->Alloc( shaderArraySize );
    m_ShaderArray = new ( m_ShaderArrayPtr )ComputeShader[ m_ShaderCount ];

    // 各シェーダを初期化
    for ( int i = 0; i < m_ShaderCount; ++i )
    {
        nn::gfx::ShaderInitializeResult shaderResult = nn::gfx::ShaderInitializeResult_SetupFailed;
        NN_UNUSED( shaderResult ); // m_ShaderCountが0になるプラットフォームでの警告対応。

        nn::gfx::ResShaderVariation* pVariation = m_pResShaderContainer->GetResShaderVariation( i );
        NN_SDK_ASSERT_NOT_NULL( pVariation );

        nn::gfx::ShaderCodeType codeType = nn::gfx::ShaderCodeType_Binary;
        nn::gfx::ResShaderProgram* binaryProgram = pVariation->GetResShaderProgram( nn::gfx::ShaderCodeType_Binary );
        nn::gfx::ResShaderProgram* irProgram = pVariation->GetResShaderProgram( nn::gfx::ShaderCodeType_Ir );
        nn::gfx::ResShaderProgram* codeProgram = pVariation->GetResShaderProgram( nn::gfx::ShaderCodeType_Source );
        NN_SDK_ASSERT( binaryProgram || irProgram || codeProgram );

        if ( binaryProgram )
        {
            shaderResult = binaryProgram->Initialize( pDevice );
        }

        if ( shaderResult != nn::gfx::ShaderInitializeResult_Success )
        {
            if ( irProgram )
            {
                shaderResult = irProgram->Initialize( pDevice );
                codeType = nn::gfx::ShaderCodeType_Ir;
            }
        }

        if ( shaderResult != nn::gfx::ShaderInitializeResult_Success )
        {
            if ( codeProgram )
            {
                shaderResult = codeProgram->Initialize( pDevice );
                codeType = nn::gfx::ShaderCodeType_Source;
            }
        }
        NN_SDK_ASSERT_EQUAL( shaderResult, nn::gfx::ShaderInitializeResult_Success );

        nn::gfx::Shader* computeShader = pVariation->GetResShaderProgram( codeType )->GetShader();
        m_ShaderArray[ i ].Initialize( computeShader );

        m_CodeType = codeType;
    }

    m_IsInitialized = true;
}

//---------------------------------------------------------------------------
//  終了処理
//---------------------------------------------------------------------------
void ComputeShaderManager::Finalize( nn::gfx::Device* pDevice, Heap* pHeap ) NN_NOEXCEPT
{
    if ( m_ShaderArray )
    {
        for ( int i = 0; i < m_ShaderCount; i++ )
        {
            nn::gfx::ResShaderVariation* pVariation = m_pResShaderContainer->GetResShaderVariation( i );
            NN_SDK_ASSERT_NOT_NULL( pVariation );

            pVariation->GetResShaderProgram( m_CodeType )->Finalize( pDevice );
        }
        pHeap->Free( m_ShaderArrayPtr );

        m_ShaderArray = NULL;
        m_ShaderArrayPtr = NULL;
    }
}

} // namespace detail
} // namespace vfx
} // namespace nn
