﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vfx/vfx_TargetDef.h>
#include <nn/vfx/vfx_Misc.h>
#include <nn/vfx/vfx_MemUtil.h>
#include <nn/vfx/vfx_Heap.h>
#include <nn/vfx/vfx_EmitterCalc.h>
#include <nn/vfx/vfx_System.h>
#include <nn/vfx/vfx_EmitterSet.h>
#include <nn/vfx/vfx_Emitter.h>

namespace nn {
namespace vfx {
namespace detail {

size_t                  g_StaticHeapAllocatedSize = 0;
Heap*                   g_pStaticHeap = NULL;
Heap*                   g_pDynamicHeap = NULL;
bool                    g_SuppressLog = false;
OutputMessageCallback   g_LogCallback = NULL;
OutputMessageCallback   g_WarningCallback = NULL;
OutputMessageCallback   g_ErrorCallback = NULL;

void**      g_DelayFreeList;
int         g_DelayFreeListCount;
int         g_DelayFreeListMax = 0;
uint32_t    g_WarningFlag = 0;

detail::Mutex   g_MutexAddDelayFreeList;


//---------------------------------------------------------------------------
//  静的ヒープからメモリを設定します。
//---------------------------------------------------------------------------
void SetStaticHeap( Heap* pHeap ) NN_NOEXCEPT
{
    if( pHeap && g_pStaticHeap )
    {
        OutputError( "StaticHeap is Initialized." );
    }

    g_pStaticHeap = pHeap;
}

//---------------------------------------------------------------------------
//  静的ヒープを取得します。
//---------------------------------------------------------------------------
Heap* GetStaticHeap() NN_NOEXCEPT
{
    return g_pStaticHeap;
}

//---------------------------------------------------------------------------
//  静的ヒープからメモリを確保します。
//---------------------------------------------------------------------------
void* AllocFromStaticHeap( size_t size, size_t alignment ) NN_NOEXCEPT
{
    size_t allocatedSize = nn::util::align_up( size, 32 );
    void* ptr = g_pStaticHeap->Alloc( allocatedSize, alignment );
    NN_SDK_ASSERT_NOT_NULL( ptr );
    g_StaticHeapAllocatedSize += allocatedSize;
    return ptr;
}

//---------------------------------------------------------------------------
//  静的ヒープからメモリを解放します。
//---------------------------------------------------------------------------
void FreeFromStaticHeap( void* ptr ) NN_NOEXCEPT
{
    return g_pStaticHeap->Free( ptr );
}

//---------------------------------------------------------------------------
//  静的ヒープから確保されたメモリサイズを取得します。
//---------------------------------------------------------------------------
size_t GetAllocatedSizeFromStaticHeap() NN_NOEXCEPT
{
    return g_StaticHeapAllocatedSize;
}

//---------------------------------------------------------------------------
//  動的ヒープからメモリを設定します。
//---------------------------------------------------------------------------
void SetDynamicHeap( Heap* pHeap ) NN_NOEXCEPT
{
    if( pHeap && g_pDynamicHeap )
    {
        OutputError( "DynamicHeap is Initialized." );
    }

    g_pDynamicHeap = pHeap;
}

Heap* GetDynamicHeap() NN_NOEXCEPT
{
    if( g_pDynamicHeap == NULL )
    {
        OutputError( "DynamicHeap is Null !!!" );
    }
    return g_pDynamicHeap;
}

//---------------------------------------------------------------------------
//  動的ヒープからメモリを確保します。
//---------------------------------------------------------------------------
void* AllocFromDynamicHeap( size_t size, size_t alignmentAddr, size_t alignmentSize ) NN_NOEXCEPT
{
    size_t allocatedSize = nn::util::align_up( size, alignmentSize );
    void* ptr = g_pDynamicHeap->Alloc( allocatedSize, alignmentAddr );
    if( !ptr )
    {
        Warning( nullptr, RuntimeWarningId_DynamicHeapAllocationFailed );
        OutputWarning( "DynamicHeap Allocation Failed. Allocation size : %d Byte.\n", allocatedSize );
    }
    return ptr;
}

//---------------------------------------------------------------------------
//  動的ヒープからメモリを解放します。
//---------------------------------------------------------------------------
void FreeFromDynamicHeap( void* ptr, bool immediate ) NN_NOEXCEPT
{
    if( immediate )
    {
        return g_pDynamicHeap->Free( ptr );
    }
    else
    {
        // 動的ヒープ free リストへつなぐ
        return AddFreeListForDynamicHeap( ptr );
    }
}

//---------------------------------------------------------------------------
//  遅延解放用のリストを初期化します。
//---------------------------------------------------------------------------
void InitializeDelayFreeList( int freeListCount ) NN_NOEXCEPT
{
    if( !g_pStaticHeap )
    {
        OutputError( "StaticHeap is not Initialized." );
    }

    g_DelayFreeListMax = freeListCount;

    g_DelayFreeList = reinterpret_cast< void** >( AllocFromStaticHeap( sizeof( void* ) * g_DelayFreeListMax ) );
    MemUtil::FillZero( g_DelayFreeList, sizeof( void* ) * g_DelayFreeListMax );
    g_DelayFreeListCount = 0;
}

//---------------------------------------------------------------------------
//  遅延解放用のリストを破棄します。
//---------------------------------------------------------------------------
void FinalizeDelayFreeList() NN_NOEXCEPT
{
    for( int i = 0; i < g_DelayFreeListCount; i++ )
    {
        g_pDynamicHeap->Free( g_DelayFreeList[ i ] );
        g_DelayFreeList[ i ] = NULL;
    }

    g_pStaticHeap->Free( g_DelayFreeList );
    g_DelayFreeList = NULL;
    g_DelayFreeListCount = 0;
}

//---------------------------------------------------------------------------
//  遅延解放の為のフリーリストへアドレスを追加する
//---------------------------------------------------------------------------
void AddFreeListForDynamicHeap( void* ptr ) NN_NOEXCEPT
{
    g_MutexAddDelayFreeList.Lock();
    {
        int index = g_DelayFreeListCount;
        g_DelayFreeList[ index ] = ptr;
        g_DelayFreeListCount++;
        NN_SDK_ASSERT( g_DelayFreeListCount <= g_DelayFreeListMax );
    }
    g_MutexAddDelayFreeList.Unlock();
}

//---------------------------------------------------------------------------
//  遅延解放用のリスト内アドレスを解放します。
//---------------------------------------------------------------------------
void FlushDelayFreeList() NN_NOEXCEPT
{
    for( int i = 0; i < g_DelayFreeListCount; i++ )
    {
        g_pDynamicHeap->Free( g_DelayFreeList[ i ] );
        g_DelayFreeList[ i ] = NULL;
    }
    g_DelayFreeListCount = 0;
}

//---------------------------------------------------------------------------
//  ランタイムログ出力を抑制行います。
//---------------------------------------------------------------------------
void SetSuppressOutputLog( bool flag ) NN_NOEXCEPT
{
    g_SuppressLog = flag;
}

//---------------------------------------------------
//  ログ出力時コールバックを設定します。
//---------------------------------------------------
void SetOutputLogCallBack( OutputMessageCallback callback ) NN_NOEXCEPT
{
    g_LogCallback = callback;
}

//---------------------------------------------------
//  ランタイム警告出力時コールバックを設定します。
//---------------------------------------------------
void SetOutputWarningCallBack( OutputMessageCallback callback ) NN_NOEXCEPT
{
    g_WarningCallback = callback;
}

//---------------------------------------------------
//! @brief  ランタイムエラー出力時コールバックを設定します。
//---------------------------------------------------
void SetOutputErrorCallBack( OutputMessageCallback callback ) NN_NOEXCEPT
{
    g_ErrorCallback = callback;
}

//---------------------------------------------------------------------------
//  ランタイムログ出力を行います。
//---------------------------------------------------------------------------
void OutputLog( const char* format, ... ) NN_NOEXCEPT
{
#if !defined( NN_SDK_BUILD_RELEASE )
    if( g_SuppressLog )
    {
        return;
    }

    if( g_LogCallback )
    {
        va_list vargs;
        va_start( vargs, format );
        g_LogCallback( "[VFX] ", vargs );
        g_LogCallback( format, vargs );
        va_end( vargs );
        return;
    }

    NN_SDK_LOG( "[VFX] " );
    va_list vargs;
    va_start( vargs, format );
    VPrintf( format, vargs );
    va_end( vargs );
#else
    NN_UNUSED( format );
#endif
}

//---------------------------------------------------------------------------
//  ランタイムログ警告出力を行います。
//---------------------------------------------------------------------------
void OutputWarning( const char* format, ... ) NN_NOEXCEPT
{
#if !defined( NN_SDK_BUILD_RELEASE )
    if( g_WarningCallback )
    {
        va_list vargs;
        va_start( vargs, format );
        g_WarningCallback( "[VFX] Warning!! ", vargs );
        g_WarningCallback( format, vargs );
        va_end( vargs );
        return;
    }

    NN_SDK_LOG( "[VFX] Warning!! " );
    va_list vlist;
    va_start( vlist, format );
    VPrintf( format, vlist );
    va_end( vlist );

#else
    NN_UNUSED( format );
#endif
}

//---------------------------------------------------------------------------
//  ランタイムログ警告出力を行います。
//---------------------------------------------------------------------------
void Warning( void* context, RuntimeWarningId warningFlag ) NN_NOEXCEPT
{
#if !defined( NN_SDK_BUILD_RELEASE )

    g_WarningFlag |= warningFlag;

    switch( warningFlag )
    {
    case RuntimeWarningId_ParticleEmissionFailure:
    {
        nn::vfx::Emitter* pEmitter = reinterpret_cast< nn::vfx::Emitter* >( context );
        OutputWarning( "Particle Emission Error.\n" );
        OutputWarning( "  %s - %s\n", pEmitter->GetEmitterSet()->GetName(), pEmitter->GetName() );
    }
        break;

    case RuntimeWarningId_ParticleMaxCountIsZero:
    {
        nn::vfx::Emitter* pEmitter = reinterpret_cast< nn::vfx::Emitter* >( context );
        OutputWarning( "Particle Max Num is Zero. PLife:%d/Interval/EmissionRate",
            pEmitter->GetResEmitter()->ptcl.life, pEmitter->GetResEmitter()->emission.interval, pEmitter->GetResEmitter()->emission.rate );
        OutputWarning( "  %s - %s - GroupId:%d\n", pEmitter->GetEmitterSet()->GetName(), pEmitter->GetName(), pEmitter->GetEmitterSet()->GetGroupId() );
    }
        break;

    case RuntimeWarningId_ParticleInstanceIsDirty:
    {
        nn::vfx::Emitter* pEmitter = reinterpret_cast< nn::vfx::Emitter* >( context );
        OutputWarning( "Particle UserData or PluginData is Dirty. \n" );
        OutputWarning( "  %s - %s - GroupId:%d\n", pEmitter->GetEmitterSet()->GetName(), pEmitter->GetName(), pEmitter->GetEmitterSet()->GetGroupId() );
    }
        break;

    case RuntimeWarningId_NoCustomActionParameter:
    {
        nn::vfx::Emitter* pEmitter = reinterpret_cast< nn::vfx::Emitter* >( context );
        OutputWarning( "CustomAction Parameter isn't Exist.\n" );
        OutputWarning( "  %s\n", pEmitter->GetEmitterSet()->GetName() );
    }
        break;

    case RuntimeWarningId_NoShaderExists:
    {
        nn::vfx::Emitter* pEmitter = reinterpret_cast< nn::vfx::Emitter* >( context );
        OutputWarning( "Shader isn't Exist.\n" );
        OutputWarning( "  %s - %s - GroupId:%d\n", pEmitter->GetEmitterSet()->GetName(), pEmitter->GetName(), pEmitter->GetEmitterSet()->GetGroupId() );
    }
        break;

    case RuntimeWarningId_NoParentParticleExists:
    {
        nn::vfx::Emitter* pEmitter = reinterpret_cast< nn::vfx::Emitter* >( context );
        OutputWarning( "Parent Particle isn't Exist.\n" );
        OutputWarning( "  %s - %s - GroupId:%d\n", pEmitter->GetEmitterSet()->GetName(), pEmitter->GetName(), pEmitter->GetEmitterSet()->GetGroupId() );
    }
        break;

    case RuntimeWarningId_LoadUnsafePrimitive:
        OutputWarning( "Unsafe Primitive Load.\n" );
        break;

    case RuntimeWarningId_IsNotManualEmitMode:
    {
        nn::vfx::EmitterSet* pEmitterSet = reinterpret_cast< nn::vfx::EmitterSet* >( context );
        OutputWarning( "Manual Emit Mode is not Set.\n" );
        OutputWarning( "  EmitterSet:%s\n", pEmitterSet->GetName() );
    }
        break;

    case RuntimeWarningId_ManualEmitSizeOver:
    {
        nn::vfx::EmitterSet* pEmitterSet = reinterpret_cast< nn::vfx::EmitterSet* >( context );
        OutputWarning( "Manual Emission: Emit reservation list is full.\n" );
        OutputWarning( "  EmitterSet:%s\n", pEmitterSet->GetName() );
    }
    break;

    case RuntimeWarningId_ManualEmitterIsFull:
        {
            nn::vfx::Emitter* pEmitter = reinterpret_cast< nn::vfx::Emitter* >( context );
            OutputWarning( "Manual Emission: Manual Emitter has no available particle.\n" );
            OutputWarning( "  EmitterSet:%s, Emitter:%s\n", pEmitter->GetEmitterSet()->GetName(), pEmitter->GetName() );
        }
        break;

    case RuntimeWarningId_DynamicHeapAllocationFailed:
    {
        // 現状フラグ操作のみ
    }
    break;

    case RuntimeWarningId_TemporaryBufferAllocationFailed:
    {
        // 現状フラグ操作のみ
    }
    break;

    case RuntimeWarningId_GpuBufferAllocationFailed:
    {
        // 現状フラグ操作のみ
    }
    break;

    case RuntimeWarningId_StripeAllocationFailed:
    {
        // 現状フラグ操作のみ
    }
    break;

    case RuntimeWarningId_StripeDynamicHeapAllocationFailed:
    {
        // 現状フラグ操作のみ
    }
    break;

    case RuntimeWarningId_StripeTemporaryBufferAllocationFailed:
    {
        OutputWarning( "Temporary Buffer for stripe history is not enough\n" );
    }
    break;

    case RuntimeWarningId_StripeGpuBufferAllocationFailed:
    {
        // 現状フラグ操作のみ
    }
    break;

    case RuntimeWarningId_None:
        break;

    default:
        NN_SDK_ASSERT( 0 );
        break;
    }
#else
    NN_UNUSED( context );
    NN_UNUSED( warningFlag );
#endif
}// NOLINT(readability/fn_size)

//---------------------------------------------------------------------------
//  ランタイム警告のビットフィールドを取得します。
//---------------------------------------------------------------------------
uint32_t GetWarningBitFlag() NN_NOEXCEPT
{
    return g_WarningFlag;
}

//---------------------------------------------------------------------------
//  ランタイムエラー出力を行います。
//---------------------------------------------------------------------------
void OutputError( const char* format, ... ) NN_NOEXCEPT
{
#if !defined( NN_SDK_BUILD_RELEASE )
    if( g_WarningCallback )
    {
        va_list vargs;
        va_start( vargs, format );
        g_LogCallback( "[VFX] Error!! ", vargs );
        g_LogCallback( format, vargs );
        va_end( vargs );
        VFX_ERROR( "[VFX] Error!! Stopped." );
        return;
    }

    NN_SDK_LOG( "[VFX] Error!! " );
    va_list vargs;
    va_start( vargs, format );
    VFX_ERROR( format, vargs );
    va_end( vargs );
#else
    NN_UNUSED( format );
#endif
}

//---------------------------------------------------------------------------
//  文字列出力を行います。
//---------------------------------------------------------------------------
void VPrintf( const char* format, va_list vlist ) NN_NOEXCEPT
{
#if !defined( NN_SDK_BUILD_RELEASE )
    NN_SDK_VLOG( format, vlist );
#else
    NN_UNUSED( format );
    NN_UNUSED( vlist );
#endif
}

//---------------------------------------------------------------------------
//  Cubic 補完関数
//---------------------------------------------------------------------------
void HermiteInterpolationOnCubic(
    nn::util::Vector3fType*         pOutPos,
    const nn::util::Vector3fType&   p1,
    const nn::util::Vector3fType&   v1,
    const nn::util::Vector3fType&   p2,
    const nn::util::Vector3fType&   v2,
    float                           relativePos ) NN_NOEXCEPT
{
    // TODO: math 最適化が不十分なので対応する。

    const nn::util::Matrix4x4fType HermiteMatrix = NN_UTIL_MATRIX_4X4F_INITIALIZER(
        2.f, -3.f, 0.f, 1.f,
        -2.f, 3.f, 0.f, 0.f,
        1.f, -2.f, 1.f, 0.f,
        1.f, -1.f, 0.f, 0.f );

    nn::util::Matrix4x4fType mat = NN_UTIL_MATRIX_4X4F_INITIALIZER(
        nn::util::VectorGetX( p1 ), nn::util::VectorGetX( p2 ), nn::util::VectorGetX( v1 ), nn::util::VectorGetX( v2 ),
        nn::util::VectorGetY( p1 ), nn::util::VectorGetY( p2 ), nn::util::VectorGetY( v1 ), nn::util::VectorGetY( v2 ),
        nn::util::VectorGetZ( p1 ), nn::util::VectorGetZ( p2 ), nn::util::VectorGetZ( v1 ), nn::util::VectorGetZ( v2 ),
        0, 0, 0, 0 );
    nn::util::MatrixMultiply( &mat, mat, HermiteMatrix );

    // t^3 + t^2 + t のベクトルを作っておく
    const float t = relativePos;
    const float vT[ 3 ] = { ( t * t * t ), ( t * t ), t };

    nn::util::Float4x4 matTemp;
    nn::util::MatrixStore( &matTemp, mat );

    nn::util::VectorSetX( pOutPos, ( matTemp.m[ 0 ][ 0 ] * vT[ 0 ] ) + ( matTemp.m[ 0 ][ 1 ] * vT[ 1 ] ) + ( matTemp.m[ 0 ][ 2 ] * vT[ 2 ] ) + matTemp.m[ 0 ][ 3 ] );
    nn::util::VectorSetY( pOutPos, ( matTemp.m[ 1 ][ 0 ] * vT[ 0 ] ) + ( matTemp.m[ 1 ][ 1 ] * vT[ 1 ] ) + ( matTemp.m[ 1 ][ 2 ] * vT[ 2 ] ) + matTemp.m[ 1 ][ 3 ] );
    nn::util::VectorSetZ( pOutPos, ( matTemp.m[ 2 ][ 0 ] * vT[ 0 ] ) + ( matTemp.m[ 2 ][ 1 ] * vT[ 1 ] ) + ( matTemp.m[ 2 ][ 2 ] * vT[ 2 ] ) + matTemp.m[ 2 ][ 3 ] );

    return;
}

//---------------------------------------------------------------------------
//  ベクトルの任意軸回転
//---------------------------------------------------------------------------
void VectorRotateArbitraryAxis(
    nn::util::Vector3fType*         pOutVec,
    const nn::util::Vector3fType&   inVec,
    const nn::util::Vector3fType&   axis,
    float                           rotationRadian ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pOutVec );

    const float ax = nn::util::VectorGetX( axis );
    const float ay = nn::util::VectorGetY( axis );
    const float az = nn::util::VectorGetZ( axis );
    const float c = nn::util::CosEst( rotationRadian );
    const float s = nn::util::SinEst( rotationRadian );
    const float cInv = ( 1 - c );

    // 任意軸回転の行列作成
    const nn::util::Matrix4x3fType mat = NN_UTIL_MATRIX_4X3F_INITIALIZER(
        ax*ax*cInv + c,    ax*ay*cInv - az*s, az*ax*cInv + ay*s,
        ax*ay*cInv + az*s, ay*ay*cInv + c,    ay*az*cInv - ax*s,
        az*ax*cInv - ay*s, ay*az*cInv + ax*s, az*az*cInv + c,
        0, 0, 0
    );

    nn::util::VectorTransform( pOutVec, inVec, mat );
}

} // namespace detail
} // namespace vfx
} // namespace nn
