﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vfx/vfx_EmitterCalc.h>
#include <nn/vfx/vfx_Random.h>

// 等分割球の定義データをインクルード
#include <nn/vfx/vfx_SameDivideSphereData.h>

namespace nn {
namespace vfx {
namespace detail {

//------------------------------------------------------------------------------
//  関数テーブル
//------------------------------------------------------------------------------
EmitterCalculator::EmitFunction EmitterCalculator::g_EmitFunctions[] =
{
    EmitterCalculator::CalculateEmitPoint,                 //!< 点エミッタ
    EmitterCalculator::CalculateEmitCircle,                //!< 円
    EmitterCalculator::CalculateEmitCircleEquallyDivided,   //!< 円(分割)
    EmitterCalculator::CalculateEmitCircleFill,            //!< 円(フィル)
    EmitterCalculator::CalculateEmitSphere,                //!< 球
    EmitterCalculator::CalculateEmitSphereEqually32Divided, //!< 球(分割)
    EmitterCalculator::CalculateEmitSphereEqually64Divided, //!< 球(分割64)
    EmitterCalculator::CalculateEmitSphereFill,            //!< 球(フィル)
    EmitterCalculator::CalculateEmitCylinder,              //!< 円柱
    EmitterCalculator::CalculateEmitCylinderFill,          //!< 円柱(フィル)
    EmitterCalculator::CalculateEmitBox,                   //!< 立方体
    EmitterCalculator::CalculateEmitBoxFill,               //!< 立方体(フィル)
    EmitterCalculator::CalculateEmitLine,                  //!< 線
    EmitterCalculator::CalculateEmitLineEquallyDivided,     //!< 線(分割)
    EmitterCalculator::CalculateEmitRectangle,             //!< 矩形
    EmitterCalculator::CalculateEmitPrimitive,             //!< プリミティブ
};

//------------------------------------------------------------------------------
//  弧の開始角度、角度幅を求めます。（パラメータ式）
//------------------------------------------------------------------------------
inline void CalculateVolumeSweepParams( float* pOutRotWidth, float* pOutRotStart, float sweepParam, float sweepStart, uint8_t isSweepStartRandom, float randomValue ) NN_NOEXCEPT
{
    // 単位：Radian
    *pOutRotWidth = sweepParam;
    *pOutRotStart = ( isSweepStartRandom ) ? ( randomValue * nn::util::FloatPi * 2.0f ) : ( sweepStart );
}

//------------------------------------------------------------------------------
//  弧の位置を計算
//------------------------------------------------------------------------------
inline float CalculateRotateForSweep( float random, float rotWidth, float rotStart ) NN_NOEXCEPT
{
    return random * rotWidth + rotStart - ( rotWidth / 2 );
}


//------------------------------------------------------------------------------
//  円周上の位置を求めます。
//------------------------------------------------------------------------------
inline void CalculateSinCosRad( const ResEmitterVolume* __restrict pEmitterVolume, Random* pRandom, float rotWidth, float rotStart, float* sinV, float* cosV ) NN_NOEXCEPT
{
    // 単位：Radian

    // 円の場合
    if( pEmitterVolume->arcType == 1 )
    {
        float rot = pRandom->GetFloat() * nn::util::FloatPi * 2.0f;
        nn::util::SinCosEst( sinV, cosV, rot );
    }
    // 弧の場合
    else
    {
        float rot = CalculateRotateForSweep( pRandom->GetFloat(), rotWidth, rotStart );
        nn::util::SinCosEst( sinV, cosV, rot );
    }
}

//------------------------------------------------------------------------------
//  Y軸上の位置を求めます。
//------------------------------------------------------------------------------
inline float CalculatePosOnAxisY( const ResEmitterVolume* __restrict pEmitterVolume, Random* pRandom ) NN_NOEXCEPT
{
    // 緯度の場合
    if( pEmitterVolume->arcType == 1 )
    {
        return 1.0f - ( 1.0f - nn::util::CosEst( pEmitterVolume->sweepLatitude ) ) * pRandom->GetFloat();
    }
    // 経度の場合
    else
    {
        return pRandom->GetFloat() * 2.0f - 1.0f;
    }
}

//------------------------------------------------------------------------------
//  球の緯度の範囲内かどうか判定します。
//------------------------------------------------------------------------------
inline bool IsLatitudeInside( const ResEmitterVolume* __restrict pEmitterVolume, const nn::util::Vector3fType& dir ) NN_NOEXCEPT
{
    // ほぼ 180 度の場合は true を返す
    if( nn::util::FloatPi - 0.0001f < pEmitterVolume->sweepLatitude )
    {
        return true;
    }

    float y = nn::util::CosEst( pEmitterVolume->sweepLatitude );
    return ( y < nn::util::VectorGetY( dir ) );
}

//------------------------------------------------------------------------------
//  球を緯度の軸に合うように回転します。
//------------------------------------------------------------------------------
inline void RotateDirection( const ResEmitterVolume* __restrict pEmitterVolume, nn::util::Vector3fType* pOutDir ) NN_NOEXCEPT
{
    static const int LatitudeDir_PlusX   = 0;
    static const int LatitudeDir_MinusX  = 1;
    static const int LatitudeDir_PlusY   = 2;
    static const int LatitudeDir_MinusY  = 3;
    static const int LatitudeDir_PlusZ   = 4;
    static const int LatitudeDir_MinusZ  = 5;

    nn::util::Vector3fType latitudeDir = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 0, 0 );
    switch( pEmitterVolume->volumeLatitudeDir )
    {
    case LatitudeDir_PlusX:  nn::util::VectorSetX( &latitudeDir, 1.0f ); break;
    case LatitudeDir_MinusX: nn::util::VectorSetX( &latitudeDir, -1.0f ); break;
    case LatitudeDir_PlusY:  nn::util::VectorSetY( &latitudeDir, 1.0f ); break;
    case LatitudeDir_MinusY: nn::util::VectorSetY( &latitudeDir, -1.0f ); break;
    case LatitudeDir_PlusZ:  nn::util::VectorSetZ( &latitudeDir, 1.0f ); break;
    case LatitudeDir_MinusZ: nn::util::VectorSetZ( &latitudeDir, -1.0f ); break;
    default: NN_SDK_ASSERT( 0 );
    }

    if( nn::util::VectorGetX( latitudeDir ) != 0.0f ||
        nn::util::VectorGetY( latitudeDir ) != 1.0f ||
        nn::util::VectorGetZ( latitudeDir ) != 0.0f )
    {
        // (0, 1, 0)を指定方向に回転させる回転行列を生成し適用する
        nn::util::Vector4fType q;
        nn::util::Vector3fType axis = NN_UTIL_VECTOR_3F_INITIALIZER( 0.0f, 1.0f, 0.0f );
        nn::util::QuaternionMakeVectorRotation( &q, axis, latitudeDir );
        nn::util::Matrix4x3fType qMatrix;
        nn::util::MatrixFromQuaternion( &qMatrix, q );
        nn::util::VectorTransform( pOutDir, *pOutDir, qMatrix );
    }
}

//------------------------------------------------------------------------------
//  点エミッタ
//------------------------------------------------------------------------------
bool EmitterCalculator::CalculateEmitPoint( nn::util::Vector3fType* pOutLocalPos, nn::util::Vector3fType* pOutLocalVec, Emitter* pEmitter, int emitIndex, int maxEmitCount, float randomValue, EmitterAnimValue* pEmitterAnimData ) NN_NOEXCEPT
{
    NN_UNUSED( emitIndex );
    NN_UNUSED( maxEmitCount );
    NN_UNUSED( randomValue );
    NN_UNUSED( pEmitterAnimData );

    // ローカル位置
    nn::util::VectorZero( pOutLocalPos );

    // ローカル速度
    const nn::util::Vector3fType& randomVec3f = pEmitter->m_Random.GetNormalizedVec3();
    const float allDirVel = pEmitterAnimData->allDirVelocity;
    nn::util::VectorMultiply( pOutLocalVec, randomVec3f, allDirVel );

    return true;
}

//---------------------------------------------------------------------------
//  円エミッタ計算処理を行います。
//---------------------------------------------------------------------------
bool EmitterCalculator::CalculateEmitCircle( nn::util::Vector3fType* pOutLocalPos, nn::util::Vector3fType* pOutLocalVec, Emitter* pEmitter, int emitIndex, int maxEmitCount, float randomValue, EmitterAnimValue* pEmitterAnimData ) NN_NOEXCEPT
{
    NN_UNUSED( emitIndex );
    NN_UNUSED( maxEmitCount );

    // 弧の開始角度、角度幅
    float rotWidth;
    float rotStart;
    const ResEmitterVolume* pEmitterVolume = &pEmitter->m_pEmitterData->volume;
    detail::CalculateVolumeSweepParams( &rotWidth, &rotStart, pEmitterVolume->sweepLongitude, pEmitterVolume->sweepStart, pEmitterVolume->sweepStartRandom, randomValue );

    float sinV, cosV;
    float emitterRotate = detail::CalculateRotateForSweep( pEmitter->m_Random.GetFloat(), rotWidth, rotStart );
    nn::util::SinCosEst( &sinV, &cosV, emitterRotate );

    float scaleX = pEmitterVolume->volumeRadius.x * pEmitterAnimData->emitterVolumeScale.x;
    float scaleZ = pEmitterVolume->volumeRadius.z * pEmitterAnimData->emitterVolumeScale.z;

    // ローカル位置
    nn::util::VectorSet( pOutLocalPos, sinV * scaleX, 0.0f, cosV * scaleZ );

    // ローカル速度
    nn::util::VectorSet( pOutLocalVec, sinV * pEmitterAnimData->allDirVelocity, 0.0f, cosV * pEmitterAnimData->allDirVelocity );

    return true;
}

//------------------------------------------------------------------------------
//  等分割円エミッタ
//------------------------------------------------------------------------------
bool EmitterCalculator::CalculateEmitCircleEquallyDivided( nn::util::Vector3fType* pOutLocalPos, nn::util::Vector3fType* pOutLocalVec, Emitter* pEmitter, int emitIndex, int maxEmitCount, float randomValue, EmitterAnimValue* pEmitterAnimData ) NN_NOEXCEPT
{
    // 単位：Radian
    NN_UNUSED( maxEmitCount );

    const ResEmitterVolume* pEmitterVolume = &pEmitter->m_pEmitterData->volume;

    // 弧の初期角度を決定
    float sweepStart = pEmitter->m_pEmitterData->volume.sweepStart;
    if( pEmitter->m_pEmitterData->volume.sweepStartRandom )
    {
        sweepStart = randomValue * nn::util::FloatPi * 2.0f;
    }

    float rot = sweepStart;

    // 弧の幅＆分割数から、インデックスごとの角度増分を決定
    int divCount = pEmitterVolume->numDivideCircle;
    if( pEmitter->m_pEmitterRes->m_pResEmitter->volume.primEmitType == PrimitiveEmissionType_Unison )
    {
        divCount -= static_cast< int >( randomValue * pEmitterVolume->numDivideCircleRandom * 0.01f * pEmitterVolume->numDivideCircle );
    }
    if( pEmitter->m_pEmitterData->volume.sweepLongitude != nn::util::FloatPi * 2.0f )
    {
        // 円でない場合は先端と末端が重ならないので「N点使って等分割（≒従ってN-1分割）」になる
        if( divCount > 1 )
        {
            divCount--;
        }
    }
    int emitFixedIndex = emitIndex;

    switch( pEmitter->m_pEmitterRes->m_pResEmitter->volume.primEmitType )
    {
    case PrimitiveEmissionType_Unison:// 一斉放出：何もしない
        break;
    case PrimitiveEmissionType_Random:// ランタイム放出タイプ
        emitFixedIndex = static_cast< int >( pEmitter->m_Random.GetFloat() * divCount );
        break;
    case PrimitiveEmissionType_IndexOrder:// インデックス放出タイプ
        emitFixedIndex = static_cast< int >( pEmitter->m_EmitByPrimitiveCounter ) % divCount;
        pEmitter->m_EmitByPrimitiveCounter++;
        if( pEmitter->m_EmitByPrimitiveCounter >= divCount )
        {
            pEmitter->m_EmitByPrimitiveCounter = 0;
        }
        break;
    default:
        NN_SDK_ASSERT( 0 );
    }

    const float longitude = pEmitter->m_pEmitterData->volume.sweepLongitude;    // 弧の幅
    const float rotPlus = longitude / divCount;

    // 角度
    rot += emitFixedIndex * rotPlus - ( longitude / 2.0f ) + ( ( pEmitter->m_Random.GetFloat() - 0.5f ) * 2 ) * pEmitter->m_pEmitterData->volume.volumeSurfacePosRand;
    float sinV, cosV;
    nn::util::SinCosEst( &sinV, &cosV, rot );

    float scaleX = pEmitterVolume->volumeRadius.x * pEmitterAnimData->emitterVolumeScale.x;
    float scaleZ = pEmitterVolume->volumeRadius.z * pEmitterAnimData->emitterVolumeScale.z;

    // ローカル位置
    nn::util::VectorSet( pOutLocalPos,
        sinV * scaleX,
        0.0f,
        cosV * scaleZ );

    // ローカル速度
    nn::util::VectorSet( pOutLocalVec,
        sinV * pEmitterAnimData->allDirVelocity,
        0.0f,
        cosV * pEmitterAnimData->allDirVelocity );

    return true;
}

//---------------------------------------------------------------------------
//  円フィルエミッタ計算処理を行います。
//---------------------------------------------------------------------------
bool EmitterCalculator::CalculateEmitCircleFill( nn::util::Vector3fType* pOutLocalPos, nn::util::Vector3fType* pOutLocalVec, Emitter* pEmitter, int emitIndex, int maxEmitCount, float randomValue, EmitterAnimValue* pEmitterAnimData ) NN_NOEXCEPT
{
    NN_UNUSED( emitIndex );
    NN_UNUSED( maxEmitCount );
    NN_UNUSED( randomValue );

    // 弧の開始角度、角度幅
    float rotWidth;
    float rotStart;
    const ResEmitterVolume* pEmitterVolume = &pEmitter->m_pEmitterData->volume;
    detail::CalculateVolumeSweepParams( &rotWidth, &rotStart, pEmitterVolume->sweepLongitude, pEmitterVolume->sweepStart, pEmitterVolume->sweepStartRandom, randomValue );

    float sinV, cosV;
    float emitterRotate = detail::CalculateRotateForSweep( pEmitter->m_Random.GetFloat(), rotWidth, rotStart );
    nn::util::SinCosEst( &sinV, &cosV, emitterRotate );

    // 方向
    float rand = pEmitter->m_Random.GetFloat();
    float inner = 1.0f - pEmitter->m_pEmitterData->volume.caliberRatio;

    // MEMO: 半径上の位置をここで乱数で補間して動かすが、
    //       線形補間すると、円の中心の密度が外周側と比べて大きくなるので、二乗したモノの線形補間のrootを取ることで、
    //       円内に均一に分布するようにする。（※外周側の方がより多く選ばれる）
    rand = SafeSqrt( rand + inner * inner * ( 1.0f - rand ) );
    float dirx = rand * sinV;
    float dirz = rand * cosV;

    float scaleX = pEmitterVolume->volumeRadius.x * pEmitterAnimData->emitterVolumeScale.x;
    float scaleZ = pEmitterVolume->volumeRadius.z * pEmitterAnimData->emitterVolumeScale.z;

    // ローカル位置
    nn::util::VectorSet( pOutLocalPos,
        dirx * scaleX,
        0.0f,
        dirz * scaleZ );

    // ローカル速度
    // MEMO: エミッタ形状に沿って歪ませたあと、一度正規化し全方向初速を乗算する。
    nn::util::Vector3fType vel = NN_UTIL_VECTOR_3F_INITIALIZER( dirx * rand, 0, dirz * rand );
    if( scaleX > scaleZ )
    {
        const float ratio = scaleZ / scaleX;
        nn::util::VectorSetZ( &vel, nn::util::VectorGetZ( vel ) * ratio );
    }
    else
    {
        const float ratio = scaleX / scaleZ;
        nn::util::VectorSetX( &vel, nn::util::VectorGetX( vel ) * ratio );
    }

    if( nn::util::VectorLengthSquared( vel ) > 0.0f )
    {
        nn::util::VectorNormalize( &vel, vel );
    }
    else
    {
        // 零ベクトルの場合（≒乱数で 0 を引いた場合）、デフォルトの方向に決め打ちする。
        nn::util::VectorSet( &vel, 0, 0, 1 );
    }
    nn::util::VectorMultiply( &vel, vel, pEmitterAnimData->allDirVelocity );
    *pOutLocalVec = vel;

    return true;
}

//---------------------------------------------------------------------------
//  球エミッタ計算処理を行います。
//---------------------------------------------------------------------------
bool EmitterCalculator::CalculateEmitSphere( nn::util::Vector3fType* pOutLocalPos, nn::util::Vector3fType* pOutLocalVec, Emitter* pEmitter, int emitIndex, int maxEmitCount, float randomValue, EmitterAnimValue* pEmitterAnimData ) NN_NOEXCEPT
{
    NN_UNUSED( emitIndex );
    NN_UNUSED( maxEmitCount );
    NN_UNUSED( randomValue );

    // 参照を取得
    const ResEmitterVolume* pEmitterVolume = &pEmitter->m_pEmitterData->volume;
    Random* rnd = &pEmitter->m_Random;

    const float scaleX = pEmitterVolume->volumeRadius.x * pEmitterAnimData->emitterVolumeScale.x;
    const float scaleY = pEmitterVolume->volumeRadius.y * pEmitterAnimData->emitterVolumeScale.y;
    const float scaleZ = pEmitterVolume->volumeRadius.z * pEmitterAnimData->emitterVolumeScale.z;
    nn::util::Vector3fType scale = NN_UTIL_VECTOR_3F_INITIALIZER( scaleX, scaleY, scaleZ );


    // 弧の開始角度、角度幅
    float rotWidth;
    float rotStart;
    if( pEmitterVolume->arcType == 1 )
    {
        // 緯度
        detail::CalculateVolumeSweepParams( &rotWidth, &rotStart, pEmitterVolume->sweepLatitude, pEmitterVolume->sweepStart, pEmitterVolume->sweepStartRandom, randomValue );
    }
    else
    {
        // 経度
        detail::CalculateVolumeSweepParams( &rotWidth, &rotStart, pEmitterVolume->sweepLongitude, pEmitterVolume->sweepStart, pEmitterVolume->sweepStartRandom, randomValue );
    }

    // 円周上の位置
    float sinV;
    float cosV;
    detail::CalculateSinCosRad( pEmitterVolume, rnd, rotWidth, rotStart, &sinV, &cosV );

    // 高さ
    float y = detail::CalculatePosOnAxisY( pEmitterVolume, rnd );

    // 半径
    float r = SafeSqrt( 1.0f - y * y );

    // 方向
    nn::util::Vector3fType dir = NN_UTIL_VECTOR_3F_INITIALIZER( r * sinV, y, r * cosV );

    // 緯度の方向回転
    if( pEmitterVolume->arcType == 1 )
    {
        detail::RotateDirection( pEmitterVolume, &dir );
    }

    // 位置
    nn::util::VectorMultiply( pOutLocalPos, dir, scale );
    // 速度
    nn::util::VectorMultiply( pOutLocalVec, dir, pEmitterAnimData->allDirVelocity );

    return true;
}

//---------------------------------------------------------------------------
//  球 32等分割エミッタ計算処理を行います。
//---------------------------------------------------------------------------
bool EmitterCalculator::CalculateEmitSphereEqually32Divided( nn::util::Vector3fType* pOutLocalPos, nn::util::Vector3fType* pOutLocalVec, Emitter* pEmitter, int emitIndex, int maxEmitCount, float randomValue, EmitterAnimValue* pEmitterAnimData ) NN_NOEXCEPT
{
    NN_UNUSED( maxEmitCount );
    NN_UNUSED( randomValue );

    // 参照を取得
    const ResEmitterVolume* pEmitterVolume = &pEmitter->m_pEmitterData->volume;

    const float scaleX = pEmitterVolume->volumeRadius.x * pEmitterAnimData->emitterVolumeScale.x;
    const float scaleY = pEmitterVolume->volumeRadius.y * pEmitterAnimData->emitterVolumeScale.y;
    const float scaleZ = pEmitterVolume->volumeRadius.z * pEmitterAnimData->emitterVolumeScale.z;
    const nn::util::Vector3fType scale = NN_UTIL_VECTOR_3F_INITIALIZER( scaleX, scaleY, scaleZ );

    // UI上のIndex値から放出レートにマッピング。
    const int pattern[] =
    {
        2, 3, 4, 6, 8, 12, 20, 32
    };

    const int vIndex = pEmitterVolume->volumeTblIndex;
    const float *sphere_tbl = detail::EquallyDividedSphereTable[ vIndex ];

    int emitFixedIndex = emitIndex;
    const int divCount = pattern[ vIndex ];
    switch( pEmitter->m_pEmitterRes->m_pResEmitter->volume.primEmitType )
    {
    case PrimitiveEmissionType_Unison:// 一斉放出
        break;
    case PrimitiveEmissionType_Random:// ランタイム放出タイプ
        emitFixedIndex = static_cast< int >( pEmitter->m_Random.GetFloat() * divCount );
        break;
    case PrimitiveEmissionType_IndexOrder:// インデックス放出タイプ
        emitFixedIndex = static_cast< int >( pEmitter->m_EmitByPrimitiveCounter ) % divCount;
        pEmitter->m_EmitByPrimitiveCounter++;
        if( pEmitter->m_EmitByPrimitiveCounter >= divCount )
        {
            pEmitter->m_EmitByPrimitiveCounter = 0;
        }
        break;
    default:
        NN_SDK_ASSERT( 0 );
    }

    // 方向
    const int index = emitFixedIndex * 3;
    nn::util::Vector3fType dir = NN_UTIL_VECTOR_3F_INITIALIZER( sphere_tbl[ index ], sphere_tbl[ index + 1 ], sphere_tbl[ index + 2 ] );

    // 経度は使わない
    {
        // 緯度でフィルタ
        if( !detail::IsLatitudeInside( pEmitterVolume, dir ) )
        {
            return false;
        }

        // 緯度の方向回転
        detail::RotateDirection( pEmitterVolume, &dir );
    }

    // 位置
    nn::util::VectorMultiply( pOutLocalPos, dir, scale );

    // 速度
    nn::util::VectorMultiply( pOutLocalVec, dir, pEmitterAnimData->allDirVelocity );

    return true;
}

//---------------------------------------------------------------------------
//  球 64等分割エミッタ計算処理を行います。
//---------------------------------------------------------------------------
bool EmitterCalculator::CalculateEmitSphereEqually64Divided( nn::util::Vector3fType* pOutLocalPos, nn::util::Vector3fType* pOutLocalVec, Emitter* pEmitter, int emitIndex, int maxEmitCount, float randomValue, EmitterAnimValue* pEmitterAnimData ) NN_NOEXCEPT
{
    NN_UNUSED( maxEmitCount );
    NN_UNUSED( randomValue );

    // 参照を取得
    const ResEmitterVolume* pEmitterVolume = &pEmitter->m_pEmitterData->volume;

    const float scaleX = pEmitterVolume->volumeRadius.x * pEmitterAnimData->emitterVolumeScale.x;
    const float scaleY = pEmitterVolume->volumeRadius.y * pEmitterAnimData->emitterVolumeScale.y;
    const float scaleZ = pEmitterVolume->volumeRadius.z * pEmitterAnimData->emitterVolumeScale.z;
    const nn::util::Vector3fType scale = NN_UTIL_VECTOR_3F_INITIALIZER( scaleX, scaleY, scaleZ );

    const int vIndex = pEmitterVolume->volumeTblIndex64 - 4;   // 分割数からインデックス値に修正
    const int tableIndex = vIndex + 2;
    const float *sphere_tbl = detail::gSameDivideSphere64Tbl[ tableIndex ];

    int emitFixedIndex = emitIndex;
    const int divCount = pEmitterVolume->volumeTblIndex64;
    switch( pEmitter->m_pEmitterRes->m_pResEmitter->volume.primEmitType )
    {
    case PrimitiveEmissionType_Unison:// 一斉放出
        break;
    case PrimitiveEmissionType_Random:// ランタイム放出タイプ
        emitFixedIndex = static_cast< int >( pEmitter->m_Random.GetFloat() * divCount );
        break;
    case PrimitiveEmissionType_IndexOrder:// インデックス放出タイプ
        emitFixedIndex = static_cast< int >( pEmitter->m_EmitByPrimitiveCounter ) % divCount;
        pEmitter->m_EmitByPrimitiveCounter++;
        if( pEmitter->m_EmitByPrimitiveCounter >= divCount )
        {
            pEmitter->m_EmitByPrimitiveCounter = 0;
        }
        break;
    default:
        NN_SDK_ASSERT( 0 );
    }

    // 方向
    const int index = emitFixedIndex * 3;
    nn::util::Vector3fType dir = NN_UTIL_VECTOR_3F_INITIALIZER(
        sphere_tbl[ index + 0 ],
        sphere_tbl[ index + 1 ],
        sphere_tbl[ index + 2 ] );

    // 経度は使わない
    {
        // 緯度でフィルタ
        if( !detail::IsLatitudeInside( pEmitterVolume, dir ) )
        {
            //nn::util::VectorSetW( pOutLocalPos, pEmitter->m_Time + pEmitter->m_pEmitterData->ptcl.life );
            return false;
        }

        // 緯度の方向回転
        detail::RotateDirection( pEmitterVolume, &dir );
    }

    // 位置
    nn::util::VectorMultiply( pOutLocalPos, dir, scale );
    // 速度
    nn::util::VectorMultiply( pOutLocalVec, dir, pEmitterAnimData->allDirVelocity );

    return true;
}

//---------------------------------------------------------------------------
//  球 フィルエミッタ計算処理を行います。
//---------------------------------------------------------------------------
bool EmitterCalculator::CalculateEmitSphereFill( nn::util::Vector3fType* pOutLocalPos, nn::util::Vector3fType* pOutLocalVec, Emitter* pEmitter, int emitIndex, int maxEmitCount, float randomValue, EmitterAnimValue* pEmitterAnimData ) NN_NOEXCEPT
{
    NN_UNUSED( emitIndex );
    NN_UNUSED( maxEmitCount );

    // 参照を取得
    const ResEmitterVolume* pEmitterVolume = &pEmitter->m_pEmitterData->volume;
    Random* rnd = &pEmitter->m_Random;

    const float scaleX = pEmitterVolume->volumeRadius.x * pEmitterAnimData->emitterVolumeScale.x;
    const float scaleY = pEmitterVolume->volumeRadius.y * pEmitterAnimData->emitterVolumeScale.y;
    const float scaleZ = pEmitterVolume->volumeRadius.z * pEmitterAnimData->emitterVolumeScale.z;
    const nn::util::Vector3fType scale = NN_UTIL_VECTOR_3F_INITIALIZER( scaleX, scaleY, scaleZ );

    // 弧の開始角度、角度幅
    float rotWidth;
    float rotStart;
    if( pEmitterVolume->arcType == 1 )
    {
        detail::CalculateVolumeSweepParams( &rotWidth, &rotStart, pEmitterVolume->sweepLatitude, pEmitterVolume->sweepStart, pEmitterVolume->sweepStartRandom, randomValue );
    }
    else
    {
        detail::CalculateVolumeSweepParams( &rotWidth, &rotStart, pEmitterVolume->sweepLongitude, pEmitterVolume->sweepStart, pEmitterVolume->sweepStartRandom, randomValue );
    }

    // 円周上の位置
    float sinV;
    float cosV;
    detail::CalculateSinCosRad( pEmitterVolume, rnd, rotWidth, rotStart, &sinV, &cosV );

    // 高さ
    float y = detail::CalculatePosOnAxisY( pEmitterVolume, rnd );

    // 半径
    float r = SafeSqrt( 1.0f - y * y );

    // 空洞状の分布
    float radius = rnd->GetFloat();
    radius = SafeSqrt( radius );
    radius = radius * pEmitterVolume->caliberRatio + 1.0f - pEmitterVolume->caliberRatio;

    // 方向
    nn::util::Vector3fType dir = NN_UTIL_VECTOR_3F_INITIALIZER( r * sinV, y, r * cosV );

    // 緯度の方向回転
    if( pEmitterVolume->arcType == 1 )
    {
        detail::RotateDirection( pEmitterVolume, &dir );
    }

    // 位置
    nn::util::VectorMultiply( pOutLocalPos, dir, scale );
    nn::util::VectorMultiply( pOutLocalPos, *pOutLocalPos, radius );
    // 速度
    nn::util::VectorMultiply( pOutLocalVec, dir, pEmitterAnimData->allDirVelocity );

    return true;
}

//---------------------------------------------------------------------------
//  シリンダー エミッタ計算処理を行います。
//---------------------------------------------------------------------------
bool EmitterCalculator::CalculateEmitCylinder( nn::util::Vector3fType* pOutLocalPos, nn::util::Vector3fType* pOutLocalVec, Emitter* pEmitter, int emitIndex, int maxEmitCount, float randomValue, EmitterAnimValue* pEmitterAnimData ) NN_NOEXCEPT
{
    bool result = CalculateEmitCircle( pOutLocalPos, pOutLocalVec, pEmitter, emitIndex, maxEmitCount, randomValue, pEmitterAnimData );
    if( !result )
    {
        return false;
    }

    const ResEmitterVolume* pEmitterVolume = &pEmitter->m_pEmitterData->volume;
    const float scaleY = pEmitterVolume->volumeRadius.y * pEmitterAnimData->emitterVolumeScale.y;

    float y = pEmitter->m_Random.GetFloat() * 2.0f - 1.0f;
    nn::util::VectorSetY( pOutLocalPos, y * scaleY );

    return true;
}

//---------------------------------------------------------------------------
//  シリンダーフィル エミッタ計算処理を行います。
//---------------------------------------------------------------------------
bool EmitterCalculator::CalculateEmitCylinderFill( nn::util::Vector3fType* pOutLocalPos, nn::util::Vector3fType* pOutLocalVec, Emitter* pEmitter, int emitIndex, int maxEmitCount, float randomValue, EmitterAnimValue* pEmitterAnimData ) NN_NOEXCEPT
{
    bool result = CalculateEmitCircleFill( pOutLocalPos, pOutLocalVec, pEmitter, emitIndex, maxEmitCount, randomValue, pEmitterAnimData );
    if( !result )
    {
        return false;
    }

    const ResEmitterVolume* pEmitterVolume = &pEmitter->m_pEmitterData->volume;
    float scaleY = pEmitterVolume->volumeRadius.y * pEmitterAnimData->emitterVolumeScale.y;

    float y = pEmitter->m_Random.GetFloat() * 2.0f - 1.0f;
    nn::util::VectorSetY( pOutLocalPos, y * scaleY );

    return true;
}

//---------------------------------------------------------------------------
//  ボックス エミッタ計算処理を行います。
//---------------------------------------------------------------------------
bool EmitterCalculator::CalculateEmitBox( nn::util::Vector3fType* pOutLocalPos, nn::util::Vector3fType* pOutLocalVec, Emitter* pEmitter, int emitIndex, int maxEmitCount, float randomValue, EmitterAnimValue* pEmitterAnimData ) NN_NOEXCEPT
{
    NN_UNUSED( emitIndex );
    NN_UNUSED( maxEmitCount );
    NN_UNUSED( randomValue );

    uint32_t coord = pEmitter->m_Random.GetUnsignedIntegerDirect();
    uint32_t sign = pEmitter->m_Random.GetUnsignedIntegerDirect();

    const nn::util::Vector3fType radRnd = NN_UTIL_VECTOR_3F_INITIALIZER(
        pEmitter->m_Random.GetFloat() * 2.0f - 1.0f,
        pEmitter->m_Random.GetFloat() * 2.0f - 1.0f,
        pEmitter->m_Random.GetFloat() * 2.0f - 1.0f );

    const ResEmitterVolume* pEmitterVolume = &pEmitter->m_pEmitterData->volume;
    const float scaleX = pEmitterVolume->volumeRadius.x * pEmitterAnimData->emitterVolumeScale.x;
    const float scaleY = pEmitterVolume->volumeRadius.y * pEmitterAnimData->emitterVolumeScale.y;
    const float scaleZ = pEmitterVolume->volumeRadius.z * pEmitterAnimData->emitterVolumeScale.z;

    // xy
    if( coord < 0xffffffff / 3 * 1 )
    {
        if( sign < 0xffffffff / 2 )
        {
            nn::util::VectorSetX( pOutLocalPos, scaleX * nn::util::VectorGetX( radRnd ) );
            nn::util::VectorSetY( pOutLocalPos, scaleY * nn::util::VectorGetY( radRnd ) );
            nn::util::VectorSetZ( pOutLocalPos, scaleZ );
        }
        else
        {
            nn::util::VectorSetX( pOutLocalPos, scaleX * nn::util::VectorGetX( radRnd ) );
            nn::util::VectorSetY( pOutLocalPos, scaleY * nn::util::VectorGetY( radRnd ) );
            nn::util::VectorSetZ( pOutLocalPos, -scaleZ );
        }
    }

    // xz
    else if( coord < 0xffffffff / 3 * 2 )
    {
        if( sign < 0xffffffff / 2 )
        {
            nn::util::VectorSetX( pOutLocalPos, scaleX * nn::util::VectorGetX( radRnd ) );
            nn::util::VectorSetY( pOutLocalPos, scaleY );
            nn::util::VectorSetZ( pOutLocalPos, scaleZ * nn::util::VectorGetZ( radRnd ) );
        }
        else
        {
            nn::util::VectorSetX( pOutLocalPos, scaleX * nn::util::VectorGetX( radRnd ) );
            nn::util::VectorSetY( pOutLocalPos, -scaleY );
            nn::util::VectorSetZ( pOutLocalPos, scaleZ * nn::util::VectorGetZ( radRnd ) );
        }
    }

    // yz
    else
    {
        if( sign < 0xffffffff / 2 )
        {
            nn::util::VectorSetX( pOutLocalPos, scaleX );
            nn::util::VectorSetY( pOutLocalPos, scaleY * nn::util::VectorGetY( radRnd ) );
            nn::util::VectorSetZ( pOutLocalPos, scaleZ * nn::util::VectorGetZ( radRnd ) );
        }
        else
        {
            nn::util::VectorSetX( pOutLocalPos, -scaleX );
            nn::util::VectorSetY( pOutLocalPos, scaleY * nn::util::VectorGetY( radRnd ) );
            nn::util::VectorSetZ( pOutLocalPos, scaleZ * nn::util::VectorGetZ( radRnd ) );
        }
    }

    // ローカル速度
    nn::util::Vector3fType dir = NN_UTIL_VECTOR_3F_INITIALIZER( 0.f, 0.f, 0.f );
    if( !nn::util::VectorIsZero( *pOutLocalPos ) )
    {
        dir = *pOutLocalPos;
        dir = *pOutLocalPos;
        dir = *pOutLocalPos;
        if( !nn::util::VectorIsZero( dir ) )
        {
            nn::util::VectorNormalize( &dir, dir );
        }
    }

    nn::util::VectorMultiply( pOutLocalVec, dir, pEmitterAnimData->allDirVelocity );

    return true;
}

//---------------------------------------------------------------------------
//  ボックス フィルエミッタ計算処理を行います。
//---------------------------------------------------------------------------
bool EmitterCalculator::CalculateEmitBoxFill( nn::util::Vector3fType* pOutLocalPos, nn::util::Vector3fType* pOutLocalVec, Emitter* pEmitter, int emitIndex, int maxEmitCount, float randomValue, EmitterAnimValue* pEmitterAnimData ) NN_NOEXCEPT
{
    NN_UNUSED( emitIndex );
    NN_UNUSED( maxEmitCount );
    NN_UNUSED( randomValue );

    const ResEmitterVolume* pEmitterVolume = &pEmitter->m_pEmitterData->volume;
    const float scaleX = pEmitterVolume->volumeRadius.x * pEmitterAnimData->emitterVolumeScale.x;
    const float scaleY = pEmitterVolume->volumeRadius.y * pEmitterAnimData->emitterVolumeScale.y;
    const float scaleZ = pEmitterVolume->volumeRadius.z * pEmitterAnimData->emitterVolumeScale.z;
    const nn::util::Vector3fType scale = NN_UTIL_VECTOR_3F_INITIALIZER( scaleX, scaleY, scaleZ );

    //if( pEmitter->emitterData->volume.caliberRatio == 0.0f )
    //{
    //    pOutLocalPos->x = scaleX * ( pEmitter->random.GetF32() * 2.0f - 1.0f );
    //    pOutLocalPos->y = scaleY * ( pEmitter->random.GetF32() * 2.0f - 1.0f );
    //    pOutLocalPos->z = scaleZ * ( pEmitter->random.GetF32() * 2.0f - 1.0f );
    //}
    //else
    {
        // 空洞状の分布
        float r = 1.0f - pEmitter->m_pEmitterData->volume.caliberRatio;
        if( r == 1.0f )
        {
            r = 0.999f;
        }

        float v = pEmitter->m_Random.GetFloat();
        float r2 = r * r;
        float r3 = r2 * r;
        float volumeA = 1.0f - r;
        float volumeB = volumeA * r;

        v = v * ( 1.0f - r3 );

        float randomx = pEmitter->m_Random.GetFloat();
        float randomy = pEmitter->m_Random.GetFloat();
        float randomz = pEmitter->m_Random.GetFloat();

        if( v < volumeA )
        {
            nn::util::VectorSet( pOutLocalPos,
                randomx,
                randomy * ( 1.0f - r ) + r,
                randomz );
        }
        else if( v < volumeA + volumeB )
        {
            nn::util::VectorSet( pOutLocalPos,
                randomx * ( 1.0f - r ) + r,
                randomy * r,
                randomz );
        }
        else
        {
            nn::util::VectorSet( pOutLocalPos,
                randomx * r,
                randomy * r,
                randomz * ( 1.0f - r ) + r );
        }

        if( pEmitter->m_Random.GetFloat() < 0.5f )
        {
            nn::util::VectorSetX ( pOutLocalPos, nn::util::VectorGetX( *pOutLocalPos) * -1.0f );
        }

        if( pEmitter->m_Random.GetFloat() < 0.5f )
        {
            nn::util::VectorSetY( pOutLocalPos, nn::util::VectorGetY( *pOutLocalPos ) * -1.0f );
        }

        if( pEmitter->m_Random.GetFloat() < 0.5f )
        {
            nn::util::VectorSetZ( pOutLocalPos, nn::util::VectorGetZ( *pOutLocalPos ) * -1.0f );
        }

        nn::util::VectorMultiply( pOutLocalPos, *pOutLocalPos, scale );
    }

    // ローカル速度
    nn::util::Vector3fType dir = NN_UTIL_VECTOR_3F_INITIALIZER( 0.f, 0.f, 0.f );
    if( !nn::util::VectorIsZero( *pOutLocalPos ) )
    {
        dir = *pOutLocalPos;

        if( nn::util::VectorIsZero( dir ) )
        {
            float randomx = pEmitter->m_Random.GetFloat();
            float randomy = pEmitter->m_Random.GetFloat();
            float randomz = pEmitter->m_Random.GetFloat();

            // この形状ではパーティクルが(0,0,0)に配置される可能性がある
            // 向きをランダムに与える
            nn::util::VectorSet( &dir,
                randomx,
                randomy,
                randomz );
        }
        nn::util::VectorNormalize( &dir, dir );
    }

    // ローカル速度
    nn::util::VectorMultiply( pOutLocalVec, dir, pEmitterAnimData->allDirVelocity );
    return true;
}

//---------------------------------------------------------------------------
//  ライン エミッタ計算処理を行います。
//---------------------------------------------------------------------------
bool EmitterCalculator::CalculateEmitLine( nn::util::Vector3fType* pOutLocalPos, nn::util::Vector3fType* pOutLocalVec, Emitter* pEmitter, int emitIndex, int maxEmitCount, float randomValue, EmitterAnimValue* pEmitterAnimData ) NN_NOEXCEPT
{
    NN_UNUSED( emitIndex );
    NN_UNUSED( maxEmitCount );
    NN_UNUSED( randomValue );

    const ResEmitterVolume* pEmitterVolume = &pEmitter->m_pEmitterData->volume;
    float scaleZ = pEmitterVolume->lineLength * pEmitterAnimData->emitterVolumeScale.z;

    // ローカル位置
    nn::util::VectorSet( pOutLocalPos,
        0.0f,
        0.0f,
        pEmitter->m_Random.GetFloat() * scaleZ - ( scaleZ + pEmitter->m_pEmitterData->volume.lineCenter * scaleZ ) / 2.0f
        );

    // ローカル速度
    nn::util::VectorSet( pOutLocalVec,
        0.0f,
        0.0f,
        pEmitterAnimData->allDirVelocity
    );

    return true;
}

//---------------------------------------------------------------------------
//  ライン 等分割エミッタ計算処理を行います。
//---------------------------------------------------------------------------
bool EmitterCalculator::CalculateEmitLineEquallyDivided( nn::util::Vector3fType* pOutLocalPos, nn::util::Vector3fType* pOutLocalVec, Emitter* pEmitter, int emitIndex, int maxEmitCount, float randomValue, EmitterAnimValue* pEmitterAnimData ) NN_NOEXCEPT
{
    NN_UNUSED( maxEmitCount );

    const ResEmitterVolume* pEmitterVolume = &pEmitter->m_pEmitterData->volume;
    float scaleZ = pEmitterVolume->lineLength * pEmitterAnimData->emitterVolumeScale.z;

    float t = 0.0f;

    int emitFixedIndex = emitIndex;
    int divCount = pEmitterVolume->numDivideLine;

    switch( pEmitter->m_pEmitterRes->m_pResEmitter->volume.primEmitType )
    {
    case PrimitiveEmissionType_Unison:// 一斉放出
        divCount -= static_cast< int >( randomValue * pEmitterVolume->numDivideLineRandom * 0.01f * pEmitterVolume->numDivideLine );
        break;
    case PrimitiveEmissionType_Random:// ランタイム放出タイプ
        emitFixedIndex = static_cast< int >( pEmitter->m_Random.GetFloat() * divCount );
        break;
    case PrimitiveEmissionType_IndexOrder:// インデックス放出タイプ
        emitFixedIndex = static_cast< int >( pEmitter->m_EmitByPrimitiveCounter ) % divCount;
        pEmitter->m_EmitByPrimitiveCounter++;
        if( pEmitter->m_EmitByPrimitiveCounter >= divCount )
        {
            pEmitter->m_EmitByPrimitiveCounter = 0;
        }
        break;
    default:
        NN_SDK_ASSERT( 0 );
    }

    float t_plus = 1.0f / divCount;
    if( divCount == 1 )
    {
        t = 0.5f;
        t_plus = 0.0f;
    }
    else
    {
        t = 0.0f;
        t_plus = 1.0f / ( divCount - 1 );
    }

    t = t_plus * emitFixedIndex + t;

    // ローカル位置
    nn::util::VectorSet( pOutLocalPos,
        0.0f,
        0.0f,
        t * scaleZ - ( scaleZ + pEmitter->m_pEmitterData->volume.lineCenter * scaleZ ) / 2.0f );

    // ローカル速度
    nn::util::VectorSet( pOutLocalVec,
        0.0f,
        0.0f,
        pEmitterAnimData->allDirVelocity );

    return true;
}

//---------------------------------------------------------------------------
//  矩形 エミッタ計算処理を行います。
//---------------------------------------------------------------------------
bool EmitterCalculator::CalculateEmitRectangle( nn::util::Vector3fType* pOutLocalPos, nn::util::Vector3fType* pOutLocalVec, Emitter* pEmitter, int emitIndex, int maxEmitCount, float randomValue, EmitterAnimValue* pEmitterAnimData ) NN_NOEXCEPT
{
    NN_UNUSED( emitIndex );
    NN_UNUSED( maxEmitCount );
    NN_UNUSED( randomValue );

    const ResEmitterVolume* pEmitterVolume = &pEmitter->m_pEmitterData->volume;
    float scaleX = pEmitterVolume->volumeRadius.x * pEmitterAnimData->emitterVolumeScale.x;
    float scaleZ = pEmitterVolume->volumeRadius.z * pEmitterAnimData->emitterVolumeScale.z;

    uint32_t coord = pEmitter->m_Random.GetUnsignedIntegerDirect();
    uint32_t sign = pEmitter->m_Random.GetUnsignedIntegerDirect();

    const float radRndX = pEmitter->m_Random.GetFloat() * 2.0f - 1.0f;
    const float radRndZ = pEmitter->m_Random.GetFloat() * 2.0f - 1.0f;

    // xy
    if( coord < 0xffffffff / 2 )
    {
        if( sign < 0xffffffff / 2 )
        {
            nn::util::VectorSetX( pOutLocalPos, scaleX * radRndX );
            nn::util::VectorSetY( pOutLocalPos, 0 );
            nn::util::VectorSetZ( pOutLocalPos, scaleZ );

        }
        else
        {
            nn::util::VectorSetX( pOutLocalPos, scaleX * radRndX );
            nn::util::VectorSetY( pOutLocalPos, 0 );
            nn::util::VectorSetZ( pOutLocalPos, -scaleZ );
        }
    }

    // yz
    else
    {
        if( sign < 0xffffffff / 2 )
        {
            nn::util::VectorSetX( pOutLocalPos, scaleX );
            nn::util::VectorSetY( pOutLocalPos, 0 );
            nn::util::VectorSetZ( pOutLocalPos, scaleZ * radRndZ );

        }
        else
        {
            nn::util::VectorSetX( pOutLocalPos, -scaleX );
            nn::util::VectorSetY( pOutLocalPos, 0 );
            nn::util::VectorSetZ( pOutLocalPos, scaleZ * radRndZ );
        }
    }

    // ローカル速度
    nn::util::Vector3fType dir = NN_UTIL_VECTOR_3F_INITIALIZER( 0.f, 0.f, 0.f );
    if( !nn::util::VectorIsZero( *pOutLocalPos ) )
    {
        dir = *pOutLocalPos;
        if( !nn::util::VectorIsZero( dir ) )
        {
            nn::util::VectorNormalize( &dir, dir );
        }
    }
    nn::util::VectorMultiply( pOutLocalVec, dir, pEmitterAnimData->allDirVelocity );

    return true;
}

//---------------------------------------------------------------------------
//  プリミティブ エミッタ計算処理を行います。
//---------------------------------------------------------------------------
bool EmitterCalculator::CalculateEmitPrimitive( nn::util::Vector3fType* pOutLocalPos, nn::util::Vector3fType* pOutLocalVec, Emitter* pEmitter, int emitIndex, int maxEmitCount, float randomValue, EmitterAnimValue* pEmitterAnimData ) NN_NOEXCEPT
{

    if( !pEmitter->m_pEmitterRes->m_pVolumePrimitive )
    {
        return CalculateEmitPoint( pOutLocalPos, pOutLocalVec, pEmitter, emitIndex, maxEmitCount, randomValue, pEmitterAnimData );
    }

    Primitive* pPrimitive = pEmitter->m_pEmitterRes->m_pVolumePrimitive;

    int emitFixedIndex = emitIndex;

    switch( pEmitter->m_pEmitterRes->m_pResEmitter->volume.primEmitType )
    {
        // ランタイム放出タイプ
    case 1:
        emitFixedIndex = static_cast< int >( pEmitter->m_Random.GetFloat() * pPrimitive->GetVertexBufferArrayCount() );
        break;

        // インデックス放出タイプ
    case 2:
        emitFixedIndex = static_cast< int >( pEmitter->m_EmitByPrimitiveCounter ) % pPrimitive->GetVertexBufferArrayCount();
        pEmitter->m_EmitByPrimitiveCounter++;
        break;
    default:
        break;
    }

    const nn::util::Vector3fType scale = NN_UTIL_VECTOR_3F_INITIALIZER(
        pEmitterAnimData->emitterVolumeScale.x,
        pEmitterAnimData->emitterVolumeScale.y,
        pEmitterAnimData->emitterVolumeScale.z );

    pPrimitive->GetPosition( pOutLocalPos, emitFixedIndex );
    nn::util::VectorMultiply( pOutLocalPos, *pOutLocalPos, scale );

    pPrimitive->GetNormal( pOutLocalVec, emitFixedIndex );
    nn::util::VectorMultiply( pOutLocalVec, *pOutLocalVec, pEmitterAnimData->allDirVelocity );

    return true;
}

} // namespace detail
} // namespace vfx
} // namespace nn
