﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


//------------------------------------------------------------------------------
#include <nn/vfx/vfx_EmitterCalc.h>
#include <nn/vfx/vfx_RenderContext.h>
#include <nn/vfx/vfx_CurlNoiseData.h>
#include <nn/vfx/vfx_System.h>

namespace nn {
namespace vfx {
namespace detail {

//---------------------------------------------------------------------------
//  エミッタが用いる通常テクスチャ（テクスチャ0/1/2、フレームバッファ、デプスバッファ、カールノイズ）をまとめて設定します。
//---------------------------------------------------------------------------
bool EmitterCalculator::SetDefaultEmitterTextures( nn::gfx::CommandBuffer* pCommandBuffer, Emitter* pEmitter, Shader* pShader, DrawParameterArg* pDrawParameterArg ) NN_NOEXCEPT
{
    // テクスチャ設定
    for( int i = 0; i < TextureSlotId_MaxStandardTextureSlotId; i++ )
    {
        nn::gfx::DescriptorSlot textureDescSlot = pEmitter->GetEmitterResource()->m_TextureDescSlot[ i ];
        if ( textureDescSlot.IsValid() )
        {
            int                     vertexLoc = pShader->GetVertexTextureSamplerLocation( static_cast< TextureSlotId >( i ) );
            int                     pixelLoc  = pShader->GetPixelTextureSamplerLocation( static_cast<TextureSlotId>( i ) );
            TextureSampler*         pSampler  = pEmitter->GetEmitterResource()->m_TextureSampler[ i ];

            if ( vertexLoc != InvalidValueId_TextureSamplerSlotId )
            {
                pCommandBuffer->SetTextureAndSampler( vertexLoc, nn::gfx::ShaderStage_Vertex,
                                                      textureDescSlot, pSampler->GetDescriptorSlot() );
            }
            if( pixelLoc != InvalidValueId_TextureSamplerSlotId )
            {
                pCommandBuffer->SetTextureAndSampler( pixelLoc, nn::gfx::ShaderStage_Pixel,
                                                      textureDescSlot, pSampler->GetDescriptorSlot() );
            }
        }
    }

    // フレームバッファテクスチャの設定
    if ( pDrawParameterArg->m_FrameBufferTexture.IsValid() )
    {
        int vertexLoc = pShader->GetFrameBufferTextureVertexSamplerLocation();
        int pixelLoc  = pShader->GetFrameBufferTexturePixelSamplerLocation();

        if ( vertexLoc != InvalidValueId_TextureSamplerSlotId )
        {
            pCommandBuffer->SetTextureAndSampler( vertexLoc, nn::gfx::ShaderStage_Vertex, pDrawParameterArg->m_FrameBufferTexture, m_pTextureSamplerForColorAndDepth->GetDescriptorSlot() );
        }
        if ( pixelLoc != InvalidValueId_TextureSamplerSlotId )
        {
            pCommandBuffer->SetTextureAndSampler( pixelLoc, nn::gfx::ShaderStage_Pixel, pDrawParameterArg->m_FrameBufferTexture, m_pTextureSamplerForColorAndDepth->GetDescriptorSlot() );
        }
    }

    // デプスバッファテクスチャの設定
    if ( pDrawParameterArg->m_DepthBufferTexture.IsValid() )
    {
        int vertexLoc = pShader->GetDepthBufferTextureVertexSamplerLocation();
        int pixelLoc  = pShader->GetDepthBufferTexturePixelSamplerLocation();

        if ( vertexLoc != InvalidValueId_TextureSamplerSlotId )
        {
            pCommandBuffer->SetTextureAndSampler( vertexLoc, nn::gfx::ShaderStage_Vertex, pDrawParameterArg->m_DepthBufferTexture, m_pTextureSamplerForColorAndDepth->GetDescriptorSlot() );
        }
        if ( pixelLoc != InvalidValueId_TextureSamplerSlotId )
        {
            pCommandBuffer->SetTextureAndSampler( pixelLoc, nn::gfx::ShaderStage_Pixel, pDrawParameterArg->m_DepthBufferTexture, m_pTextureSamplerForColorAndDepth->GetDescriptorSlot() );
        }
    }

    return true;
}

//---------------------------------------------------------------------------
//  エミッタが用いる通常定数バッファ（ビュー、静的、動的、フィールド）をまとめて設定します。
//---------------------------------------------------------------------------
bool EmitterCalculator::SetDefaultEmitterConstantBuffers( nn::gfx::CommandBuffer* pCommandBuffer, Emitter* pEmitter, Shader* pShader, DrawParameterArg* pDrawParameterArg ) NN_NOEXCEPT
{
    // ビュー定数バッファの設定
    if ( pDrawParameterArg->m_pViewGpuAddress )
    {
        int viewConstantBufferSlotV = pShader->GetVertexConstantBufferSlot( Shader::ConstantBufferType_View );
        int viewConstantBufferSlotP = pShader->GetPixelConstantBufferSlot( Shader::ConstantBufferType_View );

        if ( viewConstantBufferSlotV != InvalidValueId_ConstantBufferSlotId )
        {
            pCommandBuffer->SetConstantBuffer( viewConstantBufferSlotV, nn::gfx::ShaderStage_Vertex, *pDrawParameterArg->m_pViewGpuAddress, sizeof( ViewParam ) );
        }
        if ( viewConstantBufferSlotP != InvalidValueId_ConstantBufferSlotId )
        {
            pCommandBuffer->SetConstantBuffer( viewConstantBufferSlotP, nn::gfx::ShaderStage_Pixel, *pDrawParameterArg->m_pViewGpuAddress, sizeof( ViewParam ) );
        }
    }

    // エミッタ動的定数バッファの設定
    nn::gfx::GpuAddress address;
    pEmitter->m_GfxObjects.m_DynamicConstantBuffer->GetGpuAddress( &address, pEmitter->m_GfxObjects.m_DynamicConstantBufferAddr[pEmitter->m_BufferSide] );
    int dynamicConstantBufferSlotV = pShader->GetVertexConstantBufferSlot( Shader::ConstantBufferType_EmitterDynamic );
    int dynamicConstantBufferSlotP = pShader->GetPixelConstantBufferSlot( Shader::ConstantBufferType_EmitterDynamic );

    if( dynamicConstantBufferSlotV != InvalidValueId_ConstantBufferSlotId )
    {
        pCommandBuffer->SetConstantBuffer( dynamicConstantBufferSlotV,
                                           nn::gfx::ShaderStage_Vertex,
                                           address,
                                           sizeof( detail::EmitterDynamicConstantBuffer ) );
    }
    if( dynamicConstantBufferSlotP != InvalidValueId_ConstantBufferSlotId )
    {
        pCommandBuffer->SetConstantBuffer( dynamicConstantBufferSlotP,
                                           nn::gfx::ShaderStage_Pixel,
                                           address,
                                           sizeof( detail::EmitterDynamicConstantBuffer ) );
    }

    // エミッタ静的定数バッファの設定
    ptrdiff_t staticBufferOffset  = pEmitter->m_pEmitterRes->m_EmitterStaticBufferOffset;
    int       staticBufferSize    = sizeof( detail::ResCommon ) + sizeof( detail::EmitterStaticUniformBlock );

    pEmitter->m_pEmitterRes->m_pResourceConstantBuffer->GetGpuAddress( &address );
    address.Offset( staticBufferOffset );

    int staticConstantBufferSlotV = pShader->GetVertexConstantBufferSlot( Shader::ConstantBufferType_EmitterStatic );
    int staticConstantBufferSlotP = pShader->GetPixelConstantBufferSlot( Shader::ConstantBufferType_EmitterStatic );
    if( staticConstantBufferSlotV != InvalidValueId_ConstantBufferSlotId )
    {
        pCommandBuffer->SetConstantBuffer( staticConstantBufferSlotV, nn::gfx::ShaderStage_Vertex, address, staticBufferSize );
    }
    if( staticConstantBufferSlotP != InvalidValueId_ConstantBufferSlotId )
    {
        pCommandBuffer->SetConstantBuffer( staticConstantBufferSlotP, nn::gfx::ShaderStage_Pixel, address, staticBufferSize );
    }

#if 0   // 現状設定は不要なのでコメントアウト
    // フィールド定数バッファの設定
    int fieldBufferOffset  = pEmitter->m_pEmitterRes->m_FieldConstantBufferOffset;
    int fieldBufferSize = sizeof( detail::EmitterFieldConstantBuffer );

    pEmitter->m_pEmitterRes->m_pResourceConstantBuffer->GetGpuAddress( &address );
    address.Offset( fieldBufferOffset );

    int fieldConstantBufferSlotV = pShader->GetVertexConstantBufferSlot( Shader::ConstantBufferType_EmitterField );
    int fieldConstantBufferSlotP = pShader->GetPixelConstantBufferSlot( Shader::ConstantBufferType_EmitterField );

    if( fieldConstantBufferSlotV != InvalidValueId_ConstantBufferSlotId )
    {
        pCommandBuffer->SetConstantBuffer( fieldConstantBufferSlotV, nn::gfx::ShaderStage_Vertex, address, fieldBufferSize );
    }
    if( fieldConstantBufferSlotP != InvalidValueId_ConstantBufferSlotId )
    {
        pCommandBuffer->SetConstantBuffer( fieldConstantBufferSlotP, nn::gfx::ShaderStage_Pixel, address, fieldBufferSize );
    }
#endif

    return true;
}

//---------------------------------------------------------------------------
//  エミッタストリームアウト計算処理をします。
//---------------------------------------------------------------------------
bool EmitterCalculator::CalculateComputeShader( nn::gfx::CommandBuffer* pCommandBuffer, Emitter* pEmitter, const detail::ComputeShader* pComputeShader, int globalCounter, bool calcComputeShader, void* pUserParam, bool doFlush ) NN_NOEXCEPT
{
    NN_UNUSED( pUserParam );

    if ( !calcComputeShader )
    {
        return false;
    }

    // 1フレーム中に複数回（複数パス/複数画面描画など）計算されるのを防止
    if( pEmitter->m_ComputeShaderCounter == globalCounter )
    {
        return false;
    }

    // 演算シェーダを設定
    pCommandBuffer->SetShader( pComputeShader->GetGfxShader(), nn::gfx::ShaderStageBit_All );

    // エミッタ動的UBOの設定
    nn::gfx::GpuAddress address;
    pEmitter->m_GfxObjects.m_DynamicConstantBuffer->GetGpuAddress( &address, pEmitter->m_GfxObjects.m_DynamicConstantBufferAddr[pEmitter->m_BufferSide] );
    int dynamicConstantBufferSlotV = pComputeShader->GetEmitterDynamicBufferSlot();
    if( dynamicConstantBufferSlotV != InvalidValueId_ConstantBufferSlotId )
    {
        pCommandBuffer->SetConstantBuffer( dynamicConstantBufferSlotV, nn::gfx::ShaderStage_Compute, address, sizeof( detail::EmitterDynamicConstantBuffer ) );
    }

    // エミッタ静的定数バッファの設定
    ptrdiff_t   staticBufferOffset  = pEmitter->m_pEmitterRes->m_EmitterStaticBufferOffset;
    int         staticBufferSize    = sizeof( detail::ResCommon ) + sizeof( detail::EmitterStaticUniformBlock );

    pEmitter->m_pEmitterRes->m_pResourceConstantBuffer->GetGpuAddress( &address );
    address.Offset( staticBufferOffset );

    int staticConstantBufferSlotV = pComputeShader->GetEmitterStaticBufferSlot();
    if( staticConstantBufferSlotV != InvalidValueId_ConstantBufferSlotId )
    {
        pCommandBuffer->SetConstantBuffer( staticConstantBufferSlotV, nn::gfx::ShaderStage_Compute, address, staticBufferSize );
    }

    // フィールド定数バッファの設定
    ptrdiff_t   fieldBufferOffset  = pEmitter->m_pEmitterRes->m_FieldConstantBufferOffset;
    int         fieldBufferSize = sizeof( detail::EmitterFieldConstantBuffer );

    pEmitter->m_pEmitterRes->m_pResourceConstantBuffer->GetGpuAddress( &address );
    address.Offset( fieldBufferOffset );

    int fieldConstantBufferSlotV = pComputeShader->GetEmitterFieldBufferSlot();

    if( fieldConstantBufferSlotV != InvalidValueId_ConstantBufferSlotId )
    {
        pCommandBuffer->SetConstantBuffer( fieldConstantBufferSlotV, nn::gfx::ShaderStage_Compute, address, fieldBufferSize );
    }

    // カールノイズテクスチャを設定
    int curlNoiseTexSamplerSlot = pComputeShader->GetCurlNoiseTexSamplerSlot();
    const CurlNoiseTexture& curlNoiseTexture = detail::GetCurlNoiseTexture();
    if( curlNoiseTexSamplerSlot != InvalidValueId_TextureSamplerSlotId )
    {
        pCommandBuffer->SetTextureAndSampler( curlNoiseTexSamplerSlot,
                                              nn::gfx::ShaderStage_Compute,
                                              curlNoiseTexture.GetDescriptorSlot(),
                                              m_pTextureSamplerForCurlNoise->GetDescriptorSlot() );
    }

    size_t size   = sizeof( nn::util::Float4 ) * pEmitter->m_ParticleCount;

    int slot = pComputeShader->GetParticleAttributeBufferSlot( ComputeShader::ComputeShaderAttributeBufferIndex_Pos );
    NN_SDK_ASSERT( slot != InvalidValueId_AttributeId );
    pEmitter->m_GfxObjects.m_DynamicAllocateBuffer.GetGpuAddress( &address, pEmitter->m_ParticlePropertyFront->localPos );
    pCommandBuffer->SetUnorderedAccessBuffer( slot, nn::gfx::ShaderStage_Compute, address, size );

    slot = pComputeShader->GetParticleAttributeBufferSlot( ComputeShader::ComputeShaderAttributeBufferIndex_Vec );
    NN_SDK_ASSERT( slot != InvalidValueId_AttributeId );
    pEmitter->m_GfxObjects.m_DynamicAllocateBuffer.GetGpuAddress( &address, pEmitter->m_ParticlePropertyFront->localVec );
    pCommandBuffer->SetUnorderedAccessBuffer( slot, nn::gfx::ShaderStage_Compute, address, size );

    slot = pComputeShader->GetParticleAttributeBufferSlot( ComputeShader::ComputeShaderAttributeBufferIndex_Diff );
    NN_SDK_ASSERT( slot != InvalidValueId_AttributeId );
    pEmitter->m_GfxObjects.m_DynamicAllocateBuffer.GetGpuAddress( &address, pEmitter->m_ParticlePropertyFront->localDiff );
    pCommandBuffer->SetUnorderedAccessBuffer( slot, nn::gfx::ShaderStage_Compute, address, size );

    slot = pComputeShader->GetParticleAttributeBufferSlot( ComputeShader::ComputeShaderAttributeBufferIndex_Scale );
    if ( slot != InvalidValueId_AttributeId )
    {
        pEmitter->m_GfxObjects.m_DynamicAllocateBuffer.GetGpuAddress(&address, pEmitter->m_ParticlePropertyFront->scale);
        pCommandBuffer->SetUnorderedAccessBuffer(slot, nn::gfx::ShaderStage_Compute, address, size);
    }

    slot = pComputeShader->GetParticleAttributeBufferSlot( ComputeShader::ComputeShaderAttributeBufferIndex_Random );
    if ( slot != InvalidValueId_AttributeId )
    {
        pEmitter->m_GfxObjects.m_DynamicAllocateBuffer.GetGpuAddress(&address, pEmitter->m_ParticlePropertyFront->random);
        pCommandBuffer->SetUnorderedAccessBuffer(slot, nn::gfx::ShaderStage_Compute, address, size);
    }

    slot = pComputeShader->GetParticleAttributeBufferSlot( ComputeShader::ComputeShaderAttributeBufferIndex_MatrixSrt0 );
    if( slot != InvalidValueId_AttributeId )
    {
        if ( pEmitter->m_ParticlePropertyFront->emitterMatrixSrt0 )
        {
            pEmitter->m_GfxObjects.m_DynamicAllocateBuffer.GetGpuAddress( &address, pEmitter->m_ParticlePropertyFront->emitterMatrixSrt0 );
        }
        else
        {
            pEmitter->m_GfxObjects.m_DynamicAllocateBuffer.GetGpuAddress( &address, pEmitter->m_ParticlePropertyFront->random );
        }
        pCommandBuffer->SetUnorderedAccessBuffer( slot, nn::gfx::ShaderStage_Compute, address, size );
    }

    slot = pComputeShader->GetParticleAttributeBufferSlot( ComputeShader::ComputeShaderAttributeBufferIndex_MatrixSrt1 );
    if ( slot != InvalidValueId_AttributeId )
    {
        if ( pEmitter->m_ParticlePropertyFront->emitterMatrixSrt1 )
        {
            pEmitter->m_GfxObjects.m_DynamicAllocateBuffer.GetGpuAddress( &address, pEmitter->m_ParticlePropertyFront->emitterMatrixSrt1 );
        }
        else
        {
            pEmitter->m_GfxObjects.m_DynamicAllocateBuffer.GetGpuAddress( &address, pEmitter->m_ParticlePropertyFront->random );
        }
        pCommandBuffer->SetUnorderedAccessBuffer( slot, nn::gfx::ShaderStage_Compute, address, size );
    }

    slot = pComputeShader->GetParticleAttributeBufferSlot( ComputeShader::ComputeShaderAttributeBufferIndex_MatrixSrt2 );
    if ( slot != InvalidValueId_AttributeId )
    {
        if ( pEmitter->m_ParticlePropertyFront->emitterMatrixSrt2 )
        {
            pEmitter->m_GfxObjects.m_DynamicAllocateBuffer.GetGpuAddress( &address, pEmitter->m_ParticlePropertyFront->emitterMatrixSrt2 );
        }
        else
        {
            pEmitter->m_GfxObjects.m_DynamicAllocateBuffer.GetGpuAddress( &address, pEmitter->m_ParticlePropertyFront->random );
        }
        pCommandBuffer->SetUnorderedAccessBuffer( slot, nn::gfx::ShaderStage_Compute, address, size );
    }

    // TODO : InvalidateMemory / FlushMemory を全コンピュートシェーダ処理の前後で挟む
    pCommandBuffer->InvalidateMemory( nn::gfx::GpuAccess_UnorderedAccessBuffer );
    int dispatchCnt = 0;

    dispatchCnt = static_cast< int >( ::std::ceilf( pEmitter->m_ParticleCount / static_cast< float >( _COMPUTE_SHADER_DISPATCH_COUNT ) ) );
    pCommandBuffer->Dispatch( dispatchCnt, 1, 1 );

    if (doFlush)
    {
        pCommandBuffer->FlushMemory( nn::gfx::GpuAccess_UnorderedAccessBuffer );
    }

    pEmitter->m_ComputeShaderCounter = globalCounter;

    pEmitter->m_IsComputeShaderExecuted = true;

    return true;
} //NOLINT(impl/function_size)


//---------------------------------------------------------------------------
//  エミッタ描画します。
//---------------------------------------------------------------------------
bool EmitterCalculator::Draw( nn::gfx::CommandBuffer* pCommandBuffer, Emitter* pEmitter, void* pUserParam, DrawParameterArg* pDrawParameterArg ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pEmitter );

    if( !pEmitter->m_pEmitterRes->m_IsVisible && pEmitter->m_GroupId == ( SystemParameters_MaxGroupCount - 1 ) )
    {
        return false;
    }

    // エミッタ描画コールバック
    {
        EmitterDrawArg edArg;
        edArg.pCommandBuffer    = pCommandBuffer;
        edArg.pEmitter          = pEmitter;
        edArg.shaderType        = pDrawParameterArg->m_ShaderType;
        edArg.pUserParam        = pUserParam;
        edArg.pDrawParameterArg = pDrawParameterArg;

        if( InvokeEmitterDrawCallback( edArg ) )
        {
            pEmitter->m_IsDrawExecuted = true;
            return true;
        }
    }

    Shader* pShader = pEmitter->GetShader( pDrawParameterArg->m_ShaderType );
    if( !pShader )
    {
        Warning( pEmitter, RuntimeWarningId_NoShaderExists );
        return false;
    }
    pCommandBuffer->SetShader( pShader->GetShader(), nn::gfx::ShaderStageBit_All );

    if ( !DrawEmitterUsingBoundShader( pCommandBuffer, pEmitter, pShader, pUserParam, pDrawParameterArg ) )
    {
        return false;
    }

    pEmitter->m_IsDrawExecuted = true;

    return true;
}


//---------------------------------------------------------------------------
//  エミッタ描画処理（ 下請け ）
//---------------------------------------------------------------------------
bool EmitterCalculator::DrawEmitterUsingBoundShader( nn::gfx::CommandBuffer* pCommandBuffer, Emitter* pEmitter, Shader* pShader, void* pUserParam, DrawParameterArg* pDrawParameterArg ) NN_NOEXCEPT
{
    // 描画設定
    pCommandBuffer->SetBlendState( &pEmitter->m_pEmitterRes->m_RenderState.GetGfxBelndState() );

    // デプスステート
    pCommandBuffer->SetDepthStencilState( &pEmitter->m_pEmitterRes->m_RenderState.GetGfxDepthState() );

    // ラスタライザステート
    pCommandBuffer->SetRasterizerState( &pEmitter->m_pEmitterRes->m_RenderState.GetGfxRasterizerState() );

    // テクスチャ設定
    SetDefaultEmitterTextures( pCommandBuffer, pEmitter, pShader, pDrawParameterArg );

    // 定数バッファ設定
    SetDefaultEmitterConstantBuffers( pCommandBuffer, pEmitter, pShader, pDrawParameterArg );

    // 描画前コールバックの呼び出し
    if( !m_pSystem->InvokeBeforeRenderCallbacks( pCommandBuffer, pEmitter, pDrawParameterArg->m_ShaderType, pUserParam, pDrawParameterArg ) )
    {
        return false;
    }


    //size_t stride = sizeof( nn::util::Float4 );
    size_t size   = sizeof( nn::util::Float4 ) * pEmitter->m_ParticleCount;

    nn::gfx::GpuAddress address;

    // Property頂点バッファを設定
    // TODO : アドレスを初期化時に解決しておく
    int slot = pEmitter->m_pEmitterRes->m_ParticlePropertyBufferSlot[ ParticlePropertyAttributeBufferIndex_Pos ];
    NN_SDK_ASSERT( slot != InvalidValueId_AttributeId );
    pEmitter->m_GfxObjects.m_DynamicAllocateBuffer.GetGpuAddress( &address, pEmitter->m_ParticlePropertyFront->localPos );
    pCommandBuffer->SetVertexBuffer( slot, address, sizeof( nn::util::Float4 ), size );

    slot = pEmitter->m_pEmitterRes->m_ParticlePropertyBufferSlot[ ParticlePropertyAttributeBufferIndex_Vec ];
    NN_SDK_ASSERT( slot != InvalidValueId_AttributeId );
    pEmitter->m_GfxObjects.m_DynamicAllocateBuffer.GetGpuAddress( &address, pEmitter->m_ParticlePropertyFront->localVec );
    pCommandBuffer->SetVertexBuffer( slot, address, sizeof( nn::util::Float4 ), size );

    slot = pEmitter->m_pEmitterRes->m_ParticlePropertyBufferSlot[ ParticlePropertyAttributeBufferIndex_Diff ];
    if( slot != InvalidValueId_AttributeId )
    {
        NN_SDK_ASSERT_NOT_NULL( pEmitter->m_ParticlePropertyFront->localDiff );
        pEmitter->m_GfxObjects.m_DynamicAllocateBuffer.GetGpuAddress( &address, pEmitter->m_ParticlePropertyFront->localDiff );
        pCommandBuffer->SetVertexBuffer( slot, address, sizeof( nn::util::Float4 ), size );
    }

    slot = pEmitter->m_pEmitterRes->m_ParticlePropertyBufferSlot[ ParticlePropertyAttributeBufferIndex_Random ];
    NN_SDK_ASSERT( slot != InvalidValueId_AttributeId );
    pEmitter->m_GfxObjects.m_DynamicAllocateBuffer.GetGpuAddress( &address, pEmitter->m_ParticlePropertyFront->random );
    pCommandBuffer->SetVertexBuffer( slot, address, sizeof( nn::util::Float4 ), size );

    slot = pEmitter->m_pEmitterRes->m_ParticlePropertyBufferSlot[ ParticlePropertyAttributeBufferIndex_Scale ];
    NN_SDK_ASSERT( slot != InvalidValueId_AttributeId );
    pEmitter->m_GfxObjects.m_DynamicAllocateBuffer.GetGpuAddress( &address, pEmitter->m_ParticlePropertyFront->scale );
    pCommandBuffer->SetVertexBuffer( slot, address, sizeof( nn::util::Float4 ), size );

    slot = pEmitter->m_pEmitterRes->m_ParticlePropertyBufferSlot[ ParticlePropertyAttributeBufferIndex_Rotate ];
    if( slot != InvalidValueId_AttributeId )
    {
        NN_SDK_ASSERT_NOT_NULL( pEmitter->m_ParticlePropertyFront->initRotate );
        pEmitter->m_GfxObjects.m_DynamicAllocateBuffer.GetGpuAddress( &address, pEmitter->m_ParticlePropertyFront->initRotate );
        pCommandBuffer->SetVertexBuffer( slot, address, sizeof( nn::util::Float4 ), size );
    }

    slot = pEmitter->m_pEmitterRes->m_ParticlePropertyBufferSlot[ ParticlePropertyAttributeBufferIndex_Color0 ];
    if( slot != InvalidValueId_AttributeId )
    {
        NN_SDK_ASSERT_NOT_NULL( pEmitter->m_ParticlePropertyFront->initColor0 );
        pEmitter->m_GfxObjects.m_DynamicAllocateBuffer.GetGpuAddress( &address, pEmitter->m_ParticlePropertyFront->initColor0 );
        pCommandBuffer->SetVertexBuffer( slot, address, sizeof( nn::util::Float4 ), size );
    }

    slot = pEmitter->m_pEmitterRes->m_ParticlePropertyBufferSlot[ ParticlePropertyAttributeBufferIndex_Color1 ];
    if( slot != InvalidValueId_AttributeId )
    {
        NN_SDK_ASSERT_NOT_NULL( pEmitter->m_ParticlePropertyFront->initColor1 );
        pEmitter->m_GfxObjects.m_DynamicAllocateBuffer.GetGpuAddress( &address, pEmitter->m_ParticlePropertyFront->initColor1 );
        pCommandBuffer->SetVertexBuffer( slot, address, sizeof( nn::util::Float4 ), size );
    }

    slot = pEmitter->m_pEmitterRes->m_ParticlePropertyBufferSlot[ ParticlePropertyAttributeBufferIndex_MatrixSrt0 ];
    if( slot != InvalidValueId_AttributeId )
    {
        NN_SDK_ASSERT_NOT_NULL( pEmitter->m_ParticlePropertyFront->emitterMatrixSrt0 );
        pEmitter->m_GfxObjects.m_DynamicAllocateBuffer.GetGpuAddress( &address, pEmitter->m_ParticlePropertyFront->emitterMatrixSrt0 );
        pCommandBuffer->SetVertexBuffer( slot, address, sizeof( nn::util::Float4 ), size );
    }

    slot = pEmitter->m_pEmitterRes->m_ParticlePropertyBufferSlot[ ParticlePropertyAttributeBufferIndex_MatrixSrt1 ];
    if ( slot != InvalidValueId_AttributeId )
    {
        NN_SDK_ASSERT_NOT_NULL( pEmitter->m_ParticlePropertyFront->emitterMatrixSrt1 );
        pEmitter->m_GfxObjects.m_DynamicAllocateBuffer.GetGpuAddress( &address, pEmitter->m_ParticlePropertyFront->emitterMatrixSrt1 );
        pCommandBuffer->SetVertexBuffer( slot, address, sizeof( nn::util::Float4 ), size );
    }

    slot = pEmitter->m_pEmitterRes->m_ParticlePropertyBufferSlot[ ParticlePropertyAttributeBufferIndex_MatrixSrt2 ];
    if ( slot != InvalidValueId_AttributeId )
    {
        NN_SDK_ASSERT_NOT_NULL( pEmitter->m_ParticlePropertyFront->emitterMatrixSrt2 );
        pEmitter->m_GfxObjects.m_DynamicAllocateBuffer.GetGpuAddress( &address, pEmitter->m_ParticlePropertyFront->emitterMatrixSrt2 );
        pCommandBuffer->SetVertexBuffer( slot, address, sizeof( nn::util::Float4 ), size );
    }

    // 描画形状
    IPrimitive* pPrimitive = pEmitter->m_pEmitterRes->m_pPrimitive;
    if ( pPrimitive )
    {
        for ( int i = 0; i < pPrimitive->GetBufferCount(); i++ )
        {
            pPrimitive->GetVertexBufferGpuAddress( &address, i );

            pCommandBuffer->SetVertexBuffer( i,
                address,
                pPrimitive->GetStrideSize( i ),
                pPrimitive->GetVertexBufferSize( i ) );
        }
    }

    // 頂点ステート
    pCommandBuffer->SetVertexState( &pEmitter->m_pEmitterRes->m_VertexState );

    // 描画
    if( pEmitter->m_pEmitterData->emitter.sortType == ParticleSortType_NoSort )
    {
        // ソートせずに描画
        if ( pPrimitive )
        {
            pPrimitive->Draw( pCommandBuffer, pEmitter->m_ParticleCount, pEmitter->m_PrimitiveLodLevel );
        }
        else
        {
            pCommandBuffer->Draw( nn::gfx::PrimitiveTopology_TriangleStrip, 4, 0, pEmitter->m_ParticleCount, 0 );
        }
    }
    else
    {
        // ソートして描画
        EntrySortedParticle( pCommandBuffer, pEmitter, pShader, pPrimitive, pDrawParameterArg );
    }

    return true;
} //NOLINT(impl/function_size)

//---------------------------------------------------------------------------
//  ソート済みパーティクル描画処理を行います。
//---------------------------------------------------------------------------
bool EmitterCalculator::EntrySortedParticle( nn::gfx::CommandBuffer* pCommandBuffer, Emitter* pEmitter, Shader* pShader, IPrimitive* pPrimitive, DrawParameterArg* pDrawParameterArg ) NN_NOEXCEPT
{
    NN_UNUSED( pShader );

    // パーティクルのソート
    // ソートした配列の先頭＆配列サイズ
    int sortSize = 0;
    SortData* pSortDataHead = NULL;
    {
        const ParticleSortType sortType = static_cast<ParticleSortType>( pEmitter->m_pEmitterRes->m_pResEmitter->emitter.sortType );

        // MEMO: この時点で m_Time が一つ進んでいるので、 GPU に送った値である m_FrameRate を引いた分に戻して計算させる
        m_pSystem->GetSortedParticleList( &pSortDataHead, &sortSize, pEmitter, sortType, ( pEmitter->m_Time - pEmitter->m_FrameRate ), pDrawParameterArg->m_ProccesingIndex );
        if( sortSize == 0 )
        {
            return false;
        }
    }

    // ソートされた内容に従って１パーティクルづつ描画する。
    for( int i = 0; i < sortSize; i++ )
    {
        const int index = pSortDataHead[ i ].index;
        ParticleData* pParticleData = &pEmitter->m_pParticleData[ index ];
        const float time = pParticleData->GetTime( pEmitter->m_Time );
        const float life = pParticleData->GetLife();

        if( time <= life && pParticleData->createId != 0 && pParticleData->life > 0 )
        {
            if ( pPrimitive )
            {
                pPrimitive->Draw( pCommandBuffer, 1, pSortDataHead[ i ].index, pEmitter->m_PrimitiveLodLevel );
            }
            else
            {
                pCommandBuffer->Draw( nn::gfx::PrimitiveTopology_TriangleStrip, 4, 0, 1, pSortDataHead[ i ].index );
            }
        }
    }

    return true;
}

} // namespace detail
} // namespace vfx
} // namespace nn
