﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vfx/vfx_Emitter.h>
#include <nn/vfx/vfx_Resource.h>
#include <nn/vfx/vfx_Random.h>
#include <nn/vfx/vfx_System.h>
#include <nn/vfx/vfx_MemUtil.h>
#include <nn/vfx/vfx_UniformBlock.h>

namespace nn {
namespace vfx {


//---------------------------------------------------------------------------
//  エミッタ初期化を行います。
//---------------------------------------------------------------------------
bool Emitter::Initialize( nn::gfx::Device*       pDevice,
                          EmitterSet*            emitterSet,
                          const EmitterResource* pEmitterResource,
                          int                    maxParticleCount,
                          Heap*                  pDebugHeap,
                          Emitter*               pParentEmitter ) NN_NOEXCEPT
{
    NN_UNUSED( pDebugHeap );

    // 事前にResetが呼ばれているので必要な箇所のみ初期化
    m_pEmitterSet                = emitterSet;
    m_pEmitterData               = pEmitterResource->m_pResEmitter;
    m_pEmitterRes                = pEmitterResource;
    m_FadeOutAlphaValue          = 1.0f;
    m_FadeInAlphaValue           =
        (( pEmitterResource->m_pResEmitter->emitter.isAlphaFadeIn ) || ( pEmitterResource->m_pResEmitter->emitter.isScaleFadeIn ))
        ? 0.0f : 1.0f;    // MEMO: フェードインを使わない場合は 1.0f を先に埋めておく。
    m_MaxParticleAssignmentCount = maxParticleCount;
    m_ParticleDynamicBuffer      = NULL;
    m_ProcessEmitterFlag         = 1; // StreamOut用フラグ: 最下位ビットだけ初期値として立てておく
    nn::util::MatrixIdentity( &m_MatrixSrt );
    nn::util::MatrixIdentity( &m_MatrixRt );

    if( pEmitterResource->m_ChildEmitterResSet[ 0 ] )
    {
        for( int i = 0; i < SystemParameters_MaxEmitterInclusionCount; i++ )
        {
            m_pChildEmitterResSet[ i ] = pEmitterResource->m_ChildEmitterResSet[ i ];
        }
    }

    m_EmitRatio = 1.0f;
    m_EmitIntervalRatio = 1.0f;
    m_ParticleLifeScale = 1.0f;
    m_GroupId = m_pEmitterSet->GetGroupId();
    m_BufferSide = BufferSide_FrontBuffer;
    if ( pEmitterResource->m_IsChildEmitter )
    {
        m_IsChild = true;
    }
    else
    {
        m_IsChild = false;
    }
    m_IsVisible = true;
    m_DrawViewFlag = DrawViewFlag_All;
    m_IsEnabledEmit = true;
    m_AbortEmitSearch = false;
    m_ForceAbortEmitSearch = true;

    m_IsComputeShaderExecuted = true;

    // 乱数を初期化
    detail::RandomGenerator* rgenerator = detail::Random::GetGlobalRandom();
    const uint32_t seed = static_cast< uint32_t >( m_pEmitterData->emitter.randomSeed * 12345678901 );
    switch( m_pEmitterData->emitter.randomSeedType )
    {
    case detail::EmitterRandomSeedType_Random:
        m_Random.SetSeed( rgenerator->GetUnsignedInteger() );
        break;
    case detail::EmitterRandomSeedType_Eset:
        m_Random.SetSeed( m_pEmitterSet->GetEmitterSetRandomSeed() );
        break;
    case detail::EmitterRandomSeedType_Fixed:
        m_Random.SetSeed( seed );
        break;
    default:
        break;
    }

    // リソース更新時処理
    if( !ResourceUpdate( pDevice, pParentEmitter ) )
    {
        m_IsEmitterInitializeFailed = true;
        return false;
    }

    // 初回放出の為、インターバルを例外設定
    m_EmitInterval = 0.0f;

    // エミッタセット位置を前フレーム位置として扱う。
    nn::util::MatrixGetAxisW( &m_EmitterPrevPos, m_pEmitterSet->GetMatrixSrt() );
    nn::util::VectorSet( &m_EmitterLocalVec, 0, 0, 0 );

    // 初期化時の親エミッタセット生成IDを保持する
    m_EmitterSetCreateId = m_pEmitterSet->GetCreateId();

    // エミッタカラーを白で初期化
    nn::util::VectorSet( &m_Color0, 1, 1, 1, 1 );
    nn::util::VectorSet( &m_Color1, 1, 1, 1, 1 );

    // エミッタ初期化後処理後コールバック
    EmitterInitializeArg arg;
    arg.pEmitter = this;

    if ( InvokeEmitterInitializeCallback( arg ) )
    {
        return true;
    }

    return false;
}


//---------------------------------------------------------------------------
//  メンバを再初期化します。
//---------------------------------------------------------------------------
void Emitter::Reset() NN_NOEXCEPT
{
    const char* ptrThis = reinterpret_cast< char* >( this );
    const char* ptrGfx  = reinterpret_cast< char* >( &this->m_GfxObjects );
    const ptrdiff_t diff = ptrGfx - ptrThis;
    detail::MemUtil::FillZero( this, diff );

    m_DynamicHeap.Initialize();
}

//---------------------------------------------------------------------------
//  確保しているバッファの解放処理を行います。
//---------------------------------------------------------------------------
void Emitter::ReleaseBuffer( nn::gfx::Device* pDevice ) NN_NOEXCEPT
{
    NN_UNUSED( pDevice );

    if( m_ParticleDynamicBuffer )
    {
        m_DynamicHeap.Free( m_ParticleDynamicBuffer );
        m_ParticleDynamicBuffer = NULL;
    }

    m_GfxObjects.m_DynamicAllocateBuffer.Finalize();
}

//---------------------------------------------------------------------------
//  カスタム定数バッファの初期化を行います。
//---------------------------------------------------------------------------
bool Emitter::InitializeCustomConstantBuffer( int index, size_t customConstantBufferSize ) NN_NOEXCEPT
{
    m_GfxObjects.m_CustomConstantBuffer[ index ].Initialize( m_GfxObjects.m_pBufferAllocator, customConstantBufferSize );
    return true;
}

//---------------------------------------------------------------------------
//  カスタム頂点アトリビュートの初期化を行います。
//---------------------------------------------------------------------------
bool Emitter::InitializeCustomAttribute( int bufferCount, size_t bufferSize ) NN_NOEXCEPT
{
    return m_GfxObjects.m_ParticleEmitterPluginAttribute.Initialize( m_GfxObjects.m_pBufferAllocator, bufferCount, bufferSize );
}

//---------------------------------------------------------------------------
//  アトリビュートのダブルバッファリング状態のチェック
//---------------------------------------------------------------------------
void Emitter::CheckParticleAttributeDoubleBufferingState() NN_NOEXCEPT
{
    //---------------------------------------------------------------------------
    // 判定条件:
    // ・CPUエミッタであること
    // ・カスタムアクション／カスタムシェーダ／エミッタプラグインに ParticleCalc コールバックがある
    //---------------------------------------------------------------------------
    m_IsParticleAttributeDoubleBuffer = m_pEmitterData->emitter.calcType == detail::EmitterCalculationMode_Cpu &&
        ( ( m_pEmitterData->action.customActionIndex > 0 &&
        m_pCallbackSet[ detail::CallbackSetType_CustomAction ] != NULL && m_pCallbackSet[ detail::CallbackSetType_CustomAction ]->particleCalculate != NULL ) ||
        ( m_pEmitterData->shader.customShaderIndex > 0 &&
        m_pCallbackSet[ detail::CallbackSetType_CustomShader ] != NULL && m_pCallbackSet[ detail::CallbackSetType_CustomShader ]->particleCalculate != NULL ) ||
        ( m_pEmitterRes->m_EmitterPluginIndex > 0 &&
        m_pCallbackSet[ detail::CallbackSetType_EmitterPlugin ] != NULL && m_pCallbackSet[ detail::CallbackSetType_EmitterPlugin ]->particleCalculate != NULL ) );
}

//---------------------------------------------------------------------------
//  エミッタが必要とするパーティクルの最大確保数を計算する
//---------------------------------------------------------------------------
int Emitter::CalculateRequiredParticleAsignmentCount( const Emitter* pParentEmitter ) const NN_NOEXCEPT
{
    const detail::ResEmitter* pData = m_pEmitterData;
    int requiredParticleCount = 0;

    if( pData->emission.isEmitDistEnabled )
    {
        //---------------------------------------------------------------------------
        //  距離放出
        //---------------------------------------------------------------------------
        requiredParticleCount = pData->emission.emitDistParticleMax;  // UI で設定した値で決め打ち。
    }
    else
    {
        //---------------------------------------------------------------------------
        //  時間放出
        //---------------------------------------------------------------------------
        float maxEmitRatio          = 0; // （アニメーションを含む）最大の放出レート
        float maxParticleLife       = 0; // （アニメーションを含む）最大のパーティクル寿命
        int   maxEmitCountUntilPeak = 0; // 最初に生んだパーティクルが消えるまでに最大何回放出が行われるか

        // TODO: この計算、もうちょっと数を詰められそう
        if( m_pEmitterRes->m_EmitterAnimationArray[ detail::EmitterAnimationType_EmissionRate ] )
        {
            maxEmitRatio = GetEmissionRateMax();
            if( maxEmitRatio < 1.0f )
            {
                maxEmitRatio = 1.0f;
            }
        }
        else
        {
            maxEmitRatio = pData->emission.rate;
        }

        // TODO: この計算、もうちょっと数を詰められそう
        if( m_pEmitterRes->m_EmitterAnimationArray[ detail::EmitterAnimationType_ParticleLife ] )
        {
            maxParticleLife = static_cast< float >( GetParticleLifeMax() );
        }
        else
        {
            maxParticleLife = static_cast< float >( pData->ptcl.life );
        }

        // 放出の回数の推定
        const bool isOneTime        = ( pData->emission.isOneTime > 0 );    // ワンタイムかどうか
        const bool isLifeInfinity   = ( pData->ptcl.isLifeInfinity > 0 );   // 寿命無限かどうか
        const bool useComputeShader = ( pData->emitter.calcType == detail::EmitterCalculationMode_GpuStreamOut );         // コンピュートシェーダかどうか
        const int  emitDuration     = ( pData->emission.emitDuration );     // 放出時間
        const int  emitInterval     = ( pData->emission.interval );         // 放出間隔

        // 放出期間が終わるまでに何回放出機会があったか
        const float emitTime = ( emitDuration == 0 ) ? 1.0f : static_cast< float >( emitDuration );
        const int maxEmitCountUntilEmitEnd = static_cast< int >( ::std::ceilf( emitTime        / ( emitInterval + 1 ) ) );

        // 粒の寿命が終わるまでに何回放出機会があったか
        const int maxEmitCountUntilLifeEnd = static_cast< int >( ::std::ceilf( maxParticleLife / ( emitInterval + 1 ) ) );

        if( isLifeInfinity && isOneTime )
        {
            // 寿命無限（※ワンタイム時限定）
            // 放出回数：( 放出時間 / (放出間隔＋１) ) の切り上げ。
            maxEmitCountUntilPeak = maxEmitCountUntilEmitEnd;
        }
        else
        {
            if( isOneTime && ( maxEmitCountUntilEmitEnd < maxEmitCountUntilLifeEnd ) )
            {
                // ワンタイムの時だけ、放出時間による放出回数と比べて、小さい方を採用
                // 放出回数：( 放出時間 / (放出間隔＋１) ) の切り上げ。
                maxEmitCountUntilPeak = maxEmitCountUntilEmitEnd;
            }
            else
            {
                // それ以外のケースでは、パーティクルの寿命が尽きるまでに放出された回数を採用
                // 放出回数：(寿命÷ (放出間隔＋１) ) の切り上げ。
                maxEmitCountUntilPeak = maxEmitCountUntilLifeEnd;
            }
        }

        // 放出レートが小数の場合、多めに切り上げる
        requiredParticleCount = static_cast< int >( ::std::ceilf( maxEmitCountUntilPeak * maxEmitRatio ) + ::std::ceilf( maxEmitRatio ) );

        // 等分割系エミッタ x 一斉放出 の場合の補正
        if( pData->volume.primEmitType == detail::PrimitiveEmissionType_Unison )
        {
            // 等分割系の一斉放出の場合は、放出レートを分割数倍にする
            if( pData->volume.volumeType == detail::EmitterVolumeType_CircleEquallyDivided )
            {
                requiredParticleCount *= pData->volume.numDivideCircle;
            }
            else if( pData->volume.volumeType == detail::EmitterVolumeType_LineEquallyDivided )
            {
                requiredParticleCount *= pData->volume.numDivideLine;
            }
        }

        // ループかつコンピュートシェーダ時は、一回の放出分のバッファをさらにプラスする
        if ( !isOneTime && useComputeShader )
        {
            int addBuffer = static_cast< int >( ::std::ceilf( maxEmitRatio ) );

            if ( pData->volume.primEmitType == detail::PrimitiveEmissionType_Unison )
            {
                // 等分割系の一斉放出の場合は、放出レートを分割数倍にする
                if ( pData->volume.volumeType == detail::EmitterVolumeType_CircleEquallyDivided )
                {
                    addBuffer *= pData->volume.numDivideCircle;
                }
                else if ( pData->volume.volumeType == detail::EmitterVolumeType_LineEquallyDivided )
                {
                    addBuffer *= pData->volume.numDivideLine;
                }
            }

            requiredParticleCount += addBuffer;
        }
    }

    //---------------------------------------------------------------------------
    //  「軽量版チャイルド」の場合、エミッタは共通で持つので、
    //  最大パーティクル数を、親エミッタが出すパーティクル数（※子エミッタ数）分乗算する。
    //---------------------------------------------------------------------------
    if( m_IsChild && !m_pEmitterData->inherit.enableEmitterParticle )
    {
        NN_SDK_ASSERT( pParentEmitter );
        requiredParticleCount *= pParentEmitter->GetParticleAttrFillMax();
    }

    return requiredParticleCount;
}

//---------------------------------------------------------------------------
//  リソース更新時処理を行います。
//---------------------------------------------------------------------------
bool Emitter::ResourceUpdate( nn::gfx::Device* pDevice, Emitter* pParentEmitter ) NN_NOEXCEPT
{
    int maxParticleCount = 0;

    // ダブルバッファリング状態の更新
    CheckParticleAttributeDoubleBufferingState();

    // パーティクル運用バッファを確保する
    if( m_MaxParticleAssignmentCount != 0 )
    {
        //---------------------------------------------------------------------------
        // パーティクル最大数がアプリ側から指定された場合（マニュアル放出）
        //---------------------------------------------------------------------------
        maxParticleCount = m_MaxParticleAssignmentCount;
    }
    else
    {
        //---------------------------------------------------------------------------
        //  エミッタが必要とするパーティクルの確保数を計算する
        //---------------------------------------------------------------------------
        maxParticleCount = CalculateRequiredParticleAsignmentCount( pParentEmitter );

        //---------------------------------------------------------------------------
        //  最終的に maxParticleCount が 0 判定された場合は警告を出して処理を返す（設定がおかしい）
        //---------------------------------------------------------------------------
        if( maxParticleCount == 0 )
        {
            detail::Warning( reinterpret_cast< void* >( this ), detail::RuntimeWarningId_ParticleMaxCountIsZero );
            return false;
        }
    }

    if( m_MaxParticleAttrFill != maxParticleCount )
    {
        // パーティクル削除コールバック
        InvokeParticleRemoveCallbackAll( this );
        ReleaseBuffer( pDevice );

        // 動的ヒープからバッファを確保
        // TODO : なるべく : EffectMaker経由の生成時は、Debugバッファから確保するように
        {
            bool isCpuEmitter = false;
            if( m_pEmitterData->emitter.calcType == detail::EmitterCalculationMode_Cpu )
            {
                isCpuEmitter = true;
            }
            const bool isTripleBufferEnabled = GetEmitterSet()->GetSystem()->IsEnableTripleBuffer();

            // 確保するメモリサイズを決定する
            int tempParticleCount = maxParticleCount;
            if( tempParticleCount % 2 == 1 )
            {
                tempParticleCount++;
            }

            // コンピュートシェーダを利用する場合は、パーティクル数が 32 の倍数になるようにバッファを生成する
            if ( m_pEmitterData->emitter.calcType == detail::EmitterCalculationMode_GpuStreamOut )
            {
                tempParticleCount = nn::util::align_up( tempParticleCount, 32 );
            }

            // コンスタントバッファサイズを計算する
            detail::BufferSizeCalculator bufferSizeCalculator( pDevice, nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_UnorderedAccessBuffer );
            size_t particleBufferSize = tempParticleCount * sizeof( nn::util::Float4 );

            {
                // Dynamics
                bufferSizeCalculator.AddBufferSize( particleBufferSize );
                bufferSizeCalculator.AddBufferSize( particleBufferSize );
                bufferSizeCalculator.AddBufferSize( particleBufferSize );
                if( isCpuEmitter )
                {
                    bufferSizeCalculator.AddBufferSize( particleBufferSize );
                    bufferSizeCalculator.AddBufferSize( particleBufferSize );
                    bufferSizeCalculator.AddBufferSize( particleBufferSize );
                    if( isTripleBufferEnabled )
                    {
                        bufferSizeCalculator.AddBufferSize( particleBufferSize );
                        bufferSizeCalculator.AddBufferSize( particleBufferSize );
                        bufferSizeCalculator.AddBufferSize( particleBufferSize );
                    }
                }

                // scale / rotate
                bufferSizeCalculator.AddBufferSize( particleBufferSize );
                bufferSizeCalculator.AddBufferSize( particleBufferSize );
                if( isCpuEmitter )
                {
                    bufferSizeCalculator.AddBufferSize( particleBufferSize );
                    bufferSizeCalculator.AddBufferSize( particleBufferSize );
                    if( isTripleBufferEnabled )
                    {
                        bufferSizeCalculator.AddBufferSize( particleBufferSize );
                        bufferSizeCalculator.AddBufferSize( particleBufferSize );
                    }
                }

                // random
                bufferSizeCalculator.AddBufferSize( particleBufferSize );

                // color0 / 1
                if ( m_IsChild )
                {
                    bufferSizeCalculator.AddBufferSize( particleBufferSize );
                    bufferSizeCalculator.AddBufferSize( particleBufferSize );
                    if( isCpuEmitter )
                    {
                        bufferSizeCalculator.AddBufferSize( particleBufferSize );
                        bufferSizeCalculator.AddBufferSize( particleBufferSize );
                        if( isTripleBufferEnabled )
                        {
                            bufferSizeCalculator.AddBufferSize( particleBufferSize );
                            bufferSizeCalculator.AddBufferSize( particleBufferSize );
                        }
                    }
                }

                // matrix
                if ( m_pEmitterData->emitter.followType != detail::ParticleFollowType_EmitterFull )
                {
                    bufferSizeCalculator.AddBufferSize( particleBufferSize );
                    bufferSizeCalculator.AddBufferSize( particleBufferSize );
                    bufferSizeCalculator.AddBufferSize( particleBufferSize );
                }
            }

            // コンスタントバッファを一括で確保
            if ( !m_GfxObjects.m_DynamicAllocateBuffer.Initialize( m_GfxObjects.m_pBufferAllocator, bufferSizeCalculator.GetBufferSize() ) )
            {
                return false;
            }

            // バッファを切り分け
            {
                m_GfxObjects.m_DynamicAllocateBuffer.Begin();

                // Dynamics
                m_ParticleProperty[0].localPos              = reinterpret_cast<nn::util::Float4*>( m_GfxObjects.m_DynamicAllocateBuffer.Cut( particleBufferSize ) );
                m_ParticleProperty[0].localVec              = reinterpret_cast<nn::util::Float4*>( m_GfxObjects.m_DynamicAllocateBuffer.Cut( particleBufferSize ) );
                m_ParticleProperty[0].localDiff             = reinterpret_cast<nn::util::Float4*>( m_GfxObjects.m_DynamicAllocateBuffer.Cut( particleBufferSize ) );
                if( isCpuEmitter )
                {
                    m_ParticleProperty[1].localPos          = reinterpret_cast<nn::util::Float4*>( m_GfxObjects.m_DynamicAllocateBuffer.Cut( particleBufferSize ) );
                    m_ParticleProperty[1].localVec          = reinterpret_cast<nn::util::Float4*>( m_GfxObjects.m_DynamicAllocateBuffer.Cut( particleBufferSize ) );
                    m_ParticleProperty[1].localDiff         = reinterpret_cast<nn::util::Float4*>( m_GfxObjects.m_DynamicAllocateBuffer.Cut( particleBufferSize ) );
                    if( isTripleBufferEnabled )
                    {
                        m_ParticleProperty[ 2 ].localPos    = reinterpret_cast< nn::util::Float4* >( m_GfxObjects.m_DynamicAllocateBuffer.Cut( particleBufferSize ) );
                        m_ParticleProperty[ 2 ].localVec    = reinterpret_cast< nn::util::Float4* >( m_GfxObjects.m_DynamicAllocateBuffer.Cut( particleBufferSize ) );
                        m_ParticleProperty[ 2 ].localDiff   = reinterpret_cast< nn::util::Float4* >( m_GfxObjects.m_DynamicAllocateBuffer.Cut( particleBufferSize ) );
                    }
                }
                else
                {
                    m_ParticleProperty[1].localPos          = m_ParticleProperty[0].localPos;
                    m_ParticleProperty[1].localVec          = m_ParticleProperty[0].localVec;
                    m_ParticleProperty[1].localDiff         = m_ParticleProperty[0].localDiff;
                    m_ParticleProperty[2].localPos          = m_ParticleProperty[0].localPos;
                    m_ParticleProperty[2].localVec          = m_ParticleProperty[0].localVec;
                    m_ParticleProperty[2].localDiff         = m_ParticleProperty[0].localDiff;
                }

                // scale / rotate
                m_ParticleProperty[0].scale                 = reinterpret_cast<nn::util::Float4*>( m_GfxObjects.m_DynamicAllocateBuffer.Cut( particleBufferSize ) );
                m_ParticleProperty[0].initRotate            = reinterpret_cast<nn::util::Float4*>( m_GfxObjects.m_DynamicAllocateBuffer.Cut( particleBufferSize ) );
                if( isCpuEmitter )
                {
                    m_ParticleProperty[1].scale             = reinterpret_cast<nn::util::Float4*>( m_GfxObjects.m_DynamicAllocateBuffer.Cut( particleBufferSize ) );
                    m_ParticleProperty[1].initRotate        = reinterpret_cast<nn::util::Float4*>( m_GfxObjects.m_DynamicAllocateBuffer.Cut( particleBufferSize ) );
                    if( isTripleBufferEnabled )
                    {
                        m_ParticleProperty[2].scale         = reinterpret_cast< nn::util::Float4* >( m_GfxObjects.m_DynamicAllocateBuffer.Cut( particleBufferSize ) );
                        m_ParticleProperty[2].initRotate    = reinterpret_cast< nn::util::Float4* >( m_GfxObjects.m_DynamicAllocateBuffer.Cut( particleBufferSize ) );
                    }
                }
                else
                {
                    m_ParticleProperty[1].scale             = m_ParticleProperty[0].scale;
                    m_ParticleProperty[1].initRotate        = m_ParticleProperty[0].initRotate;
                    m_ParticleProperty[2].scale             = m_ParticleProperty[0].scale;
                    m_ParticleProperty[2].initRotate        = m_ParticleProperty[0].initRotate;
                }

                // random
                m_ParticleProperty[0].random                = reinterpret_cast<nn::util::Float4*>( m_GfxObjects.m_DynamicAllocateBuffer.Cut( particleBufferSize ) );
                m_ParticleProperty[1].random                = m_ParticleProperty[0].random;
                m_ParticleProperty[2].random                = m_ParticleProperty[0].random;

                // color0 / 1
                if ( m_IsChild )
                {
                    m_ParticleProperty[0].initColor0        = reinterpret_cast<nn::util::Float4*>( m_GfxObjects.m_DynamicAllocateBuffer.Cut( particleBufferSize ) );
                    m_ParticleProperty[0].initColor1        = reinterpret_cast<nn::util::Float4*>( m_GfxObjects.m_DynamicAllocateBuffer.Cut( particleBufferSize ) );

                    if( isCpuEmitter )
                    {
                        m_ParticleProperty[1].initColor0    = reinterpret_cast<nn::util::Float4*>( m_GfxObjects.m_DynamicAllocateBuffer.Cut( particleBufferSize ) );
                        m_ParticleProperty[1].initColor1    = reinterpret_cast<nn::util::Float4*>( m_GfxObjects.m_DynamicAllocateBuffer.Cut( particleBufferSize ) );
                        if( isTripleBufferEnabled )
                        {
                            m_ParticleProperty[2].initColor0 = reinterpret_cast<nn::util::Float4*>( m_GfxObjects.m_DynamicAllocateBuffer.Cut( particleBufferSize ) );
                            m_ParticleProperty[2].initColor1 = reinterpret_cast<nn::util::Float4*>( m_GfxObjects.m_DynamicAllocateBuffer.Cut( particleBufferSize ) );
                        }
                    }
                    else
                    {
                        m_ParticleProperty[1].initColor0    = m_ParticleProperty[0].initColor0;
                        m_ParticleProperty[1].initColor1    = m_ParticleProperty[0].initColor1;
                        m_ParticleProperty[2].initColor0    = m_ParticleProperty[0].initColor0;
                        m_ParticleProperty[2].initColor1    = m_ParticleProperty[0].initColor1;
                    }
                }
                else
                {
                    m_ParticleProperty[0].initColor0        = NULL;
                    m_ParticleProperty[0].initColor1        = NULL;
                    m_ParticleProperty[1].initColor0        = NULL;
                    m_ParticleProperty[1].initColor1        = NULL;
                }

                // matrix
                if ( m_pEmitterData->emitter.followType != detail::ParticleFollowType_EmitterFull )
                {
                    m_ParticleProperty[0].emitterMatrixSrt0   = reinterpret_cast<nn::util::Float4*>( m_GfxObjects.m_DynamicAllocateBuffer.Cut( particleBufferSize ) );
                    m_ParticleProperty[0].emitterMatrixSrt1   = reinterpret_cast<nn::util::Float4*>( m_GfxObjects.m_DynamicAllocateBuffer.Cut( particleBufferSize ) );
                    m_ParticleProperty[0].emitterMatrixSrt2   = reinterpret_cast<nn::util::Float4*>( m_GfxObjects.m_DynamicAllocateBuffer.Cut( particleBufferSize ) );
                    m_ParticleProperty[1].emitterMatrixSrt0   = m_ParticleProperty[0].emitterMatrixSrt0;
                    m_ParticleProperty[1].emitterMatrixSrt1   = m_ParticleProperty[0].emitterMatrixSrt1;
                    m_ParticleProperty[1].emitterMatrixSrt2   = m_ParticleProperty[0].emitterMatrixSrt2;
                    m_ParticleProperty[2].emitterMatrixSrt0   = m_ParticleProperty[0].emitterMatrixSrt0;
                    m_ParticleProperty[2].emitterMatrixSrt1   = m_ParticleProperty[0].emitterMatrixSrt1;
                    m_ParticleProperty[2].emitterMatrixSrt2   = m_ParticleProperty[0].emitterMatrixSrt2;
                }
                else
                {
                    m_ParticleProperty[0].emitterMatrixSrt0   = NULL;
                    m_ParticleProperty[0].emitterMatrixSrt1   = NULL;
                    m_ParticleProperty[0].emitterMatrixSrt2   = NULL;
                    m_ParticleProperty[1].emitterMatrixSrt0   = NULL;
                    m_ParticleProperty[1].emitterMatrixSrt1   = NULL;
                    m_ParticleProperty[1].emitterMatrixSrt2   = NULL;
                    m_ParticleProperty[2].emitterMatrixSrt0   = NULL;
                    m_ParticleProperty[2].emitterMatrixSrt1   = NULL;
                    m_ParticleProperty[2].emitterMatrixSrt2   = NULL;
                }

                m_GfxObjects.m_DynamicAllocateBuffer.End();
            }

            m_ParticlePropertyFront = &m_ParticleProperty[ m_BufferSide ];

            // パーティクルデータ用バッファ( CPUからのアクセスのみ )
            {
                BufferCutter cutter;

                const size_t particleDataSize = tempParticleCount * sizeof( detail::ParticleData );
                const size_t parentParticleDataSize = ( m_pEmitterRes->m_ChildEmitterResCount ) ? tempParticleCount * sizeof( detail::ParentParticleData ) : 0;

                // CPU演算用パーティクルバッファ
                int attrCount = 0;
                if( isCpuEmitter )
                {
                    attrCount = 6;

                    // カラー用アトリビュート
                    if ( m_IsChild )
                    {
                        attrCount += 2;
                    }
                    // カラー用アトリビュート
                    if ( m_pEmitterData->emitter.followType != detail::ParticleFollowType_EmitterFull )
                    {
                        attrCount += 6;
                    }
                }

                const size_t cpuParticleBufferSize = tempParticleCount * sizeof( nn::util::Float4 ) * attrCount;
                const size_t workSize = particleDataSize + parentParticleDataSize + cpuParticleBufferSize;

                m_ParticleDynamicBuffer = m_DynamicHeap.Alloc( workSize, 32 );
                if( !m_ParticleDynamicBuffer )
                {
                    return false;
                }
                cutter.Initialize( m_ParticleDynamicBuffer, workSize );

                m_pParticleData = reinterpret_cast< detail::ParticleData* >( cutter.Cut( particleDataSize ) );
                detail::MemUtil::FillZero( m_pParticleData, particleDataSize );

                if( isCpuEmitter )
                {
                    m_CpuParticleProperty.localPos          = reinterpret_cast< nn::util::Float4* >( cutter.Cut( tempParticleCount * sizeof( nn::util::Float4 ) ) );
                    m_CpuParticleProperty.localVec          = reinterpret_cast< nn::util::Float4* >( cutter.Cut( tempParticleCount * sizeof( nn::util::Float4 ) ) );
                    m_CpuParticleProperty.localDiff         = reinterpret_cast< nn::util::Float4* >( cutter.Cut( tempParticleCount * sizeof( nn::util::Float4 ) ) );
                    m_CpuParticleProperty.scale             = reinterpret_cast< nn::util::Float4* >( cutter.Cut( tempParticleCount * sizeof( nn::util::Float4 ) ) );
                    m_CpuParticleProperty.random            = reinterpret_cast< nn::util::Float4* >( cutter.Cut( tempParticleCount * sizeof( nn::util::Float4 ) ) );
                    m_CpuParticleProperty.initRotate        = reinterpret_cast< nn::util::Float4* >( cutter.Cut( tempParticleCount * sizeof( nn::util::Float4 ) ) );
                    if ( m_IsChild )
                    {
                        m_CpuParticleProperty.initColor0        = reinterpret_cast< nn::util::Float4* >( cutter.Cut( tempParticleCount * sizeof( nn::util::Float4 ) ) );
                        m_CpuParticleProperty.initColor1        = reinterpret_cast< nn::util::Float4* >( cutter.Cut( tempParticleCount * sizeof( nn::util::Float4 ) ) );
                    }
                    if ( m_pEmitterData->emitter.followType != detail::ParticleFollowType_EmitterFull )
                    {
                        m_CpuParticleProperty.emitterMatrixSrt0 = reinterpret_cast< nn::util::Float4* >( cutter.Cut( tempParticleCount * sizeof( nn::util::Float4 ) ) );
                        m_CpuParticleProperty.emitterMatrixSrt1 = reinterpret_cast< nn::util::Float4* >( cutter.Cut( tempParticleCount * sizeof( nn::util::Float4 ) ) );
                        m_CpuParticleProperty.emitterMatrixSrt2 = reinterpret_cast< nn::util::Float4* >( cutter.Cut( tempParticleCount * sizeof( nn::util::Float4 ) ) );
                    }
                }
                else
                {
                    // MEMO: CPUエミッタでない場合は、0番バッファのアドレスを入れておく。
                    //       マニュアル放出時にコールバック経由で ParticleProperty を書き換える必要があるため。
                    m_CpuParticleProperty.localPos          = m_ParticleProperty[0].localPos;
                    m_CpuParticleProperty.localVec          = m_ParticleProperty[0].localVec;
                    m_CpuParticleProperty.localDiff         = m_ParticleProperty[0].localDiff;
                    m_CpuParticleProperty.scale             = m_ParticleProperty[0].scale;
                    m_CpuParticleProperty.random            = m_ParticleProperty[0].random;
                    m_CpuParticleProperty.initRotate        = m_ParticleProperty[0].initRotate;
                    m_CpuParticleProperty.initColor0        = m_ParticleProperty[0].initColor0;
                    m_CpuParticleProperty.initColor1        = m_ParticleProperty[0].initColor1;
                    m_CpuParticleProperty.emitterMatrixSrt0 = m_ParticleProperty[0].emitterMatrixSrt0;
                    m_CpuParticleProperty.emitterMatrixSrt1 = m_ParticleProperty[0].emitterMatrixSrt1;
                    m_CpuParticleProperty.emitterMatrixSrt2 = m_ParticleProperty[0].emitterMatrixSrt2;
                }

                if( m_pEmitterRes->m_ChildEmitterResCount )
                {
                    // 親パーティクル用ワーク
                    m_pParentParticleData = reinterpret_cast< detail::ParentParticleData* >( cutter.Cut( parentParticleDataSize ) );
                    detail::MemUtil::FillZero( m_pParentParticleData, parentParticleDataSize );
                }
                else
                {
                    m_pParentParticleData = NULL;
                }
            }
        }

        m_MaxParticleAttrFill = maxParticleCount;
        m_ParticleAttrFillIndex = 0;
        m_ParticleCount = 0;
    }

    // マトリクスを更新する
    CreateResMatrix();

    // エミッタアニメーション計算用バッファを初期化
    m_EmitterAnimationVec[ detail::EmitterAnimationType_Scale ] = m_pEmitterRes->m_pResEmitter->emitter.scale;
    m_EmitterAnimationVec[ detail::EmitterAnimationType_Rotate ] = m_pEmitterRes->m_pResEmitter->emitter.rotate;
    m_EmitterAnimationVec[ detail::EmitterAnimationType_Translate ] = m_pEmitterRes->m_pResEmitter->emitter.trans;
    detail::Float3Copy( &m_EmitterAnimationVec[ detail::EmitterAnimationType_Color0 ], m_pEmitterRes->m_pResEmitter->emitter.color0 );
    detail::Float3Copy( &m_EmitterAnimationVec[ detail::EmitterAnimationType_Color1 ], m_pEmitterRes->m_pResEmitter->emitter.color1 );
    m_EmitterAnimationVec[ detail::EmitterAnimationType_ParticleScale ] = m_pEmitterRes->m_pResEmitter->ptclScale.base;
    m_EmitterAnimationVec[ detail::EmitterAnimationType_EmitterVolumeScale ] = m_pEmitterRes->m_pResEmitter->volume.volumeFormScale;

    m_EmitterAnimationVec[ detail::EmitterAnimationType_EmissionRate ].x = m_pEmitterRes->m_pResEmitter->emission.rate;
    m_EmitterAnimationVec[ detail::EmitterAnimationType_ParticleLife ].x = static_cast< float >( m_pEmitterRes->m_pResEmitter->ptcl.life );
    m_EmitterAnimationVec[ detail::EmitterAnimationType_Alpha0 ].x = m_pEmitterRes->m_pResEmitter->emitter.color0.w;
    m_EmitterAnimationVec[ detail::EmitterAnimationType_Alpha1 ].x = m_pEmitterRes->m_pResEmitter->emitter.color1.w;
    m_EmitterAnimationVec[ detail::EmitterAnimationType_AllDirectionalVelocity ].x = m_pEmitterRes->m_pResEmitter->ptclVel.allDirection;
    m_EmitterAnimationVec[ detail::EmitterAnimationType_DesignatedDirectionalVelocity ].x = m_pEmitterRes->m_pResEmitter->ptclVel.designatedDirScale;
    m_EmitterAnimationVec[ detail::EmitterAnimationType_GravityScale ].x = m_pEmitterRes->m_pResEmitter->emission.gravityScale;

    return true;
}// NOLINT(readability/fn_size)

//---------------------------------------------------------------------------
//  エミッタ終了処理を行います。
//---------------------------------------------------------------------------
void Emitter::Finalize( nn::gfx::Device* pDevice, bool invokeCallback ) NN_NOEXCEPT
{
    m_pEmitterCalculator = NULL;

    if ( invokeCallback  )
    {
        // パーティクル削除コールバック
        {
            InvokeParticleRemoveCallbackAll( this );
        }

        // エミッタ終了処理後コールバック
        {
            EmitterFinalizeArg arg;
            arg.pEmitter = this;
            InvokeEmitterFinalizeCallback( arg );
        }
    }

    ReleaseBuffer( pDevice );

    for ( int i = 0; i < CustomShaderConstantBufferIndex_MaxIndexCount; i++ )
    {
        m_GfxObjects.m_CustomConstantBuffer[i].Finalize();
    }

    m_GfxObjects.m_ParticleEmitterPluginAttribute.Finalize();

    return;
}

//---------------------------------------------------------------------------
//  放出毎に更新が必要な情報を更新します。
//---------------------------------------------------------------------------
void Emitter::UpdateByEmit( float* pOutEmitInterval ) NN_NOEXCEPT
{
    // 放出間隔の更新
    *pOutEmitInterval = static_cast< float >( m_pEmitterData->emission.interval + 1.0f ) +
        m_Random.GetInteger( m_pEmitterData->emission.intervalRandom );
    *pOutEmitInterval *= m_EmitIntervalRatio;
    *pOutEmitInterval *= m_pEmitterSet->GetEmissionIntervalScale();

    // エミッタマトリクス更新
    if( !m_pEmitterData->emitter.isUpdateMatrixByEmit )
    {
        return;
    }
    CreateResMatrix();
}

//---------------------------------------------------------------------------
//  マトリクス情報を更新します。
//---------------------------------------------------------------------------
void Emitter::CreateResMatrix() NN_NOEXCEPT
{
    nn::util::Vector3fType scale;
    nn::util::Vector3fType rotate;
    nn::util::Vector3fType translate;

    nn::util::VectorSetX( &rotate, m_pEmitterData->emitter.rotate.x + ( m_Random.GetFloat() * 2.0f - 1.0f ) * m_pEmitterData->emitter.rotateRand.x );
    nn::util::VectorSetY( &rotate, m_pEmitterData->emitter.rotate.y + ( m_Random.GetFloat() * 2.0f - 1.0f ) * m_pEmitterData->emitter.rotateRand.y );
    nn::util::VectorSetZ( &rotate, m_pEmitterData->emitter.rotate.z + ( m_Random.GetFloat() * 2.0f - 1.0f ) * m_pEmitterData->emitter.rotateRand.z );

    nn::util::VectorSetX( &translate, m_pEmitterData->emitter.trans.x + ( m_Random.GetFloat() * 2.0f - 1.0f ) * m_pEmitterData->emitter.transRand.x );
    nn::util::VectorSetY( &translate, m_pEmitterData->emitter.trans.y + ( m_Random.GetFloat() * 2.0f - 1.0f ) * m_pEmitterData->emitter.transRand.y );
    nn::util::VectorSetZ( &translate, m_pEmitterData->emitter.trans.z + ( m_Random.GetFloat() * 2.0f - 1.0f ) * m_pEmitterData->emitter.transRand.z );

    nn::util::VectorSetX( &scale, m_pEmitterData->emitter.scale.x );
    nn::util::VectorSetY( &scale, m_pEmitterData->emitter.scale.y );
    nn::util::VectorSetZ( &scale, m_pEmitterData->emitter.scale.z );

    detail::MatrixCreateSrtXyz( &m_ResMatrixSrt, scale, rotate, translate );
    nn::util::VectorSet( &scale, 1, 1, 1 );
    detail::MatrixCreateSrtXyz( &m_ResMatrixRt, scale, rotate, translate );
}

//---------------------------------------------------------------------------
//  バッファをスワップします。
//---------------------------------------------------------------------------
bool Emitter::SwapBuffer( BufferSwapMode bufferSwapMode ) NN_NOEXCEPT
{
    if ( ( bufferSwapMode == BufferSwapMode_Auto && ( m_IsDrawExecuted || m_IsComputeShaderExecuted ) ) ||
           bufferSwapMode == BufferSwapMode_Swap )
    {
        m_IsDrawExecuted = false;
        m_IsComputeShaderExecuted = false;

        const int maxBufferCount = GetEmitterSet()->GetSystem()->IsEnableTripleBuffer() ? 3 : 2;
        m_BufferSide = static_cast< nn::vfx::BufferSide >( ( m_BufferSide + 1 ) % maxBufferCount );

        m_ParticlePropertyFront = &m_ParticleProperty[ m_BufferSide ];

        return true;
    }

    return false;
}


//---------------------------------------------------------------------------
//  エミッタがフレームバッファテクスチャを要求するか。
//---------------------------------------------------------------------------
bool Emitter::IsRequestFrameBufferTexture() const NN_NOEXCEPT
{
    bool req = false;

    for( int i = 0; i < ShaderType_MaxShaderType; i++ )
    {
        if( m_pEmitterRes->m_pShader[ i ] &&
          ( m_pEmitterRes->m_pShader[ i ]->GetFrameBufferTexturePixelSamplerLocation() != InvalidValueId_TextureLocationId ||
            m_pEmitterRes->m_pShader[ i ]->GetFrameBufferTextureVertexSamplerLocation()  != InvalidValueId_TextureLocationId ) )
        {
            req = true;
        }
    }
    return req;
}

//---------------------------------------------------------------------------
//  エミッタがデプスバッファテクスチャを要求するか。
//---------------------------------------------------------------------------
bool Emitter::IsRequestDepthTexture() const NN_NOEXCEPT
{
    bool req = false;

    for( int i = 0; i < ShaderType_MaxShaderType; i++ )
    {
        if( m_pEmitterRes->m_pShader[ i ] &&
          ( m_pEmitterRes->m_pShader[ i ]->GetDepthBufferTexturePixelSamplerLocation()  != InvalidValueId_TextureLocationId ||
            m_pEmitterRes->m_pShader[ i ]->GetDepthBufferTextureVertexSamplerLocation() != InvalidValueId_TextureLocationId ) )
        {
            req = true;
        }
    }

    return req;
}

//---------------------------------------------------------------------------
//  描画パスを設定します。
//---------------------------------------------------------------------------
void Emitter::SetDrawPath( uint32_t drawPathFlag ) NN_NOEXCEPT
{
    m_DrawPathFlag = drawPathFlag;
}

//---------------------------------------------------------------------------
//  マニュアルエミッタが消えても大丈夫な状態かを取得します。
//---------------------------------------------------------------------------
bool Emitter::IsManualEmitterReadyToExit() const NN_NOEXCEPT
{
    if ( !m_pEmitterSet->IsManualParticleEmission() )
    {
        // マニュアルエミッタセットでない場合は false
        return false;
    }
    if ( m_pEmitterSet->IsResidentManualEmitterSet() )
    {
        // 常駐型マニュアルエミッタセットなら false
        return false;
    }
    if ( !m_pEmitterSet->IsResidentManualEmitterSet() &&
         ( m_Time < m_EmitLastFrame + m_pEmitterData->ptcl.life + m_pEmitterSet->GetResidentEmitterTime() ) )
    {
        // 非常駐型のマニュアルエミッタセットで、まだ待機時間が経過していない場合も false
        return false;
    }

    if ( ( GetCalculationType() == detail::EmitterCalculationMode_Cpu ) &&
         ( m_ParticleProcessingCount > 0 ) )
    {
        // まだ粒が生きている場合も false
        return false;
    }

    // 以上の条件をすべて通過している場合、もう消しても大丈夫なので true
    return true;
}

//---------------------------------------------------------------------------
//  カラー0 RGBA値(乗算値)を設定します。
//---------------------------------------------------------------------------
void Emitter::SetColor0( const nn::util::Vector4fType& color0 ) NN_NOEXCEPT
{
    m_Color0 = color0;
}

//---------------------------------------------------------------------------
//  カラー1 RGBA値(乗算値)を設定します。
//---------------------------------------------------------------------------
void Emitter::SetColor1( const nn::util::Vector4fType& color1 ) NN_NOEXCEPT
{
    m_Color1 = color1;
}

//---------------------------------------------------------------------------
//  今の時点でエミッタが持つ正確なパーティクルの数を計測します。
//---------------------------------------------------------------------------
int Emitter::CalculatePreciseParticleCount() const NN_NOEXCEPT
{
    if( m_pEmitterData->emitter.calcType == nn::vfx::detail::EmitterCalculationMode_Cpu )
    {
        // CPU エミッタの場合は、Calculate() で計測が行われているのでその結果を返す。
        return m_ParticleProcessingCount;
    }
    else
    {
        // すべての粒について生存チェックを行い、生きているものの数を数え上げる
        // GPU / GPU+SO の場合は、時間と寿命で死亡判断を行う
        int particleCount = 0;
        for( int i = 0; i < m_ParticleCount; i++ )
        {
            const float time = m_pParticleData[ i ].GetTime( m_Time );
            const float life = m_pParticleData[ i ].GetLife();
            const bool  isAlive = ( time - life ) <= 0.0f;
            if( isAlive )
            {
                particleCount++;
            }
        }
        return particleCount;
    }
}

} // namespace vfx
} // namespace nn
