﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/ui2d/detail/ui2d_Log.h>
#if defined(NN_BUILD_CONFIG_OS_WIN)

#include <exception>

#include <string>
#include <cstring>

#include <nn/util/util_StringUtil.h>

#include <nn/ui2d.h>
#include <nn/font.h>

#include <nn/ui2d/viewer/win/ui2d_DirResourceAccessorWin.h>

#include <nn/nn_windows.h>

#if defined(NN_UI2D_VIEWER_ENABLED)

namespace nn
{
namespace ui2d
{
namespace viewer
{

namespace
{

    bool IsResTextureFileInitialized(nn::gfx::ResTextureFile* pResTextureFile)
    {
        if(pResTextureFile == NULL)
        {
            return false;
        }

        const nn::gfx::MemoryPool* pMemPool = reinterpret_cast<nn::gfx::MemoryPool*>(pResTextureFile->ToData().textureContainerData.pTextureMemoryPool.Get());
        return pMemPool->ToData()->state == nn::gfx::MemoryPool::DataType::State_Initialized;
    }

    bool IsIndividualArchiveShaderVariationTable(int firstBlend)
    {
        if (firstBlend == IndividualArchiveShaderVariationTableFirstBlendWithDetailedCombiner ||
            firstBlend == IndividualArchiveShaderVariationTableFirstBlendWithCombinerUserShader)
        {
            return true;
        }
        return false;
    }

}

class DirResourceAccessorImpl : public DirResourceAccessor
{
    static const size_t  ResourceAlignment = 4;
    static const size_t  ResourceAlignmentForFont = 1024 * 4;
    static const size_t  ResourceAlignmentForTexture = 1024 * 4;
    static const size_t  ResourceAlignmentForShader = 1024 * 4;

public:
    explicit DirResourceAccessorImpl(const char* pBasePath);

    virtual ~DirResourceAccessorImpl();

    virtual void Finalize(nn::gfx::Device* pDevice);

    // 指定したパス以下のリソースをアクセサに関連付けます。
    virtual void Attach(const char* resPath);

    // アクセサに関連づけられたリソースを破棄します。
    virtual void Detach();

    // リソースの関連づけが行われているか調べます。
    virtual bool IsAttached() const
    {
        return !m_BasePath.empty();
    }

    // 関連付けられたリソースのパスを取得します。
    virtual const char* GetResourceBasePath() const
    {
        return m_BasePath.c_str();
    }

    virtual const char* FindFile(
        nn::ui2d::ResType resType,
        const char* fileName);

    virtual void* FindResourceByName(
        size_t * pSize,
        nn::ui2d::ResType resType,
        const char* name);

    virtual void FindResourceByType(
        ResType resType,
        FindResourceByTypeCallback pCallback,
        void* pParam
    ) const NN_OVERRIDE;

    virtual const nn::ui2d::TextureInfo* AcquireTexture(nn::gfx::Device* pDevice, const char* name);

    virtual nn::font::Font* AcquireFont(nn::gfx::Device* pDevice, const char* name);

    virtual const nn::ui2d::ShaderInfo* AcquireShader(nn::gfx::Device* pDevice, const char* name);

    //! @brief フォントを登録します。
    //!
    //! @param[in] name フォント名です。
    //! @param[in] pFont フォントです。
    //!
    virtual void RegisterFont(const char* name, nn::font::Font* pFont);

    /**
    * @brief テクスチャービューをディスクリプタプールに登録します。
    *
    * @param[in] regsiterTextureViewSlot     ディスクリプタスロット確保関数です。
    *
    */
    virtual void RegisterTextureViewToDescriptorPool(RegisterTextureViewSlot regsiterTextureViewSlot, void* pUserData)
    {
        m_FontList.RegisterTextureViewToDescriptorPool(regsiterTextureViewSlot, pUserData);
        m_TextureList.RegisterTextureViewToDescriptorPool(regsiterTextureViewSlot, pUserData);
    }

    /**
    * @brief テクスチャービューをディスクリプタプールから解放します。
    *
    * @param[in] unregsiterTextureViewSlot     ディスクリプタスロット解放関数です。
    *
    */
    virtual void UnregisterTextureViewFromDescriptorPool(UnregisterTextureViewSlot unregsiterTextureViewSlot, void* pUserData)
    {
        m_FontList.UnregisterTextureViewFromDescriptorPool(unregsiterTextureViewSlot, pUserData);
        m_TextureList.UnregisterTextureViewFromDescriptorPool(unregsiterTextureViewSlot, pUserData);
    }

    //! @brief  キャプチャテクスチャをリソースへ登録します。
    //!
    //! @param[in]  pName   登録するキャプチャテクスチャの名前です。
    //!
    //! @return 登録した RenderTargetTextureInfo のポインタです。
    //!
    virtual RenderTargetTextureInfo* RegisterRenderTargetTexture(const char* pName)
    {
        // レンダーターゲットテクスチャの名前が重複すると最初に登録したものしか見つからなくなり
        // 意図したテクスチャが参照されないケースが発生するためアサートで止める。
        NN_SDK_ASSERT(m_TextureList.FindTextureByName(pName) == NULL);

        // ユーザーが用意したリソースとしてキャプチャテクスチャを登録する。
        return m_TextureList.RegisterRenderTargetTexture(pName, false);
    }

    //! @brief キャプチャテクスチャの登録を解除します。
    //!
    //! @param[in] pTexInfo テクスチャ情報です。
    //!
    //! @sa RegisterRenderTargetTexture
    //!
    virtual void UnregisterRenderTargetTexture(TextureInfo* pTexInfo)
    {
        m_TextureList.UnregisterTexture(pTexInfo);
    }

protected:

    virtual nn::font::Font* ReferFont(const char* name);

    virtual void AddFont(const char* name, nn::font::Font* pFont, bool own);

    virtual bool LoadTexture(ResourceTextureInfo* textureInfo, nn::gfx::Device* pDevice, const char *name);

    virtual bool LoadShader(ShaderInfo* pShaderInfo, nn::gfx::Device* pDevice, const char *name);

private:

    class DirResourceLink
    {
    public:
        DirResourceLink()
        : m_ResType(0)
        , m_pResource(NULL)
        , m_Size(0)
        {}

        void Set(
            nn::ui2d::ResType resType,
            void* pResource,
            size_t  size,
            const char* pName)
        {
            m_ResType = resType;
            m_pResource = pResource;
            m_Size = static_cast<uint32_t>(size);
            m_Name = pName;
        }

        nn::util::IntrusiveListNode m_Link;

        nn::ui2d::ResType m_ResType;
        void* m_pResource;
        uint32_t  m_Size;
        std::string m_Name;
    };

    typedef nn::util::IntrusiveList<DirResourceLink, nn::util::IntrusiveListMemberNodeTraits<DirResourceLink, &DirResourceLink::m_Link> > DirResourceList;

    void*
    SearchFile(
        size_t * pSize,
        nn::ui2d::ResType resType,
        const char* pBasePath,
        const char* pName
        );

    DirResourceLink* SearchResource(
        nn::ui2d::ResType resType,
        const char* pName);

    bool RegisterResource(
        nn::ui2d::ResType resType,
        void* pResource,
        size_t  size,
        const char* pName);

    const std::string GetFilePath(
        const char* pResPath,
        nn::ui2d::ResType resType,
        const char* pFileName,
        const char* pDirSeparator);

    const char*
    GetArchiveShaderRegistrationName(const char* name);

    DirResourceList m_ResourceList;
    std::string m_BasePath;
    nn::ui2d::FontContainer m_FontList;
    nn::ui2d::TextureContainer m_TextureList;
    nn::ui2d::ShaderContainer m_ShaderList;

    char m_FoundPath[MAX_PATH];
    bool m_FindContinue;
    nn::gfx::ResTextureFile* m_pResTextureFile;
};

DirResourceAccessor*
DirResourceAccessor::Create(const char *pBasePath)
{
    return new DirResourceAccessorImpl(pBasePath);
}

/*---------------------------------------------------------------------------*
  @brief コンストラクタ
 *---------------------------------------------------------------------------*/
DirResourceAccessorImpl::DirResourceAccessorImpl(
    const char* pBasePath
)
: m_BasePath(pBasePath)
, m_FindContinue(false)
, m_pResTextureFile(NULL)
{
}

//-------------------------------------------------------------------------

DirResourceAccessorImpl::~DirResourceAccessorImpl()
{
}

//-------------------------------------------------------------------------

void DirResourceAccessorImpl::Finalize(nn::gfx::Device* pDevice)
{
    m_FontList.Finalize(pDevice);
    m_TextureList.Finalize(pDevice);
    m_ShaderList.Finalize(pDevice);

    ResourceAccessor::Finalize(pDevice);

    if(m_pResTextureFile != NULL)
    {
        m_pResTextureFile->Finalize(pDevice);
        m_pResTextureFile = NULL;
    }

    while (!m_ResourceList.empty())
    {
        DirResourceList::iterator top = m_ResourceList.begin();
        DirResourceLink* pResLink = &(*top);
        nn::ui2d::Layout::FreeMemory(pResLink->m_pResource);
        m_ResourceList.erase(top);
        delete pResLink;
    }

}

//-------------------------------------------------------------------------

DirResourceAccessorImpl::DirResourceLink*
DirResourceAccessorImpl::SearchResource(
    nn::ui2d::ResType resType,
    const char* name
)
{
    for (DirResourceList::iterator it = m_ResourceList.begin(); it != m_ResourceList.end(); ++it)
    {
        if (it->m_ResType == resType && it->m_Name == name)
        {
            return &(*it);
        }
    }

    return NULL;
}

//-------------------------------------------------------------------------
// リソースパスの設定
void DirResourceAccessorImpl::Attach(const char* resPath)
{
    NN_SDK_ASSERT_NOT_NULL(resPath);
    NN_SDK_ASSERT(!IsAttached());

    if (!IsAttached())
    {
        m_BasePath = resPath;
    }
}

//-------------------------------------------------------------------------
// 読み込んでいる全てのリソースを開放
void DirResourceAccessorImpl::Detach()
{
    m_BasePath = "";
}

//-------------------------------------------------------------------------

const char* DirResourceAccessorImpl::FindFile(
    nn::ui2d::ResType resType,
    const char* pPrevFilePath)
{
    // フォルダがロックされてしまう問題があるため、効率は良くないが毎回一から探す
    std::string findPattern = this->GetFilePath(m_BasePath.c_str(), resType, "*", "\\");
    WIN32_FIND_DATAA findData;
    HANDLE handle = ::FindFirstFileA(findPattern.c_str(), &findData);

    if (handle == INVALID_HANDLE_VALUE)
    {
        return NULL;
    }
    else
    {
        if (pPrevFilePath == NULL)
        {
            // 最初のファイル
            // 隠しファイルは無視する。
            while (findData.cFileName[0] == '.')
            {
                if (! ::FindNextFileA(handle, &findData))
                {
                    ::FindClose(handle);
                    return NULL;
                }
            }
        }
        else
        {
            // 二つ目以降のファイル
            while (::strcmp(findData.cFileName, pPrevFilePath) != 0)
            {
                if (! ::FindNextFileA(handle, &findData))
                {
                    ::FindClose(handle);
                    return NULL;
                }
            }

            // ひとつ前に探したファイルが見つかった。その次を探す
            if (! ::FindNextFileA(handle, &findData))
            {
                ::FindClose(handle);
                return NULL;
            }
        }

        ::FindClose(handle);
        nn::util::Strlcpy(m_FoundPath, findData.cFileName, sizeof(m_FoundPath));
        return m_FoundPath;
    }
}

//-------------------------------------------------------------------------

bool
DirResourceAccessorImpl::RegisterResource(
    nn::ui2d::ResType resType,
    void*       pResource,
    size_t          size,
    const char* pName)
{
    NN_SDK_ASSERT_NOT_NULL(pResource);
    NN_SDK_ASSERT_NOT_NULL(pName);

    DirResourceLink* pDirResourceLink = new DirResourceLink();
    NN_SDK_ASSERT_NOT_NULL(pDirResourceLink);

    pDirResourceLink->Set(resType, pResource, size, pName);
    m_ResourceList.push_back(*pDirResourceLink);

    return true;
}

//-------------------------------------------------------------------------

void*
DirResourceAccessorImpl::FindResourceByName(
    size_t * pSize,
    nn::ui2d::ResType resType,
    const char* name
)
{
    NN_SDK_ASSERT_NOT_NULL(name);

    size_t len = ::strlen(name);
    if (len == 0)
    {
        return NULL;
    }

    DirResourceLink* pDirResourceLink = this->SearchResource(resType, name);
    if (pDirResourceLink)
    {
        if (pSize)
        {
            *pSize = pDirResourceLink->m_Size;
        }
        return pDirResourceLink->m_pResource;
    }
    else
    {
        size_t  size;
        void* buffer = NULL;

        if ((buffer = SearchFile(&size, resType, m_BasePath.c_str(), name)) != 0)
        {
            RegisterResource(resType, buffer, size, name);
            if (pSize)
            {
                *pSize = size;
            }
            return buffer;
        }

        return NULL;
    }
}

//-------------------------------------------------------------------------

void
DirResourceAccessorImpl::FindResourceByType(
    ResType resType,
    FindResourceByTypeCallback pCallback,
    void* pParam
) const
{
    NN_UNUSED(resType);
    NN_UNUSED(pCallback);
    NN_UNUSED(pParam);
    NN_SDK_ASSERT(false, "DirResourceAccessorImpl::FindResourceByType is not implemented.");
}

//-------------------------------------------------------------------------

nn::font::Font*
DirResourceAccessorImpl::AcquireFont(nn::gfx::Device* pDevice, const char* name)
{
    nn::font::Font* pFont = this->ReferFont(name);

    if (pFont == NULL)
    {
        pFont = this->LoadFont(pDevice, name);

        if (pFont != NULL)
        {
            this->AddFont(name, pFont, true);
        }
    }

    return pFont;
}

//-------------------------------------------------------------------------

nn::font::Font*
DirResourceAccessorImpl::ReferFont(const char* name)
{
    // フォントを所有しているか。
    {
        nn::font::Font* pFont = m_FontList.FindFontByName(name);
        if (pFont != NULL)
        {
            return pFont;
        }
    }

    // .bffnt のフォント生成はライブラリに任せる。
    return NULL;
}

//-------------------------------------------------------------------------

void
DirResourceAccessorImpl::AddFont(const char* name, nn::font::Font* pFont, bool own)
{
    (void)m_FontList.RegisterFont(name, pFont, own);
}

void
DirResourceAccessorImpl::RegisterFont(const char* name, nn::font::Font* pFont)
{
    (void)m_FontList.RegisterFont(name, pFont, false);
}

//-------------------------------------------------------------------------

bool
DirResourceAccessorImpl::LoadTexture(ResourceTextureInfo* textureInfo, nn::gfx::Device* pDevice, const char *name)
{
    void* pTexRes = NULL;
    size_t size = 0;

    // まず、bntx を探してその中からテクスチャが無いか探します。
    if(m_pResTextureFile == NULL)
    {
        pTexRes = FindResourceByName(&size, ResourceTypeTexture, "__Combined.bntx");
        if (pTexRes != NULL && size != 0)
        {
            // 発見した bntx から テクスチャを取得します。
            m_pResTextureFile = nn::gfx::ResTextureFile::ResCast(pTexRes);
            if(!IsResTextureFileInitialized(m_pResTextureFile))
            {
                m_pResTextureFile->Initialize(pDevice);
            }
        }
    }

    pTexRes = NULL;

    // bntx から テクスチャを取得します。
    if (m_pResTextureFile != NULL)
    {
        int resTextureIndex = m_pResTextureFile->GetTextureDic()->FindIndex(name);
        if(resTextureIndex != -1)
        {
            pTexRes = m_pResTextureFile->GetResTexture(resTextureIndex);
        }
    }

    // bntx から発見できない場合、指定された名前そのままで検索する
    if(pTexRes == NULL)
    {
        pTexRes = FindResourceByName(&size, ResourceTypeTexture, name);
        return false;
    }

    if (!ui2d::LoadTexture(textureInfo, pDevice, pTexRes))
    {
        NN_DETAIL_UI2D_ERROR("texture object unavailable. - %s", name);
        return false;
    }

    return true;
}

//-------------------------------------------------------------------------
bool
DirResourceAccessorImpl::LoadShader(ShaderInfo* pShaderInfo, nn::gfx::Device* pDevice, const char *name)
{
    int firstBlend;
    int secondBlend;
    bool isCombined = ConvertArchiveShaderNameToBlends(&firstBlend, &secondBlend, name);

    if (isCombined)
    {
        if (!IsIndividualArchiveShaderVariationTable(firstBlend))
        {
            // 統合済みアーカイブシェーダから探します。
            size_t size = 0;
            void* pShaderRes = FindResourceByName(&size, ResourceTypeShader, ArchiveShaderFileName);
            if (pShaderRes != NULL && size != 0)
            {
                void* pVariationTable = FindResourceByName(&size, ResourceTypeShader, ArchiveShaderVariationTableFileName);
                if (pVariationTable != NULL && size != 0)
                {
                    int variationIndex = SearchShaderVariationIndexFromTable(pVariationTable, firstBlend, secondBlend);
                    if (variationIndex != -1)
                    {
                        ui2d::LoadArchiveShader(pShaderInfo, pDevice, pShaderRes, pVariationTable);
                        return true;
                    }
                }
            }
        }
        else
        {
            // 個別のテーブルを利用する
            int stageBits[DetailedCombinerStageBitsCountWithVariationTable];
            ConvertArchiveShaderDetailedCombinerNameToStageBits(stageBits, name);

            size_t size = 0;
            void* pShaderRes = FindResourceByName(&size, ResourceTypeShader, ArchiveShaderFileName);
            NN_SDK_ASSERT(pShaderRes != NULL, "%s ArchiveShaderFile does not exist", ArchiveShaderFileName);
            if (pShaderRes != NULL && size != 0)
            {
                //! バリエーション変数版の詳細コンバイナ用バリエーションテーブルのファイル名
                char individualArchiveShaderVariationTableName[MAX_PATH];
                std::sprintf(individualArchiveShaderVariationTableName, "%s_%d.%s",
                    IndividualArchiveShaderVariationTableBaseName,
                    firstBlend,
                    IndividualArchiveShaderVariationTableExtensionName);
                NN_SDK_ASSERT(std::strlen(individualArchiveShaderVariationTableName) <= MAX_PATH, "\"%s\" file name is too long", individualArchiveShaderVariationTableName);
                void* pDetailedCombinerVariationTable = FindResourceByName(&size, ResourceTypeShader, individualArchiveShaderVariationTableName);
                NN_SDK_ASSERT(pDetailedCombinerVariationTable != NULL, "%s IndividualArchiveShaderVariationTableFile does not exist", individualArchiveShaderVariationTableName);

                if (pShaderRes != NULL && size != 0)
                {
                    const int variationIndex = SearchShaderVariationDetailedCombinerIndexFromTable(pDetailedCombinerVariationTable, stageBits);
                    NN_SDK_ASSERT(variationIndex != -1, "VariationIndex[%d] does not exist in %s IndividualArchiveShaderVariationTableFile", variationIndex, individualArchiveShaderVariationTableName);
                    if (variationIndex != -1)
                    {
                        const int texMapCount = firstBlend == IndividualArchiveShaderVariationTableFirstBlendWithCombinerUserShader ? CombinerUserShaderTexMapMax : TexMapMax;
                        // 拡張用のテーブル利用時は pDetailedCombinerVariationTable が GetVariationTable() から得られる事に注意してください。
                        ui2d::LoadArchiveShader(pShaderInfo, pDevice, pShaderRes, pDetailedCombinerVariationTable, NULL, 0, 0, texMapCount);
                        return true;
                    }
                }
            }
        }
    }
    else
    {
        // 指定された名前そのままで検索します。
        char str[256];
        ResourceAccessor::GetArchiveShaderResourceName(str, name, sizeof(str));
        size_t size = 0;
        void* pShaderRes = FindResourceByName(&size, ResourceTypeShader, str);
        if (pShaderRes != NULL && size != 0)
        {
            ui2d::LoadArchiveShader(pShaderInfo, pDevice, pShaderRes, NULL);
            return true;
        }
        NN_DETAIL_UI2D_ERROR("shader resource not found. - %s\n", str);
    }

    return false;
}

//-------------------------------------------------------------------------

const nn::ui2d::TextureInfo*
DirResourceAccessorImpl::AcquireTexture(nn::gfx::Device* pDevice, const char *name)
{
    const nn::ui2d::TextureInfo* texInfo = m_TextureList.FindTextureByName(name);
    if (texInfo)
    {
        return texInfo;
    }
    else
    {
        nn::ui2d::ResourceTextureInfo* newTextureInfo = m_TextureList.RegisterResourceTexture(name);

        if (newTextureInfo)
        {
            this->LoadTexture(newTextureInfo, pDevice, name);

            if (!newTextureInfo->IsValid())
            {
                NN_DETAIL_UI2D_ERROR("Can't load texture : %s\n", name);
            }
        }
        else
        {
            NN_DETAIL_UI2D_ERROR("Can't register texture : %s\n", name);
        }

        return newTextureInfo;
    }
}

//-------------------------------------------------------------------------

void*
DirResourceAccessorImpl::SearchFile(
    size_t * pSize,
    nn::ui2d::ResType resType,
    const char* pBasePath,
    const char* pName
)
{
    NN_SDK_ASSERT_NOT_NULL(pName);

    HANDLE hFindDir;
    WIN32_FIND_DATAA findDataDir;

    std::string searchDir = std::string(pBasePath) + "/*";

    if ((hFindDir = ::FindFirstFileA(searchDir.c_str(), &findDataDir)) != INVALID_HANDLE_VALUE)
    {
        do
        {
            if (::strcmp(".", findDataDir.cFileName) == 0 ||
                ::strcmp("..", findDataDir.cFileName) == 0)
            {
                continue;
            }

            std::string dirName = std::string(pBasePath) + "/" + findDataDir.cFileName + "/";
            std::string searchFile = dirName + pName;

            HANDLE hFindFile;
            WIN32_FIND_DATAA findDataFile;

            if ((hFindFile = ::FindFirstFileA(searchFile.c_str(), &findDataFile)) == INVALID_HANDLE_VALUE)
            {
                continue;
            }

            std::string fileName = dirName + findDataFile.cFileName;
            HANDLE handle = CreateFileA(fileName.c_str(), GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
            if(handle == INVALID_HANDLE_VALUE)
            {
                NN_SDK_ASSERT(false, "can not open file (%s).", fileName.c_str());
            }

            size_t  alignment = ResourceAlignment;
            switch (resType)
            {
            case nn::ui2d::ResourceTypeFont:
            case nn::ui2d::ResourceTypeArchiveFont:
                alignment = ResourceAlignmentForFont;
                break;

            case nn::ui2d::ResourceTypeTexture:
                alignment = ResourceAlignmentForTexture;
                break;
            case nn::ui2d::ResourceTypeShader:
                alignment = ResourceAlignmentForShader;
                break;

            default:
                break;
            }

            /// 下位32bitのみ使う
            DWORD file_size_high = 0;
            DWORD fileSize = GetFileSize(handle, &file_size_high);
            void* buffer = nn::ui2d::Layout::AllocateMemory(fileSize, alignment);
            NN_SDK_ASSERT_NOT_NULL(buffer);

            DWORD read_size = 0;
            BOOL read_success = ReadFile(handle, buffer, fileSize, &read_size, NULL);
            NN_SDK_ASSERT(read_success);
            NN_UNUSED(read_success);
            CloseHandle(handle);

            ::FindClose(hFindFile);
            ::FindClose(hFindDir);

            if (pSize)
            {
                *pSize = fileSize;
            }

            return buffer;

        } while (::FindNextFileA(hFindDir, &findDataDir));

        ::FindClose(hFindDir);
    }

    return NULL;
}

//-------------------------------------------------------------------------

const std::string
DirResourceAccessorImpl::GetFilePath(
    const char* pResPath,
    nn::ui2d::ResType resType,
    const char* pFileName,
    const char* pDirSeparator)
{
    std::string path;

    if (pResPath)
    {
        path += pResPath;
        path += pDirSeparator;
    }

    path += char(resType >> 24);
    path += char(resType >> 16);
    path += char(resType >> 8);
    path += char(resType >> 0);

    if (pFileName)
    {
        path += pDirSeparator;
        path += pFileName;
    }

    return path;
}

//-------------------------------------------------------------------------
const char*
DirResourceAccessorImpl::GetArchiveShaderRegistrationName(const char* name)
{
    // 統合済みアーカイブシェーダの場合は特別な名前で登録する
    int firstBlend;
    int secondBlend;
    bool isCombined = ConvertArchiveShaderNameToBlends(&firstBlend, &secondBlend, name);
    if (isCombined)
    {
        if (firstBlend == IndividualArchiveShaderVariationTableFirstBlendWithDetailedCombiner)
        {
            return ArchiveShaderRegistrationNameWithDetailedCombiner;
        }
        else if (firstBlend == IndividualArchiveShaderVariationTableFirstBlendWithCombinerUserShader)
        {
            return ArchiveShaderRegistrationNameWithCombinerUserShader;
        }
        else
        {
            return ArchiveShaderRegistrationName;
        }
    }
    else
    {
        return name;
    }
}

//-------------------------------------------------------------------------

const nn::ui2d::ShaderInfo*
DirResourceAccessorImpl::AcquireShader(nn::gfx::Device* pDevice, const char* name)
{
    const char* registrationName = GetArchiveShaderRegistrationName(name);

    // アーカイブ内のリソースがロードされていれば使う
    const ShaderInfo* pShaderInfo = m_ShaderList.FindShaderByName(registrationName);
    if (pShaderInfo != NULL)
    {
        return pShaderInfo;
    }

    // アーカイブ内のリソースをロードする
    {
        ShaderInfo* pNewShaderInfo = m_ShaderList.RegisterShader(registrationName, true); // アーカイブ内のリソース用リスト

        if (pNewShaderInfo)
        {
            this->LoadShader(pNewShaderInfo, pDevice, name);
        }
        else
        {
            NN_DETAIL_UI2D_ERROR("Can't register shader: %s\n", name);
        }

        return pNewShaderInfo;
    }
}

} // namespace viewer
} // namespace ui2d
} // namespace nn

#endif // NN_UI2D_VIEWER_ENABLED

#endif // NN_BUILD_CONFIG_OS_WIN
