﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkLog.h>

#include <nn/ui2d/ui2d_ShaderInfo.h>
#include <nn/ui2d/ui2d_Layout.h>
#include <nn/ui2d/ui2d_Util.h>

namespace nn
{
namespace ui2d
{

//----------------------------------------
void ShaderInfo::Initialize(nn::gfx::Device* pDevice, void* pShader)
{
    InitializeWithVariationTable(pDevice, pShader, NULL);
}

//----------------------------------------
void ShaderInfo::InitializeWithVariationTable(nn::gfx::Device* pDevice, void* pShader, const void* pVariationTable, nn::gfx::MemoryPool* pMemoryPool, ptrdiff_t memoryPoolOffset, size_t memoryPoolSize, int textureSlotCount)
{
    NN_SDK_ASSERT_NOT_NULL(pShader);

    SetTextureSlotCount(textureSlotCount);

    // シェーダの読み込み
    {
        m_pResShaderFile = nn::gfx::ResShaderFile::ResCast(pShader);

        nn::gfx::ResShaderContainer* pContainer = m_pResShaderFile->GetShaderContainer();
        NN_SDK_ASSERT_NOT_NULL(pContainer);
        if(!IsResShaderContainerInitialized(pContainer))
        {
            if (pMemoryPool == NULL)
            {
                pContainer->Initialize(pDevice);
            }
            else
            {
                nn::util::BytePtr resShaderFileTop(pShader);
                pContainer->Initialize(pDevice, pMemoryPool, resShaderFileTop.Distance(pContainer) + memoryPoolOffset, memoryPoolSize);
            }
        }

        nn::gfx::ResShaderVariation* pFirstVariation = pContainer->GetResShaderVariation(0);
        NN_SDK_ASSERT_NOT_NULL(pFirstVariation);

        const nn::gfx::ShaderCodeType codeType = TryInitializeAndGetShaderCodeType(pDevice, pFirstVariation);
        SetShaderCodeType(codeType);

        // 残りのバリエーションを初期化
        for (int i = 1, variationCount = GetVariationCount(); i < variationCount; i++)
        {
            nn::gfx::ResShaderVariation* pVariation = pContainer->GetResShaderVariation(i);
            NN_SDK_ASSERT_NOT_NULL(pVariation);

            const bool isShaderInitialize = IsResShaderProgramInitialized(pVariation->GetResShaderProgram(codeType));
            nn::gfx::ShaderInitializeResult shaderResult = isShaderInitialize ? nn::gfx::ShaderInitializeResult_Success : pVariation->GetResShaderProgram(codeType)->Initialize(pDevice);
            NN_SDK_ASSERT(shaderResult == nn::gfx::ShaderInitializeResult_Success);
            NN_UNUSED(pVariation);
            NN_UNUSED(shaderResult);
        }
    }

    {
        int variationCount = GetVariationCount();
        m_pVertexStates = Layout::NewArray<nn::gfx::VertexState>(variationCount);
        m_pVertexShaderSlots = static_cast<int*>(Layout::AllocateMemory(sizeof(int) * variationCount));
        m_pGeometryShaderSlots = static_cast<int*>(Layout::AllocateMemory(sizeof(int) * variationCount));
        m_pPixelShaderSlots = static_cast<int*>(Layout::AllocateMemory(sizeof(int) * variationCount));
        m_pTextureSlots = static_cast<int(*)>(Layout::AllocateMemory(sizeof(int) * variationCount * textureSlotCount));
    }

    for (int i = 0, variationCount = GetVariationCount(); i < variationCount; i++)
    {
        const nn::gfx::Shader* pVertexShader = GetVertexShader(i);
        const nn::gfx::Shader* pGeometryShader = GetGeometryShader(i);
        const nn::gfx::Shader* pPixelShader = GetPixelShader(i);
        m_pVertexShaderSlots[i] = pVertexShader->GetInterfaceSlot(nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "uConstantBufferForVertexShader");
        m_pGeometryShaderSlots[i] = pGeometryShader->GetInterfaceSlot(nn::gfx::ShaderStage_Geometry, nn::gfx::ShaderInterfaceType_ConstantBuffer, "uConstantBufferForGeometryShader");
        m_pPixelShaderSlots[i] = pPixelShader->GetInterfaceSlot(nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_ConstantBuffer, "uConstantBufferForPixelShader");

        const int samplerNameBufferSize = 32;
        char samplerNameBuffer[samplerNameBufferSize];
        for (int j = 0; j < textureSlotCount; j++)
        {
            nn::util::SNPrintf(samplerNameBuffer, samplerNameBufferSize, "uTexture%d", j);
            *(m_pTextureSlots + i * textureSlotCount + j) = pPixelShader->GetInterfaceSlot(nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, samplerNameBuffer);
        }
    }

    // 頂点ステートを初期化
    InitializeVertexStates(pDevice);

    // バリエーションテーブルの設定
    m_pVariationTable = pVariationTable;
}

//----------------------------------------
void ShaderInfo::InitializeVertexStates(nn::gfx::Device* pDevice)
{
    for (int i = 0, variationCount = GetVariationCount(); i < variationCount; i++)
    {
        const nn::gfx::Shader* pVertexShader = GetVertexShader(i);
        nn::gfx::VertexState::InfoType info;
        info.SetDefault();

        nn::gfx::VertexAttributeStateInfo attribs[3];
        int attribCount = 0;
        int stride = 0;


        if (pVertexShader->GetInterfaceSlot(nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "aVertexIndex") >= 0)
        {
            attribs[attribCount].SetDefault();
            attribs[attribCount].SetNamePtr("aVertexIndex");
            attribs[attribCount].SetBufferIndex(0);
            attribs[attribCount].SetFormat(nn::gfx::AttributeFormat_32_32_Float);
            attribs[attribCount].SetOffset(0);
            ++attribCount;
            stride += sizeof(float) * 2;
        }

        if (pVertexShader->GetInterfaceSlot(nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "aPosition") >= 0)
        {
            attribs[attribCount].SetDefault();
            attribs[attribCount].SetNamePtr("aPosition");
            attribs[attribCount].SetBufferIndex(0);
            attribs[attribCount].SetFormat(nn::gfx::AttributeFormat_32_32_32_Float);
            attribs[attribCount].SetOffset(stride);
            ++attribCount;
            stride += sizeof(float) * 3;
        }

        if (pVertexShader->GetInterfaceSlot(nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "aTexCoord") >= 0)
        {
            attribs[attribCount].SetDefault();
            attribs[attribCount].SetNamePtr("aTexCoord");
            attribs[attribCount].SetBufferIndex(0);
            attribs[attribCount].SetFormat(nn::gfx::AttributeFormat_32_32_Float);
            attribs[attribCount].SetOffset(stride);
            ++attribCount;
            stride += sizeof(float) * 2;
        }

        nn::gfx::VertexBufferStateInfo buffer[1];
        {
            buffer[0].SetDefault();
            buffer[0].SetStride(stride);
            buffer[0].SetDivisor(0);
        }
        info.SetVertexAttributeStateInfoArray(attribs, attribCount);
        info.SetVertexBufferStateInfoArray(buffer, 1);
        size_t size = nn::gfx::VertexState::GetRequiredMemorySize(info);
        m_pVertexStates[i].SetMemory(Layout::AllocateMemory(size, nn::gfx::VertexState::RequiredMemoryInfo_Alignment), size);
        m_pVertexStates[i].Initialize(pDevice, info, pVertexShader);
    }
}

//----------------------------------------
void ShaderInfo::Finalize(nn::gfx::Device* pDevice, bool skipFinalizingResShader)
{
    for (int i = 0, variationCount = GetVariationCount(); i < variationCount; i++)
    {
        void* pBuffer = m_pVertexStates[i].GetMemory();
        m_pVertexStates[i].Finalize(pDevice);
        Layout::FreeMemory(pBuffer);

        if (!skipFinalizingResShader)
        {
            nn::gfx::ResShaderVariation* pVariation = m_pResShaderFile->GetShaderContainer()->GetResShaderVariation(i);
            if (IsResShaderProgramInitialized(pVariation->GetResShaderProgram(GetShaderCodeType())))
            {
                pVariation->GetResShaderProgram(GetShaderCodeType())->Finalize(pDevice);
            }
        }
    }

    if (!skipFinalizingResShader)
    {
        nn::gfx::ResShaderContainer* pContainer = m_pResShaderFile->GetShaderContainer();
        NN_SDK_ASSERT_NOT_NULL(pContainer);
        if (IsResShaderContainerInitialized(pContainer))
        {
            pContainer->Finalize(pDevice);
        }
    }

    Layout::DeleteArray(m_pVertexStates, GetVariationCount());
    Layout::FreeMemory(m_pVertexShaderSlots);
    if (m_pGeometryShaderSlots != NULL)
    {
        Layout::FreeMemory(m_pGeometryShaderSlots);
    }
    Layout::FreeMemory(m_pPixelShaderSlots);
    Layout::FreeMemory(m_pTextureSlots);
}

//----------------------------------------
void ShaderInfo::SetShader(nn::gfx::CommandBuffer& commandBuffer, int variation) const
{
    NN_SDK_ASSERT(0 <= variation && variation < GetVariationCount());
    nn::gfx::ResShaderVariation* pVariation = m_pResShaderFile->GetShaderContainer()->GetResShaderVariation(variation);
    commandBuffer.SetShader(pVariation->GetResShaderProgram(GetShaderCodeType())->GetShader(), nn::gfx::ShaderStageBit_All);
    commandBuffer.SetVertexState(&m_pVertexStates[variation]);
}

//----------------------------------------
const nn::gfx::Shader* ShaderInfo::GetVertexShader(int variation) const
{
    return m_pResShaderFile->GetShaderContainer()->GetResShaderVariation(variation)->GetResShaderProgram(GetShaderCodeType())->GetShader();
}

//----------------------------------------
const nn::gfx::Shader* ShaderInfo::GetGeometryShader(int variation) const
{
    return m_pResShaderFile->GetShaderContainer()->GetResShaderVariation(variation)->GetResShaderProgram(GetShaderCodeType())->GetShader();
}

//----------------------------------------
const nn::gfx::Shader* ShaderInfo::GetPixelShader(int variation) const
{
    return m_pResShaderFile->GetShaderContainer()->GetResShaderVariation(variation)->GetResShaderProgram(GetShaderCodeType())->GetShader();
}

//----------------------------------------
int ShaderInfo::GetVariationCount() const
{
    return m_pResShaderFile->GetShaderContainer()->GetShaderVariationCount();
}

//----------------------------------------
void ShaderInfo::SetShaderCodeType(nn::gfx::ShaderCodeType code)
{
     m_Flags = detail::SetBits(m_Flags, PosShaderCodeType, BitsShaderCodeType, static_cast<uint32_t>(code));
}
nn::gfx::ShaderCodeType ShaderInfo::GetShaderCodeType() const
{
     return static_cast<nn::gfx::ShaderCodeType>(detail::GetBits(m_Flags, PosShaderCodeType, BitsShaderCodeType));
}
void ShaderInfo::SetTextureSlotCount(int textureSlotCount)
{
     m_Flags = detail::SetBits(m_Flags, PosTextureSlotCount, BitsTextureSlotCount, static_cast<uint32_t>(textureSlotCount));
}
int ShaderInfo::GetTextureSlotCount() const
{
     return static_cast<int>(detail::GetBits(m_Flags, PosTextureSlotCount, BitsTextureSlotCount));
}

} // namespace nn::ui2d
} // namespace nn
