﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_Arithmetic.h>

#include <nn/ui2d/ui2d_Common.h>
#include <nn/ui2d/ui2d_DrawInfo.h>
#include <nn/ui2d/ui2d_GraphicsResource.h>
#include <nn/ui2d/ui2d_Material.h>
#include <nn/ui2d/ui2d_Layout.h>
#include <nn/ui2d/ui2d_Resources.h>
#include <nn/ui2d/ui2d_Animation.h>
#include <nn/ui2d/ui2d_ResourceAccessor.h>
#include <nn/ui2d/ui2d_Util.h>
#include <nn/ui2d/ui2d_MaterialHelper.h>
#include <nn/ui2d/ui2d_Capture.h>
#include <nn/util/util_Matrix.h>

#include <nn/perf.h>

// アルファテスト用にインクルード
#if defined(NN_BUILD_CONFIG_SPEC_GENERIC)
#elif defined(NN_BUILD_CONFIG_SPEC_NX)
#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>
#endif

namespace nn
{
namespace ui2d
{

namespace
{
    // TODO: scale や translate を引数に取る関数が nn::util に実装されたら置き換える
    NN_FORCEINLINE void MatrixTextureProjectionOrthographicRightHanded(nn::util::MatrixT4x3fType* pOutValue, float left, float right, float bottom, float top, float scaleS, float scaleT, float translateS, float translateT) NN_NOEXCEPT
    {
        float reciprocalWidth = 1.f / (right - left);
        float reciprocalHeight = 1.f / (top - bottom);

        nn::util::MatrixSet(
            pOutValue,

            2.f * reciprocalWidth * scaleS,
            0.f,
            0.f,

            0.f,
            2.f * reciprocalHeight * scaleT,
            0.f,

            0.f,
            0.f,
            0.f,

            -(right + left) * reciprocalWidth * scaleS + translateS,
            -(top + bottom) * reciprocalHeight * scaleT + translateT,
            1.f
        );
    }

    // ResTexSrt から V 方向の座標系の補正がかかっていないテクスチャ SRT 行列を作成します。
    NN_FORCEINLINE void MakeTextureSrtMatrix(
        nn::util::MatrixT4x3fType* pTexSrtMtx,
        const ResTexSrt& resTexSrt)
    {
        NN_SDK_ASSERT_NOT_NULL(pTexSrtMtx);

        float texMtx[2][3];
        TexMap  dummyTexMap;
        Material::CalculateTextureMtx(texMtx, resTexSrt, dummyTexMap);

        // CalcualteTextureMtx で作成される情報は V 方向の座標系が反転されているためキャンセルしつつ行列を作成する。
        nn::util::Vector3fType texSrtAxisX = NN_UTIL_VECTOR_3F_INITIALIZER(texMtx[0][0], -texMtx[1][0], 0.0f);
        nn::util::Vector3fType texSrtAxisY = NN_UTIL_VECTOR_3F_INITIALIZER(texMtx[0][1], -texMtx[1][1], 0.0f);
        nn::util::Vector3fType texSrtAxisZ = NN_UTIL_VECTOR_3F_INITIALIZER(0.0f, 0.0f, 0.0f);
        nn::util::Vector3fType texSrtAxisW = NN_UTIL_VECTOR_3F_INITIALIZER(texMtx[0][2], -texMtx[1][2] + 1.0f, 0.0f);
        nn::util::MatrixSetAxisX(pTexSrtMtx, texSrtAxisX);
        nn::util::MatrixSetAxisY(pTexSrtMtx, texSrtAxisY);
        nn::util::MatrixSetAxisZ(pTexSrtMtx, texSrtAxisZ);
        nn::util::MatrixSetAxisW(pTexSrtMtx, texSrtAxisW);
    }

    // 透視投影テクスチャ投影を使用する際に正射影に合わせた投影位置に調整する行列を作成します。
    NN_FORCEINLINE void MakePerspectiveProjectionArrangeMatrix(
        nn::util::MatrixT4x3fType* pArrangeMtx,
        const DrawInfo& drawInfo,
        const float baseWidth,
        const float baseHeight,
        const float offsetX,
        const float offsetY,
        const ResProjectionTexGenParameters& texGen
    )
    {
        NN_SDK_ASSERT(
            drawInfo.IsValidDefaultViewportInfoSet(),
            "You must set default viewport and scissor information by DrawInfo::SetDefaultViewportScissorInfo() to use perspective texture projection function.");

        const gfx::ViewportStateInfo& viewportInfo = drawInfo.GetDefaultViewportInfo();

        const float fHorizontalRate = (float)viewportInfo.GetWidth() / (float)baseWidth;
        const float fVerticalRate = (float)viewportInfo.GetHeight() / (float)baseHeight;

        nn::util::Vector3fType  arrangeTrans
            = NN_UTIL_VECTOR_3F_INITIALIZER(0.5f - offsetX, 0.5f + offsetY, 0.0f);
        nn::util::Vector3fType  arrangeScale
            = NN_UTIL_VECTOR_3F_INITIALIZER(0.5f * fHorizontalRate / texGen.scale.x, -0.5f * fVerticalRate / texGen.scale.y, 1.0f);
        nn::util::MatrixIdentity(pArrangeMtx);
        nn::util::MatrixSetScale(pArrangeMtx, arrangeScale);
        nn::util::MatrixSetTranslate(pArrangeMtx, arrangeTrans);
    }



    // テクスチャ投影行列を計算します
    NN_FORCEINLINE void CalculateTextureProjectionMatrix(
        nn::util::MatrixT4x4fType* pTexMtx,
        const DrawInfo& drawInfo,
        const nn::util::MatrixT4x3fType& paneGlbMtx,
        const float baseWidth,
        const float baseHeight,
        const ResProjectionTexGenParameters& texGen,
        const TexGenSrc texGenSrc,
        const ResTexSrt& resTexSrt,
        const bool isAdjustProjectionSREnabled)
    {
        const nn::util::MatrixT4x3fType& modelViewMtx = drawInfo.GetModelViewMtx();

        const float halfWidth = baseWidth * 0.5f;
        const float halfHeight = baseHeight * 0.5f;

        switch(texGenSrc)
        {
        case TexGenSrc_PaneBaseOrthoProjection:
            if (isAdjustProjectionSREnabled)
            {
                nn::util::MatrixT4x3fType matrix4x3;
                MatrixTextureProjectionOrthographicRightHanded(
                    &matrix4x3,
                    -halfWidth,
                    halfWidth,
                    halfHeight,
                    -halfHeight,
                    0.5f / texGen.scale.x,
                    0.5f / texGen.scale.y,
                    0.5f - texGen.translate.x / texGen.scale.x / baseWidth,
                    0.5f + texGen.translate.y / texGen.scale.y / baseHeight);
                nn::util::MatrixT4x3fType inv;
                nn::util::MatrixInverse(&inv, paneGlbMtx);
                nn::util::MatrixMultiply(&matrix4x3, inv, matrix4x3);
                nn::util::MatrixConvert(pTexMtx, matrix4x3);
            }
            else
            {
                nn::util::Vector3f axisW;
                nn::util::MatrixGetAxisW(&axisW, paneGlbMtx);
                nn::util::MatrixT4x3fType matrix4x3;
                MatrixTextureProjectionOrthographicRightHanded(
                    &matrix4x3,
                    -halfWidth,
                    halfWidth,
                    halfHeight,
                    -halfHeight,
                    0.5f / texGen.scale.x,
                    0.5f / texGen.scale.y,
                    0.5f - (texGen.translate.x + axisW.GetX()) / texGen.scale.x / baseWidth,
                    0.5f + (texGen.translate.y + axisW.GetY()) / texGen.scale.y / baseHeight);
                nn::util::MatrixConvert(pTexMtx, matrix4x3);
            }

            {
                nn::util::MatrixT4x3fType matrix4x3;
                nn::util::MatrixConvert(&matrix4x3, *pTexMtx);
                nn::util::MatrixMultiply(
                    &matrix4x3,
                    modelViewMtx,
                    matrix4x3);
                nn::util::MatrixConvert(pTexMtx, matrix4x3);
            }
            break;
        case TexGenSrc_OrthoProjection:
            {
                nn::util::MatrixT4x3fType matrix4x3;
                MatrixTextureProjectionOrthographicRightHanded(
                    &matrix4x3,
                    -halfWidth,
                    halfWidth,
                    halfHeight,
                    -halfHeight,
                    0.5f / texGen.scale.x,
                    0.5f / texGen.scale.y,
                    0.5f - texGen.translate.x / texGen.scale.x / baseWidth,
                    0.5f + texGen.translate.y / texGen.scale.y / baseHeight);
                nn::util::MatrixConvert(pTexMtx, matrix4x3);
            }

            {
                nn::util::MatrixT4x3fType matrix4x3;
                nn::util::MatrixConvert(&matrix4x3, *pTexMtx);
                nn::util::MatrixMultiply(
                    &matrix4x3,
                    modelViewMtx,
                    matrix4x3);
                nn::util::MatrixConvert(pTexMtx, matrix4x3);
            }
            break;
        case TexGenSrc_PaneBasePerspectiveProjection:
            {
                nn::util::Vector3f axisW;
                nn::util::MatrixGetAxisW(&axisW, paneGlbMtx);
                const float offsetX = (texGen.translate.x + axisW.GetX()) / texGen.scale.x / baseWidth;
                const float offsetY = (texGen.translate.y + axisW.GetY()) / texGen.scale.y / baseHeight;

                // 投影位置調整用の行列を作成
                nn::util::MatrixT4x3fType    arrangeMtx;
                MakePerspectiveProjectionArrangeMatrix(&arrangeMtx, drawInfo, baseWidth, baseHeight, offsetX, offsetY, texGen);

                // テクスチャ SRT 行列の作成
                nn::util::MatrixT4x3fType    texSrtMtx;
                MakeTextureSrtMatrix(&texSrtMtx, resTexSrt);

                nn::util::MatrixT4x4fType    mvpMtx;

                // ペインの SR を考慮して投影機能は透視投影でも平行投影でも同じ結果になり紛らわしいため、透視投影の場合は OFF の前提になっています。
                nn::util::MatrixMultiply(&mvpMtx, modelViewMtx, drawInfo.GetProjectionMtx());

                nn::util::MatrixT4x3fType    arrangeSrtMtx;
                nn::util::MatrixMultiply(&arrangeSrtMtx, arrangeMtx, texSrtMtx);
                nn::util::MatrixMultiply(pTexMtx, mvpMtx, arrangeSrtMtx);
            }
            break;

        case TexGenSrc_PerspectiveProjection:
            {
                const float offsetX = texGen.translate.x / texGen.scale.x / baseWidth;
                const float offsetY = texGen.translate.y / texGen.scale.y / baseHeight;

                // 投影位置調整用の行列を作成
                nn::util::MatrixT4x3fType    arrangeMtx;
                MakePerspectiveProjectionArrangeMatrix(&arrangeMtx, drawInfo, baseWidth, baseHeight, offsetX, offsetY, texGen);

                // テクスチャ SRT 行列の作成
                nn::util::MatrixT4x3fType    texSrtMtx;
                MakeTextureSrtMatrix(&texSrtMtx, resTexSrt);

                nn::util::MatrixT4x4fType    mvpMtx;
                nn::util::MatrixMultiply(&mvpMtx, modelViewMtx, drawInfo.GetProjectionMtx());
                nn::util::MatrixT4x3fType    arrangeSrtMtx;
                nn::util::MatrixMultiply(&arrangeSrtMtx, arrangeMtx, texSrtMtx);
                nn::util::MatrixMultiply(pTexMtx, mvpMtx, arrangeSrtMtx);
            }
            break;
        default:
            NN_SDK_ASSERT(false, "This type of TexGenSrc is not supported(%d).", texGenSrc);
            break;
        }
    }// NOLINT(impl/function_size)
}

//----------------------------------------
Material::Material()
{
    Initialize();
}

//----------------------------------------
void
Material::InitializeMaterialImpl(
    BuildResultInformation* pOutBuildResultInformation,
    nn::gfx::Device* pDevice,
    const ResMaterial* pBaseRes,
    const ResMaterial* pOverrideRes,
    const BuildArgSet& buildArgSet,
    CaptureTextureCopyInfo* pCaptureTextureCopyInfo
)
{
    Initialize();
    // 上書き設定の解決
    const ResMaterial* pRes = NULL;
    const ResMaterial* pColorRes = NULL;
    const ResMaterial* pTextureRes = NULL;
    const ResMaterial* pTexMapNumRes = NULL;
    const BuildResSet* pBuildResSet = NULL;
    if (pOverrideRes)
    {
        // 上書きのリソースがある
        if (buildArgSet.overrideMaterialUsageFlag != 0)
        {
            // 上書きがない状態から部分上書きを一つずつ適用させる
            pRes = pBaseRes;
            pColorRes = pBaseRes;
            pTextureRes = pBaseRes;
            pTexMapNumRes = pBaseRes;
            pBuildResSet = buildArgSet.pCurrentBuildResSet;

            if (detail::TestBit(buildArgSet.overrideMaterialUsageFlag, MaterialOverrideUsageFlag_InterpolateColorEnabled))
            {
                // カラーの部分上書き
                pColorRes = pOverrideRes;
            }

            if (detail::TestBit(buildArgSet.overrideMaterialUsageFlag, MaterialOverrideUsageFlag_TextureEnabled))
            {
                if (pBaseRes->resCount.GetTexMapCount() == pOverrideRes->resCount.GetTexMapCount())
                {
                    // テクスチャの部分上書き
                    pTextureRes = pOverrideRes;
                    pBuildResSet = buildArgSet.pOverrideBuildResSet;
                }
                else
                {
                    // テクスチャの部分上書きで TexMapNum の数が変わった場合、
                    // 仕様としては不正とみなし assert で止め、
                    // 挙動としては部分上書きをスキップする
                    NN_SDK_ASSERT(false, "TexMapNum must not be changed when overwriting. (BaseTexMapNum=%d, TargetTexMapNum=%d, Material=%s, Layout=%s)", pBaseRes->resCount.GetTexMapCount(), pOverrideRes->resCount.GetTexMapCount(), GetName(), pBuildResSet->pLayout->GetName());
                }
            }
        }
        else if (buildArgSet.overrideUsageFlag != 0)
        {
            // 一つでもTextBoxの部分上書きのフラグが立っていたら、マテリアルは上書きしない
            pRes = pBaseRes;
            pColorRes = pBaseRes;
            pTextureRes = pBaseRes;
            pTexMapNumRes = pBaseRes;
            pBuildResSet = buildArgSet.pCurrentBuildResSet;
        }
        else
        {
            // 完全に上書き
            pRes = pOverrideRes;
            pColorRes = pOverrideRes;
            pTextureRes = pOverrideRes;
            pTexMapNumRes = pOverrideRes;
            pBuildResSet = buildArgSet.pOverrideBuildResSet;
        }
    }
    else
    {
        // 上書きがない
        pRes = pBaseRes;
        pColorRes = pBaseRes;
        pTextureRes = pBaseRes;
        pTexMapNumRes = pBaseRes;
        pBuildResSet = buildArgSet.pCurrentBuildResSet;
    }

    SetName(pRes->name);

    uint32_t  resOffset = sizeof(ResMaterial);

    const ResConstantColor *const pResConstantColor = nn::util::ConstBytePtr(pColorRes, resOffset).Get<ResConstantColor>();

    // constant color
    // float カラーが使用されている場合は m_Colors 内に直接値を持てないので
    // カラーデータを保存する領域を動的に確保する。
    bool bUseDynamicAllocateColor = false;
    for (int i = 0; i < MatColorMax; ++i)
    {
        if (detail::TestBit(pResConstantColor->types, i))
        {
            bUseDynamicAllocateColor = true;
            break;
        }
    }

    if(bUseDynamicAllocateColor)
    {
        m_Colors.pFloatColor = static_cast<nn::util::Float4*>(Layout::AllocateMemory(sizeof(nn::util::Float4) * 2));
        m_Flag = detail::SetBits(m_Flag, Flags_DynamicAllocatedColorData, 1, uint8_t(1));

        for (int i = 0; i < MatColorMax; ++i)
        {
            m_Flag = detail::SetBits(m_Flag, Flags_BlackColorFloat + i, 1, uint8_t(1));
            m_Colors.pFloatColor[i] = *pResConstantColor->GetColorAsFloat(i);
        }
    }
    else
    {
        for (int i = 0; i < MatColorMax; ++i)
        {
            m_Colors.byteColor[i] = *pResConstantColor->GetColorAsByte(i);
        }
    }

    resOffset += static_cast<uint32_t>(pResConstantColor->CalculateSize());

    const ResTexMap *const pResTexMap = nn::util::ConstBytePtr(pTextureRes, resOffset).Get<ResTexMap>();
    resOffset += sizeof(ResTexMap) * pTextureRes->resCount.GetTexMapCount();

    // ResTexmapAdditionalInfo は HasTexMapAdditionalInfo() の設定によって(例えばキャプチャテクスチャの使用の有無)によってデータブロックが存在しないこともある。
    // resOffset は pRes からの読み込みに使用されるため pRes の状態に合わせてオフセットを進める。
    const ResTexMapAdditionalInfo *const pResTexMapAdditionalInfo = nn::util::ConstBytePtr(pTextureRes, resOffset).Get<ResTexMapAdditionalInfo>();
    resOffset += pRes->resCount.HasTexMapAdditionalInfo() ? sizeof(ResTexMapAdditionalInfo) * pRes->resCount.GetTexMapCount() : 0;

    const ResTexSrt *const pResTexSrts = nn::util::ConstBytePtr(pRes, resOffset).Get<ResTexSrt>();
    resOffset += sizeof(ResTexSrt) * pRes->resCount.GetTexSrtCount();

    const ResTexCoordGen *const pResTexCoordGens = nn::util::ConstBytePtr(pRes, resOffset).Get<ResTexCoordGen>();
    resOffset += sizeof(ResTexCoordGen) * pRes->resCount.GetTexCoordGenCount();

    const ResTevStage *const pResTevStages = nn::util::ConstBytePtr(pRes, resOffset).Get<ResTevStage>();
    resOffset += sizeof(ResTevStage) * pRes->resCount.GetTevStageCount();

    const ResAlphaCompare *const pResAlphaCompare = nn::util::ConstBytePtr(pRes, resOffset).Get<ResAlphaCompare>();
    resOffset += pRes->resCount.HasAlphaCompare() ? sizeof(ResAlphaCompare) : 0;


    const ResBlendMode *const pResBlendMode = nn::util::ConstBytePtr(pRes, resOffset).Get<ResBlendMode>();
    resOffset += pRes->resCount.HasBlendMode() ? sizeof(ResBlendMode) : 0;

    const ResBlendMode *const pResBlendModeAlpha = nn::util::ConstBytePtr(pRes, resOffset).Get<ResBlendMode>();
    resOffset += pRes->resCount.IsSeparateBlendMode() ? sizeof(ResBlendMode) : 0;

    const ResIndirectParameter *const pResIndirectParameter = nn::util::ConstBytePtr(pRes, resOffset).Get<ResIndirectParameter>();
    resOffset += pRes->resCount.HasIndirectParameter() ? sizeof(ResIndirectParameter) : 0;

    const ResDetailedCombinerStageInfo *const pCombinerInfoParameter = pRes->resCount.HasDetailedCombiner() ? nn::util::ConstBytePtr(pRes, resOffset).Get<ResDetailedCombinerStageInfo>() : NULL;
    resOffset += pRes->resCount.HasDetailedCombiner() ? sizeof(ResDetailedCombinerStageInfo) : 0;

    const ResDetailedCombinerStage * const pCombinerStage = pRes->resCount.HasDetailedCombiner() ? nn::util::ConstBytePtr(pRes, resOffset).Get<ResDetailedCombinerStage>() : NULL;
    resOffset += pRes->resCount.HasDetailedCombiner() ? sizeof(ResDetailedCombinerStage) * pRes->resCount.GetTevStageCount() : 0;

    const ResProjectionTexGenParameters *const pResProjectionTexGenParameter = nn::util::ConstBytePtr(pRes, resOffset).Get<ResProjectionTexGenParameters>();
    resOffset += sizeof(ResProjectionTexGenParameters) * pRes->resCount.GetProjectionTexGenCount();

    const ResFontShadowParameter *const pFontShadowParameter = nn::util::ConstBytePtr(pRes, resOffset).Get<ResFontShadowParameter>();
    resOffset += pRes->resCount.HasFontShadowParameter() ? sizeof(ResFontShadowParameter) : 0;

    const ResCombinerUserShader *const pCombinerUserShader = nn::util::ConstBytePtr(pRes, resOffset).Get<ResCombinerUserShader>();
    resOffset += pRes->resCount.HasCombinerUserShader() ? sizeof(ResCombinerUserShader) : 0;

    const uint8_t  texMapNum = std::min(pTexMapNumRes->resCount.GetTexMapCount(), static_cast<uint8_t>(TexMapMax));

    const uint8_t  texSrtNum = std::min(pRes->resCount.GetTexSrtCount(), static_cast<uint8_t>(TexMapMax));
    const uint8_t  texCoordGenNum = std::min(pRes->resCount.GetTexCoordGenCount(), static_cast<uint8_t>(TexMapMax));
    const uint8_t  tevStageNum = std::min(pRes->resCount.GetTevStageCount(), static_cast<uint8_t>(TevStageMax));
    const uint8_t  projectionTexGenNum = std::min(pRes->resCount.GetProjectionTexGenCount(), static_cast<uint8_t>(TevStageMax));
    const bool allocAlpComp = pRes->resCount.HasAlphaCompare();
    const bool allocBlendMode = pRes->resCount.HasBlendMode();
    const bool allocBlendModeAlpha = pRes->resCount.IsSeparateBlendMode();
    const bool allocIndirectParameter = pRes->resCount.HasIndirectParameter();
    const bool allocFontShadowParameter = pRes->resCount.HasFontShadowParameter();
    const bool allocCombinerParameter = pRes->resCount.HasDetailedCombiner();
    const bool allocCombinerUserShaderParameter = pRes->resCount.HasCombinerUserShader();
    uint8_t  blendModeNum;
    if (allocBlendModeAlpha) {
        blendModeNum = 2;
    }
    else {
        if (allocBlendMode) {
            blendModeNum = 1;
        }
        else {
            blendModeNum = 0;
        }
    }

    SetThresholdingAlphaInterpolation(pRes->resCount.IsThresholdingAlphaInterpolation());

    // TODO
    ReserveMem(
        texMapNum,
        texSrtNum,
        texCoordGenNum,
        tevStageNum,
        allocAlpComp,
        blendModeNum,
        allocIndirectParameter,
        projectionTexGenNum,
        allocFontShadowParameter,
        allocCombinerParameter,
        allocCombinerUserShaderParameter
    );

    this->SetTextureOnly(pRes->resCount.IsTextureOnly());

    if (m_pMem)
    {
        // texmap
        for (uint32_t i = 0; i < m_MemCap.texMap; ++i)
        {
            new (GetTexMapArray() + i) TexMap();
        }

        SetTexMapCount(texMapNum);

        // TexMap の初期化処理
        if (texMapNum > 0)
        {
            NN_SDK_ASSERT(pBuildResSet->pTextureList, "pBuildResSet->pTextureList must not be NULL for Material[%s] Layout[%s]", GetName(), pBuildResSet->pLayout->GetName());

            const ResTexture *const pResTextures = nn::util::ConstBytePtr(pBuildResSet->pTextureList, sizeof(*pBuildResSet->pTextureList)).Get<ResTexture>();
            TexMap *const pTexMapArray = GetTexMapArray();

            uint8_t  di = 0;
            for (uint8_t si = 0; si < texMapNum; ++si)
            {
                int texIdx = pResTexMap[si].texIdx;
                NN_SDK_ASSERT(texIdx < pBuildResSet->pTextureList->texCount,
                    "out of bouds: texIdx[%d] < pBuildResSet->pTextureList->texCount[%d] for Material[%s] Layout[%s]",
                    texIdx, static_cast<int>(pBuildResSet->pTextureList->texCount), GetName(), pBuildResSet->pLayout->GetName());
                const char* pFileName = nn::util::ConstBytePtr(pResTextures, pResTextures[texIdx].nameStrOffset).Get<char>();
                const TextureInfo* texInfo = NULL;

                // キャプチャテクスチャの場合はレイアウトとパーツの階層構造から絶対パスを作成して一意になるように名前を付ける。
                if (pTextureRes->resCount.HasTexMapAdditionalInfo() &&
                    detail::TestBit(pResTexMapAdditionalInfo[si].info, ResTexMapAdditionalInfo::InfoType::InfoType_CaptureTextureEnabled))
                {
                    // キャプチャテクスチャの使用状況をコピーする
                    if (pCaptureTextureCopyInfo != NULL)
                    {
                        detail::SetBit(&(pCaptureTextureCopyInfo->useFlags), si, 1);
                    }

                    // プリフィックスが設定されている場合はつなげた名前でキャプチャテクスチャを作成する。
                    bool overrided = pBuildResSet != buildArgSet.pCurrentBuildResSet;
                    if (pCaptureTextureCopyInfo != NULL)
                    {
                        texInfo = AcquireCaptureTextureWithResolvePrefix(
                            pCaptureTextureCopyInfo->names[si],
                            CaptureTexturePathMax,
                            buildArgSet,
                            overrided,
                            pDevice,
                            pBuildResSet->pResAccessor,
                            pFileName);
                    }
                    else
                    {
                        texInfo = AcquireCaptureTextureWithResolvePrefix(NULL, 0, buildArgSet, overrided, pDevice, pBuildResSet->pResAccessor, pFileName);
                    }
                }
                else
                {
                    texInfo = pBuildResSet->pResAccessor->AcquireTexture(pDevice, pFileName);
                }

                pTexMapArray[di].Set(texInfo);
                pTexMapArray[di].SetWrapMode(pResTexMap[si].GetWarpModeS(), pResTexMap[si].GetWarpModeT());
                pTexMapArray[di].SetFilter(pResTexMap[si].GetMinFilter(), pResTexMap[si].GetMagFilter());
                ++di;
            }
        }

        // テクスチャSrt
        ResTexSrt* pTexSrtArray = GetTexSrtArray();
        for (uint32_t i = 0; i < m_MemCap.texSrt; ++i)
        {
            memset(&pTexSrtArray[i], 0, sizeof(ResTexSrt));
        }
        SetTexSrtCount(texSrtNum);
        for (int i = 0; i < texSrtNum; ++i)
        {
            pTexSrtArray[i].translate = pResTexSrts[i].translate;
            pTexSrtArray[i].rotate = pResTexSrts[i].rotate;
            pTexSrtArray[i].scale = pResTexSrts[i].scale;
        }

        // テクスチャ座標生成
        ResTexCoordGen* pTexCoordGenArray = GetTexCoordGenArray();
        for (uint32_t i = 0; i < m_MemCap.texCoordGen; ++i)
        {
            new(static_cast<void*>(&pTexCoordGenArray[i])) ResTexCoordGen();
        }
        SetTexCoordGenCount(texCoordGenNum);
        uint32_t  numProjectionTexGen = 0;
        for (uint32_t i = 0; i < m_MemCount.texCoordGen; ++i)
        {
            pTexCoordGenArray[i] = pResTexCoordGens[i];

            ResProjectionTexGenParameters *const projectionTexGen = GetProjectionTexGenArray();
            if (pTexCoordGenArray[i].IsProjection())
            {
                NN_SDK_ASSERT(numProjectionTexGen < projectionTexGenNum,
                    "out of bouds: numProjectionTexGen[%d] < projectionTexGenNum[%d] for Material[%s] Layout[%s]",
                    numProjectionTexGen, projectionTexGenNum, GetName(), pBuildResSet->pLayout->GetName());
                pTexCoordGenArray[i].SetProjectionTexGenParameters(&projectionTexGen[numProjectionTexGen]);
                numProjectionTexGen++;
            }
            else {
                pTexCoordGenArray[i].SetProjectionTexGenParameters(NULL);
            }
        }

        if (tevStageNum > 0)
        {
            ResTevStage *const pTevStageArray = GetTevStageArray();
            for (uint32_t i = 0; i < m_MemCap.tevStage; ++i)
            {
                new(static_cast<void*>(&pTevStageArray[i])) ResTevStage();
            }
            SetTevStageCount(tevStageNum);
            for (int i = 0; i < tevStageNum; ++i)
            {
                pTevStageArray[i] = pResTevStages[i];
            }
        }

        if (allocAlpComp)
        {
            NN_SDK_ASSERT(IsAlphaCompareCap(), "IsAlphaCompareCap() must be true for Material[%s] Layout[%s]",
                GetName(), pBuildResSet->pLayout->GetName());
            *GetAlphaComparePtr() = *pResAlphaCompare;
        }

        if (allocBlendMode)
        {
            NN_SDK_ASSERT(IsBlendModeCap(), "IsBlendModeCap() must be true for Material[%s] Layout[%s]",
                GetName(), pBuildResSet->pLayout->GetName());
            *GetBlendModePtr() = *pResBlendMode;
        }

        if (allocBlendModeAlpha)
        {
            NN_SDK_ASSERT(IsBlendModeAlphaCap(), "IsBlendModeAlphaCap() must be true for Material[%s] Layout[%s]",
                GetName(), pBuildResSet->pLayout->GetName());
            *GetBlendModeAlphaPtr() = *pResBlendModeAlpha;
        }

        if (allocIndirectParameter)
        {
            NN_SDK_ASSERT(IsIndirectParameterCap(), "IsIndirectParameterCap() must be true for Material[%s] Layout[%s]",
                GetName(), pBuildResSet->pLayout->GetName());
            *GetIndirectParameterPtr() = *pResIndirectParameter;
        }

        if (allocCombinerParameter && pCombinerInfoParameter != NULL)
        {
            NN_SDK_ASSERT(UseDetailedCombinerCap(), "UseCombinerCap() must be true for Material[%s] Layout[%s]",
                GetName(), pBuildResSet->pLayout->GetName());

            *GetDetailedCombinerStageInfoPtr() = *pCombinerInfoParameter;
            ResDetailedCombinerStage *const pCombinerStageArray = GetDetailedCombinerStageAry();
            for (int i = 0; i < tevStageNum; i++)
            {
                pCombinerStageArray[i] = pCombinerStage[i];
            }
        }

        if (projectionTexGenNum > 0)
        {
            ResProjectionTexGenParameters *const pProjectionTexGenArray = GetProjectionTexGenArray();
            for (uint32_t i = 0; i < m_MemCap.projectionTexGen; ++i)
            {
                memset(&pProjectionTexGenArray[i], 0, sizeof(ResProjectionTexGenParameters));
            }
            SetProjectionTexGenCount(projectionTexGenNum);
            for (int i = 0; i < projectionTexGenNum; ++i)
            {
                pProjectionTexGenArray[i] = pResProjectionTexGenParameter[i];
            }
        }

        if (allocFontShadowParameter)
        {
            NN_SDK_ASSERT(IsFontShadowParameterCap(), "IsFontShadowParameterCap() must be true for Material[%s] Layout[%s]",
                GetName(), pBuildResSet->pLayout->GetName());
            *GetFontShadowParameterPtr() = *pFontShadowParameter;
        }

        if (allocCombinerUserShaderParameter)
        {
            NN_SDK_ASSERT(UseCombinerUserShaderCap(), "UseCombinerUserShaderCap() must be true for Material[%s] Layout[%s]",
                GetName(), pBuildResSet->pLayout->GetName());
            *GetCombinerUserShaderPtr() = *pCombinerUserShader;
        }
    }

    // シェーダーのセットアップ
    SetupShader(pDevice, buildArgSet, pBuildResSet);

    // コンスタントバッファの使用サイズを収集する
    CollectConstantBufferSize(pOutBuildResultInformation, pDevice);

    InitializeBlendInformationImpl(pDevice);
}// NOLINT(impl/function_size)


void Material::ModifyShaderKeyForProceduralShape(int* pKey0, int* pKey1, const BuildArgSet& buildArgSet)
{
    NN_SDK_ASSERT_NOT_NULL(pKey0);
    NN_SDK_ASSERT_NOT_NULL(pKey1);

    // 拡張ユーザーデータに角丸データが存在するかどうかを調べて使用するシェーダーを切り替えます。
    if (buildArgSet.pExtUserDataList != NULL)
    {
        const ResExtUserData* pFirstData = nn::util::ConstBytePtr(buildArgSet.pExtUserDataList, sizeof(*buildArgSet.pExtUserDataList)).Get<const ResExtUserData>();
        if (pFirstData->GetType() == ExtUserDataType::ExtUserDataType_SystemData)
        {
            int systemDataCount = pFirstData->GetSystemDataCount();
            for (int i = 0; i < systemDataCount; ++i)
            {
                const void* pData = pFirstData->GetSystemData(i);
                if (*(static_cast<const PaneSystemDataType*>(pData)) == PaneSystemDataType_ProceduralShape)
                {
                    if (m_MemCount.texMap < 3)
                    {
                        // 標準シェーダーの並びに合わせたシェーダーを選択
                        *pKey0 = ProceduralShapeDefaultShader;
                        *pKey1 = m_ShaderId;
                    }
                    else
                    {
                        *pKey0 += ProceduralShapeBlendShaderOffset;
                    }
                    break;
                }
            }
        }
    }

}

//----------------------------------------
void Material::SetupShader(nn::gfx::Device* pDevice, const BuildArgSet& buildArgSet, const BuildResSet* pBuildResSet)
{
    // シェーダの選択とインダイレクトテクスチャの設定。
    int shaderKey0 = -1;
    int shaderKey1 = -1;
    char shaderNameBuf[6];
    shaderNameBuf[0] = '\0';
    m_pShaderInfo = NULL;

    UserShaderInformation  userShaderInformation;
    userShaderInformation.SetDefault();

    if (buildArgSet.pGetUserShaderInformationFromUserDataCallback != NULL &&
        (*buildArgSet.pGetUserShaderInformationFromUserDataCallback)(userShaderInformation, buildArgSet.pExtUserDataList, buildArgSet.pGetUserShaderInformationFromUserDataCallbackUserData))
    {
        NN_SDK_ASSERT(m_MemCount.detailedCombinerParameter == 0, "UserShader with detailed combiner set can't be used.");

        strcpy(shaderNameBuf, userShaderInformation.userShaderName);

        SetupUserShaderConstantBufferInformation(userShaderInformation);
    }
    else
    {
        if (m_MemCount.detailedCombinerParameter == 1)
        {
            if (!pBuildResSet->isDetailedCombinerWithVariationVariable)
            {
                // 動的な分岐を行う詳細コンバイナモード
                GetShaderForDetailedCombiner(&shaderKey0, &shaderKey1);
            }
            else
            {
                // バリエーション変数を利用した静的分岐を行う詳細コンバイナモード
                shaderKey0 = IndividualArchiveShaderVariationTableFirstBlendWithDetailedCombiner;
                shaderKey1 = 0;
            }
        }
        else if (m_MemCount.combinerUserShaderParameter == 1)
        {
            shaderKey0 = IndividualArchiveShaderVariationTableFirstBlendWithCombinerUserShader;
            shaderKey1 = 0;
        }
        else
        {
            switch (m_MemCount.texMap)
            {
            case 0:
                m_ShaderId = ShaderId_NullTexture;
                break;
            case 1:
                m_ShaderId = ShaderId_SingleTexture;
                break;
            case 2:
                if (IsIndirectBlendUsed())
                {
                    m_ShaderId = ShaderId_DoubleIndirectTexture;
                }
                else
                {
                    // tevStage.GetCombineRgb() の 値は、nn::ui2d::TevMode の値と一致している。
                    const ResTevStage& tevStage = this->GetTevStage(0);
                    m_ShaderId = static_cast<uint8_t>(tevStage.GetCombineRgb() + ShaderId_DoubleTextureBaseIdx);
                }
                break;
            case 3:
                GetShaderBlend(&shaderKey0, &shaderKey1);
                break;
            default:
                NN_SDK_ASSERT(false, "");
                break;
            }

            // 透視投影テクスチャプロジェクションを使用する場合はシェーダーを切り替えるためにシェーダーキーを書き換える
            // 通常テクスチャが 2 枚以下の場合は標準シェーダーが使用されるが、透視投影テクスチャプロジェクションの場合はすべてのシェーダーがアーカイブシェーダーに含まれている。
            // 2 枚以下の場合は shaderKey0 が 250, shaderKey1 が ShaderId に対応した値でバリエーションが定義されている。
            // 3 枚ブレンドの場合はブレンド設定から作成した Key0 と Key1 に対して、 Key0 に 200 を加算した場所にバリエーションが定義されている。
            if (IsPerspectiveTextureProjectionUsed())
            {
                if (m_MemCount.texMap < 3)
                {
                    // 標準シェーダーの並びに合わせたシェーダーを選択
                    shaderKey0 = PerspectiveTextureProjectionDefaultShader;
                    shaderKey1 = m_ShaderId;
                }
                else
                {
                    shaderKey0 += PerspectiveTextureProjectionBlendShaderOffset;
                }
            }
        }
    }

    // 拡張ユーザーデータに角丸データが存在するかどうかを調べて使用するシェーダーを切り替えます。
    ModifyShaderKeyForProceduralShape(&shaderKey0, &shaderKey1, buildArgSet);

    // shaderNameBufが設定されていればユーザが作成したアーカイブシェーダを使用する
    if (shaderNameBuf[0] != '\0')
    {
        m_pShaderInfo = pBuildResSet->pLayout->AcquireArchiveShader(pDevice, shaderNameBuf);
        NN_SDK_ASSERT(m_pShaderInfo != NULL, "ShaderInfo[%s] not found for Material[%s].", shaderNameBuf, GetName());
        m_ShaderVariation = 0;
        m_ShaderId = ShaderId_ArchiveShader;
    }
    else if (shaderKey0 != -1 && shaderKey1 != -1)
    {
        m_ShaderId = ShaderId_ArchiveShader;
        if(shaderKey0 != IndividualArchiveShaderVariationTableFirstBlendWithDetailedCombiner &&
           shaderKey0 != IndividualArchiveShaderVariationTableFirstBlendWithCombinerUserShader)
        {
            char archiveShaderName[8];
            ConvertBlendsToArchiveShaderName(archiveShaderName, shaderKey0, shaderKey1);
            // 統合されたアーカイブシェーダを使用
            m_pShaderInfo = pBuildResSet->pLayout->AcquireArchiveShader(pDevice, archiveShaderName);
            NN_SDK_ASSERT(m_pShaderInfo != NULL, "ShaderInfo[%s] not found for Material[%s].", archiveShaderName, GetName());

            const void* pVariationTable = m_pShaderInfo->GetVariationTable();
            NN_SDK_ASSERT(pVariationTable != NULL, "VariationTable not found for Material[%s].", GetName());

            // バリエーションを検索して設定
            const int variationIndex = SearchShaderVariationIndexFromTable(pVariationTable, shaderKey0, shaderKey1);
            NN_SDK_ASSERT(variationIndex != -1, "The shader variation table is broken for Material[%s].", GetName());
            m_ShaderVariation = static_cast<uint8_t>(variationIndex);
        }
        else
        {
            // バリエーション変数を利用した詳細コンバイナ用インデックスの検索
            int stageBits[DetailedCombinerStageBitsCountWithVariationTable];
            memset(stageBits, 0, sizeof(int) * DetailedCombinerStageBitsCountWithVariationTable);

            if (shaderKey0 == IndividualArchiveShaderVariationTableFirstBlendWithDetailedCombiner)
            {
                const nn::ui2d::ResDetailedCombinerStage* pDetailedCombinerStage = GetDetailedCombinerStageAry();
                for (int i = 0; i < GetTevStageCount(); i++)
                {
                    stageBits[i * 4 + 0] = pDetailedCombinerStage[i].bits0;
                    stageBits[i * 4 + 1] = pDetailedCombinerStage[i].bits1;
                    stageBits[i * 4 + 2] = pDetailedCombinerStage[i].bits2;
                    stageBits[i * 4 + 3] = pDetailedCombinerStage[i].bits3;
                }

                // バイナリのバージョンを見て、バリエーション変数を利用した詳細コンバイナを利用するか決めます。
                const uint8_t val = uint8_t(pBuildResSet != NULL ? pBuildResSet->isDetailedCombinerWithVariationVariable : false);
                m_Flag = detail::SetBits(m_Flag, Flags_UseDetailedCombinerWithVariationVariable, 1, val);
            }
            else
            {
                NN_SDK_ASSERT(shaderKey0 == IndividualArchiveShaderVariationTableFirstBlendWithCombinerUserShader);
                const ui2d::ResCombinerUserShader* pCombinerUserShader = GetCombinerUserShaderPtr();
                memcpy(stageBits, pCombinerUserShader->GetKeyCode(), DetailedCombinerStageBitsCountWithVariationTable * sizeof(int));
            }

            // 詳細コンバイナ用文字  "110_000" + "_00000000" * DetailedCombinerStageBitsCountWithVariationTable + "\0"
            const int archiveShaderNameSize = 8 + (9 * DetailedCombinerStageBitsCountWithVariationTable);
            char archiveShaderName[archiveShaderNameSize];
            ConvertStageBitsToArchiveShaderDetailedCombinerName(archiveShaderName, archiveShaderNameSize, shaderKey0, shaderKey1, stageBits);
            // 統合されたアーカイブシェーダを使用
            m_pShaderInfo = pBuildResSet->pLayout->AcquireArchiveShader(pDevice, archiveShaderName);
            NN_SDK_ASSERT(m_pShaderInfo != NULL, "ShaderInfo[%s] not found for Material[%s].", archiveShaderName, GetName());

            const void* pExtensionVariationTable = m_pShaderInfo->GetVariationTable();
            NN_SDK_ASSERT(pExtensionVariationTable != NULL, "ExtensionVariationTable not found for Material[%s].", GetName());

            // バリエーションを検索して設定
            const int variationIndex = SearchShaderVariationDetailedCombinerIndexFromTable(pExtensionVariationTable, stageBits);
            NN_SDK_ASSERT(variationIndex != -1,
                "0x%08x, 0x%08x, 0x%08x, 0x%08x,\n"
                "0x%08x, 0x%08x, 0x%08x, 0x%08x,\n"
                "0x%08x, 0x%08x, 0x%08x, 0x%08x,\n"
                "0x%08x, 0x%08x, 0x%08x, 0x%08x,\n"
                "0x%08x, 0x%08x, 0x%08x, 0x%08x,\n"
                "0x%08x, 0x%08x, 0x%08x, 0x%08x,\n"
                "The shader variation table is broken for Material[%s].",
                stageBits[ 0], stageBits[ 1], stageBits[ 2], stageBits[ 3],
                stageBits[ 4], stageBits[ 5], stageBits[ 6], stageBits[ 7],
                stageBits[ 8], stageBits[ 9], stageBits[10], stageBits[11],
                stageBits[12], stageBits[13], stageBits[14], stageBits[15],
                stageBits[16], stageBits[17], stageBits[18], stageBits[19],
                stageBits[20], stageBits[21], stageBits[22], stageBits[23],
                GetName());
            m_ShaderVariation = static_cast<uint8_t>(variationIndex);
        }
    }

    // アーカイブシェーダーの場合
    // バイナリバージョンを見て、ブレンド毎バリエーション数を決めます。
    if (m_ShaderId == ShaderId_ArchiveShader)
    {
        const uint8_t val = uint8_t(pBuildResSet != NULL ? pBuildResSet->isFlytOlderThanBinary8200 : false);
        m_Flag = detail::SetBits(m_Flag, Flags_Use2ForShaderVariationCountPerBlend, 1, val);
    }

} // NOLINT(impl/function_size)

//----------------------------------------
bool Material::IsIndirectBlendUsed() const
{
    // インダイレクトブレンドが使われているか判定します。
    for (int idx = 0; idx < this->GetTevStageCount(); ++idx)
    {
        TevMode tevMode = this->GetTevStage(idx).GetCombineRgb();
        if (tevMode == TevMode_Indirect ||
            tevMode == TevMode_BlendIndirect ||
            tevMode == TevMode_EachIndirect)
        {
            return true;
        }
    }

    return false;
}

//----------------------------------------
bool Material::IsPerspectiveTextureProjectionUsed() const
{
    // PerspectiveTextureProjection が使われているか判定します。
    for (uint8_t idx = 0; idx < this->GetTexMapCount(); ++idx)
    {
        if (GetTexCoordGen(idx).IsPerspectiveProjection())
        {
            return true;
        }
    }

    return false;
}


//----------------------------------------
void Material::SetupUserShaderConstantBufferInformation(const UserShaderInformation& userShaderInformation)
{
    m_pUserShaderConstantBufferInformation = static_cast<UserShaderConstantBufferInformation*>(Layout::AllocateMemory(sizeof(UserShaderConstantBufferInformation)));
    m_pUserShaderConstantBufferInformation->SetDefault();

    m_pUserShaderConstantBufferInformation->m_ConstantBufferExtendAreaSizeForVertexShader = userShaderInformation.vertexShaderConstantBufferExtendSize;
    m_pUserShaderConstantBufferInformation->m_ConstantBufferExtendAreaSizeForPixelShader = userShaderInformation.pixelShaderConstantBufferExtendSize;
    m_pUserShaderConstantBufferInformation->m_ConstantBufferExtendAreaSizeForGeometryShader = userShaderInformation.geometryShaderConstantBufferExtendSize;
}

//----------------------------------------
void Material::CopyUserShaderConstantBufferInformation(const Material& material)
{
    if (material.m_pUserShaderConstantBufferInformation != NULL)
    {
        m_pUserShaderConstantBufferInformation = static_cast<UserShaderConstantBufferInformation*>(Layout::AllocateMemory(sizeof(UserShaderConstantBufferInformation)));

        // いったんすべてコピーしてオフセットだけクリアする。
        *m_pUserShaderConstantBufferInformation = *material.m_pUserShaderConstantBufferInformation;
        m_pUserShaderConstantBufferInformation->m_GeometryShaderConstantBufferOffset = 0;
    }
    else
    {
        m_pUserShaderConstantBufferInformation = NULL;
    }
}

//----------------------------------------
void Material::CollectConstantBufferSize(BuildResultInformation* pOutBuildResultInformation, nn::gfx::Device* pDevice) const
{
    if (pOutBuildResultInformation != NULL)
    {
        pOutBuildResultInformation->requiredUi2dConstantBufferSize += GetAlignedBufferSize(pDevice, nn::gfx::GpuAccess_ConstantBuffer, GetVertexShaderConstantBufferSize());
        if (!UseDetailedCombinerCap() && !UseCombinerUserShaderCap())
        {
            pOutBuildResultInformation->requiredUi2dConstantBufferSize += GetAlignedBufferSize(pDevice, nn::gfx::GpuAccess_ConstantBuffer, GetPixelShaderConstantBufferSize());
        }
        else if (UseDetailedCombinerCap())
        {
            pOutBuildResultInformation->requiredUi2dConstantBufferSize += GetAlignedBufferSize(pDevice, nn::gfx::GpuAccess_ConstantBuffer, GetPixelShaderDetailedCombinerConstantBufferSize());
        }
        else
        {
            NN_SDK_ASSERT(UseCombinerUserShaderCap());
            pOutBuildResultInformation->requiredUi2dConstantBufferSize += GetAlignedBufferSize(pDevice, nn::gfx::GpuAccess_ConstantBuffer, GetPixelShaderCombinerUserShaderConstantBufferSize());
        }


        if (m_pUserShaderConstantBufferInformation != NULL &&
            m_pUserShaderConstantBufferInformation->m_ConstantBufferExtendAreaSizeForGeometryShader > 0)
        {
            pOutBuildResultInformation->requiredUi2dConstantBufferSize += GetAlignedBufferSize(pDevice, nn::gfx::GpuAccess_ConstantBuffer, GetGeometryShaderConstantBufferSize());
        }
    }
}

//----------------------------------------
void Material::InitializeBlendInformationImpl(nn::gfx::Device* pDevice)
{
    PresetBlendStateId presetBlendStateId = GetBlendStateId();

    if (presetBlendStateId == PresetBlendStateId_None)
    {
        NN_SDK_ASSERT(m_pBlendState == NULL);

        // プリセットの BlendState でなかった場合は初期化する
        m_pBlendState = Layout::AllocateAndConstruct<nn::gfx::BlendState>();

        const ResBlendMode* pBlendMode = &this->GetBlendMode();
        const ResBlendMode* pBlendModeAlpha = IsBlendModeAlphaCap() ? &this->GetBlendModeAlpha() : pBlendMode;
        nn::gfx::BlendState::InfoType blendStateInfo;
        nn::gfx::BlendTargetStateInfo blendTargetStateInfo;
        size_t size = GraphicsResource::SetupBlendStateInfo(&blendStateInfo, &blendTargetStateInfo, pBlendMode, pBlendModeAlpha);
        m_pBlendState->SetMemory(Layout::AllocateMemory(size, nn::gfx::BlendState::RequiredMemoryInfo_Alignment), size);
        m_pBlendState->Initialize(pDevice, blendStateInfo);
    }
    else
    {
        // プリセットの BlendState だった場合はひとまず NULL を設定し、
        // Calculate のときにプリセットのものを参照する
        m_pBlendState = NULL;
    }
}

//----------------------------------------
void Material::FinalizeBlendInformationImpl(nn::gfx::Device* pDevice)
{
    if (m_pBlendState != NULL)
    {
        if (GetBlendStateId() == PresetBlendStateId_None)
        {
            // プリセットの BlendState でなかった場合は解放する
            void* pBuffer = m_pBlendState->GetMemory();
            m_pBlendState->Finalize(pDevice);
            Layout::FreeMemory(pBuffer);
            Layout::DeleteObj(m_pBlendState);
        }
        m_pBlendState = NULL;
    }
}

//----------------------------------------
PresetBlendStateId Material::GetBlendStateId() const
{
    PresetBlendStateId presetBlendStateId = PresetBlendStateId_None;
    if (IsBlendModeCap())
    {
        const ResBlendMode* pBlendMode = &this->GetBlendMode();
        const ResBlendMode* pBlendModeAlpha = IsBlendModeAlphaCap() ? &this->GetBlendModeAlpha() : pBlendMode;
        presetBlendStateId = GraphicsResource::GetPresetBlendStateId(pBlendMode, pBlendModeAlpha);
    }
    else
    {
        presetBlendStateId = GraphicsResource::GetDefaultPresetBlendStateId();
    }
    return presetBlendStateId;
}

//----------------------------------------
void Material::SetupBlendState(const DrawInfo* pDrawInfo)
{
    if (m_pBlendState != NULL)
    {
        return;
    }

    // BlendState が NULL の場合は GraphicsResource 内の共有のものを使う
    PresetBlendStateId presetBlendStateId = GetBlendStateId();
    NN_SDK_ASSERT(presetBlendStateId != PresetBlendStateId_None);
    m_pBlendState = pDrawInfo->GetGraphicsResource()->GetPresetBlendState(presetBlendStateId);
}

//----------------------------------------
size_t Material::GetVertexShaderConstantBufferSize() const
{
    return sizeof(Material::ConstantBufferForVertexShader) +
            (m_pUserShaderConstantBufferInformation == NULL ?
            0 : m_pUserShaderConstantBufferInformation->m_ConstantBufferExtendAreaSizeForVertexShader);
}

//----------------------------------------
size_t Material::GetGeometryShaderConstantBufferSize() const
{
    return m_pUserShaderConstantBufferInformation != NULL
            ? m_pUserShaderConstantBufferInformation->m_ConstantBufferExtendAreaSizeForGeometryShader
            : 0;
}

//----------------------------------------
size_t Material::GetPixelShaderConstantBufferSize() const
{
    return sizeof(Material::ConstantBufferForPixelShader) +
            (m_pUserShaderConstantBufferInformation == NULL ?
            0 : m_pUserShaderConstantBufferInformation->m_ConstantBufferExtendAreaSizeForPixelShader);
}

//----------------------------------------
size_t Material::GetPixelShaderDetailedCombinerConstantBufferSize() const
{
    return sizeof(Material::ConstantBufferForDetailedCombinerPixelShader);
}

//----------------------------------------
size_t Material::GetPixelShaderCombinerUserShaderConstantBufferSize() const
{
    return sizeof(Material::ConstantBufferForCombinerUserShaderPixelShader);
}

//----------------------------------------
Material::Material(const Material& material, nn::gfx::Device* pDevice)
{
    MaterialCopyContext copyContext;
    copyContext.pDevice = pDevice;
    copyContext.pResAccessor = NULL;
    copyContext.pCaptureTextureCopyInfo = NULL;
    copyContext.pNewRootName = NULL;

    CopyMaterialImpl(material, copyContext);
}


//----------------------------------------
Material::Material(const Material& material, MaterialCopyContext& copyContext)
{
    CopyMaterialImpl(material, copyContext);
}

//----------------------------------------
void Material::CopyMaterialImpl(const Material& material, MaterialCopyContext& copyContext)
{
    Initialize();

    m_pShaderInfo = material.m_pShaderInfo;
    m_AllowableTextureQuantity = material.m_AllowableTextureQuantity;
    m_ShaderId = material.m_ShaderId;
    m_Flag = material.m_Flag;
    m_ShaderVariation = material.m_ShaderVariation;

    SetName(material.GetName());

    // コンスタントバッファのセットアップに必要なため先に
    // ユーザーシェーダーコンスタントバッファ関連のパラメータをコピーする
    CopyUserShaderConstantBufferInformation(material);

    // constant color
    if (detail::TestBit(material.m_Flag, Flags_DynamicAllocatedColorData))
    {
        m_Colors.pFloatColor = static_cast<nn::util::Float4*>(Layout::AllocateMemory(sizeof(nn::util::Float4) * 2));

        memcpy(m_Colors.pFloatColor, material.m_Colors.pFloatColor, sizeof(nn::util::Float4) * 2);
    }
    else
    {
        for (int i = 0; i < MatColorMax; ++i)
        {
            m_Colors.byteColor[i] = material.m_Colors.byteColor[i];
        }
    }

    InitializeMatMemCount(&m_MemCap);
    InitializeMatMemCount(&m_MemCount);

    ReserveMem(
        material.m_MemCap.texMap,
        material.m_MemCap.texSrt,
        material.m_MemCap.texCoordGen,
        material.m_MemCap.tevStage,
        material.m_MemCap.alpComp > 0,
        material.m_MemCap.blendMode,
        material.m_MemCap.indirectParameter > 0,
        material.m_MemCap.projectionTexGen,
        material.m_MemCap.fontShadowParameter,
        material.m_MemCap.detailedCombinerParameter > 0,
        material.m_MemCap.combinerUserShaderParameter > 0
        );

    // TextureOnlyフラグはm_Flagのコピーでコピーされる

    if (m_pMem)
    {
        SetTexMapCount(material.m_MemCount.texMap);
        SetTexSrtCount(material.m_MemCount.texSrt);
        SetTexCoordGenCount(material.m_MemCount.texCoordGen);
        SetTevStageCount(material.m_MemCount.tevStage);
        SetProjectionTexGenCount(material.m_MemCount.projectionTexGen);
        SetFontShadowParameterCount(material.m_MemCount.fontShadowParameter);

        size_t copySize = CalculateReserveMemSize(m_MemCount);
        std::memcpy(m_pMem, material.m_pMem, copySize);
    }

    // キャプチャテクスチャの参照の付け替え処理
    // キャプチャテクスチャはレイアウトインスタンスに属しているためそのままコピーするとコピー元のレイアウトが削除されたときに不正な参照となる。
    // レイアウトインスタンスコピー時にコピー先に新しい名前でキャプチャテクスチャが作成されるためそのテクスチャを参照するように TexMap を変更する。
    if (copyContext.pCaptureTextureCopyInfo != NULL &&
        copyContext.pDevice != NULL &&
        copyContext.pResAccessor != NULL &&
        copyContext.pNewRootName != NULL)
    {
        int texMapNum = GetTexMapCount();

        if (texMapNum > 0)
        {
            TexMap *const pTexMapArray = GetTexMapArray();

            uint8_t  di = 0;
            for (uint8_t si = 0; si < texMapNum; ++si)
            {
                if (detail::TestBit(copyContext.pCaptureTextureCopyInfo->useFlags, si))
                {
                    const TextureInfo* texInfo = AcquireCaptureTexture(
                        NULL,
                        0,
                        copyContext.pDevice,
                        copyContext.pResAccessor,
                        copyContext.pNewRootName,
                        copyContext.pCaptureTextureCopyInfo->names[si]);
                    pTexMapArray[di].Set(texInfo);
                }
                ++di;
            }
        }
    }

    InitializeBlendInformationImpl(copyContext.pDevice);
}

//----------------------------------------
void
Material::Initialize()
{
    nn::util::Unorm8x4 interpolateColorBlack = { { 0, 0, 0, 0 } };
    nn::util::Unorm8x4 interpolateColorWhite = { { std::numeric_limits<uint8_t>::max(), std::numeric_limits<uint8_t>::max(), std::numeric_limits<uint8_t>::max(), std::numeric_limits<uint8_t>::max() } };

    InitializeMatMemCount(&m_MemCap);
    InitializeMatMemCount(&m_MemCount);

    m_Colors.byteColor[InterpolateColor_Black] = interpolateColorBlack;
    m_Colors.byteColor[InterpolateColor_White] = interpolateColorWhite;

    m_pName = NULL;
    m_Flag = 0;
    m_pMem = 0;
    m_pShaderInfo = NULL;
    m_ShaderId = static_cast<uint8_t>(ShaderId_Undefined);
    m_pBlendState = NULL;
    m_AllowableTextureQuantity = 0;

    m_VertexShaderConstantBufferOffset = 0;
    m_PixelShaderConstantBufferOffset = 0;

    m_ShaderVariation = 0;

    m_pUserShaderConstantBufferInformation = NULL;
}

//----------------------------------------
Material::~Material()
{
    NN_SDK_ASSERT(m_pMem == 0);
}

//----------------------------------------
void
Material::Finalize(nn::gfx::Device* pDevice)
{
    FinalizeBlendInformationImpl(pDevice);

    m_PixelShaderConstantBufferOffset = 0;
    m_VertexShaderConstantBufferOffset = 0;

    if ( m_pUserShaderConstantBufferInformation != NULL)
    {
        Layout::FreeMemory(m_pUserShaderConstantBufferInformation);
        m_pUserShaderConstantBufferInformation = NULL;
    }

    // テクスチャ
    for (uint32_t  i = 0; i < m_MemCap.texMap; ++i)
    {
        GetTexMapArray()[i].Finalize();
        GetTexMapArray()[i].~TexMap();
    }

    if (m_pMem)
    {
        Layout::FreeMemory(m_pMem);
        m_pMem = 0;
    }

    if (detail::TestBit(m_Flag, Flags_DynamicAllocatedColorData))
    {
        Layout::FreeMemory(m_Colors.pFloatColor);
        m_Colors.pFloatColor = NULL;
    }
}

//----------------------------------------
void
Material::InitializeMatMemCount(detail::MatMemCount* pMatMemCount) const
{
    pMatMemCount->texMap = 0;
    pMatMemCount->texSrt = 0;
    pMatMemCount->texCoordGen = 0;
    pMatMemCount->tevStage = 0;
    pMatMemCount->alpComp = 0;
    pMatMemCount->blendMode = 0;
    pMatMemCount->indirectParameter = 0;
    pMatMemCount->projectionTexGen = 0;
    pMatMemCount->fontShadowParameter = 0;
    pMatMemCount->detailedCombinerParameter = 0;
    pMatMemCount->combinerUserShaderParameter = 0;
}

//----------------------------------------
void
Material::GetShaderBlend(int* pFirstBlend, int* pSecondBlend) const
{
    *pFirstBlend = static_cast<int32_t>(this->GetTevStage(0).GetCombineRgb());
    *pSecondBlend = static_cast<int32_t>(this->GetTevStage(1).GetCombineRgb());
}

//----------------------------------------
void
Material::GetShaderForDetailedCombiner(int* pFirstBlend, int* pSecondBlend) const
{
    int first = 0;
    int second = 0;
    for (int i = 0; i < GetTevStageCount(); i++)
    {
        int rgbSourceCount = GetDetailedCombinerStageAry()[i].GetCombineRgbSourceCount();
        int alphaSourceCount = GetDetailedCombinerStageAry()[i].GetCombineAlphaSourceCount();
        first = std::max(first, rgbSourceCount);
        second = std::max(second, alphaSourceCount);
    }

    *pFirstBlend = ArchiveShaderId_DetailedCombinerBaseIdx + first;
    *pSecondBlend = ArchiveShaderId_DetailedCombinerBaseIdx + second;
}

//----------------------------------------
size_t
Material::CalculateReserveMemSize(
    int texMapNum,
    int texSrtNum,
    int texCoordGenNum,
    int tevStageNum,
    int alpCompNum,
    int blendModeNum,
    int indirectParameterNum,
    int projectionTexGenNum,
    int fontShadowParameterNum,
    int detailedCombinerParameterNum,
    int combinerUserShaderParameterNum
)
{
    return sizeof(TexMap) * texMapNum +
        sizeof(ResTexSrt) * texSrtNum +
        sizeof(ResTexCoordGen) * texCoordGenNum +
        sizeof(ResAlphaCompare) * alpCompNum +
        sizeof(ResBlendMode) * blendModeNum +
        sizeof(ResIndirectParameter) * indirectParameterNum +
        sizeof(ResTevStage) * tevStageNum +
        sizeof(ResProjectionTexGenParameters) * projectionTexGenNum +
        sizeof(ResFontShadowParameter) * fontShadowParameterNum +
        sizeof(ResDetailedCombinerStageInfo) * detailedCombinerParameterNum +
        sizeof(ResDetailedCombinerStage) * detailedCombinerParameterNum * tevStageNum +
        sizeof(ResCombinerUserShader) * combinerUserShaderParameterNum;
}

//----------------------------------------
void
Material::ResParameterToMemory(
    int texMapNum,
    int texSrtNum,
    int texCoordGenNum,
    int tevStageNum,
    int alpCompNum,
    int blendModeNum,
    int indirectParameterNum,
    int projectionTexGenNum,
    int fontShadowParameterNum,
    int combineParameterNum,
    int combinerUserShaderParameterNum
)
{
    m_MemCap.texMap = texMapNum;
    m_MemCap.texSrt = texSrtNum;
    m_MemCap.texCoordGen = texCoordGenNum;
    m_MemCap.tevStage = tevStageNum;
    m_MemCap.alpComp = alpCompNum;
    m_MemCap.blendMode = blendModeNum;
    m_MemCap.indirectParameter = indirectParameterNum;
    m_MemCap.projectionTexGen = projectionTexGenNum;
    m_MemCap.fontShadowParameter = fontShadowParameterNum;
    m_MemCap.detailedCombinerParameter = combineParameterNum;
    m_MemCap.combinerUserShaderParameter = combinerUserShaderParameterNum;

    m_MemCount.texSrt = m_MemCap.texSrt;
    InitializeTexSrt(GetTexSrtArray(), m_MemCount.texSrt);

    m_MemCount.alpComp = m_MemCap.alpComp;
    if (m_MemCount.alpComp)
    {
        *GetAlphaComparePtr() = ResAlphaCompare();
    }

    m_MemCount.blendMode = m_MemCap.blendMode;
    if (m_MemCount.blendMode > 0)
    {
        *GetBlendModePtr() = ResBlendMode();
        if (m_MemCount.blendMode == 2)
        {
            *GetBlendModeAlphaPtr() = ResBlendMode();
        }
    }

    m_MemCount.indirectParameter = m_MemCap.indirectParameter;
    if (m_MemCount.indirectParameter)
    {
        *GetIndirectParameterPtr() = ResIndirectParameter();
    }

    m_MemCount.fontShadowParameter = m_MemCap.fontShadowParameter;
    if (m_MemCount.fontShadowParameter)
    {
        new (GetFontShadowParameterPtr()) ResFontShadowParameter();
    }

    m_MemCount.detailedCombinerParameter = m_MemCap.detailedCombinerParameter;
    if (m_MemCount.detailedCombinerParameter)
    {
        *GetDetailedCombinerStageInfoPtr() = ResDetailedCombinerStageInfo();
        for (int i = 0; i < tevStageNum; i++)
        {
            GetDetailedCombinerStageAry()[i] = ResDetailedCombinerStage();
        }
    }

    m_MemCount.combinerUserShaderParameter = m_MemCap.combinerUserShaderParameter;
    if (m_MemCount.combinerUserShaderParameter)
    {
        *GetCombinerUserShaderPtr() = ResCombinerUserShader();
    }
}

//----------------------------------------
bool
Material::ReserveMem(
    int  texMapNum,
    int  texSrtNum,
    int  texCoordGenNum,
    int  tevStageNum,
    bool allocAlpComp,
    int  blendModeNum,
    bool allocIndirectParameter,
    int  projectionTexGenNum,
    bool fontShadowParameter,
    bool detailedCombinerParameter,
    bool combinerUserShaderParameter
)
{
    bool result = true;

    NN_SDK_ASSERT(texMapNum <= TexMapMax, "out of bounds: texMapNum[%d] <= TexMapMax for Material[%s]", texMapNum, GetName());
    NN_SDK_ASSERT(texSrtNum <= TexMapMax, "out of bounds: texSrtNum[%d] <= TexMapMax for Material[%s]", texSrtNum, GetName());
    NN_SDK_ASSERT(texCoordGenNum <= TexMapMax, "out of bounds: texCoordGenNum[%d] <= TexMapMax for Material[%s]", texCoordGenNum, GetName());
    NN_SDK_ASSERT(tevStageNum <= TevStageMax, "out of bounds: tevStageNum[%d] <= TevStageMax for Material[%s]", tevStageNum, GetName());
    NN_SDK_ASSERT(projectionTexGenNum <= TevStageMax, "out of bounds: projectionTexGenNum[%d] <= TevStageMax for Material[%s]", projectionTexGenNum, GetName());

    const uint32_t  alpCompNum = static_cast<uint32_t >(allocAlpComp ? 1: 0);
    const uint32_t  indirectParameterNum = static_cast<uint32_t >(allocIndirectParameter ? 1: 0);
    const uint32_t  fontShadowParameterNum = static_cast<uint32_t >(fontShadowParameter ? 1 : 0);
    const uint32_t  combineParameterNum = static_cast<uint32_t >(detailedCombinerParameter ? 1 : 0);
    const uint32_t  combinerUserShaderParameterNum = static_cast<uint32_t>(combinerUserShaderParameter ? 1 : 0);

    // いずれかが要求しているサイズのほうが大きい場合
    if ( static_cast<int>(m_MemCap.texMap) < texMapNum
      || static_cast<int>(m_MemCap.texSrt) < texSrtNum
      || static_cast<int>(m_MemCap.texCoordGen) < texCoordGenNum
      || static_cast<int>(m_MemCap.tevStage) < tevStageNum
      || static_cast<uint32_t>(m_MemCap.alpComp) < alpCompNum
      || static_cast<int>(m_MemCap.blendMode) < blendModeNum
      || static_cast<uint32_t>(m_MemCap.indirectParameter) < indirectParameterNum
      || static_cast<int>(m_MemCap.projectionTexGen) < projectionTexGenNum
      || static_cast<uint32_t>(m_MemCap.fontShadowParameter) < fontShadowParameterNum
      || static_cast<uint32_t>(m_MemCap.detailedCombinerParameter) < combineParameterNum
      || static_cast<uint32_t>(m_MemCap.combinerUserShaderParameter) < combinerUserShaderParameterNum
    )
    {
        if (m_pMem)
        {
            Layout::FreeMemory(m_pMem);
            m_pMem = 0;

            InitializeMatMemCount(&m_MemCap);
            InitializeMatMemCount(&m_MemCount);
        }

        m_pMem = Layout::AllocateMemory(
            CalculateReserveMemSize(
                texMapNum,
                texSrtNum,
                texCoordGenNum,
                tevStageNum,
                alpCompNum,
                blendModeNum,
                indirectParameterNum,
                projectionTexGenNum,
                fontShadowParameterNum,
                combineParameterNum,
                combinerUserShaderParameterNum)
            );

        if (m_pMem)
        {
            ResParameterToMemory(
                texMapNum,
                texSrtNum,
                texCoordGenNum,
                tevStageNum,
                alpCompNum,
                blendModeNum,
                indirectParameterNum,
                projectionTexGenNum,
                fontShadowParameterNum,
                combineParameterNum,
                combinerUserShaderParameterNum);
        }
        else
        {
            result = false;
        }
    }

    return result;
}

//----------------------------------------
void
Material::BindAnimation(AnimTransform* pAnimTrans)
{
    pAnimTrans->BindMaterial(this);
}

//----------------------------------------
void
Material::UnbindAnimation(AnimTransform* pAnimTrans)
{
    pAnimTrans->UnbindMaterial(this);
}

//----------------------------------------
int
Material::GetProjectionTexGenParametersIdxFromTexCoordGenIdx(int  texCoordGenIdx) const
{
    int  texMapNum = std::min<int>(std::min<int>(GetTexMapCount(), GetTexCoordGenCount()), TexMapMax);
    int projectionTexIndex = 0;

    for (int  i = 0; i < texMapNum; ++i)
    {
        const ResTexCoordGen& texCoordGen = this->GetTexCoordGen(i);
        if (texCoordGen.IsProjection())
        {
            if (i == texCoordGenIdx) {
                return projectionTexIndex;
            }
            projectionTexIndex++;
        }
    }

    NN_SDK_ASSERT(false, "texCoordGenIdx[%d] must be index of projection maped TEXGENSRC for Material[%s]", texCoordGenIdx, GetName());
    return 0;
}

//----------------------------------------
void
Material::CalculateTextureMtx(float pTexMtx[2][3], const ResTexSrt& texSrt, const TexMap& texMap)
{
    NN_UNUSED(texMap);

    float (*texMtx)[3] = pTexMtx;

    const nn::util::Float2 center = NN_UTIL_FLOAT_2_INITIALIZER(0.5f, 0.5f);

    float sinR, cosR;

    const nn::util::AngleIndex index = nn::util::DegreeToAngleIndex(texSrt.rotate);
    nn::util::SinCosTable(&sinR, &cosR, index);

    float a0, a1;

    a0 = cosR * texSrt.scale.x;
    a1 = -sinR * texSrt.scale.y;

    // Srt の順
    texMtx[0][0] = a0;
    texMtx[0][1] = a1;
    texMtx[0][2] = texSrt.translate.x + center.v[0] + a0 * (-center.v[0]) + a1 * (-center.v[1]);

    a0 = sinR * texSrt.scale.x;
    a1 = cosR * texSrt.scale.y;
    texMtx[1][0] = a0;
    texMtx[1][1] = a1;
    texMtx[1][2] = texSrt.translate.y + center.v[1] + a0 * (-center.v[0]) + a1 * (-center.v[1]);

    // math::MTX23 m(
    //      1,   0,  0,
    //      0, - 1,  1);
    // math::MTX23Mult(&texMtx, &m, &texMtx);
    texMtx[1][0] *= -1.f;
    texMtx[1][1] *= -1.f;
    texMtx[1][2] = texMtx[1][2] * -1.f + 1.0f;
}

//----------------------------------------
void
Material::CalculateIndirectMtx(float pIndirectMtx[2][3], const float rotate, const ResVec2& scale)
{
    NN_SDK_ASSERT_NOT_NULL(pIndirectMtx);

    float (*mtx)[3] = pIndirectMtx;

    const nn::util::Float2 center = NN_UTIL_FLOAT_2_INITIALIZER(0.5f, 0.5f);

    float sinR, cosR;
    const nn::util::AngleIndex index = nn::util::DegreeToAngleIndex(rotate);
    nn::util::SinCosTable(&sinR, &cosR, index);

    float a0, a1;

    // Srt の順
    a0 = cosR * scale.x * 2.0f;
    a1 = -sinR * scale.y * 2.0f;
    mtx[0][0] = a0;
    mtx[0][1] = a1;
    mtx[0][2] = a0 * (-center.v[0]) + a1 * (-center.v[1]);

    a0 = sinR * scale.x * 2.0f;
    a1 = cosR * scale.y * 2.0f;
    mtx[1][0] = a0;
    mtx[1][1] = a1;
    mtx[1][2] = a0 * (-center.v[0]) + a1 * (-center.v[1]);
}

//----------------------------------------
void
Material::SetupGraphics(
    DrawInfo& drawInfo,
    uint8_t  alpha,
    ShaderVariation shaderVariation,
    bool bInitFrameTransform,
    const nn::util::MatrixT4x3fType& paneGlbMtx,
    const nn::ui2d::Size* pPaneSize,
    const ResExtUserData* pExtUserData,
    uint16_t extUserDataCount
)
{
    NN_PERF_AUTO_MEASURE_INDEX_NAME(0, "nn::ui2d::Material::SetupGraphics");

    {
        const bool useTextureOnly = GetTextureOnly();
        if(!useTextureOnly)
        {
            drawInfo.SetAllowableTextureQuantity(std::min(static_cast<int>(m_MemCount.texMap), static_cast<int>(TexMapMax)));
        }

        drawInfo.ResetModelViewMtxLoaded();

        // 頂点カラーを使う必要があるかどうかがアニメーション等により動的に変わるため、
        // 毎フレーム実行して使用するシェーダを切り替える必要がある
        nn::ui2d::ShaderId shaderId = GetShaderId();
        NN_SDK_ASSERT(shaderId != ShaderId_Undefined, "A shaderId is undefined. Set shaderId with nn::ui2d::Material::SetShaderId().");
        if (shaderId == ShaderId_ArchiveShader)
        {
            // アーカイブシェーダ
            int variation = static_cast<int>(shaderVariation);

            // シェーダバリエーションを、頂点あり／なしの基点に揃えてから、頂点あり／なしを再設定する。
            const bool isUse2ForShaderVariationCountPerBlend = detail::TestBit(m_Flag, Flags_Use2ForShaderVariationCountPerBlend);
            const int variationCountPerBlend = isUse2ForShaderVariationCountPerBlend ? 2 : ShaderVariationCountPerBlend;
            m_ShaderVariation = static_cast<uint8_t>(m_ShaderVariation / variationCountPerBlend * variationCountPerBlend + variation);
        }
        else
        {
            // 共通シェーダ
            if (m_pShaderInfo == NULL)
            {
                const ShaderInfo* pShaderInfo = drawInfo.GetGraphicsResource()->GetCommonShaderInfo();
                m_pShaderInfo = pShaderInfo;
            }
            int variation = GetShaderIndex(GetShaderId(), shaderVariation);
            m_ShaderVariation = static_cast<uint8_t>(variation);
        }

        ConstantBufferForVertexShader* pVertexShaderConstantBuffer = GetConstantBufferForVertexShader(drawInfo);
        const float scale = 1.0f / 255.0f; // 頂点カラーを[0, 1]に正規化するためのスケールです。
        switch (shaderVariation)
        {
        case ShaderVariation_WithoutVertexColor:
            {
                const uint8_t max = std::numeric_limits<uint8_t>::max();
                static const nn::util::Unorm8x4 colors[VertexColor_MaxVertexColor] = {
                    { { max, max, max, max } },
                    { { max, max, max, max } },
                    { { max, max, max, max } },
                    { { max, max, max, max } }
                };
                detail::SetupVertexColor(drawInfo, *this, colors);
                // 頂点カラーへ積算するカラーを設定します。
                pVertexShaderConstantBuffer->color[0] = scale * 1.0f;
                pVertexShaderConstantBuffer->color[1] = scale * 1.0f;
                pVertexShaderConstantBuffer->color[2] = scale * 1.0f;
                pVertexShaderConstantBuffer->color[3] = scale * scale * alpha;
            }
            break;
        default:
            {
                // 頂点カラーへ積算するカラーを設定します。
                pVertexShaderConstantBuffer->color[0] = scale * 1.0f;
                pVertexShaderConstantBuffer->color[1] = scale * 1.0f;
                pVertexShaderConstantBuffer->color[2] = scale * 1.0f;
                pVertexShaderConstantBuffer->color[3] = scale * scale * alpha;
            }
            break;
        }
        drawInfo.LoadProjectionMtx(pVertexShaderConstantBuffer->projection);

        if (bInitFrameTransform)
        {
            pVertexShaderConstantBuffer->frameSpec = TextureFlip_None;
        }

        SetupSubmaterialOf_TextureMatrix(drawInfo, paneGlbMtx, pPaneSize);
        SetRcpTexSize(drawInfo);

        if (!useTextureOnly)
        {
            SetupSubmaterialOf_Tev(drawInfo, pExtUserData, extUserDataCount, pPaneSize);
        }
    }
    SetAllowableTextureQuantity(drawInfo);
}

//----------------------------------------
void
Material::SetupSubmaterialOf_TextureMatrix(DrawInfo& drawInfo, const nn::util::MatrixT4x3fType& paneGlbMtx, const nn::ui2d::Size* pPaneSize)
{
    NN_PERF_AUTO_MEASURE_INDEX_NAME(0, "nn::ui2d::Material::SetupSubmaterialOf_TextureMatrix");

    // テクスチャ行列
    if (m_MemCount.texSrt > 0)
    {
        const uint32_t  num = std::min(
            static_cast<int>(std::min(std::min(static_cast<uint32_t>(m_MemCount.texSrt), static_cast<uint32_t>(this->GetTexMapCount())), static_cast<uint32_t>(TexMapMax))),
            drawInfo.GetAllowableTextureQuantity());
        for (uint32_t  i = 0; i < num; ++i)
        {
            const ResTexCoordGen& texCoordGen = this->GetTexCoordGen(i);
            if (!texCoordGen.IsPerspectiveProjection())
            {
                const ResTexSrt& texSrt = this->GetTexSrt(i);
                const TexMap& texMap = this->GetTexMap(i);

                float texMtx[2][3];
                CalculateTextureMtx(texMtx, texSrt, texMap);

                LoadTexCoordMatrix(drawInfo, *this, i, texMtx);
            }
            else
            {
                // 透視投影の場合はテクスチャプロジェクション行列にテクスチャ SRT 行列も含めて実装してあるため、ここでは変換されないパラメータを設定する。
                float texMtx[2][3] =
                {
                    { 1.0f, 0.0f, 0.0f },
                    { 0.0f, -1.0f, 1.0f }
                };

                LoadTexCoordMatrix(drawInfo, *this, i, texMtx);
            }
        }
    }

    if (m_MemCount.texCoordGen > 0)
    {
        uint32_t  i = 0;
        const uint32_t  num = std::min(uint32_t (m_MemCount.texCoordGen), uint32_t (TexMapMax));
        for (; i < num; ++i)
        {
            const ResTexCoordGen& texCoordGen = this->GetTexCoordGen(i);
            drawInfo.SetTexCoordSrc(static_cast<int>(i), texCoordGen.GetTexGenSrc());
        }
        for (; i < TexMapMax; ++i)
        {
            drawInfo.SetTexCoordSrc(static_cast<int>(i), -1);
        }
    }
    else
    {
        drawInfo.SetTexCoordSrc(0, -1);
    }

    if (GetTexMapCap())
    {
        int generatingTexCoord[TexMapMax];

        const int texMapNum = std::min<int>(std::min<int>(GetTexMapCount(), GetTexCoordGenCount()), TexMapMax);
        int projectionTexIndex = 0;
        for (int i = 0; i < texMapNum; ++i)
        {
            const ResTexCoordGen& texCoordGen = this->GetTexCoordGen(i);

            generatingTexCoord[i] = texCoordGen.IsProjection() ? 1 : 0;

            if (texCoordGen.IsProjection())
            {
                NN_SDK_ASSERT(projectionTexIndex < GetProjectionTexGenCount(),
                    "out of bouds: projectionTexIndex[%d] < GetProjectionTexGenCount[%d] for Material[%s]",
                    projectionTexIndex, GetProjectionTexGenCount(), GetName());

                const ResProjectionTexGenParameters& texGen = GetProjectionTexGenArray()[projectionTexIndex];
                ++projectionTexIndex;

                SetupSubmaterialOf_TextureProjectionMatrix(drawInfo, paneGlbMtx, pPaneSize, texGen, i);
            }
        }
        for (int i = texMapNum; i < TexMapMax; ++i) { generatingTexCoord[i] = 0; }

        ConstantBufferForVertexShader* pVertexShaderConstantBuffer = GetConstantBufferForVertexShader(drawInfo);
        pVertexShaderConstantBuffer->generatingTexCoord[0] = generatingTexCoord[0];
        pVertexShaderConstantBuffer->generatingTexCoord[1] = generatingTexCoord[1];
        pVertexShaderConstantBuffer->generatingTexCoord[2] = generatingTexCoord[2];
    }
}// NOLINT(impl/function_size)

//----------------------------------------
void
Material::SetupSubmaterialOf_TextureProjectionMatrix(
    DrawInfo& drawInfo,
    const nn::util::MatrixT4x3fType& paneGlbMtx,
    const nn::ui2d::Size* pPaneSize,
    const ResProjectionTexGenParameters& texGen,
    int textureIndex) const
{
    const bool isFittingLayoutSize = (texGen.flag & (1 << TexProjectionFlag_FittingLayoutSize)) != 0;
    const bool isFittingPaneSizeEnabled = (texGen.flag & (1 << TexProjectionFlag_FittingPaneSizeEnabled)) != 0;
    const bool isAdjustProjectionSREnabled = (texGen.flag & (1 << TexProjectionFlag_AdjustProjectionScaleRotateEnabled)) != 0;

    // 投影範囲となるサイズを求める
    float baseWidth;
    float baseHeight;
    if (isFittingLayoutSize)
    {
        const Size& layoutSize = drawInfo.GetLayout()->GetLayoutSize();
        baseWidth = layoutSize.width;
        baseHeight = layoutSize.height;
    }
    else if (isFittingPaneSizeEnabled)
    {
        NN_SDK_ASSERT(pPaneSize != NULL);
        baseWidth = pPaneSize->width;
        baseHeight = pPaneSize->height;
    }
    else
    {
        const TexMap& texMap = GetTexMap(textureIndex);

        if (texMap.GetTextureInfo() != NULL)
        {
            const TexSize texSize = texMap.GetTextureInfo()->GetSize();

            baseWidth = texSize.width;
            baseHeight = texSize.height;
        }
        else
        {
            baseWidth = 0;
            baseHeight = 0;
        }
    }

    // 投影行列を求める
    const ResTexCoordGen& texCoordGen = GetTexCoordGen(textureIndex);

    nn::util::MatrixT4x4fType texMtx;
    CalculateTextureProjectionMatrix(
        &texMtx, drawInfo, paneGlbMtx,
        baseWidth, baseHeight, texGen, texCoordGen.GetTexGenSrc(), GetTexSrt(textureIndex), isAdjustProjectionSREnabled );

    void* vertexTexCoord = NULL;
    switch (textureIndex)
    {
    case 0: vertexTexCoord = GetConstantBufferForVertexShader(drawInfo)->vertexTexCoord0; break;
    case 1: vertexTexCoord = GetConstantBufferForVertexShader(drawInfo)->vertexTexCoord1; break;
    case 2: vertexTexCoord = GetConstantBufferForVertexShader(drawInfo)->vertexTexCoord2; break;
    default:
        NN_SDK_ASSERT(false, "Wrong texture coord.");
        return;
    }
    nn::util::MatrixStore(static_cast<nn::util::FloatT4x4*>(vertexTexCoord), texMtx);
}

//----------------------------------------
void
Material::SetRcpTexSize(const DrawInfo& drawInfo) const
{
    if (m_MemCount.texMap > 0 && TexMapMax > 0 && drawInfo.GetAllowableTextureQuantity() > 0)
    {
        const TexMap& texMap = this->GetTexMap(0);
        ConstantBufferForVertexShader* pVertexShaderConstantBuffer = GetConstantBufferForVertexShader(drawInfo);
        const TextureInfo* pTextureInfo = texMap.GetTextureInfo();
        if (pTextureInfo != NULL)
        {
            const TexSize texSize = pTextureInfo->GetSize();
            pVertexShaderConstantBuffer->rcpTexSize0[0] = 1.0f / texSize.width;
            pVertexShaderConstantBuffer->rcpTexSize0[1] = 1.0f / texSize.height;
            pVertexShaderConstantBuffer->rcpTexSize0[2] = 0.0f;
            pVertexShaderConstantBuffer->rcpTexSize0[3] = 0.0f;
        }
    }
}

//----------------------------------------
void
Material::SetupSubmaterialOf_Texture(DrawInfo& drawInfo, nn::gfx::CommandBuffer& commandBuffer) const
{
    NN_PERF_AUTO_MEASURE_INDEX_NAME(0, "nn::ui2d::Material::SetupSubmaterialOf_Texture");

    // テクスチャ
    if (m_MemCount.texMap > 0)
    {
        const int maxTexMapCount = static_cast<int>(std::min(static_cast<uint32_t>(m_MemCount.texMap), static_cast<uint32_t>(TexMapMax)));
        const int count = std::min(maxTexMapCount, static_cast<int>(m_AllowableTextureQuantity));

#if NN_GFX_IS_TARGET_GL || (NN_GFX_IS_TARGET_NVN && defined(NN_BUILD_CONFIG_OS_HORIZON)) || NN_GFX_IS_TARGET_VK
        const int textureCount = count;
#elif NN_GFX_IS_TARGET_NVN
        const int textureCount = TexMapMax;
#else
        #error Un konown API Type.
#endif

        for (int  i = 0; i < textureCount; ++i)
        {
#if NN_GFX_IS_TARGET_GL || (NN_GFX_IS_TARGET_NVN && defined(NN_BUILD_CONFIG_OS_HORIZON)) || NN_GFX_IS_TARGET_VK
            const TexMap& texMap = this->GetTexMap(i);
#elif NN_GFX_IS_TARGET_NVN
            // 詳細コンバイナを利用時は全てのスロットに設定を行います。
            int texMapNum = 0;
            if (i < count)
            {
                texMapNum = i;
            }
            else if (!this->UseDetailedCombinerCap())
            {
                break;
            }
            const TexMap& texMap = this->GetTexMap(texMapNum);
#else
            #error Un konown API Type.
#endif

            NN_SDK_ASSERT(texMap.GetTextureInfo()->IsValid(), "texMap.GetTextureObject() must not be invalid for Material[%s]. If you used CaptureTexture, Please call Layout::UpdateCaptureTexture() before Layout::Draw() Call.", GetName());

            const int slotTex = m_pShaderInfo->GetTextureSlot(m_ShaderVariation, i);

            NN_SDK_ASSERT(slotTex >= 0, "Binding texture failed[%s]. Check the number of textures in the layout data and the number of samplers used in the shader.", GetName() );

            nn::gfx::DescriptorSlot samplerDescriptor =
                drawInfo.GetGraphicsResource()->GetSamplerDescriptorSlot(texMap.GetWrapModeS(), texMap.GetWrapModeT(), texMap.GetMinFilter(), texMap.GetMagFilter());

            NN_SDK_ASSERT(texMap.GetTextureInfo()->IsTextureDescriptorSlotReady(),
                "You must call ResourceAccessor::RegisterTextureViewToDescriptorPool() before you use the texture.");

            commandBuffer.SetTextureAndSampler(slotTex, nn::gfx::ShaderStage_Pixel, *texMap.GetTextureInfo()->GetTextureDescriptorSlot(), samplerDescriptor);
        }
    }

    // 現状、フレームバッファテクスチャを利用する機能は
    // コンバイナユーザーシェーダ以外では利用しません。
    if (IsUseFramebufferTexture())
    {
        SetupSubmaterialOf_FramebufferTexture(drawInfo, commandBuffer);
    }
}

//----------------------------------------
bool Material::IsUseFramebufferTexture() const
{
    return m_MemCount.combinerUserShaderParameter == 1;
}

//----------------------------------------
void
Material::SetupSubmaterialOf_FramebufferTexture(DrawInfo& drawInfo, nn::gfx::CommandBuffer& commandBuffer) const
{
    NN_SDK_ASSERT(m_pShaderInfo->GetTextureSlotCount() == CombinerUserShaderTexMapMax);

    // シェーダコード内でフレームバッファスロットが利用されない場合は、最適化によってスロット変数定義が無くなる為 '-1' となる為、
    // テクスチャーとサンプラーをセットするコマンドを設定しません。
    const int slotTex = m_pShaderInfo->GetTextureSlot(m_ShaderVariation, CombinerUserShaderTexMapMax - 1);
    if (slotTex < 0)
    {
        return;
    }

    NN_SDK_ASSERT(drawInfo.GetFramebufferTextureDescriptorSlot() != NULL, "TextureDescriptorSlot for use with CombinerUserShader is not set. Please call GetFramebufferTextureDescriptorSlot() function");
    NN_SDK_ASSERT(drawInfo.GetFramebufferSamplerDescriptorSlot() != NULL, "SamplerDescriptorSlot for use with CombinerUserShader is not set. Please call GetFramebufferSamplerDescriptorSlot() function");

    commandBuffer.SetTextureAndSampler(
        slotTex, nn::gfx::ShaderStage_Pixel,
        *drawInfo.GetFramebufferTextureDescriptorSlot(),
        *drawInfo.GetFramebufferSamplerDescriptorSlot());
}

//----------------------------------------
void
Material::SetupSubmaterialOf_Tev(DrawInfo& drawInfo, const ResExtUserData* pExtUserData, uint16_t extUserDataCount, const nn::ui2d::Size* pPaneSize) const
{
    NN_PERF_AUTO_MEASURE_INDEX_NAME(0, "nn::ui2d::Material::SetupSubmaterialOf_Tev");

    if (!m_MemCount.detailedCombinerParameter &&
        !m_MemCount.combinerUserShaderParameter)
    {
        GraphicsResource& graphicsResource = *drawInfo.GetGraphicsResource();
        ConstantBufferForPixelShader* pPixelShaderConstantBuffer = GetConstantBufferForPixelShader(drawInfo);

        // 白黒補間の値を設定（現状では、TEVの設定に関係なく白黒補間は常に有効）
        if (m_MemCount.texMap == 0) {
            nn::util::Float4  whiteAsFloat;
            GetColorWithFloatConversion(whiteAsFloat, InterpolateColor_White);

            NN_UNUSED(graphicsResource);

            pPixelShaderConstantBuffer->interpolateWidth[0] = whiteAsFloat.v[0];
            pPixelShaderConstantBuffer->interpolateWidth[1] = whiteAsFloat.v[1];
            pPixelShaderConstantBuffer->interpolateWidth[2] = whiteAsFloat.v[2];
            pPixelShaderConstantBuffer->interpolateWidth[3] = whiteAsFloat.v[3];

            pPixelShaderConstantBuffer->interpolateOffset[0] = 0.0f;
            pPixelShaderConstantBuffer->interpolateOffset[1] = 0.0f;
            pPixelShaderConstantBuffer->interpolateOffset[2] = 0.0f;
            pPixelShaderConstantBuffer->interpolateOffset[3] = 0.0f;
        }
        else {
            nn::util::Float4  whiteAsFloat;
            nn::util::Float4  blackAsFloat;
            GetColorWithFloatConversion(whiteAsFloat, InterpolateColor_White);
            GetColorWithFloatConversion(blackAsFloat, InterpolateColor_Black);

            pPixelShaderConstantBuffer->interpolateWidth[0] = whiteAsFloat.v[0] - blackAsFloat.v[0];
            pPixelShaderConstantBuffer->interpolateWidth[1] = whiteAsFloat.v[1] - blackAsFloat.v[1];
            pPixelShaderConstantBuffer->interpolateWidth[2] = whiteAsFloat.v[2] - blackAsFloat.v[2];
            pPixelShaderConstantBuffer->interpolateOffset[0] = blackAsFloat.v[0];
            pPixelShaderConstantBuffer->interpolateOffset[1] = blackAsFloat.v[1];
            pPixelShaderConstantBuffer->interpolateOffset[2] = blackAsFloat.v[2];
            if (GetThresholdingAlphaInterpolation())
            {
                const float a = 1.0f / (whiteAsFloat.v[3] - blackAsFloat.v[3] + 0.0001f); // 傾きaを求めます。発散を防ぐために微小な値を足しています。
                pPixelShaderConstantBuffer->interpolateWidth[3] = a;
                pPixelShaderConstantBuffer->interpolateOffset[3] = -blackAsFloat.v[3] * a;
            }
            else
            {
                pPixelShaderConstantBuffer->interpolateWidth[3] = whiteAsFloat.v[3] - blackAsFloat.v[3];
                pPixelShaderConstantBuffer->interpolateOffset[3] = blackAsFloat.v[3];
            }
        }

        //  TEV
        if (m_MemCount.tevStage > 0)
        {
            if (m_MemCount.texMap >= 2)
            {
                // アルファモードを pPixelShaderConstantBuffer->textureMode に設定します。
                {
                    int textureMode = 0;
                    for (int i = 0; i < this->GetTevStageCount(); ++i)
                    {
                        const ResTevStage& tevStage = this->GetTevStage(i);
                        if (tevStage.GetCombineAlpha() == TevMode_Modulate)
                        {
                            // TEV設定のステージ1のアルファステージのコンバインモードがTEVMODE_MODULATE
                            // ならアルファは最小値を取る、それ以外なら最大値を取る
                            const int TEXTURE_MODE_BITS = 8;
                            textureMode |= 1 << (i * TEXTURE_MODE_BITS + 7);
                        }
                    }

                    pPixelShaderConstantBuffer->textureMode = textureMode;
                }

                if (IsIndirectParameterCap())
                {
                    const ResIndirectParameter& param = GetIndirectParameter();

                    float mtx[2][3];
                    CalculateIndirectMtx(mtx, param.rotate, param.scale);

                    pPixelShaderConstantBuffer->indirectMtx0[0] = mtx[0][0];
                    pPixelShaderConstantBuffer->indirectMtx0[1] = mtx[0][1];
                    pPixelShaderConstantBuffer->indirectMtx0[2] = 0.0f;
                    pPixelShaderConstantBuffer->indirectMtx0[3] = mtx[0][2];

                    pPixelShaderConstantBuffer->indirectMtx1[0] = mtx[1][0];
                    pPixelShaderConstantBuffer->indirectMtx1[1] = mtx[1][1];
                    pPixelShaderConstantBuffer->indirectMtx1[2] = 0.0f;
                    pPixelShaderConstantBuffer->indirectMtx1[3] = mtx[1][2];
                }
            }
            else
            {
                pPixelShaderConstantBuffer->textureMode = m_MemCount.texMap;
            }
        }
    }
    else if (m_MemCount.detailedCombinerParameter)
    {
        SetupSubmaterialOf_DetailedCombiner(drawInfo);
    }
    else
    {
        SetupSubmaterialOf_CombinerUserShader(drawInfo, pExtUserData, extUserDataCount, pPaneSize);
    }

}// NOLINT(impl/function_size)

void
Material::SetupSubmaterialOf_DetailedCombiner(DrawInfo& drawInfo) const
{
    ConstantBufferForDetailedCombinerPixelShader* pDetailedCombinerConstantBuffer = GetConstantBufferForDetailedCombinerPixelShader(drawInfo);
    NN_SDK_ASSERT(pDetailedCombinerConstantBuffer != NULL, "Invalid ConstantBufferForDetailedCombinerPixelShader for Material[%s]", GetName());

    const bool useDetailedCombinerWithVariationVariable = detail::TestBit(m_Flag, Flags_UseDetailedCombinerWithVariationVariable);
    if (!useDetailedCombinerWithVariationVariable)
    {
        pDetailedCombinerConstantBuffer->stageCountMax = m_MemCount.tevStage;

        for (int i = 0; i < pDetailedCombinerConstantBuffer->stageCountMax; i++)
        {
            pDetailedCombinerConstantBuffer->stage_bit[i][0] = GetDetailedCombinerStageAry()[i].bits0;
            pDetailedCombinerConstantBuffer->stage_bit[i][1] = GetDetailedCombinerStageAry()[i].bits1;
            pDetailedCombinerConstantBuffer->stage_bit[i][2] = GetDetailedCombinerStageAry()[i].bits2;
            pDetailedCombinerConstantBuffer->stage_bit[i][3] = GetDetailedCombinerStageAry()[i].bits3;
        }
    }

    // 最適化によるシンボル除去を防ぐために固定値を利用します。
    pDetailedCombinerConstantBuffer->padding1[0] = 0.0f;

    nn::util::Float4 blackAsFloat;
    nn::util::Float4 whiteAsFloat;
    GetColorWithFloatConversion(blackAsFloat, InterpolateColor_Black);
    GetColorWithFloatConversion(whiteAsFloat, InterpolateColor_White);

    memcpy(pDetailedCombinerConstantBuffer->constantColor[0], &blackAsFloat, sizeof(float) * 4);
    memcpy(pDetailedCombinerConstantBuffer->constantColor[1], &whiteAsFloat, sizeof(float) * 4);

    for (int i = 0; i < DetailedCombinerConstantColor; i++)
    {
        const nn::util::Unorm8x4& constantColor = GetDetailedCombinerStageInfoPtr()->GetConstantColor(i);
        pDetailedCombinerConstantBuffer->constantColor[2 + i][0] = static_cast<float>(constantColor.v[0]) / static_cast<float>(std::numeric_limits<uint8_t>::max());
        pDetailedCombinerConstantBuffer->constantColor[2 + i][1] = static_cast<float>(constantColor.v[1]) / static_cast<float>(std::numeric_limits<uint8_t>::max());
        pDetailedCombinerConstantBuffer->constantColor[2 + i][2] = static_cast<float>(constantColor.v[2]) / static_cast<float>(std::numeric_limits<uint8_t>::max());
        pDetailedCombinerConstantBuffer->constantColor[2 + i][3] = static_cast<float>(constantColor.v[3]) / static_cast<float>(std::numeric_limits<uint8_t>::max());
    }
}

void
Material::SetupConstantBufferColor_for_CombinerUserShader(ConstantBufferForCombinerUserShaderPixelShader* pCombinerUserShaderConstantBuffer) const
{
    NN_SDK_ASSERT_NOT_NULL(pCombinerUserShaderConstantBuffer);

    nn::util::Float4 blackAsFloat;
    nn::util::Float4 whiteAsFloat;
    GetColorWithFloatConversion(blackAsFloat, InterpolateColor_Black);
    GetColorWithFloatConversion(whiteAsFloat, InterpolateColor_White);

    memcpy(pCombinerUserShaderConstantBuffer->constantColor[0], &blackAsFloat, sizeof(float) * 4);
    memcpy(pCombinerUserShaderConstantBuffer->constantColor[1], &whiteAsFloat, sizeof(float) * 4);

    for (int i = 0; i < DetailedCombinerConstantColor; i++)
    {
        const nn::util::Unorm8x4& constantColor = GetCombinerUserShaderPtr()->GetConstantColor(i);
        pCombinerUserShaderConstantBuffer->constantColor[2 + i][0] = static_cast<float>(constantColor.v[0]) / static_cast<float>(std::numeric_limits<uint8_t>::max());
        pCombinerUserShaderConstantBuffer->constantColor[2 + i][1] = static_cast<float>(constantColor.v[1]) / static_cast<float>(std::numeric_limits<uint8_t>::max());
        pCombinerUserShaderConstantBuffer->constantColor[2 + i][2] = static_cast<float>(constantColor.v[2]) / static_cast<float>(std::numeric_limits<uint8_t>::max());
        pCombinerUserShaderConstantBuffer->constantColor[2 + i][3] = static_cast<float>(constantColor.v[3]) / static_cast<float>(std::numeric_limits<uint8_t>::max());
    }
}

void
Material::SetupConstantBufferPosture_for_CombinerUserShader(ConstantBufferForCombinerUserShaderPixelShader* pCombinerUserShaderConstantBuffer, DrawInfo& drawInfo) const
{
    NN_SDK_ASSERT_NOT_NULL(pCombinerUserShaderConstantBuffer);

    const nn::util::MatrixT4x3fType& modelViewMatrix = drawInfo.GetModelViewMtx();
    const nn::util::MatrixT4x3fType& viewMatrix = drawInfo.GetViewMtx();
    nn::util::MatrixT4x3fType mViewInv;
    nn::util::MatrixInverse(&mViewInv, viewMatrix);
    nn::util::MatrixT4x3fType m;
    nn::util::MatrixMultiply(&m, modelViewMatrix, mViewInv);
    nn::util::Vector3fType cameraPosition;
    nn::util::MatrixGetAxisW(&cameraPosition, mViewInv);

    nn::util::MatrixStore(reinterpret_cast<nn::util::FloatT4x3*>(pCombinerUserShaderConstantBuffer->modelViewMatrix), modelViewMatrix);
    nn::util::MatrixStore(reinterpret_cast<nn::util::FloatT4x3*>(pCombinerUserShaderConstantBuffer->modelMatrix), m);
    nn::util::MatrixStore(reinterpret_cast<nn::util::FloatT4x3*>(pCombinerUserShaderConstantBuffer->viewMatrix), viewMatrix);
    nn::util::VectorStore(reinterpret_cast<nn::util::Float3*>(pCombinerUserShaderConstantBuffer->cameraPosition), cameraPosition);
}

void
Material::SetupConstantBufferExData_for_CombinerUserShader(
    ConstantBufferForCombinerUserShaderPixelShader* pCombinerUserShaderConstantBuffer,
    const ResExtUserData* pExtUserData,
    uint16_t extUserDataCount) const
{
    NN_SDK_ASSERT_NOT_NULL(pCombinerUserShaderConstantBuffer);

    if (pExtUserData == NULL || extUserDataCount <= 0)
    {
        return;
    }

    for (int i = 0; i < extUserDataCount; ++i, ++pExtUserData)
    {
        for(int t = 0; t < NN_ARRAY_SIZE(pCombinerUserShaderExtUserData_Vec2); t++)
        {
            if (0 == std::strcmp(pCombinerUserShaderExtUserData_Vec2[t], pExtUserData->GetName()))
            {
                pCombinerUserShaderConstantBuffer->userData_vec2[t][0] = pExtUserData->GetFloatArray()[0];
                pCombinerUserShaderConstantBuffer->userData_vec2[t][1] = pExtUserData->GetFloatArray()[1];
            }
        }

        for(int t = 0; t < NN_ARRAY_SIZE(pCombinerUserShaderExtUserData_Vec3); t++)
        {
            if (0 == std::strcmp(pCombinerUserShaderExtUserData_Vec3[t], pExtUserData->GetName()))
            {
                pCombinerUserShaderConstantBuffer->userData_vec3[t][0] = pExtUserData->GetFloatArray()[0];
                pCombinerUserShaderConstantBuffer->userData_vec3[t][1] = pExtUserData->GetFloatArray()[1];
                pCombinerUserShaderConstantBuffer->userData_vec3[t][2] = pExtUserData->GetFloatArray()[2];
            }
        }

        for(int t = 0; t < NN_ARRAY_SIZE(pCombinerUserShaderExtUserData_Rgba); t++)
        {
            if (0 == std::strcmp(pCombinerUserShaderExtUserData_Rgba[t], pExtUserData->GetName()))
            {
                pCombinerUserShaderConstantBuffer->userData_rgba[t][0] = pExtUserData->GetIntArray()[0];
                pCombinerUserShaderConstantBuffer->userData_rgba[t][1] = pExtUserData->GetIntArray()[1];
                pCombinerUserShaderConstantBuffer->userData_rgba[t][2] = pExtUserData->GetIntArray()[2];
                pCombinerUserShaderConstantBuffer->userData_rgba[t][3] = pExtUserData->GetIntArray()[3];
            }
        }

        for(int t = 0; t < NN_ARRAY_SIZE(pCombinerUserShaderExtUserData_IVec2); t++)
        {
            if (0 == std::strcmp(pCombinerUserShaderExtUserData_IVec2[t], pExtUserData->GetName()))
            {
                pCombinerUserShaderConstantBuffer->userData_ivec2[t][0] = pExtUserData->GetIntArray()[0];
                pCombinerUserShaderConstantBuffer->userData_ivec2[t][1] = pExtUserData->GetIntArray()[1];
            }
        }

        for(int t = 0; t < NN_ARRAY_SIZE(pCombinerUserShaderExtUserData_Int); t++)
        {
            if (0 == std::strcmp(pCombinerUserShaderExtUserData_Int[t], pExtUserData->GetName()))
            {
                pCombinerUserShaderConstantBuffer->userData_int[t] = pExtUserData->GetIntArray()[0];
            }
        }

        for(int t = 0; t < NN_ARRAY_SIZE(pCombinerUserShaderExtUserData_Float); t++)
        {
            if (0 == std::strcmp(pCombinerUserShaderExtUserData_Float[t], pExtUserData->GetName()))
            {
                pCombinerUserShaderConstantBuffer->userData_float[t] = pExtUserData->GetFloatArray()[0];
            }
        }
    }
}
void
Material::SetupConstantBufferTextureData_for_CombinerUserShader(ConstantBufferForCombinerUserShaderPixelShader* pCombinerUserShaderConstantBuffer) const
{
    NN_STATIC_ASSERT(TexMapMax == 3); // glsl 内のコンスタントバッファに定義されている配列の数 '3' と合わせる事
    NN_SDK_ASSERT_NOT_NULL(pCombinerUserShaderConstantBuffer);

    for (int i = 0; i < GetTexMapCount(); i++)
    {
        const TexMap& texMap = GetTexMap(i);

        if (texMap.GetTextureInfo() != NULL)
        {
            const TexSize texSize = texMap.GetTextureInfo()->GetSize();

            pCombinerUserShaderConstantBuffer->textureSize[i][0] = texSize.width;
            pCombinerUserShaderConstantBuffer->textureSize[i][1] = texSize.height;
        }
    }
}

void
Material::SetupSubmaterialOf_CombinerUserShader(DrawInfo& drawInfo, const ResExtUserData* pExtUserData, uint16_t extUserDataCount, const nn::ui2d::Size* pPaneSize) const
{
    NN_SDK_ASSERT(m_MemCount.combinerUserShaderParameter == 1);

    ConstantBufferForCombinerUserShaderPixelShader* pCombinerUserShaderConstantBuffer = GetConstantBufferForCombinerUserShaderPixelShader(drawInfo);
    NN_SDK_ASSERT(pCombinerUserShaderConstantBuffer != NULL, "Invalid ConstantBufferForCombinerUserShaderPixelShader for Material[%s]", GetName());
    memset(pCombinerUserShaderConstantBuffer, 0, sizeof(ConstantBufferForCombinerUserShaderPixelShader));

    // 最適化によるシンボル除去を防ぐために固定値を利用します。
    pCombinerUserShaderConstantBuffer->fixedZero = 0.0f;

    // ペインのサイズ
    pCombinerUserShaderConstantBuffer->paneWidth = pPaneSize->width;
    pCombinerUserShaderConstantBuffer->paneHeight = pPaneSize->height;

    // カラー設定
    SetupConstantBufferColor_for_CombinerUserShader(pCombinerUserShaderConstantBuffer);

    // ペインの姿勢や各マトリクス情報をコンスタントバッファに用意します。
    SetupConstantBufferPosture_for_CombinerUserShader(pCombinerUserShaderConstantBuffer, drawInfo);

    // 拡張ユーザーデータ内のコンバイナユーザーシェーダ用の変数を更新します。
    SetupConstantBufferExData_for_CombinerUserShader(pCombinerUserShaderConstantBuffer, pExtUserData, extUserDataCount);

    // テクスチャ情報をコンスタントバッファに用意
    SetupConstantBufferTextureData_for_CombinerUserShader(pCombinerUserShaderConstantBuffer);
}

//----------------------------------------
void Material::AllocateConstantBuffer(DrawInfo& drawInfo)
{
    NN_SDK_ASSERT_NOT_NULL(drawInfo.GetGraphicsResource());
    // １フレームの描画に使用するコンスタントバッファの領域を確定します。
    // このメソッドの呼び出し後であれば ConsantBuffer の領域へアクセスすることが出来ます。
    // アロケートした領域(オフセット値)はリセットされないため次にこのメソッドが呼び出されるまで値が保持されます。
    size_t  constantBufferAlignment = drawInfo.GetGraphicsResource()->GetConstantBufferAlignment();

    size_t  offset = drawInfo.GetUi2dConstantBuffer()->Allocate(nn::util::align_up(GetVertexShaderConstantBufferSize(), constantBufferAlignment));
    NN_SDK_ASSERT(offset <= size_t(0xFFFFFFFF), "offset is over the limit for Material[%s]", GetName());
    m_VertexShaderConstantBufferOffset = static_cast<uint32_t>(offset);

    if (!UseDetailedCombinerCap() &&
        !UseCombinerUserShaderCap())
    {
        offset = drawInfo.GetUi2dConstantBuffer()->Allocate(nn::util::align_up(GetPixelShaderConstantBufferSize(), constantBufferAlignment));
        NN_SDK_ASSERT(offset <= size_t(0xFFFFFFFF), "offset is over the limit for Material[%s]", GetName());
        m_PixelShaderConstantBufferOffset = static_cast<uint32_t>(offset);
    }
    else if (UseDetailedCombinerCap())
    {
        offset = drawInfo.GetUi2dConstantBuffer()->Allocate(nn::util::align_up(GetPixelShaderDetailedCombinerConstantBufferSize(), constantBufferAlignment));
        NN_SDK_ASSERT(offset <= size_t(0xFFFFFFFF), "offset is over the limit for Material[%s]", GetName());
        m_PixelShaderConstantBufferOffset = static_cast<uint32_t>(offset);
    }
    else
    {
        NN_SDK_ASSERT(UseCombinerUserShaderCap());
        offset = drawInfo.GetUi2dConstantBuffer()->Allocate(nn::util::align_up(GetPixelShaderCombinerUserShaderConstantBufferSize(), constantBufferAlignment));
        NN_SDK_ASSERT(offset <= size_t(0xFFFFFFFF), "offset is over the limit for Material[%s]", GetName());
        m_PixelShaderConstantBufferOffset = static_cast<uint32_t>(offset);
    }
    size_t geometryShaderConstantBufferSize = GetGeometryShaderConstantBufferSize();
    if (geometryShaderConstantBufferSize > 0 )
    {
        NN_SDK_ASSERT_NOT_NULL(m_pUserShaderConstantBufferInformation);
        offset = drawInfo.GetUi2dConstantBuffer()->Allocate(nn::util::align_up(geometryShaderConstantBufferSize, constantBufferAlignment));
        NN_SDK_ASSERT(offset <= size_t(0xFFFFFFFF), "offset is over the limit for Material[%s]", GetName());
        m_pUserShaderConstantBufferInformation->m_GeometryShaderConstantBufferOffset = static_cast<uint32_t>(offset);
    }
}

//----------------------------------------
Material::ConstantBufferForVertexShader* Material::GetConstantBufferForVertexShader(const DrawInfo& drawInfo) const
{
    if (drawInfo.GetUi2dConstantBuffer()->GetMappedPointer() != NULL)
    {
        nn::util::BytePtr   pConstantBufferForVertexShader(drawInfo.GetUi2dConstantBuffer()->GetMappedPointer());
        return static_cast<ConstantBufferForVertexShader*>(pConstantBufferForVertexShader.Advance(m_VertexShaderConstantBufferOffset).Get());
    }

    return NULL;
}

//----------------------------------------
Material::ConstantBufferForPixelShader* Material::GetConstantBufferForPixelShader(const DrawInfo& drawInfo) const
{
    if (drawInfo.GetUi2dConstantBuffer()->GetMappedPointer() != NULL)
    {
        nn::util::BytePtr   pConstantBufferForPixelShader(drawInfo.GetUi2dConstantBuffer()->GetMappedPointer());
        return static_cast<ConstantBufferForPixelShader*>(pConstantBufferForPixelShader.Advance(m_PixelShaderConstantBufferOffset).Get());
    }

    return NULL;
}

//----------------------------------------
Material::ConstantBufferForDetailedCombinerPixelShader* Material::GetConstantBufferForDetailedCombinerPixelShader(const DrawInfo& drawInfo) const
{
    return reinterpret_cast<ConstantBufferForDetailedCombinerPixelShader*>(GetConstantBufferForPixelShader(drawInfo));
}

//----------------------------------------
Material::ConstantBufferForCombinerUserShaderPixelShader* Material::GetConstantBufferForCombinerUserShaderPixelShader(const DrawInfo& drawInfo) const
{
    return reinterpret_cast<ConstantBufferForCombinerUserShaderPixelShader*>(GetConstantBufferForPixelShader(drawInfo));
}
//----------------------------------------
void* Material::GetConstantBufferForUserVertexShader(const DrawInfo& drawInfo) const
{
    if (drawInfo.GetUi2dConstantBuffer()->GetMappedPointer() != NULL &&
        m_pUserShaderConstantBufferInformation != NULL &&
        m_pUserShaderConstantBufferInformation->m_ConstantBufferExtendAreaSizeForVertexShader > 0)
    {
        nn::util::BytePtr   pConstantBufferForUserVertexShader(drawInfo.GetUi2dConstantBuffer()->GetMappedPointer(),
                                                m_VertexShaderConstantBufferOffset + sizeof(ConstantBufferForVertexShader));
        return pConstantBufferForUserVertexShader.Get();
    }

    return NULL;
}

//----------------------------------------
void* Material::GetConstantBufferForUserPixelShader(const DrawInfo& drawInfo) const
{
    if (drawInfo.GetUi2dConstantBuffer()->GetMappedPointer() != NULL &&
        m_pUserShaderConstantBufferInformation != NULL &&
        m_pUserShaderConstantBufferInformation->m_ConstantBufferExtendAreaSizeForPixelShader > 0)
    {
        nn::util::BytePtr   pConstantBufferForUserPixelShader(drawInfo.GetUi2dConstantBuffer()->GetMappedPointer(),
                                                m_PixelShaderConstantBufferOffset + sizeof(ConstantBufferForPixelShader));
        return pConstantBufferForUserPixelShader.Get();
    }

    return NULL;
}

//----------------------------------------
void* Material::GetConstantBufferForUserGeometryShader(const DrawInfo& drawInfo) const
{
    if (drawInfo.GetUi2dConstantBuffer()->GetMappedPointer() != NULL &&
        m_pUserShaderConstantBufferInformation != NULL &&
        m_pUserShaderConstantBufferInformation->m_ConstantBufferExtendAreaSizeForGeometryShader > 0 )
    {
        nn::util::BytePtr   pConstantBufferForUserGeometryShader(drawInfo.GetUi2dConstantBuffer()->GetMappedPointer(),
                                                m_pUserShaderConstantBufferInformation->m_GeometryShaderConstantBufferOffset);
        return pConstantBufferForUserGeometryShader.Get();
    }

    return NULL;
}

//----------------------------------------
void Material::SetCommandBuffer(nn::gfx::CommandBuffer& commandBuffer, DrawInfo& drawInfo) const
{
    NN_SDK_ASSERT_NOT_NULL(drawInfo.GetUi2dConstantBuffer());
    NN_PERF_AUTO_MEASURE_INDEX_NAME(0, "nn::ui2d::Material::SetCommandBuffer");

    if (IsAlphaCompareCap())
    {
        // アルファテストを設定して描画する
        EnableAlphaTest(commandBuffer);
        commandBuffer.SetBlendState(m_pBlendState);
        DisableAlphaTest();
    }
    else
    {
        commandBuffer.SetBlendState(m_pBlendState);
    }

    ApplyVertexShaderConstantBuffer(commandBuffer, drawInfo);
    ApplyGeometryShaderConstantBuffer(commandBuffer, drawInfo);
    ApplyPixelShaderConstantBuffer(commandBuffer, drawInfo);
}

//----------------------------------------
void Material::ApplyVertexShaderConstantBuffer(nn::gfx::CommandBuffer& commandBuffer, DrawInfo& drawInfo) const
{
    nn::gfx::GpuAddress gpuAddr;
    gpuAddr = *drawInfo.GetUi2dConstantBuffer()->GetGpuAddress();
    gpuAddr.Offset(m_VertexShaderConstantBufferOffset);

    size_t vertexShaderConstantBufferSize = sizeof(Material::ConstantBufferForVertexShader);

    if (m_pUserShaderConstantBufferInformation != NULL)
    {
        vertexShaderConstantBufferSize += m_pUserShaderConstantBufferInformation->m_ConstantBufferExtendAreaSizeForVertexShader;
    }

    NN_SDK_ASSERT(m_pShaderInfo->GetVertexShaderSlot(m_ShaderVariation) >= 0, "An invalid shader-slot(for uConstantBufferForVertexShader)was detected at [m_ShaderVariation=%d]! Are you sure that you intend to use vertex-shader ? ", m_ShaderVariation);

    commandBuffer.SetConstantBuffer(m_pShaderInfo->GetVertexShaderSlot(m_ShaderVariation), nn::gfx::ShaderStage_Vertex, gpuAddr, vertexShaderConstantBufferSize);
}

//----------------------------------------
void Material::ApplyGeometryShaderConstantBuffer(nn::gfx::CommandBuffer& commandBuffer, DrawInfo& drawInfo) const
{
    if (m_pUserShaderConstantBufferInformation != NULL &&
        m_pUserShaderConstantBufferInformation->m_GeometryShaderConstantBufferOffset != 0)
    {
        nn::gfx::GpuAddress gpuAddr;
        gpuAddr = *drawInfo.GetUi2dConstantBuffer()->GetGpuAddress();
        gpuAddr.Offset(m_pUserShaderConstantBufferInformation->m_GeometryShaderConstantBufferOffset);

        NN_SDK_ASSERT(m_pShaderInfo->GetGeometryShaderSlot(m_ShaderVariation) >= 0, "An invalid shader-slot(for uConstantBufferForGeometryShader) was detected at [m_ShaderVariation=%d]! Are you sure that you intend to use geometry-shader ?", m_ShaderVariation);

        commandBuffer.SetConstantBuffer(m_pShaderInfo->GetGeometryShaderSlot(m_ShaderVariation), nn::gfx::ShaderStage_Geometry, gpuAddr, m_pUserShaderConstantBufferInformation->m_ConstantBufferExtendAreaSizeForGeometryShader);
    }
}

//----------------------------------------
void Material::ApplyPixelShaderConstantBufferDefault(nn::gfx::CommandBuffer& commandBuffer, DrawInfo& drawInfo) const
{
    nn::gfx::GpuAddress gpuAddr;
    gpuAddr = *drawInfo.GetUi2dConstantBuffer()->GetGpuAddress();
    gpuAddr.Offset(m_PixelShaderConstantBufferOffset);

    size_t pixelShaderConstantBufferSize = sizeof(Material::ConstantBufferForPixelShader);

    if (m_pUserShaderConstantBufferInformation != NULL)
    {
        pixelShaderConstantBufferSize += m_pUserShaderConstantBufferInformation->m_ConstantBufferExtendAreaSizeForPixelShader;
    }

    NN_SDK_ASSERT(m_pShaderInfo->GetPixelShaderSlot(m_ShaderVariation) >= 0, "An invalid shader-slot(for uConstantBufferForPixelShader) was detected at [m_ShaderVariation=%d]!  Are you sure that you intend to use pixel-shader ?", m_ShaderVariation);

    commandBuffer.SetConstantBuffer(m_pShaderInfo->GetPixelShaderSlot(m_ShaderVariation), nn::gfx::ShaderStage_Pixel, gpuAddr, pixelShaderConstantBufferSize);
}

//----------------------------------------
void Material::ApplyPixelShaderConstantBuffer(nn::gfx::CommandBuffer& commandBuffer, DrawInfo& drawInfo) const
{
    if (!UseDetailedCombinerCap() &&
        !UseCombinerUserShaderCap())
    {
        ApplyPixelShaderConstantBufferDefault(commandBuffer, drawInfo);
    }
    else if (UseDetailedCombinerCap())
    {
        ApplyPixelShaderDetailedCombinerConstantBuffer(commandBuffer, drawInfo);
    }
    else
    {
        NN_SDK_ASSERT(UseCombinerUserShaderCap());
        ApplyPixelShaderCombinerUserShaderConstantBuffer(commandBuffer, drawInfo);
    }
}

//----------------------------------------
void Material::ApplyPixelShaderDetailedCombinerConstantBuffer(nn::gfx::CommandBuffer& commandBuffer, DrawInfo& drawInfo) const
{
    nn::gfx::GpuAddress gpuAddr;
    gpuAddr = *drawInfo.GetUi2dConstantBuffer()->GetGpuAddress();
    gpuAddr.Offset(m_PixelShaderConstantBufferOffset);

    size_t pixelShaderCombinerConstantBufferSize = sizeof(Material::ConstantBufferForDetailedCombinerPixelShader);

    NN_SDK_ASSERT(m_pShaderInfo->GetPixelShaderSlot(m_ShaderVariation) >= 0, "An invalid shader-slot(for uConstantBufferForPixelShader) was detectedat at [m_ShaderVariation=%d]!  Are you sure that you intend to use pixel-shader ?", m_ShaderVariation);

    commandBuffer.SetConstantBuffer(m_pShaderInfo->GetPixelShaderSlot(m_ShaderVariation), nn::gfx::ShaderStage_Pixel, gpuAddr, pixelShaderCombinerConstantBufferSize);
}

//----------------------------------------
void Material::ApplyPixelShaderCombinerUserShaderConstantBuffer(nn::gfx::CommandBuffer& commandBuffer, DrawInfo& drawInfo) const
{
    nn::gfx::GpuAddress gpuAddr;
    gpuAddr = *drawInfo.GetUi2dConstantBuffer()->GetGpuAddress();
    gpuAddr.Offset(m_PixelShaderConstantBufferOffset);

    size_t pixelShaderCombinerUserShaderConstantBufferSize = sizeof(Material::ConstantBufferForCombinerUserShaderPixelShader);

    NN_SDK_ASSERT(m_pShaderInfo->GetPixelShaderSlot(m_ShaderVariation) >= 0, "An invalid shader-slot(for uConstantBufferForPixelShader) was detectedat at [m_ShaderVariation=%d]!  Are you sure that you intend to use pixel-shader ?", m_ShaderVariation);

    commandBuffer.SetConstantBuffer(m_pShaderInfo->GetPixelShaderSlot(m_ShaderVariation), nn::gfx::ShaderStage_Pixel, gpuAddr, pixelShaderCombinerUserShaderConstantBufferSize);
}

//----------------------------------------
void Material::SetCommandBufferOnlyBlend(nn::gfx::CommandBuffer& commandBuffer) const
{
    if (IsAlphaCompareCap())
    {
        // アルファテストを設定して描画する
        EnableAlphaTest(commandBuffer);
        commandBuffer.SetBlendState(m_pBlendState);
        DisableAlphaTest();
    }
    else
    {
        commandBuffer.SetBlendState(m_pBlendState);
    }
}

//----------------------------------------
void Material::SetShader(nn::gfx::CommandBuffer& commandBuffer) const
{
    m_pShaderInfo->SetShader(commandBuffer, m_ShaderVariation);
}

//----------------------------------------
void Material::EnableAlphaTest(nn::gfx::CommandBuffer& commandBuffer) const
{
    #if defined(NN_BUILD_CONFIG_SPEC_GENERIC)
    // OpenGL 4.5 以降ではアルファテストが仕様から消滅したので対応しません。
    // 最終的にはアルファテストはシェーダで実装します。
    NN_UNUSED(commandBuffer);
    #elif defined(NN_BUILD_CONFIG_SPEC_NX)
    #if(NN_GFX_IS_TARGET_VK) || (NN_GFX_IS_TARGET_GL)
    // Vulkanにはアルファテストがありません。
    NN_UNUSED(commandBuffer);
    #else
    const ResAlphaCompare& alphaCompare = this->GetAlphaCompare();
    static const NVNalphaFunc functions[AlphaTest_MaxAlphaTest] =
    {
        NVN_ALPHA_FUNC_NEVER,
        NVN_ALPHA_FUNC_LESS,
        NVN_ALPHA_FUNC_LEQUAL,
        NVN_ALPHA_FUNC_EQUAL,
        NVN_ALPHA_FUNC_NOTEQUAL,
        NVN_ALPHA_FUNC_GEQUAL,
        NVN_ALPHA_FUNC_GREATER,
        NVN_ALPHA_FUNC_ALWAYS,
    };
    {
        nn::gfx::BlendState::DataType& data = nn::gfx::AccessorToData(m_pBlendState);
        pfnc_nvnColorStateSetAlphaTest(reinterpret_cast<NVNcolorState*>(&data.nvnColorState), functions[alphaCompare.GetFunc()]);
    }
    {
        nn::gfx::CommandBuffer::DataType& data = nn::gfx::AccessorToData(&commandBuffer);
        pfnc_nvnCommandBufferSetAlphaRef(data.pNvnCommandBuffer, alphaCompare.GetRef());
    }
    #endif
    #endif
}

//----------------------------------------
void Material::DisableAlphaTest() const
{
    #if defined(NN_BUILD_CONFIG_SPEC_GENERIC)
    // OpenGL 4.5 以降ではアルファテストが仕様から消滅したので対応しません。
    // 最終的にはアルファテストはシェーダで実装します。
    #elif defined(NN_BUILD_CONFIG_SPEC_NX)
    #if(NN_GFX_IS_TARGET_VK) || (NN_GFX_IS_TARGET_GL)
    // Vulkanにはアルファテストがありません。
    #else
    // NVN のドキュメントに "Using AlphaFunc::ALWAYS disables the alpha test entirely." とあるため、
    // これによりアルファテストを無効化する。
    {
        nn::gfx::BlendState::DataType& data = nn::gfx::AccessorToData(m_pBlendState);
        pfnc_nvnColorStateSetAlphaTest(reinterpret_cast<NVNcolorState*>(&data.nvnColorState), NVN_ALPHA_FUNC_ALWAYS);
    }
    #endif
    #endif
}

//----------------------------------------
bool Material::CompareCopiedInstanceTest(const Material& target) const
{
    if (memcmp(&m_MemCap, &target.m_MemCap, sizeof(detail::MatMemCount)) != 0 ||
        memcmp(&m_MemCount, &target.m_MemCount, sizeof(detail::MatMemCount)) != 0)
    {
        return false;
    }

    if (target.m_pMem != NULL)
    {
        if (m_pMem == NULL)
        {
            return false;
        }

        size_t  dataSize = sizeof(TexMap) * m_MemCap.texMap +
            sizeof(ResTexSrt) * m_MemCap.texSrt +
            sizeof(ResTexCoordGen) * m_MemCap.texCoordGen +
            sizeof(ResAlphaCompare) * m_MemCap.alpComp +
            sizeof(ResBlendMode) * m_MemCap.blendMode +
            sizeof(ResIndirectParameter) * m_MemCap.indirectParameter +
            sizeof(ResTevStage) * m_MemCap.tevStage +
            sizeof(ResProjectionTexGenParameters) * m_MemCap.projectionTexGen +
            sizeof(ResFontShadowParameter) * m_MemCap.fontShadowParameter +
            sizeof(ResDetailedCombinerStageInfo) * m_MemCap.detailedCombinerParameter +
            sizeof(ResDetailedCombinerStage) * m_MemCap.detailedCombinerParameter * m_MemCap.tevStage +
            sizeof(ResCombinerUserShader) * m_MemCap.combinerUserShaderParameter;

        if (memcmp(m_pMem, target.m_pMem, dataSize) != 0)
        {
            return false;
        }
    }

    if (m_pShaderInfo != target.m_pShaderInfo ||
        m_pName != target.m_pName)
    {
        return false;
    }

    if (target.m_pUserShaderConstantBufferInformation != NULL)
    {
        if (m_pUserShaderConstantBufferInformation == NULL)
        {
            return false;
        }
        //  ディープコピーのためメモリ内容を比較。
        if (memcmp(m_pUserShaderConstantBufferInformation, target.m_pUserShaderConstantBufferInformation, sizeof(UserShaderConstantBufferInformation)) != 0)
        {
            return false;
        }
    }

    // ブレンドステートは BlendStateId が同じになっていれば問題ないとみなす。
    if (GetBlendStateId() != target.GetBlendStateId())
    {
        return false;
    }

    if (m_AllowableTextureQuantity != target.m_AllowableTextureQuantity ||
        m_ShaderId != target.m_ShaderId ||
        m_Flag != target.m_Flag ||
        m_ShaderVariation != target.m_ShaderVariation)
    {
        return false;
    }

    if (detail::TestBit(m_Flag, Flags_DynamicAllocatedColorData))
    {
        if (memcmp(m_Colors.pFloatColor, target.m_Colors.pFloatColor, sizeof(nn::util::Float4)) != 0)
        {
            return false;
        }
    }
    else
    {
        if (memcmp(m_Colors.byteColor, target.m_Colors.byteColor, sizeof(nn::util::Unorm8x4)) != 0)
        {
            return false;
        }
    }

    return true;
}


} // namespace nn::ui2d
} // namespace nn
