﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ui2d/ui2d_Capture.h>
#include <nn/ui2d/ui2d_DrawInfo.h>
#include <nn/ui2d/ui2d_GraphicsResource.h>
#include <nn/ui2d/ui2d_ResourceAccessor.h>
#include <nn/ui2d/ui2d_Resources.h>
#include <nn/ui2d/ui2d_PaneEffect.h>

#if NN_GFX_IS_TARGET_NVN
#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>
#elif NN_GFX_IS_TARGET_GL
#define GLEW_NO_GLU
#include <GL/glew.h>
#elif NN_GFX_IS_TARGET_VK
#include <vulkan/vulkan.h>
#endif

namespace nn
{
namespace ui2d
{

Capture::Capture(
    const ResCapture* pBaseBlock,
    const ResCapture* pOverrideBlock,
    const BuildArgSet& buildArgSet
)
: Base(NULL, NULL, pBaseBlock, buildArgSet)
{
    NN_UNUSED(pOverrideBlock);
}

Capture::Capture(const Capture& capture)
: Base(capture)
{
}

Capture::~Capture()
{
    // OSReport("Capture::~Capture()\n");
}

void
Capture::Calculate(DrawInfo& drawInfo, Pane::CalculateContext& context, bool isDirtyParentMtx)
{
    const nn::util::MatrixT4x4fType   drawInfoProjMtx = drawInfo.GetProjectionMtx();

    const float halfWidth = GetSize().width * 0.5f;
    const float halfHeight = GetSize().height * 0.5f;

    nn::util::MatrixT4x4fType   projMtx;
    nn::util::MatrixOrthographicOffCenterRightHanded(&projMtx, -halfWidth, halfWidth, halfHeight, -halfHeight, 0.0f, 500.0f);

    drawInfo.SetProjectionMtx(projMtx);

    // キャプチャペインの子供ペインをキャプチャするテクスチャの原点付近へ描画するため
    //  親となっているキャプチャペインのグローバルマトリックスに単位行列を設定する。
    nn::util::MatrixT4x3fType glbMtx;

    detail::CalculateCaptureRootMatrix(glbMtx, drawInfo);
    SetGlobalMtx(glbMtx);

    Pane::Calculate(drawInfo, context, isDirtyParentMtx);

    drawInfo.SetProjectionMtx(drawInfoProjMtx);

    // ユーザーシェーダー更新コールバックが設定されていたら呼び出す。
    UpdateUserShaderCallback pCallback = drawInfo.GetUpdateUserShaderCallback();

    if (pCallback != NULL)
    {
        (*pCallback)(drawInfo, this, drawInfo.GetUpdateUserShaderCallbackUserData());
    }
}

void
Capture::Draw(DrawInfo& drawInfo, nn::gfx::CommandBuffer& commandBuffer)
{
    NN_UNUSED(drawInfo);
    NN_UNUSED(commandBuffer);
}

bool
Capture::CompareCopiedInstanceTest(const Capture& target) const
{
    NN_UNUSED(target);
    // コピーコンストラクタで何もコピーしていないため比較するものはない。
    return true;
}

//----------------------------------------------------------------------
CaptureTexture::CaptureTexture()
{
    InitializeParams();
}

//----------------------------------------------------------------------
CaptureTexture::CaptureTexture(const CaptureTexture& src, ResourceAccessor* pResAccessor, const char* pPrefix)
{
    InitializeParams();

    RegisterMaterialReferencedTextureInfo(pResAccessor, src.m_pCaptureTextureName, pPrefix);
}

//----------------------------------------------------------------------
CaptureTexture::~CaptureTexture()
{
}

//-------------------------------------------------------------------------------------------------
void
CaptureTexture::RegisterMaterialReferencedTextureInfo(ResourceAccessor* pResAcsr, const char* pPaneName, const char* pNamePrefix)
{
    const size_t prefixStringLength = pNamePrefix  != NULL ? strlen(pNamePrefix) : 0;
    const size_t paneNameStringLength = strlen(pPaneName);

    if (prefixStringLength > 0)
    {
        const size_t stringLength = prefixStringLength + paneNameStringLength + 1 + 1;
        m_pCaptureTextureName = static_cast<char*>(Layout::AllocateMemory(stringLength));

        MakeCaptureTextureName(m_pCaptureTextureName, stringLength, pNamePrefix, pPaneName);

        m_pFilteredTexture = pResAcsr->RegisterRenderTargetTexture(m_pCaptureTextureName);
    }
    else
    {
        m_pFilteredTexture = pResAcsr->RegisterRenderTargetTexture(pPaneName);

        m_pCaptureTextureName = static_cast<char*>(Layout::AllocateMemory(paneNameStringLength + 1));
        std::strncpy(m_pCaptureTextureName, pPaneName, paneNameStringLength + 1);
    }
}

//-------------------------------------------------------------------------------------------------
template<typename T>
void
CaptureTexture::Initialize(nn::gfx::Device* pDevice, const T* const pResCaptureTexture, Pane* pTargetPane)
{
    Initialize<T>(pDevice, NULL, pResCaptureTexture, pTargetPane);
}

template void CaptureTexture::Initialize<ResCaptureTexture>(nn::gfx::Device* pDevice, const ResCaptureTexture* const pResCaptureTexture, Pane* pTargetPane);
template void CaptureTexture::Initialize<ResCaptureTextureOld>(nn::gfx::Device* pDevice, const ResCaptureTextureOld* const pResCaptureTexture, Pane* pTargetPane);

//-------------------------------------------------------------------------------------------------
template<typename T>
void
CaptureTexture::Initialize(nn::gfx::Device* pDevice, const Layout* pLayout, const T* const pResCaptureTexture, Pane* pTargetPane)
{
    m_pCaptureTexture = Layout::AllocateAndConstruct<RenderTargetTextureInfo>();

    m_pTargetPane = pTargetPane;
    m_pOwnerLayout = pLayout;

    if (pResCaptureTexture->framebufferCaptureEnabled)
    {
        detail::SetBit(&m_Flags, Flags_CaptureFrameBuffer, 1);
    }

    if (pResCaptureTexture->captureOnlyFirstFrame)
    {
        detail::SetBit(&m_Flags, Flags_CaptureOnlyFirstFrame, 1);
    }

    m_ImageFormat = static_cast<nn::gfx::ImageFormat>(pResCaptureTexture->textureFormat);

    const float* pClearColor = pResCaptureTexture->clearColor;
    m_ClearColor.Set(pClearColor[0], pClearColor[1], pClearColor[2], pClearColor[3]);

    m_FilterCount = pResCaptureTexture->filterCount;

    // フィルターの追加情報を取得する。
    size_t resOffset = sizeof(T);

    for (int i = 0; i < m_FilterCount; ++i)
    {
        const ResCaptureTextureFilter *const pResCaptureTextureFilter = nn::util::ConstBytePtr(pResCaptureTexture, resOffset).Get<ResCaptureTextureFilter>();
        resOffset += sizeof(ResCaptureTextureFilter);

        // type に応じて追加で読み込むリソースの型を切り替える。
        // 現状はコピー用のシェーダーしか存在しないため固定で読み取っている。
        m_Scale = pResCaptureTextureFilter->textureScale;
    }

    // 初回キャプチャのみのテクスチャを初期化する。
    // ここで初期化テクスチャは Lifetime_Layout のヒントを与えられ、レイアウトインスタンスの終了処理時に破棄される。
    if (detail::TestBit(m_Flags, Flags_CaptureOnlyFirstFrame))
    {
        InitializeResources(pDevice, m_pOwnerLayout, RenderTargetTextureLifetime_Layout);
    }
}

template void CaptureTexture::Initialize<ResCaptureTexture>(nn::gfx::Device* pDevice, const Layout* pLayout, const ResCaptureTexture* const pResCaptureTexture, Pane* pTargetPane);
template void CaptureTexture::Initialize<ResCaptureTextureOld>(nn::gfx::Device* pDevice, const Layout* pLayout, const ResCaptureTextureOld* const pResCaptureTexture, Pane* pTargetPane);

//-------------------------------------------------------------------------------------------------
void
CaptureTexture::Initialize(nn::gfx::Device* pDevice, const CaptureTexture& src, Pane* pTargetPane)
{
    Initialize(pDevice, NULL, src, pTargetPane);
}

//-------------------------------------------------------------------------------------------------
void
CaptureTexture::Initialize(nn::gfx::Device* pDevice, const Layout* pLayout, const CaptureTexture& src, Pane* pTargetPane)
{
    m_pCaptureTexture = Layout::AllocateAndConstruct<RenderTargetTextureInfo>();

    m_pOwnerLayout = pLayout;
    m_pTargetPane = pTargetPane;
    if (detail::TestBit(src.m_Flags, Flags_CaptureFrameBuffer))
    {
        detail::SetBit(&m_Flags, Flags_CaptureFrameBuffer, 1);
    }

    if (detail::TestBit(src.m_Flags, Flags_CaptureOnlyFirstFrame))
    {
        detail::SetBit(&m_Flags, Flags_CaptureOnlyFirstFrame, 1);
    }

    m_ImageFormat = src.m_ImageFormat;
    m_ClearColor = src.m_ClearColor;
    m_FilterCount = src.m_FilterCount;
    m_Scale = src.m_Scale;

    // 初回キャプチャのみのテクスチャを初期化する。
    // ここで初期化テクスチャは Lifetime_Layout のヒントを与えられ、レイアウトインスタンスの終了処理時に破棄される。
    if (detail::TestBit(m_Flags, Flags_CaptureOnlyFirstFrame))
    {
        InitializeResources(pDevice, m_pOwnerLayout, RenderTargetTextureLifetime_Layout);
    }
}

//----------------------------------------------------------------------
void CaptureTexture::InitializeParams()
{
    m_pCaptureTextureName = NULL;
    m_pCaptureTexture = NULL;
    m_pFilteredTexture = NULL;
    m_pTargetPane = NULL;
    m_pOwnerLayout = NULL;
    m_FilterCount = 0;
    m_Scale = 1.0f;
    m_ImageFormat = nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm;
    m_Flags = 0;
#if NN_GFX_IS_TARGET_GL || NN_GFX_IS_TARGET_VK
    m_TextureCopyInfo = {};
#endif
}

//-------------------------------------------------------------------------------------------------
bool
CaptureTexture::InitializeResources(nn::gfx::Device* pDevice, const Layout* pLayout, RenderTargetTextureLifetime lifetimeHint)
{
    NN_SDK_ASSERT_NOT_NULL(m_pCaptureTexture);
    NN_SDK_ASSERT_NOT_NULL(m_pFilteredTexture);
    NN_SDK_ASSERT_NOT_NULL(m_pTargetPane);

    float   x;
    float   y;
    float fCaptureWidth;
    float fCaptureHeight;
    GetCaptureAreaInfo(&x, &y, &fCaptureWidth, &fCaptureHeight);

    const int captureWidth = std::max((int)fCaptureWidth, 1);
    const int captureHeight = std::max((int)fCaptureHeight, 1);

    // テクスチャ
    {
        nn::gfx::Texture::InfoType info;
        info.SetDefault();
        info.SetWidth(captureWidth);
        info.SetHeight(captureHeight);
        info.SetImageFormat(m_ImageFormat);
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_ColorBuffer|nn::gfx::GpuAccess_Texture);
        info.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
        info.SetMipCount(1);

        if (!m_pCaptureTexture->IsValid())
        {
            m_pCaptureTexture->Initialize(pDevice, pLayout, info, lifetimeHint);
        }

        int width = std::max((int)(fCaptureWidth * m_Scale), 1);
        int height = std::max((int)(fCaptureHeight * m_Scale), 1);

        info.SetWidth(width);
        info.SetHeight(height);

        if (!m_pFilteredTexture->IsValid())
        {
            m_pFilteredTexture->Initialize(pDevice, pLayout, info, lifetimeHint);
        }
    }

    // 終了時まで再初期化が必要ないかチェックする
    if (m_pCaptureTexture->GetLifetime() == RenderTargetTextureLifetime_Layout &&
        m_pFilteredTexture->GetLifetime() == RenderTargetTextureLifetime_Layout)
    {
        detail::SetBit(&m_Flags, Flags_NotNeedPerFrameReconstruct, 1);
    }

    return true;
}

//-------------------------------------------------------------------------------------------------
void
CaptureTexture::Finalize(nn::gfx::Device* pDevice, ResourceAccessor* pResAcsr)
{
    if (m_pFilteredTexture != NULL)
    {
        m_pFilteredTexture->Finalize(pDevice);
        pResAcsr->UnregisterRenderTargetTexture(m_pFilteredTexture);
        Layout::DeleteObj(m_pFilteredTexture);
        m_pFilteredTexture = NULL;
    }

    if (m_pCaptureTexture != NULL)
    {
        m_pCaptureTexture->Finalize(pDevice);
        Layout::DeleteObj(m_pCaptureTexture);
        m_pCaptureTexture = NULL;
    }

    if (m_pCaptureTextureName != NULL)
    {
        Layout::FreeMemory(m_pCaptureTextureName);
        m_pCaptureTextureName = NULL;
    }
}

//-------------------------------------------------------------------------------------------------
void
CaptureTexture::Draw(nn::gfx::Device* pDevice, DrawInfo& drawInfo, nn::gfx::CommandBuffer& commandBuffer)
{
    // ペインのコンスタントバッファが作成されない状態ではキャプチャも行わない。
    if (!m_pTargetPane->IsConstantBufferUpdateNeeded())
    {
        return;
    }

    const bool  paneEffectEnabled = m_pTargetPane->IsPaneEffectEnabled();

    if (paneEffectEnabled)
    {
        // ペインエフェクト機能が有効な場合は対象ペインだけしか描画を行わないため
        // 対象ペインの描画準備ができているときのみキャプチャ処理を行います。
        if (!m_pTargetPane->IsDrawSelfReady())
        {
            return;
        }
    }

    // 初回更新テクスチャは未処理の時のみ更新する。
    if (detail::TestBit(m_Flags, Flags_CaptureOnlyFirstFrame))
    {
        if (detail::TestBit(m_Flags, Flags_CaptureOnlyFirstFrameUpdated))
        {
            return;
        }

        // 処理済みのフラグを立てる。
        detail::SetBit(&m_Flags, Flags_CaptureOnlyFirstFrameUpdated, 1);
    }

    // 動的に内容が更新されるテクスチャの終了・初期化処理
    // ユーザーがテクスチャ作成コールバックで RenderTargetLifetime_Layout に変更した場合、以下の初期化・終了処理は無駄なため
    //  Flags_NotNeedPerFrameReconstruct フラグを見て処理するかどうか決定する。
    // Flags_NotNeedPerFrameReconstruct は InitializeReosurces 内で初期化後のユーザーの設定を確認して設定される。
    if (!detail::TestBit(m_Flags, Flags_NotNeedPerFrameReconstruct))
    {
        // 終了処理
        if (m_pCaptureTexture->IsValid())
        {
            m_pCaptureTexture->Finalize(pDevice);
        }

        if (m_pFilteredTexture->IsValid())
        {
            m_pFilteredTexture->Finalize(pDevice);
        }

        InitializeResources(pDevice, m_pOwnerLayout, RenderTargetTextureLifetime_OneFrame);
    }

    nn::gfx::ColorTargetView*   pTarget = m_pCaptureTexture->GetColorTargetView();

    commandBuffer.ClearColor(
        pTarget,
        m_ClearColor.GetX(),
        m_ClearColor.GetY(),
        m_ClearColor.GetZ(),
        m_ClearColor.GetW(),
        NULL);

    if (detail::TestBit(m_Flags, Flags_CaptureFrameBuffer))
    {
        CopyFramebufferTexture(drawInfo, commandBuffer);
    }

    commandBuffer.SetRenderTargets(1, &pTarget, NULL);
    commandBuffer.SetViewportScissorState(m_pCaptureTexture->GetViewportScissorState());

    if (paneEffectEnabled)
    {
        m_pTargetPane->DrawSelf(drawInfo, commandBuffer);
    }
    else
    {
        // 子供のペインを描画
        m_pTargetPane->DrawChildren(drawInfo, commandBuffer);
    }

    // フィルターを適用する
    ApplyFilterToCaptureTexture(drawInfo, commandBuffer);

    // 次のキャプチャペインの子ペイン描画に備えてリセットしておく
    drawInfo.ResetCurrentShader();
    drawInfo.ResetVertexBufferState();
}

#if NN_GFX_IS_TARGET_GL
//-------------------------------------------------------------------------------------------------
void CopyImageWithYMirroring(const void* pParam)
{
    // OpenGL 版のテクスチャイメージ反転コピー実装です。
    // FBO を描画毎に作成・破棄していますが、FBO を ui2d ランタイム内で適切に管理できないための暫定実装です。
    // 将来的に gfx 側でミラーリングしてコピーする機能が実装される予定があるとのことで、機能が追加された際に置き換えられる予定です。
    const TextureCopyInfo* pInfo = static_cast<const TextureCopyInfo*>(pParam);

    uint32_t    tmpFbo[2];
    glGenFramebuffers(2, tmpFbo);

    glEnable(GL_FRAMEBUFFER_SRGB);

    GLboolean backScissorTest = glIsEnabled(GL_SCISSOR_TEST);
    if (backScissorTest == GL_TRUE)
    {
        glDisable(GL_SCISSOR_TEST);
    }

    GLint tmpFboDraw;
    glGetIntegerv(GL_DRAW_FRAMEBUFFER_BINDING, &tmpFboDraw);

    glBindFramebuffer(GL_DRAW_FRAMEBUFFER, tmpFbo[0]);
    glBindFramebuffer(GL_READ_FRAMEBUFFER, tmpFbo[1]);
    GLenum attachment = GL_COLOR_ATTACHMENT0;
    glFramebufferDrawBuffersEXT(tmpFbo[0], 1, &attachment);

    glNamedFramebufferTextureEXT(tmpFbo[1],
        GL_COLOR_ATTACHMENT0, pInfo->pSrcTexture->ToData()->hTexture, 0);
    glNamedFramebufferTextureEXT(tmpFbo[0],
        GL_COLOR_ATTACHMENT0, pInfo->pDstTexture->ToData()->hTexture, 0);

    glBlitFramebuffer(pInfo->srcX0, pInfo->srcY0,
        pInfo->srcX1, pInfo->srcY1, pInfo->dstX0, pInfo->dstY0,
        pInfo->dstX1, pInfo->dstY1, GL_COLOR_BUFFER_BIT, GL_NEAREST);

    glBindFramebuffer(GL_FRAMEBUFFER, tmpFboDraw);

    if (backScissorTest == GL_TRUE)
    {
        glEnable(GL_SCISSOR_TEST);
    }

    glDeleteFramebuffers(2, tmpFbo);

    GLenum error;
    NN_SDK_ASSERT(GL_NO_ERROR == (error = ::glGetError()), "GL Error: %s (0x%x)\n", error);
    NN_UNUSED(error);
}
#endif

#if NN_GFX_IS_TARGET_VK
//-------------------------------------------------------------------------------------------------
void CopyImageWithYMirroringVk(VkCommandBuffer commandBuffer, const void* pParam)
{
    // Vulkan 版のテクスチャイメージ反転コピー実装です。
    // 将来的に gfx 側でミラーリングしてコピーする機能が実装される予定があるとのことで、機能が追加された際に置き換えられる予定です。
    const TextureCopyInfo* pInfo = static_cast<const TextureCopyInfo*>(pParam);

#if defined( NN_BUILD_CONFIG_OS_WIN ) && !defined( NN_BUILD_CONFIG_CPU_X64 )
    VkImage srcImage = pInfo->pSrcTexture->ToData()->hImage;
    VkImage dstImage = pInfo->pDstTexture->ToData()->hImage;
#else
    VkImage srcImage = reinterpret_cast< VkImage >( pInfo->pSrcTexture->ToData()->hImage );
    VkImage dstImage = reinterpret_cast< VkImage >( pInfo->pDstTexture->ToData()->hImage );
#endif
    VkImageBlit region;

    region.srcOffsets[ 0 ].x = pInfo->srcX0;
    region.srcOffsets[ 0 ].y = pInfo->srcY0;
    region.srcOffsets[ 0 ].z = 0;
    region.srcOffsets[ 1 ].x = pInfo->srcX1;
    region.srcOffsets[ 1 ].y = pInfo->srcY1;
    region.srcOffsets[ 1 ].z = 1;
    region.srcSubresource.aspectMask = VK_IMAGE_ASPECT_COLOR_BIT;
    region.srcSubresource.baseArrayLayer = 0;
    region.srcSubresource.layerCount = 1;
    region.srcSubresource.mipLevel = 0;

    region.dstOffsets[ 0 ].x = pInfo->dstX0;
    region.dstOffsets[ 0 ].y = pInfo->dstY0;
    region.dstOffsets[ 0 ].z = 0;
    region.dstOffsets[ 1 ].x = pInfo->dstX1;
    region.dstOffsets[ 1 ].y = pInfo->dstY1;
    region.dstOffsets[ 1 ].z = 1;
    region.dstSubresource.aspectMask = VK_IMAGE_ASPECT_COLOR_BIT;
    region.dstSubresource.baseArrayLayer = 0;
    region.dstSubresource.layerCount = 1;
    region.dstSubresource.mipLevel = 0;

    vkCmdBlitImage( commandBuffer, srcImage, VK_IMAGE_LAYOUT_GENERAL, dstImage, VK_IMAGE_LAYOUT_GENERAL, 1, &region, VK_FILTER_NEAREST );
}
#endif

//-------------------------------------------------------------------------------------------------
void
CaptureTexture::CopyFramebufferTexture(DrawInfo& drawInfo, nn::gfx::CommandBuffer& commandBuffer)
{
    NN_SDK_ASSERT_NOT_NULL(drawInfo.GetFramebufferTexture());

    const int framebufferWidth = drawInfo.GetFramebufferWidth();
    const int framebufferHeight = drawInfo.GetFramebufferHeight();
    nn::gfx::TextureSubresource ts;
    ts.SetDefault();
    nn::gfx::TextureCopyRegion  srcCopyRegion;
    nn::gfx::TextureCopyRegion  dstCopyRegion;

    srcCopyRegion.SetDefault();
    dstCopyRegion.SetDefault();

    float   x;
    float   y;
    float fCaptureWidth;
    float fCaptureHeight;
    GetCaptureAreaInfo(&x, &y, &fCaptureWidth, &fCaptureHeight);

    int copyWidth = (int)fCaptureWidth;
    int copyHeight = (int)fCaptureHeight;

    int srcX = (int)((framebufferWidth / 2) + (int)x - (int)fCaptureWidth / 2);
    int srcY = (int)((framebufferHeight / 2) - (int)y - (int)fCaptureHeight / 2);
    int destX = 0;
    int destY = 0;

    if (srcX < 0)
    {
        copyWidth += srcX;
        destX = -srcX;
        srcX = 0;
    }

    if (srcY < 0)
    {
        copyHeight += srcY;
        destY = -srcY;
        srcY = 0;
    }

    if (srcX + copyWidth > framebufferWidth)
    {
        int protrudeSize = srcX + copyWidth - framebufferWidth;
        copyWidth -= protrudeSize;
    }

    if (srcY + copyHeight > framebufferHeight)
    {
        int protrudeSize = srcY + copyHeight - framebufferHeight;
        copyHeight -= protrudeSize;
    }

    if (srcX >= framebufferWidth ||
        srcX + copyWidth <= 0 ||
        srcY >= framebufferHeight ||
        srcY + copyHeight <= 0)
    {
        // 全くキャプチャする範囲がないため何もしない。
        return;
    }

    // 左上がウインドウ原点のシステムではキャプチャ元の Y 軸が反転しているため逆転する。
    if (drawInfo.IsLeftTopWindowOrigin())
    {
        srcY = framebufferHeight - (srcY + copyHeight);
    }

    srcCopyRegion.SetOffsetU(srcX);
    srcCopyRegion.SetOffsetV(((framebufferHeight - copyHeight) - srcY));
    srcCopyRegion.SetWidth(copyWidth);
    srcCopyRegion.SetHeight(copyHeight);

    const nn::gfx::Texture* pSrcTexture = drawInfo.GetFramebufferTexture();
    const nn::gfx::Texture* pDstTexture = m_pCaptureTexture->GetTexture();

    // 左下原点のテクスチャをイメージとしてコピーすると反転するため、コピー時に Y 軸反転させる。
    // gfx のコピー処理では反転できないためそれぞれの環境ごとに実装分岐して専用のものを使用する。
#if NN_GFX_IS_TARGET_NVN
    // NVN 版
    const nn::gfx::TextureSubresource& dstSubresource = ts;

    NN_SDK_ASSERT_NOT_NULL(pSrcTexture->ToData()->pNvnTexture.ptr);
    NN_SDK_ASSERT_NOT_NULL(pDstTexture->ToData()->pNvnTexture.ptr);

    NVNtextureTarget target = nvnTextureGetTarget(pSrcTexture->ToData()->pNvnTexture);

    int srcV = srcCopyRegion.GetOffsetV();
    int srcHeight = srcCopyRegion.GetHeight();
    int srcW = srcCopyRegion.GetOffsetW();
    int srcDepth = 1;

    NVNcopyRegion srcRegion;
    srcRegion.xoffset = srcCopyRegion.GetOffsetU();
    srcRegion.yoffset = srcV;
    srcRegion.zoffset = srcW;
    srcRegion.width = srcCopyRegion.GetWidth();
    srcRegion.height = srcHeight;
    srcRegion.depth = srcDepth;
    NVNcopyRegion dstRegion;
    dstRegion.xoffset = destX;
    dstRegion.yoffset = target == NVN_TEXTURE_TARGET_1D_ARRAY ? dstSubresource.GetArrayIndex() : destY;
    dstRegion.zoffset = (target == NVN_TEXTURE_TARGET_2D_ARRAY ||
        target == NVN_TEXTURE_TARGET_2D_MULTISAMPLE_ARRAY) ? dstSubresource.GetArrayIndex() : 0;
    dstRegion.width = srcRegion.width;
    dstRegion.height = srcRegion.height;
    dstRegion.depth = srcRegion.depth;

    NVNtextureView srcView;
    nvnTextureViewSetDefaults(&srcView);
    nvnTextureViewSetLevels(
        &srcView, srcCopyRegion.GetSubresource().GetMipLevel(), 1);

    NVNtextureView dstView;
    nvnTextureViewSetDefaults(&dstView);
    nvnTextureViewSetLevels(
        &dstView, dstSubresource.GetMipLevel(), 1);

    nvnCommandBufferCopyTextureToTexture(commandBuffer.ToData()->pNvnCommandBuffer,
        pSrcTexture->ToData()->pNvnTexture, &srcView, &srcRegion,
        pDstTexture->ToData()->pNvnTexture, &dstView, &dstRegion, NVN_COPY_FLAGS_MIRROR_Y_BIT);
#elif NN_GFX_IS_TARGET_GL
    // PC/OpenGL 版
    m_TextureCopyInfo.pSrcTexture = pSrcTexture;
    m_TextureCopyInfo.pDstTexture = pDstTexture;

    m_TextureCopyInfo.srcX0 = srcCopyRegion.GetOffsetU();
    m_TextureCopyInfo.srcX1 = srcCopyRegion.GetOffsetU() + srcCopyRegion.GetWidth();
    m_TextureCopyInfo.srcY0 = srcCopyRegion.GetOffsetV();
    m_TextureCopyInfo.srcY1 = srcCopyRegion.GetOffsetV() + srcCopyRegion.GetHeight();
    m_TextureCopyInfo.dstX0 = destX;
    m_TextureCopyInfo.dstX1 = srcCopyRegion.GetWidth() + destX;
    m_TextureCopyInfo.dstY0 = srcCopyRegion.GetHeight() + destY;
    m_TextureCopyInfo.dstY1 = destY;

    commandBuffer.Gl4SetUserCommand(CopyImageWithYMirroring, &m_TextureCopyInfo);
#elif NN_GFX_IS_TARGET_VK

    // Vulkan版
    m_TextureCopyInfo.pSrcTexture = pSrcTexture;
    m_TextureCopyInfo.pDstTexture = pDstTexture;

    m_TextureCopyInfo.srcX0 = srcCopyRegion.GetOffsetU();
    m_TextureCopyInfo.srcX1 = srcCopyRegion.GetOffsetU() + srcCopyRegion.GetWidth();
    m_TextureCopyInfo.srcY0 = srcCopyRegion.GetOffsetV();
    m_TextureCopyInfo.srcY1 = srcCopyRegion.GetOffsetV() + srcCopyRegion.GetHeight();
    m_TextureCopyInfo.dstX0 = destX;
    m_TextureCopyInfo.dstX1 = srcCopyRegion.GetWidth() + destX;
    m_TextureCopyInfo.dstY0 = srcCopyRegion.GetHeight() + destY;
    m_TextureCopyInfo.dstY1 = destY;

    commandBuffer.Vk1SetUserCommandDynamic(CopyImageWithYMirroringVk, &m_TextureCopyInfo, sizeof( m_TextureCopyInfo ) );

#else
    dstCopyRegion.SetOffsetU(destX);
    dstCopyRegion.SetOffsetV(destY);
    dstCopyRegion.SetWidth(copyWidth);
    dstCopyRegion.SetHeight(copyHeight);

    commandBuffer.BlitImage(m_pCaptureTexture->GetTexture(), dstCopyRegion, drawInfo.GetFramebufferTexture(), srcCopyRegion, 0);
#endif
}// NOLINT(impl/function_size)

//-------------------------------------------------------------------------------------------------
void
CaptureTexture::ApplyFilterToCaptureTexture(DrawInfo& drawInfo, nn::gfx::CommandBuffer& commandBuffer)
{
    ApplyCaptureTextureFilterCallback   pCallback = drawInfo.GetApplyCaptureTextureFilterCallback();
    bool    copyImage = true;

    if (pCallback != NULL)
    {
        // コールバック内で描画が行われたかどうかをコールバック関数の返り値で判断して
        // 描画されなかった場合は通常のコピー処理を行う。
        copyImage = !(*pCallback)(
            commandBuffer,
            m_pFilteredTexture->GetColorTargetView(),
            *(m_pFilteredTexture->GetTextureDescriptorSlot()),
            *(m_pCaptureTexture->GetTextureDescriptorSlot()),
            *(m_pFilteredTexture->GetViewportScissorState()),
            m_pTargetPane,
            drawInfo.GetApplyCaptureTextureFilterCallbackUserData());
    }

    if (copyImage)
    {
        nn::gfx::TextureCopyRegion  dstRegion;
        dstRegion.SetDefault();
        dstRegion.SetOffsetU(0);
        dstRegion.SetOffsetV(0);
        dstRegion.SetWidth(m_pFilteredTexture->GetSize().width);
        dstRegion.SetHeight(m_pFilteredTexture->GetSize().height);

        nn::gfx::TextureCopyRegion  srcRegion;

        srcRegion.SetDefault();

        srcRegion.SetOffsetU(0);
        srcRegion.SetOffsetV(0);
        srcRegion.SetWidth(m_pCaptureTexture->GetSize().width);
        srcRegion.SetHeight(m_pCaptureTexture->GetSize().height);

        commandBuffer.BlitImage(m_pFilteredTexture->GetTexture(), dstRegion, m_pCaptureTexture->GetTexture(), srcRegion, 0);
    }
}

//-------------------------------------------------------------------------------------------------
void CaptureTexture::GetCaptureAreaInfo(float* pX, float* pY, float* pWidth, float* pHeight) const
{
    NN_SDK_ASSERT_NOT_NULL(m_pTargetPane);

    *pWidth = m_pTargetPane->GetSize().width;
    *pHeight = m_pTargetPane->GetSize().height;

    nn::util::Vector3f  vGlobalTrans;
    nn::util::Vector3f  vLocalTrans;
    nn::util::Float3    localTrans = m_pTargetPane->GetTranslate();
    nn::util::VectorSet(&vLocalTrans, localTrans.x, localTrans.y, localTrans.z);
    nn::util::VectorTransform(&vGlobalTrans, vLocalTrans, m_pTargetPane->GetParent()->GetGlobalMtx());

    *pX = vGlobalTrans.GetX();
    *pY = vGlobalTrans.GetY();
}

} // namespace nn::ui2d
} // namespace nn
