﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ui2d/ui2d_ButtonGroup.h>
#include <nn/ui2d/ui2d_Common.h>
#include <nn/ui2d/ui2d_Layout.h>

namespace nn
{
namespace ui2d
{

ButtonGroup::ButtonGroup()
: m_HitButton(NULL)
, m_DraggingButton(NULL)
{
    m_Flag.Clear();
    m_Flag.SetAllBitOn(FlagMask_UpdateHitBox | FlagMask_ExcludeDown | FlagMask_HitTestEnabled);
}

ButtonGroup::~ButtonGroup()
{
}

void ButtonGroup::Update(const nn::util::Float2* pPos, bool isDown, bool isRelease /* = false */)
{
    // 当たりの領域の更新
    if (IsUpdateHitBox()) {
        ButtonList::iterator endIter = m_ButtonList.end();
        for (ButtonList::iterator iter = m_ButtonList.begin(); iter != endIter; ++iter)
        {
            iter->UpdateHitBox();
        }
    }

    {
        AnimButton* pHitButton = NULL;
        bool isExcludeDown = false;
        if (IsExcludeDown()) {
            // ボタンの中から、押されたときに他のボタンが押されるのを防ぐ設定になっていて、
            // かつ現在押されているボタンを探し、もしあったらこのフレームボタンには当たり
            // が発生しないようにする。
            ButtonList::iterator endIter = m_ButtonList.end();
            for (ButtonList::iterator iter = m_ButtonList.begin(); iter != endIter; ++iter)
            {
                if (iter->IsExcludeDown() && iter->IsDowning())
                {
                    isExcludeDown = true;
                    break;
                }
            }
        }

        if (m_DraggingButton) {
            if (IsHitTestEnabled() && ! isExcludeDown && pPos) {
                if (m_DraggingButton->IsActive() && m_DraggingButton->IsHit(*pPos)) {
                    pHitButton = m_DraggingButton;
                }
            }
            m_DraggingButton->UpdateDragPosition(pPos);
            if (isRelease) {
                m_DraggingButton->Cancel();
                if (pHitButton != m_DraggingButton) {
                    m_DraggingButton->Off();
                }
                m_DraggingButton = NULL;
            }
        } else {
            if (IsHitTestEnabled() && ! isExcludeDown && pPos) {
                ButtonList::iterator endIter = m_ButtonList.end();
                for (ButtonList::iterator iter = m_ButtonList.begin(); iter != endIter; ++iter) {
                    if (iter->IsActive() && iter->IsHit(*pPos)) {
                        pHitButton = &(*iter);
                        break;
                    }
                }
            }
        }

        if (m_HitButton != pHitButton) {
            if (m_HitButton) {
                m_HitButton->Off();
            }
            if (pHitButton) {
                pHitButton->On();
            }
            m_HitButton = pHitButton;
        }

        if (m_HitButton) {
            bool isDownActually = ( ! isExcludeDown && isDown);
            if (isDownActually) {
                m_HitButton->Down();
            }
            AnimButton::DragMode dragMode = m_HitButton->GetDragMode();
            if ((dragMode == AnimButton::DragMode_OnHit || (dragMode == AnimButton::DragMode_OnDown && isDownActually)) && m_DraggingButton == NULL && ! isRelease) {
                m_DraggingButton = m_HitButton;
                m_DraggingButton->InitializeDragPosition(*pPos);
            }
        }
    }

    // 状態の更新
    {
        ButtonList::iterator endIter = m_ButtonList.end();
        for (ButtonList::iterator iter = m_ButtonList.begin(); iter != endIter; ++iter)
        {
            iter->Update();
        }
    }
}

AnimButton* ButtonGroup::FindDownButton()
{
    ButtonList::iterator endIter = m_ButtonList.end();
    for (ButtonList::iterator iter = m_ButtonList.begin(); iter != endIter; ++iter)
    {
        if (iter->IsDowning()) {
            return &(*iter);
        }
    }
    return NULL;
}

AnimButton* ButtonGroup::FindButtonByName(const char* pButtonName)
{
    ButtonList::iterator endIter = m_ButtonList.end();
    for (ButtonList::iterator iter = m_ButtonList.begin(); iter != endIter; ++iter)
    {
        if (ui2d::detail::EqualsLayoutName(pButtonName, iter->GetName())) {
            return &(*iter);
        }
    }
    return NULL;
}

const AnimButton* ButtonGroup::FindButtonByName(const char* pButtonName) const
{
    return const_cast<ButtonGroup*>(this)->FindButtonByName(pButtonName);
}

AnimButton* ButtonGroup::FindButtonByNameReverse(const char* pButtonName)
{
    ButtonList::reverse_iterator endIter = m_ButtonList.rend();
    for (ButtonList::reverse_iterator iter = m_ButtonList.rbegin(); iter != endIter; ++iter)
    {
        if (ui2d::detail::EqualsLayoutName(pButtonName, iter->GetName())) {
            return &(*iter);
        }
    }
    return NULL;
}

const AnimButton* ButtonGroup::FindButtonByNameReverse(const char* pButtonName) const
{
    return const_cast<ButtonGroup*>(this)->FindButtonByNameReverse(pButtonName);
}

AnimButton* ButtonGroup::FindButtonByTag(int tag)
{
    ButtonList::iterator endIter = m_ButtonList.end();
    for (ButtonList::iterator iter = m_ButtonList.begin(); iter != endIter; ++iter)
    {
        if (tag == iter->GetTag()) {
            return &(*iter);
        }
    }
    return NULL;
}

const AnimButton* ButtonGroup::FindButtonByTag(int tag) const
{
    return const_cast<ButtonGroup*>(this)->FindButtonByTag(tag);
}

void ButtonGroup::ForceOffAll()
{
    ButtonList::iterator endIter = m_ButtonList.end();
    for (ButtonList::iterator iter = m_ButtonList.begin(); iter != endIter; ++iter)
    {
        iter->ForceOff();
    }
}

void ButtonGroup::ForceOnAll()
{
    ButtonList::iterator endIter = m_ButtonList.end();
    for (ButtonList::iterator iter = m_ButtonList.begin(); iter != endIter; ++iter)
    {
        iter->ForceOn();
    }
}

void ButtonGroup::ForceDownAll()
{
    ButtonList::iterator endIter = m_ButtonList.end();
    for (ButtonList::iterator iter = m_ButtonList.begin(); iter != endIter; ++iter)
    {
        iter->ForceDown();
    }
}

void ButtonGroup::UpdateHitBoxAll()
{
    ButtonList::iterator endIter = m_ButtonList.end();
    for (ButtonList::iterator iter = m_ButtonList.begin(); iter != endIter; ++iter)
    {
        iter->UpdateHitBox();
    }
}

void ButtonGroup::SetStateChangeCallbackAll(AnimButton::StateChangeCallback pCallback, void* pParam)
{
    ButtonList::iterator endIter = m_ButtonList.end();
    for (ButtonList::iterator iter = m_ButtonList.begin(); iter != endIter; ++iter)
    {
        iter->SetStateChangeCallback(pCallback, pParam);
    }
}

void ButtonGroup::CancelAll()
{
    ButtonList::iterator endIter = m_ButtonList.end();
    for (ButtonList::iterator iter = m_ButtonList.begin(); iter != endIter; ++iter)
    {
        iter->Cancel();
    }
}

void ButtonGroup::FreeAll()
{
    for (ButtonList::iterator iter = m_ButtonList.begin(); iter != m_ButtonList.end();)
    {
        ButtonList::iterator currIter = iter++;
        m_ButtonList.erase(currIter);
        ui2d::Layout::DeleteObj(&(*currIter));
    }
}

} // namespace nn::ui2d
} // namespace nn
