﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../precompiled.h"

#ifdef NN_BUILD_CONFIG_SPY_ENABLED

#include <nn/spy/detail/spy_SpyPacketReader.h>

#include <nn/spy/detail/fnd/basis/spyfnd_Memory.h>

#if defined(NN_SDK_BUILD_DEBUG)
//#define STATE_DEBUG_ENABLED
//#define COM_DEBUG_ENABLED
#endif

namespace nn {
namespace spy {
namespace detail {

//----------------------------------------------------------
SpyPacketReader::SpyPacketReader() NN_NOEXCEPT
    : m_Channel(NULL)
    , m_State(State_NotInitialized)
    , m_pPacketHeader(NULL)
    , m_ReadPacketLength(0)
{
}

//----------------------------------------------------------
void
SpyPacketReader::Initialize(nn::spy::detail::fnd::HioChannel* channel, void* buffer) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Channel == NULL);
    NN_SDK_ASSERT_NOT_NULL(channel);
    NN_SDK_ASSERT_NOT_NULL(buffer);

    m_Channel = channel;
    m_pPacketHeader = reinterpret_cast<PacketHeader*>(buffer);
    m_ReadPacketLength = 0;
    m_State = State_Idle;
}

//----------------------------------------------------------
void
SpyPacketReader::Finalize() NN_NOEXCEPT
{
    m_Channel = NULL;
    m_pPacketHeader = NULL;
    m_ReadPacketLength = 0;
    m_State = State_NotInitialized;
}

//----------------------------------------------------------
bool
SpyPacketReader::ReadPacketHeader() NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        return false;
    }

    if(State_Idle < m_State && m_State < State_ReadBody)
    {
        NN_SDK_ASSERT(false, "packet not completed!\n");
        return false;
    }

    NN_SDK_ASSERT(m_State == State_Idle || m_State == State_ReadBody);

    if(!m_Channel->IsOpened())
    {
        m_State = State_Idle;
        return false;
    }

#if defined(COM_DEBUG_ENABLED)
    NN_DETAIL_SPY_INFO("[SpyPacketReader] ReadPacketHeader().\n");
#endif

    m_State = State_ReadingHeader;

    // HioChannel::Read()は指定の長さ未満で戻ることがあるので
    // 指定の長さに達するまで繰り返し呼び出す。
    const size_t partBegin = 0;
    const size_t partEnd = partBegin + sizeof(PacketHeader);
    while (m_ReadPacketLength < partEnd)
    {
        size_t readLength = partEnd - m_ReadPacketLength;
        size_t offset = m_ReadPacketLength - partBegin;

        size_t readResult = m_Channel->Read(nn::spy::detail::fnd::AddOffsetToPtr(m_pPacketHeader, offset), readLength);
        if (readResult == static_cast<uint32_t>(-1) || readResult == 0)
        {
            break;
        }

        m_ReadPacketLength += readResult;
    }

    if(m_ReadPacketLength < partEnd)
    {
        // ヘッダを完全に読み終わるまではアイドル状態にとどまる。
        m_State = State_Idle;
        return false;
    }

    if(m_pPacketHeader->bodyLength == 0)
    {
        m_State = State_ReadBody;
    }
    else
    {
        m_State = State_ReadHeader;
    }

    return true;
}

//----------------------------------------------------------
bool
SpyPacketReader::ReadPacketBody(void* buffer, size_t length) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(buffer);

    NN_UNUSED(length);

    if(!IsInitialized())
    {
        return false;
    }

    if(m_State == State_ReadBody)
    {
        return false;
    }

    if(m_ReadPacketLength < sizeof(PacketHeader) || m_State != State_ReadHeader)
    {
        NN_SDK_ASSERT(false, "packet header not read!\n");
        return false;
    }

    const PacketHeader* packetHeader = GetCurrentPacketHeader();
    NN_SDK_ASSERT_NOT_NULL(packetHeader);
    NN_SDK_ASSERT(m_ReadPacketLength < packetHeader->bodyLength + sizeof(PacketHeader));
    // bufferにはパケットボディをすべて格納できるサイズが必要。
    NN_SDK_ASSERT(packetHeader->bodyLength <= length);

#if defined(COM_DEBUG_ENABLED)
    NN_DETAIL_SPY_INFO("[SpyPacketReader] ReadPacketBody() : bodyLength=%d.\n", packetHeader->bodyLength);
#endif

    m_State = State_ReadingBody;

    // HioChannel::Read()は指定の長さ未満で戻ることがあるので
    // 指定の長さに達するまで繰り返し呼び出す。
    const size_t partBegin = sizeof(PacketHeader);
    const size_t partEnd = partBegin + packetHeader->bodyLength;
    while (m_ReadPacketLength < partEnd)
    {
        size_t readLength = partEnd - m_ReadPacketLength;
        size_t offset = m_ReadPacketLength - partBegin;

        size_t readResult = m_Channel->Read(nn::spy::detail::fnd::AddOffsetToPtr(buffer, offset), readLength);
        if (readResult == static_cast<size_t>(-1) || readResult == 0)
        {
            break;
        }

        m_ReadPacketLength += readResult;
    }

    if(m_ReadPacketLength < partEnd)
    {
        // ボディを完全に読み終わるまではヘッダ読込済み状態にとどまる。
        m_State = State_ReadHeader;
        return false;
    }

    m_State = State_ReadBody;

    return true;
}

//----------------------------------------------------------
void
SpyPacketReader::Next() NN_NOEXCEPT
{
    if(m_State == State_Idle)
    {
        return;
    }

    NN_SDK_ASSERT(m_State == State_ReadBody, "invalid state(%s).\n", StateToString(m_State));

    m_ReadPacketLength = 0;
    m_State = State_Idle;
}

//----------------------------------------------------------
const char*
SpyPacketReader::StateToString(State value) NN_NOEXCEPT
{
#if defined(STATE_DEBUG_ENABLED)
    static const char* strings[] = {
        "State_NotInitialized",
        "State_Idle",
        "State_ReadingHeader",
        "State_ReadHeader",
        "State_ReadingBody",
        "State_ReadBody"
    };

    return strings[value];
#else
    NN_UNUSED(value);
    return NULL;
#endif
}

} // namespace nn::spy::detail
} // namespace nn::spy
} // namespace nn

#endif // NN_BUILD_CONFIG_SPY_ENABLED
