﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../../precompiled.h"

#ifdef NN_BUILD_CONFIG_SPY_ENABLED

#include <nn/spy/detail/fnd/hio/spyfnd_HioChannel.h>

#include <cafe/os.h>
#include <cafe/hio.h>
#include <nn/spy/detail/fnd/string/spyfnd_String.h>

#if defined(NN_SDK_BUILD_DEBUG)
#define STATUS_DEBUG_ENABLED
#endif

namespace nn {
namespace spy {
namespace detail {
namespace fnd {

namespace {

static const HIOHandle InvalidHandle = -1;

}

NN_DEFINE_STATIC_CONSTANT(const uint32_t HioChannel::MaxPortLength);

//----------------------------------------------------------
HioChannel::HioChannel() NN_NOEXCEPT
    : m_Handle(InvalidHandle)
    , m_IsConnected(false)
    , m_ConnectEvent(true, true)
{
    m_Port[0] = '\0';
}

//----------------------------------------------------------
bool
HioChannel::Open(const char* port) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(port);
    NN_SDK_ASSERT(port[0] != '\0');

    if(IsOpened())
    {
        return true;
    }

    HIOHandle handle = HIOOpen(port, ConnectionCallback, this, HIO_CHANNEL_OPTION_READ_WRITE, 0);

    if(handle < 0)
    {
        NN_DETAIL_SPY_WARN("[%s] failed to HioOpen : port=%s.\n", NN_CURRENT_FUNCTION_NAME, port);
        return false;
    }

    String::Copy(m_Port, MaxPortLength, port);
    m_Handle = handle;

    m_ConnectEvent.Reset();

#ifdef STATUS_DEBUG_ENABLED
    NN_DETAIL_SPY_INFO(
        "[nn::spy::detail::fnd::HioChannel] Open(%s) : isOpened=%d, isConnected=%d.\n",
        port,
        IsOpened(),
        IsConnected());
#endif

    return true;
}

//----------------------------------------------------------
void
HioChannel::Close() NN_NOEXCEPT
{
    if(m_Handle != InvalidHandle)
    {
        HIOClose(m_Handle);

        m_Handle = InvalidHandle;
        m_Port[0] = '\0';
        m_IsConnected = false;

#ifdef STATUS_DEBUG_ENABLED
        NN_DETAIL_SPY_INFO(
            "[nn::spy::detail::fnd::HioChannel] Close : isOpened=%d, isConnected=%d.\n",
            IsOpened(),
            IsConnected());
#endif

        m_ConnectEvent.Set();
    }
}

//----------------------------------------------------------
bool
HioChannel::Connect() NN_NOEXCEPT
{
    while(IsOpened() && !m_IsConnected)
    {
        // コールバックを受け取るまで待機する
        m_ConnectEvent.Wait();
    }

    return m_IsConnected;
}

//----------------------------------------------------------
void
HioChannel::Disconnect() NN_NOEXCEPT
{
    // 何もしない
}

//----------------------------------------------------------
bool
HioChannel::IsOpened() const NN_NOEXCEPT
{
    return m_Port[0] != '\0';
}

//----------------------------------------------------------
bool
HioChannel::IsConnected() const NN_NOEXCEPT
{
    return m_IsConnected;
}

//----------------------------------------------------------
const char*
HioChannel::GetPort() const NN_NOEXCEPT
{
    return m_Port;
}

//----------------------------------------------------------
size_t
HioChannel::Read(void* buffer, size_t length) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(buffer);

    if(!IsOpened())
    {
        return 0;
    }

    if(length == 0)
    {
        return 0;
    }

    HIOStatus result = HIO_STATUS_OK;

    for(;;)
    {
        result = HIORead(m_Handle, length, buffer);

        if(result != HIO_STATUS_NO_CLIENT_TXN_BUF_AVAILABLE)
        {
            if (result < 0)
            {
                // 読み込みに失敗しました。
                return static_cast<size_t>(-1);
            }

            break;
        }
    }

    return result;
}

//----------------------------------------------------------
size_t
HioChannel::Write(const void* buffer, size_t length) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(buffer);

    if(!IsOpened())
    {
        return 0;
    }

    if(length == 0)
    {
        return 0;
    }

    HIOStatus result = HIO_STATUS_OK;

    for(;;)
    {
        result = HIOWrite(m_Handle, length, buffer);

        if(result != HIO_STATUS_NO_CLIENT_TXN_BUF_AVAILABLE)
        {
            break;
        }
    }

    return result;
}

//----------------------------------------------------------
void
HioChannel::ConnectionCallback(HIOStatus status, void* context) NN_NOEXCEPT
{
    HioChannel* target = reinterpret_cast<HioChannel*>(context);
    NN_SDK_ASSERT_NOT_NULL(target);

#ifdef STATUS_DEBUG_ENABLED
    NN_DETAIL_SPY_INFO("[nn::spy::detail::fnd::HioChannel] ConnectionCallback : port=%s, status=%d.\n", target->m_Port, status);
#endif

    // 新たに接続された。
    if(status == HIO_STATUS_NEW_CONNECTION)
    {
        target->m_IsConnected = true;
        target->m_ConnectEvent.Set();

#ifdef STATUS_DEBUG_ENABLED
        NN_DETAIL_SPY_INFO(
            "[nn::spy::detail::fnd::HioChannel] HIO_STATUS_NEW_CONNECTION : port=%s, isOpened=%d, isConnected=%d.\n",
            target->m_Port,
            target->IsOpened(),
            target->IsConnected());
#endif
        return;
    }

    // すべての接続が切断された。
    // Cafe の HIO では、複数の PC アプリが同一ポートに接続可能。
    if(status == HIO_STATUS_NO_CONNECTIONS )
    {
        target->m_IsConnected = false;

#ifdef STATUS_DEBUG_ENABLED
        NN_DETAIL_SPY_INFO(
            "[nn::spy::detail::fnd::HioChannel] HIO_STATUS_NO_CONNECTIONS : port=%s, isOpened=%d, isConnected=%d.\n",
            target->m_Port,
            target->IsOpened(),
            target->IsConnected());
#endif

        target->Close();
        target->m_ConnectEvent.Set();
        return;
    }

    // エラー。
    if(status < 0)
    {
        target->m_IsConnected = false;

#ifdef STATUS_DEBUG_ENABLED
        NN_DETAIL_SPY_INFO(
            "[nn::spy::detail::fnd::HioChannel] ConnectionCallback : port=%s, isOpened=%d, isConnected=%d.\n",
            target->m_Port,
            target->IsOpened(),
            target->IsConnected());
#endif
        return;
    }
}

} // namespace nn::spy::detail::fnd
} // namespace nn::spy::detail
} // namespace nn
} // namespace nn

#endif // NN_BUILD_CONFIG_SPY_ENABLED
