﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/spy/audio/spy_AudioSpyModule.h>

#ifdef NN_BUILD_CONFIG_SPY_ENABLED

#include <new>

#include <nn/nn_SdkAssert.h>
#include <nn/util/util_BytePtr.h>
#include <nn/os/os_Tick.h>
#include <nn/spy/audio/detail/spy_PlotAudioPerformanceMetrics.h>
#include <nn/spy/audio/detail/spy_WaveformModule.h>
#include <nn/spy/detail/fnd/basis/spyfnd_Inlines.h>

namespace nn {
namespace spy {
namespace audio {

namespace {

    const detail::PlotAudioPerformanceMetrics::InitializeArg
        ToPlotAudioPerformanceMetricsInitializeArg(
            const AudioSpyModule::InitializeArg& args) NN_NOEXCEPT
    {
        detail::PlotAudioPerformanceMetrics::InitializeArg initializeArg;
        initializeArg.pAudioRendererParameter = args.pAudioRendererParameter;
        return initializeArg;
    }

    const detail::WaveformModule::InitializeArg
        ToWaveformModuleInitializeArg(
            const AudioSpyModule::InitializeArg& args) NN_NOEXCEPT
    {
        detail::WaveformModule::InitializeArg initializeArg;
        initializeArg.pAudioRendererParameter = args.pAudioRendererParameter;
        initializeArg.pWaveformChannelTypes = args.pWaveformChannelTypes;
        initializeArg.waveformChannelCount = args.waveformChannelCount;
        initializeArg.waveformSampleFormat = args.waveformSampleFormat;
        return initializeArg;
    }

}

// 現在 AudioSpyModule は固有のデータを使用していないためデータの名前とバージョンに意味はありません。
// バージョン 0.0.0.0 は不正なので 0.0.0.1 としてあります。
AudioSpyModule::AudioSpyModule() NN_NOEXCEPT
    : nn::spy::SpyModule("NnSpyAudio", NN_DETAIL_SPY_FND_BINARY_MAKE_U32_SIGNATURE(0, 0, 0, 1))
    , m_pPlotAudioPerformanceMetrics(nullptr)
    , m_pWaveformModule(nullptr)
    , m_IsInitialized(false)
{
}

size_t AudioSpyModule::GetRequiredMemorySize(
    const InitializeArg& args) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(args.pAudioRendererParameter);

    return sizeof(detail::PlotAudioPerformanceMetrics)
        + detail::PlotAudioPerformanceMetrics::GetRequiredMemorySize(
            ToPlotAudioPerformanceMetricsInitializeArg(args))
        + sizeof(detail::WaveformModule);
}

bool AudioSpyModule::Initialize(
    const InitializeArg& args,
    void* buffer,
    size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!IsInitialized());
    NN_SDK_REQUIRES(!IsRegistered());
    NN_SDK_REQUIRES_NOT_NULL(args.pAudioRendererParameter);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_ALIGNED(buffer, ::nn::DefaultAlignment);
    NN_SDK_REQUIRES(bufferSize >= GetRequiredMemorySize(args));

    NN_UNUSED(bufferSize); // Release ビルドで Spy を有効にした場合

    m_IsInitialized = true;

    nn::util::BytePtr bytePtr(buffer);

    {
        const auto initializeArg = ToPlotAudioPerformanceMetricsInitializeArg(args);
        auto* pPlot = new(bytePtr.Get()) detail::PlotAudioPerformanceMetrics();
        bytePtr.Advance(sizeof(detail::PlotAudioPerformanceMetrics));
        size_t plotBufferSize = detail::PlotAudioPerformanceMetrics::GetRequiredMemorySize(initializeArg);
        void* plotBuffer = bytePtr.Get();
        bytePtr.Advance(plotBufferSize);
        if (!pPlot->Initialize(initializeArg, plotBuffer, plotBufferSize))
        {
            Finalize();
            return false;
        }
        m_pPlotAudioPerformanceMetrics = pPlot;
    }

    {
        const auto initializeArg = ToWaveformModuleInitializeArg(args);
        auto* pWaveformModule = new(bytePtr.Get()) detail::WaveformModule();
        bytePtr.Advance(sizeof(detail::WaveformModule));
        if (!pWaveformModule->Initialize(initializeArg))
        {
            Finalize();
            return false;
        }
        m_pWaveformModule = pWaveformModule;
    }

    return true;
}

void AudioSpyModule::Finalize() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(!IsRegistered());

    if (m_pPlotAudioPerformanceMetrics != nullptr)
    {
        m_pPlotAudioPerformanceMetrics->Finalize();
        m_pPlotAudioPerformanceMetrics = nullptr;
    }

    if (m_pWaveformModule != nullptr)
    {
        m_pWaveformModule->Finalize();
        m_pWaveformModule = nullptr;
    }

    m_IsInitialized = false;
}

bool AudioSpyModule::IsInitialized() const NN_NOEXCEPT
{
    return m_IsInitialized;
}

bool AudioSpyModule::PushPerformanceMetrics(
    const void* performanceFrameBuffer,
    size_t performanceFrameBufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());

    return PushPerformanceMetrics(
        performanceFrameBuffer,
        performanceFrameBufferSize,
        nn::os::GetSystemTick());
}

bool AudioSpyModule::PushPerformanceMetrics(
    const void* performanceFrameBuffer,
    size_t performanceFrameBufferSize,
    nn::os::Tick tick) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_ASSERT_NOT_NULL(m_pPlotAudioPerformanceMetrics);

    return m_pPlotAudioPerformanceMetrics->PushPerformanceMetrics(
        performanceFrameBuffer,
        performanceFrameBufferSize,
        tick);
}

bool AudioSpyModule::PushWaveform(
    const void* buffer,
    size_t bufferSize,
    nn::os::Tick tick) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_ASSERT_NOT_NULL(m_pWaveformModule);

    if(m_pWaveformModule->IsRegistered() == false)
    {
        return false;
    }

    if (bufferSize == 0)
    {
        return true;
    }

    return m_pWaveformModule->PushWaveform(buffer, bufferSize, tick);
}

void AudioSpyModule::OnRegistered() NN_NOEXCEPT
{
    Base::OnRegistered();

    NN_SDK_ASSERT(IsInitialized());
    NN_SDK_ASSERT_NOT_NULL(GetController());
    NN_SDK_ASSERT_NOT_NULL(m_pPlotAudioPerformanceMetrics);
    NN_SDK_ASSERT_NOT_NULL(m_pWaveformModule);

    m_pPlotAudioPerformanceMetrics->AttachToModules(*GetController());

    GetController()->RegisterModule(*m_pWaveformModule);
}

void AudioSpyModule::OnUnregistered() NN_NOEXCEPT
{
    Base::OnUnregistered();

    NN_SDK_ASSERT(IsInitialized());
    NN_SDK_ASSERT_NOT_NULL(GetController());
    NN_SDK_ASSERT_NOT_NULL(m_pPlotAudioPerformanceMetrics);
    NN_SDK_ASSERT_NOT_NULL(m_pWaveformModule);

    m_pPlotAudioPerformanceMetrics->DetachFromModules();

    GetController()->UnregisterModule(*m_pWaveformModule);
}

} // namespace audio
} // namespace spy
} // namespace nn

#endif
