﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ige/ige_Node.h>
#include <nn/ige/ige_UiService.h>
#include <nn/ige/ige_ControlFactory.h>
#include <nn/ige/ige_BufferWriter.h>
#include <nn/ige/ige_GroupBox.h>
#include <nn/ige/ige_Panel.h>
#include "detail/ige_Allocator.h"

namespace nn { namespace ige {

void ControlFactory::InitializeFactory(detail::IgeAllocator* pAllocator, NodeHolder* pParent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAllocator);
    NN_SDK_REQUIRES_NOT_NULL(pParent);
    m_pAllocator = pAllocator;
    m_pParent = pParent;
}

void ControlFactory::FinalizeFactory() NN_NOEXCEPT
{
    m_pAllocator = NULL;
    m_pParent = NULL;
}

Button* ControlFactory::CreateButton(ButtonOption option) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAllocator);

    void* buffer = m_pAllocator->Allocate<Button>();
    auto pButton = new (buffer) Button(option);
    RegisterNode(&pButton->m_Node);

    return pButton;
}

TextBox* ControlFactory::CreateTextBox(char* buffer, size_t size, TextBoxOption option) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAllocator);

    void* buf = m_pAllocator->Allocate<TextBox>();
    auto pTextBox = new (buf) TextBox(buffer, size, option);
    RegisterNode(&pTextBox->m_Node);

    return pTextBox;
}

Label* ControlFactory::CreateLabel(LabelOption option) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAllocator);

    void* buf = m_pAllocator->Allocate<Label>();
    auto pLabel = new (buf) Label(option);
    RegisterNode(&pLabel->m_Node);

    return pLabel;
}

CheckBox* ControlFactory::CreateCheckBox(bool* pIsChecked, CheckBoxOption option) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAllocator);

    void* buffer = m_pAllocator->Allocate<CheckBox>();
    auto pCheckBox = new (buffer) CheckBox(pIsChecked, option);
    RegisterNode(&pCheckBox->m_Node);

    return pCheckBox;
}

Slider<int>* ControlFactory::CreateSlider(int* pValue, int min, int max, SliderOption option) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAllocator);
    NN_SDK_REQUIRES(min <= max);

    void* buffer = m_pAllocator->Allocate<Slider<int>>();
    auto pSlider = new (buffer) Slider<int>(pValue, min, max, option);
    RegisterNode(&pSlider->m_Node);

    return pSlider;
}

Slider<float>* ControlFactory::CreateSlider(float* pValue, float min, float max, SliderOption option) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAllocator);
    NN_SDK_REQUIRES(min <= max);

    void* buffer = m_pAllocator->Allocate<Slider<float>>();
    auto pSlider = new (buffer) Slider<float>(pValue, min, max, option);
    RegisterNode(&pSlider->m_Node);

    return pSlider;
}

GroupBox* ControlFactory::CreateGroupBox(GroupBoxOption option) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAllocator);

    void* buffer = m_pAllocator->Allocate<GroupBox>();
    auto pGroupBox = new (buffer) GroupBox(m_pAllocator, option);
    RegisterNode(&pGroupBox->m_Node);

    return pGroupBox;
}

Panel* ControlFactory::CreatePanel(PanelOption option) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAllocator);

    void* buffer = m_pAllocator->Allocate<Panel>();
    auto pPanel = new (buffer) Panel(m_pAllocator, option);
    RegisterNode(&pPanel->m_Node);

    return pPanel;
}

bool ControlFactory::Contains(const Control* pControl) const NN_NOEXCEPT
{
    if (pControl == NULL) return false;

    for (Node* pCurrent = m_pParent->GetChild(); pCurrent != NULL; pCurrent = pCurrent->GetNext())
    {
        Control* pTarget = pCurrent->GetUserData<Control>();
        if (pTarget == pControl)
        {
            return true;
        }
    }
    return false;
}

bool ControlFactory::Delete(Control* pControl) NN_NOEXCEPT
{
    if (m_pAllocator == NULL || !Contains(pControl))
    {
        return false;
    }

    pControl->~Control();
    m_pAllocator->Free(pControl);
    return true;
}

void ControlFactory::DeleteAll() NN_NOEXCEPT
{
    if (m_pAllocator == NULL)
    {
        return;
    }

    for (Node* pCurrent = m_pParent->GetChild(), *pNext; pCurrent != NULL; pCurrent = pNext)
    {
        pNext = pCurrent->GetNext();

        Control* pControl = pCurrent->GetUserData<Control>();
        pControl->~Control();
        m_pAllocator->Free(pControl);
    }
}

void ControlFactory::RegisterNode(Node* pNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pParent);
    NN_SDK_REQUIRES_NOT_NULL(pNode);
    m_pParent->Register(pNode);
}

}} // namespace nn::ige
