﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_BytePtr.h>
#include <nn/ige/ige_BufferWriter.h>
#include "detail/ige_Allocator.h"

namespace nn { namespace ige {

    BufferWriter::BufferWriter(MemoryResource* pResource) NN_NOEXCEPT
        : m_pResource(pResource)
        , m_WorkBuffer()
        , m_Size(0)
    {
        NN_SDK_ASSERT_NOT_NULL(m_pResource);
    }

    BufferWriter::BufferWriter(BufferReference buffer) NN_NOEXCEPT
        : m_pResource(NULL)
        , m_WorkBuffer(buffer)
        , m_Size(0)
    {
        NN_SDK_ASSERT_NOT_NULL(buffer.GetPtr());
    }

    BufferWriter::~BufferWriter() NN_NOEXCEPT
    {
        if (m_pResource)
        {
            if (void* ptr = m_WorkBuffer.GetPtr())
            {
                m_pResource->deallocate(ptr, m_WorkBuffer.GetSize());
                m_WorkBuffer.Set(NULL, 0);
            }

            m_pResource = NULL;
        }
    }

namespace {
    const int s_ExpandRate = 2;
} // namespace anonymous

    void BufferWriter::Write(const void* ptr, int startIndex, size_t count) NN_NOEXCEPT
    {
        size_t currentSize = m_WorkBuffer.GetSize();
        size_t requiredSize = m_Size + count;

        if (currentSize < requiredSize)
        {
            size_t newSize = std::max NN_PREVENT_MACRO_FUNC (requiredSize, currentSize * s_ExpandRate);
            ExpandBuffer(newSize);
        }

        util::BytePtr dstPtr = util::BytePtr(m_WorkBuffer.GetPtr(), static_cast<ptrdiff_t>(m_Size));
        util::ConstBytePtr srcPtr = util::ConstBytePtr(ptr, static_cast<ptrdiff_t>(startIndex));

        std::memcpy(dstPtr.Get(), srcPtr.Get(), count);
         m_Size += count;
    }

    void BufferWriter::ExpandBuffer(size_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pResource);

        BufferReference oldBuffer = m_WorkBuffer;

        size_t currentSize = oldBuffer.GetSize();

        if (currentSize >= size)
        {
            return;
        }

        BufferReference newBuffer(m_pResource->allocate(size), size);

        if (void* ptr = oldBuffer.GetPtr())
        {
            std::memcpy(newBuffer.GetPtr(), oldBuffer.GetPtr(), m_Size);
            m_pResource->deallocate(ptr, oldBuffer.GetSize());
        }

        m_WorkBuffer = newBuffer;
    }

}} // namespace nn::ige
