﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/util/util_StringUtil.h>
#include <nn/htcs.h>

#include "util/ige_ScopedLock.h"
#include "ige_ServerSocketHtcs.h"
#include "ige_DetailDefine.h"

namespace {
    const int s_InvalidDescriptor = -1;
} // anonymous namespace

namespace nn { namespace ige { namespace detail {

NN_DEFINE_STATIC_CONSTANT( const int ServerSocketHtcs::CHANNEL_NAME_LENGTH );

ServerSocketHtcs::ServerSocketHtcs() NN_NOEXCEPT
    : m_ServerDescriptor(s_InvalidDescriptor)
    , m_ClientDescriptor(s_InvalidDescriptor)
    , m_WriteMutex(true)
    , m_ReadMutex(true)
{
}

bool ServerSocketHtcs::InitializeInternal(const SetupArg& arg) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(::nn::htcs::IsInitialized());

    m_ServerDescriptor = s_InvalidDescriptor;
    m_ClientDescriptor = s_InvalidDescriptor;

    int nameLength = static_cast<int>(arg.channelName.length());

    NN_SDK_REQUIRES_RANGE(nameLength, 1, CHANNEL_NAME_LENGTH);

    // 1 + null終端分 <= count <= CHANNEL_NAME_LENGTH
    int count = std::min NN_PREVENT_MACRO_FUNC (nameLength + 1, CHANNEL_NAME_LENGTH);
    util::Strlcpy(m_ChannelName, arg.channelName.data(), count);

    return true;
}

bool ServerSocketHtcs::OpenInternal() NN_NOEXCEPT
{
    if (m_ServerDescriptor != s_InvalidDescriptor)
    {
        return false;
    }

    // ソケットの作成
    int newSocket = ::nn::htcs::Socket();
    if (newSocket == -1)
    {
        return false;
    }
    m_ServerDescriptor = newSocket;

    // ノンブロッキングモードに設定
    nn::htcs::FcntlFlag  nonBlock = nn::htcs::HTCS_O_NONBLOCK;
    ::nn::htcs::Fcntl(m_ServerDescriptor, nn::htcs::HTCS_F_SETFL, nonBlock);

    // ソケットの設定
    nn::htcs::SockAddrHtcs addr = {};
    addr.family = nn::htcs::HTCS_AF_HTCS;
    util::Strlcpy(addr.peerName.name, ::nn::htcs::GetPeerNameAny().name, nn::htcs::PeerNameBufferLength);
    util::Strlcpy(addr.portName.name, m_ChannelName, nn::htcs::PortNameBufferLength);

    ::nn::htcs::Bind(m_ServerDescriptor, &addr);

    // TCPクライアントからの接続要求を待てる状態にする
    int listenResult = ::nn::htcs::Listen(m_ServerDescriptor, 5);
    if (listenResult == -1)
    {
        NN_IGE_WARNING_LOG("Socket open error %d\n", ::nn::htcs::GetLastError());
        CloseSocket();
        return false;
    }

    return true;
}

void ServerSocketHtcs::CloseSocket() NN_NOEXCEPT
{
    if (m_ServerDescriptor != s_InvalidDescriptor)
    {
        ::nn::htcs::Close(m_ServerDescriptor);
        m_ServerDescriptor = s_InvalidDescriptor;
    }
}

void ServerSocketHtcs::CloseClientSocket() NN_NOEXCEPT
{
    ResetReadFlag();
    ResetWriteFlag();

    if (IsConnected())
    {
        ::nn::htcs::Close(m_ClientDescriptor);
        m_ClientDescriptor = s_InvalidDescriptor;
    }
}

void ServerSocketHtcs::CloseInternal() NN_NOEXCEPT
{
    CloseClientSocket();
    CloseSocket();
}

// 処理を開始できたかどうかを返す。
bool ServerSocketHtcs::WriteAsyncInternal(const void* buf, size_t size) NN_NOEXCEPT
{
    ScopedLock locker(&m_WriteMutex);
    NN_UNUSED(locker);

    if (!IsConnected())
    {
        return false;
    }

    if (IsWriting())
    {
        return false;
    }

    m_pWriteBuffer = static_cast<const uint8_t*>(buf);
    m_WriteBufferSize = size;
    m_WriteSizeRemained = size;

    return WriteAsyncForPoll();
}

// 正常に処理を継続している状態かどうかを返す。
bool ServerSocketHtcs::WriteAsyncForPoll() NN_NOEXCEPT
{
    ScopedLock locker(&m_WriteMutex);
    NN_UNUSED(locker);

    if (!IsConnected())
    {
        return false;
    }

    if (!IsWriting()) // 書き込むべきものがない場合は処理を抜ける
    {
        return true;
    }

    int bufferSize = static_cast<int>(m_WriteSizeRemained);
    const void* buffer = m_pWriteBuffer + m_WriteBufferSize - bufferSize;

    nn::htcs::ssize_t result = ::nn::htcs::Send(m_ClientDescriptor, buffer, bufferSize, 0);

    if (m_IsWriteLogEnabled)
    {
        NN_IGE_DEBUG_PRINT("Send: result = %zd\n", result);
    }

    if (result <= 0)
    {
        bool isGracefulClose = result == 0; // ソケットが使えなくなっている場合は、ソケットを閉じる

        if (::nn::htcs::GetLastError() != nn::htcs::HTCS_EAGAIN || isGracefulClose )
        {
            ResetWriteFlag();
            CloseClientSocket();

            return false;
        }

        return true;
    }

    m_WriteSizeRemained -= static_cast<size_t>(result);
    return true;
}

// 処理を開始できたかどうかを返す。
bool ServerSocketHtcs::ReadAsyncInternal(void* buf, size_t size) NN_NOEXCEPT
{
    ScopedLock locker(&m_ReadMutex);
    NN_UNUSED(locker);

    if (!IsConnected())
    {
        return false;
    }

    if (IsReading())
    {
        return false;
    }

    m_pReadBuffer = static_cast<uint8_t*>(buf);
    m_ReadBufferSize = size;
    m_ReadSizeRemained = size;

    return ReadAsyncForPoll();
}

// 正常に処理を継続している状態かどうかを返す。
bool ServerSocketHtcs::ReadAsyncForPoll() NN_NOEXCEPT
{
    ScopedLock locker(&m_ReadMutex);
    NN_UNUSED(locker);

    if (!IsConnected()) // ツール側から通信を切った時の対処
    {
        return false;
    }

    if (!IsReading()) // 読み込むべきものがない場合は処理を抜ける
    {
        return true;
    }

    int bufferSize = static_cast<int>(m_ReadSizeRemained);
    void* buffer = m_pReadBuffer + m_ReadBufferSize - bufferSize;

    nn::htcs::ssize_t result = ::nn::htcs::Recv(m_ClientDescriptor, buffer, bufferSize, 0);

    if (result <= 0)
    {
        // ソケットが使えなくなっている場合は、ソケットを閉じる
        bool isGracefulClose = result == 0;
        if (::nn::htcs::GetLastError() != nn::htcs::HTCS_EAGAIN || isGracefulClose)
        {
            ResetReadFlag();
            CloseClientSocket();

            return false;
        }

        return true;
    }

    m_ReadSizeRemained -= static_cast<size_t>(result);
    return true;
}

void ServerSocketHtcs::PollInternal() NN_NOEXCEPT
{
    // サーバーソケットが無効ならば、処理を抜ける
    if (m_ServerDescriptor == s_InvalidDescriptor)
    {
        return;
    }

    // クライアントの接続ができている場合は処理を抜ける
    if (IsConnected())
    {
        if (ReadAsyncForPoll() && WriteAsyncForPoll())
        {
            return;
        }

        CloseClientSocket();
    }

    // TCPクライアントからの接続要求を受け付ける
    nn::htcs::SockAddrHtcs client;
    int clientSocket = ::nn::htcs::Accept(m_ServerDescriptor, &client);
    if (clientSocket != -1)
    {
        m_ClientDescriptor = clientSocket;
    }
}

bool ServerSocketHtcs::IsConnected() const NN_NOEXCEPT
{
    return m_ClientDescriptor != s_InvalidDescriptor;
}

}}} // namespace nn::ige::detail
