﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>

#include <nn/util/util_Endian.h>

#include "ige_Channel.h"
#include "ige_ServerSocketHtcs.h"
#include "ige_DetailDefine.h"
#include "util/ige_ScopedLock.h"

namespace nn { namespace ige { namespace detail {

Channel::Channel(const util::string_view& channelName, IgeAllocator* pAllocator) NN_NOEXCEPT
    : m_pAllocator(pAllocator)
    , m_pSocket(NULL)
    , m_WriteMutex(true)
    , m_ReadMutex(true)
    , m_ReadPacketQueue(*pAllocator)
    , m_ReadState(ReadState_Ready)
    , m_pReadingPayload(NULL)
    , m_Signature()
    , m_HasSignature(false)
{
    NN_SDK_REQUIRES_NOT_NULL(pAllocator);

    Initialize(channelName);
}

Channel::Channel(const util::string_view& channelName, IgeAllocator* pAllocator, int32_t signature) NN_NOEXCEPT
    : m_pAllocator(pAllocator)
    , m_pSocket(NULL)
    , m_WriteMutex(true)
    , m_ReadMutex(true)
    , m_ReadPacketQueue(*pAllocator)
    , m_ReadState(ReadState_Ready)
    , m_pReadingPayload(NULL)
    , m_Signature(signature)
    , m_HasSignature(true)
{
    NN_SDK_REQUIRES_NOT_NULL(pAllocator);

    Initialize(channelName);
}

Channel::~Channel() NN_NOEXCEPT
{
    Close();

    m_pSocket->~ServerSocket();
    m_pAllocator->Free(m_pSocket);
    m_pSocket = NULL;
    m_pAllocator = NULL;
}

void Channel::Initialize(const util::string_view& channelName) NN_NOEXCEPT
{
    void* buffer = m_pAllocator->Allocate<ServerSocketHtcs>();
    m_pSocket = new (buffer) ServerSocketHtcs();

    ServerSocket::SetupArg arg;
    arg.channelName = channelName;
    m_pSocket->Initialize(arg);
    m_pSocket->RequestOpen();
    m_pSocket->Open();
    ResetReadState();
}

bool Channel::IsOpened() const NN_NOEXCEPT
{
    return m_pSocket->IsOpened();
}

bool Channel::IsConnected() const NN_NOEXCEPT
{
    return m_pSocket->IsConnected();
}

void Channel::Close() NN_NOEXCEPT
{
    ScopedLock readLock(&m_ReadMutex);
    ScopedLock writeLock(&m_WriteMutex);
    NN_UNUSED(readLock);
    NN_UNUSED(writeLock);

    m_pSocket->Close();
    ResetReadState();

    for (auto buffer : m_ReadPacketQueue)
    {
        m_pAllocator->FreeBuffer(&buffer);
    }
    m_ReadPacketQueue.clear();
}

void Channel::ResetConnection() NN_NOEXCEPT
{
    Close();
    m_pSocket->RequestOpen();
}

void Channel::ResetReadState() NN_NOEXCEPT
{
    m_ReadingSignature = 0;
    m_ReadingLength = 0;
    m_pAllocator->Free(m_pReadingPayload);
    m_pReadingPayload = NULL;
    m_ReadState = ReadState_Ready;
}

void Channel::SendPacket(const BufferView& packet) NN_NOEXCEPT
{
    if (!IsConnected())
    {
        return;
    }

    ScopedLock lock(&m_WriteMutex);
    NN_UNUSED(lock);

    if (!WritePacket(packet))
    {
        ResetConnection();
    }
}

void Channel::Poll() NN_NOEXCEPT
{
    m_pSocket->Poll();

    if (!IsConnected())
    {
        return;
    }

    {
        ScopedLock lock(&m_ReadMutex);
        NN_UNUSED(lock);

        // とりあえず読めるだけ全部読む。重そうなら割合で上限数を設ける
        while (ReadPacket())
        {
            BufferReference buffer(m_pReadingPayload, m_ReadingLength);
            m_ReadPacketQueue.push_back(buffer);

            // 次回読み込み時の ResetReadState で pReadingPayload が指す領域を解放しないように NULL をセットしておく。
            // pReadingPayload が指す領域は ReceivePacket 時に読み取り処理の後で解放される。
            m_pReadingPayload = NULL;
        }
    }

    // ReadPacket() 中に ResetConnection() する場合があるのでもう一回接続チェック
    if (!IsConnected())
    {
        return;
    }

    // 接続状態確認のためのテストパケットを送る。受信側では無視される
    if (!WriteTestPacket())
    {
        ResetConnection();
    }
}

// 送信時は現状は全て同期処理。非同期にする場合は WriteState を作って管理
bool Channel::WritePacket(const BufferView& buffer) NN_NOEXCEPT
{
    if (m_HasSignature && !m_pSocket->Write(&m_Signature, sizeof(m_Signature)))
    {
        return false;
    }

    auto length = static_cast<int32_t>(buffer.GetSize());
    length = util::LoadLittleEndian(&length);

    if (!m_pSocket->Write(&length, sizeof(length)))
    {
        return false;
    }

    return length > 0 ? m_pSocket->Write(buffer.GetPtr(), buffer.GetSize()) : true;
}

bool Channel::WriteTestPacket() NN_NOEXCEPT
{
    // length = 0 のパケット
    BufferView dummy;
    return WritePacket(dummy);
}

bool Channel::ReadPacket() NN_NOEXCEPT
{
    if (!IsConnected())
    {
        ResetReadState();
        return false;
    }

    if (m_ReadState == ReadState_Ready)
    {
        ResetReadState();
        m_ReadState = m_HasSignature ? ReadState_Signature : ReadState_Length;
    }

    if (m_ReadState == ReadState_Signature)
    {
        m_ReadState = m_pSocket->ReadAsync(&m_ReadingSignature, sizeof(m_ReadingSignature)) ?
            ReadState_SignatureReading :
            ReadState_Invalid;
    }

    if (m_ReadState == ReadState_SignatureReading)
    {
        if (m_pSocket->IsReading())
        {
            return false;
        }

        if (m_ReadingSignature != m_Signature)
        {
            NN_IGE_WARNING_LOG("received signature '%d' is not met. expected '%d'.", m_ReadingSignature, m_Signature);
            m_ReadState = ReadState_Invalid;
        }

        m_ReadState = ReadState_Length;
    }

    if (m_ReadState == ReadState_Length)
    {
        m_ReadState = m_pSocket->ReadAsync(&m_ReadingLength, sizeof(m_ReadingLength)) ?
            ReadState_LengthReading :
            ReadState_Invalid;
    }

    if (m_ReadState == ReadState_LengthReading)
    {
        if (m_pSocket->IsReading())
        {
            return false;
        }

        m_pAllocator->Free(m_pReadingPayload);

        m_ReadingLength = util::LoadLittleEndian(&m_ReadingLength);
        if (m_ReadingLength == 0)
        {
            m_ReadState = ReadState_Ready;
            return true;
        }

        m_pReadingPayload = m_pAllocator->Allocate(m_ReadingLength);

        m_ReadState = m_pSocket->ReadAsync(m_pReadingPayload, m_ReadingLength) ?
            ReadState_PayloadReading :
            ReadState_Invalid;
    }

    if (m_ReadState == ReadState_PayloadReading)
    {
        if (m_pSocket->IsReading())
        {
            return false;
        }

        // ここでクリアすると読み込み完了したバッファを利用できないので
        // 次に ReadPacket() が呼ばれるときにバッファをクリアする。
        m_ReadState = ReadState_Ready;
        return true;
    }

    // ReadState_Invalid
    ResetConnection();
    return false;
}

}}} // namespace nn::ige::detail
