﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/g3d/viewer/g3d_ViewerServer.h>
#include <nn/diag/text/diag_SdkTextG3dviewer.h>

#if !NN_DETAIL_G3D_VIEWER_CONFIG_IS_VIEWER_EMPTY

#include "detail/g3d_ViewerServerImpl.h"
#include "detail/g3d_ViewerDetailDefine.h"
#include "detail/g3d_Allocator.h"
#include "detail/g3d_ResourceManager.h"
#include <nn/g3d/viewer/g3d_ViewerCallbackUtility.h>

#include <nn/g3d/g3d_ResShader.h>
#include <nn/g3d/g3d_ModelObj.h>

#include <nn/nn_SdkAssert.h>
#include <nn/fs.h>

#include <nn/htcs.h>

//--------------------------------------------------------------------------------------------------
// ミドルウェア情報
#include <nn/nn_Middleware.h>
#include <nn/nn_Version.h>

#define NN_DETAIL_XX_MACRO_TO_STRING(x) NN_DETAIL_XX_MACRO_TO_STRING_(x)
#define NN_DETAIL_XX_MACRO_TO_STRING_(x) #x

#define NW_G3D_VIEWER_MIDDLEWARE_SYMBOL(buildOption) "NintendoWare_G3d_Viewer_For_Develop-" NN_DETAIL_XX_MACRO_TO_STRING(NN_NX_ADDON_VERSION_MAJOR) "_" \
NN_DETAIL_XX_MACRO_TO_STRING(NN_NX_ADDON_VERSION_MINOR) "_" NN_DETAIL_XX_MACRO_TO_STRING(NN_NX_ADDON_VERSION_MICRO) "-" #buildOption

namespace {
#if defined(NN_SDK_BUILD_DEBUG)
    NN_DEFINE_MIDDLEWARE(g_MiddlewareInfo, "Nintendo", NW_G3D_VIEWER_MIDDLEWARE_SYMBOL(Debug));
#elif defined(NN_SDK_BUILD_DEVELOP)
    NN_DEFINE_MIDDLEWARE(g_MiddlewareInfo, "Nintendo", NW_G3D_VIEWER_MIDDLEWARE_SYMBOL(Develop));
#elif defined(NN_SDK_BUILD_RELEASE)
    NN_DEFINE_MIDDLEWARE(g_MiddlewareInfo, "Nintendo", NW_G3D_VIEWER_MIDDLEWARE_SYMBOL(Release));
#endif
}

//--------------------------------------------------------------------------------------------------

using namespace nn::g3d::viewer;
using namespace nn::g3d::viewer::detail;

// ユーザがテクスチャバインドをライブラリ任せにした際のバインドコールバック
nn::g3d::TextureRef ViewerServer::ViewerServerProxy::ViewerDefaultTextureBindCallback(
    const char* name, const nn::gfx::ResTextureFile* pResTextureFile, void* pUserData) NN_NOEXCEPT
{
    NN_UNUSED(pUserData);
    if (pResTextureFile == nullptr)
    {
        // ユーザがテクスチャ割り当てのないサンプラに対して、
        // ダミーテクスチャを割り当てられるように ResTextureFile が nullptr で
        // コールバックが呼ばれる場合がある
        return nn::g3d::TextureRef();
    }

    nn::g3d::TextureRef textureRef;
    const nn::util::ResDic* pDic = pResTextureFile->GetTextureDic();
    int index = pDic->FindIndex(name);
    if (index == nn::util::ResDic::Npos)
    {
        return textureRef;
    }

    const nn::gfx::ResTexture* pResTexture = pResTextureFile->GetResTexture(index);
    nn::gfx::DescriptorSlot textureDescriptorSlot;
    pResTexture->GetUserDescriptorSlot(&textureDescriptorSlot);
    textureRef.SetTextureView(pResTexture->GetTextureView());
    textureRef.SetDescriptorSlot(textureDescriptorSlot);
    return textureRef;
}

void ViewerServer::ViewerServerProxy::CallViewerCallbacks(void* pOutArg, const void* pInArg, CallbackType type, void* pUserData) NN_NOEXCEPT
{
    NN_UNUSED(pOutArg);
    NN_UNUSED(pInArg);

    ViewerCallbacks* pCallback = static_cast<ViewerCallbacks*>(pUserData);
    NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(pCallback, NN_TEXT_G3DVIEWER("初期化時にコールバッククラスをユーザデータに設定して下さい"));

    NN_G3D_VIEWER_DEBUG_PRINT("ViewerCallback %d called\n", type);

    switch (type)
    {
    case CallbackType_ModelFileLoaded:
        {
            const ModelFileLoadedArg* pCastedArg = static_cast<const ModelFileLoadedArg*>(pInArg);
            NN_G3D_VIEWER_ASSERT_NOT_NULL(pCastedArg);
            ModelFileLoadedOutArg* pCastedOutArg = static_cast<ModelFileLoadedOutArg*>(pOutArg);
            NN_G3D_VIEWER_ASSERT_NOT_NULL(pCastedOutArg);
            pCallback->ModelFileLoaded(*pCastedOutArg, *pCastedArg);
        }
        return;
    case CallbackType_ModelFileUnloaded:
        {
            const ModelFileUnloadedArg* pCastedArg = static_cast<const ModelFileUnloadedArg*>(pInArg);
            NN_G3D_VIEWER_ASSERT_NOT_NULL(pCastedArg);
            pCallback->ModelFileUnloaded(*pCastedArg);
        }
        return;
    case CallbackType_ModelAttached:
        {
            const ModelAttachedArg* pCastedArg = static_cast<const ModelAttachedArg*>(pInArg);
            NN_G3D_VIEWER_ASSERT_NOT_NULL(pCastedArg);
            pCallback->ModelAttached(*pCastedArg);
        }
        return;
    case CallbackType_ModelDetached:
        {
            const ModelDetachedArg* pCastedArg = static_cast<const ModelDetachedArg*>(pInArg);
            NN_G3D_VIEWER_ASSERT_NOT_NULL(pCastedArg);
            pCallback->ModelDetached(*pCastedArg);
        }
        return;
    case CallbackType_MaterialUpdated:
        {
            const MaterialUpdatedArg* pCastedArg = static_cast<const MaterialUpdatedArg*>(pInArg);
            NN_G3D_VIEWER_ASSERT_NOT_NULL(pCastedArg);
            pCallback->MaterialUpdated(*pCastedArg);
        }
        return;
    case CallbackType_ShaderAssignUpdated:
        {
            const ShaderAssignUpdatedArg* pCastedArg = static_cast<const ShaderAssignUpdatedArg*>(pInArg);
            NN_G3D_VIEWER_ASSERT_NOT_NULL(pCastedArg);
            pCallback->ShaderAssignUpdated(*pCastedArg);
        }
        return;
    case CallbackType_RenderInfoUpdated:
        {
            const RenderInfoUpdatedArg* pCastedArg = static_cast<const RenderInfoUpdatedArg*>(pInArg);
            NN_G3D_VIEWER_ASSERT_NOT_NULL(pCastedArg);
            pCallback->RenderInfoUpdated(*pCastedArg);
        }
        return;
    case CallbackType_SendRenderInfoRequested:
        {
            const SendRenderInfoRequestedArg* pCastedArg = static_cast<const SendRenderInfoRequestedArg*>(pInArg);
            NN_G3D_VIEWER_ASSERT_NOT_NULL(pCastedArg);
            pCallback->SendRenderInfoRequested(*pCastedArg);
        }
        return;
    case CallbackType_BoneBindUpdated:
        {
            const BoneBindUpdatedArg* pCastedArg = static_cast<const BoneBindUpdatedArg*>(pInArg);
            NN_G3D_VIEWER_ASSERT_NOT_NULL(pCastedArg);
            pCallback->BoneBindUpdated(*pCastedArg);
        }
        return;
    case CallbackType_ModelLayoutUpdated:
        {
            const ModelLayoutUpdatedArg* pCastedArg = static_cast<const ModelLayoutUpdatedArg*>(pInArg);
            NN_G3D_VIEWER_ASSERT_NOT_NULL(pCastedArg);
            pCallback->ModelLayoutUpdated(*pCastedArg);
        }
        return;
    case CallbackType_SendModelLayoutRequested:
        {
            SendModelLayoutRequestedOutArg* pCastedOutArg = static_cast<SendModelLayoutRequestedOutArg*>(pOutArg);
            NN_G3D_VIEWER_ASSERT_NOT_NULL(pCastedOutArg);
            const SendModelLayoutRequestedArg* pCastedArg = static_cast<const SendModelLayoutRequestedArg*>(pInArg);
            NN_G3D_VIEWER_ASSERT_NOT_NULL(pCastedArg);
            pCallback->SendModelLayoutRequested(*pCastedOutArg, *pCastedArg);
        }
        return;
    case CallbackType_TargetSelected:
        {
            const TargetSelectedArg* pCastedArg = static_cast<const TargetSelectedArg*>(pInArg);
            NN_G3D_VIEWER_ASSERT_NOT_NULL(pCastedArg);
            pCallback->TargetSelected(*pCastedArg);
        }
        return;
    case CallbackType_SamplerParamUpdated:
        {
            const SamplerParamUpdatedArg* pCastedArg = static_cast<const SamplerParamUpdatedArg*>(pInArg);
            NN_G3D_VIEWER_ASSERT_NOT_NULL(pCastedArg);
            pCallback->SamplerParamUpdated(*pCastedArg);
        }
        return;
    case CallbackType_ShapeUpdated:
        {
            const ShapeUpdatedArg* pCastedArg = static_cast<const ShapeUpdatedArg*>(pInArg);
            NN_G3D_VIEWER_ASSERT_NOT_NULL(pCastedArg);
            pCallback->ShapeUpdated(*pCastedArg);
        }
        return;
    case CallbackType_ModelUserScriptExecuted:
        {
            const ModelUserScriptExecutedArg* pCastedArg = static_cast<const ModelUserScriptExecutedArg*>(pInArg);
            NN_G3D_VIEWER_ASSERT_NOT_NULL(pCastedArg);
            pCallback->ModelUserScriptExecuted(*pCastedArg);
        }
        return;
    case CallbackType_SceneAnimBound:
        {
            const SceneAnimBoundArg* pCastedArg = static_cast<const SceneAnimBoundArg*>(pInArg);
            NN_G3D_VIEWER_ASSERT_NOT_NULL(pCastedArg);
            pCallback->SceneAnimBound(*pCastedArg);
        }
        return;
    case CallbackType_SceneAnimUnbound:
        {
            const SceneAnimUnboundArg* pCastedArg = static_cast<const SceneAnimUnboundArg*>(pInArg);
            NN_G3D_VIEWER_ASSERT_NOT_NULL(pCastedArg);
            pCallback->SceneAnimUnbound(*pCastedArg);
        }
        return;
    case CallbackType_ApplySceneAnimRequested:
        {
            const ApplySceneAnimRequestedArg* pCastedArg = static_cast<const ApplySceneAnimRequestedArg*>(pInArg);
            NN_G3D_VIEWER_ASSERT_NOT_NULL(pCastedArg);
            pCallback->ApplySceneAnimRequested(*pCastedArg);
        }
        return;
    case CallbackType_ShaderAttached:
        {
            const ShaderAttachedArg* pCastedArg = static_cast<const ShaderAttachedArg*>(pInArg);
            NN_G3D_VIEWER_ASSERT_NOT_NULL(pCastedArg);
            pCallback->ShaderAttached(*pCastedArg);
        }
        return;
    case CallbackType_ShaderDetached:
        {
            const ShaderDetachedArg* pCastedArg = static_cast<const ShaderDetachedArg*>(pInArg);
            NN_G3D_VIEWER_ASSERT_NOT_NULL(pCastedArg);
            pCallback->ShaderDetached(*pCastedArg);
        }
        return;
    case CallbackType_ShaderProgramUpdated:
        {
            const ShaderProgramUpdatedArg* pCastedArg = static_cast<const ShaderProgramUpdatedArg*>(pInArg);
            NN_G3D_VIEWER_ASSERT_NOT_NULL(pCastedArg);
            pCallback->ShaderProgramUpdated(*pCastedArg);
        }
        return;
    case CallbackType_ShadingModelUpdated:
        {
            const ShadingModelUpdatedArg* pCastedArg = static_cast<const ShadingModelUpdatedArg*>(pInArg);
            NN_G3D_VIEWER_ASSERT_NOT_NULL(pCastedArg);
            pCallback->ShadingModelUpdated(*pCastedArg);
        }
        return;
    case CallbackType_TextureFileLoaded:
        {
            const nn::g3d::viewer::TextureFileLoadedArg* pCastedArg = static_cast<const nn::g3d::viewer::TextureFileLoadedArg*>(pInArg);
            NN_G3D_VIEWER_ASSERT_NOT_NULL(pCastedArg);
            pCallback->TextureFileLoaded(*pCastedArg);
        }
        return;
    case CallbackType_TextureFileUnloaded:
        {
            const nn::g3d::viewer::TextureFileUnloadedArg* pCastedArg = static_cast<const nn::g3d::viewer::TextureFileUnloadedArg*>(pInArg);
            NN_G3D_VIEWER_ASSERT_NOT_NULL(pCastedArg);
            pCallback->TextureFileUnloaded(*pCastedArg);
        }
        return;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
} // NOLINT (readability/fn_size)

ViewerResult ViewerServer::ViewerServerProxy::Initialize(const InitializeArg& arg) NN_NOEXCEPT
{
    NN_USING_MIDDLEWARE(g_MiddlewareInfo);

    NN_SDK_REQUIRES(nn::htcs::IsInitialized(), NN_TEXT_G3DVIEWER("nn::g3d::viewer を初期化する前に nn::htcs を初期化して下さい"));

    // TODO: 将来的にどのドライブがマウントされているか調べるAPIや、別名でドライブをマウントできるようになるので、
    //       そうなったら、すでにドライブがマウント済みかを事前条件チェックするか、初期化時に g3d::viewer 用のマウントを行うかを検討する

    NN_SDK_REQUIRES_NOT_NULL(arg.m_AllocateFunction);
    NN_SDK_REQUIRES_NOT_NULL(arg.m_FreeFunction);
    NN_SDK_REQUIRES_NOT_NULL(arg.m_pDevice);

    ViewerServer& instance = GetInstance();

    Allocator* pAllocator = nullptr;
    {
        void* pBuffer = arg.m_AllocateFunction(sizeof(Allocator), nn::g3d::detail::Alignment_Default, arg.m_pAllocateFunctionUserData);
        if (pBuffer == nullptr)
        {
            return ViewerResult_MemoryAllocationFailed;
        }

        pAllocator = new (pBuffer) Allocator(
            arg.m_AllocateFunction,
            arg.m_FreeFunction,
            arg.m_pAllocateFunctionUserData,
            arg.m_pFreeUserData);
    }

    {
        void* pBuffer = pAllocator->Allocate(sizeof(ViewerServerProxy::Impl), nn::g3d::detail::Alignment_Default, AllocateType_Other);
        if (pBuffer == nullptr)
        {
            return ViewerResult_MemoryAllocationFailed;
        }

        instance.m_Proxy.m_pImpl = new (pBuffer) ViewerServerProxy::Impl(
            arg.m_pDevice,
            pAllocator,
            arg.m_CallbackFunction, arg.m_pCallbackFunctionUserData,
            arg.m_TextureBindCallback, arg.m_pTextureBindCallbackUserData);
    }

    ViewerResult resultInitializeImpl = instance.m_Proxy.m_pImpl->Initialize(arg);
    if (resultInitializeImpl != ViewerResult_Success)
    {
        Finalize();
        return resultInitializeImpl;
    }

    return ViewerResult_Success;
}

void ViewerServer::ViewerServerProxy::Finalize() NN_NOEXCEPT
{
    ViewerServer& instance = ViewerServer::ViewerServerProxy::GetInstance();
    if (IsInitialized())
    {
        instance.m_Proxy.m_pImpl->CloseForce();

        Allocator* pAllocator = &instance.m_Proxy.m_pImpl->GetAllocator();

        instance.m_Proxy.m_pImpl->~Impl();
        pAllocator->Free(instance.m_Proxy.m_pImpl);
        instance.m_Proxy.m_pImpl = nullptr;

        int memoryLeakAddressCount = pAllocator->FindMemoryLeakCount();
        NN_G3D_VIEWER_ASSERT_DETAIL(memoryLeakAddressCount == 0, "Detected %d memory leaks", memoryLeakAddressCount);

        void* pUserData = pAllocator->GetFreeUserData();
        const nn::FreeFunctionWithUserData freeFunc = pAllocator->GetFreeFunction();
        pAllocator->~Allocator();
        freeFunc(pAllocator, pUserData);
    }
}

ViewerServer& ViewerServer::ViewerServerProxy::GetInstance() NN_NOEXCEPT
{
    static ViewerServer s_Instance;
    return s_Instance;
}

bool ViewerServer::ViewerServerProxy::IsInitialized() NN_NOEXCEPT
{
    return ViewerServer::ViewerServerProxy::GetInstance().m_Proxy.m_pImpl != nullptr;
}

bool ViewerServer::ViewerServerProxy::IsConnected() const NN_NOEXCEPT
{
    if (!IsInitialized())
    {
        return false;
    }

    return m_pImpl->IsConnected();
}

bool ViewerServer::ViewerServerProxy::IsOpened() const NN_NOEXCEPT
{
    if (!IsInitialized())
    {
        return false;
    }

    return m_pImpl->IsOpened();
}

void ViewerServer::ViewerServerProxy::Poll() NN_NOEXCEPT
{
    if (!IsInitialized())
    {
        return;
    }

    m_pImpl->PollDataCommunication();
}

void ViewerServer::ViewerServerProxy::ExecuteCommands() NN_NOEXCEPT
{
    if (!IsInitialized())
    {
        return;
    }

    m_pImpl->PollDataEdit();
    m_pImpl->GetResourceManager().UpdateBlink();
}

ViewerResult ViewerServer::ViewerServerProxy::Open() NN_NOEXCEPT
{
    if (!IsInitialized())
    {
        return ViewerResult_ServerNotInitialized;
    }

    return m_pImpl->Open();
}

void ViewerServer::ViewerServerProxy::Close() NN_NOEXCEPT
{
    if (!IsInitialized())
    {
        return;
    }

    m_pImpl->Close();
}

ViewerResult ViewerServer::ViewerServerProxy::QueueAttachModelCommand(nn::g3d::ModelObj*  pAttachModelObj, const char* attachFileName) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAttachModelObj);

    if (!IsInitialized())
    {
        return ViewerResult_ServerNotInitialized;
    }

    if (!IsConnected()) // 接続されていない場合は失敗
    {
        return ViewerResult_NotConnected;
    }

    if (IsModelAttached(pAttachModelObj) || m_pImpl->GetEditCommandManager().IsAttaching(pAttachModelObj))
    {
        return ViewerResult_AlreadyAttached;
    }

    // TODO: 同じリソースを共有する複数の ModelObj をアタッチしたいときに、ひとつずつ登録してしまうと、
    // 全部を登録し終わる前に並列で走っている通信スレッドが処理を進めてしまう可能性がある。
    // 利用者側で通信スレッドを止めれば対処可能だが、利用者からはわかりにく仕様なので、もし問題が報告された場合は
    // 複数の ModelObj を一括でアタッチできるような API の提供や通信処理を内部でブロックする API の提供を考える。
    return m_pImpl->GetEditCommandManager().QueueAttachModel(&pAttachModelObj, 1, attachFileName);
}

ViewerResult ViewerServer::ViewerServerProxy::QueueDetachModelCommand(const nn::g3d::ModelObj* pDetachModelObj) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDetachModelObj);
    if (!IsInitialized())
    {
        return ViewerResult_ServerNotInitialized;
    }

    return m_pImpl->GetEditCommandManager().QueueDetachModel(pDetachModelObj);
}

bool ViewerServer::ViewerServerProxy::IsModelAttaching(const nn::g3d::ModelObj* pModelObj) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pModelObj);
    if (!IsInitialized())
    {
        return false;
    }

    return m_pImpl->GetEditCommandManager().IsAttaching(pModelObj);
}

bool ViewerServer::ViewerServerProxy::IsModelAttached(const nn::g3d::ModelObj* pModelObj) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pModelObj);
    if (!IsInitialized())
    {
        return false;
    }

    return m_pImpl->GetResourceManager().HasModelObj(pModelObj);
}

ViewerResult ViewerServer::ViewerServerProxy::QueueAttachShaderArchiveCommand(nn::g3d::ResShaderArchive* pAttachResShaderArchive, const char* attachFileName) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAttachResShaderArchive);
    if (!IsInitialized())
    {
        return ViewerResult_ServerNotInitialized;
    }

    if (!IsConnected()) // 接続されていない場合は失敗
    {
        return ViewerResult_NotConnected;
    }

    if (IsShaderArchiveAttached(pAttachResShaderArchive) || IsShaderArchiveAttaching(pAttachResShaderArchive))
    {
        return ViewerResult_AlreadyAttached;
    }

    // 強制バリエーションされていると .fsdb とブランチ情報が異なるため許可しない
    if (pAttachResShaderArchive->ToData().flag & ResShaderArchive::Flag_ForceVariation)
    {
        return ViewerResult_InvalidArgument;
    }

    return m_pImpl->GetEditCommandManager().QueueAttachShaderArchive(pAttachResShaderArchive, attachFileName);
}

ViewerResult ViewerServer::ViewerServerProxy::QueueDetachShaderArchiveCommand(const nn::g3d::ResShaderArchive* pDetachResShaderArchive) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDetachResShaderArchive);
    if (!IsInitialized())
    {
        return ViewerResult_ServerNotInitialized;
    }

    return m_pImpl->GetEditCommandManager().QueueDetachShaderArchive(pDetachResShaderArchive);
}

bool ViewerServer::ViewerServerProxy::IsShaderArchiveAttaching(const nn::g3d::ResShaderArchive* pShaderArchive) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pShaderArchive);
    if (!IsInitialized())
    {
        return false;
    }

    return m_pImpl->GetEditCommandManager().IsShaderArchiveAttaching(pShaderArchive);
}

bool ViewerServer::ViewerServerProxy::IsShaderArchiveAttached(const nn::g3d::ResShaderArchive* pShaderArchive) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pShaderArchive);
    if (!IsInitialized())
    {
        return false;
    }

    return m_pImpl->GetResourceManager().HasShaderArchive(pShaderArchive);
}

void ViewerServer::ViewerServerProxy::CalculateAnimations() NN_NOEXCEPT
{
    if (!IsInitialized())
    {
        return;
    }

    m_pImpl->GetResourceManager().CalculateAnimations();
}

void ViewerServer::ViewerServerProxy::CalculateSkeletalAnimations(const nn::g3d::ModelObj* modelObj) NN_NOEXCEPT
{
    if (!IsInitialized())
    {
        return;
    }

    m_pImpl->GetResourceManager().CalculateSkeletalAnimations(modelObj);
}


ViewerResult ViewerServer::ViewerServerProxy::QueueAddRenderInfoStringChoiceCommand(const char* labelName, const char* itemName, const char* aliasItemName) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(labelName);
    NN_SDK_REQUIRES_NOT_NULL(itemName);

    if (!IsInitialized())
    {
        return ViewerResult_ServerNotInitialized;
    }

    EditRenderInfo* pRenderInfoDefinition = m_pImpl->GetCurrentEditTargetRenderInfoDefinition();
    if (pRenderInfoDefinition == nullptr)
    {
        return ViewerResult_SendRenderInfoNotRequested;
    }

    return pRenderInfoDefinition->PushChoice(labelName, itemName, aliasItemName);
}

ViewerResult ViewerServer::ViewerServerProxy::QueueSetRenderInfoIntRangeCommand(const char* labelName, int minValue, int maxValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(labelName);
    NN_SDK_REQUIRES(minValue <= maxValue);

    if (!IsInitialized())
    {
        return ViewerResult_ServerNotInitialized;
    }

    EditRenderInfo* pRenderInfoDefinition = m_pImpl->GetCurrentEditTargetRenderInfoDefinition();
    if (pRenderInfoDefinition == nullptr)
    {
        return ViewerResult_SendRenderInfoNotRequested;
    }

    return pRenderInfoDefinition->PushChoice(labelName, minValue, maxValue);
}

ViewerResult ViewerServer::ViewerServerProxy::QueueSetRenderInfoFloatRangeCommand(const char* labelName, float minValue, float maxValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(labelName);
    NN_SDK_REQUIRES(minValue <= maxValue);

    if (!IsInitialized())
    {
        return ViewerResult_ServerNotInitialized;
    }

    EditRenderInfo* pRenderInfoDefinition = m_pImpl->GetCurrentEditTargetRenderInfoDefinition();
    if (pRenderInfoDefinition == nullptr)
    {
        return ViewerResult_SendRenderInfoNotRequested;
    }

    return pRenderInfoDefinition->PushChoice(labelName, minValue, maxValue);
}


ViewerResult ViewerServer::ViewerServerProxy::QueuePushRenderInfoStringDefaultCommand(const char* labelName, const char* value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(labelName);
    NN_SDK_REQUIRES_NOT_NULL(value);
    if (!IsInitialized())
    {
        return ViewerResult_ServerNotInitialized;
    }

    EditRenderInfo* pRenderInfoDefinition = m_pImpl->GetCurrentEditTargetRenderInfoDefinition();
    if (pRenderInfoDefinition == nullptr)
    {
        return ViewerResult_SendRenderInfoNotRequested;
    }

    return pRenderInfoDefinition->PushDefault(labelName, value);
}

ViewerResult ViewerServer::ViewerServerProxy::QueuePushRenderInfoIntDefaultCommand(const char* labelName, int value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(labelName);
    if (!IsInitialized())
    {
        return ViewerResult_ServerNotInitialized;
    }

    EditRenderInfo* pRenderInfoDefinition = m_pImpl->GetCurrentEditTargetRenderInfoDefinition();
    if (pRenderInfoDefinition == nullptr)
    {
        return ViewerResult_SendRenderInfoNotRequested;
    }

    return pRenderInfoDefinition->PushDefault(labelName, value);
}

ViewerResult ViewerServer::ViewerServerProxy::QueuePushRenderInfoFloatDefaultCommand(const char* labelName, float value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(labelName);
    if (!IsInitialized())
    {
        return ViewerResult_ServerNotInitialized;
    }

    EditRenderInfo* pRenderInfoDefinition = m_pImpl->GetCurrentEditTargetRenderInfoDefinition();
    if (pRenderInfoDefinition == nullptr)
    {
        return ViewerResult_SendRenderInfoNotRequested;
    }

    return pRenderInfoDefinition->PushDefault(labelName, value);
}

ViewerResult ViewerServer::ViewerServerProxy::QueueLayoutModelCommand(const LayoutModelArg& arg) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(arg.m_pModelObj);
    if (!IsInitialized())
    {
        return ViewerResult_ServerNotInitialized;
    }

    if (!IsModelAttached(arg.m_pModelObj))
    {
        return ViewerResult_NotAttached;
    }

    ViewerKeyType modelObjKey = ViewerKeyManager::GetInstance().FindKey(arg.m_pModelObj);
    NN_G3D_VIEWER_ASSERT_VALID_KEY(modelObjKey);
    return m_pImpl->GetEditCommandManager().AddModelLayoutQueue(
        modelObjKey,
        arg.m_Scale, arg.m_Rotate, arg.m_Translate);
}

ViewerResult ViewerServer::ViewerServerProxy::QueueSendUserMessageCommand(const SendUserMessageArg& arg) NN_NOEXCEPT
{
    if (!IsInitialized())
    {
        return ViewerResult_ServerNotInitialized;
    }

    return m_pImpl->GetEditCommandManager().QueueNotifyMessageCommand(
        arg.m_Message,
        arg.m_MessageType,
        arg.m_MessageDestination);
}

ViewerResult ViewerServer::ViewerServerProxy::QueueSelectMaterialCommand(const nn::g3d::ModelObj* pModelObj, int materialIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pModelObj);
    NN_SDK_REQUIRES(materialIndex >= 0, "%s\n", pModelObj->GetResource()->GetName());
    if (!IsInitialized())
    {
        return ViewerResult_ServerNotInitialized;
    }

    return m_pImpl->GetEditPickup().PushMaterialPickup(pModelObj, materialIndex);
}

ViewerResult ViewerServer::ViewerServerProxy::QueueClearMaterialSelectionCommand() NN_NOEXCEPT
{
    if (!IsInitialized())
    {
        return ViewerResult_ServerNotInitialized;
    }

    return m_pImpl->GetEditPickup().PushMaterialPickup(nullptr, -1);
}


bool ViewerServer::ViewerServerProxy::IsAnimPlaying() const NN_NOEXCEPT
{
    if (!IsInitialized())
    {
        return false;
    }

    return m_pImpl->GetResourceManager().IsAnimationPlaying();
}

void ViewerServer::ViewerServerProxy::SetAnimPlaying(bool isPlaying) NN_NOEXCEPT
{
    if (!IsInitialized())
    {
        return;
    }

    m_pImpl->GetResourceManager().SetAnimPlaying(isPlaying);
}

bool ViewerServer::ViewerServerProxy::IsAnimPlayPolicyLoop() const NN_NOEXCEPT
{
    if (!IsInitialized())
    {
        return false;
    }

    return m_pImpl->GetResourceManager().IsLoopAnim();
}

float ViewerServer::ViewerServerProxy::GetCurrentFrame() const NN_NOEXCEPT
{
    if (!IsInitialized())
    {
        return 0.0f;
    }

    return m_pImpl->GetResourceManager().GetFrame();
}

float ViewerServer::ViewerServerProxy::GetFrameStep() const NN_NOEXCEPT
{
    if (!IsInitialized())
    {
        return 0.0f;
    }

    return m_pImpl->GetResourceManager().GetFrameStep();
}

void ViewerServer::ViewerServerProxy::SetFrameStep(float step) NN_NOEXCEPT
{
    if (!IsInitialized())
    {
        return;
    }

    m_pImpl->GetResourceManager().SetFrameStep(step);
}

float ViewerServer::ViewerServerProxy::GetPreviewFrameStepRate() const NN_NOEXCEPT
{
    if (!IsInitialized())
    {
        return 0.0f;
    }

    return m_pImpl->GetResourceManager().GetPreviewFrameStepRate();
}

void ViewerServer::ViewerServerProxy::SetPreviewFrameStepRate(float rate) NN_NOEXCEPT
{
    if (!IsInitialized())
    {
        return;
    }

    return m_pImpl->GetResourceManager().SetPreviewFrameStepRate(rate);
}

float ViewerServer::ViewerServerProxy::GetFrameCount() const NN_NOEXCEPT
{
    if (!IsInitialized())
    {
        return 0.0f;
    }

    return m_pImpl->GetResourceManager().GetFrameCount();
}

void ViewerServer::ViewerServerProxy::SetCurrentFrame(float frame) NN_NOEXCEPT
{
    if (!IsInitialized())
    {
        return;
    }

    m_pImpl->GetResourceManager().SetFrame(frame);
}

int ViewerServer::ViewerServerProxy::GetModelAnimCount() const NN_NOEXCEPT
{
    if (!IsInitialized())
    {
        return 0;
    }

    return m_pImpl->GetResourceManager().GetEditAnimCount();
}

int ViewerServer::ViewerServerProxy::GetModelAnimBoundCount(const nn::g3d::ModelObj* modelObj) const NN_NOEXCEPT
{
    if (!IsInitialized())
    {
        return 0;
    }

    return m_pImpl->GetEditCommandManager().GetModelAnimBoundCount(modelObj);
}

int ViewerServer::ViewerServerProxy::GetActiveModelAnimIndex(const nn::g3d::ModelObj* modelObj, int boundAnimIndex) const NN_NOEXCEPT
{
    if (!IsInitialized())
    {
        return 0;
    }

    return m_pImpl->GetEditCommandManager().GetActiveEditAnimIndex(modelObj, boundAnimIndex);
}

float ViewerServer::ViewerServerProxy::GetModelAnimFrameCount(int modelAnimIndex) const NN_NOEXCEPT
{
    if (!IsInitialized())
    {
        return 0.0f;
    }

    return m_pImpl->GetResourceManager().GetEditAnimFrameCount(modelAnimIndex);
}

const char* ViewerServer::ViewerServerProxy::GetModelAnimName(int modelAnimindex) const NN_NOEXCEPT
{
    if (!IsInitialized())
    {
        return nullptr;
    }

    return m_pImpl->GetResourceManager().GetEditAnimName(modelAnimindex);
}

ViewerAnimKind ViewerServer::ViewerServerProxy::GetModelAnimKind(int modelAnimIndex) const NN_NOEXCEPT
{
    if (!IsInitialized())
    {
        return static_cast<ViewerAnimKind>(-1);
    }

    return m_pImpl->GetResourceManager().GetViewerAnimKind(modelAnimIndex);
}

#endif
