﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "g3d_Map.h"
#include "g3d_SynchronizedObject.h"
#include "g3d_ScopedLock.h"

namespace nn { namespace g3d { namespace viewer { namespace detail {

template<typename TKey, typename TValue>
class SynchronizedMap : public SynchronizedObject
{
public:
    explicit SynchronizedMap(
        nn::g3d::viewer::detail::Allocator* pAllocator,
        TKey errorKey,
        TValue errorValue) NN_NOEXCEPT
        : m_Map(pAllocator, errorKey, errorValue)
    {
    }

    bool Register(TKey key, TValue value) NN_NOEXCEPT
    {
        ScopedLock scopedLock(GetMutex());
        return m_Map.Register(key, value);
    }

    // @brief 既存のキーに割り当てられている値を別の値で上書きします。
    bool UpdateValue(TKey key, TValue value) NN_NOEXCEPT
    {
        ScopedLock scopedLock(GetMutex());
        return m_Map.UpdateValue(key, value);
    }

    bool Unregister(TKey key) NN_NOEXCEPT
    {
        ScopedLock scopedLock(GetMutex());
        return m_Map.Unregister(key);
    }

    TKey FindKey(TValue value) const NN_NOEXCEPT
    {
        ScopedLock scopedLock(*const_cast<nn::os::Mutex*>(&GetMutex()));
        return m_Map.FindKey(value);
    }

    TValue FindValue(TKey key) const NN_NOEXCEPT
    {
        ScopedLock scopedLock(*const_cast<nn::os::Mutex*>(&GetMutex()));
        return m_Map.FindValue(key);
    }

    int GetCount() const NN_NOEXCEPT
    {
        ScopedLock scopedLock(*const_cast<nn::os::Mutex*>(&GetMutex()));
        return m_Map.GetCount();
    }

    TKey GetKey(int index) const NN_NOEXCEPT
    {
        ScopedLock scopedLock(*const_cast<nn::os::Mutex*>(&GetMutex()));
        return m_Map.GetKey(index);
    }

    TValue GetValue(int index) const NN_NOEXCEPT
    {
        ScopedLock scopedLock(*const_cast<nn::os::Mutex*>(&GetMutex()));
        return m_Map.GetValue(index);
    }

    bool IsKeyRegistered(TKey key) const NN_NOEXCEPT
    {
        ScopedLock scopedLock(*const_cast<nn::os::Mutex*>(&GetMutex()));
        return m_Map.IsKeyRegistered(key);
    }

    bool IsValueRegistered(TValue value) const NN_NOEXCEPT
    {
        ScopedLock scopedLock(*const_cast<nn::os::Mutex*>(&GetMutex()));
        return m_Map.IsValueRegistered(value);
    }

private:
    Map<TKey, TValue> m_Map;
};

}}}} // namespace nn::g3d::viewer::detail
