﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "g3d_DynamicPtrArray.h"
#include "g3d_SynchronizedObject.h"
#include "g3d_ScopedLock.h"

namespace nn { namespace g3d { namespace viewer { namespace detail {

template<typename T>
class SynchronizedDynamicPtrArray : public SynchronizedObject
{
public:
    SynchronizedDynamicPtrArray(nn::g3d::viewer::detail::Allocator* pAllocator, size_t alignment) NN_NOEXCEPT
        : m_DynamicPtrArray(pAllocator, alignment)
    {}

    explicit SynchronizedDynamicPtrArray(size_t alignment) NN_NOEXCEPT
        : m_DynamicPtrArray(alignment)
    {}

    T* At(int index) const NN_NOEXCEPT
    {
        ScopedLock scopedLock(*const_cast<nn::os::Mutex*>(&GetMutex()));
        return m_DynamicPtrArray.At(index);
    }

    T* UnsafeAt(int index) const NN_NOEXCEPT
    {
        ScopedLock scopedLock(*const_cast<nn::os::Mutex*>(&GetMutex()));
        return m_DynamicPtrArray.UnsafeAt(index);
    }

    T* Front() const NN_NOEXCEPT
    {
        ScopedLock scopedLock(*const_cast<nn::os::Mutex*>(&GetMutex()));
        return m_DynamicPtrArray.Front();
    }

    T* Back() const NN_NOEXCEPT
    {
        ScopedLock scopedLock(*const_cast<nn::os::Mutex*>(&GetMutex()));
        return m_DynamicPtrArray.Back();
    }

    bool PushBack(T* ptr) NN_NOEXCEPT
    {
        ScopedLock scopedLock(GetMutex());
        return m_DynamicPtrArray.PushBack(ptr);
    }

    T* PopBack() NN_NOEXCEPT
    {
        ScopedLock scopedLock(GetMutex());
        return m_DynamicPtrArray.PopBack();
    }

    template<typename TKey>
    T* Find(const TKey* key, int (*compareCallback)(const T* a, const TKey* b)) const NN_NOEXCEPT
    {
        ScopedLock scopedLock(*const_cast<nn::os::Mutex*>(&GetMutex()));
        return m_DynamicPtrArray.Find(key, compareCallback);
    }

    template<typename TKey>
    int Search(const TKey* key, int (*compareCallback)(const T* a, const TKey* b)) const NN_NOEXCEPT
    {
        ScopedLock scopedLock(*const_cast<nn::os::Mutex*>(&GetMutex()));
        return m_DynamicPtrArray.Search(key, compareCallback);
    }

    int IndexOf(const T* ptr) const NN_NOEXCEPT
    {
        ScopedLock scopedLock(*const_cast<nn::os::Mutex*>(&GetMutex()));
        return m_DynamicPtrArray.IndexOf(ptr);
    }

    int GetCount() const NN_NOEXCEPT
    {
        ScopedLock scopedLock(*const_cast<nn::os::Mutex*>(&GetMutex()));
        return m_DynamicPtrArray.GetCount();
    }

    void Clear() NN_NOEXCEPT
    {
        ScopedLock scopedLock(GetMutex());
        m_DynamicPtrArray.Clear();
    }

    void Destroy() NN_NOEXCEPT
    {
        ScopedLock scopedLock(GetMutex());
        m_DynamicPtrArray.Destroy();
    }

    void EraseByIndex(int pos) NN_NOEXCEPT
    {
        ScopedLock scopedLock(GetMutex());
        m_DynamicPtrArray.EraseByIndex(pos);
    }

    class Iter<T*> Begin() NN_NOEXCEPT
    {
        return m_DynamicPtrArray.Begin();
    }

    class Iter<T*> End() NN_NOEXCEPT
    {
        return m_DynamicPtrArray.End();
    }

    class Iter<T*> Erase(class Iter<T*> iter) NN_NOEXCEPT
    {
        return m_DynamicPtrArray.Erase(iter);
    }

private:
    DynamicPtrArray<T> m_DynamicPtrArray;
};

}}}} // namespace nn::g3d::viewer::detail
