﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "g3d_DynamicArray.h"
#include "g3d_SynchronizedObject.h"
#include "g3d_ScopedLock.h"

namespace nn { namespace g3d { namespace viewer { namespace detail {

template<typename T>
class SynchronizedDynamicArray : public SynchronizedObject
{
public:
    SynchronizedDynamicArray(nn::g3d::viewer::detail::IAllocator* pAllocator, size_t alignment, T errorValue) NN_NOEXCEPT
        : m_DynamicArray(pAllocator, alignment, errorValue)
    {}

    T Front() const NN_NOEXCEPT
    {
        ScopedLock scopedLock(*const_cast<nn::os::Mutex*>(&GetMutex()));
        return m_DynamicArray.Front();
    }

    T Back() const NN_NOEXCEPT
    {
        ScopedLock scopedLock(*const_cast<nn::os::Mutex*>(&GetMutex()));
        return m_DynamicArray.Back();
    }

    bool PushBack(T elem) NN_NOEXCEPT
    {
        ScopedLock scopedLock(GetMutex());
        return m_DynamicArray.PushBack(elem);
    }

    T PopBack() NN_NOEXCEPT
    {
        ScopedLock scopedLock(GetMutex());
        return m_DynamicArray.PopBack();
    }

    template<typename TKey>
    T Find(const TKey* key, int (*compareCallback)(const T a, const TKey b)) const NN_NOEXCEPT
    {
        ScopedLock scopedLock(*const_cast<nn::os::Mutex*>(&GetMutex()));
        return m_DynamicArray.Find(key, compareCallback);
    }

    int IndexOf(const T ptr) const NN_NOEXCEPT
    {
        ScopedLock scopedLock(*const_cast<nn::os::Mutex*>(&GetMutex()));
        return m_DynamicArray.IndexOf(ptr);
    }

    int GetCount() const NN_NOEXCEPT
    {
        ScopedLock scopedLock(*const_cast<nn::os::Mutex*>(&GetMutex()));
        return m_DynamicArray.GetCount();
    }

    void Clear() NN_NOEXCEPT
    {
        ScopedLock scopedLock(GetMutex());
        m_DynamicArray.Clear();
    }

    void Destroy() NN_NOEXCEPT
    {
        ScopedLock scopedLock(GetMutex());
        m_DynamicArray.Destroy();
    }

    void EraseByIndex(int pos) NN_NOEXCEPT
    {
        ScopedLock scopedLock(GetMutex());
        m_DynamicArray.EraseByIndex(pos);
    }

    class Iter<const T> Begin() const NN_NOEXCEPT
    {
        return m_DynamicArray.Begin();
    }

    class Iter<const T> End() const NN_NOEXCEPT
    {
        return m_DynamicArray.End();
    }

    class Iter<T> Begin() NN_NOEXCEPT
    {
        return m_DynamicArray.Begin();
    }

    class Iter<T> End() NN_NOEXCEPT
    {
        return m_DynamicArray.End();
    }

    class Iter<T> Erase(class Iter<T> iter) NN_NOEXCEPT
    {
        return m_DynamicArray.Erase(iter);
    }

    T& operator[] (int index) NN_NOEXCEPT
    {
        ScopedLock scopedLock(GetMutex());
        return m_DynamicArray[index];
    }

private:
    DynamicArray<T> m_DynamicArray;
};

}}}} // namespace nn::g3d::viewer::detail
