﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3d_EditShaderProgram.h"



#include "../g3d_EditSocket.h"
#include "../util/g3d_ViewerUtility.h"
#include "../g3d_Allocator.h"

#include <nn/g3d/g3d_ResShader.h>

namespace nn { namespace g3d { namespace viewer { namespace detail {

EditShaderProgram::EditShaderProgram(
    nn::gfx::Device* pDevice,
    Allocator* allocator,
    ViewerKeyType shaderArchiveKey,
    int shadingModelIndex,
    int shaderProgramIndex,
    ResShaderProgram* pResShaderProgram) NN_NOEXCEPT
    : DeviceDependentObj(pDevice, allocator)
    , m_pTargetResShaderProgram(pResShaderProgram)
    , m_pUpdateResShaderArchive(nullptr)
    , m_WorkBuffer(allocator)
    , m_OptionInfos(allocator, nn::g3d::detail::Alignment_Default)
    , m_ShaderArchiveKey(shaderArchiveKey)
    , m_ShadingModelIndex(shadingModelIndex)
    , m_ShaderProgramIndex(shaderProgramIndex)
    , m_PacketEnableFlag(false)
    , m_SendInfoCallbackFlag(false)
{
    NN_G3D_VIEWER_ASSERT_NOT_NULL(pResShaderProgram);
    m_OriginalFlag = pResShaderProgram->ToData().flag;
    m_OriginalSamplerCount = pResShaderProgram->ToData().samplerCount;
    m_OriginalUniformBlockCount = pResShaderProgram->ToData().uniformBlockCount;
    m_OriginalAttribuActiveFlag = pResShaderProgram->ToData().attribActiveFlag;
    m_OriginalOffsetSamplerTable = pResShaderProgram->ToData().pSamplerTable.Get();
    m_OriginalOffsetUniformBlockTable = pResShaderProgram->ToData().pUniformBlockTable.Get();
    m_pOriginalShaderStorageBlockTable = pResShaderProgram->ToData().pShaderStorageBlockTable.Get();
    m_OriginalShaderStorageBlockCount = pResShaderProgram->ToData().shaderStorageBlockCount;
    m_OriginalOffsetShader = pResShaderProgram->ToData().pShader.Get();
}

void
EditShaderProgram::ResetToOriginal() NN_NOEXCEPT
{
    // 初期値に戻す
    {
        // オリジナルのプログラムの場合は Cleanup しない。
        // Cafe 版は GX2 構造体がプログラム間で共有されている可能性があるため。
        // GL 版は オリジナルのハンドルを破棄しないようにするため。
        // TODO: ↑gfx では問題ない?しばらく様子を見て問題なければこのif分を消す
        //if (m_pUpdateResShaderArchive != nullptr)
        {
            m_pTargetResShaderProgram->Cleanup(m_pDevice);
        }

        // TODO: フラグを書き換えてしまうとgfx以下の低レイヤーAPIのフラグと整合性がとれなくなる?
        // 書き換えるなら低レイヤーAPIのフラグも編集しないといけなそうなので、コメントアウトして様子見
        // 問題なければ消す
        //m_pTargetResShaderProgram->ToData().flag = m_OriginalFlag;

        m_pTargetResShaderProgram->ToData().samplerCount = m_OriginalSamplerCount;
        m_pTargetResShaderProgram->ToData().uniformBlockCount = m_OriginalUniformBlockCount;
        m_pTargetResShaderProgram->ToData().attribActiveFlag = m_OriginalAttribuActiveFlag;
        m_pTargetResShaderProgram->ToData().pSamplerTable.Set(static_cast<int32_t*>(m_OriginalOffsetSamplerTable));
        m_pTargetResShaderProgram->ToData().pUniformBlockTable.Set(static_cast<int32_t*>(m_OriginalOffsetUniformBlockTable));
        m_pTargetResShaderProgram->ToData().pShaderStorageBlockTable.Set(m_pOriginalShaderStorageBlockTable);
        m_pTargetResShaderProgram->ToData().shaderStorageBlockCount = m_OriginalShaderStorageBlockCount;
        m_pTargetResShaderProgram->ToData().pShader.Set(static_cast<nn::gfx::ResShaderVariation*>(m_OriginalOffsetShader));

        m_pTargetResShaderProgram->Setup(m_pDevice);
    }

    ClearOptionInfo();

    m_WorkBuffer.Clear();
}

void
EditShaderProgram::Destroy() NN_NOEXCEPT
{
    m_OptionInfos.Destroy();
    m_pAllocator = nullptr;
}

bool
EditShaderProgram::SendModifiedShaderProgram(EditSocketBase* socket) NN_NOEXCEPT
{
    ScopedLock scopedLock(m_WorkBuffer);
    NN_G3D_VIEWER_ASSERT_NOT_NULL(socket);

    if (!m_PacketEnableFlag)
    {
        return false;
    }

    bool result = socket->WriteSync(m_WorkBuffer.GetWorkBufferPtr(), m_ModifiedShaderProgramBufferSize);

    if (result)
    {
        m_PacketEnableFlag = false;
    }
    return result;
}

bool
EditShaderProgram::MakeModifiedShaderProgramPacket() NN_NOEXCEPT
{
    m_PacketEnableFlag = false;
    size_t bufferSize = CalculateWorkBufferSize();
    if (bufferSize == 0)
    {
        return false;
    }

    if (!m_WorkBuffer.Resize(bufferSize))
    {
        return false;
    }

    m_ModifiedShaderProgramBufferSize = bufferSize;
    MakeModifiedShaderProgramBuffer();
    m_PacketEnableFlag = true;
    return true;
}

bool
EditShaderProgram::PushOptionInfo(const char* pOption, const char* pChoice) NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT_NOT_NULL(pOption);
    NN_G3D_VIEWER_ASSERT_NOT_NULL(pChoice);

    size_t bufferSize = sizeof(OptionInfo);
    void* buffer = m_pAllocator->Allocate(bufferSize, nn::g3d::detail::Alignment_Default, AllocateType_Communication);
    if (buffer == nullptr)
    {
        return false;
    }

    // オプション情報登録作業
    {
        OptionInfo* optionInfo = new (buffer) OptionInfo(m_pAllocator);
        if (!optionInfo->Setup(pOption, pChoice))
        {
            optionInfo->~OptionInfo();
            m_pAllocator->Free(optionInfo);
            return false;
        }

        if (!m_OptionInfos.PushBack(optionInfo))
        {
            optionInfo->~OptionInfo();
            m_pAllocator->Free(optionInfo);
            return false;
        }
    }
    return true;
}

void
EditShaderProgram::UpdateShaderProgram(ResShaderArchive* pNewResShaderArchive) NN_NOEXCEPT
{
    NN_G3D_VIEWER_DEBUG_PRINT("Updateded Target Program: 0x%p\n", m_pTargetResShaderProgram);
    NN_G3D_VIEWER_DEBUG_PRINT("Updateded New Archive: 0x%p\n", pNewResShaderArchive);
    NN_G3D_VIEWER_ASSERT_NOT_NULL(pNewResShaderArchive);
    NN_G3D_VIEWER_ASSERT_DETAIL(pNewResShaderArchive->GetShadingModelCount() > 0, "%s\n", NN_G3D_VIEWER_RES_NAME(pNewResShaderArchive, GetName()));
    ResShadingModel* pNewResShadingModel = pNewResShaderArchive->GetShadingModel(0);

    NN_G3D_VIEWER_ASSERT_DETAIL(pNewResShadingModel->GetShaderProgramCount() > 0, "%s\n", NN_G3D_VIEWER_RES_NAME(pNewResShadingModel, GetName()));

    ResShaderProgram* pNewResShaderProgram = pNewResShadingModel->GetShaderProgram(0);
    ResShaderProgramData* pNewResShaderProgramData = &pNewResShaderProgram->ToData();

    bool isProgramInitialized = m_pTargetResShaderProgram->IsInitialized();
    if (isProgramInitialized)
    {
        m_pTargetResShaderProgram->Cleanup(m_pDevice);
    }

    m_pTargetResShaderProgram->ToData().attribActiveFlag = pNewResShaderProgramData->attribActiveFlag;

    m_pTargetResShaderProgram->ToData().pShaderStorageBlockTable.Set(pNewResShaderProgramData->pShaderStorageBlockTable.Get());
    m_pTargetResShaderProgram->ToData().shaderStorageBlockCount = pNewResShaderProgramData->shaderStorageBlockCount;

    m_pTargetResShaderProgram->ToData().pSamplerTable.Set(pNewResShaderProgramData->pSamplerTable.Get());
    m_pTargetResShaderProgram->ToData().samplerCount = pNewResShaderProgramData->samplerCount;

    m_pTargetResShaderProgram->ToData().pUniformBlockTable.Set(pNewResShaderProgramData->pUniformBlockTable.Get());
    m_pTargetResShaderProgram->ToData().uniformBlockCount = pNewResShaderProgramData->uniformBlockCount;

    m_pTargetResShaderProgram->ToData().pShader.Set(pNewResShaderProgramData->pShader.Get());

    m_pUpdateResShaderArchive = pNewResShaderArchive;

    if (isProgramInitialized)
    {
        m_pTargetResShaderProgram->Setup(m_pDevice);
        m_pTargetResShaderProgram->Update(m_pDevice);
    }
}

void
EditShaderProgram::ClearOptionInfo() NN_NOEXCEPT
{
    int size = m_OptionInfos.GetCount();
    for (int i = 0; i < size; ++i)
    {
        OptionInfo* optionInfo = m_OptionInfos.UnsafeAt(i);
        optionInfo->~OptionInfo();
        m_pAllocator->Free(optionInfo);
        optionInfo = nullptr;
    }
    m_OptionInfos.Clear();
}

size_t
EditShaderProgram::CalculateWorkBufferSize() NN_NOEXCEPT
{
    int optionInfoSize = m_OptionInfos.GetCount();

    size_t size = sizeof(PacketHeader);
    size += sizeof(ShaderProgramSendInfo);

    if (optionInfoSize == 0)
    {
        return size;
    }

    size += sizeof(ShaderProgramOptionInfo) * optionInfoSize;
    {
        for (int i = 0; i < optionInfoSize; ++i)
        {
            OptionInfo* optionInfo = m_OptionInfos.UnsafeAt(i);
            optionInfo->m_OptionOffset = size;
            size += optionInfo->m_OptionBinarySize;
            optionInfo->m_ChoiceOffset = size;
            size += optionInfo->m_ChoiceBinarySize;
        }
    }
    return size;
}

void
EditShaderProgram::MakeModifiedShaderProgramBuffer() NN_NOEXCEPT
{
    ScopedLock scopedLock(m_WorkBuffer);

    // CalculateWorkBufferSize が成功している前提で実行
    void* buffer = m_WorkBuffer.GetWorkBufferPtr();
    NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(buffer, "%s\n", NN_G3D_VIEWER_RES_NAME(m_pUpdateResShaderArchive, GetName()));

    PacketHeader* header = static_cast<PacketHeader*>(buffer);
    header->magic = NN_G3D_EDIT_MAGIC;
    header->verWord = NN_G3D_EDIT_VERSION;
    header->command = EDIT_SEND_MODIFIED_SHADER_COMMAND_FLAG;
    // TODO: 後でsize_tに直す
    header->dataSize = static_cast<int32_t>(m_ModifiedShaderProgramBufferSize - sizeof(PacketHeader));

    int optionInfoCount = m_OptionInfos.GetCount();

    ShaderProgramSendInfo* sendInfo = AddOffset<ShaderProgramSendInfo>(buffer, sizeof(PacketHeader));
    sendInfo->shaderArchiveKey = m_ShaderArchiveKey;
    sendInfo->shadingModelIndex = m_ShadingModelIndex;
    sendInfo->shaderProgramIndex = m_ShaderProgramIndex;
    sendInfo->optionInfoNum = static_cast<uint32_t>(optionInfoCount);

    size_t optionInfoSize = sizeof(ShaderProgramOptionInfo);
    ShaderProgramOptionInfo* firstShaderProgramOptionInfo = AddOffset<ShaderProgramOptionInfo>(sendInfo, sizeof(ShaderProgramSendInfo));
    {
        for (int i = 0; i < optionInfoCount; ++i)
        {
            ShaderProgramOptionInfo* shaderProgramOptionInfo = AddOffset<ShaderProgramOptionInfo>(firstShaderProgramOptionInfo, i * optionInfoSize);
            OptionInfo* optionInfo = m_OptionInfos.UnsafeAt(i);
            // TODO: 後でsize_tに直す
            shaderProgramOptionInfo->optionOffset = static_cast<uint32_t>(optionInfo->m_OptionOffset);
            shaderProgramOptionInfo->choiceOffset = static_cast<uint32_t>(optionInfo->m_ChoiceOffset);
        }
    }

    {
        for (int i = 0; i < optionInfoCount; ++i)
        {
            OptionInfo* optionInfo = m_OptionInfos.UnsafeAt(i);
            {
                char* name = AddOffset<char>(buffer, optionInfo->m_OptionOffset);
                memcpy(name, optionInfo->m_Option.GetStr(), optionInfo->m_Option.GetLength());
                name[optionInfo->m_Option.GetLength()] = 0;
            }

            {
                char* name = AddOffset<char>(buffer, optionInfo->m_ChoiceOffset);
                memcpy(name, optionInfo->m_Choice.GetStr(), optionInfo->m_Choice.GetLength());
                name[optionInfo->m_Choice.GetLength()] = 0;
            }
        }
    }
}

bool
EditShaderProgram::OptionInfo::Setup(const char* option, const char* choice) NN_NOEXCEPT
{
    bool isFailed = false;
    isFailed |= m_Option.Assign(option) != ViewerResult_Success;
    isFailed |= m_Choice.Assign(choice) != ViewerResult_Success;

    if (isFailed)
    {
        m_Option.Clear();
        m_Choice.Clear();
    }

    m_OptionBinarySize = nn::util::align_up<size_t>(m_Option.GetLength() + 1, nn::g3d::detail::Alignment_Default);
    m_ChoiceBinarySize = nn::util::align_up<size_t>(m_Choice.GetLength() + 1, nn::g3d::detail::Alignment_Default);
    return !isFailed;
}

}}}}

