﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3d_EditRenderInfo.h"



#include "../g3d_EditSocket.h"
#include "../util/g3d_ViewerUtility.h"

namespace nn { namespace g3d { namespace viewer { namespace detail {

ViewerResult
EditRenderInfo::PushChoice(const char* labelName, const char* itemName, const char* aliasItemName) NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT_NOT_NULL(labelName);
    NN_G3D_VIEWER_ASSERT_NOT_NULL(itemName);

    int labelIndex = FindLabelIndex(labelName);
    if (labelIndex == -1)
    {
        ViewerResult pushResult = PushLabel(labelName, ResRenderInfo::Type_String);
        if (pushResult != ViewerResult_Success)
        {
            return pushResult;
        }
    }
    labelIndex = FindLabelIndex(labelName);
    NN_G3D_VIEWER_ASSERT(labelIndex >= 0);

    return PushItem(labelIndex, itemName, aliasItemName);
}

ViewerResult
EditRenderInfo::PushChoice(const char* labelName, int minValue, int maxValue) NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT_NOT_NULL(labelName);

    int labelIndex = FindLabelIndex(labelName);
    if (labelIndex == -1)
    {
        ViewerResult pushResult = PushLabel(labelName, ResRenderInfo::Type_Int);
        if (pushResult != ViewerResult_Success)
        {
            return pushResult;
        }
    }
    labelIndex = FindLabelIndex(labelName);
    NN_G3D_VIEWER_ASSERT(labelIndex >= 0);
    RenderInfoLabel* label = m_Labels.UnsafeAt(labelIndex);

    // タイプが違う場合は不正な処理なので止める
    NN_G3D_VIEWER_ASSERT(label->m_Type == ResRenderInfo::Type_Int);
    label->m_Min.iValue = minValue;
    label->m_Max.iValue = maxValue;
    return ViewerResult_Success;
}

ViewerResult
EditRenderInfo::PushChoice(const char* labelName, float minValue, float maxValue) NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT_NOT_NULL(labelName);

    int labelIndex = FindLabelIndex(labelName);
    if (labelIndex == -1)
    {
        ViewerResult pushResult = PushLabel(labelName, ResRenderInfo::Type_Float);
        if (pushResult != ViewerResult_Success)
        {
            return pushResult;
        }
    }
    labelIndex = FindLabelIndex(labelName);
    NN_G3D_VIEWER_ASSERT(labelIndex >= 0);
    RenderInfoLabel* label = m_Labels.UnsafeAt(labelIndex);

    // タイプが違う場合は不正な処理なので止める
    NN_G3D_VIEWER_ASSERT(label->m_Type == ResRenderInfo::Type_Float);
    label->m_Min.fValue = minValue;
    label->m_Max.fValue = maxValue;
    return ViewerResult_Success;
}

ViewerResult
EditRenderInfo::PushDefault(const char* labelName, const char* value) NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT_NOT_NULL(labelName);
    NN_G3D_VIEWER_ASSERT_NOT_NULL(value);

    int labelIndex = FindLabelIndex(labelName);
    if (labelIndex == -1)
    {
        ViewerResult pushResult = PushLabel(labelName, ResRenderInfo::Type_String);
        if (pushResult != ViewerResult_Success)
        {
            return pushResult;
        }
    }
    labelIndex = FindLabelIndex(labelName);
    NN_G3D_VIEWER_ASSERT(labelIndex >= 0);

    return PushValue(labelIndex, value);
}

ViewerResult
EditRenderInfo::PushDefault(const char* labelName, int value) NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT_NOT_NULL(labelName);

    int labelIndex = FindLabelIndex(labelName);
    if (labelIndex == -1)
    {
        ViewerResult pushResult = PushLabel(labelName, ResRenderInfo::Type_Int);
        if (pushResult != ViewerResult_Success)
        {
            return pushResult;
        }
    }
    labelIndex = FindLabelIndex(labelName);
    NN_G3D_VIEWER_ASSERT(labelIndex >= 0);

    return PushValue(labelIndex, value);
}

ViewerResult
EditRenderInfo::PushDefault(const char* labelName, float value) NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT_NOT_NULL(labelName);

    int labelIndex = FindLabelIndex(labelName);
    if (labelIndex == -1)
    {
        ViewerResult pushResult = PushLabel(labelName, ResRenderInfo::Type_Float);
        if (pushResult != ViewerResult_Success)
        {
            return pushResult;
        }
    }
    labelIndex = FindLabelIndex(labelName);
    NN_G3D_VIEWER_ASSERT(labelIndex >= 0);

    return PushValue(labelIndex, value);
}

void
EditRenderInfo::Clear() NN_NOEXCEPT
{
    {
        int size = m_Values.GetCount();
        for (int i = 0; i < size; ++i)
        {
            RenderInfoValue* value = m_Values.UnsafeAt(i);
            value->~RenderInfoValue();
            m_pAllocator->Free(value);
        }
        m_Values.Clear();
    }

    {
        int size = m_Items.GetCount();
        for (int i = 0;i < size; ++i)
        {
            RenderInfoItem* item = m_Items.UnsafeAt(i);
            item->~RenderInfoItem();
            m_pAllocator->Free(item);
        }
        m_Items.Clear();
    }

    {
        int size = m_Labels.GetCount();
        for (int i = 0; i < size; ++i)
        {
            RenderInfoLabel* label = m_Labels.UnsafeAt(i);
            label->~RenderInfoLabel();
            m_pAllocator->Free(label);
        }
        m_Labels.Clear();
    }
}

bool
EditRenderInfo::MakeRenderInfoPacket() NN_NOEXCEPT
{
    size_t bufferSize = CalculateWorkBufferSize();
    if (bufferSize == 0)
    {
        return false;
    }

    if (!m_WorkBuffer.Resize(bufferSize))
    {
        return false;
    }

    m_RenderInfoBufferSize = bufferSize;
    MakeRenderInfoBuffer();
    return true;
}

bool
EditRenderInfo::SendRenderInfo(EditSocketBase* pSocket) NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT_NOT_NULL(pSocket);

    bool success = pSocket->WriteSync(m_WorkBuffer.GetWorkBufferPtr(), m_RenderInfoBufferSize);

    return success;
}

ViewerResult
EditRenderInfo::PushLabel(const char* labelName, ResRenderInfo::Type type) NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT_NOT_NULL(labelName);
    // 指定したlabelName が存在しない場合に実行するので存在チェックは行わない。

    size_t bufferSize = sizeof(RenderInfoLabel);
    void* buffer = m_pAllocator->Allocate(bufferSize, nn::g3d::detail::Alignment_Default, AllocateType_Communication);
    if (buffer == nullptr)
    {
        return ViewerResult_MemoryAllocationFailed;
    }

    // ラベル登録作業
    {
        RenderInfoLabel* label = new (buffer) RenderInfoLabel(m_pAllocator);
        ViewerResult setupResult = label->Setup(labelName, type);
        if (setupResult != ViewerResult_Success)
        {
            label->~RenderInfoLabel();
            m_pAllocator->Free(label);
            return setupResult;
        }

        if (!m_Labels.PushBack(label))
        {
            label->~RenderInfoLabel();
            m_pAllocator->Free(label);
            return ViewerResult_MemoryAllocationFailed;
        }
    }
    return ViewerResult_Success;
}

int
EditRenderInfo::FindLabelIndex(const char* labelName) const NN_NOEXCEPT
{
    int size = m_Labels.GetCount();
    for (int i = 0; i < size; ++i)
    {
        RenderInfoLabel* label = m_Labels.UnsafeAt(i);
        if (label->IsEqual(labelName))
        {
            return i;
        }
    }
    return -1;
}

ViewerResult
EditRenderInfo::PushItem(int labelIndex, const char* itemName, const char* aliasItemName) NN_NOEXCEPT
{
    if (FindItemIndex(labelIndex, itemName) >= 0)
    {
        return ViewerResult_Success;
    }

    size_t bufferSize = sizeof(RenderInfoItem);
    void* buffer = m_pAllocator->Allocate(bufferSize, nn::g3d::detail::Alignment_Default, AllocateType_Communication);
    if (buffer == nullptr)
    {
        return ViewerResult_MemoryAllocationFailed;
    }

    RenderInfoItem* item = new (buffer) RenderInfoItem(m_pAllocator);
    ViewerResult setupResult = item->Setup(labelIndex, itemName, aliasItemName);
    if (setupResult != ViewerResult_Success)
    {
        item->~RenderInfoItem();
        m_pAllocator->Free(item);
        return setupResult;
    }

    if (!m_Items.PushBack(item))
    {
        item->~RenderInfoItem();
        m_pAllocator->Free(item);
        return ViewerResult_MemoryAllocationFailed;
    }
    return ViewerResult_Success;
}

int
EditRenderInfo::FindItemIndex(int labelIndex, const char* itemName) const NN_NOEXCEPT
{
    int size = m_Items.GetCount();
    for (int i = 0; i < size; ++i)
    {
        RenderInfoItem* item = m_Items.UnsafeAt(i);
        if (item->IsEqual(labelIndex, itemName))
        {
            return i;
        }
    }
    return -1;
}

ViewerResult
EditRenderInfo::PushValue(int labelIndex, const char* value) NN_NOEXCEPT
{
    size_t bufferSize = sizeof(RenderInfoValue);
    void* buffer = m_pAllocator->Allocate(bufferSize, nn::g3d::detail::Alignment_Default, AllocateType_Communication);
    if (buffer == nullptr)
    {
        return ViewerResult_MemoryAllocationFailed;
    }

    RenderInfoValue* renderInfoValue = new (buffer) RenderInfoValue(ResRenderInfo::Type_String, m_pAllocator);
    ViewerResult setupResult = renderInfoValue->Setup(labelIndex, value);
    if (setupResult != ViewerResult_Success)
    {
        renderInfoValue->~RenderInfoValue();
        m_pAllocator->Free(renderInfoValue);
        return setupResult;
    }

    if (!m_Values.PushBack(renderInfoValue))
    {
        renderInfoValue->~RenderInfoValue();
        m_pAllocator->Free(renderInfoValue);
        return ViewerResult_MemoryAllocationFailed;
    }
    return ViewerResult_Success;
}

ViewerResult
EditRenderInfo::PushValue(int labelIndex, int value) NN_NOEXCEPT
{
    size_t bufferSize = sizeof(RenderInfoValue);
    void* buffer = m_pAllocator->Allocate(bufferSize, nn::g3d::detail::Alignment_Default, AllocateType_Communication);
    if (buffer == nullptr)
    {
        return ViewerResult_MemoryAllocationFailed;
    }

    RenderInfoValue* renderInfoValue = new (buffer) RenderInfoValue(ResRenderInfo::Type_Int, m_pAllocator);
    ViewerResult setupResult = renderInfoValue->Setup(labelIndex, value);
    if (setupResult != ViewerResult_Success)
    {
        renderInfoValue->~RenderInfoValue();
        m_pAllocator->Free(renderInfoValue);
        return setupResult;
    }

    if (!m_Values.PushBack(renderInfoValue))
    {
        renderInfoValue->~RenderInfoValue();
        m_pAllocator->Free(renderInfoValue);
        return ViewerResult_MemoryAllocationFailed;
    }
    return ViewerResult_Success;
}

ViewerResult
EditRenderInfo::PushValue(int labelIndex, float value) NN_NOEXCEPT
{
    size_t bufferSize = sizeof(RenderInfoValue);
    void* buffer = m_pAllocator->Allocate(bufferSize, nn::g3d::detail::Alignment_Default, AllocateType_Communication);
    if (buffer == nullptr)
    {
        return ViewerResult_MemoryAllocationFailed;
    }

    RenderInfoValue* renderInfoValue = new (buffer) RenderInfoValue(ResRenderInfo::Type_Float, m_pAllocator);
    ViewerResult setupResult = renderInfoValue->Setup(labelIndex, value);
    if (setupResult != ViewerResult_Success)
    {
        renderInfoValue->~RenderInfoValue();
        m_pAllocator->Free(renderInfoValue);
        return setupResult;
    }

    if (!m_Values.PushBack(renderInfoValue))
    {
        renderInfoValue->~RenderInfoValue();
        m_pAllocator->Free(renderInfoValue);
        return ViewerResult_MemoryAllocationFailed;
    }
    return ViewerResult_Success;
}

size_t
EditRenderInfo::CalculateWorkBufferSize() NN_NOEXCEPT
{
    int labelSize = m_Labels.GetCount();
    int itemSize = m_Items.GetCount();
    int valueSize = m_Values.GetCount();

    if (labelSize <= 0)
    {
        return 0;
    }

    size_t size = sizeof(PacketHeader);
    size += sizeof(RenderInfoSendInfo);

    size_t itemOffsetBufferSize = sizeof(RenderInfoChoiceInfo) * itemSize;
    size_t valueOffsetBufferSize = sizeof(size_t) * valueSize;
    size += sizeof(RenderInfoLabelInfo) * labelSize;

    size_t itemOffsetArray = size;
    size += itemOffsetBufferSize;

    size_t valueOffsetArray = size;
    size += valueOffsetBufferSize;

    // Choiceの前処理設定
    {
        for (int i = 0; i < itemSize; ++i)
        {
            RenderInfoItem* item = m_Items.UnsafeAt(i);
            RenderInfoLabel* label = m_Labels.UnsafeAt(item->GetLabelIndex());
            item->m_ItemIndex = label->m_ItemCount;
            ++label->m_ItemCount;
        }
    }

    // Defaultの前処理設定
    {
        for (int i = 0; i < valueSize; ++i)
        {
            RenderInfoValue* value = m_Values.UnsafeAt(i);
            RenderInfoLabel* label = m_Labels.UnsafeAt(value->GetLabelIndex());
            value->m_ValueIndex = label->m_ValueCount;
            ++label->m_ValueCount;
        }
    }

    {
        for (int i = 0; i < labelSize; ++i)
        {
            RenderInfoLabel* label = m_Labels.UnsafeAt(i);

            label->m_Offset = size;
            size += label->m_LabelBinarySize;

            label->m_ValueOffset = valueOffsetArray;
            valueOffsetArray += sizeof(size_t) * label->m_ValueCount;

            // RenderInfoのタイプが STRING 以外はアイテムが存在しないので処理を飛ばす
            if (label->m_Type != ResRenderInfo::Type_String)
            {
                continue;
            }

            label->m_ItemOffset = itemOffsetArray;
            itemOffsetArray += sizeof(RenderInfoChoiceInfo) * label->m_ItemCount;
        }
    }

    // Choice のオフセット決定
    {
        for (int i = 0; i < itemSize; ++i)
        {
            RenderInfoItem* item = m_Items.UnsafeAt(i);
            item->m_Offset = size;
            size += item->m_ItemBinarySize;
        }
    }

    // Choice のAliasオフセット決定
    {
        for (int i = 0; i < itemSize; ++i)
        {
            RenderInfoItem* item = m_Items.UnsafeAt(i);
            if (item->m_AliasBinarySize > 0)
            {
                item->m_AliasOffset = size;
                size += item->m_AliasBinarySize;
            }
        }
    }

    // Default のオフセット決定
    {
        for (int i = 0; i < valueSize; ++i)
        {
            RenderInfoValue* value = m_Values.UnsafeAt(i);
            if (value->m_DefaultBinarySize > 0) // 文字列型のみオフセットを設定
            {
                value->m_Offset = size;
                size += value->m_DefaultBinarySize;
            }
        }
    }

    return size;
}

void
EditRenderInfo::MakeRenderInfoBuffer() NN_NOEXCEPT
{
    // CalculateWorkBufferSize が成功している前提で実行

    void* buffer = m_WorkBuffer.GetWorkBufferPtr();
    NN_G3D_VIEWER_ASSERT_NOT_NULL(buffer);

    PacketHeader* header = static_cast<PacketHeader*>(buffer);
    header->magic = NN_G3D_EDIT_MAGIC;
    header->verWord = NN_G3D_EDIT_VERSION;
    header->command = EDIT_SEND_RENDER_INFO_COMMAND_FLAG;
    // TODO: 後でsize_tに直す
    header->dataSize = static_cast<int32_t>(m_RenderInfoBufferSize - sizeof(PacketHeader));

    RenderInfoSendInfo* sendInfo = AddOffset<RenderInfoSendInfo>(buffer, sizeof(PacketHeader));

    int labelSize = m_Labels.GetCount();
    int itemSize = m_Items.GetCount();
    int valueSize = m_Values.GetCount();

    sendInfo->labelInfoNum = static_cast<uint32_t>(labelSize);
    sendInfo->modelKey = m_ModelKey;
    sendInfo->materialIndex = m_MaterialIndex;
    sendInfo->padding = 0;

    size_t labelInfoSize = sizeof(RenderInfoLabelInfo);
    RenderInfoLabelInfo* firstLabelInfo = AddOffset<RenderInfoLabelInfo>(sendInfo, sizeof(RenderInfoSendInfo));
    {
        for (int i = 0; i < labelSize; ++i)
        {
            RenderInfoLabelInfo* labelInfo = AddOffset<RenderInfoLabelInfo>(firstLabelInfo, i * labelInfoSize);
            RenderInfoLabel* label = m_Labels.UnsafeAt(i);

            // TODO: 後でsize_tに直す
            labelInfo->labelOffset = static_cast<uint32_t>(label->m_Offset);
            labelInfo->renderInfoType = label->m_Type;

            labelInfo->valueNum = label->m_ValueCount;
            // TODO: 後でsize_tに直す
            labelInfo->valueOffset = static_cast<uint32_t>(label->m_ValueOffset);

            switch(label->m_Type)
            {
            case ResRenderInfo::Type_String:
                labelInfo->itemNum = label->m_ItemCount;
                // TODO: 後でsize_tに直す
                labelInfo->itemOffset = static_cast<uint32_t>(label->m_ItemOffset);
                break;
            case ResRenderInfo::Type_Int:
                labelInfo->iMinValue = label->m_Min.iValue;
                labelInfo->iMaxValue = label->m_Max.iValue;
                break;
            case ResRenderInfo::Type_Float:
                labelInfo->fMinValue = label->m_Min.fValue;
                labelInfo->fMaxValue = label->m_Max.fValue;
                break;

            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
    }

    // Choice のオフセット配列設定
    {
        for (int i = 0; i < itemSize; ++i)
        {
            RenderInfoItem* item = m_Items.UnsafeAt(i);
            RenderInfoLabel* label = m_Labels.UnsafeAt(item->GetLabelIndex());
            NN_G3D_VIEWER_ASSERT(label->m_Type == ResRenderInfo::Type_String);

            RenderInfoChoiceInfo* itemOffsetArray = AddOffset<RenderInfoChoiceInfo>(buffer, label->m_ItemOffset);
            RenderInfoChoiceInfo* itemOffset = AddOffset<RenderInfoChoiceInfo>(itemOffsetArray, sizeof(RenderInfoChoiceInfo) * item->m_ItemIndex);
            // TODO: 後でsize_tに直す
            itemOffset->choiceOffset = static_cast<uint32_t>(item->m_Offset);
            itemOffset->padding = 0;
            if (item->m_AliasBinarySize > 0)
            {
                // TODO: 後でsize_tに直す
                itemOffset->aliasOffset = static_cast<uint32_t>(item->m_AliasOffset);
                itemOffset->aliasSize = static_cast<uint32_t>(item->m_AliasLength);
            }
            else
            {
                itemOffset->aliasOffset = 0;
                itemOffset->aliasSize = 0;
            }
        }
    }

    // Default のオフセット配列設定
    {
        for (int i = 0; i < valueSize; ++i)
        {
            RenderInfoValue* value = m_Values.UnsafeAt(i);
            RenderInfoLabel* label = m_Labels.UnsafeAt(value->GetLabelIndex());

            if (label->m_ValueCount == 0) // Defaultがない場合はスキップ
            {
                continue;;
            }

            size_t* valueOffsetArray = AddOffset<size_t>(buffer, label->m_ValueOffset);
            size_t* valueOffset = AddOffset<size_t>(valueOffsetArray, sizeof(uint32_t) * value->m_ValueIndex);
            *valueOffset = value->m_Offset; // union なので、型を無視して値を設定
        }
    }

    {
        for (int i = 0; i < labelSize; ++i)
        {
            RenderInfoLabelInfo* labelInfo = AddOffset<RenderInfoLabelInfo>(firstLabelInfo, i * labelInfoSize);
            RenderInfoLabel* label = m_Labels.UnsafeAt(i);

            // TODO: 後でsize_tに直す
            labelInfo->labelOffset = static_cast<uint32_t>(label->m_Offset);// 必要？
            labelInfo->valueNum = label->m_ValueCount;
            // TODO: 後でsize_tに直す
            labelInfo->valueOffset = static_cast<uint32_t>(label->m_ValueOffset);

            if (label->m_Type != ResRenderInfo::Type_String)
            {
                continue;;
            }

            labelInfo->itemNum = label->m_ItemCount;
            // TODO: 後でsize_tに直す
            labelInfo->itemOffset = static_cast<uint32_t>(label->m_ItemOffset);
        }
    }

    // 描画情報名を出力
    {
        for (int i = 0; i < labelSize; ++i)
        {
            RenderInfoLabel* label = m_Labels.UnsafeAt(i);
            char* name = AddOffset<char>(buffer, label->m_Offset);
            memcpy(name, label->GetName(), label->GetNameLength());
            name[label->GetNameLength()] = 0;
        }
    }

    // Choiceの文字列を出力
    {
        for (int i = 0; i < itemSize; ++i)
        {
            RenderInfoItem* item = m_Items.UnsafeAt(i);
            char* name = AddOffset<char>(buffer, item->m_Offset);
            memcpy(name, item->GetName(), item->GetNameLength());
            name[item->GetNameLength()] = 0;
        }
    }

    // Choice の Aliasを出力
    {
        for (int i = 0;i < itemSize; ++i)
        {
            RenderInfoItem* item = m_Items.UnsafeAt(i);
            if (item->m_AliasBinarySize == 0)
            {
                continue;
            }

            uint8_t* alias = AddOffset<uint8_t>(buffer, item->m_AliasOffset);
            memcpy(alias, item->m_Alias.GetWorkBufferPtr(), item->m_AliasLength);
            alias[item->m_AliasLength] = 0;
        }
    }

    // 文字列型のDefaultの文字列を出力
    {
        for (int i = 0; i < valueSize; ++i)
        {
            RenderInfoValue* value = m_Values.UnsafeAt(i);
            if (value->m_Type != ResRenderInfo::Type_String)
            {
                continue;
            }

            char* name = AddOffset<char>(buffer, value->m_Offset);
            memcpy(name, value->GetStringDefault(), value->GetStringDefaultLength());
            name[value->GetStringDefaultLength()] = 0;
        }
    }
} // NOLINT (readability/fn_size)

ViewerResult
EditRenderInfo::RenderInfoLabel::Setup(const char* labelName, ResRenderInfo::Type type) NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT_NOT_NULL(labelName);
    ViewerResult setupResult = m_Label.Assign(labelName);
    if (setupResult != ViewerResult_Success)
    {
        return setupResult;
    }

    m_LabelBinarySize = nn::util::align_up<size_t>(m_Label.GetLength() + 1, nn::g3d::detail::Alignment_Default);
    m_Type = type;
    return ViewerResult_Success;
}

bool
EditRenderInfo::RenderInfoLabel::IsEqual(const char* labelName) const NN_NOEXCEPT
{
    if (strcmp(m_Label.GetStr(), labelName) != 0)
    {
        return false;
    }
    return true;
}

ViewerResult
EditRenderInfo::RenderInfoItem::Setup(int labelIndex, const char* itemName, const char* aliasItemName) NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT(labelIndex >= 0);
    NN_G3D_VIEWER_ASSERT_NOT_NULL(itemName);
    ViewerResult assignResult = m_Item.Assign(itemName);
    if (assignResult != ViewerResult_Success)
    {
        return assignResult;
    }

    if (aliasItemName != nullptr)
    {
        size_t length = strlen(aliasItemName);
        m_AliasBinarySize = nn::util::align_up<size_t>(length + 1, nn::g3d::detail::Alignment_Default);
        if (!m_Alias.Resize(m_AliasBinarySize))
        {
            m_Item.Clear();
            return ViewerResult_MemoryAllocationFailed;
        }
        void* buffer = m_Alias.GetWorkBufferPtr();
        memcpy(buffer, aliasItemName, length + 1);
        m_AliasLength = length;
    }

    m_LabelIndex = labelIndex;
    m_ItemBinarySize = nn::util::align_up<size_t>(m_Item.GetLength() + 1, nn::g3d::detail::Alignment_Default);
    return ViewerResult_Success;
}

bool
EditRenderInfo::RenderInfoItem::IsEqual(int labelIndex, const char* itemName) const NN_NOEXCEPT
{
    if (m_LabelIndex != labelIndex)
    {
        return false;
    }
    if (strcmp(m_Item.GetStr(), itemName) != 0)
    {
        return false;
    }
    return true;
}

ViewerResult
EditRenderInfo::RenderInfoValue::Setup(int labelIndex, const char* value) NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT(labelIndex >= 0);
    NN_G3D_VIEWER_ASSERT(m_Type == ResRenderInfo::Type_String);// タイプが違う場合は不正な処理なので止める
    NN_G3D_VIEWER_ASSERT_NOT_NULL(value);
    ViewerResult assignResult = m_DefaultString.Assign(value);
    if (assignResult != ViewerResult_Success)
    {
        return assignResult;
    }
    m_Offset = 0;
    m_DefaultBinarySize = nn::util::align_up<size_t>(m_DefaultString.GetLength() + 1, nn::g3d::detail::Alignment_Default);
    m_LabelIndex = labelIndex;
    return ViewerResult_Success;
}

ViewerResult
EditRenderInfo::RenderInfoValue::Setup(int labelIndex, int value) NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT(labelIndex >= 0);
    NN_G3D_VIEWER_ASSERT(m_Type == ResRenderInfo::Type_Int);// タイプが違う場合は不正な処理なので止める
    m_iValue = value;
    m_DefaultBinarySize = 0;
    m_LabelIndex = labelIndex;
    return ViewerResult_Success;
}

ViewerResult
EditRenderInfo::RenderInfoValue::Setup(int labelIndex, float value) NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT(labelIndex >= 0);
    NN_G3D_VIEWER_ASSERT(m_Type == ResRenderInfo::Type_Float);// タイプが違う場合は不正な処理なので止める
    m_fValue = value;
    m_DefaultBinarySize = 0;
    m_LabelIndex = labelIndex;
    return ViewerResult_Success;
}

}}}} // namespace nn::g3d::viewer::detail


