﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3d_EditModelObj.h"



#include "../g3d_Allocator.h"
#include "../anim/g3d_EditAnimObj.h"
#include "../util/g3d_EditWorkBuffer.h"
#include "../util/g3d_BufferSwitcher.h"
#include "g3d_EditMaterialObj.h"
#include "g3d_EditShapeObj.h"

#include <nn/g3d/g3d_ResFile.h>
#include <nn/g3d/g3d_ResShader.h>
#include <nn/g3d/g3d_ResModel.h>
#include <nn/diag/text/diag_SdkTextG3dviewer.h>
#include <nn/mem/mem_StandardAllocator.h>

using namespace nn::g3d;
using namespace nn::g3d::viewer;
using namespace nn::g3d::viewer::detail;

namespace {

RuntimeErrorCode ValidateResModel(const ResModel* pTargetResModel, const ResModel* pOriginalResModel) NN_NOEXCEPT
{
    if (pOriginalResModel->GetMaterialCount() != pTargetResModel->GetMaterialCount())
    {
        return RUNTIME_ERROR_CODE_INVALID_MATERIAL_COUNT;
    }

    if (pTargetResModel->GetShapeCount() != pOriginalResModel->GetShapeCount())
    {
        return RUNTIME_ERROR_CODE_INVALID_SHAPE_COUNT;
    }

    for (int shapeIndex = 0, shapeCount = pTargetResModel->GetShapeCount(); shapeIndex < shapeCount; ++shapeIndex)
    {
        const ResShape* pResShape = pTargetResModel->GetShape(shapeIndex);
        const ResShape* pOrigianlResShape = pOriginalResModel->GetShape(shapeIndex);
        if (pResShape->GetMeshCount() != pOrigianlResShape->GetMeshCount())
        {
            return RUNTIME_ERROR_CODE_INVALID_MESH_COUNT;
        }

        for (int meshIndex = 0, meshCount = pResShape->GetMeshCount(); meshIndex < meshCount; ++meshIndex)
        {
            const ResMesh* pResMesh = pResShape->GetMesh(meshIndex);
            const ResMesh* pOrigianlResMesh = pOrigianlResShape->GetMesh(meshIndex);
            if (pResMesh->GetSubMeshCount() != pOrigianlResMesh->GetSubMeshCount())
            {
                return RUNTIME_ERROR_CODE_INVALID_SUBMESH_COUNT;
            }
        }
    }

    return RUNTIME_ERROR_CODE_NO_ERROR;
}

}

namespace nn { namespace g3d { namespace viewer { namespace detail {

EditModelObj::EditModelObj(
    nn::gfx::Device* pDevice,
    Allocator* pAllocator,
    ResModel* pOriginalResModel,
    CallbackCaller* pCallbackCaller) NN_NOEXCEPT
    : DeviceDependentObj(pDevice, pAllocator)
    , m_EditModelDataSets(pAllocator, nn::g3d::detail::Alignment_Default, EditModelDataSet())
    , m_pOriginalResModel(pOriginalResModel)
    , m_pOriginalResFileManagedByUser(nullptr)
    , m_pCurrentAttachedResFile(nullptr)
    , m_pCurrentAttachedResModel(nullptr)
    , m_BoundAnimPtrArray(pAllocator, nn::g3d::detail::Alignment_Default)
    , m_HasBlinkingMaterials(false)
    , m_HasBlinkingShapes(false)
    , m_ShapeLodLevel(-1)
    , m_pUpdateResFile(nullptr)
    , m_pModelName(nullptr)
    , m_pCallbackCaller(pCallbackCaller)
    , m_BoundTextureKeys(pAllocator, Alignment_Default, InvalidKey)
    , m_MemoryPoolManager(nullptr)
{
    NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(pAllocator, "%s\n", NN_G3D_VIEWER_RES_NAME(pOriginalResModel, GetName()));
    NN_G3D_VIEWER_ASSERT_NOT_NULL(pOriginalResModel);
    NN_G3D_VIEWER_ASSERT_NOT_NULL(m_pCallbackCaller);

    {
        void* buffer = m_pAllocator->Allocate(sizeof(MemoryPoolManager), nn::DefaultAlignment, AllocateType_Other);
        NN_G3D_VIEWER_ASSERT_NOT_NULL(buffer);

        m_MemoryPoolManager = new (buffer) MemoryPoolManager(UniformBlockBufferingCount, pDevice, pAllocator);
    }
}

EditModelObj::~EditModelObj() NN_NOEXCEPT
{
    DestroyEditModelObj();
    m_MemoryPoolManager->~MemoryPoolManager();
    m_pAllocator->Free(m_MemoryPoolManager);
    m_MemoryPoolManager = nullptr;
}

int EditModelObj::GetShaderCount() const NN_NOEXCEPT
{
    return m_MaterialShaderInfoArray.GetCount();
}

const ResModel* EditModelObj::GetResModel() const NN_NOEXCEPT
{
    return m_pOriginalResModel;
}

bool EditModelObj::AddBoundAnim(EditAnimObj* pEditAnimObj) NN_NOEXCEPT
{
    return m_BoundAnimPtrArray.PushBack(pEditAnimObj);
}

bool EditModelObj::RemoveBoundAnim(int index) NN_NOEXCEPT
{
    if (index < 0 || index >= m_BoundAnimPtrArray.GetCount())
    {
        return false;
    }

    EditAnimObj* pEditAnimObj = m_BoundAnimPtrArray.UnsafeAt(index);
    return RemoveBoundAnim(pEditAnimObj);
}

bool EditModelObj::RemoveBoundAnim(EditAnimObj* pEditAnimObj) NN_NOEXCEPT
{
    int index = m_BoundAnimPtrArray.IndexOf(pEditAnimObj);
    if (index < 0)
    {
        return false;
    }

    m_BoundAnimPtrArray.EraseByIndex(index);

    return true;
}

RuntimeErrorCode EditModelObj::ResetTextureRefToOriginal() NN_NOEXCEPT
{
    for (int modelIndex = 0, modelCount = m_EditModelDataSets.GetCount(); modelIndex < modelCount; ++modelIndex)
    {
        EditModelDataSet model = m_EditModelDataSets[modelIndex];
        ModelObj* pModelObj = model.pOriginalModelObj;
        for (int materialIndex = 0, materialCount = pModelObj->GetMaterialCount(); materialIndex < materialCount; ++materialIndex)
        {
            MaterialObj* pMaterialObj = pModelObj->GetMaterial(materialIndex);
            pMaterialObj->ClearTexture();
        }
    }

    return RUNTIME_ERROR_CODE_NO_ERROR;
}

int EditModelObj::GetBoundAnimCount() const NN_NOEXCEPT
{
    return m_BoundAnimPtrArray.GetCount();
}

EditAnimObj* EditModelObj::GetBoundAnimAt(int index) NN_NOEXCEPT
{
    return m_BoundAnimPtrArray.UnsafeAt(index);
}

int EditModelObj::GetIndexOfBoundAnim(EditAnimObj* pEditAnimObj) NN_NOEXCEPT
{
    return m_BoundAnimPtrArray.IndexOf(pEditAnimObj);
}

void EditModelObj::PrintBoundAnimInfo() NN_NOEXCEPT
{
    int animCount = m_BoundAnimPtrArray.GetCount();
    NN_G3D_VIEWER_DEBUG_PRINT("Bound Anims for Model %s\n", GetName());
    for (int animIndex = 0; animIndex < animCount; ++animIndex)
    {
        EditAnimObj* pEditAnimObj = m_BoundAnimPtrArray.UnsafeAt(animIndex);
        NN_UNUSED(pEditAnimObj);
        NN_G3D_VIEWER_DEBUG_PRINT("  [%d] %s(%d)\n", animIndex, pEditAnimObj->GetName(), pEditAnimObj->GetResFileKey());
    }
}

void
EditModelObj::UpdateRenderInfo(
    int materialIndex,
    const void* pRenderInfoUpdateCommandData, size_t renderInfoUpdateCommandDataSize,
    ptrdiff_t renderInfoArrayOffset) NN_NOEXCEPT
{
    for (int index = 0, end = m_EditModelDataSets.GetCount(); index < end; ++index)
    {
        NN_G3D_VIEWER_ASSERT_DETAIL(
            materialIndex >= 0 && materialIndex < m_EditModelDataSets[index].materialArray.GetCount(), "%s\n", m_pOriginalResModel->GetName());
        EditMaterialObj* pMaterialObj = m_EditModelDataSets[index].materialArray.UnsafeAt(materialIndex);
        pMaterialObj->UpdateRenderInfo(pRenderInfoUpdateCommandData, renderInfoUpdateCommandDataSize,
            renderInfoArrayOffset);
    }
}

RuntimeErrorCode
EditModelObj::AttachModelObjs(
    nn::g3d::ResFile* pAttachModelResFile, nn::g3d::ModelObj** pAttachTargetModelObjs, int modelObjCount) NN_NOEXCEPT
{
    NN_G3D_VIEWER_REQUIRES_NOT_NULL_DETAIL(pAttachTargetModelObjs, m_pOriginalResModel->GetName());
    if (modelObjCount == 0)
    {
        return RUNTIME_ERROR_CODE_NO_ERROR;
    }

    {
        NN_G3D_VIEWER_ASSERT(m_EditModelDataSets.GetCount() == 0);

        for (int modelIndex = 0; modelIndex < modelObjCount; ++modelIndex)
        {
            EditModelDataSet editData;;
            nn::g3d::ModelObj* pModelObj = pAttachTargetModelObjs[modelIndex];
            editData.pOriginalModelObj = pModelObj;
            editData.modelObjKey = ViewerKeyManager::GetInstance().FindKey(pModelObj);
            NN_G3D_VIEWER_DEBUG_PRINT("AttachModelObjs: EditModelObj = 0x%p, pModelObj = 0x%p(key = %d)\n", this, pModelObj, editData.modelObjKey);

            memcpy(&editData.destroyCheckData, pModelObj, s_CheckByteSize);

            // スケルトン初期化
            {
                void* buffer = m_pAllocator->Allocate(sizeof(EditSkeletonObj), Alignment_Default, AllocateType_EditObj);
                if (buffer == nullptr) // バッファ確保失敗
                {
                    return RUNTIME_ERROR_CODE_INSUFFICIENT_MEMORY;
                }

                editData.pEditSkeletonObj = new (buffer) EditSkeletonObj(pModelObj, pModelObj->GetSkeleton(), m_pDevice, m_pAllocator);
           }

            // マテリアル初期化
            {
                int materialCount = pModelObj->GetMaterialCount();
                if (materialCount > 0)
                {
                    size_t bufferSize = FixedSizeArray<EditMaterialObj>::CalculateBufferSize(materialCount);
                    void* buffer = m_pAllocator->Allocate(bufferSize, nn::g3d::detail::Alignment_Default, AllocateType_EditObj);
                    if (buffer == nullptr) // バッファ確保失敗
                    {
                        editData.pEditSkeletonObj->ResetToOriginal();
                        editData.pEditSkeletonObj->~EditSkeletonObj();
                        m_pAllocator->Free(editData.pEditSkeletonObj);
                        return RUNTIME_ERROR_CODE_INSUFFICIENT_MEMORY;
                    }

                    editData.materialArray.SetArrayBuffer(buffer, materialCount);
                    for(int materialIndex = 0; materialIndex < materialCount; ++materialIndex)
                    {
                        nn::g3d::MaterialObj* pMaterialObj = pModelObj->GetMaterial(materialIndex);
                        EditMaterialObj* pEditMaterialObj = editData.materialArray.UnsafeAt(materialIndex);
                        new (pEditMaterialObj) EditMaterialObj(m_pDevice, m_pAllocator, materialIndex, pModelObj, pMaterialObj);
                    }
                }
            }

            // シェイプ初期化
            {
                int shapeCount = pModelObj->GetShapeCount();
                if (shapeCount > 0)
                {
                    size_t bufferSize = FixedSizeArray<EditShapeObj>::CalculateBufferSize(shapeCount);
                    void* buffer = m_pAllocator->Allocate(bufferSize, nn::g3d::detail::Alignment_Default, AllocateType_EditObj);
                    if (buffer == nullptr) // バッファ確保失敗
                    {
                        ResetMaterialsToOriginal();
                        editData.pEditSkeletonObj->ResetToOriginal();
                        editData.pEditSkeletonObj->~EditSkeletonObj();
                        m_pAllocator->Free(editData.pEditSkeletonObj);
                        return RUNTIME_ERROR_CODE_INSUFFICIENT_MEMORY;
                    }

                    editData.shapeArray.SetArrayBuffer(buffer, shapeCount);
                    for(int shapeIndex = 0; shapeIndex < shapeCount; ++shapeIndex)
                    {
                        nn::g3d::ShapeObj* pShapeObj = pModelObj->GetShape(shapeIndex);
                        EditShapeObj* pEditShapeObj = editData.shapeArray.UnsafeAt(shapeIndex);
                        new (pEditShapeObj) EditShapeObj(m_pDevice, m_pAllocator, shapeIndex, pModelObj, pShapeObj);
                    }
                }
            }

            pModelObj->UpdateViewDependency();

            m_EditModelDataSets.PushBack(editData);
        }

        {
            NN_G3D_VIEWER_ASSERT(m_pModelName == nullptr);
            size_t length = strlen(m_pOriginalResModel->GetName());
            m_pModelName = reinterpret_cast<char*>(m_pAllocator->Allocate(length + 1, Alignment_Default, AllocateType_Other));
            strcpy(m_pModelName, m_pOriginalResModel->GetName());
        }
    }

    return ReloadResFile(pAttachModelResFile);
} // NOLINT (readability/fn_size)

void
EditModelObj::DestroyEditModelObj() NN_NOEXCEPT
{
    m_BoundAnimPtrArray.Destroy();

    for (int modelIndex = 0, modelCount = m_EditModelDataSets.GetCount(); modelIndex < modelCount; ++modelIndex)
    {
        EditModelDataSet* pEditData = &m_EditModelDataSets[modelIndex];

        // マテリアルの破棄
        for(int materialIndex = 0, materialCount = pEditData->materialArray.GetCount(); materialIndex < materialCount; ++materialIndex)
        {
            EditMaterialObj* pEditMaterialObj = pEditData->materialArray.UnsafeAt(materialIndex);
            pEditMaterialObj->~EditMaterialObj();
        }

        void* matBuffer = pEditData->materialArray.GetBufferPtr();
        if (matBuffer != nullptr)
        {
            pEditData->materialArray.Clear();
            m_pAllocator->Free(matBuffer);
        }

        // シェイプの破棄
        {
            for(int shapeIndex = 0, shapeCount = pEditData->shapeArray.GetCount(); shapeIndex < shapeCount; ++shapeIndex)
            {
                EditShapeObj* pEditShapeObj = pEditData->shapeArray.UnsafeAt(shapeIndex);
                pEditShapeObj->~EditShapeObj();
            }

            void* shapeBuffer = pEditData->shapeArray.GetBufferPtr();
            if (shapeBuffer != nullptr)
            {
                pEditData->shapeArray.Clear();
                m_pAllocator->Free(shapeBuffer);
            }
        }

        pEditData->pEditSkeletonObj->ClearBuffers();
        pEditData->pEditSkeletonObj->~EditSkeletonObj();
        m_pAllocator->Free(pEditData->pEditSkeletonObj);
    }

    DestroyShaders();

    if (m_pModelName != nullptr)
    {
        m_pAllocator->Free(m_pModelName);
        m_pModelName = nullptr;
    }
}

RuntimeErrorCode
EditModelObj::ResetToOriginal() NN_NOEXCEPT
{
    if (m_pCurrentAttachedResModel == nullptr)
    {
        // 未アタッチ状態は正常系とする
        return RUNTIME_ERROR_CODE_NO_ERROR;
    }

    // オリジナルモデルに変更があるかをチェックする
    // 変更があった場合はアプリケーションの実装の問題で元モデルが破棄されてしまった可能性がある
    NN_G3D_VIEWER_ASSERT_NOT_NULL(m_pModelName);
    for (int i = 0, end = m_EditModelDataSets.GetCount(); i < end; ++i)
    {
        EditModelDataSet* pEditData = &m_EditModelDataSets[i];
        NN_G3D_VIEWER_ASSERT_DETAIL(0 == memcmp(&pEditData->destroyCheckData, pEditData->pOriginalModelObj, s_CheckByteSize),
            "Model \"%s\" may have been destoyed or rewritten illegally before detatching from edit library",
            m_pModelName);
    }

    RuntimeErrorCode errorCode = SwapResModel(m_pCurrentAttachedResModel, m_pOriginalResModel, false, m_pCallbackCaller);
    if (errorCode != RUNTIME_ERROR_CODE_NO_ERROR) {
        return errorCode;
    }

    // サンプラやシェーダーアサインに変更がある可能性があるので、
    // テクスチャパターンアニメーションとシェーダーパラメータアニメーションを再バインドする
    bool successRebindTexPatternAnim = RebindTexPatternOrShaderParamAnims();
    NN_UNUSED(successRebindTexPatternAnim);
    // とりあえず再バインド失敗は無視しておく

    ResetModelsToOriginal();

    if (IsOriginalModelFrom3dEditor())
    {
        UnbindAllTexturesWithoutCallback();
    }

    for (int i = 0, end = m_EditModelDataSets.GetCount(); i < end; ++i)
    {
        ModelDetachedArg arg;
        arg.pModelObj = m_EditModelDataSets[i].pOriginalModelObj;
        m_pCallbackCaller->Call(CallbackType_ModelDetached, &arg);
    }

    return RUNTIME_ERROR_CODE_NO_ERROR;
}

RuntimeErrorCode
EditModelObj::ReloadResFile(nn::g3d::ResFile* pResFile) NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT_NOT_NULL(pResFile);
    NN_G3D_VIEWER_ASSERT_DETAIL(pResFile->GetModelCount() > 0, "%s\n", NN_G3D_VIEWER_RES_NAME(pResFile, GetName()));

    nn::g3d::ResModel* resModel = pResFile->GetModel(0);
    NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(resModel, "%s\n", NN_G3D_VIEWER_RES_NAME(pResFile, GetName()));

    RuntimeErrorCode checkResult = ValidateResModel(resModel, m_pOriginalResModel);
    if (checkResult != RUNTIME_ERROR_CODE_NO_ERROR)
    {
        return checkResult;
    }

    RuntimeErrorCode errorCode = SwapResModel(m_pCurrentAttachedResModel, resModel, false, m_pCallbackCaller);

    m_pCurrentAttachedResFile = pResFile;
    m_pCurrentAttachedResModel = resModel;

    // サンプラやシェーダーアサインに変更がある可能性があるので、
    // テクスチャパターンアニメーションとシェーダーパラメータアニメーションを再バインドする
    bool successRebindTexPatternAnim = RebindTexPatternOrShaderParamAnims();
    NN_UNUSED(successRebindTexPatternAnim);
    // とりあえず再バインド失敗は無視しておく

    return errorCode;
}

void
EditModelObj::ResetModelsToOriginal() NN_NOEXCEPT
{
    m_BoundAnimPtrArray.Clear();

    ResetMaterialsToOriginal();
    ResetShapesToOriginal();
    for (int modelIndex = 0, modelCount = m_EditModelDataSets.GetCount(); modelIndex < modelCount; ++modelIndex)
    {
        EditModelDataSet* pEditData = &m_EditModelDataSets[modelIndex];
        pEditData->pEditSkeletonObj->ResetToOriginal();
    }
}

void
EditModelObj::ResetMaterialsToOriginal() NN_NOEXCEPT
{
    for (int modelIndex = 0, modelCount = m_EditModelDataSets.GetCount(); modelIndex < modelCount; ++modelIndex)
    {
        EditModelDataSet* pEditData = &m_EditModelDataSets[modelIndex];
        for(int materialIndex = 0, materialCount = pEditData->materialArray.GetCount(); materialIndex < materialCount; ++materialIndex)
        {
            EditMaterialObj* pEditMaterialObj = pEditData->materialArray.UnsafeAt(materialIndex);
            pEditMaterialObj->ResetToOriginal();
        }
    }
}

void
EditModelObj::ResetShapesToOriginal() NN_NOEXCEPT
{
    for (int modelIndex = 0, modelCount = m_EditModelDataSets.GetCount(); modelIndex < modelCount; ++modelIndex)
    {
        EditModelDataSet* pEditData = &m_EditModelDataSets[modelIndex];
        for(int shapeIndex = 0, shapeCount = pEditData->shapeArray.GetCount(); shapeIndex < shapeCount; ++shapeIndex)
        {
            EditShapeObj* pEditShapeObj = pEditData->shapeArray.UnsafeAt(shapeIndex);
            pEditShapeObj->ResetToOriginal();
        }
    }
}

void
EditModelObj::DestroyShaders() NN_NOEXCEPT
{
    if (void* buffer = m_MaterialShaderInfoArray.GetBufferPtr())
    {
        m_MaterialShaderInfoArray.Clear();
        m_pAllocator->Free(buffer);
    }
}

void
EditModelObj::UpdateMaterialBlinking() NN_NOEXCEPT
{
    if (!m_HasBlinkingMaterials)
    {
        return;
    }

    m_HasBlinkingMaterials = false;
    for (int modelIndex = 0, modelCount = m_EditModelDataSets.GetCount(); modelIndex < modelCount; ++modelIndex)
    {
        EditModelDataSet* pEditData = &m_EditModelDataSets[modelIndex];
        for(int materialIndex = 0, materialCount = pEditData->materialArray.GetCount(); materialIndex < materialCount; ++materialIndex)
        {
            EditMaterialObj* pEditMaterialObj = pEditData->materialArray.UnsafeAt(materialIndex);
            if (!pEditMaterialObj->IsBlinking())
            {
                continue;
            }

            pEditMaterialObj->UpdateBlinking();
            m_HasBlinkingMaterials = true;
        }
    }
}

void
EditModelObj::StartMaterialBlinking(const TargetSelectedArg& arg) NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(arg.index, "%s\n", m_pOriginalResModel->GetName());
    NN_G3D_VIEWER_ASSERT_DETAIL(arg.indexCount > 0, "%s\n", m_pOriginalResModel->GetName());

    for (int i = 0; i < arg.indexCount; ++i)
    {
        for (int modelIndex = 0, modelCount = m_EditModelDataSets.GetCount(); modelIndex < modelCount; ++modelIndex)
        {
            EditModelDataSet* pEditData = &m_EditModelDataSets[modelIndex];
            EditMaterialObj* pEditMaterialObj = pEditData->materialArray.UnsafeAt(arg.index[i]);
            if (pEditData->pOriginalModelObj->IsMaterialVisible(arg.index[i]))
            {
                pEditMaterialObj->StartBlinking();
            }
        }
    }

    m_HasBlinkingMaterials = true;
}

void
EditModelObj::BreakMaterialBlinking(const TargetSelectedArg& arg) NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(arg.index, "%s\n", m_pOriginalResModel->GetName());
    NN_G3D_VIEWER_ASSERT_DETAIL(arg.indexCount > 0, "%s\n", m_pOriginalResModel->GetName());

    for (int i = 0; i < arg.indexCount; ++i)
    {
        for (int modelIndex = 0, modelCount = m_EditModelDataSets.GetCount(); modelIndex < modelCount; ++modelIndex)
        {
            EditModelDataSet* pEditData = &m_EditModelDataSets[modelIndex];
            EditMaterialObj* pEditMaterialObj = pEditData->materialArray.UnsafeAt(arg.index[i]);
            pEditMaterialObj->BreakBlinking();
        }
    }
}

void EditModelObj::StartShapeBlinking(const TargetSelectedArg& arg) NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(arg.index, "%s\n", m_pOriginalResModel->GetName());
    NN_G3D_VIEWER_ASSERT_DETAIL(arg.indexCount > 0, "%s\n", m_pOriginalResModel->GetName());

    for (int modelIndex = 0, modelCount = m_EditModelDataSets.GetCount(); modelIndex < modelCount; ++modelIndex)
    {
        EditModelDataSet* pEditData = &m_EditModelDataSets[modelIndex];
        for (int i = 0; i < arg.indexCount; ++i)
        {
            EditShapeObj* pEditShapeObj = pEditData->shapeArray.UnsafeAt(arg.index[i]);
            pEditShapeObj->StartBlinking();
        }
    }

    m_HasBlinkingShapes = true;
}

void EditModelObj::StartModelBlinking() NN_NOEXCEPT
{
    const ResModel* pResModel = m_pOriginalResModel;
    int shapeCount = pResModel->GetShapeCount();
    for (int modelIndex = 0, modelCount = m_EditModelDataSets.GetCount(); modelIndex < modelCount; ++modelIndex)
    {
        EditModelDataSet* pEditData = &m_EditModelDataSets[modelIndex];
        for (int shapeIndex = 0; shapeIndex < shapeCount; ++shapeIndex)
        {
            EditShapeObj* pEditShapeObj = pEditData->shapeArray.UnsafeAt(shapeIndex);
            pEditShapeObj->StartBlinking();
        }
    }

    m_HasBlinkingShapes = true;
}

void
EditModelObj::UpdateShapeBlinking() NN_NOEXCEPT
{
    if (!m_HasBlinkingShapes)
    {
        return;
    }

    m_HasBlinkingShapes = false;
    bool isShapeUpdated = false;
    for (int modelIndex = 0, modelCount = m_EditModelDataSets.GetCount(); modelIndex < modelCount; ++modelIndex)
    {
        EditModelDataSet* pEditData = &m_EditModelDataSets[modelIndex];
        for(int shapeIndex = 0, shapeCount = pEditData->shapeArray.GetCount(); shapeIndex < shapeCount; ++shapeIndex)
        {
            EditShapeObj* pEditShapeObj = pEditData->shapeArray.UnsafeAt(shapeIndex);
            if (!pEditShapeObj->IsBlinking())
            {
                continue;
            }

            isShapeUpdated |= pEditShapeObj->UpdateBlinking();
            m_HasBlinkingShapes = true;
        }
    }

    if (isShapeUpdated)
    {
        CallShapeUpdatedCallback(m_pCallbackCaller);
    }
}

void
EditModelObj::SetLodLevel(int level) NN_NOEXCEPT
{
    m_ShapeLodLevel = level;

    for (int modelIndex = 0, modelCount = m_EditModelDataSets.GetCount(); modelIndex < modelCount; ++modelIndex)
    {
        EditModelDataSet* pEditData = &m_EditModelDataSets[modelIndex];
        for (int shapeIndex = 0, shapeCount = pEditData->shapeArray.GetCount(); shapeIndex < shapeCount; ++shapeIndex)
        {
            EditShapeObj* pEditShapeObj = pEditData->shapeArray.UnsafeAt(shapeIndex);
            pEditShapeObj->SetLodLevel(level);
        }
    }

    CallShapeUpdatedCallback(m_pCallbackCaller);
}

void
EditModelObj::ResetLodLevel() NN_NOEXCEPT
{
    m_ShapeLodLevel = -1;

    for (int modelIndex = 0, modelCount = m_EditModelDataSets.GetCount(); modelIndex < modelCount; ++modelIndex)
    {
        EditModelDataSet* pEditData = &m_EditModelDataSets[modelIndex];
        for (int shapeIndex = 0, shapeCount = pEditData->shapeArray.GetCount(); shapeIndex < shapeCount; ++shapeIndex)
        {
            EditShapeObj* pEditShapeObj = pEditData->shapeArray.UnsafeAt(shapeIndex);
            pEditShapeObj->ResetLodLevel();
        }
    }

    CallShapeUpdatedCallback(m_pCallbackCaller);
}

void EditModelObj::ResetSkeletonsToOriginal() NN_NOEXCEPT
{
    for (int modelIndex = 0, modelCount = m_EditModelDataSets.GetCount(); modelIndex < modelCount; ++modelIndex)
    {
        EditModelDataSet* pEditData = &m_EditModelDataSets[modelIndex];
        pEditData->pEditSkeletonObj->ResetToOriginal();
    }
}

void EditModelObj::CallShaderAssignUpdatedCallback(
        nn::g3d::ResModel* pOldResModel,
        nn::g3d::ResModel* pNewResModel,
        bool useShaders,
        CallbackCaller* pCallbackCaller) NN_NOEXCEPT
{
    int materialCount = GetResModel()->GetMaterialCount();
    size_t shaderArchiveArraySize = sizeof(nn::g3d::ResShaderArchive*) * materialCount;
    ScopedAllocator scopedAllocator(*m_pAllocator, shaderArchiveArraySize, nn::DefaultAlignment, AllocateType_Resource);

    for (int modelIndex = 0, modelCount = m_EditModelDataSets.GetCount(); modelIndex < modelCount; ++modelIndex)
    {
        EditModelDataSet* pEditData = &m_EditModelDataSets[modelIndex];

        ShaderAssignUpdatedArg arg;
        arg.pOldResModel = pOldResModel;
        arg.pNewResModel = pNewResModel;
        arg.pModelObj = pEditData->pOriginalModelObj;
        arg.state = ShaderAssignUpdatedArg::State_ModelUpdate;
        arg.pMaterialShaderInfos = nullptr;
        arg.pShaderArchives = nullptr;

        if (useShaders)
        {
            arg.pShaderArchives = scopedAllocator.GetAllocatedBuffer<nn::g3d::ResShaderArchive*>();
            ShaderAssignUpdatedArg::MaterialShaderInfo* pMaterialShaderInfos = m_MaterialShaderInfoArray.GetArrayBufferPtr();
            arg.pMaterialShaderInfos = m_MaterialShaderInfoArray.GetArrayBufferPtr();
            for (int matIndex = 0; matIndex < materialCount; ++matIndex)
            {
                arg.pShaderArchives[matIndex] = pMaterialShaderInfos[matIndex].pResShaderArchive;
            }
        }

        // nullptr ならばオリジナルのResModelを入れる。
        // 更新前のResModelがオリジナルの場合はフラグを有効にする。
        // モデルをアタッチした場合は以下の状態にする。
        if (pOldResModel == nullptr)
        {
            arg.pOldResModel = m_pOriginalResModel;
            arg.state = ShaderAssignUpdatedArg::State_ModelBegin;
        }

        // 更新用のResModelがオリジナルの場合はフラグを有効にする。
        // モデルをデタッチした場合は以下の状態にする。
        if (m_pOriginalResModel == pNewResModel)
        {
            arg.state = ShaderAssignUpdatedArg::State_ModelEnd;
        }

        pCallbackCaller->Call(CallbackType_ShaderAssignUpdated, &arg);
    }
}

void EditModelObj::CallUpdateMaterialsCallback(
        nn::g3d::ResModel* pOldResModel,
        nn::g3d::ResModel* pNewResModel,
        CallbackCaller* pCallbackCaller) NN_NOEXCEPT
{
    for (int modelIndex = 0, modelCount = m_EditModelDataSets.GetCount(); modelIndex < modelCount; ++modelIndex)
    {
        EditModelDataSet* pEditData = &m_EditModelDataSets[modelIndex];

        MaterialUpdatedArg arg;
        arg.pModelObj = pEditData->pOriginalModelObj;
        arg.state = MaterialUpdatedArg::State_Update;

        // pOldResModel が nullptr の場合は初回実行
        if (pOldResModel == nullptr)
        {
            arg.state = MaterialUpdatedArg::State_Begin;
        }

        // pNewResModel が m_pOriginalResModel と同一の場合は終了時実行
        if (m_pOriginalResModel == pNewResModel)
        {
            arg.state = MaterialUpdatedArg::State_End;
        }

        pCallbackCaller->Call(CallbackType_MaterialUpdated, &arg);
    }
}

void EditModelObj::CallModelAttachedCallback(
    CallbackCaller* pCallbackCaller) NN_NOEXCEPT
{
    for (int modelIndex = 0, modelCount = m_EditModelDataSets.GetCount(); modelIndex < modelCount; ++modelIndex)
    {
        EditModelDataSet* pEditData = &m_EditModelDataSets[modelIndex];
        nn::g3d::viewer::ModelAttachedArg arg;
        arg.pModelObj = pEditData->pOriginalModelObj;
        pCallbackCaller->Call(CallbackType_ModelAttached, &arg);
    }
}

void EditModelObj::CallShapeUpdatedCallback(CallbackCaller* pCallback) NN_NOEXCEPT
{
    for (int modelIndex = 0, modelCount = GetAttachedModelObjCount(); modelIndex < modelCount; ++modelIndex)
    {
        ShapeUpdatedArg arg;
        arg.pModelObj = GetTargetModelObj(modelIndex);
        pCallback->Call(CallbackType_ShapeUpdated, &arg);
    }
}

size_t
EditModelObj::CalculateBlockBufferSize(const nn::g3d::ResModel* pNewResModel) const NN_NOEXCEPT
{
    size_t bufferSize = 0;

    for (int modelIndex = 0, modelCount = m_EditModelDataSets.GetCount(); modelIndex < modelCount; ++modelIndex)
    {
        const EditModelDataSet* pEditData = &m_EditModelDataSets[modelIndex];

        // Skeleton
        const nn::g3d::ResSkeleton* pNewResSkeleton = pNewResModel->GetSkeleton();
        bufferSize += nn::util::align_up(pEditData->pEditSkeletonObj->CalculateBlockBufferSize(pNewResSkeleton),
            nn::gfx::util::MemoryPoolAllocator::AllocatorUnitSize);

        // Shape
        for (int shapeIndex = 0, shapeCount = pEditData->shapeArray.GetCount(); shapeIndex < shapeCount; ++shapeIndex)
        {
            const nn::g3d::ResShape* pNewResShape = pNewResModel->GetShape(shapeIndex);
            const ResBone* pNewResBone = pNewResSkeleton->GetBone(pNewResShape->GetBoneIndex());
            bool isViewDependent = (pNewResBone->GetBillboardMode() != ResBone::Flag_BillboardNone) ? true : false;

            const EditShapeObj* pEditShapeObj = pEditData->shapeArray.UnsafeAt(shapeIndex);
            bufferSize += nn::util::align_up(pEditShapeObj->CalculateBlockBufferSize(isViewDependent, pNewResShape),
                nn::gfx::util::MemoryPoolAllocator::AllocatorUnitSize);
        }

        // Material
        for (int materialIndex = 0, materialCount = pEditData->materialArray.GetCount(); materialIndex < materialCount; ++materialIndex)
        {
            const nn::g3d::ResMaterial* pResMaterial = pNewResModel->GetMaterial(materialIndex);
            const EditMaterialObj* pEditMaterialObj = pEditData->materialArray.UnsafeAt(materialIndex);
            bufferSize += nn::util::align_up(pEditMaterialObj->CalculateBlockBufferSize(pResMaterial),
                nn::gfx::util::MemoryPoolAllocator::AllocatorUnitSize);
        }
    }
    return bufferSize;
}

RuntimeErrorCode
EditModelObj::SwapResModel(
    nn::g3d::ResModel* pOldResModel,
    nn::g3d::ResModel* pNewResModel,
    bool useShaders,
    CallbackCaller* pCallbackCaller) NN_NOEXCEPT
{
    // シェーダーアサイン情報が変わったことを通知
    CallShaderAssignUpdatedCallback(pOldResModel, pNewResModel, useShaders, pCallbackCaller);

    // 一度、オリジナルに戻す
    ResetMaterialsToOriginal();
    ResetShapesToOriginal();
    ResetSkeletonsToOriginal();

    // モデルインスタンスを再構築する前に必要なサイズのメモリプールを確保
    {
        m_MemoryPoolManager->SwitchCurrentBuffer();

        // モデル内のユニフォームブロックを切り替え
        // 既に確保済みのブロックバッファが存在する場合は破棄します。
        SwitchWorkBlockBuffers();

        size_t workBlockSize = CalculateBlockBufferSize(pNewResModel);
        if (workBlockSize > 0)
        {
            // TODO: 本当はここでリングバッファが一巡したら、MemoryPoolAllocator の Free を呼び出したいが、
            // 現状、EditShapeObj 等 が最新のオフセットのみ保持しており、呼び出せる状況にありません
            bool success = m_MemoryPoolManager->ResizeMemoryPoolBuffer(workBlockSize);
            if (!success)
            {
                return RUNTIME_ERROR_CODE_INSUFFICIENT_MEMORY;
            }
        }
    }

    // ModelObj を新しいリソースで再構築
    RebuildModelObj(pNewResModel);

    CallShapeUpdatedCallback(pCallbackCaller);
    CallUpdateMaterialsCallback(pOldResModel, pNewResModel, pCallbackCaller);

    if (m_ShapeLodLevel >= 0 && m_pOriginalResModel != pNewResModel)
    {
        SetLodLevel(m_ShapeLodLevel);
    }

    if (pOldResModel == nullptr)
    {
        CallModelAttachedCallback(pCallbackCaller);
    }

    return RUNTIME_ERROR_CODE_NO_ERROR;
}

void
EditModelObj::ResizeAndResetMaterialShaderInfoArray(int size) NN_NOEXCEPT
{
    if (size > m_MaterialShaderInfoArray.GetCount())
    {
        // 配列のリアロケート
        void* buffer = m_MaterialShaderInfoArray.GetBufferPtr();
        if (buffer != nullptr)
        {
            m_pAllocator->Free(buffer);
            m_MaterialShaderInfoArray.Clear();
            buffer = nullptr;
        }

        {
            size_t bufferSize = m_MaterialShaderInfoArray.CalculateBufferSize(size);
            buffer = m_pAllocator->Allocate(bufferSize, nn::g3d::detail::Alignment_Default, AllocateType_Other);
            NN_G3D_VIEWER_ASSERT_NOT_NULL(buffer);
            memset(buffer, 0, bufferSize);
            m_MaterialShaderInfoArray.SetArrayBuffer(buffer, size);
        }
    }

    // 要素の初期化
    for (int infoIndex = 0, infoCount = m_MaterialShaderInfoArray.GetCount(); infoIndex < infoCount; ++infoIndex)
    {
        ShaderAssignUpdatedArg::MaterialShaderInfo* pInfo = m_MaterialShaderInfoArray.UnsafeAt(infoIndex);
        *pInfo = ShaderAssignUpdatedArg::MaterialShaderInfo();
    }
}

void
EditModelObj::SetMaterialShaderInfo(int index, const ShaderAssignUpdatedArg::MaterialShaderInfo& info) NN_NOEXCEPT
{
    ShaderAssignUpdatedArg::MaterialShaderInfo* pTarget = m_MaterialShaderInfoArray.UnsafeAt(index);
    NN_G3D_VIEWER_ASSERT(pTarget->pResShaderArchive == nullptr);
    *pTarget = info;
}


RuntimeErrorCode
EditModelObj::UpdateShaders(bool useShaders) NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT(m_MaterialShaderInfoArray.GetCount() > 0 && m_MaterialShaderInfoArray.GetArrayBufferPtr() != nullptr);

    NN_G3D_VIEWER_ASSERT_NOT_NULL(m_pUpdateResFile);
    NN_G3D_VIEWER_ASSERT(m_pUpdateResFile->GetModelCount() > 0);

    nn::g3d::ResModel* resModel = m_pUpdateResFile->GetModel(0);
    NN_G3D_VIEWER_ASSERT_NOT_NULL(resModel);

    RuntimeErrorCode checkResult = ValidateResModel(resModel, m_pOriginalResModel);
    if (checkResult != RUNTIME_ERROR_CODE_NO_ERROR)
    {
        return checkResult;
    }

    RuntimeErrorCode errorCode = SwapResModel(m_pCurrentAttachedResModel, resModel, useShaders, m_pCallbackCaller);
    if (errorCode != RUNTIME_ERROR_CODE_NO_ERROR) {
        return errorCode;
    }

    m_pCurrentAttachedResFile = m_pUpdateResFile;
    m_pCurrentAttachedResModel = resModel;

    // サンプラやシェーダーアサインに変更がある可能性があるので、
    // テクスチャパターンアニメーションとシェーダーパラメータアニメーションを再バインドする
    bool successRebindTexPatternAnim = RebindTexPatternOrShaderParamAnims();
    NN_UNUSED(successRebindTexPatternAnim);
    // とりあえず再バインド失敗は無視しておく

    SetUpdateResFile(nullptr);

    return RUNTIME_ERROR_CODE_NO_ERROR;
}

bool EditModelObj::RebindTexPatternOrShaderParamAnims() NN_NOEXCEPT
{
    bool success = true;
    for (int animIndex = 0, animCount = m_BoundAnimPtrArray.GetCount(); animIndex < animCount; ++animIndex)
    {
        EditAnimObj* pEditAnimObj = m_BoundAnimPtrArray.UnsafeAt(animIndex);
        ViewerAnimKind kind = pEditAnimObj->GetAnimKind();
        if ((kind != ViewerAnimKind_TexturePatternAnim)
         && (kind != ViewerAnimKind_MaterialAnim)
         && (kind != ViewerAnimKind_ShaderParamAnim))
        {
            continue;
        }

        NN_G3D_VIEWER_ASSERT(pEditAnimObj->IsModelBound(this));
        pEditAnimObj->UnbindModelObj(this);
        BindResult result = pEditAnimObj->BindModelObj(this);
        if (!result.IsComplete())
        {
            // サンプラがないのにテクスチャパターンアニメーションは設定されている場合や
            // シェーダーパラメータアニメーションが設定された状態でシェーダーアサインが変更された場合などは
            // 正常系でバインドが失敗するのでエラーにせず無視するが、不具合でバインドできない場合のために念のためエラーログだけ出しておく
            NN_G3D_VIEWER_ERROR_LOG(
                NN_TEXT_G3DVIEWER("モデル \"%s\" へのアニメーション \"%s\" の再バインドに失敗しました。\n"),
                this->GetName(), pEditAnimObj->GetName());

            // TODO: 作業中のデザイナが気づけるようにユーザメッセージ機能を応用して警告ログを出すようにするのを検討する
        }

        success &= result.IsComplete();
    }

    return success;
}

void EditModelObj::ReloadTexture(const nn::gfx::ResTextureFile* pOldResTextureFile, const nn::gfx::ResTextureFile* pResTextureFile, CallbackCaller& callback) NN_NOEXCEPT
{
    if (!ExaminesTextureFileUsed(pOldResTextureFile))
    {
        return;
    }

    ForceBindTexture(pResTextureFile, callback);
}

void EditModelObj::ForceBindTexture(const nn::gfx::ResTextureFile* pResTextureFile, CallbackCaller& callback) NN_NOEXCEPT
{
    for (int modelIndex = 0, modelCount = m_EditModelDataSets.GetCount(); modelIndex < modelCount; ++modelIndex)
    {
        EditModelDataSet model = m_EditModelDataSets[modelIndex];
        nn::g3d::viewer::detail::ForceBindTexture(model.pOriginalModelObj, pResTextureFile, callback);
    }

    if (GetResource() != nullptr)
    {
        nn::g3d::viewer::detail::ForceBindTexture(GetResource(), pResTextureFile, callback);
    }
}

void EditModelObj::UnbindTexture(const nn::gfx::ResTextureFile* pResTextureFile, CallbackCaller& callback) NN_NOEXCEPT
{
    if (!ExaminesTextureFileUsed(pResTextureFile))
    {
        return;
    }

    for (int modelIndex = 0, modelCount = m_EditModelDataSets.GetCount(); modelIndex < modelCount; ++modelIndex)
    {
        EditModelDataSet model = m_EditModelDataSets[modelIndex];
        ForceUnbindTexture(model.pOriginalModelObj, pResTextureFile, callback);
    }

    if (GetResource() != nullptr)
    {
        ForceUnbindTexture(GetResource(), pResTextureFile, callback);
    }
}

void EditModelObj::UpdateTextureBindings(TextureBindingBlock* pBlock) NN_NOEXCEPT
{
    m_BoundTextureKeys.Clear();
    for (int index = 0; index < pBlock->textureKeyArrayCount; ++index)
    {
        m_BoundTextureKeys.PushBack(static_cast<ViewerKeyType>(pBlock->textureKeyArrayData[index]));
    }
}

void EditModelObj::UnbindAllTexturesWithoutCallback() NN_NOEXCEPT
{
    for (int modelIndex = 0, modelCount = m_EditModelDataSets.GetCount(); modelIndex < modelCount; ++modelIndex)
    {
        EditModelDataSet model = m_EditModelDataSets[modelIndex];
        ModelObj* pModelObj = model.pOriginalModelObj;
        for (int materialIndex = 0, materialCount = pModelObj->GetMaterialCount(); materialIndex < materialCount; ++materialIndex)
        {
            MaterialObj* pMaterialObj = pModelObj->GetMaterial(materialIndex);
            ResMaterial* pResMaterial = const_cast<ResMaterial*>(pMaterialObj->GetResource());
            pResMaterial->ClearSamplerDescriptorSlot();
            for (int samplerIndex = 0, samplerCount = pResMaterial->GetSamplerCount();
                samplerIndex < samplerCount; ++samplerIndex)
            {
                pResMaterial->SetTexture(samplerIndex, nullptr);
            }

            pMaterialObj->ClearTexture();
        }
    }
}

void EditModelObj::RebuildModelObj(nn::g3d::ResModel* pNewResModel) NN_NOEXCEPT
{
    for (int modelIndex = 0, modelCount = m_EditModelDataSets.GetCount(); modelIndex < modelCount; ++modelIndex)
    {
        EditModelDataSet* pEditData = &m_EditModelDataSets[modelIndex];

        {
            nn::g3d::ResSkeleton* pResSkeleton = pNewResModel->GetSkeleton();

            // WorkBlock 用のメモリプールをアロケート
            // スキニングされない場合は size == 0 になるので、size > 0 のときのみアロケートする
            size_t memoryPoolSize = pEditData->pEditSkeletonObj->CalculateBlockBufferSize(pResSkeleton);
            ptrdiff_t memoryPoolOffset = 0;
            if (memoryPoolSize > 0) {
                size_t memoryPoolAlignment = pEditData->pEditSkeletonObj->GetBlockBufferAlignment(pResSkeleton);

                memoryPoolOffset = m_MemoryPoolManager->AllocateMemoryPool(memoryPoolSize, memoryPoolAlignment);
                NN_G3D_VIEWER_ASSERT(memoryPoolOffset != nn::gfx::util::MemoryPoolAllocator::InvalidOffset);
            }
            pEditData->pEditSkeletonObj->SetupSkeletonInstance(pResSkeleton, m_MemoryPoolManager->GetMemoryPool(), memoryPoolOffset, memoryPoolSize);
        }

        for (int shapeIndex = 0, shapeCount = pEditData->shapeArray.GetCount(); shapeIndex < shapeCount; ++shapeIndex)
        {
            nn::g3d::ResShape* pNewResShape = pNewResModel->GetShape(shapeIndex);
            EditShapeObj* pEditShapeObj = pEditData->shapeArray.UnsafeAt(shapeIndex);

            const nn::g3d::ResSkeleton* pNewResSkeleton = pNewResModel->GetSkeleton();
            const ResBone* pNewResBone = pNewResSkeleton->GetBone(pNewResShape->GetBoneIndex());
            pEditShapeObj->SetViewDependent(pNewResBone->GetBillboardMode() != ResBone::Flag_BillboardNone);

            bool result = pEditShapeObj->ReinitBuffer(pNewResShape);
            NN_G3D_VIEWER_ASSERT(result); // 今は止める

            // WorkBlock 用のメモリプールをアロケート
            size_t memoryPoolSize = pEditShapeObj->CalculateBlockBufferSize(pNewResBone->GetBillboardMode() != ResBone::Flag_BillboardNone, pNewResShape);
            size_t memoryPoolAlignment = pEditShapeObj->GetBlockBufferAlignment();
            ptrdiff_t memoryPoolOffset = m_MemoryPoolManager->AllocateMemoryPool(memoryPoolSize, memoryPoolAlignment);
            NN_G3D_VIEWER_ASSERT(memoryPoolOffset != nn::gfx::util::MemoryPoolAllocator::InvalidOffset);

            pEditShapeObj->SetupShapeInstance(pNewResShape, m_MemoryPoolManager->GetMemoryPool(), memoryPoolOffset, memoryPoolSize);
        }

        for (int materialIndex = 0, materialCount = pEditData->materialArray.GetCount(); materialIndex < materialCount; ++materialIndex)
        {
            nn::g3d::ResMaterial* pResMaterial = pNewResModel->GetMaterial(materialIndex);
            EditMaterialObj* pEditMaterialObj = pEditData->materialArray.UnsafeAt(materialIndex);

            bool result = pEditMaterialObj->ReinitBuffer(pResMaterial);
            NN_G3D_VIEWER_ASSERT(result); // 今は止める

            // WorkBlock 用のメモリプールをアロケート
            ptrdiff_t   memoryPoolOffset = nn::gfx::util::MemoryPoolAllocator::InvalidOffset;
            size_t      memoryPoolSize = pEditMaterialObj->CalculateBlockBufferSize(pResMaterial);
            if (memoryPoolSize > 0)
            {
                size_t memoryPoolAlignment = pEditMaterialObj->GetBlockBufferAlignment(pResMaterial);
                memoryPoolOffset = m_MemoryPoolManager->AllocateMemoryPool(memoryPoolSize, memoryPoolAlignment);
                NN_G3D_VIEWER_ASSERT(memoryPoolOffset != nn::gfx::util::MemoryPoolAllocator::InvalidOffset);
            }

            pEditMaterialObj->SetupMaterialInstance(pResMaterial, m_MemoryPoolManager->GetMemoryPool(), memoryPoolOffset, memoryPoolSize);
        }

        pEditData->pOriginalModelObj->UpdateViewDependency();
        pEditData->pOriginalModelObj->ClearBoneVisible();
        pEditData->pOriginalModelObj->ClearMaterialVisible();
    }
}

void EditModelObj::SwitchWorkBlockBuffers() NN_NOEXCEPT
{
    for (int modelIndex = 0, modelCount = m_EditModelDataSets.GetCount(); modelIndex < modelCount; ++modelIndex)
    {
        EditModelDataSet* pEditData = &m_EditModelDataSets[modelIndex];

        // スケルトン
        pEditData->pEditSkeletonObj->SwitchBlockBuffer();

        // シェイプ
        for (int shapeIndex = 0, shapeCount = pEditData->shapeArray.GetCount(); shapeIndex < shapeCount; ++shapeIndex)
        {
            EditShapeObj* pEditShapeObj = pEditData->shapeArray.UnsafeAt(shapeIndex);
            pEditShapeObj->SwitchBlockBuffer();
        }

        // マテリアル
        for (int materialIndex = 0, materialCount = pEditData->materialArray.GetCount(); materialIndex < materialCount; ++materialIndex)
        {
            EditMaterialObj* pEditMaterialObj = pEditData->materialArray.UnsafeAt(materialIndex);
            pEditMaterialObj->SwitchBlockBuffer();
        }
    }
}

}}}} // namespace nn::g3d::viewer::detail


