﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3d_EditMaterialObj.h"

#include "../util/g3d_DynamicArray.h"
#include "../g3d_Allocator.h"
#include "../g3d_CallbackCaller.h"
#include "../util/g3d_BufferSwitcher.h"
#include "../util/g3d_ViewerUtility.h"

#include <nn/g3d/g3d_ModelObj.h>
#include <nn/g3d/g3d_MaterialObj.h>
#include <nn/g3d/g3d_ResMaterial.h>

using namespace nn::g3d;
using namespace nn::g3d::viewer;

namespace nn { namespace g3d { namespace viewer { namespace detail {

EditMaterialObj::EditMaterialObj(
    nn::gfx::Device* pDevice,
    Allocator* pAllocator,
    int index,
    nn::g3d::ModelObj* pOwnerModelObj,
    nn::g3d::MaterialObj* pMaterialObj) NN_NOEXCEPT
    : BlinkingObj(pDevice, pAllocator)
    , m_Index(index)
    , m_pOwnerModel(pOwnerModelObj)
    , m_pMaterialObj(pMaterialObj)
    , m_WorkBuffer(pAllocator, MaterialObj::Alignment_Buffer, AllocateType_DynamicBuffer)
    , m_BlockBufferObjectManager(UniformBlockBufferingCount, pAllocator)
    , m_RenderInfo(pAllocator, pMaterialObj)
    , m_TextureRefArray(pAllocator, NN_ALIGNOF(nn::g3d::TextureRef), nn::g3d::TextureRef())
    , m_pEditedSrcParam(nullptr)
    , m_pShaderParamEditFlagBuffer(nullptr)
{
    NN_SDK_REQUIRES_NOT_NULL(pDevice);
    NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(pMaterialObj, "%s\n", NN_G3D_VIEWER_RES_NAME(m_pOwnerModel->GetResource(), GetName()));
    nn::g3d::ResMaterial* pResMaterial = const_cast<nn::g3d::ResMaterial*>(pMaterialObj->GetResource());
    NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(pResMaterial, "%s\n", NN_G3D_VIEWER_RES_NAME(m_pOwnerModel->GetResource(), GetName()));
    m_pOriginalMaterial = pResMaterial;

    m_pOriginalBuffer = pMaterialObj->GetBufferPtr();
    MaterialObj::Builder builder(pResMaterial);
    builder.BufferingCount(pMaterialObj->GetBufferingCount());
    builder.CalculateMemorySize();
    m_OriginalBufferSize = builder.GetWorkMemorySize();
    m_OriginalBufferingCount = pMaterialObj->GetBufferingCount();

    m_pOriginalBlockMemoryPool = pMaterialObj->GetMemoryPoolPtr();
    m_OriginalBlockMemoryPoolOffset = pMaterialObj->GetMemoryPoolOffset();
    m_OriginalBlockBufferSize = pMaterialObj->CalculateBlockBufferSize(m_pDevice);
    m_pOriginalBlockBufferArray = pMaterialObj->GetMaterialBlockArray();

    m_EditedSrcParamSize = m_pMaterialObj->GetResource()->GetSrcParamSize();
    if (m_EditedSrcParamSize > 0)
    {
        m_pEditedSrcParam = m_pAllocator->Allocate(m_EditedSrcParamSize, DefaultAlignment, AllocateType_Other);
    }

    {
        int dirtyFlagCount = m_pMaterialObj->GetResource()->GetShaderParamCount();
        size_t bufferSize = FlagSet::CalcBufferSize(dirtyFlagCount, m_OriginalBufferingCount);
        if (bufferSize > 0)
        {
            m_pShaderParamEditFlagBuffer = m_pAllocator->Allocate(bufferSize, DefaultAlignment, AllocateType_Other);
        }

        // バッファーサイズが 0 の場合でも初期化可能。
        m_ShaderParamEditFlags.Initialize(dirtyFlagCount, m_OriginalBufferingCount, m_pShaderParamEditFlagBuffer, bufferSize);
        m_ShaderParamEditFlags.Clear();
    }

    // 初期値を保存
    m_MaterialVisibilityEnable = m_pOwnerModel->IsMaterialVisible(m_Index);

    {
        int textureCount = m_pMaterialObj->GetTextureCount();
        for (int textureIndex = 0; textureIndex < textureCount; ++textureIndex)
        {
            m_TextureRefArray.PushBack(m_pMaterialObj->GetTexture(textureIndex));
        }
    }
}

EditMaterialObj::~EditMaterialObj() NN_NOEXCEPT
{
    // オリジナルのマテリアルに戻す前に、RenderInfoDicも元に戻す
    m_RenderInfo.Destroy();

    m_WorkBuffer.Clear();

    if (m_pEditedSrcParam != nullptr)
    {
        m_pAllocator->Free(m_pEditedSrcParam);
        m_pEditedSrcParam = nullptr;
    }
    if (m_pShaderParamEditFlagBuffer != nullptr)
    {
        m_pAllocator->Free(m_pShaderParamEditFlagBuffer);
        m_pShaderParamEditFlagBuffer = nullptr;
    }

    for (int bufferIndex = 0, bufferCount = m_BlockBufferObjectManager.GetBufferCount(); bufferIndex < bufferCount; ++bufferIndex)
    {
        if (m_BlockBufferObjectManager.GetBufferSize(bufferIndex) > 0)
        {
            nn::gfx::Buffer* pMaterialBlockArray = m_BlockBufferObjectManager.GetBuffer<nn::gfx::Buffer>(bufferIndex);
            FinalizeBuffers(m_pDevice, pMaterialBlockArray, m_OriginalBufferingCount);
        }
    }
}

void EditMaterialObj::ResetToOriginal() NN_NOEXCEPT
{
    m_RenderInfo.Cleanup();

    if (m_pMaterialObj->IsBlockBufferValid())
    {
        // UserPtr を保持するために一時退避
        void* userPtr = m_pMaterialObj->GetUserPtr();

        if (m_OriginalBlockBufferSize > 0)
        {
            FinalizeBuffers(m_pDevice, m_pOriginalBlockBufferArray, m_OriginalBufferingCount);
        }

        {
            MaterialObj::Builder builder(m_pOriginalMaterial);
            builder.BufferingCount(m_OriginalBufferingCount);
            builder.CalculateMemorySize();
            bool success = builder.Build(m_pMaterialObj, m_pOriginalBuffer, m_OriginalBufferSize);
            NN_G3D_VIEWER_ASSERT_DETAIL(success, "Failed to build MaterialObj");
        }

        if (m_OriginalBlockBufferSize > 0)
        {
            bool success = m_pMaterialObj->SetupBlockBuffer(m_pDevice, m_pOriginalBlockMemoryPool, m_OriginalBlockMemoryPoolOffset, m_OriginalBlockBufferSize);
            NN_G3D_VIEWER_ASSERT_DETAIL(success, "Failed to setup block buffer");
        }

        // テクスチャの設定やビジビリティの変更時にはコールバック関数が呼び出されます。
        // コールバック関数にて UserPtr を用いた操作が行われる可能性があるため、このタイミングで設定を行います。
        m_pMaterialObj->SetUserPtr(userPtr);

        // 初期値に戻します。
        {
            for (int textureIndex = 0, textureCount = m_TextureRefArray.GetCount(); textureIndex < textureCount; ++textureIndex)
            {
                m_pMaterialObj->SetTexture(textureIndex, m_TextureRefArray[textureIndex]);
            }
        }

        m_pOwnerModel->SetMaterialVisible(m_Index, m_MaterialVisibilityEnable);
    }

    ReinitEditedShaderParam(m_pOriginalMaterial);
}

void EditMaterialObj::SetMaterialVisible(bool visible) NN_NOEXCEPT
{
    m_MaterialVisibilityEnable = visible;
}

void EditMaterialObj::UpdateMaterialVisible() NN_NOEXCEPT
{
    m_pOwnerModel->SetMaterialVisible(m_Index, m_MaterialVisibilityEnable);
}

void EditMaterialObj::SwitchBlockBuffer() NN_NOEXCEPT
{
    m_BlockBufferObjectManager.SwitchCurrentBuffer();
    if (m_BlockBufferObjectManager.GetCurrentBufferSize() > 0)
    {
        // 過去に使っていたバッファーを使い回すので一度終了処理する
        nn::gfx::Buffer* pOldMaterialBlockArray = m_BlockBufferObjectManager.GetCurrentBuffer<nn::gfx::Buffer>();
        FinalizeBuffers(m_pDevice, pOldMaterialBlockArray, m_OriginalBufferingCount);

        // 終了処理が済んでいるのでサイズを 0 にする
        m_BlockBufferObjectManager.ResizeCurrentBuffer(0);
    }
}

void EditMaterialObj::UpdateRenderInfo(
        const void* pRenderInfoUpdateCommandData, size_t renderInfoUpdateCommandDataSize,
        ptrdiff_t renderInfoArrayOffset) NN_NOEXCEPT
{
    m_RenderInfo.Update(pRenderInfoUpdateCommandData, renderInfoUpdateCommandDataSize,
        renderInfoArrayOffset);
}

bool EditMaterialObj::ReinitBuffer(const nn::g3d::ResMaterial* pResMaterial) NN_NOEXCEPT
{
    nn::g3d::MaterialObj::Builder builder(pResMaterial);
    builder.BufferingCount(m_OriginalBufferingCount);
    builder.CalculateMemorySize();
    size_t size = builder.GetWorkMemorySize();
    return m_WorkBuffer.Resize(size);
}

size_t EditMaterialObj::CalculateBlockBufferSize(const nn::g3d::ResMaterial* pResMaterial) const NN_NOEXCEPT
{
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetSize(pResMaterial->GetRawParamSize());
    info.SetGpuAccessFlags(nn::gfx::GpuAccess_ConstantBuffer);
    size_t alignment = nn::gfx::Buffer::GetBufferAlignment(m_pDevice, info);

    return nn::util::align_up(pResMaterial->GetRawParamSize(), alignment) * m_OriginalBufferingCount;
}

size_t EditMaterialObj::GetBlockBufferAlignment(const nn::g3d::ResMaterial* pResMaterial) const NN_NOEXCEPT
{
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetSize(pResMaterial->GetRawParamSize());
    info.SetGpuAccessFlags(nn::gfx::GpuAccess_ConstantBuffer);

    return nn::gfx::Buffer::GetBufferAlignment(m_pDevice, info);
}

void EditMaterialObj::SetupMaterialInstance(
    nn::g3d::ResMaterial* pResMaterial,
    nn::gfx::MemoryPool* pMemoryPool,
    ptrdiff_t memoryPoolOffset,
    size_t memoryPoolSize) NN_NOEXCEPT
{
    m_RenderInfo.SetUpdateMaterial(pResMaterial);

    nn::g3d::MaterialObj::Builder builder(pResMaterial);
    builder.BufferingCount(m_OriginalBufferingCount);

    {
        void* userPtr = m_pMaterialObj->GetUserPtr();

        builder.CalculateMemorySize();
        NN_G3D_VIEWER_ASSERT(builder.GetWorkMemorySize() == m_WorkBuffer.GetSize());
        bool success = builder.Build(m_pMaterialObj, m_WorkBuffer.GetWorkBufferPtr(), m_WorkBuffer.GetSize());
        NN_G3D_VIEWER_ASSERT_DETAIL(success, NN_G3D_VIEWER_RES_NAME(m_pOwnerModel->GetResource(), GetName()));

        m_pMaterialObj->SetUserPtr(userPtr);
    }

    if (memoryPoolSize > 0)
    {
        m_BlockBufferObjectManager.ResizeCurrentBuffer(sizeof(nn::gfx::Buffer) * m_OriginalBufferingCount);
        m_pMaterialObj->SetMaterialBlockArray(new (m_BlockBufferObjectManager.GetCurrentBuffer()) nn::gfx::Buffer());

        NN_G3D_VIEWER_ASSERT(memoryPoolSize == m_pMaterialObj->CalculateBlockBufferSize(m_pDevice));
        m_pMaterialObj->SetupBlockBuffer(m_pDevice, pMemoryPool, memoryPoolOffset, memoryPoolSize);
    }

    ReinitEditedShaderParam(pResMaterial);
}

void EditMaterialObj::Show() NN_NOEXCEPT
{
    m_pOwnerModel->SetMaterialVisible(m_Index, true);
}

void EditMaterialObj::Hide() NN_NOEXCEPT
{
    m_pOwnerModel->SetMaterialVisible(m_Index, false);
}

bool EditMaterialObj::IsVisibleByDefault() const NN_NOEXCEPT
{
    return m_MaterialVisibilityEnable;
}

void EditMaterialObj::ReinitEditedShaderParam(const nn::g3d::ResMaterial* pResMaterial) NN_NOEXCEPT
{
    // 破棄処理
    // 新しいマテリアルにシェーダーパラメーターが存在しない可能性があるので、nullptr を代入しておかないと多重解放してしまう。

    if (m_pEditedSrcParam)
    {
        m_pAllocator->Free(m_pEditedSrcParam);
        m_pEditedSrcParam = nullptr;
    }
    if (m_pShaderParamEditFlagBuffer)
    {
        m_pAllocator->Free(m_pShaderParamEditFlagBuffer);
        m_pShaderParamEditFlagBuffer = nullptr;
    }

    // 初期化処理
    m_EditedSrcParamSize = pResMaterial->GetSrcParamSize();
    if (m_EditedSrcParamSize > 0)
    {
        m_pEditedSrcParam = m_pAllocator->Allocate(m_EditedSrcParamSize, DefaultAlignment, AllocateType_Other);
    }

    {
        int dirtyFlagCount = pResMaterial->GetShaderParamCount();
        size_t bufferSize = FlagSet::CalcBufferSize(dirtyFlagCount, m_OriginalBufferingCount);
        if (bufferSize > 0)
        {
            m_pShaderParamEditFlagBuffer = m_pAllocator->Allocate(bufferSize, DefaultAlignment, AllocateType_Other);
        }

        // バッファーサイズが 0 の場合でも初期化可能。
        m_ShaderParamEditFlags.Initialize(dirtyFlagCount, m_OriginalBufferingCount, m_pShaderParamEditFlagBuffer, bufferSize);
        m_ShaderParamEditFlags.Clear();
    }
}

void EditMaterialObj::RestoreEditedShaderParam() NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT_EQUAL(m_pMaterialObj->GetResource()->GetSrcParamSize(), m_EditedSrcParamSize);
    m_pMaterialObj->ClearShaderParam();

    int shaderParamCount = m_pMaterialObj->GetShaderParamCount();
    for (int paramIndex = 0; paramIndex < shaderParamCount; ++paramIndex)
    {
        // 編集されたものは編集状態を反映する
        if (m_ShaderParamEditFlags.IsFlagDirty(paramIndex))
        {
            const ResShaderParam* pParam = m_pMaterialObj->GetResShaderParam(paramIndex);
            void* pShaderParam = m_pMaterialObj->EditShaderParam(paramIndex);
            void* pEditParam = AddOffset(m_pEditedSrcParam, pParam->GetSrcOffset());
            size_t paramSize = pParam->GetSize();
            memcpy(pShaderParam, pEditParam, paramSize);
        }
    }
}

void EditMaterialObj::EditTextureSrt(
    const char* paramName,
    TexSrt::Mode mode,
    ResShaderParam::Type paramType,
    const float* pValues,
    int valueCount) NN_NOEXCEPT
{
    int paramIndex = m_pMaterialObj->FindShaderParamIndex(paramName);
    if (paramIndex == -1)
    {
        return;
    }

    NN_G3D_VIEWER_ASSERT_INDEX_BOUNDS(paramIndex, m_pMaterialObj->GetShaderParamCount());

    const ResShaderParam* pResShaderParam = m_pMaterialObj->GetResShaderParam(paramIndex);
    if (pResShaderParam->GetType() != paramType)
    {
        return;
    }

    TexSrt* texSrtValue = m_pMaterialObj->EditShaderParam<TexSrt>(paramIndex);
    texSrtValue->mode = mode;
    float* pEditValues = &texSrtValue->sx;
    for (int idxValue = 0; idxValue < valueCount; ++idxValue)
    {
        pEditValues[idxValue] = pValues[idxValue];
    }

    TexSrt* pEditValueBackup = AddOffset<TexSrt>(m_pEditedSrcParam, pResShaderParam->GetSrcOffset());
    memcpy(pEditValueBackup, texSrtValue, pResShaderParam->GetSize());

    m_ShaderParamEditFlags.Set(paramIndex);
}

bool EditMaterialObj::ExaminesTextureFileUsed(const nn::gfx::ResTextureFile* pResTextureFile) const NN_NOEXCEPT
{
    for (int idxTex = 0, texCount = pResTextureFile->GetTextureDic()->GetCount(); idxTex < texCount; ++idxTex)
    {
        const nn::gfx::ResTexture* pResTexture = pResTextureFile->GetResTexture(idxTex);
        for (int idxMatTex = 0, matTexCount = m_pMaterialObj->GetTextureCount(); idxMatTex < matTexCount; ++idxMatTex)
        {
            nn::g3d::TextureRef texRef = m_pMaterialObj->GetTexture(idxMatTex);
            if (texRef.GetTextureView() == pResTexture->GetTextureView())
            {
                return true;
            }
        }
    }

    return false;
}

}}}} // namespace nn::g3d::viewer::detail


