﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "g3d_ViewerDetailDefine.h"
#include "g3d_DeviceDependentObj.h"
#include "util\g3d_DynamicArray.h"
#include "util\g3d_SynchronizedDynamicPtrArray.h"

#include <nn/g3d/viewer/detail/g3d_ViewerPacketDefine.h>
#include <nn/g3d/viewer/g3d_ViewerDefine.h>
#include <nn/g3d/viewer/g3d_ViewerCallback.h>
#include <nn/nn_Macro.h>

namespace nn { namespace g3d {

class ModelObj;
class ModelAnimObj;

class ResModel;
class ResShaderArchive;
class ResFile;
class ResShaderFile;

namespace viewer {

class ViewerServer;

namespace detail {

class Allocator;
class EditModelObj;
class EditShaderArchive;
class EditAnimObj;
class EditSceneAnimObj;
class CallbackCaller;
class EditSocketBase;
class ResourceDestructionScheduler;

struct ResourceInfo
{
    FileDataKind kind;
    ViewerKeyType resFileKey;

    // モデル、アニメーション用
    nn::g3d::ResFile* pResFile;

    // シェーダアーカイブ用
    nn::g3d::ResShaderFile* pResShaderFile;
    nn::g3d::ResShaderArchive* pResShaderArchive;

    // テクスチャ用
    nn::gfx::ResTextureFile* pResTextureFile;

    // EditModelObj用(メモリープールを持っている)
    EditModelObj* pEditModelObj;

    ResourceInfo() NN_NOEXCEPT
        : kind(FILEDATA_KIND_ERROR)
        , resFileKey(0)
        , pResFile(nullptr)
        , pResShaderFile(nullptr)
        , pResShaderArchive(nullptr)
        , pResTextureFile(nullptr)
        , pEditModelObj(nullptr)
    {
    }

    ResourceInfo(nn::g3d::ResFile* pResFile, FileDataKind kind) NN_NOEXCEPT
        : kind(kind)
        , resFileKey(0)
        , pResFile(pResFile)
        , pResShaderFile(nullptr)
        , pResShaderArchive(nullptr)
        , pResTextureFile(nullptr)
        , pEditModelObj(nullptr)
    {
    }

    ResourceInfo(nn::g3d::ResFile* pResFile, FileDataKind kind, ViewerKeyType key) NN_NOEXCEPT
        : kind(kind)
        , resFileKey(key)
        , pResFile(pResFile)
        , pResShaderFile(nullptr)
        , pResShaderArchive(nullptr)
        , pResTextureFile(nullptr)
        , pEditModelObj(nullptr)
    {
    }

    ResourceInfo(nn::gfx::ResTextureFile* pResTextureFile, ViewerKeyType key) NN_NOEXCEPT
        : kind(FILEDATA_TEXTURE)
        , resFileKey(key)
        , pResFile(nullptr)
        , pResShaderFile(nullptr)
        , pResShaderArchive(nullptr)
        , pResTextureFile(pResTextureFile)
        , pEditModelObj(nullptr)
    {
    }

    ResourceInfo(nn::g3d::ResShaderFile* pResShaderFile, nn::g3d::ResShaderArchive* pResShaderArchive) NN_NOEXCEPT
        : kind(FILEDATA_SHADER_ARCHIVE)
        , resFileKey(0)
        , pResFile(nullptr)
        , pResShaderFile(pResShaderFile)
        , pResShaderArchive(pResShaderArchive)
        , pResTextureFile(nullptr)
        , pEditModelObj(nullptr)
    {
    }

    explicit ResourceInfo(EditModelObj* pEditModelObj) NN_NOEXCEPT
        : kind(FILEDATA_EDIT_MODEL_OBJ)
        , resFileKey(0)
        , pResFile(nullptr)
        , pResShaderFile(nullptr)
        , pResShaderArchive(nullptr)
        , pResTextureFile(nullptr)
        , pEditModelObj(pEditModelObj)
    {
    }

};


class ResourceManager : public DeviceDependentObj
{
    NN_DISALLOW_COPY(ResourceManager);

public:
    ResourceManager(nn::gfx::Device* pDevice, Allocator* pAllocator, CallbackCaller* pCallbackCaller) NN_NOEXCEPT;
    virtual ~ResourceManager() NN_NOEXCEPT;

    void Reset() NN_NOEXCEPT;

    //! @brief 管理しているリソース数を取得します。
    int GetResourceCount() const NN_NOEXCEPT;
    int GetEditAnimCount() const NN_NOEXCEPT;

    //! @brief すべてのリソースの遅延破棄を開始します。
    void ScheduleDestructionAll() NN_NOEXCEPT;
    void ScheduleDestroyUnusedResFiles() NN_NOEXCEPT;
    void ScheduleDestroyAnimResFile(ViewerKeyType resFileKey) NN_NOEXCEPT;

    Iter<EditModelObj*> ScheduleDestructEditModelObj(EditModelObj* pDeleteEditModel) NN_NOEXCEPT;

    //! @brief 遅延破棄可能なものを破棄します。破棄不可能なものは破棄状態を更新します。
    void TryDestroyUnreferencedResources() NN_NOEXCEPT;

    RuntimeErrorCode CreateEditModelObj(
        EditModelObj** pOut, nn::g3d::ResFile* pOriginalResFile) NN_NOEXCEPT;

    RuntimeErrorCode CreateAttachedEditModelObj(
        nn::g3d::ModelObj** pAttachTargetModelObjs, int modelObjCount, ResFile* pFirstLoadResFile) NN_NOEXCEPT;

    EditShaderArchive* CreateAttachEditShaderArchive(nn::g3d::ResShaderArchive* pTargetResShaderArchive) NN_NOEXCEPT;

    EditAnimObj* AddEditAnimObj(nn::g3d::ResFile* pResFile, FileDataKind kind) NN_NOEXCEPT;
    EditSceneAnimObj* AddEditSceneAnimObj(nn::g3d::ResFile* targetResFile) NN_NOEXCEPT;
    bool AddEditShaderArchive(EditShaderArchive* pEditShaderArchive) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(pEditShaderArchive);
        return m_EditShaderArchiveArray.PushBack(pEditShaderArchive);
    }

    RuntimeErrorCode AttachModelObjs(EditModelObj* pTarget, nn::g3d::ResFile* pAttachModelResFile, nn::g3d::ModelObj** pModelObjs, int modelObjCount) NN_NOEXCEPT;

    EditModelObj* FindEditModelObj(ViewerKeyType key) NN_NOEXCEPT;
    EditModelObj* FindEditModelObj(const nn::g3d::ModelObj* pTargetModelObj) NN_NOEXCEPT;
    EditModelObj*  FindUnattachedEditModelObj(ViewerKeyType resModelKey) NN_NOEXCEPT;
    EditShaderArchive* FindEditShaderArchive(const nn::g3d::ResShaderArchive* pResShaderArchive) NN_NOEXCEPT;
    const EditShaderArchive* FindEditShaderArchive(ViewerKeyType key) const NN_NOEXCEPT;
    EditShaderArchive* FindEditShaderArchive(ViewerKeyType key) NN_NOEXCEPT;
    EditAnimObj* FindEditAnimObj(ViewerKeyType resFileKey) NN_NOEXCEPT;
    EditSceneAnimObj* FindEditSceneAnimObj(ViewerKeyType resFileKey) NN_NOEXCEPT;
    const EditAnimObj* FindEditAnimObj(ViewerKeyType resFileKey) const NN_NOEXCEPT;
    int FindModelAnimBoundCount(const ModelObj* modelObj) const NN_NOEXCEPT;
    int FindActiveEditAnimIndex(const nn::g3d::ModelObj* pModelObj, int boundAnimIndex) const NN_NOEXCEPT;
    EditAnimObj* FindBoundEditAnimObj(ViewerKeyType modelKey, ViewerKeyType boundAnimationResFileKey) NN_NOEXCEPT;

    bool HasModelObj(const nn::g3d::ModelObj* pModelObj) const NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(pModelObj);
        return FindSameModelObj(pModelObj) != nullptr;
    }

    bool HasShaderArchive(const nn::g3d::ResShaderArchive* pResShaderArchive) const NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(pResShaderArchive);
        return FindSameResShaderArchive(pResShaderArchive) != nullptr;
    }

    void BindSceneAnimations(const BindAnimArg& arg) NN_NOEXCEPT;
    void UnbindSceneAnimations(const BindAnimArg& arg) NN_NOEXCEPT;

    void CalculateAnimations() NN_NOEXCEPT;
    void CalculateSkeletalAnimations(const nn::g3d::ModelObj* pModelObj) NN_NOEXCEPT;

    void SendModifiedShaderPrograms(EditSocketBase* pSocket) NN_NOEXCEPT;

    void DeleteEditAnimObjBindingForAllModels(EditAnimObj* pTarget) NN_NOEXCEPT;
    void DeleteEditShaderArchive(ViewerKeyType key) NN_NOEXCEPT;
    void DeleteEditAnimObj(ViewerKeyType key) NN_NOEXCEPT;
    void DeleteEditSceneAnimObj(ViewerKeyType key) NN_NOEXCEPT;

    RuntimeErrorCode ReloadAnimResource(ViewerKeyType resFileKey, nn::g3d::ResFile* pResFile, FileDataKind kind) NN_NOEXCEPT;
    RuntimeErrorCode ReloadTextureResource(ViewerKeyType resFileKey, nn::gfx::ResTextureFile* pResTextureFile) NN_NOEXCEPT;

    RuntimeErrorCode SetRetargetingHostModel(ViewerKeyType animResFileKey, ViewerKeyType retargetHostModelObjKey) NN_NOEXCEPT;
    void UnsetRetargetingHostModel(ViewerKeyType animResFileKey) NN_NOEXCEPT;

    void SetPlayMotionMirroringEnabled(ViewerKeyType animResFileKey, bool isEnabled) NN_NOEXCEPT;

    nn::g3d::ResFile* LoadResFileWithoutCopyAndRegister(void** pFileBuffer, FileDataKind kind) NN_NOEXCEPT;
    nn::g3d::ResFile* LoadAnimResFileWithoutCopyAndRegister(void** pFileBuffer, FileDataKind kind) NN_NOEXCEPT;
    nn::g3d::ResFile* LoadAnimResFileWithoutCopy(void** pFileBuffer) NN_NOEXCEPT;
    nn::g3d::ResShaderArchive* LoadResShaderArchiveWithoutCopyAndRegister(void** pFileBuffer) NN_NOEXCEPT;
    nn::gfx::ResTextureFile* LoadResTextureFileWithoutCopyAndRegister(void** pFileBuffer) NN_NOEXCEPT;
    nn::gfx::ResTextureFile* LoadResTextureFileWithoutCopy(void** pFileBuffer) NN_NOEXCEPT;
    void UnloadResTextureFile(ViewerKeyType resFileKey) NN_NOEXCEPT;

    RuntimeErrorCode ReloadEditSceneAnimObj(EditSceneAnimObj* pEditSceneAnimObj, nn::g3d::ResFile* pResFile) NN_NOEXCEPT;

    void ForceBindExistingTextures(EditModelObj* pEditModelObj) NN_NOEXCEPT;
    void BindExistingTextures(nn::g3d::ResFile* pResFile) NN_NOEXCEPT;
    void ForceBindTextureForAllObjs(const nn::gfx::ResTextureFile* pResTextureFile) NN_NOEXCEPT;
    void ForceBindExistingTextures(EditAnimObj* pEditAnimObj) NN_NOEXCEPT;

    void UpdateBlink() NN_NOEXCEPT;
    void StartBlinking(const TargetSelectedArg& arg) NN_NOEXCEPT;

    float GetEditAnimFrameCount(int animIndex) const NN_NOEXCEPT;
    const char* GetEditAnimName(int animIndex) const NN_NOEXCEPT;
    ViewerAnimKind GetViewerAnimKind(int animIndex) const NN_NOEXCEPT;

    bool IsAnimationPlaying() const NN_NOEXCEPT;
    void SetAnimPlaying(bool enable) NN_NOEXCEPT;
    bool IsLoopAnim() const NN_NOEXCEPT;
    float GetFrame() const NN_NOEXCEPT;
    float GetFrameStep() const NN_NOEXCEPT;
    void SetFrameStep(float frameStep) NN_NOEXCEPT;
    float GetPreviewFrameStepRate() const NN_NOEXCEPT;
    void SetPreviewFrameStepRate(float rate) NN_NOEXCEPT;
    float GetFrameCount() const NN_NOEXCEPT;
    void SetFrame(float frame) NN_NOEXCEPT;
    void SetStartFrame(float startFrame) NN_NOEXCEPT;
    void SetFrameCount(float frameCount) NN_NOEXCEPT;
    void SetPlayPolicy(EditPlayPolicyKind kind) NN_NOEXCEPT;

    void ClearCalculateFlagModelAnims() NN_NOEXCEPT;

    void EditMaterials(const EditMaterialArg& arg) NN_NOEXCEPT;
    void EditBones(const EditBoneArg& arg) NN_NOEXCEPT;
    void EditAnimCurve(const EditAnimCurveArg& arg) NN_NOEXCEPT;
    void EditSceneAnimCurve(const EditAnimCurveArg& arg) NN_NOEXCEPT;
    RuntimeErrorCode UpdateTextureBindings(TextureBindingBlock* pBlock) NN_NOEXCEPT;

    void PrintAllResFiles() NN_NOEXCEPT;
    void PrintAllEditModelObjs() NN_NOEXCEPT;
    void PrintAllEditAnimObjs() NN_NOEXCEPT;
    void PrintEditAnimObjs(ViewerKeyType resFileKey, ViewerKeyType boundModelKey) NN_NOEXCEPT;

    void BindAnimations(const BindAnimArg& arg) NN_NOEXCEPT;
    void UnbindAnimations(const BindAnimArg& arg) NN_NOEXCEPT;


    void UpdateEditShaderArchive(ViewerKeyType key, int shadingModelIndices[], int indexCount) NN_NOEXCEPT;

private:
    bool AddEditModelObj(EditModelObj* pEditModelObj) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(pEditModelObj);
        return m_EditModelArray.PushBack(pEditModelObj);
    }

    Iter<EditShaderArchive*> DeleteEditShaderArchive(EditShaderArchive* pTarget) NN_NOEXCEPT;
    Iter<EditAnimObj*> DeleteEditAnimObj(EditAnimObj* pDeleteTargetEditAnim) NN_NOEXCEPT;
    Iter<EditSceneAnimObj*> DeleteEditSceneAnimObj(EditSceneAnimObj* pDeleteTargetEditAnim) NN_NOEXCEPT;


    EditModelObj* FindSameModelObj(const ModelObj* modelObj) const NN_NOEXCEPT;
    EditShaderArchive* FindSameResShaderArchive(const nn::g3d::ResShaderArchive* resShaderArchive) const NN_NOEXCEPT;
    int FindResFileInfoIndex(ViewerKeyType resFileKey) NN_NOEXCEPT;

    void ForceBindExistingTexturesForAllResources() NN_NOEXCEPT;
    template<typename TTarget>
    void ForceBindExistingTexturesImpl(TTarget* pTarget) NN_NOEXCEPT;
    void UnbindTexturesFromAllObj(ViewerKeyType resFileKey) NN_NOEXCEPT;


    bool IsBound(EditModelObj* pEditModelObj, ViewerKeyType animResFileKey) const NN_NOEXCEPT;

    void UnbindAnimFromEditModelObj(EditAnimObj* pEditAnimObj, EditModelObj* pTargetEditModelObj) NN_NOEXCEPT;
    void UnbindAnimation(EditModelObj* pTargetEditModelObj, ViewerKeyType animResFileKey) NN_NOEXCEPT;
    void ForceUnbindAnimations(EditModelObj* pEditModelObj) NN_NOEXCEPT;


    bool IsShaderArchiveUsed(const ResShaderArchive* pShaderArchive) const NN_NOEXCEPT;
    bool ExaminesTextureFileUsed(const nn::gfx::ResTextureFile* pResTextureFile) const NN_NOEXCEPT;
    bool IsResFileUsedForModel(ResFile* pResFile) NN_NOEXCEPT;

    nn::g3d::ResFile* SetupBufferAsResFile(nn::gfx::Device* pDevice, void* buffer) NN_NOEXCEPT;

private:
    class AnimResourceManager;

    SynchronizedDynamicPtrArray<EditModelObj> m_EditModelArray;
    SynchronizedDynamicPtrArray<EditShaderArchive> m_EditShaderArchiveArray;

    // EditCommandManagerが確保したリソースを管理するための配列
    DynamicArray<ResourceInfo> m_ResFileInfoArray;

    CallbackCaller* m_pViewerCallback;
    AnimResourceManager* m_pAnimResourceManager;
    ResourceDestructionScheduler* m_pDestructionScheduler;
};

}}}} // namespace nn::g3d::edit::detail


