﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3d_ResourceDestructionScheduler.h"

#include "g3d_ViewerKeyManager.h"
#include "model\g3d_EditModelObj.h"
#include "util\g3d_ViewerUtility.h"

#include <nn/g3d/g3d_ResShader.h>

using namespace nn::g3d::viewer::detail;

ResourceDestructionScheduler::ResourceType ResourceDestructionScheduler::ConvertToResourceType(const ResourceInfo& resourceInfo) NN_NOEXCEPT
{
    switch (resourceInfo.kind)
    {
    case FILEDATA_MODEL:
    case FILEDATA_SCENE_ANIM:
    case FILEDATA_SHADER_PARAM_ANIM:
    case FILEDATA_MAT_COLOR_ANIM:
    case FILEDATA_TEXTURE_SRT_ANIM:
    case FILEDATA_TEXTURE_PATTERN_ANIM:
    case FILEDATA_SKELETAL_ANIM:
    case FILEDATA_BONE_VISIBILITY_ANIM:
    case FILEDATA_MAT_VISIBILITY_ANIM:
    case FILEDATA_SHAPE_ANIM:
    case FILEDATA_MATERIAL_ANIM:
        {
            return ResourceType_ResFile;
        }
        break;
    case FILEDATA_TEXTURE:
        {
            return ResourceType_ResTextureFile;
        }
        break;
    case FILEDATA_SHADER_ARCHIVE:
        {
            return ResourceType_ResShaderFile;
        }
        break;
    case FILEDATA_EDIT_MODEL_OBJ:
        {
            return ResourceType_EditModelObj;
        }
        break;
    case FILEDATA_KIND_ERROR:
        {
            return ResourceType_Unknown;
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void ResourceDestructionScheduler::Register(ResourceInfo resourceInfo) NN_NOEXCEPT
{
    ResourceType type = ConvertToResourceType(resourceInfo);

    // 各種リソースのキー登録解除
    switch (type)
    {
    case ResourceType_ResFile:
        {
            UnregisterKey(resourceInfo.pResFile);
        }
        break;
    case ResourceType_ResTextureFile:
        {
            UnregisterKey(resourceInfo.pResTextureFile);
        }
        break;
    case ResourceType_ResShaderFile:
        {
            UnregisterKey(resourceInfo.pResShaderFile);
        }
        break;
    case ResourceType_EditModelObj:
        {
            bool success;
            EditModelObj* pEditModelObj = resourceInfo.pEditModelObj;
            for (int modelIndex = 0, modelCount = pEditModelObj->GetAttachedModelObjCount(); modelIndex < modelCount; ++modelIndex)
            {
                ViewerKeyType modelObjKey = pEditModelObj->GetModelObjKey(modelIndex);
                success = ViewerKeyManager::GetInstance().Unregister(modelObjKey);
                NN_G3D_VIEWER_ASSERT_DETAIL(success, "Tried to unregister the key which has not been registered");
            }

            ViewerKeyType resModelKey = pEditModelObj->GetResModelKey();
            NN_G3D_VIEWER_ASSERT_VALID_KEY(resModelKey);
            success = ViewerKeyManager::GetInstance().Unregister(resModelKey);
            NN_G3D_VIEWER_ASSERT_DETAIL(success, "Tried to unregister the key %d which has not been registered", resModelKey);

            if (pEditModelObj->IsOriginalModelFrom3dEditor())
            {
                nn::g3d::ResFile* pOriginalResFile = pEditModelObj->GetOriginalResFileManagedByUser();
                NN_G3D_VIEWER_ASSERT_NOT_NULL(pOriginalResFile);

                success = ViewerKeyManager::GetInstance().Unregister(pOriginalResFile);
                NN_G3D_VIEWER_ASSERT_DETAIL(success, "Tried to unregister the key which has not been registered");

                // 3DEditor からロードされたモデルであれば、コールバックでアプリケーションに解放を要求する
                ModelFileUnloadedArg arg;
                arg.pResFile = pOriginalResFile;
                m_pCallbackCaller->Call(CallbackType_ModelFileUnloaded, &arg);
            }
        }
        break;
    case ResourceType_Unknown:
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    m_DestructionScheduledResources.PushBack(DestructionInfo(resourceInfo, type));
}

void ResourceDestructionScheduler::UpdateSchedule() NN_NOEXCEPT
{
    for (Iter<DestructionInfo> iter = m_DestructionScheduledResources.Begin(),
        end = m_DestructionScheduledResources.End(); iter != end; ++iter)
    {
        DestructionInfo* pInfo = &*iter;
        if (!pInfo->IsDestructionScheduled())
        {
            continue;
        }

        if (pInfo->IsDestructible())
        {
            switch (pInfo->type)
            {
            case ResourceType_ResFile:
                {
                    NN_G3D_VIEWER_ASSERT_NOT_NULL(pInfo->resourceInfo.pResFile);
                    DeleteResFile(pInfo->resourceInfo.pResFile);
                    iter = m_DestructionScheduledResources.Erase(iter);
                }
                break;
            case ResourceType_ResTextureFile:
                {
                    NN_G3D_VIEWER_ASSERT_NOT_NULL(pInfo->resourceInfo.pResTextureFile);
                    DeleteResTextureFile(pInfo->resourceInfo.pResTextureFile);
                    iter = m_DestructionScheduledResources.Erase(iter);
                }
                break;
            case ResourceType_ResShaderFile:
                {
                    NN_G3D_VIEWER_ASSERT_NOT_NULL(pInfo->resourceInfo.pResShaderFile);
                    DeleteResShaderFile(pInfo->resourceInfo.pResShaderFile);
                    iter = m_DestructionScheduledResources.Erase(iter);
                }
                break;
            case ResourceType_EditModelObj:
                {
                    NN_G3D_VIEWER_ASSERT_NOT_NULL(pInfo->resourceInfo.pEditModelObj);
                    DeleteEditModelObj(pInfo->resourceInfo.pEditModelObj);
                    iter = m_DestructionScheduledResources.Erase(iter);
                }
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
        else
        {
            pInfo->UpdateDestructionSchedule();
        }
    }
}

void ResourceDestructionScheduler::DeleteResFile(nn::g3d::ResFile* pResFile) NN_NOEXCEPT
{
    NN_G3D_VIEWER_DEBUG_PRINT("Free ResFile 0x%p\n", pResFile);

    pResFile->Cleanup(m_pDevice);
    pResFile->~ResFile();
    m_pAllocator->Free(pResFile);
}

void ResourceDestructionScheduler::DeleteResTextureFile(nn::gfx::ResTextureFile* pResTextureFile) NN_NOEXCEPT
{
    NN_G3D_VIEWER_DEBUG_PRINT("Free ResTextureFile 0x%p\n", pResTextureFile);

    if (IsTextureFileInitialized(pResTextureFile))
    {
        // テクスチャのアンロードコールバック内で Finalize が行われるため、g3d::viewer が管理しているリソースから
        // 参照されなくなってから数フレーム遅延させてコールバックを呼ぶ必要がある(コマンドバッファが参照しているため)
        nn::g3d::viewer::TextureFileUnloadedArg inArg;
        inArg.pResTextureFile = pResTextureFile;
        m_pCallbackCaller->Call(nn::g3d::viewer::CallbackType_TextureFileUnloaded, &inArg);
    }

    pResTextureFile->~ResTextureFile();
    m_pAllocator->Free(pResTextureFile);
}

void ResourceDestructionScheduler::DeleteResShaderFile(nn::g3d::ResShaderFile* pResShaderFile) NN_NOEXCEPT
{
    nn::g3d::ResShaderArchive* pResShaderArchive = pResShaderFile->GetResShaderArchive();
    NN_G3D_VIEWER_DEBUG_PRINT(
        "Free ResShaderFile 0x%p, ResShaderArhive 0x%p\n",
        pResShaderFile, pResShaderArchive);

    pResShaderArchive->Cleanup(m_pDevice);

    pResShaderFile->~ResShaderFile();
    m_pAllocator->Free(pResShaderFile);
}

void ResourceDestructionScheduler::DeleteEditModelObj(EditModelObj* pEditModelObj) NN_NOEXCEPT
{
    NN_G3D_VIEWER_DEBUG_PRINT("Free EditModelObj 0x%p\n", pEditModelObj);

    pEditModelObj->~EditModelObj();
    m_pAllocator->Free(pEditModelObj);
}
