﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/g3d/viewer/g3d_ViewerDefine.h>

#include "g3d_ViewerDetailDefine.h"
#include <nn/fs/fs_File.h>

namespace nn { namespace g3d { namespace viewer {
namespace detail {

class HostFileDeviceBase
{
public:
    enum OpenFlag
    {
        OpenFlag_ReadOnly,
    };

public:
    HostFileDeviceBase() NN_NOEXCEPT
        : m_IsFileOpened(false)
    {
    }
    ~HostFileDeviceBase() NN_NOEXCEPT
    {
    }

    ViewerResult Initialize() NN_NOEXCEPT
    {
        return InitializeInternal();
    }

    bool Open(const char* fileName, OpenFlag flag) NN_NOEXCEPT
    {
        m_IsFileOpened = OpenInternal(fileName, flag);
        return IsFileOpened();
    }

    void Close() NN_NOEXCEPT
    {
        if (!IsFileOpened())
        {
            return;
        }

        CloseInternal();
        m_IsFileOpened = false;
    }

    bool ReadASync(void* buffer, size_t size) NN_NOEXCEPT
    {
        if (!IsFileOpened())
        {
            return false;
        }

        return ReadASyncInternal(buffer, size);
    }

    bool IsReading() const NN_NOEXCEPT
    {
        return IsFileOpened() && IsReadingInternal();
    }

    bool IsFileOpened() const NN_NOEXCEPT
    {
        return m_IsFileOpened;
    }

protected:
    virtual ViewerResult InitializeInternal() = 0;
    virtual bool OpenInternal(const char* fileName, OpenFlag flag) = 0;
    virtual void CloseInternal() = 0;
    virtual bool ReadASyncInternal(void* buffer, size_t size) = 0;
    virtual bool IsReadingInternal() const = 0;

private:
    bool m_IsFileOpened;
};

class HostFileDeviceGeneric : public HostFileDeviceBase
{
    NN_DISALLOW_COPY(HostFileDeviceGeneric);
public:
    HostFileDeviceGeneric() NN_NOEXCEPT;
    ~HostFileDeviceGeneric() NN_NOEXCEPT;

private:
    virtual ViewerResult InitializeInternal() NN_NOEXCEPT;
    virtual bool OpenInternal(const char* fileName, OpenFlag flag) NN_NOEXCEPT;
    virtual void CloseInternal() NN_NOEXCEPT;
    virtual bool ReadASyncInternal(void* buffer, size_t size) NN_NOEXCEPT;
    virtual bool IsReadingInternal() const NN_NOEXCEPT;

private:
    // windows.h をヘッダーにインクルードしないための対処です。
    struct Impl;
    Impl* m_Impl;
    uint8_t m_FileIoMemberBuffer[sizeof(uint32_t) * 16]; // バッファは多めにとっています。

    char*           m_pWriteBuffer;
    char*           m_pReadBuffer;

    nn::fs::FileHandle m_FileHandle;
    size_t m_ReadingSize;
};

}}}} // namespace nn::g3d::viewer::detail


