﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/g3d/viewer/g3d_ViewerDefine.h>

#include "g3d_ViewerDetailDefine.h"
#include "g3d_Allocator.h"
#include <nn/g3d/viewer/detail/g3d_ViewerPacketDefine.h>
#include <nn/htcs/htcs_Types.h>
#include <nn/os/os_Mutex.h>

namespace nn { namespace g3d { namespace viewer {

class EditServer;

namespace detail {
    class EditSocketBase
    {
        NN_DISALLOW_COPY(EditSocketBase);
    public:
        struct SetupArg
        {
            int portNumber;
            const char* channelName; // チャンネル名を指定します。

            SetupArg() NN_NOEXCEPT
            {
                portNumber = EDIT_DEFAULT_PORT_NUMBER; // 未使用ポート番号を利用します。
                channelName = EDIT_CHANNEL_NAME;
            }
        };

    public:
        virtual ~EditSocketBase() NN_NOEXCEPT;

        ViewerResult Initialize(const SetupArg& arg) NN_NOEXCEPT
        {
            m_pWriteBuffer = nullptr;
            m_pReadBuffer = nullptr;
            m_WriteBufferSize = 0;
            m_ReadBufferSize = 0;

            m_WriteSize = 0;
            m_WritingSize = 0;
            m_TotalWritingSize = 0;
            m_IsWriteEnd = true;

            m_ReadSize = 0;
            m_ReadingSize = 0;
            m_TotalReadingSize = 0;
            m_IsReadEnd = true;
            return InitializeInternal(arg);
        }

        void RequestOpen() NN_NOEXCEPT
        {
            m_IsOpenRequested = true;
        }

        bool OpenSocket() NN_NOEXCEPT
        {
            m_IsOpened = OpenInternal();
            return m_IsOpened;
        }

        //! @brief ソケットがオープンされているかどうかを取得します。
        bool IsOpened() const NN_NOEXCEPT
        {
            return m_IsOpened;
        }

        //! @brief ソケットオープンの要求がされているかどうかを取得します。
        bool IsOpenRequested() const NN_NOEXCEPT
        {
            return m_IsOpenRequested;
        }

        void Close() NN_NOEXCEPT
        {
            CloseInternal();
            ResetReadFlag();
            ResetWriteFlag();
            m_IsOpenRequested = false;
            m_IsOpened = false;
        }

        bool WriteASync(void* buf, size_t size) NN_NOEXCEPT;
        bool ReadASync(void* buf, size_t size) NN_NOEXCEPT;

        bool WriteSync(void* buf, size_t size) NN_NOEXCEPT;

        bool IsWriting() const NN_NOEXCEPT
        {
            return m_WriteBufferSize > m_TotalWritingSize;
        }

        bool IsReading() const NN_NOEXCEPT
        {
            return m_ReadBufferSize > m_TotalReadingSize;
        }

        void ResetReadFlag() NN_NOEXCEPT
        {
            m_TotalReadingSize = m_ReadSize = m_ReadingSize = 0;
            m_IsReadEnd = true;
            m_ReadBufferSize = 0;
            m_pReadBuffer = nullptr;
        }

        void ResetWriteFlag() NN_NOEXCEPT
        {
            m_TotalWritingSize = m_WriteSize = m_WritingSize = 0;
            m_IsWriteEnd = true;
            m_WriteBufferSize = 0;
            m_pWriteBuffer = nullptr;
        }

        void Poll() NN_NOEXCEPT
        {
            if (!IsConnected() && m_IsOpenRequested)
            {
                OpenSocket();
            }

            PollInternal();
        }

        void SetWriteLogEnabled() NN_NOEXCEPT
        {
            m_IsWriteLogEnabled = true;
        }

        void SetWriteLogDisabled() NN_NOEXCEPT
        {
            m_IsWriteLogEnabled = false;
        }

        virtual bool IsConnected() const = 0;

    protected:
        EditSocketBase() NN_NOEXCEPT;

        virtual ViewerResult InitializeInternal(const SetupArg& arg) = 0;

        virtual bool OpenInternal() = 0;
        virtual void CloseInternal() = 0;

        virtual bool WriteASyncInternal(void* buf, size_t size) = 0;
        virtual bool ReadASyncInternal(void* buf, size_t size) = 0;

        virtual void ReadASyncForPoll() = 0;
        virtual void WriteASyncForPoll() = 0;

        virtual void PollInternal() = 0;

        volatile size_t    m_WriteBufferSize;
        volatile size_t    m_ReadBufferSize;

        volatile size_t    m_WriteSize;
        volatile size_t    m_WritingSize;
        volatile size_t    m_TotalWritingSize;

        volatile size_t    m_ReadSize;
        volatile size_t    m_ReadingSize;
        volatile size_t    m_TotalReadingSize;

        volatile bool   m_IsWriteEnd;
        volatile bool   m_IsReadEnd;
        volatile bool m_IsOpenRequested;
        volatile bool m_IsOpened;
        volatile bool m_IsWriteLogEnabled;

        uint8_t* m_pWriteBuffer;
        uint8_t* m_pReadBuffer;
    };

    class EditSocketGeneric : public EditSocketBase
    {
        NN_DISALLOW_COPY(EditSocketGeneric) NN_NOEXCEPT;
    public:
        EditSocketGeneric() NN_NOEXCEPT;
        virtual bool IsConnected() const NN_NOEXCEPT;


    private:
        virtual ViewerResult InitializeInternal(const SetupArg& arg) NN_NOEXCEPT;

        virtual bool OpenInternal() NN_NOEXCEPT;
        virtual void CloseInternal() NN_NOEXCEPT;

        virtual bool WriteASyncInternal(void* buf, size_t size) NN_NOEXCEPT;
        virtual bool ReadASyncInternal(void* buf, size_t size) NN_NOEXCEPT;
        virtual void WriteASyncForPoll() NN_NOEXCEPT;

        virtual void PollInternal() NN_NOEXCEPT;

        virtual void ReadASyncForPoll() NN_NOEXCEPT;

        void CloseSocket() NN_NOEXCEPT;
        void CloseClientSocket() NN_NOEXCEPT;

    private:
        struct Impl
        {
            int socket;
            int clientSocket;
            int portNumber;
        };
        Impl m_Impl;

        static const int CHANNEL_NAME_LENGTH = 15;
        char        m_ChannelName[CHANNEL_NAME_LENGTH + 1];
        nn::os::Mutex m_WriteMutex;
        nn::os::Mutex m_ReadMutex;
    };

}}}} // namespace nn::g3d::viewer::detail
