﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3d_CommandMonitor.h"



#include "g3d_Allocator.h"
#include "util/g3d_ViewerUtility.h"

using namespace nn::g3d::viewer::detail;

namespace
{
bool IsAbnormalPacket(const PacketHeader* pPacketHeader) NN_NOEXCEPT
{
    if (pPacketHeader->magic != NN_G3D_EDIT_MAGIC ||
        pPacketHeader->verWord != NN_G3D_EDIT_VERSION)
    {
        NN_G3D_WARNING(false, "Version check failed (packet:'%d.%d.%d.%d' lib:'%d.%d.%d.%d').\n",
            pPacketHeader->version[0],
            pPacketHeader->version[1],
            pPacketHeader->version[2],
            pPacketHeader->version[3],
            NN_G3D_VERSION_EDITMAJOR,
            NN_G3D_VERSION_EDITMINOR,
            NN_G3D_VERSION_EDITMICRO,
            NN_G3D_VERSION_EDITBUGFIX);
        NN_G3D_VIEWER_LOG(
            "0x%08x != 0x%08x || 0x%08x != 0x%08x\n",
            pPacketHeader->magic, NN_G3D_EDIT_MAGIC,
            pPacketHeader->verWord, NN_G3D_EDIT_VERSION);
        return true;
    }
    return false;
}
}

namespace nn {
namespace g3d {
namespace viewer {
namespace detail {

CommandMonitor::CommandMonitor(
    CommandMonitor::AnalyzeCommandResult (*pAnalyzeCommandCallback)(AnalyzeCommandArg&),
    CommandMonitor::ProcessCommandResult (*pProcessCommandCallback)(ProcessCommandArg&)) NN_NOEXCEPT
    : m_pAllocator(nullptr)
    , m_pWorkBuffer(nullptr)
    , m_pSocket(nullptr)
    , m_PollState(kReady)
    , m_FileLoadState(kBegin)
    , m_FileLoadIndex(0)
    , m_IsReadStarted(false)
    , m_IsCommandAnalyzing(false)
    , m_pAnalyzeCommandCallback(pAnalyzeCommandCallback)
    , m_pProcessCommandCallback(pProcessCommandCallback)
{
    m_PacketErrorHeader.magic = NN_G3D_EDIT_MAGIC;
    m_PacketErrorHeader.verWord = NN_G3D_EDIT_VERSION;
    m_PacketErrorHeader.dataSize = 0;
    m_PacketErrorHeader.command = SYSTEM_PACKET_VERSION_ERROR_COMMAND_FLAG;
}

ViewerResult CommandMonitor::Initialize(const InitArg& arg) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(arg.pAllocator);
    m_pAllocator = arg.pAllocator;

    {
        void* buffer = m_pAllocator->Allocate(sizeof(EditSocketGeneric), Alignment_Default, AllocateType_Communication);
        m_pSocket = new (buffer) EditSocketGeneric();
        if (buffer == nullptr)
        {
            return ViewerResult_MemoryAllocationFailed;
        }

        ViewerResult result = m_pSocket->Initialize(EditSocketBase::SetupArg());
        if (result != ViewerResult_Success)
        {
            return result;
        }

        m_pSocket->SetWriteLogEnabled();
    }

    {
        void* pBuffer = m_pAllocator->Allocate(sizeof(EditWorkBuffer), nn::g3d::detail::Alignment_Default, AllocateType_Communication);
        if (pBuffer == nullptr)
        {
            return ViewerResult_MemoryAllocationFailed;
        }

        m_pWorkBuffer = new (pBuffer) EditWorkBuffer(m_pAllocator, AllocateType_Communication);
    }

    return ViewerResult_Success;
}

CommandMonitor::~CommandMonitor() NN_NOEXCEPT
{
    if (m_pSocket != nullptr)
    {
        Close();
        m_pSocket->~EditSocketBase();
        m_pAllocator->Free(m_pSocket);
        m_pSocket = nullptr;
    }

    if (m_pWorkBuffer != nullptr)
    {
        m_pWorkBuffer->~EditWorkBuffer();
        m_pAllocator->Free(m_pWorkBuffer);
        m_pWorkBuffer = nullptr;
    }

    m_pAllocator = nullptr;
}

bool
CommandMonitor::IsConnected() const NN_NOEXCEPT
{
    return m_pSocket->IsConnected();
}

bool
CommandMonitor::RequestOpen() NN_NOEXCEPT
{
    m_pSocket->RequestOpen();
    return true;
}

void
CommandMonitor::Close() NN_NOEXCEPT
{
    m_pSocket->Close();
}

void CommandMonitor::PollSocket() NN_NOEXCEPT
{
    if (!m_pSocket->IsWriting())
    {
        m_pSocket->ResetWriteFlag();
    }

    if (!m_pSocket->IsReading())
    {
        m_pSocket->ResetReadFlag();
    }

    m_pSocket->Poll();
}

bool CommandMonitor::PollDataCommunication() NN_NOEXCEPT
{
    PollSocket();

    for (int idx = 0; idx < kEnd; ++idx)
    {
        PollState startState = m_PollState;

        if (!m_pSocket->IsConnected())
        {
            ClearState();
            return false;
        }

        switch (m_PollState)
        {
        case kReady:
            {
                if (m_pSocket->ReadASync(&m_PacketHeader, sizeof(PacketHeader)))
                {
                    m_PollState = kPacketAnalyzing;
                }
            }
            break;
        case kPacketAnalyzing:
            {
                if (!m_pSocket->IsReading())
                {
                    m_pSocket->ResetReadFlag();
                    m_IsReadStarted = false;
                    bool success = AnalyzePacket();
                    if (!success)
                    {
                        ClearState();
                        return false;
                    }

                    m_PollState = m_IsCommandAnalyzing? kCommandAnalyzing : kReady;
                }
            }
            break;
        case kCommandAnalyzing:
            {
                if (!m_pSocket->IsReading())
                {
                    AnalyzeCommandArg arg;
                    arg.command = static_cast<CommandFlag>(m_PacketHeader.command);
                    arg.pWorkBuffer = m_pWorkBuffer->GetWorkBufferPtr();
                    // TODO: 後でsize_tにする
                    arg.workBufferSize = static_cast<int>(m_pWorkBuffer->GetSize());
                    AnalyzeCommandResult result = m_pAnalyzeCommandCallback(arg);
                    switch (result)
                    {
                    case AnalyzeCommandResult_Error:
                        {
                            ClearState();
                            return false;
                        }
                    case AnalyzeCommandResult_Finish:
                        {
                            m_PollState = kReady;
                        }
                        break;
                    case AnalyzeCommandResult_RequestProcessCommand:
                        {
                            m_PollState = kCommandProcessing;
                        }
                        break;
                    default:
                        NN_UNEXPECTED_DEFAULT;
                    }

                    m_IsCommandAnalyzing = false;
                    m_pSocket->ResetReadFlag();
                }
            }
            break;
        default:
            break;
        }

        if (startState == m_PollState)
        {
            return true;
        }
    }

    return true;
}

bool CommandMonitor::PollDataEdit() NN_NOEXCEPT
{
    switch (m_PollState)
    {
    case kCommandProcessing:
        {
            ProcessCommandArg arg;
            arg.command = static_cast<CommandFlag>(m_PacketHeader.command);
            arg.pWorkBuffer = m_pWorkBuffer->GetWorkBufferPtr();
            // TODO: 後でsize_tにする
            arg.workBufferSize = static_cast<int>(m_pWorkBuffer->GetSize());
            ProcessCommandResult result = m_pProcessCommandCallback(arg);
            if (result == ProcessCommandResult_Error)
            {
                m_PollState = kReady;
                return false;
            }

            m_PollState = kReady;
        }
        break;
    default:
        break;
    }

    return true;
}

void
CommandMonitor::ClearState() NN_NOEXCEPT
{
    m_pSocket->ResetWriteFlag();
    m_pSocket->ResetReadFlag();

    m_PollState = kReady;
    m_FileLoadState = kEnd;

    ClearWorkBuffer();
}

void
CommandMonitor::ClearWorkBuffer() NN_NOEXCEPT
{
    m_pWorkBuffer->Clear();
}

void
CommandMonitor::TryResizeWorkBuffer(int bufferSize) NN_NOEXCEPT
{
    m_pWorkBuffer->Resize(bufferSize);
}

bool CommandMonitor::AnalyzePacket() NN_NOEXCEPT
{
    // 正常なパケットではない場合は、処理を抜ける
    bool isAbnormalPacket = IsAbnormalPacket(&m_PacketHeader);
    if (isAbnormalPacket)
    {
        NN_G3D_VIEWER_LOG("Abnormal packet received.\n");
        m_PacketHeader.command = SYSTEM_PACKET_VERSION_ERROR_COMMAND_FLAG;
        m_PacketHeader.dataSize = 0;

        // 今の所、ここには、パケットエラーの時しか処理がこないはず
        m_pSocket->WriteSync(&m_PacketErrorHeader, sizeof(m_PacketErrorHeader));
        return false;
    }

    uint32_t commandCategory = m_PacketHeader.command & COMMAND_CATEGORY_FLAG_MASK;
    switch(commandCategory)
    {
    case FILEDATA_CATEGORY_FLAG:
    case MATERIAL_CATEGORY_FLAG:
    case EDIT_CATEGORY_FLAG:
    case ANIMATION_CATEGORY_FLAG:
    case SHADER_CATEGORY_FLAG:
    case BONE_CATEGORY_FLAG:
    case MODEL_CATEGORY_FLAG:
    case MODEL_ANIMATION_CATEGORY_FLAG:
    case SCENE_ANIMATION_CATEGORY_FLAG:
    case PICK_CATEGORY_FLAG:
    case OTHER_CATEGORY_FLAG:
    case TEXTURE_CATEGORY_FLAG:
        {
            TryResizeWorkBuffer(m_PacketHeader.dataSize);
            m_IsCommandAnalyzing = m_pSocket->ReadASync(m_pWorkBuffer->GetWorkBufferPtr(), m_PacketHeader.dataSize);
        }
        return true;
    case SYSTEM_CATEGORY_FLAG:
        if (m_PacketHeader.command != SYSTEM_PACKET_VERSION_ERROR_COMMAND_FLAG)
        {
            TryResizeWorkBuffer(m_PacketHeader.dataSize);
            m_IsCommandAnalyzing = m_pSocket->ReadASync(m_pWorkBuffer->GetWorkBufferPtr(), m_PacketHeader.dataSize);
        }
        return true;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

bool
CommandMonitor::SendBeginFreeze() NN_NOEXCEPT
{
    if (!IsConnected())
    {
        return false;
    }
    m_SendPacketHeader.command = SYSTEM_BEGIN_FREEZE_COMMAND_FLAG;
    m_SendPacketHeader.dataSize = 0;
    m_SendPacketHeader.magic = NN_G3D_EDIT_MAGIC;
    m_SendPacketHeader.verWord = NN_G3D_EDIT_VERSION;
    return m_pSocket->WriteSync(&m_SendPacketHeader, sizeof(m_SendPacketHeader));
}

bool
CommandMonitor::SendEndFreeze() NN_NOEXCEPT
{
    if (!IsConnected())
    {
        return false;
    }
    m_SendPacketHeader.command = SYSTEM_END_FREEZE_COMMAND_FLAG;
    m_SendPacketHeader.dataSize = 0;
    m_SendPacketHeader.magic = NN_G3D_EDIT_MAGIC;
    m_SendPacketHeader.verWord = NN_G3D_EDIT_VERSION;
    return m_pSocket->WriteSync(&m_SendPacketHeader, sizeof(m_SendPacketHeader));
}

}}}} // namespace evfl::viewer::internal

