﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/g3d/g3d_SkeletonObj.h>
#include <nn/g3d/detail/g3d_Flag.h>
#include <nn/g3d/detail/g3d_Perf.h>
#include <nn/g3d/g3d_Billboard.h>
#include <nn/gfx/util/gfx_ObjectDebugLabel.h>

namespace nn { namespace g3d {

// メンバ関数ポインターはサイズの解釈が一定でないためヘッダに出さない
class SkeletonObj::Impl
{
public:
    static void (SkeletonObj::* const s_pFuncCalculateWorld[])(const nn::util::Matrix4x3fType&);
    static void (SkeletonObj::* const s_pFuncCalculateWorldWithCallback[])(const nn::util::Matrix4x3fType&);
};

class CalculateWorldNoScale;
class CalculateWorldStd;
class CalculateWorldMaya;
class CalculateWorldSoftimage;

void (SkeletonObj::* const SkeletonObj::Impl::s_pFuncCalculateWorld[])(const nn::util::Matrix4x3fType&) =
{
    &SkeletonObj::CalculateWorldImpl<CalculateWorldNoScale, false>,
    &SkeletonObj::CalculateWorldImpl<CalculateWorldStd, false>,
    &SkeletonObj::CalculateWorldImpl<CalculateWorldMaya, false>,
    &SkeletonObj::CalculateWorldImpl<CalculateWorldSoftimage, false>
};

void (SkeletonObj::* const SkeletonObj::Impl::s_pFuncCalculateWorldWithCallback[])(const nn::util::Matrix4x3fType&) =
{
    &SkeletonObj::CalculateWorldImpl<CalculateWorldNoScale, true>,
    &SkeletonObj::CalculateWorldImpl<CalculateWorldStd, true>,
    &SkeletonObj::CalculateWorldImpl<CalculateWorldMaya, true>,
    &SkeletonObj::CalculateWorldImpl<CalculateWorldSoftimage, true>
};

void SkeletonObj::InitializeArgument::CalculateMemorySize() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(GetBufferingCount() > 0);
    const ResSkeleton* pRes = GetResource();
    int boneCount = pRes->GetBoneCount();
    int bufferCount = GetBufferingCount();
    int scaleMode = pRes->GetScaleMode();

    for (int blockIndex = 0; blockIndex < MemoryBlockIndex_End; blockIndex++)
    {
        m_MemoryBlock[blockIndex].Initialize();
    }

    // サイズ計算
    m_MemoryBlock[MemoryBlockIndex_WorldMtx].SetAlignment(MatrixVectorAlignment, 0);
    m_MemoryBlock[MemoryBlockIndex_WorldMtx].SetSizeBy<nn::util::Matrix4x3fType>(MatrixVectorAlignment, boneCount);
    m_MemoryBlock[MemoryBlockIndex_LocalMtx].SetAlignment(MatrixVectorAlignment, 0);
    m_MemoryBlock[MemoryBlockIndex_LocalMtx].SetSizeBy<LocalMtx>(MatrixVectorAlignment, boneCount);
    if (scaleMode == ScaleMode_Softimage)
    {
        m_MemoryBlock[MemoryBlockIndex_Scale].SetAlignment(MatrixVectorAlignment, 0);
        m_MemoryBlock[MemoryBlockIndex_Scale].SetSizeBy<nn::util::Vector3fType>(MatrixVectorAlignment, boneCount);
    }
    else
    {
        m_MemoryBlock[MemoryBlockIndex_Scale].SetSize(0);
    }
    m_MemoryBlock[MemoryBlockIndex_Buffer].SetSizeBy<nn::gfx::Buffer>(1, bufferCount);

    m_WorkMemory.Initialize();
    for (int blockIndex = 0; blockIndex < MemoryBlockIndex_End; blockIndex++)
    {
        m_WorkMemory.Append(&m_MemoryBlock[blockIndex]);
    }
}

bool SkeletonObj::Initialize(const InitializeArgument& arg, void* pBuffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(pBuffer != NULL);
    NN_SDK_REQUIRES(IsAligned(pBuffer, Alignment_Buffer));

    if (arg.IsMemoryCalculated() == false)
    {
        return false;
    }
    if (arg.GetWorkMemorySize() > bufferSize)
    {
        // バッファーが必要なサイズに満たない場合は失敗。
        return false;
    }

    const ResSkeleton* pRes = arg.GetResource();

    // メンバの初期化。
    m_pRes               = pRes;
    m_Flag               = static_cast<Bit16>(pRes->ToData().flag);
    m_BufferingCount     = static_cast<uint8_t>(arg.GetBufferingCount());
    m_pBoneArray         = pRes->ToData().pBoneArray.Get();
    m_pLocalMtxArray     = arg.GetBuffer<LocalMtx>(pBuffer, InitializeArgument::MemoryBlockIndex_LocalMtx);
    m_pMemLocalMtxArray  = m_pLocalMtxArray;
    m_pWorldMtxArray     = arg.GetBuffer<nn::util::Matrix4x3fType>(pBuffer, InitializeArgument::MemoryBlockIndex_WorldMtx);
    m_pMemWorldMtxArray  = m_pWorldMtxArray;
    m_pScaleArray        = arg.GetBuffer<nn::util::Vector3fType>(pBuffer, InitializeArgument::MemoryBlockIndex_Scale);
    m_pMtxBlockArray     = arg.GetBuffer<nn::gfx::Buffer>(pBuffer, InitializeArgument::MemoryBlockIndex_Buffer);
    m_BoneCount          = static_cast<uint16_t>(pRes->GetBoneCount());
    m_CallbackBone       = ICalculateWorldCallback::InvalidBone;
    m_pCallback          = NULL;
    m_pUserPtr           = NULL;
    m_pBufferPtr         = pBuffer;
    m_SizePerMtxBlock    = 0;
    m_pMemoryPool        = NULL;
    m_MemoryPoolOffset   = 0;

    ClearLocalMtx();

    // 各バッファーは計算結果が参照されるべきなのでここでは初期化しない。

    return true;
}

size_t SkeletonObj::GetBlockBufferAlignment(nn::gfx::Device* pDevice) const NN_NOEXCEPT
{
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetSize(sizeof(nn::util::FloatColumnMajor4x3) * m_pRes->GetMtxCount());
    info.SetGpuAccessFlags( nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_UnorderedAccessBuffer );

    return nn::gfx::Buffer::GetBufferAlignment(pDevice, info);
}

size_t SkeletonObj::CalculateBlockBufferSize(nn::gfx::Device* pDevice) const NN_NOEXCEPT
{
    return nn::util::align_up(sizeof(nn::util::FloatColumnMajor4x3) * m_pRes->GetMtxCount(), GetBlockBufferAlignment(pDevice)) * m_BufferingCount;
}

void SkeletonObj::SetupBlockBufferImpl(nn::gfx::Device* pDevice, nn::gfx::MemoryPool* pMemoryPool, ptrdiff_t offset, size_t memoryPoolSize) NN_NOEXCEPT
{
    NN_UNUSED(memoryPoolSize);

    // 定数バッファーおよびビュー作成。
    m_SizePerMtxBlock = sizeof(nn::util::FloatColumnMajor4x3) * m_pRes->GetMtxCount();
    ptrdiff_t memoryPoolOffset = offset;
    for (int bufferIndex = 0; bufferIndex < m_BufferingCount; ++bufferIndex)
    {
        nn::gfx::Buffer::InfoType bufferInfo;
        bufferInfo.SetDefault();
        bufferInfo.SetSize(m_SizePerMtxBlock);
        bufferInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_UnorderedAccessBuffer);
        nn::gfx::Buffer* pMtxBlock = new(&m_pMtxBlockArray[bufferIndex]) nn::gfx::Buffer;
        pMtxBlock->Initialize(pDevice, bufferInfo, pMemoryPool, memoryPoolOffset, m_SizePerMtxBlock);
        nn::gfx::util::SetBufferDebugLabel(pMtxBlock, "g3d_SkeletonUniformBlock");
        memoryPoolOffset += nn::util::align_up(m_SizePerMtxBlock, GetBlockBufferAlignment(pDevice));
    }

    // 各バッファーは計算結果が参照されるべきなのでここでは初期化しない。

    m_Flag |= Flag_BlockBufferValid;

    return;
}

bool SkeletonObj::SetupBlockBuffer(nn::gfx::Device* pDevice, nn::gfx::MemoryPool* pMemoryPool, ptrdiff_t offset, size_t memoryPoolSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(pMemoryPool != NULL || memoryPoolSize == 0);
    NN_SDK_REQUIRES(IsBlockBufferValid() == false);

    size_t size = CalculateBlockBufferSize(pDevice);

    if (size > memoryPoolSize)
    {
        // バッファーが必要なサイズに満たない場合は失敗。
        return false;
    }

    // sizeが0の場合、nn::gfx::bufferを作成する必要がないので、リターンする
    if (size == 0)
    {
        return true;
    }

    m_pMemoryPool = pMemoryPool;
    m_MemoryPoolOffset = offset;

    SetupBlockBufferImpl(pDevice, pMemoryPool, offset, memoryPoolSize);

    return true;
}

void SkeletonObj::CleanupBlockBuffer(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsBlockBufferValid() == true);

    for (int blockIndex = 0; blockIndex < m_BufferingCount; ++blockIndex)
    {
        nn::gfx::Buffer& mtxBlock = m_pMtxBlockArray[blockIndex];
        mtxBlock.Finalize(pDevice);
        mtxBlock.nn::gfx::Buffer::~TBuffer();
    }

    m_Flag &= ~Flag_BlockBufferValid;
    m_SizePerMtxBlock = 0;

    m_pMemoryPool = NULL;
    m_MemoryPoolOffset = 0;
}

//--------------------------------------------------------------------------------------------------

void SkeletonObj::ClearLocalMtx() NN_NOEXCEPT
{
    LocalMtx* pLocalMtxArray = GetLocalMtxArray();
    if (RotateMode_Quat != GetRotateMode())
    {
        // Euler
        int boneCount = m_pRes->GetBoneCount();
        for (int idxBone = 0; idxBone < boneCount; ++idxBone)
        {
            const ResBone* pBone = m_pRes->GetBone(idxBone);
            LocalMtx& local = pLocalMtxArray[idxBone];
            local.flag = pBone->ToData().flag & (ResBone::Mask_Transform | ResBone::Mask_Billboard);
            VectorLoad(&local.scale, pBone->GetScale());
            nn::util::Vector3fType vectorRotateEuler;
            VectorLoad(&vectorRotateEuler, pBone->GetRotateEuler());
            nn::util::Vector3fType vectorTranslate;
            VectorLoad(&vectorTranslate, pBone->GetTranslate());
            MatrixSetRotateXyz(&local.mtxRT, vectorRotateEuler);
            MatrixSetTranslate(&local.mtxRT, vectorTranslate);
        }
    }
    else
    {
        // Quaternion
        int boneCount = m_pRes->GetBoneCount();
        for (int idxBone = 0; idxBone < boneCount; ++idxBone)
        {
            const ResBone* pBone = m_pRes->GetBone(idxBone);
            LocalMtx& local = pLocalMtxArray[idxBone];
            local.flag = pBone->ToData().flag & (ResBone::Mask_Transform | ResBone::Mask_Billboard);
            VectorLoad(&local.scale, pBone->GetScale());
            nn::util::Vector4fType vectorQuatanion;
            VectorLoad(&vectorQuatanion, pBone->GetRotateQuat());
            nn::util::Vector3fType vectorTranslate;
            VectorLoad(&vectorTranslate, pBone->GetTranslate());
            MatrixSetRotate(&local.mtxRT, vectorQuatanion);
            MatrixSetTranslate(&local.mtxRT, vectorTranslate);
        }
    }
}

void SkeletonObj::CalculateBillboardMtx(
    nn::util::Matrix4x3fType* pMtx, const nn::util::Matrix4x3fType& cameraMtx, int boneIndex, bool combineWorld) const NN_NOEXCEPT
{
    NN_G3D_PERF_LEVEL1("SkeletonObj::CalculateBillboardMtx");
    NN_G3D_REQUIRES(pMtx != NULL, NN_G3D_RES_GET_NAME(GetResBone(boneIndex), GetName()));

    const nn::util::Matrix4x3fType* pWorldMtxArray = GetWorldMtxArray();
    Bit32 bbMode = GetResBone(boneIndex)->GetBillboardMode();
    const nn::util::Matrix4x3fType& world = pWorldMtxArray[boneIndex];

    nn::util::Matrix4x3fType worldView;
    MatrixMultiply(&worldView, world, cameraMtx);
    Billboard::Calculate(bbMode, &worldView, cameraMtx, world, m_pLocalMtxArray[boneIndex].mtxRT);

    // 階層ビルボードのためビルボード行列単体を取り出す。
    nn::util::Matrix4x3fType inverseMtx;
    bool result = MatrixInverse(&inverseMtx, cameraMtx);
    NN_G3D_WARNING(result == true, "Fail to calculate inverse matrix");
    MatrixMultiply(pMtx, worldView, inverseMtx);

    if (combineWorld)
    {
        return;
    }

    result = MatrixInverse(&inverseMtx, world);
    NN_G3D_WARNING(result == true, "Fail to calculate inverse matrix");
    MatrixMultiply(pMtx, inverseMtx, *pMtx);
}

void SkeletonObj::CalculateWorldMtx(const nn::util::Matrix4x3fType& baseMtx) NN_NOEXCEPT
{
    NN_G3D_PERF_LEVEL1("SkeletonObj::CalculateWorldMtx");
    int funcIndex = (m_Flag & Mask_Scale) >> (Shift_Scale);
    NN_SDK_ASSERT(funcIndex < sizeof(Impl::s_pFuncCalculateWorld) / sizeof(*Impl::s_pFuncCalculateWorld));

#if defined(NN_BUILD_CONFIG_OS_COS)
    if (m_pWorldMtxArray == m_pMemWorldMtxArray)
    {
        // ワールド行列の領域をキャッシュに乗せます。サイズはアライメントまで切り上げられます。
        DCZeroRange(m_pWorldMtxArray, sizeof(nn::util::Matrix4x3fType) * GetBoneCount());
    }
#endif
    if (m_pCallback && m_CallbackBone != ICalculateWorldCallback::InvalidBone)
    {
        (this->*Impl::s_pFuncCalculateWorldWithCallback[funcIndex])(baseMtx);
    }
    else
    {
        (this->*Impl::s_pFuncCalculateWorld[funcIndex])(baseMtx);
    }
}

void SkeletonObj::CalculateSkeleton(int bufferIndex) NN_NOEXCEPT
{
    NN_G3D_PERF_LEVEL1("SkeletonObj::CalculateSkeleton");

    if (m_SizePerMtxBlock == 0)
    {
        return; // 更新不要。
    }

    // TODO: 最適化
    const int16_t* pMtxToBoneTable = m_pRes->ToData().pMtxToBoneTable.Get();
    const nn::util::FloatColumnMajor4x3* pInvModelMatrixArray = m_pRes->ToData().pInvModelMatrixArray.Get();
    int idxMtx = 0;
    nn::util::FloatColumnMajor4x3* pMtxArray = NULL;

    // 更新領域をマップ
    pMtxArray = m_pMtxBlockArray[bufferIndex].Map< nn::util::FloatColumnMajor4x3 >();

    if (IsBlockSwapEnabled())
    {
        // スムーススキン
        int smoothSkinMtxCount = m_pRes->GetSmoothMtxCount();
        for (; idxMtx < smoothSkinMtxCount; ++idxMtx)
        {
            int idxBone = pMtxToBoneTable[idxMtx];
            const nn::util::Matrix4x3fType& worldMtx = m_pWorldMtxArray[idxBone];
            const nn::util::FloatColumnMajor4x3& invModelMtx = pInvModelMatrixArray[idxMtx];
            nn::util::Matrix4x3fType invModelMtx4x3;
            MatrixLoad(&invModelMtx4x3, invModelMtx);
            nn::util::Matrix4x3fType mtx;
            MatrixMultiply(&mtx, invModelMtx4x3, worldMtx);
            MatrixStore(&pMtxArray[idxMtx], mtx);
            MatrixSwapEndian(&pMtxArray[idxMtx], pMtxArray[idxMtx]);
        }

        // リジッドスキン
        int mtxCount = m_pRes->GetMtxCount();
        for (; idxMtx < mtxCount; ++idxMtx)
        {
            int idxBone = pMtxToBoneTable[idxMtx];
            MatrixStore(&pMtxArray[idxMtx], m_pWorldMtxArray[idxBone]);
            MatrixSwapEndian(&pMtxArray[idxMtx], pMtxArray[idxMtx]);
        }
    }
    else
    {
        // スムーススキン
        int smoothSkinMtxCount = m_pRes->GetSmoothMtxCount();
        for (; idxMtx < smoothSkinMtxCount; ++idxMtx)
        {
            int idxBone = pMtxToBoneTable[idxMtx];
            const nn::util::Matrix4x3fType& worldMtx = m_pWorldMtxArray[idxBone];
            const nn::util::FloatColumnMajor4x3& invModelMtx = pInvModelMatrixArray[idxMtx];
            nn::util::Matrix4x3fType invModelMtx4x3;
            MatrixLoad(&invModelMtx4x3, invModelMtx);
            nn::util::Matrix4x3fType mtx;
            MatrixMultiply(&mtx, invModelMtx4x3, worldMtx);
            MatrixStore(&pMtxArray[idxMtx], mtx);
        }

        // リジッドスキン
        int mtxCount = m_pRes->GetMtxCount();
        for (; idxMtx < mtxCount; ++idxMtx)
        {
            int idxBone = pMtxToBoneTable[idxMtx];
            MatrixStore(&pMtxArray[idxMtx], m_pWorldMtxArray[idxBone]);
        }
    }

    m_pMtxBlockArray[bufferIndex].FlushMappedRange(0, m_SizePerMtxBlock);
    m_pMtxBlockArray[bufferIndex].Unmap();
}

//--------------------------------------------------------------------------------------------------

template<typename CalculateType, bool useCallback>
void SkeletonObj::CalculateWorldImpl(const nn::util::Matrix4x3fType& baseMtx)
{
    int boneCount = m_BoneCount;
    NN_SDK_ASSERT(boneCount > 0);

    CalculateType op(GetLocalMtxArray(), m_pWorldMtxArray, m_pScaleArray);

    // ベース変換の都合上、ルートボーンは別処理にする。
    op.CalculateRootBone(baseMtx);
    int idxBone = 0;
    ICalculateWorldCallback::CallbackArg callbackArg(idxBone, m_CallbackBone);

    if (NN_STATIC_CONDITION(useCallback) && callbackArg.GetCallbackBoneIndex() == idxBone)
    {
        WorldMtxManip manip(
            &m_pWorldMtxArray[idxBone], op.GetScale(idxBone), &m_pLocalMtxArray[idxBone].flag);
        m_pCallback->Exec(callbackArg, manip);
    }

    for (++idxBone; idxBone < boneCount; ++idxBone)
    {
        const ResBone* pBone = GetResBone(idxBone);

        // モード毎の処理
        op.CalculateBone(pBone);

        if (NN_STATIC_CONDITION(useCallback) && callbackArg.GetCallbackBoneIndex() == idxBone)
        {
            WorldMtxManip manip(
                &m_pWorldMtxArray[idxBone], op.GetScale(idxBone), &m_pLocalMtxArray[idxBone].flag);
            m_pCallback->Exec(callbackArg, manip);
        }
    }

    op.CalculateScale(boneCount);
}

// CalculateWorld の雛形クラス
class CalculateWorldBase
{
public:
    CalculateWorldBase(LocalMtx* pLocalMtxArray, nn::util::Matrix4x3fType* pWorldMtxArray, nn::util::Vector3fType* pScaleArray)
        : m_pLocalMtxArray(pLocalMtxArray)
        , m_pWorldMtxArray(pWorldMtxArray)
        , m_pScaleArray(pScaleArray)
    {
    }

    //! ルートボーン以外のワールド計算
    NN_FORCEINLINE void CalculateBone( const ResBone* /*bone*/)
    {
        NN_SDK_ASSERT(false);
    }

    //! ルートボーンのワールド計算
    NN_FORCEINLINE
    void CalculateRootBone(const nn::util::Matrix4x3fType& baseMtx)
    {
        LocalMtx& local = m_pLocalMtxArray[0];
        if (CheckFlag(local.flag, ResBone::Flag_RotTransZero))
        {
            m_pWorldMtxArray[0] = baseMtx;
        }
        else
        {
            MatrixMultiply(&m_pWorldMtxArray[0], local.mtxRT, baseMtx);
        }

        // baseMtx は単位スケール扱いとする。
        AccumulateFlag(local.flag, ResBone::Flag_ScaleOne | ResBone::Flag_HiScaleOne);
    }

    //! 各ボーンへのスケールの反映
    NN_FORCEINLINE
    void CalculateScale(int boneCount)
    {
        for (int idxBone = 0; idxBone < boneCount; ++idxBone)
        {
            const LocalMtx& local = m_pLocalMtxArray[idxBone];
            if (!CheckFlag(local.flag, ResBone::Flag_ScaleOne))
            {
                nn::util::Matrix4x3fType& worldMtx = m_pWorldMtxArray[idxBone];
                MatrixScaleBase(&worldMtx, worldMtx, local.scale);
            }
        }
    }

    //! スケールの取得
    NN_FORCEINLINE
    nn::util::Vector3fType* GetScale(int idxBone)
    {
        return &m_pLocalMtxArray[idxBone].scale;
    }

protected:
    //! SRT のフラグを累積する
    NN_FORCEINLINE void AccumulateFlag(Bit32& flag, Bit32 parentFlag)
    {
        flag &= ~ResBone::Flag_HiIdentity;
        flag |= (parentFlag & (flag << ResBone::Shift_Hierarchy)) &
            static_cast<Bit32>(ResBone::Flag_HiIdentity);
    }

    LocalMtx* m_pLocalMtxArray;
    nn::util::Matrix4x3fType* m_pWorldMtxArray;
    nn::util::Vector3fType* m_pScaleArray;
};

//--------------------------------------------------------------------------------------------------

class CalculateWorldNoScale : public CalculateWorldBase
{
public:
    CalculateWorldNoScale(LocalMtx* pLocalMtxArray, nn::util::Matrix4x3fType* pWorldMtxArray, nn::util::Vector3fType* pScaleArray)
        : CalculateWorldBase(pLocalMtxArray, pWorldMtxArray, pScaleArray)
    {
    }

    NN_FORCEINLINE
    void CalculateBone(const ResBone* bone)
    {
        NN_G3D_PERF_LEVEL2("CalculateWorldNoScale::CalculateBone");
        int index = bone->GetIndex();
        int parentIndex = bone->GetParentIndex();
        nn::util::Matrix4x3fType& worldMtx = m_pWorldMtxArray[index];
        LocalMtx& local = m_pLocalMtxArray[index];
        const nn::util::Matrix4x3fType& parentMtx = m_pWorldMtxArray[parentIndex];
        const LocalMtx& parent = m_pLocalMtxArray[parentIndex];
        AccumulateFlag(local.flag, parent.flag);

        if (CheckFlag(local.flag, ResBone::Flag_RotTransZero))
        {
            worldMtx = parentMtx;
        }
        else
        {
            MatrixMultiply(&worldMtx, local.mtxRT, parentMtx);
        }
    }

    NN_FORCEINLINE
    void CalculateScale(int boneCount)
    {
        (void)boneCount;
    }

    //! スケールの取得
    NN_FORCEINLINE
    nn::util::Vector3fType* GetScale(int idxBone)
    {
        (void)idxBone;
        return NULL;
    }
};

class CalculateWorldStd : public CalculateWorldBase
{
public:
    CalculateWorldStd(LocalMtx* pLocalMtxArray, nn::util::Matrix4x3fType* pWorldMtxArray, nn::util::Vector3fType* pScaleArray)
        : CalculateWorldBase(pLocalMtxArray, pWorldMtxArray, pScaleArray)
    {
    }

    NN_FORCEINLINE
    void CalculateBone(const ResBone* bone)
    {
        NN_G3D_PERF_LEVEL2("CalculateWorldStd::CalculateBone");
        int index = bone->GetIndex();
        int parentIndex = bone->GetParentIndex();
        nn::util::Matrix4x3fType& worldMtx = m_pWorldMtxArray[index];
        LocalMtx& local = m_pLocalMtxArray[index];
        const nn::util::Matrix4x3fType& parentMtx = m_pWorldMtxArray[parentIndex];
        const LocalMtx& parent = m_pLocalMtxArray[parentIndex];
        AccumulateFlag(local.flag, parent.flag);

        if (CheckFlag(parent.flag, ResBone::Flag_ScaleOne))
        {
            if (CheckFlag(local.flag, ResBone::Flag_RotTransZero))
            {
                worldMtx = parentMtx;
            }
            else
            {
                MatrixMultiply(&worldMtx, local.mtxRT, parentMtx);
            }
        }
        else
        {
            MatrixScaleBase(&worldMtx, parentMtx, parent.scale);
            if (!CheckFlag(local.flag, ResBone::Flag_RotTransZero))
            {
                MatrixMultiply(&worldMtx, local.mtxRT, worldMtx);
            }
        }
    }
};

class CalculateWorldMaya : public CalculateWorldBase
{
public:
    CalculateWorldMaya(LocalMtx* pLocalMtxArray, nn::util::Matrix4x3fType* pWorldMtxArray, nn::util::Vector3fType* pScaleArray)
        : CalculateWorldBase(pLocalMtxArray, pWorldMtxArray, pScaleArray)
    {
    }

    NN_FORCEINLINE
    void CalculateBone(const ResBone* bone)
    {
        NN_G3D_PERF_LEVEL2("CalculateWorldMaya::CalculateBone");
        int index = bone->GetIndex();
        int parentIndex = bone->GetParentIndex();
        nn::util::Matrix4x3fType& worldMtx = m_pWorldMtxArray[index];
        LocalMtx& local = m_pLocalMtxArray[index];
        const nn::util::Matrix4x3fType& parentMtx = m_pWorldMtxArray[parentIndex];
        const LocalMtx& parent = m_pLocalMtxArray[parentIndex];
        AccumulateFlag(local.flag, parent.flag);

        if (CheckFlag(local.flag, ResBone::Flag_RotTransZero))
        {
            if ((local.flag & ResBone::Flag_SegmentScaleCompensate) ||
                CheckFlag(parent.flag, ResBone::Flag_ScaleOne))
            {
                worldMtx = parentMtx; // T がゼロなので、SSC でなければスケールの処理も省略する。
            }
            else
            {
                MatrixScaleBase(&worldMtx, parentMtx, parent.scale);
            }
        }
        else
        {
            if (CheckFlag(parent.flag, ResBone::Flag_ScaleOne))
            {
                MatrixMultiply(&worldMtx, local.mtxRT, parentMtx);
            }
            else if (local.flag & ResBone::Flag_SegmentScaleCompensate)
            {
                worldMtx = local.mtxRT;
                nn::util::Vector3fType vectorW;
                MatrixGetAxisW(&vectorW, worldMtx);
                VectorMultiply(&vectorW, vectorW, parent.scale);
                MatrixSetAxisW(&worldMtx, vectorW);
                MatrixMultiply(&worldMtx, worldMtx, parentMtx);
            }
            else
            {
                MatrixScaleBase(&worldMtx, parentMtx, parent.scale);
                MatrixMultiply(&worldMtx, local.mtxRT, worldMtx);
            }
        }
    }
};

class CalculateWorldSoftimage : public CalculateWorldBase
{
public:
    CalculateWorldSoftimage(
        LocalMtx* pLocalMtxArray, nn::util::Matrix4x3fType* pWorldMtxArray, nn::util::Vector3fType* pScaleArray)
        : CalculateWorldBase(pLocalMtxArray, pWorldMtxArray, pScaleArray)
    {
    }

    NN_FORCEINLINE
    void CalculateRootBone(const nn::util::Matrix4x3fType& baseMtx)
    {
        // baseMtx は Hierarchical Scaling ではなく標準の SRT として扱う。
        // この挙動が問題となる場合はルートボーンを baseMtx の代用として回避する。

        CalculateWorldBase::CalculateRootBone(baseMtx);
        LocalMtx& local = m_pLocalMtxArray[0];
        if (CheckFlag(local.flag, ResBone::Flag_ScaleOne))
        {
            VectorSet(&m_pScaleArray[0], 1.0f, 1.0f, 1.0f);
        }
        else
        {
            m_pScaleArray[0] = local.scale;
        }
    }

    NN_FORCEINLINE
    void CalculateBone(const ResBone* bone)
    {
        NN_G3D_PERF_LEVEL2("CalculateWorldSoftimage::CalculateBone");
        int index = bone->GetIndex();
        int parentIndex = bone->GetParentIndex();
        nn::util::Vector3fType& scale = m_pScaleArray[index];
        nn::util::Matrix4x3fType& worldMtx = m_pWorldMtxArray[index];
        LocalMtx& local = m_pLocalMtxArray[index];
        const nn::util::Vector3fType& parentScale = m_pScaleArray[parentIndex];
        const nn::util::Matrix4x3fType& parentMtx = m_pWorldMtxArray[parentIndex];
        const LocalMtx& parent = m_pLocalMtxArray[parentIndex];
        AccumulateFlag(local.flag, parent.flag);

        if (CheckFlag(local.flag, ResBone::Flag_RotTransZero))
        {
            // T がゼロなので、スケールの処理も省略する。
            worldMtx = parentMtx;
        }
        else
        {
            if (CheckFlag(parent.flag, ResBone::Flag_HiScaleOne))
            {
                MatrixMultiply(&worldMtx, local.mtxRT, parentMtx);
            }
            else
            {
                worldMtx = local.mtxRT;
                nn::util::Vector3fType vectorW;
                MatrixGetAxisW(&vectorW, worldMtx);
                VectorMultiply(&vectorW, vectorW, parent.scale);
                MatrixSetAxisW(&worldMtx, vectorW);
                MatrixMultiply(&worldMtx, worldMtx, parentMtx);
            }
        }

        if (CheckFlag(local.flag, ResBone::Flag_ScaleOne))
        {
            scale = parentScale;
        }
        else
        {
            VectorMultiply(&scale, parentScale, local.scale);
        }
    }

    NN_FORCEINLINE
    void CalculateScale(int boneCount)
    {
        for (int idxBone = 0; idxBone < boneCount; ++idxBone)
        {
            const LocalMtx& local = m_pLocalMtxArray[idxBone];
            if (!CheckFlag(local.flag, ResBone::Flag_HiScaleOne))
            {
                nn::util::Matrix4x3fType& worldMtx = m_pWorldMtxArray[idxBone];
                MatrixScaleBase(&worldMtx, worldMtx, m_pScaleArray[idxBone]);
            }
        }
    }

    //! スケールの取得
    NN_FORCEINLINE
    nn::util::Vector3fType* GetScale(int idxBone)
    {
        return &m_pScaleArray[idxBone];
    }
};

}} // namespace nn::g3d
