﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/g3d/g3d_ResShapeAnim.h>
#include <nn/g3d/g3d_ResFile.h>

NN_PRAGMA_PUSH_WARNINGS
NN_DISABLE_WARNING_SHADOW

namespace nn { namespace g3d {
void ResVertexShapeAnim::Evaluate(
    float* pResult, float frame, const int8_t* pSubBindIndex) const NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pResult != NULL,       NN_G3D_RES_GET_NAME(this, GetName()));
    NN_G3D_REQUIRES(pSubBindIndex != NULL, NN_G3D_RES_GET_NAME(this, GetName()));

    const ResKeyShapeAnimInfo* pInfo = ToData().pKeyShapeAnimInfoArray.Get();

    for (int idxKeyShapeAnim = 0, keyShapeAnimCount = GetKeyShapeAnimCount();
        idxKeyShapeAnim < keyShapeAnimCount; ++idxKeyShapeAnim, ++pInfo)
    {
        if (pSubBindIndex[idxKeyShapeAnim] == -1 ||
            pInfo->curveIndex == -1)
        {
            continue;
        }

        const ResAnimCurve* pCurve = GetCurve(pInfo->curveIndex);
        pResult[idxKeyShapeAnim] = pCurve->EvaluateFloat(frame);
    }
}

void ResVertexShapeAnim::Evaluate(
    float* pResult, float frame, const int8_t* pSubBindIndex,
    AnimFrameCache* pFrameCache) const NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pResult != NULL,       NN_G3D_RES_GET_NAME(this, GetName()));
    NN_G3D_REQUIRES(pSubBindIndex != NULL, NN_G3D_RES_GET_NAME(this, GetName()));

    const ResKeyShapeAnimInfo* pInfo = ToData().pKeyShapeAnimInfoArray.Get();

    for (int idxKeyShapeAnim = 0, keyShapeAnimCount = GetKeyShapeAnimCount();
        idxKeyShapeAnim < keyShapeAnimCount; ++idxKeyShapeAnim, ++pInfo)
    {
        if (pSubBindIndex[idxKeyShapeAnim] == -1 ||
            pInfo->curveIndex == -1)
        {
            continue;
        }

        const ResAnimCurve* pCurve = GetCurve(pInfo->curveIndex);
        pResult[idxKeyShapeAnim] = pCurve->EvaluateFloat(frame, &pFrameCache[pInfo->curveIndex]);
    }
}

BindResult ResVertexShapeAnim::PreBind(const ResShape* pTarget) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pTarget != NULL, NN_G3D_RES_GET_NAME(this, GetName()));

    ResKeyShapeAnimInfo* pInfo = ToData().pKeyShapeAnimInfoArray.Get();

    BindResult result;
    for (int idxKeyShapeAnim = 0, keyShapeAnimCount = GetKeyShapeAnimCount();
        idxKeyShapeAnim < keyShapeAnimCount; ++idxKeyShapeAnim, ++pInfo)
    {
        const char* pName = pInfo->pName.Get()->GetData();
        int idxTarget = pTarget->FindKeyShapeIndex(pName);
        NN_G3D_ASSERT(idxTarget >= 0, NN_G3D_RES_GET_NAME(this, GetName())); // シェイプアニメーションはすべてバインドしなければならない。
        pInfo->subbindIndex = static_cast<int8_t>(idxTarget);
        if (idxTarget == nn::util::ResDic::Npos)
        {
            result.SetFailureBit();
        }
        else
        {
            result.SetSuccessBit();
        }
    }
    return result;
}

BindResult ResVertexShapeAnim::BindCheck(const ResShape* pTarget) const NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pTarget != NULL, NN_G3D_RES_GET_NAME(this, GetName()));

    const ResKeyShapeAnimInfo* pInfo = ToData().pKeyShapeAnimInfoArray.Get();

    BindResult result;
    for (int idxKeyShapeAnim = 0, keyShapeAnimCount = GetKeyShapeAnimCount();
        idxKeyShapeAnim < keyShapeAnimCount; ++idxKeyShapeAnim, ++pInfo)
    {
        const char* pName = pInfo->pName.Get()->GetData();
        int idxTarget = pTarget->FindKeyShapeIndex(pName);
        if (idxTarget == nn::util::ResDic::Npos)
        {
            result.SetFailureBit();
        }
        else
        {
            result.SetSuccessBit();
        }
    }
    return result;
}

void ResVertexShapeAnim::Initialize(float* pResult) const NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pResult != NULL, NN_G3D_RES_GET_NAME(this, GetName()));

    // バインドされていない部分も初期化してしまう。
    // ベースシェイプは 0.0f で初期化
    pResult[0] = 0.0f;
    const float* pBaseValueArray = ToData().pBaseValueArray.Get();
    for (int idxKeyShapeAnim = 1, keyShapeAnimCount = GetKeyShapeAnimCount();
        idxKeyShapeAnim < keyShapeAnimCount; ++idxKeyShapeAnim)
    {
        pResult[idxKeyShapeAnim] = pBaseValueArray[idxKeyShapeAnim - 1];
    }
}

//--------------------------------------------------------------------------------------------------

BindResult ResShapeAnim::PreBind(const ResModel* pModel) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pModel != NULL, NN_G3D_RES_GET_NAME(this, GetName()));

    ToData().pBindModel.Set(const_cast<ResModel*>(pModel));
    uint16_t* pBindIndexArray = ToData().pBindIndexArray.Get();

    BindResult result;
    for (int idxAnim = 0, animCount = GetVertexShapeAnimCount(); idxAnim < animCount; ++idxAnim)
    {
        ResVertexShapeAnim* pVertexShapeAnim = GetVertexShapeAnim(idxAnim);

        const char* pName = pVertexShapeAnim->ToData().pName.Get()->GetData();
        const ResShape* pShape = pModel->FindShape(pName);
        if (pShape != NULL)
        {
            pBindIndexArray[idxAnim] = static_cast<uint16_t>(pShape->GetIndex());
            BindResult subBindResult = pVertexShapeAnim->PreBind(pShape);
            result |= subBindResult;
        }
        else
        {
            pBindIndexArray[idxAnim] = AnimFlag_NotBound;
            result.SetFailureBit();
        }
    }
    return result;
}

BindResult ResShapeAnim::BindCheck(const ResModel* pModel) const NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pModel != NULL, NN_G3D_RES_GET_NAME(this, GetName()));

    BindResult result;
    for (int idxAnim = 0, animCount = GetVertexShapeAnimCount(); idxAnim < animCount; ++idxAnim)
    {
        const ResVertexShapeAnim* pVertexShapeAnim = GetVertexShapeAnim(idxAnim);

        const char* pName = pVertexShapeAnim->ToData().pName.Get()->GetData();
        const ResShape* pShape = pModel->FindShape(pName);
        if (pShape != NULL)
        {
            BindResult subBindResult = pVertexShapeAnim->BindCheck(pShape);
            result |= subBindResult;
        }
        else
        {
            result.SetFailureBit();
        }
    }
    return result;
}

bool ResShapeAnim::BakeCurve(void* pBuffer, size_t bufferSize) NN_NOEXCEPT
{
    if (bufferSize == 0)
    {
        return true;
    }
    if (pBuffer == NULL ||
        bufferSize < GetBakedSize())
    {
        return false;
    }
    for (int idxAnim = 0, animCount = GetVertexShapeAnimCount(); idxAnim < animCount; ++idxAnim)
    {
        ResVertexShapeAnim* pVertexShapeAnim = GetVertexShapeAnim(idxAnim);
        for (int idxCurve = 0, curveCount = pVertexShapeAnim->GetCurveCount();
            idxCurve < curveCount; ++idxCurve)
        {
            ResAnimCurve* curve = pVertexShapeAnim->GetCurve(idxCurve);
            size_t size = curve->CalculateBakedFloatSize();
            curve->BakeFloat(pBuffer, size);
            pBuffer = AddOffset(pBuffer, size);
        }
    }
    ToData().flag |= Flag_CurveBaked;

    return true;
}

void* ResShapeAnim::ResetCurve() NN_NOEXCEPT
{
    void* pBuffer = NULL;
    if (IsCurveBaked())
    {
        bool foundCurve = false;
        for (int idxAnim = 0, animCount = GetVertexShapeAnimCount(); idxAnim < animCount; ++idxAnim)
        {
            ResVertexShapeAnim* pVertexShapeAnim = GetVertexShapeAnim(idxAnim);
            for (int idxCurve = 0, curveCount = pVertexShapeAnim->GetCurveCount();
                idxCurve < curveCount; ++idxCurve)
            {
                ResAnimCurve* curve = pVertexShapeAnim->GetCurve(idxCurve);
                Bit32 type = curve->ToData().flag & ResAnimCurve::Mask_Curve;
                if (!foundCurve && type == ResAnimCurve::CurveType_BakedFloat)
                {
                    pBuffer = curve->ToData().pKeyArray.Get();
                    foundCurve = true;
                }
                curve->Reset();
            }
        }
        ToData().flag ^= Flag_CurveBaked;
    }

    return pBuffer;
}

void ResShapeAnim::Reset() NN_NOEXCEPT
{
    ToData().pBindModel.Set(NULL);
    uint16_t* pBindIndexArray = ToData().pBindIndexArray.Get();
    for (int idxAnim = 0, animCount = GetVertexShapeAnimCount(); idxAnim < animCount; ++idxAnim)
    {
        ResVertexShapeAnim* pVertexShapeAnim = GetVertexShapeAnim(idxAnim);
        ResKeyShapeAnimInfo* pInfo = pVertexShapeAnim->ToData().pKeyShapeAnimInfoArray.Get();
        for (int idxKeyShapeAnim = 0, keyShapeAnimCount = pVertexShapeAnim->GetKeyShapeAnimCount();
            idxKeyShapeAnim < keyShapeAnimCount; ++idxKeyShapeAnim, ++pInfo)
        {
            pInfo->subbindIndex = -1;
        }
        pBindIndexArray[idxAnim] = AnimFlag_NotBound;
    }
    ResetCurve();
}

}} // namespace nn::g3d

NN_PRAGMA_POP_WARNINGS
