﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/g3d/g3d_ResShader.h>
#include <nn/g3d/g3d_ResMaterial.h>
#include <nn/g3d/detail/g3d_Flag.h>
#include <nn/gfx/gfx_Variation-api.gl.h>

NN_PRAGMA_PUSH_WARNINGS
NN_DISABLE_WARNING_SHADOW

namespace nn { namespace g3d {

namespace {

void UpdateShaderProgramCallback(nn::gfx::Device* pDevice, ResShadingModel* pShaderModel, int programIndex)
{
    ResShaderProgram* pShaderProgram = pShaderModel->GetShaderProgram(programIndex);

    // シェーダーを初期化
    ResShaderArchive* pShaderArchive = pShaderModel->ToData().pShaderArchive.Get();
    pShaderProgram->Initialize(pDevice, !pShaderArchive->IsBinaryAvailable());

    // GL の場合はサンプラーとユニフォームブロックのロケーションテーブルを更新
    const nn::gfx::ResShaderFile* pResShaderFile = pShaderModel->GetGfxResShaderFile();
    const nn::gfx::ResShaderContainer* pResShaderContainer = pResShaderFile->GetShaderContainer();
    if (pResShaderContainer->ToData().targetApiType == nn::gfx::ApiTypeGl::value)
    {
        pShaderProgram->UpdateTable();
    }

    // 下のフラグ操作とこれより前の処理がアウトオブオーダーで実行されることはないと思うが、念のためメモリフェンスを入れておく。
    nn::os::FenceMemoryStoreStore();

    pShaderProgram->ToData().flag &= ~ResShaderProgram::Flag_UpdateRequired;
}

// ShaderKey 二分探索用のイテレータと比較関数
class KeyIter : public std::iterator<std::random_access_iterator_tag, const Bit32*, std::ptrdiff_t, const Bit32* const*, const Bit32* const&>
{
public:
    KeyIter(const Bit32* pKey, size_t length, size_t stride)
        : pKey(pKey), length(length), stride(stride)
    {
    }

    reference operator*() const { return pKey; }
    KeyIter& operator++() { pKey += stride; return *this; }
    KeyIter& operator+=(difference_type diff) { pKey += stride * diff; return *this; }
    KeyIter operator+(difference_type diff) const { return KeyIter(*this) += diff; }
    difference_type operator-(const KeyIter& rhs) const { return (pKey - rhs.pKey) / stride; }

    bool operator==(const KeyIter& rhs) const { return pKey == rhs.pKey; }
#if defined( _MSC_VER ) // VC 実装では使われる。
    bool operator!=(const KeyIter& rhs) const { return pKey != rhs.pKey; }
    bool operator<(const KeyIter& rhs) const { return pKey < rhs.pKey; }
#endif

#if 0 // std::lower_bound() では使われない演算子。
    KeyIter& operator--() { pKey += stride; return *this; }
    KeyIter operator++(int) { KeyIter tmp(*this); ++(*this); return tmp; }
    KeyIter operator--(int) { KeyIter tmp(*this); --(*this); return tmp; }
    KeyIter& operator-=(difference_type diff) { pKey -= stride * diff; return *this; }
    KeyIter operator-(difference_type diff) const { return KeyIter(*this) -= diff; }
    bool operator>(const KeyIter& rhs) const { return pKey > rhs.pKey; }
    bool operator<=(const KeyIter& rhs) const { return pKey <= rhs.pKey; }
    bool operator>=(const KeyIter& rhs) const { return pKey >= rhs.pKey; }
#endif

private:
    const Bit32* pKey;
    size_t length NN_IS_UNUSED_MEMBER;
    size_t stride;
};

struct KeyCmp
{
    explicit KeyCmp(size_t length) : length(length)
    {
    }

    bool operator()(const Bit32* lhs, const Bit32* rhs) const
    {
        for (size_t word = 0; word < length; ++word, ++lhs, ++rhs)
        {
            if (*lhs != *rhs)
            {
                return *lhs < *rhs;
            }
        }
        return false;
    }

    size_t length;
};

} // anonymous namespace

//--------------------------------------------------------------------------------------------------

bool ResShaderProgram::IsBinaryAvailable(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    #if (NN_GFX_IS_TARGET_VK)
    bool result = InitializePerType(pDevice, nn::gfx::ShaderCodeType_Ir);
    #else
    bool result = InitializePerType(pDevice, nn::gfx::ShaderCodeType_Binary);
    #endif
    Cleanup(pDevice);
    return result;
}

void ResShaderProgram::Setup(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    NN_UNUSED(pDevice);
    // 遅延コンパイルのために更新フラグを立てる。
    ToData().flag |= Flag_UpdateRequired;
}

void ResShaderProgram::Cleanup(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    // シェーダーが初期化済みなら、破棄する
    if (flag & Flag_Initialized)
    {
        // すべてのタイプを破棄する
        nn::gfx::ResShaderVariation* pShaderVariation = pShader.Get();
        if (flag & Flag_Binary)
        {
            pShaderVariation->GetResShaderProgram(nn::gfx::ShaderCodeType_Binary)->Finalize(pDevice);
        }
        if (flag & Flag_Ir)
        {
            pShaderVariation->GetResShaderProgram(nn::gfx::ShaderCodeType_Ir)->Finalize(pDevice);
        }
        if (flag & Flag_Source)
        {
            pShaderVariation->GetResShaderProgram(nn::gfx::ShaderCodeType_Source)->Finalize(pDevice);
        }
    }
    flag = Flag_None;
}

bool ResShaderProgram::InitializePerType(nn::gfx::Device* pDevice, nn::gfx::ShaderCodeType type) NN_NOEXCEPT
{
    nn::gfx::ResShaderVariation* pShaderVariation = pShader.Get();
    nn::gfx::ResShaderProgram* pShaderProgram = pShaderVariation->GetResShaderProgram(type);
    if (pShaderProgram == NULL)
    {
        return false;
    }
    if (pShaderProgram->Initialize(pDevice) == nn::gfx::ShaderInitializeResult_Success)
    {
        switch (type)
        {
            case nn::gfx::ShaderCodeType_Binary:
                flag |= Flag_Binary;
                break;
            case nn::gfx::ShaderCodeType_Ir:
                flag |= Flag_Ir;
                break;
            case nn::gfx::ShaderCodeType_Source:
                flag |= Flag_Source;
                break;
            default:
                NN_G3D_ASSERT(false, "Unknown shader program type");
                break;
        }
        flag |= Flag_Initialized;
        return true;
    }
    else
    {
        return false;
    }
}

void ResShaderProgram::Initialize(nn::gfx::Device* pDevice, bool skipBinaryCheck) NN_NOEXCEPT
{
    bool result = false;

    if (!skipBinaryCheck)
    {
        // バイナリーが使えるか
        result = InitializePerType(pDevice, nn::gfx::ShaderCodeType_Binary);
    }

    if (result == false)
    {
        // 中間言語が使えるか
        result = InitializePerType(pDevice, nn::gfx::ShaderCodeType_Ir);
        if (result == false)
        {
            // ソースが使えるか
            result = InitializePerType(pDevice, nn::gfx::ShaderCodeType_Source);
            NN_SDK_ASSERT(result == true, "Can't initialize the shader program");
        }
    }
}

void ResShaderProgram::UpdateTable() NN_NOEXCEPT
{
    ResShadingModel* pResShadingModel = pShadingModel.Get();

    nn::gfx::ShaderCodeType type = nn::gfx::ShaderCodeType_End;
    if (flag & Flag_Binary)
    {
        type = nn::gfx::ShaderCodeType_Binary;
    }
    else if (flag & Flag_Ir)
    {
        type = nn::gfx::ShaderCodeType_Ir;
    }
    else if (flag & Flag_Source)
    {
        type = nn::gfx::ShaderCodeType_Source;
    }

    NN_SDK_ASSERT(type != nn::gfx::ShaderCodeType_End);

    const nn::gfx::ResShaderVariation* pShaderVariation =pShader.Get();
    const nn::gfx::ResShaderProgram* pShaderProgram = pShaderVariation->GetResShaderProgram(type);
    const nn::gfx::Shader* pShader = pShaderProgram->GetShader();

    // シンボルテーブル取得
    const ResShaderInfo* pShaderInfo = pResShadingModel->GetShaderInfo();
    // サンプラーロケーションテーブルの書き換え
    int32_t* pSamplerLocation = pSamplerTable.Get();
    const nn::util::BinPtrToString* pSamplerSymbol = pShaderInfo->pSamplerTable.Get();
    for (int idxSampler = 0, samplerCount = pResShadingModel->GetSamplerCount();
        idxSampler < samplerCount; ++idxSampler)
    {
        for (int stage = 0; stage < Stage_End; ++stage)
        {
            int32_t location = -1;
            const nn::util::BinString* pName = pSamplerSymbol->Get();
            if (pName != NULL)
            {
                nn::gfx::ShaderStage gfxStage = ConvertToGfxStage(static_cast<Stage>(stage));
                location = pShader->GetInterfaceSlot(gfxStage, nn::gfx::ShaderInterfaceType_Sampler, pName->GetData());
            }
            ++pSamplerSymbol;

            pSamplerLocation[idxSampler * Stage_End + stage] = location;
        }
    }

    // ブロックロケーションテーブルの書き換え
    int32_t* pBlockLocation = pUniformBlockTable.Get();
    const nn::util::BinPtrToString* pUniformBlockSymbol = pShaderInfo->pUniformBlockTable.Get();
    for (int idxBlock = 0, blockCount = pResShadingModel->GetUniformBlockCount();
        idxBlock < blockCount; ++idxBlock)
    {
        for (int stage = 0; stage < Stage_End; ++stage)
        {
            int32_t location = -1;
            const nn::util::BinString* pName = pUniformBlockSymbol->Get();
            if (pName != NULL)
            {
                nn::gfx::ShaderStage gfxStage = ConvertToGfxStage(static_cast<Stage>(stage));
                location = pShader->GetInterfaceSlot(gfxStage, nn::gfx::ShaderInterfaceType_ConstantBuffer, pName->GetData());
            }
            ++pUniformBlockSymbol;

            pBlockLocation[idxBlock * Stage_End + stage] = location;
        }
    }

    // シェーダーストレージブロックロケーションテーブルの書き換え
    int32_t* pShaderStorageBlockLocation = pShaderStorageBlockTable.Get();
    const nn::util::BinPtrToString* pShaderStorageBlockSymbol = pShaderInfo->pShaderStorageBlockTable.Get();
    for (int idxBlock = 0, blockCount = pResShadingModel->GetShaderStorageBlockCount();
        idxBlock < blockCount; ++idxBlock)
    {
        for (int stage = 0; stage < Stage_End; ++stage)
        {
            int32_t location = -1;
            const nn::util::BinString* pName = pShaderStorageBlockSymbol->Get();
            if (pName != NULL)
            {
                nn::gfx::ShaderStage gfxStage = ConvertToGfxStage(static_cast<Stage>(stage));
                location = pShader->GetInterfaceSlot(gfxStage, nn::gfx::ShaderInterfaceType_UnorderedAccessBuffer, pName->GetData());
            }
            ++pShaderStorageBlockSymbol;

            pShaderStorageBlockLocation[idxBlock * Stage_End + stage] = location;
        }
    }
}

void ResShaderProgram::Update(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    // 遅延初期化が必要なら、初期化する
    if (ToData().flag & Flag_UpdateRequired)
    {
        ResShadingModel* pShadingModel = ToData().pShadingModel.Get();
        nn::os::MutexType* pMutexType = pShadingModel->ToData().pMutexTypePtr.Get();
        if (pMutexType)
        {
            nn::os::LockMutex(pMutexType);
            // ロック後、再度状態が変わっていないかチェックし、変わってなければ初期化
            if (ToData().flag & Flag_UpdateRequired)
            {
                int idxProgram = pShadingModel->GetProgramIndex(this);
                pShadingModel->UpdateProgram(pDevice, idxProgram);
            }
            nn::os::UnlockMutex(pMutexType);
        }
        else
        {
            int idxProgram = pShadingModel->GetProgramIndex(this);
            pShadingModel->UpdateProgram(pDevice, idxProgram);
        }
    }
}

void ResShaderProgram::Load(nn::gfx::CommandBuffer* pCommandBuffer) const NN_NOEXCEPT
{
    nn::gfx::ShaderCodeType type = nn::gfx::ShaderCodeType_End;
    if (flag & Flag_Binary)
    {
        type = nn::gfx::ShaderCodeType_Binary;
    }
    else if (flag & Flag_Ir)
    {
        type = nn::gfx::ShaderCodeType_Ir;
    }
    else if (flag & Flag_Source)
    {
        type = nn::gfx::ShaderCodeType_Source;
    }
    else
    {
        NN_SDK_ASSERT(false, "Can't find appropriate shader type");
        return;
    }

    const nn::gfx::ResShaderVariation* pShaderVariation = pShader.Get();
    pCommandBuffer->SetShader(pShaderVariation->GetResShaderProgram(type)->GetShader(), nn::gfx::ShaderStageBit_All);
}

const nn::gfx::Shader* ResShaderProgram::GetShader() const NN_NOEXCEPT
{
    nn::gfx::ShaderCodeType type = nn::gfx::ShaderCodeType_End;
    if (flag & Flag_Binary)
    {
        type = nn::gfx::ShaderCodeType_Binary;
    }
    else if (flag & Flag_Ir)
    {
        type = nn::gfx::ShaderCodeType_Ir;
    }
    else if (flag & Flag_Source)
    {
        type = nn::gfx::ShaderCodeType_Source;
    }
    else
    {
        return NULL;
    }

    const nn::gfx::ResShaderVariation* pShaderVariation = pShader.Get();
    return pShaderVariation->GetResShaderProgram(type)->GetShader();
}

//--------------------------------------------------------------------------------------------------

void ResShaderOption::WriteStaticKey(Bit32* pStaticKey, int choiceIndex) const NN_NOEXCEPT
{
    NN_G3D_REQUIRES(IsStatic() == true, GetName());
    NN_G3D_REQUIRES(pStaticKey != NULL, GetName());
    NN_G3D_REQUIRES_RANGE(choiceIndex, 0, GetChoiceCount(), GetName());
    Bit32& key32 = pStaticKey[ToData().bit32Index];
    key32 = (key32 & ~ToData().bit32Mask) | static_cast<Bit32>(choiceIndex << ToData().bit32Shift);
}

int ResShaderOption::ReadStaticKey(const Bit32* pStaticKey) const NN_NOEXCEPT
{
    NN_G3D_REQUIRES(IsStatic() == true, GetName());
    NN_G3D_REQUIRES(pStaticKey != NULL, GetName());
    Bit32 key32 = pStaticKey[ToData().bit32Index];
    return static_cast<int>((key32 & ToData().bit32Mask) >> ToData().bit32Shift);
}

void ResShaderOption::WriteDynamicKey(Bit32* pDynamicKey, int choiceIndex) const NN_NOEXCEPT
{
    NN_G3D_REQUIRES(IsDynamic() == true, GetName());
    NN_G3D_REQUIRES(pDynamicKey != NULL, GetName());
    NN_G3D_REQUIRES_RANGE(choiceIndex, 0, GetChoiceCount(), GetName());
    Bit32& key32 = pDynamicKey[ToData().bit32Index - ToData().keyOffset];
    key32 = (key32 & ~ToData().bit32Mask) | static_cast<Bit32>(choiceIndex << ToData().bit32Shift);
}

int ResShaderOption::ReadDynamicKey(const Bit32* pDynamicKey) const NN_NOEXCEPT
{
    NN_G3D_REQUIRES(IsDynamic() == true, GetName());
    NN_G3D_REQUIRES(pDynamicKey != NULL, GetName());
    Bit32 key32 = pDynamicKey[ToData().bit32Index - ToData().keyOffset];
    return static_cast<int>((key32 & ToData().bit32Mask) >> ToData().bit32Shift);
}

//--------------------------------------------------------------------------------------------------

void ResShadingModel::Relocate() NN_NOEXCEPT
{
    // リロケーション
    nn::gfx::ResShaderFile* pResShaderFile = pShaderFile.Get();
    nn::util::BinaryFileHeader* pFileHeader = pResShaderFile->GetBinaryFileHeader();
    if (!pFileHeader->IsRelocated())
    {
        pFileHeader->GetRelocationTable()->Relocate();
    }

    return;
}

void ResShadingModel::Unrelocate() NN_NOEXCEPT
{
    // アンリロケーション
    nn::gfx::ResShaderFile* pResShaderFile = pShaderFile.Get();
    nn::util::BinaryFileHeader* pFileHeader = pResShaderFile->GetBinaryFileHeader();
    if (pFileHeader->IsRelocated())
    {
        pFileHeader->GetRelocationTable()->Unrelocate();
    }

    return;
}

bool ResShadingModel::IsBinaryAvailable(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    bool result = false;

    if (GetShaderProgramCount() > 0)
    {
        nn::gfx::ResShaderFile* pResShaderFile = pShaderFile.Get();
        nn::gfx::ResShaderContainer* pResShaderContainer = pResShaderFile->GetShaderContainer();
        pResShaderContainer->Initialize(pDevice);
        result = GetShaderProgram(0)->IsBinaryAvailable(pDevice);
        pResShaderContainer->Finalize(pDevice);
    }

    return result;
}

void ResShadingModel::Setup(nn::gfx::Device* pDevice, nn::os::MutexType* pMutexType) NN_NOEXCEPT
{
    nn::gfx::ResShaderFile* pResShaderFile = pShaderFile.Get();
    nn::gfx::ResShaderContainer* pResShaderContainer = pResShaderFile->GetShaderContainer();
    pResShaderContainer->Initialize(pDevice);

    for (int idxProgram = 0, programCount = GetShaderProgramCount(); idxProgram < programCount; ++idxProgram)
    {
        ResShaderProgram* pProgram = GetShaderProgram(idxProgram);
        pProgram->Setup(pDevice);
    }

    pMutexTypePtr.Set(pMutexType);
}

void ResShadingModel::Setup(nn::gfx::Device* pDevice, nn::gfx::MemoryPool* pMemoryPool, ptrdiff_t memoryPoolOffset, size_t memoryPoolSize, nn::os::MutexType* pMutexType) NN_NOEXCEPT
{
    nn::gfx::ResShaderFile* pResShaderFile = pShaderFile.Get();
    nn::gfx::ResShaderContainer* pResShaderContainer = pResShaderFile->GetShaderContainer();

    // シェーダーコンテナへのオフセットを追加する
    ptrdiff_t shaderContainerOffset = memoryPoolOffset;
    shaderContainerOffset += nn::util::BytePtr(this).Distance(pResShaderContainer);

    pResShaderContainer->Initialize(pDevice, pMemoryPool, shaderContainerOffset, memoryPoolSize);

    for (int idxProgram = 0, programCount = GetShaderProgramCount(); idxProgram < programCount; ++idxProgram)
    {
        ResShaderProgram* pProgram = GetShaderProgram(idxProgram);
        pProgram->Setup(pDevice);
    }

    pMutexTypePtr.Set(pMutexType);
}

void ResShadingModel::Cleanup(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    for (int idxProgram = 0, programCount = GetShaderProgramCount(); idxProgram < programCount; ++idxProgram)
    {
        ResShaderProgram* pProgram = GetShaderProgram(idxProgram);
        pProgram->Cleanup(pDevice);
    }

    nn::gfx::ResShaderFile* pResShaderFile = ToData().pShaderFile.Get();
    nn::gfx::ResShaderContainer* pResShaderContainer = pResShaderFile->GetShaderContainer();
    pResShaderContainer->Finalize(pDevice);

    pMutexTypePtr.Set(NULL);
}

void ResShadingModel::WriteDefaultStaticKey(Bit32* pStaticKey) const NN_NOEXCEPT
{
    int idxDefaultProgram = GetDefaultProgramIndex();
    if (idxDefaultProgram == ShaderLocationProgramNone)
    {
        // 全てのオプションがデフォルトのシェーダーが存在しない場合はオプションからキーを作ります。
        std::fill_n(pStaticKey, GetStaticKeyLength(), 0);
        for (int idxOption = 0, optionCount = GetStaticOptionCount();
            idxOption < optionCount; ++idxOption)
        {
            const ResShaderOption* pOption = GetStaticOption(idxOption);
            pOption->WriteStaticKey(pStaticKey, pOption->GetDefaultIndex());
        }
    }
    else
    {
        // 全てのオプションがデフォルトのシェーダーが存在する場合はキーをコピーします。
        const Bit32* pDefaultKey = GetStaticKey(idxDefaultProgram);
        std::copy(pDefaultKey, pDefaultKey + GetStaticKeyLength(), pStaticKey);
    }
}

void ResShadingModel::WriteDefaultDynamicKey(Bit32* pDynamicKey) const NN_NOEXCEPT
{
    int idxDefaultProgram = GetDefaultProgramIndex();
    if (idxDefaultProgram == ShaderLocationProgramNone)
    {
        // 全てのオプションがデフォルトのシェーダーが存在しない場合はオプションからキーを作ります。
        std::fill_n(pDynamicKey, GetDynamicKeyLength(), 0);
        for (int idxOption = 0, optionCount = GetDynamicOptionCount();
            idxOption < optionCount; ++idxOption)
        {
            const ResShaderOption* pOption = GetDynamicOption(idxOption);
            pOption->WriteDynamicKey(pDynamicKey, pOption->GetDefaultIndex());
        }
    }
    else
    {
        // 全てのオプションがデフォルトのシェーダーが存在する場合はキーをコピーします。
        const Bit32* pDefaultKey = GetDynamicKey(idxDefaultProgram);
        std::copy(pDefaultKey, pDefaultKey + GetDynamicKeyLength(), pDynamicKey);
    }
}

void ResShadingModel::WriteInvalidDynamicKey(Bit32* pDynamicKey) const NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pDynamicKey != NULL, GetName());
    // システムで予約されているビットがあるため全て1のキーは存在しません。
    std::fill_n(pDynamicKey, GetDynamicKeyLength(), Bit32(~0));
}

int ResShadingModel::FindProgramIndex(const Bit32* pKey) const NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pKey != NULL, GetName());

    int keyLength = GetKeyLength();
    KeyIter iterBegin(ToData().pKeyTable.Get(), keyLength, keyLength);
    KeyIter iterEnd(iterBegin + GetShaderProgramCount());
    KeyIter iterFound = std::lower_bound(iterBegin, iterEnd, pKey, KeyCmp(keyLength));
    if (iterFound == iterEnd || KeyCmp(keyLength)(pKey, *iterFound))
    {
        return -1;
    }
    else
    {
        return static_cast<int>(std::distance(iterBegin, iterFound));
    }
}

int ResShadingModel::FindProgramIndex(const ShaderRange& range, const Bit32* pDynamicKey) const NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pDynamicKey != NULL || GetDynamicKeyLength() == 0, GetName());

    int keyStride = GetKeyLength();
    int keyOffset = GetStaticKeyLength();
    int keyLength = GetDynamicKeyLength();
    KeyIter iterBegin(range.pBegin, keyLength, keyStride);
    KeyIter iterEnd(range.pEnd, keyLength, keyStride);
    KeyIter iterFound = std::lower_bound(iterBegin, iterEnd, pDynamicKey, KeyCmp(keyLength));
    if (iterFound == iterEnd || KeyCmp(keyLength)(pDynamicKey, *iterFound))
    {
        return -1;
    }
    else
    {
        KeyIter iterBase(ToData().pKeyTable.Get() + keyOffset, keyLength, keyStride);
        return static_cast<int>(std::distance(iterBase, iterFound));
    }
}

bool ResShadingModel::FindProgramRange(ShaderRange* pRange, const Bit32* pStaticKey) const NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pRange != NULL, GetName());
    NN_G3D_REQUIRES(pStaticKey != NULL || GetStaticKeyLength() == 0, GetName());

    int keyStride = GetKeyLength();
    int keyLength = GetStaticKeyLength();
    KeyIter iterBegin(ToData().pKeyTable.Get(), keyLength, keyStride);
    KeyIter iterEnd(iterBegin + GetShaderProgramCount());
    std::pair<KeyIter, KeyIter> rangeFound =
        std::equal_range(iterBegin, iterEnd, pStaticKey, KeyCmp(keyLength));
    pRange->pBegin = *rangeFound.first + keyLength; // dynamic オプションのキーとして記録します。
    pRange->pEnd = *rangeFound.second + keyLength;
    return pRange->pBegin != pRange->pEnd;
}

void ResShadingModel::UpdateProgram(nn::gfx::Device* pDevice, int programIndex) NN_NOEXCEPT
{
    const ResShaderArchive* pArchive = ToData().pShaderArchive.Get();
    const UpdateProgramCallback pCallback = pArchive->GetUpdateProgramCallback();
    if (pCallback)
    {
        pCallback(pDevice, this, programIndex);
    }
}

const Bit32* ResShadingModel::GetKey(int programIndex) const NN_NOEXCEPT
{
    NN_G3D_REQUIRES_RANGE(programIndex, 0, GetShaderProgramCount(), GetName());
    const Bit32* pKeyTable = ToData().pKeyTable.Get();
    int keyStride = sizeof(Bit32) * GetKeyLength();
    const Bit32* pKey = AddOffset<Bit32>(pKeyTable, keyStride * programIndex);
    return pKey;
}

const Bit32* ResShadingModel::GetStaticKey(int programIndex) const NN_NOEXCEPT
{
    NN_G3D_REQUIRES_RANGE(programIndex, 0, GetShaderProgramCount(), GetName());
    const Bit32* pKeyTable = ToData().pKeyTable.Get();
    int keyStride = sizeof(Bit32) * GetKeyLength();
    const Bit32* pKey = AddOffset<Bit32>(pKeyTable, keyStride * programIndex);
    return pKey;
}

const Bit32* ResShadingModel::GetDynamicKey(int programIndex) const NN_NOEXCEPT
{
    NN_G3D_REQUIRES_RANGE(programIndex, 0, GetShaderProgramCount(), GetName());
    const Bit32* pKeyTable = ToData().pKeyTable.Get() + GetStaticKeyLength();
    int keyStride = sizeof(Bit32) * GetKeyLength();
    const Bit32* pKey = AddOffset<Bit32>(pKeyTable, keyStride * programIndex);
    return pKey;
}

int ResShadingModel::PrintKeyTo(char* pStr, size_t strLength, const Bit32* pKey, int keyLength) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(pStr != NULL || strLength == 0);
    NN_SDK_REQUIRES(pKey != NULL || keyLength == 0);

    const int lenPerBit32 = 9; // 8桁 + 区切り文字
    size_t strLengthToWrite = 0;
    if (keyLength > 0)
    {
        strLengthToWrite = keyLength * lenPerBit32 - 1; // 最後は区切り文字を書きません。
    }

    if (pStr == NULL)
    {
        return static_cast<int>(strLengthToWrite);
    }

    NN_SDK_ASSERT(strLength > 0); // 少なくとも終端文字分は必要です。

    if (strLength < strLengthToWrite + 1)
    {
        *pStr = '\0';
        return -1;
    }

    // 実際の書き込み処理を行います。

    if (strLengthToWrite == 0)
    {
        *pStr = '\0';
        return 0;
    }

    int idxKey = 0;
    do
    {
        sprintf(pStr, "%08X_", pKey[idxKey]); // 書き込み先サイズはチェック済みです。
        pStr += lenPerBit32;
    } while (++idxKey < keyLength);

    *(--pStr) = '\0'; // 最後の '_' を終端文字に置き換えます。
    return static_cast<int>(strLengthToWrite);
}

int ResShadingModel::PrintStaticOptionTo(char* pStr, size_t strLength, const Bit32* pKey) const NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pStr != NULL || strLength == 0, GetName());
    NN_G3D_REQUIRES(pKey != NULL || GetStaticKeyLength() == 0, GetName());

    // デバッグ機能なので簡潔さのためにループを2周します。
    size_t strLengthToWrite = 0;
    for (int idxOption = 0, optionCount = GetStaticOptionCount(); idxOption < optionCount; ++idxOption)
    {
        const ResShaderOption* pOption = GetStaticOption(idxOption);
        const char* pOptionName = pOption->ToData().pName.Get()->GetData();
        strLengthToWrite += std::strlen(pOptionName);

        int choice = pOption->ReadStaticKey(pKey);
        const nn::util::ResDic* pDic = pOption->ToData().pChoiceDic.Get();
        const char* pChoiceName = pDic->GetKey(choice).data();
        strLengthToWrite += std::strlen(pChoiceName);

        strLengthToWrite += 2; // 区切り文字分です。(Option:Choice\t)
    }
    if (strLengthToWrite > 0)
    {
        --strLengthToWrite; // 最後は区切り文字を書きません。
    }

    if (pStr == NULL)
    {
        return static_cast<int>(strLengthToWrite);
    }

    NN_G3D_ASSERT(strLength > 0, GetName()); // 少なくとも終端文字分は必要です。

    if (strLength < strLengthToWrite + 1)
    {
        *pStr = '\0';
        return -1;
    }

    // 実際の書き込み処理を行います。

    if (strLengthToWrite == 0)
    {
        *pStr = '\0';
        return 0;
    }

    for (int idxOption = 0, optionCount = GetStaticOptionCount(); idxOption < optionCount; ++idxOption)
    {
        const ResShaderOption* pOption = GetStaticOption(idxOption);
        const char* pOptionName = pOption->ToData().pName.Get()->GetData();
        size_t length = std::strlen(pOptionName);
        memcpy(pStr, pOptionName, length);
        pStr += length;
        *pStr++ = ':';

        int choice = pOption->ReadStaticKey(pKey);
        const nn::util::ResDic* pDic = pOption->ToData().pChoiceDic.Get();
        const char* pChoiceName = pDic->GetKey(choice).data();
        length = std::strlen(pChoiceName);
        memcpy(pStr, pChoiceName, length);
        pStr += length;
        *pStr++ = '\t';
    }

    *(--pStr) = '\0'; // 最後の '\t' を終端文字に置き換えます。
    return static_cast<int>(strLengthToWrite);
}

int ResShadingModel::PrintDynamicOptionTo(char* pStr, size_t strLength, const Bit32* pKey) const NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pStr != NULL || strLength == 0, GetName());
    NN_G3D_REQUIRES(pKey != NULL || GetDynamicKeyLength() == 0, GetName());

    // デバッグ機能なので簡潔さのためにループを2周します。
    size_t strLengthToWrite = 0;
    for (int idxOption = 0, optionCount = GetDynamicOptionCount(); idxOption < optionCount; ++idxOption)
    {
        const ResShaderOption* pOption = GetDynamicOption(idxOption);
        const char* pOptionName = pOption->ToData().pName.Get()->GetData();
        strLengthToWrite += std::strlen(pOptionName);

        int choice = pOption->ReadDynamicKey(pKey);
        const nn::util::ResDic* pDic = pOption->ToData().pChoiceDic.Get();
        const char* pChoiceName = pDic->GetKey(choice).data();
        strLengthToWrite += std::strlen(pChoiceName);

        strLengthToWrite += 2; // 区切り文字分です。(Option:Choice\t)
    }
    if (strLengthToWrite > 0)
    {
        --strLengthToWrite; // 最後は区切り文字を書きません。
    }

    if (pStr == NULL)
    {
        return static_cast<int>(strLengthToWrite);
    }

    NN_G3D_ASSERT(strLength > 0, GetName()); // 少なくとも終端文字分は必要です。

    if (strLength < strLengthToWrite + 1)
    {
        *pStr = '\0';
        return -1;
    }

    // 実際の書き込み処理を行います。

    if (strLengthToWrite == 0)
    {
        *pStr = '\0';
        return 0;
    }

    for (int idxOption = 0, optionCount = GetDynamicOptionCount(); idxOption < optionCount; ++idxOption)
    {
        const ResShaderOption* pOption = GetDynamicOption(idxOption);
        const char* pOptionName = pOption->ToData().pName.Get()->GetData();
        size_t length = std::strlen(pOptionName);
        memcpy(pStr, pOptionName, length);
        pStr += length;
        *pStr++ = ':';

        int choice = pOption->ReadDynamicKey(pKey);
        const nn::util::ResDic* pDic = pOption->ToData().pChoiceDic.Get();
        const char* pChoiceName = pDic->GetKey(choice).data();
        length = std::strlen(pChoiceName);
        memcpy(pStr, pChoiceName, length);
        pStr += length;
        *pStr++ = '\t';
    }

    *(--pStr) = '\0'; // 最後の '\t' を終端文字に置き換えます。
    return static_cast<int>(strLengthToWrite);
}

//--------------------------------------------------------------------------------------------------

void ResShaderArchive::Setup(nn::gfx::Device* pDevice, void* pWorkMemory, size_t workMemorySize) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pDevice != NULL, GetName());
    NN_G3D_REQUIRES((pWorkMemory != NULL && workMemorySize >= GetWorkMemorySize()) || pWorkMemory == NULL, GetName());
    NN_UNUSED(workMemorySize);

    SetUpdateProgramCallback(UpdateShaderProgramCallback);

    int shadingModelCount = GetShadingModelCount();

    // バイナリー使用可能フラグがセットされていない場合、バイナリーが使用可能かチェック
    if ((flag & Flag_Binary) && !(flag & Flag_BinaryAvailable))
    {
        if (shadingModelCount > 0)
        {
            if (GetShadingModel(0)->IsBinaryAvailable(pDevice) == true)
            {
                flag |= Flag_BinaryAvailable;
            }
        }
    }

    nn::util::BytePtr bytePtr(pWorkMemory);
    for (int idxShadingModel = 0; idxShadingModel < shadingModelCount; ++idxShadingModel)
    {
        ResShadingModel* pShadingModel = GetShadingModel(idxShadingModel);
        if (pWorkMemory)
        {
            nn::os::MutexType* pMutexType = bytePtr.Get<nn::os::MutexType>();
            nn::os::InitializeMutex(pMutexType, false, 0);
            bytePtr.Advance(sizeof(nn::os::MutexType));
            pShadingModel->Setup(pDevice, pMutexType);

        }
        else
        {
            pShadingModel->Setup(pDevice);
        }
    }

    pWorkMemoryPtr.Set(pWorkMemory);
    flag |= Flag_Initialized;
}

void ResShaderArchive::Setup(nn::gfx::Device* pDevice, nn::gfx::MemoryPool* pMemoryPool, ptrdiff_t memoryPoolOffset, size_t memoryPoolSize, void* pWorkMemory, size_t workMemorySize) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pDevice != NULL, GetName());
    NN_G3D_REQUIRES((pWorkMemory != NULL && workMemorySize >= GetWorkMemorySize()) || pWorkMemory == NULL, GetName());
    NN_UNUSED(workMemorySize);

    SetUpdateProgramCallback(UpdateShaderProgramCallback);

    int shadingModelCount = GetShadingModelCount();

    // バイナリー使用可能フラグがセットされていない場合、バイナリーが使用可能かチェック
    if ((flag & Flag_Binary) && !(flag & Flag_BinaryAvailable))
    {
        if (shadingModelCount > 0)
        {
            if (GetShadingModel(0)->IsBinaryAvailable(pDevice) == true)
            {
                flag |= Flag_BinaryAvailable;
            }
        }
    }

    nn::util::BytePtr bytePtr(pWorkMemory);
    for (int idxShadingModel = 0; idxShadingModel < shadingModelCount; ++idxShadingModel)
    {
        ResShadingModel* pShadingModel = GetShadingModel(idxShadingModel);

        // シェーディングモデルへのオフセットを追加する
        ptrdiff_t shadingModelOffset = memoryPoolOffset;
        shadingModelOffset += nn::util::BytePtr(this).Distance(pShadingModel);

        if (pWorkMemory)
        {
            nn::os::MutexType* pMutexType = bytePtr.Get<nn::os::MutexType>();
            nn::os::InitializeMutex(pMutexType, false, 0);
            bytePtr.Advance(sizeof(nn::os::MutexType));
            pShadingModel->Setup(pDevice, pMemoryPool, shadingModelOffset, memoryPoolSize, pMutexType);
        }
        else
        {
            pShadingModel->Setup(pDevice, pMemoryPool, shadingModelOffset, memoryPoolSize);
        }
    }

    pWorkMemoryPtr.Set(pWorkMemory);
    flag |= Flag_Initialized;
}

void ResShaderArchive::Cleanup(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    nn::util::BytePtr bytePtr(pWorkMemoryPtr.Get());
    for (int idxShadingModel = 0, shadingModelCount = GetShadingModelCount(); idxShadingModel < shadingModelCount; ++idxShadingModel)
    {
        ResShadingModel* pShadingModel = GetShadingModel(idxShadingModel);
        pShadingModel->Cleanup(pDevice);
        if (pWorkMemoryPtr.Get())
        {
            nn::os::MutexType* pMutexType = bytePtr.Get<nn::os::MutexType>();
            nn::os::FinalizeMutex(pMutexType);
            bytePtr.Advance(sizeof(nn::os::MutexType));
        }
    }

    pWorkMemoryPtr.Set(NULL);
    flag &= ~Flag_Initialized;
}

//--------------------------------------------------------------------------------------------------
const int64_t ResShaderFile::Signature = NN_UTIL_CREATE_SIGNATURE_8('F', 'S', 'H', 'A', ' ', ' ', ' ', ' ');

void ResShaderFile::Relocate() NN_NOEXCEPT
{
    // リロケーション
    if( !this->fileHeader.IsRelocated() )
    {
        this->fileHeader.GetRelocationTable()->Relocate();
    }

    ResShaderArchive* pResShaderArchive = GetResShaderArchive();
    for (int idxShadingModel = 0, shadingModelCount = pResShaderArchive->GetShadingModelCount(); idxShadingModel < shadingModelCount; ++idxShadingModel)
    {
        ResShadingModel* pResShadingModel = pResShaderArchive->GetShadingModel(idxShadingModel);
        pResShadingModel->Relocate();
    }

    return;
}

void ResShaderFile::Unrelocate() NN_NOEXCEPT
{
    // アンリロケーション
    ResShaderArchive* pResShaderArchive = GetResShaderArchive();
    for (int idxShadingModel = 0, shadingModelCount = pResShaderArchive->GetShadingModelCount(); idxShadingModel < shadingModelCount; ++idxShadingModel)
    {
        ResShadingModel* pResShadingModel = pResShaderArchive->GetShadingModel(idxShadingModel);
        pResShadingModel->Unrelocate();
    }
    if( this->fileHeader.IsRelocated() )
    {
        this->fileHeader.GetRelocationTable()->Unrelocate();
    }

    return;
}

ResShaderFile* ResShaderFile::ResCast(void* ptr) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(ptr != NULL);
    ResShaderFileData* pData = static_cast<ResShaderFileData*>(ptr);
    ResShaderFile* pFile = static_cast<ResShaderFile*>(pData);

    NN_SDK_ASSERT(pFile->fileHeader.IsAlignmentValid());

    // リロケーション
    pFile->Relocate();

    // エンディアン変換
    if (pFile->fileHeader.IsEndianReverse())
    {
        NN_G3D_NOT_IMPLEMENTED();
        // エンディアンを変換する。
        //Endian<true>::Swap(pData);
    }

    return pFile;
}

void ResShaderFile::Setup(nn::gfx::Device* pDevice, nn::gfx::MemoryPool* pMemoryPool, ptrdiff_t memoryPoolOffset, size_t memoryPoolSize, void* pWorkMemory, size_t workMemorySize) NN_NOEXCEPT
{
    // メモリープールのサイズを確認
    NN_SDK_ASSERT(ToData().fileHeader.GetFileSize() < memoryPoolSize);

    // シェーダーアーカイブのオフセットを計算
    ptrdiff_t shaderArchiveOffset = memoryPoolOffset;
    shaderArchiveOffset += nn::util::BytePtr(this).Distance(ToData().pShaderArchive.Get());

    ToData().pShaderArchive.Get()->Setup(pDevice, pMemoryPool, shaderArchiveOffset, memoryPoolSize, pWorkMemory, workMemorySize);
}

bool ResShaderFile::IsValid(const void* ptr) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(ptr != NULL);
    const nn::util::BinaryFileHeader* pHeader = static_cast<const nn::util::BinaryFileHeader*>(ptr);

    return pHeader->IsValid(Signature,
        NN_G3D_SHADER_BINARY_VERSION_MAJOR,
        NN_G3D_SHADER_BINARY_VERSION_MINOR,
        NN_G3D_SHADER_BINARY_VERSION_MICRO);
}

}} // namespace nn::g3d

NN_PRAGMA_POP_WARNINGS
