﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/g3d/g3d_ResFile.h>
//#include <nn/g3d/g3d_ResUtility.h>
#include <nn/gfx/util/gfx_ObjectDebugLabel.h>

//--------------------------------------------------------------------------------------------------
// ミドルウェア情報
#include <nn/nn_Middleware.h>
#include <nn/nn_Version.h>

#define NN_DETAIL_XX_MACRO_TO_STRING(x) NN_DETAIL_XX_MACRO_TO_STRING_(x)
#define NN_DETAIL_XX_MACRO_TO_STRING_(x) #x

#define NW_G3D_MIDDLEWARE_SYMBOL(buildOption) "NintendoWare_G3d-" NN_DETAIL_XX_MACRO_TO_STRING(NN_NX_ADDON_VERSION_MAJOR) "_" \
NN_DETAIL_XX_MACRO_TO_STRING(NN_NX_ADDON_VERSION_MINOR) "_" NN_DETAIL_XX_MACRO_TO_STRING(NN_NX_ADDON_VERSION_MICRO) "-" #buildOption

namespace {
#if defined(NN_SDK_BUILD_DEBUG)
    NN_DEFINE_MIDDLEWARE(g_MiddlewareInfo, "Nintendo", NW_G3D_MIDDLEWARE_SYMBOL(Debug));
#elif defined(NN_SDK_BUILD_DEVELOP)
    NN_DEFINE_MIDDLEWARE(g_MiddlewareInfo, "Nintendo", NW_G3D_MIDDLEWARE_SYMBOL(Develop));
#elif defined(NN_SDK_BUILD_RELEASE)
    NN_DEFINE_MIDDLEWARE(g_MiddlewareInfo, "Nintendo", NW_G3D_MIDDLEWARE_SYMBOL(Release));
#endif
}
//--------------------------------------------------------------------------------------------------

NN_PRAGMA_PUSH_WARNINGS
NN_DISABLE_WARNING_SHADOW

namespace nn { namespace g3d {

const int64_t ResFile::Signature = NN_UTIL_CREATE_SIGNATURE_8('F', 'R', 'E', 'S', ' ', ' ', ' ', ' ');

bool ResFile::IsValid(const void* ptr) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ptr);
    const nn::util::BinaryFileHeader* pHeader = static_cast<const nn::util::BinaryFileHeader*>(ptr);

    return pHeader->IsValid(Signature,
                            NN_G3D_MODEL_BINARY_VERSION_MAJOR,
                            NN_G3D_MODEL_BINARY_VERSION_MINOR,
                            NN_G3D_MODEL_BINARY_VERSION_MICRO);

}

void ResFile::Relocate() NN_NOEXCEPT
{
    // リロケーション
    if( !this->fileHeader.IsRelocated() )
    {
        this->fileHeader.GetRelocationTable()->Relocate();
    }
    return;
}

void ResFile::Unrelocate() NN_NOEXCEPT
{
    // アンリロケーション
    if( this->fileHeader.IsRelocated() )
    {
        this->fileHeader.GetRelocationTable()->Unrelocate();
    }
    return;
}

ResFile* ResFile::ResCast(void* ptr) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ptr);

    ResFileData* pData = static_cast<ResFileData*>(ptr);
    ResFile* pFile = static_cast<ResFile*>(pData);

    NN_SDK_ASSERT(pFile->fileHeader.IsAlignmentValid());

    // リロケーション
    pFile->Relocate();

    // エンディアン変換
    if (pData->fileHeader.IsEndianReverse())
    {
        NN_G3D_NOT_IMPLEMENTED();
        // エンディアンを変換する。
        //Endian<true>::Swap(pData);
    }

    return pFile;
}

BindResult ResFile::BindTexture(TextureBindCallback pCallback, void* pUserData)
{
    NN_G3D_REQUIRES(pCallback != NULL, NN_G3D_RES_GET_NAME(this, GetName()));

    BindResult result;

    for (int idxModel = 0, modelCount = GetModelCount(); idxModel < modelCount; ++idxModel)
    {
        ResModel* pModel = GetModel(idxModel);
        result |= pModel->BindTexture(pCallback, pUserData);
    }

    for (int idxMatAnim = 0, materialAnimCount = GetMaterialAnimCount(); idxMatAnim < materialAnimCount; ++idxMatAnim)
    {
        ResMaterialAnim* pMatAnim = GetMaterialAnim(idxMatAnim);
        result |= pMatAnim->BindTexture(pCallback, pUserData);
    }

    return result;
}

void ResFile::ReleaseTexture() NN_NOEXCEPT
{
    for (int idxModel = 0, modelCount = GetModelCount(); idxModel < modelCount; ++idxModel)
    {
        ResModel* pModel = GetModel(idxModel);
        pModel->ReleaseTexture();
    }

    for (int idxMatAnim = 0, materialAnimCount = GetMaterialAnimCount(); idxMatAnim < materialAnimCount; ++idxMatAnim)
    {
        ResMaterialAnim* pMatAnim = GetMaterialAnim(idxMatAnim);
        pMatAnim->ReleaseTexture();
    }
}

void ResFile::Setup(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    NN_USING_MIDDLEWARE(g_MiddlewareInfo);

    // アニメーションだけ持つリソースの場合はメモリプールが存在しない
    if (pBufferMemoryPool.Get())
    {
        // メモリプールの初期化
        nn::gfx::MemoryPool::InfoType* pInfo = pBufferMemoryPoolInfo.Get();
        if (pInfo)
        {
            nn::gfx::MemoryPool* pMemoryPool = new (pBufferMemoryPool.Get()) nn::gfx::MemoryPool();
            pMemoryPool->Initialize(pDevice, *pInfo);
            nn::gfx::util::SetMemoryPoolDebugLabel(pMemoryPool, "g3d");
        }
    }

    int modelCount = this->GetModelCount();
    for (int modelIndex = 0; modelIndex < modelCount; ++modelIndex)
    {
        this->GetModel(modelIndex)->Setup(pDevice);
    }
}

void ResFile::Setup(nn::gfx::Device* pDevice, nn::gfx::MemoryPool* pMemoryPool, ptrdiff_t memoryPoolOffset, size_t memoryPoolSize) NN_NOEXCEPT
{
    // メモリプールのサイズを確認
    NN_SDK_ASSERT(fileHeader.GetFileSize() < memoryPoolSize);
    NN_UNUSED(memoryPoolSize);

    // リソースの先頭から頂点バッファー・インデックスバッファーへのオフセットを追加する
    ptrdiff_t bufferOffset = memoryPoolOffset;

    // アニメーションだけ持つリソースの場合はメモリプールが存在しない
    if (pBufferMemoryPool.Get())
    {
        nn::gfx::MemoryPool::InfoType* pInfo = pBufferMemoryPoolInfo.Get();
        if (pInfo)
        {
            bufferOffset += nn::util::BytePtr(this).Distance(pInfo->GetPoolMemory());
        }
    }

    int modelCount = this->GetModelCount();
    for (int modelIndex = 0; modelIndex < modelCount; ++modelIndex)
    {
        this->GetModel(modelIndex)->Setup(pDevice, pMemoryPool, bufferOffset);
    }
}

void ResFile::Cleanup(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    int modelCount = this->GetModelCount();
    for (int modelIndex = 0; modelIndex < modelCount; ++modelIndex)
    {
        this->GetModel(modelIndex)->Cleanup(pDevice);
    }

    // メモリプールの破棄
    if( pBufferMemoryPoolInfo.Get() )
    {
        nn::gfx::MemoryPool* pMemoryPool = pBufferMemoryPool.Get();
        if (nn::gfx::IsInitialized(*pMemoryPool) == true)
        {
            pMemoryPool->Finalize(pDevice);
            pMemoryPool->nn::gfx::MemoryPool::~TMemoryPool();
        }
    }
}

void ResFile::Reset() NN_NOEXCEPT
{
    int modelCount = this->GetModelCount();
    for (int modelIndex = 0; modelIndex < modelCount; ++modelIndex)
    {
        this->GetModel(modelIndex)->Reset();
    }

    int skeletalAnimCount = this->GetSkeletalAnimCount();
    for (int skeletalAnimIndex = 0; skeletalAnimIndex < skeletalAnimCount; ++skeletalAnimIndex)
    {
        this->GetSkeletalAnim(skeletalAnimIndex)->Reset();
    }

    int materialAnimCount = this->GetMaterialAnimCount();
    for (int materialAnimIndex = 0; materialAnimIndex < materialAnimCount; ++materialAnimIndex)
    {
        this->GetMaterialAnim(materialAnimIndex)->Reset();
    }

    int shapeAnimCount = this->GetShapeAnimCount();
    for (int shapeAnimIndex = 0; shapeAnimIndex < shapeAnimCount; ++shapeAnimIndex)
    {
        this->GetShapeAnim(shapeAnimIndex)->Reset();
    }

    int sceneAnimCount = this->GetSceneAnimCount();
    for (int sceneAnimIndex = 0; sceneAnimIndex < sceneAnimCount; ++sceneAnimIndex)
    {
        this->GetSceneAnim(sceneAnimIndex)->Reset();
    }

    ToData().pUserPtr.Set(NULL);
}

}} // namespace nn::g3d::res

NN_PRAGMA_POP_WARNINGS
