﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/g3d/g3d_ModelObj.h>

namespace nn { namespace g3d {

void ModelObj::InitializeArgument::CalculateMemorySize() NN_NOEXCEPT
{
    NN_G3D_REQUIRES(GetViewCount() > 0,              NN_G3D_RES_GET_NAME(GetResource(), GetName()));
    NN_G3D_REQUIRES(GetSkeletonBufferingCount() > 0, NN_G3D_RES_GET_NAME(GetResource(), GetName()));
    NN_G3D_REQUIRES(GetShapeBufferingCount() > 0,    NN_G3D_RES_GET_NAME(GetResource(), GetName()));
    NN_G3D_REQUIRES(GetMaterialBufferingCount() > 0, NN_G3D_RES_GET_NAME(GetResource(), GetName()));

    const ResModel* pRes = GetResource();

    // Skeleton のサイズ計算
    size_t       sizeSkeleton = 0;
    const ResSkeleton* pResSkeleton = pRes->GetSkeleton();
    SkeletonObj* pSkeleton    = GetSkeleton();
    if (pSkeleton)
    {
        // SkeletonObj を渡された場合はそれを使用する。
        NN_SDK_ASSERT(pResSkeleton == pSkeleton->GetResource(), "The skeleton resource is not same. Name : %s\n",
                                                                NN_G3D_RES_GET_NAME(GetResource(), GetName()));
    }
    else
    {
        // SkeletonObj も構築する。
        SkeletonObj::Builder builder(pResSkeleton);
        builder.BufferingCount(GetSkeletonBufferingCount());
        builder.CalculateMemorySize();
        size_t size = builder.GetWorkMemorySize();
        sizeSkeleton = nn::util::align_up(size, SkeletonObj::Alignment_Buffer);
    }

    int shapeCount    = pRes->GetShapeCount();
    int materialCount = pRes->GetMaterialCount();
    int boneCount     = pResSkeleton->GetBoneCount();
    int viewCount     = GetViewCount();

    // Shape を辿ってサイズ計算
    size_t sizeShape = 0;
    for (int idxShape = 0; idxShape < shapeCount; ++idxShape)
    {
        const ResShape* pResShape = pRes->GetShape(idxShape);
        ShapeObj::Builder builder(pResShape);
        builder.BufferingCount(GetShapeBufferingCount());
        const ResBone* pResBone = pResSkeleton->GetBone(pResShape->GetBoneIndex());
        if (pResBone->GetBillboardMode() == ResBone::Flag_BillboardNone)
        {
            builder.ViewIndependent();
        }
        else
        {
            builder.ViewDependent();
        }
        builder.ViewCount(viewCount);
        if (IsBoundingEnabled())
        {
            builder.SetBoundingEnabled();
        }
        else
        {
            builder.SetBoundingDisabled();
        }
        builder.UserAreaSize(GetShapeUserAreaSize());
        builder.CalculateMemorySize();
        size_t size = builder.GetWorkMemorySize();
        sizeShape += nn::util::align_up(size, ShapeObj::Alignment_Buffer);
    }

    // Material を辿ってサイズ計算
    size_t sizeMaterial = 0;
    for (int idxMaterial = 0; idxMaterial < materialCount; ++idxMaterial)
    {
        const ResMaterial* pResMaterial = pRes->GetMaterial(idxMaterial);
        MaterialObj::Builder builder(pResMaterial);
        builder.BufferingCount(GetMaterialBufferingCount());
        builder.CalculateMemorySize();
        size_t size = builder.GetWorkMemorySize();
        sizeMaterial += nn::util::align_up(size, MaterialObj::Alignment_Buffer);
    }

    for (int blockIndex = 0; blockIndex < MemoryBlockIndex_End; ++blockIndex)
    {
        m_MemoryBlock[blockIndex].Initialize();
    }

    // サイズ計算
    m_MemoryBlock[MemoryBlockIndex_SkeletonBuffer].SetAlignment(SkeletonObj::Alignment_Buffer);
    m_MemoryBlock[MemoryBlockIndex_SkeletonBuffer].SetSize(sizeSkeleton);
    m_MemoryBlock[MemoryBlockIndex_ShapeBuffer].SetAlignment(ShapeObj::Alignment_Buffer);
    m_MemoryBlock[MemoryBlockIndex_ShapeBuffer].SetSize(sizeShape);
    m_MemoryBlock[MemoryBlockIndex_MaterialBuffer].SetAlignment(MaterialObj::Alignment_Buffer);
    m_MemoryBlock[MemoryBlockIndex_MaterialBuffer].SetSize(sizeMaterial);
    m_MemoryBlock[MemoryBlockIndex_Skeleton].SetSizeBy<SkeletonObj>(1, 1);
    m_MemoryBlock[MemoryBlockIndex_ShapeArray].SetSizeBy<ShapeObj>(1, shapeCount);
    m_MemoryBlock[MemoryBlockIndex_MaterialArray].SetSizeBy<MaterialObj>(1, materialCount);
    m_MemoryBlock[MemoryBlockIndex_BoneVisArray].SetSize(nn::util::align_up(boneCount, 32) >> 3); // 1ビジビリティ1ビット
    m_MemoryBlock[MemoryBlockIndex_MatVisArray].SetSize(nn::util::align_up(materialCount, 32) >> 3); // 1ビジビリティ1ビット
    m_MemoryBlock[MemoryBlockIndex_Bounding].SetAlignment(MatrixVectorAlignment);
    if (IsBoundingEnabled() == true)
    {
        // LOD数分確保する
        m_MemoryBlock[MemoryBlockIndex_Bounding].SetSizeBy<Sphere>(1, m_LodCount);
    }
    else
    {
        m_MemoryBlock[MemoryBlockIndex_Bounding].SetSize(0);
    }

    m_WorkMemory.Initialize();
    for (int blockIndex = 0; blockIndex < MemoryBlockIndex_End; ++blockIndex)
    {
        m_WorkMemory.Append(&m_MemoryBlock[blockIndex]);
    }
} // NOLINT

//--------------------------------------------------------------------------------------------------

bool ModelObj::Initialize(const InitializeArgument& arg, void* pBuffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pBuffer != NULL,                      NN_G3D_RES_GET_NAME(arg.GetResource(), GetName()));
    NN_G3D_REQUIRES(IsAligned(pBuffer, Alignment_Buffer), NN_G3D_RES_GET_NAME(arg.GetResource(), GetName()));

    if (arg.IsMemoryCalculated() == false)
    {
        return false;
    }
    if (arg.GetWorkMemorySize() > bufferSize)
    {
        // バッファーが必要なサイズに満たない場合は失敗。
        return false;
    }

    ResModel* pRes            = const_cast<ResModel*>(arg.GetResource());
    ResSkeleton* pResSkeleton = pRes->GetSkeleton();
    int shapeCount            = pRes->GetShapeCount();
    int materialCount         = pRes->GetMaterialCount();
    int viewCount             = arg.GetViewCount();

    // メンバの初期化。
    m_pRes = pRes;
    m_pBoneVisArray  = arg.GetBuffer<Bit32>(pBuffer, InitializeArgument::MemoryBlockIndex_BoneVisArray);
    m_pMatVisArray   = arg.GetBuffer<Bit32>(pBuffer, InitializeArgument::MemoryBlockIndex_MatVisArray);
    m_ViewCount      = static_cast<uint8_t>(viewCount);
    m_Flag           = 0;
    m_ShapeCount     = static_cast<uint16_t>(shapeCount);
    m_MaterialCount  = static_cast<uint16_t>(materialCount);
    m_pSkeleton      = NULL;
    m_pShapeArray    = arg.GetBuffer<ShapeObj>(pBuffer, InitializeArgument::MemoryBlockIndex_ShapeArray);
    m_pMaterialArray = arg.GetBuffer<MaterialObj>(pBuffer, InitializeArgument::MemoryBlockIndex_MaterialArray);
    m_pBounding      = arg.GetBuffer<Sphere>(pBuffer, InitializeArgument::MemoryBlockIndex_Bounding);
    m_pUserPtr       = NULL;
    m_pBufferPtr     = pBuffer;
    m_pMemoryPool    = NULL;
    m_MemoryPoolOffset = 0;
    m_LodCount       = arg.GetLodCount();

    // Skeleton を構築。
    SkeletonObj* pSkeleton = arg.GetSkeleton();
    if (pSkeleton)
    {
        m_pSkeleton = pSkeleton;
    }
    else
    {
        SkeletonObj::Builder builder(pResSkeleton);
        builder.BufferingCount(arg.GetSkeletonBufferingCount());
        builder.CalculateMemorySize();
        size_t size = builder.GetWorkMemorySize();

        void* pSkeletonObjBuffer = arg.GetBuffer(pBuffer, InitializeArgument::MemoryBlockIndex_Skeleton);
        void* pSkeletonBuffer = arg.GetBuffer(pBuffer, InitializeArgument::MemoryBlockIndex_SkeletonBuffer);
        m_pSkeleton = new(pSkeletonObjBuffer) SkeletonObj();
        builder.Build(m_pSkeleton, pSkeletonBuffer, size);
    }

    // 配下の Shape と Material を構築。

    void* pShapeBuffer = arg.GetBuffer(pBuffer, InitializeArgument::MemoryBlockIndex_ShapeBuffer);
    int viewDependent = 0;
    for (int idxShape = 0; idxShape < shapeCount; ++idxShape)
    {
        ResShape* pResShape = pRes->GetShape(idxShape);
        ShapeObj::Builder builder(pResShape);
        builder.BufferingCount(arg.GetShapeBufferingCount());
        ResBone* pResBone = pResSkeleton->GetBone(pResShape->GetBoneIndex());
        if (pResBone->GetBillboardMode() == ResBone::Flag_BillboardNone)
        {
            builder.ViewIndependent();
        }
        else
        {
            builder.ViewDependent();
            viewDependent = 1;
        }
        builder.ViewCount(viewCount);
        if (arg.IsBoundingEnabled())
        {
            builder.SetBoundingEnabled();
        }
        else
        {
            builder.SetBoundingDisabled();
        }
        builder.UserAreaSize(arg.GetShapeUserAreaSize());
        builder.CalculateMemorySize();
        size_t size = builder.GetWorkMemorySize();
        size = nn::util::align_up(size, ShapeObj::Alignment_Buffer);

        ShapeObj* pShape = new(&m_pShapeArray[idxShape]) ShapeObj();
        builder.Build(pShape, pShapeBuffer, size);

        pShapeBuffer = AddOffset(pShapeBuffer, size);
    }
    m_ViewDependent = static_cast<uint8_t>(viewDependent);

    void* pMaterialBuffer = arg.GetBuffer(pBuffer,  InitializeArgument::MemoryBlockIndex_MaterialBuffer);
    for (int idxMaterial = 0; idxMaterial < materialCount; ++idxMaterial)
    {
        ResMaterial* pResMaterial = pRes->GetMaterial(idxMaterial);

        MaterialObj::Builder builder(pResMaterial);
        builder.BufferingCount(arg.GetMaterialBufferingCount());
        builder.CalculateMemorySize();
        size_t size = builder.GetWorkMemorySize();
        size = nn::util::align_up(size, MaterialObj::Alignment_Buffer);

        MaterialObj* pMaterial = new(&m_pMaterialArray[idxMaterial]) MaterialObj();
        builder.Build(pMaterial, pMaterialBuffer, size);

        pMaterialBuffer = AddOffset(pMaterialBuffer, size);
    }

    ClearBoneVisible();
    ClearMaterialVisible();

    if (m_pBounding)
    {
        memset(m_pBounding, 0, sizeof(*m_pBounding)); // 念のため初期化しておく。
    }

    return true;
} //! NOLINT

size_t ModelObj::GetBlockBufferAlignment(nn::gfx::Device* pDevice) const NN_NOEXCEPT
{
    size_t alignment = 1;
    if (m_pSkeleton && !m_pSkeleton->IsBlockBufferValid())
    {
        alignment = std::max(alignment, m_pSkeleton->GetBlockBufferAlignment(pDevice));
    }
    for (int idxShape = 0; idxShape < m_ShapeCount; ++idxShape)
    {
        ShapeObj& shape = m_pShapeArray[idxShape];
        if (!shape.IsBlockBufferValid()) // 既に構築されている場合は何もしない。
        {
            alignment = std::max(alignment, shape.GetBlockBufferAlignment(pDevice));
        }
    }
    for (int idxMaterial = 0; idxMaterial < m_MaterialCount; ++idxMaterial)
    {
        MaterialObj& material = m_pMaterialArray[idxMaterial];
        if (!material.IsBlockBufferValid()) // 既に構築されている場合は何もしない。
        {
            alignment = std::max(alignment, material.GetBlockBufferAlignment(pDevice));
        }
    }

    return alignment;
}

size_t ModelObj::CalculateBlockBufferSize(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    size_t size = 0;

    if (m_pSkeleton && !m_pSkeleton->IsBlockBufferValid())
    {
        // 共有スケルトンは既に構築されている可能性がある。
        size += m_pSkeleton->CalculateBlockBufferSize(pDevice);
    }

    for (int idxShape = 0; idxShape < m_ShapeCount; ++idxShape)
    {
        ShapeObj& shape = m_pShapeArray[idxShape];
        if (!shape.IsBlockBufferValid()) // 既に構築されている場合は何もしない。
        {
            size = nn::util::align_up(size, shape.GetBlockBufferAlignment(pDevice));
            size +=  shape.CalculateBlockBufferSize(pDevice);
        }
    }

    for (int idxMaterial = 0; idxMaterial < m_MaterialCount; ++idxMaterial)
    {
        MaterialObj& material = m_pMaterialArray[idxMaterial];
        if (! m_pMaterialArray[idxMaterial].IsBlockBufferValid()) // 既に構築されている場合は何もしない。
        {
            size = nn::util::align_up(size, material.GetBlockBufferAlignment(pDevice));
            size += material.CalculateBlockBufferSize(pDevice);
        }
    }

    return size;
}

bool ModelObj::SetupBlockBuffer(nn::gfx::Device* pDevice, nn::gfx::MemoryPool* pMemoryPool, ptrdiff_t offset, size_t memoryPoolSize) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pMemoryPool != NULL,           NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    NN_G3D_REQUIRES(IsBlockBufferValid() == false, NN_G3D_RES_GET_NAME(m_pRes, GetName()));

    if (CalculateBlockBufferSize(pDevice) > memoryPoolSize)
    {
        // バッファーが必要なサイズに満たない場合は失敗。
        return false;
    }

    m_pMemoryPool = pMemoryPool;
    m_MemoryPoolOffset = offset;

    bool success = true;
    ptrdiff_t addOffset = 0;
    if (m_pSkeleton && !m_pSkeleton->IsBlockBufferValid())
    {
        // 共有スケルトンは既に構築されている可能性がある。
        size_t size = m_pSkeleton->CalculateBlockBufferSize(pDevice);
        success = m_pSkeleton->SetupBlockBuffer(pDevice, pMemoryPool, offset + addOffset, size) && success;
        addOffset += size;
    }

    for (int idxShape = 0; idxShape < m_ShapeCount; ++idxShape)
    {
        ShapeObj& shape = m_pShapeArray[idxShape];
        if (! m_pShapeArray[idxShape].IsBlockBufferValid()) // 既に構築されている場合は何もしない。
        {
            addOffset = nn::util::align_up(addOffset, shape.GetBlockBufferAlignment(pDevice));
            size_t size =  shape.CalculateBlockBufferSize(pDevice);
            success = shape.SetupBlockBuffer(pDevice, pMemoryPool, offset + addOffset, size) && success;
            addOffset += size;
        }
    }

    for (int idxMaterial = 0; idxMaterial < m_MaterialCount; ++idxMaterial)
    {
        MaterialObj& material = m_pMaterialArray[idxMaterial];
        if (!m_pMaterialArray[idxMaterial].IsBlockBufferValid()) // 既に構築されている場合は何もしない。
        {
            addOffset = nn::util::align_up(addOffset, material.GetBlockBufferAlignment(pDevice));
            size_t size = m_pMaterialArray[idxMaterial].CalculateBlockBufferSize(pDevice);
            success = m_pMaterialArray[idxMaterial].SetupBlockBuffer(pDevice, pMemoryPool, offset + addOffset, size) && success;
            addOffset += size;
        }
    }

    if (success)
    {
        m_Flag |= Flag_BlockBufferValid;
    }

    return success;
}

void ModelObj::CleanupBlockBuffer(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(IsBlockBufferValid() == true, NN_G3D_RES_GET_NAME(m_pRes, GetName()));

    if (m_pSkeleton && m_pSkeleton->IsBlockBufferValid())
    {
        // 共有スケルトンは既に破棄されている可能性がある。
        m_pSkeleton->CleanupBlockBuffer(pDevice);
    }

    for (int idxShape = 0; idxShape < m_ShapeCount; ++idxShape)
    {
        ShapeObj& shape = m_pShapeArray[idxShape];
        if (shape.IsBlockBufferValid()) // 既に解放されている場合は何もしない。
        {
            shape.CleanupBlockBuffer(pDevice);
        }
    }

    for (int idxMaterial = 0; idxMaterial < m_MaterialCount; ++idxMaterial)
    {
        MaterialObj& material = m_pMaterialArray[idxMaterial];
        if (material.IsBlockBufferValid()) // 既に解放されている場合は何もしない。
        {
            material.CleanupBlockBuffer(pDevice);
        }
    }

    m_Flag ^= Flag_BlockBufferValid;

    m_pMemoryPool = NULL;
    m_MemoryPoolOffset = 0;
}

void ModelObj::CalculateWorld(const nn::util::Matrix4x3fType& baseMtx) NN_NOEXCEPT
{
    m_pSkeleton->CalculateWorldMtx(baseMtx);
}

void ModelObj::CalculateBounding(int lodLevel) NN_NOEXCEPT
{
    NN_G3D_REQUIRES_RANGE(lodLevel, 0, m_LodCount, NN_G3D_RES_GET_NAME(m_pRes, GetName()))
    if (m_pBounding == NULL)
    {
        return;
    }
    Sphere& bounding = m_pBounding[lodLevel];
    if (GetShapeCount() >= 1)
    {
        ShapeObj& shape = m_pShapeArray[0];
        if (lodLevel < shape.GetMeshCount())
        {
            shape.CalculateBounding(m_pSkeleton, lodLevel);
            bounding = *shape.GetBounding(lodLevel);
        }
        // 該当のLODレベルを持たないなら、LODレベル0で代用
        else
        {
            shape.CalculateBounding(m_pSkeleton, 0);
            bounding = *shape.GetBounding(0);
        }
    }
    for (int idxShape = 1, shapeCount = GetShapeCount(); idxShape < shapeCount; ++idxShape)
    {
        ShapeObj& shape = m_pShapeArray[idxShape];
        if (lodLevel < shape.GetMeshCount())
        {
            shape.CalculateBounding(m_pSkeleton, lodLevel);
            bounding.Merge(bounding, *shape.GetBounding(lodLevel));
        }
        // 該当のLODレベルを持たないなら、LODレベル0で代用
        else
        {
            shape.CalculateBounding(m_pSkeleton, 0);
            bounding.Merge(bounding, *shape.GetBounding(0));
        }
    }
}

void ModelObj::CalculateSkeleton(int bufferIndex) NN_NOEXCEPT
{
    m_pSkeleton->CalculateSkeleton(bufferIndex);
}

void ModelObj::CalculateView(int viewIndex, const nn::util::Matrix4x3fType& cameraMtx, int bufferIndex) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(viewIndex < GetViewCount(), NN_G3D_RES_GET_NAME(m_pRes, GetName()));

    if (!IsViewDependent())
    {
        return; // 将来ビューに依存する処理が増えた場合はこの処理はスキップできない可能性がある。
    }

    // 直近のビルボード行列をキャッシュする。
    // キャッシュはシェイプがボーンの階層構造順に並んでいることに依存する。
    int lastBoneIndex = -1;
    nn::util::Matrix4x3fType billboardMtx;

    const nn::util::Matrix4x3fType* pWorldMtxArray = m_pSkeleton->GetWorldMtxArray();
    for (int idxShape = 0; idxShape < m_ShapeCount; ++idxShape)
    {
        ShapeObj& shape = m_pShapeArray[idxShape];
        if (shape.IsRigidBody() || m_ShapeUserAreaForceUpdate)
        {
            if (shape.IsViewDependent())
            {
                // ビューに依存しないシェイプは CalcShape で計算する。
                int idxBone = shape.GetBoneIndex();
                const ResBone* pBone = m_pSkeleton->GetResBone(idxBone);

                Bit32 bbMode = pBone->GetBillboardMode();
                NN_G3D_ASSERT(bbMode <= ResBone::Flag_BillboardMax, NN_G3D_RES_GET_NAME(m_pRes, GetName()));
                if (bbMode == ResBone::Flag_BillboardNone)
                {
                    // ビルボードでない場合は通常のワールド行列を書き込む。
                    shape.CalculateShape(viewIndex, pWorldMtxArray[idxBone], bufferIndex);
                    continue;
                }

                int bbBoneIndex = pBone->ToData().billboardIndex;
                if (bbBoneIndex == ResBone::Flag_BillboardIndexNone)
                {
                    // 階層ビルボードではないのでワールド行列と乗算されたビルボード行列を取得する。
                    if (idxBone != lastBoneIndex) // 非階層ビルボードは自分自身のインデックスを見る。
                    {
                        // キャッシュされていない場合はビルボード行列を計算する。
                        m_pSkeleton->CalculateBillboardMtx(&billboardMtx, cameraMtx, idxBone, true);
                        lastBoneIndex = idxBone;
                    }
                    shape.CalculateShape(viewIndex, billboardMtx, bufferIndex);
                    continue;
                }

                if (bbBoneIndex != lastBoneIndex)
                {
                    // キャッシュされていない場合はビルボード行列を計算する。
                    m_pSkeleton->CalculateBillboardMtx(&billboardMtx, cameraMtx, bbBoneIndex, false);
                    lastBoneIndex = bbBoneIndex;
                }

                // ビルボード行列をかけて回転させる。
                nn::util::Matrix4x3fType worldMtx;
                MatrixMultiply(&worldMtx, pWorldMtxArray[idxBone], billboardMtx);
                //worldMtx.Mul(billboardMtx, pWorldMtxArray[idxBone]);
                shape.CalculateShape(viewIndex, worldMtx, bufferIndex);
            }
        }
    }
}

void ModelObj::CalculateShape(int bufferIndex) NN_NOEXCEPT
{
    const nn::util::Matrix4x3fType* pWorldMtxArray = m_pSkeleton->GetWorldMtxArray();
    nn::util::Matrix4x3fType identityMtx;
    nn::util::MatrixIdentity(&identityMtx);
    for (int idxShape = 0; idxShape < m_ShapeCount; ++idxShape)
    {
        ShapeObj& shape = m_pShapeArray[idxShape];
        if (shape.IsRigidBody() || m_ShapeUserAreaForceUpdate)
        {
            if (!shape.IsViewDependent())
            {
                // ビューに依存するシェイプは CalcView で計算する。
                int idxBone = shape.GetBoneIndex();
                shape.CalculateShape(0, pWorldMtxArray[idxBone], bufferIndex);
            }
        }
    }
}

void ModelObj::CalculateMaterial(int bufferIndex) NN_NOEXCEPT
{
    for (int idxMaterial = 0; idxMaterial < m_MaterialCount; ++idxMaterial)
    {
        MaterialObj& material = m_pMaterialArray[idxMaterial];
        material.CalculateMaterial(bufferIndex);
    }
}

void ModelObj::UpdateViewDependency() NN_NOEXCEPT
{
    int viewDependent = 0;
    for (int idxShape = 0; idxShape < m_ShapeCount; ++idxShape)
    {
        if (m_pShapeArray[idxShape].IsViewDependent())
        {
            viewDependent = 1;
        }
    }
    m_ViewDependent = static_cast<uint8_t>(viewDependent);
}

void ModelObj::SetShapeAnimCalculationEnabled() NN_NOEXCEPT
{
    for (int shapeIndex = 0, shapeCount = GetShapeCount(); shapeIndex < shapeCount; ++shapeIndex)
    {
        m_pShapeArray[shapeIndex].SetShapeAnimCalculationEnabled();
    }
}

void ModelObj::SetShapeAnimCalculationDisabled() NN_NOEXCEPT
{
    for (int shapeIndex = 0, shapeCount = GetShapeCount(); shapeIndex < shapeCount; ++shapeIndex)
    {
        m_pShapeArray[shapeIndex].SetShapeAnimCalculationDisabled();
    }
}

void ModelObj::ClearBoneVisible() NN_NOEXCEPT
{
    for (int idxBone = 0, boneCount = m_pSkeleton->GetBoneCount(); idxBone < boneCount; ++idxBone)
    {
        const ResBone* pBone = m_pSkeleton->GetResBone(idxBone);
        bool visible = pBone->IsVisible();
        SetBoneVisible(idxBone, visible);
    }
}

void ModelObj::ClearMaterialVisible() NN_NOEXCEPT
{
    for (int idxMat = 0, materialCount = GetMaterialCount(); idxMat < materialCount; ++idxMat)
    {
        const ResMaterial* pResMaterial = GetMaterial(idxMat)->GetResource();
        bool visible = pResMaterial->ToData().flag & ResMaterial::Flag_Visibility ? true : false;
        SetMaterialVisible(idxMat, visible);
    }
}

void ModelObj::SetTextureChangeCallback(MaterialObj::TextureChangeCallback pCallback) NN_NOEXCEPT
{
    for (int idxMat = 0, materialCount = GetMaterialCount(); idxMat < materialCount; ++idxMat)
    {
        GetMaterial(idxMat)->SetTextureChangeCallback(pCallback);
    }
}

}} // namespace nn::g3d

