﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/font/font_CharStrmReader.h>
#include <nn/font/font_Util.h>

namespace nn {
namespace font {
namespace detail {

#if defined(NN_FONT_UNREFERENCE)
const uint32_t  SjisLowWidth    = 0xBC;
const uint32_t  SjisLowBase     = 0x40;
#endif
const uint32_t  SjisHigh0Start     = 0x81;
#if defined(NN_FONT_UNREFERENCE)
const uint32_t  SjisHigh0End     = 0x85;
const uint32_t  SjisHigh1Start     = 0x87;
const uint32_t  SjisHigh1End     = 0x88;
const uint32_t  SjisHigh2Start     = 0x88;
#endif
const uint32_t  SjisHigh2End     = 0xA0;
const uint32_t  SjisHigh3Start     = 0xE0;
#if defined(NN_FONT_UNREFERENCE)
const uint32_t  SjisHigh3End     = 0x100;
const uint32_t  AsciiStart          = 0x20;
const uint32_t  AsciiEnd          = 0x80;
const uint32_t  SjisHankanaStart   = 0xA0;
const uint32_t  SjisHankanaEnd   = 0xE0;
#endif

/*!--------------------------------------------------------------------------*
  @brief        対象のバイトがSJIS文字のリードバイトがどうか判定します。

  @param[in]    c   判定対象のバイトデータ。

  @return       c がリードバイトなら true そうでなければ false。
 *---------------------------------------------------------------------------*/
inline
bool
IsSjisLeadByte(uint8_t  c)
{
    return ((SjisHigh0Start <= c) && (c < SjisHigh2End))
            || (SjisHigh3Start <= c);
}

}   // namespace detail

uint32_t CharStrmReader::ReadNextCharUtf8()
{
    NN_SDK_ASSERT_NOT_NULL( this );
    NN_SDK_ASSERT_NOT_NULL( m_pCharStream );

    // ストリームから UTF-8 のコードポイントまでを切り出して、ストリームのポインタを進める
    const char*& pCharStrm = reinterpret_cast<const char*&>(m_pCharStream);
    char buffer[4];
    nn::util::CharacterEncodingResult result = nn::util::PickOutCharacterFromUtf8String(buffer, &pCharStrm);
    NN_SDK_ASSERT(result == nn::util::CharacterEncodingResult_Success, "Wrong character code: 0x%02x", static_cast<uint8_t>(*pCharStrm));
    NN_UNUSED(result);
    return ConvertCharUtf8ToUtf32(buffer);
}

uint32_t CharStrmReader::ReadNextCharUtf16()
{
    NN_SDK_ASSERT_NOT_NULL( this );
    NN_SDK_ASSERT_NOT_NULL( m_pCharStream );
    NN_SDK_ASSERT(detail::IsAligned(m_pCharStream, 2));

    // UTF-16 の読み込みはサロゲートペアには対応させない
    uint16_t code = GetChar<uint16_t >();
    StepStream<uint16_t >();

    return static_cast<uint32_t>(code);
}

uint32_t CharStrmReader::ReadNextCharCp1252()
{
    NN_SDK_ASSERT_NOT_NULL( this );
    NN_SDK_ASSERT_NOT_NULL( m_pCharStream );
    uint16_t  code;

    code = GetChar<uint8_t >();
    StepStream<uint8_t >();

    return static_cast<uint32_t>(code);
}

uint32_t CharStrmReader::ReadNextCharSjis()
{
    NN_SDK_ASSERT_NOT_NULL( this );
    NN_SDK_ASSERT_NOT_NULL( m_pCharStream );
    uint16_t  code;

    if (detail::IsSjisLeadByte(GetChar<uint8_t >()))
    {
        //---- 2バイト文字
        code = static_cast<uint16_t >((GetChar<uint8_t >(0) << 8) | GetChar<uint8_t >(1));
        StepStream<uint8_t >(2);
    }
    else
    {
        //---- 1バイト文字
        code = GetChar<uint8_t >();
        StepStream<uint8_t >();
    }

    return static_cast<uint32_t>(code);
}

}   // namespace font
}   // namespace nn
