﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>

#include <nn/atk2/detail/atk2_Voice.h>

#include <nn/atk2/atk2_AudioEngine.h>
#include <nn/atk2/detail/atk2_Filter.h>
#include <nn/atk2/detail/atk2_CommandManager.h>

namespace nn { namespace atk2 { namespace detail {

Voice::Voice() NN_NOEXCEPT
    : m_VoiceState(VoiceState_Stop)
{
}

Voice::~Voice() NN_NOEXCEPT
{
}

bool Voice::AllocVoice(uint32_t priority, AudioEngine* pAudioEngine) NN_NOEXCEPT
{
    m_VoiceId = pAudioEngine->GetVirtualVoiceManager().AllocVirtualVoice();
    if ( m_VoiceId == VirtualVoiceManager::InvalidVoiceId )
    {
        return false;
    }

    VoiceCommandManager& cmdmgr = pAudioEngine->GetVoiceCommandManager();
    VoiceCommandAlloc* command = cmdmgr.AllocCommand<VoiceCommandAlloc>( false );
    //@@@ NN_DETAIL_ATK_INFO("__ALLOC__\n");
    if ( command == NULL )
    {
        //NN_ATK_WARNING("[%s:%d] cannot alloc command", __FUNCTION__, __LINE__);
        return false;
    }
    command->id = VoiceCommandId_AllocVoice;
    command->voiceId = m_VoiceId;
    command->priority = priority;
    command->userId = this;
    m_CommandTag = cmdmgr.PushCommand(command);

    Initialize( priority, pAudioEngine );

    return true;
}

void Voice::Free() NN_NOEXCEPT
{
    VoiceCommandManager& cmdmgr = m_pAudioEngine->GetVoiceCommandManager();
    VoiceCommandFree* command = cmdmgr.AllocCommand<VoiceCommandFree>( false );
    //@@@ NN_DETAIL_ATK_INFO("__FREE__\n");
    if ( command == NULL )
    {
        //NN_ATK_WARNING("[%s:%d] cannot alloc command", __FUNCTION__, __LINE__);
        m_WaveBufferList.Clear();
        return;
    }
    command->id = VoiceCommandId_FreeVoice;
    command->voiceId = m_VoiceId;
    cmdmgr.PushCommand(command);

    m_WaveBufferList.Clear();

    if ( m_VoiceId != VirtualVoiceManager::InvalidVoiceId )
    {
        m_pAudioEngine->GetVirtualVoiceManager().FreeVirtualVoice( m_VoiceId );
        m_VoiceId = VirtualVoiceManager::InvalidVoiceId;
    }
    m_PlaySamplePosition = 0;
}

void Voice::AppendWaveBuffer(WaveBuffer* waveBuffer) NN_NOEXCEPT
{
    VoiceCommandManager& cmdmgr = m_pAudioEngine->GetVoiceCommandManager();
    VoiceCommandAppendWaveBuffer* command = cmdmgr.AllocCommand<VoiceCommandAppendWaveBuffer>( false );
    //@@@ NN_DETAIL_ATK_INFO("__APPEND__\n");
    if ( command == NULL )
    {
        //NN_ATK_WARNING("[%s:%d] cannot alloc command", __FUNCTION__, __LINE__);
        return;
    }
    command->id = VoiceCommandId_AppendWaveBuffer;
    command->voiceId = m_VoiceId;
    command->tag = waveBuffer;
    command->bufferAddress = waveBuffer->bufferAddress;
    command->bufferSize = waveBuffer->bufferSize;
    command->sampleLength = waveBuffer->sampleLength;
    command->sampleOffset = waveBuffer->sampleOffset;
    if ( waveBuffer->pAdpcmContext != NULL )
    {
        command->adpcmContext = *(waveBuffer->pAdpcmContext);
        command->adpcmContextEnable = true;
    }
    else
    {
        command->adpcmContextEnable = false;
    }
    command->loopFlag = waveBuffer->loopFlag;
    cmdmgr.PushCommand(command);

    m_WaveBufferList.PushBack(waveBuffer);
}

bool Voice::IsAvailable() const NN_NOEXCEPT
{
    if ( m_VoiceId != VirtualVoiceManager::InvalidVoiceId )
    {
        RendererVoice* pVoice = m_pAudioEngine->GetVirtualVoiceManager().GetRendererVoice(m_VoiceId);
        if (pVoice != NULL)
        {
            return pVoice->IsAvailable();
        }

        return true;
    }
    return false;
}

ptrdiff_t Voice::GetPlaySamplePosition() const NN_NOEXCEPT
{
    return m_PlaySamplePosition;
}

void Voice::SetSampleRate(uint32_t sampleRate) NN_NOEXCEPT
{
    m_SampleRate = sampleRate;
}

void Voice::SetSampleFormat(SampleFormat format) NN_NOEXCEPT
{
    m_SampleFormat = format;
}

void Voice::SetAdpcmParam(const AdpcmParam& param) NN_NOEXCEPT
{
    m_AdpcmParam = param;
}

void Voice:: SetPriority(uint32_t priority) NN_NOEXCEPT
{
    m_Priority = priority;
}

uint32_t Voice::GetPriority() const NN_NOEXCEPT
{
    return m_Priority;
}

void Voice::SetVolume(float volume) NN_NOEXCEPT
{
    m_Volume = volume;
}

float Voice::GetVolume() const NN_NOEXCEPT
{
    return m_Volume;
}

void Voice::SetPitch(float pitch) NN_NOEXCEPT
{
    m_Pitch = pitch;
}

float Voice::GetPitch() const NN_NOEXCEPT
{
    return m_Pitch;
}

//void Voice::SetInterpolationType( uint8_t interpolationType ) NN_NOEXCEPT
//{
//    m_InterpolationType = interpolationType;
//}

void Voice::SetMonoFilter( bool enable, uint16_t cutoff ) NN_NOEXCEPT
{
    m_MonoFilterFlag = enable;
    if (enable)
    {
        m_MonoFilterCutoff = cutoff;
    }
}

void Voice::SetBiquadFilter( bool enable, const BiquadFilterCoefficients* coef /* = NULL */ ) NN_NOEXCEPT
{
    m_BiquadFilterFlag = enable;
    if (enable)
    {
        m_BiquadFilterCoefficients = *coef;
    }
}

void Voice::SetOutputReceiver( OutputReceiver* pOutputReceiver ) NN_NOEXCEPT
{
    m_pOutputReceiver = pOutputReceiver;
}

// 内部管理用

void Voice::SetVoiceState(VoiceState voiceState) NN_NOEXCEPT
{
    m_VoiceState = voiceState;

    if ( voiceState == VoiceState_Play )
    {
        VoiceCommandManager& cmdmgr = m_pAudioEngine->GetVoiceCommandManager();
        VoiceCommandPlay* command = cmdmgr.AllocCommand<VoiceCommandPlay>( false );
        if ( command == nullptr )
        {
            NN_SDK_LOG("[%s:%d] cannot alloc command\n", __FUNCTION__, __LINE__);
            return;
        }
        command->id = VoiceCommandId_Play;
        command->voiceId = m_VoiceId;
        command->sampleFormat = m_SampleFormat;
        command->sampleRate = m_SampleRate;
        command->adpcmParam = m_AdpcmParam;
        command->pOutputReceiver = m_pOutputReceiver;
        cmdmgr.PushCommand(command);
    }
    else if ( voiceState == VoiceState_Pause )
    {
        VoiceCommandManager& cmdmgr = m_pAudioEngine->GetVoiceCommandManager();
        VoiceCommandPause* command = cmdmgr.AllocCommand<VoiceCommandPause>( false );
        //@@@ NN_DETAIL_ATK_INFO("__PAUSE__\n");
        if ( command == NULL )
        {
            NN_SDK_LOG("[%s:%d] cannot alloc command\n", __FUNCTION__, __LINE__);
            return;
        }
        command->id = VoiceCommandId_Pause;
        command->voiceId = m_VoiceId;
        cmdmgr.PushCommand(command);
    }
    else if ( voiceState == VoiceState_Stop )
    {
        VoiceCommandManager& cmdmgr = m_pAudioEngine->GetVoiceCommandManager();
        VoiceCommandFree* command = cmdmgr.AllocCommand<VoiceCommandFree>( false );
        //@@@ NN_DETAIL_ATK_INFO("__FREE__\n");
        if ( command == NULL )
        {
            NN_SDK_LOG("[%s:%d] cannot alloc command\n", __FUNCTION__, __LINE__);
            m_WaveBufferList.Clear();
            return;
        }
        command->id = VoiceCommandId_Stop;
        command->voiceId = m_VoiceId;
        cmdmgr.PushCommand(command);

        m_WaveBufferList.Clear();
    }
}

VoiceState Voice::GetVoiceState() const NN_NOEXCEPT
{
    return m_VoiceState;
}

void Voice::UpdateParam() NN_NOEXCEPT
{
    VoiceCommandManager& cmdmgr = m_pAudioEngine->GetVoiceCommandManager();
    VoiceCommandParam* command = cmdmgr.AllocCommand<VoiceCommandParam>( false );
    //@@@ NN_DETAIL_ATK_INFO("__PARAM__\n");
    if ( command == NULL )
    {
        NN_SDK_LOG("[%s:%d] cannot alloc command", __FUNCTION__, __LINE__);
        return;
    }
    command->id = VoiceCommandId_UpdateParam;
    command->voiceId = m_VoiceId;
    command->volume = m_Volume;
    command->pitch = m_Pitch;
    command->monoFilterFlag = m_MonoFilterFlag;
    command->monoFilterCutoff = m_MonoFilterCutoff;
    command->biquadFilterFlag = m_BiquadFilterFlag;
    command->biquadFilterCoefficients = m_BiquadFilterCoefficients;
    cmdmgr.PushCommand(command);
}

// private

void Voice::Initialize(uint32_t priority, AudioEngine* pAudioEngine) NN_NOEXCEPT
{
    m_SampleRate = 48000;
    m_SampleFormat = SampleFormat_PcmS16;
    //m_AdpcmParam;

    m_Volume = 1.0f;
    m_Pitch = 1.0f;
    //m_TvMix = DefaultTvMix;
    m_MonoFilterFlag = false;
    m_BiquadFilterFlag = false;
    m_BiquadFilterCoefficients = BiquadFilterCoefZero;
    m_MonoFilterCutoff = 0;
    //m_InterpolationType = 0;
    m_Priority = priority;
    m_pOutputReceiver = nullptr;

    m_VoiceState = VoiceState_Stop;

    //m_VoiceId;
    m_PlaySamplePosition = 0;
    m_VoiceInfoEnableFlag = true;
    //m_CommandTag;

    m_pAudioEngine = pAudioEngine;
}

}}}
