﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/viewer/detail/atk_SoundArchiveParameterEditor.h>

#ifdef NN_ATK_CONFIG_ENABLE_DEV

#include <nn/atk/atk_Util.h>
#include <nn/atk/atk_SoundArchive.h>
#include <nn/atk/viewer/detail/atk_SoundArchiveEditController.h>
#include <nn/atk/viewer/detail/res/atk_ResourceManager.h>
#include <nn/atk/detail/atk_SoundArchiveManager.h>

namespace nn {
namespace atk {
namespace viewer {
namespace detail {

//----------------------------------------------------------
SoundArchiveParameterEditor::SoundArchiveParameterEditor() NN_NOEXCEPT :
m_SoundArchiveEditController(NULL),
m_ResourceManager(NULL),
m_SoundArchiveManager(NULL)
{
    m_SoundArchiveHook.Initialize(*this);
}

//----------------------------------------------------------
Result
SoundArchiveParameterEditor::Initialize(
    SoundArchiveEditController* editController,
    ResourceManager* resourceManager,
    nn::atk::detail::SoundArchiveManager* soundArchiveManager) NN_NOEXCEPT
{
    if(editController == NULL ||
        soundArchiveManager == NULL)
    {
        NN_SDK_ASSERT(false, "invalid arguments.\n");
        return Result(ResultType_Failed);
    }

    m_SoundArchiveEditController = editController;
    m_ResourceManager = resourceManager;
    m_SoundArchiveManager = soundArchiveManager;

    return Result(ResultType_True);
}

//----------------------------------------------------------
void
SoundArchiveParameterEditor::Finalize() NN_NOEXCEPT
{
    m_SoundArchiveEditController = NULL;
    m_ResourceManager = NULL;
    m_SoundArchiveManager = NULL;
}

//----------------------------------------------------------
bool SoundArchiveParameterEditor::IsInitialized() const NN_NOEXCEPT
{
    return m_SoundArchiveManager != NULL && m_SoundArchiveManager->GetMainSoundArchive() != NULL;
}

//----------------------------------------------------------
void
SoundArchiveParameterEditor::Start() NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsInitialized(), "SoundArchiveParameterEditor is not initialized.");
    m_SoundArchiveManager->SetParametersHook(&m_SoundArchiveHook);
}

//----------------------------------------------------------
void
SoundArchiveParameterEditor::Stop() NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsInitialized(), "SoundArchiveParameterEditor is not initialized.");
    m_SoundArchiveManager->SetParametersHook(NULL);
}

//----------------------------------------------------------
bool
SoundArchiveParameterEditor::Hook::IsTargetItemImpl(const char* itemLabel) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(itemLabel);
    NN_SDK_ASSERT(*itemLabel != '\0');

    NN_SDK_ASSERT_NOT_NULL(m_Owner);
    NN_SDK_ASSERT(m_Owner->IsInitialized(), "SoundArchiveFileEditor is not initialized.");

    return m_Owner->m_SoundArchiveEditController->GetItemCacheState(itemLabel) == CacheState_Cached;
}

//----------------------------------------------------------
const char*
SoundArchiveParameterEditor::Hook::GetItemLabelImpl(SoundArchive::ItemId id) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_Owner);
    NN_SDK_ASSERT(m_Owner->IsInitialized(), "SoundArchiveFileEditor is not initialized.");

    EditItemInfo editItemInfo;
    if(!m_Owner->m_ResourceManager->TryGetEditItemInfoFromID(id, &editItemInfo))
    {
        return NULL;
    }

    return editItemInfo.name;
}

//----------------------------------------------------------
SoundArchive::ItemId
SoundArchiveParameterEditor::Hook::GetItemIdImpl(const char* itemLabel) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_Owner);
    NN_SDK_ASSERT(m_Owner->IsInitialized(), "SoundArchiveFileEditor is not initialized.");

    EditItemInfo editItemInfo;
    if(!m_Owner->m_ResourceManager->TryGetEditItemInfo(itemLabel, &editItemInfo))
    {
        return SoundArchive::InvalidId;
    }

    return editItemInfo.id;
}

//----------------------------------------------------------
SoundArchive::SoundType
SoundArchiveParameterEditor::Hook::GetSoundTypeImpl(const char* itemLabel) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_Owner);
    NN_SDK_ASSERT(m_Owner->IsInitialized(), "SoundArchiveFileEditor is not initialized.");

    ResDataType dataType = ResDataType_Unknown;
    m_Owner->m_SoundArchiveEditController->GetSoundInfo(itemLabel, NULL, &dataType);

    return ResDataTypeToSoundType(dataType);
}

//----------------------------------------------------------
bool
SoundArchiveParameterEditor::Hook::ReadSoundInfoImpl(
    SoundArchive::ItemId soundId,
    SoundArchive::SoundInfo* info) const NN_NOEXCEPT
{
    return ReadSoundInfoImpl(GetItemName(soundId), info);
}

//----------------------------------------------------------
bool
SoundArchiveParameterEditor::Hook::ReadSoundInfoImpl(
    const char* soundName,
    SoundArchive::SoundInfo* info) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(info);

    NN_SDK_ASSERT_NOT_NULL(m_Owner);
    NN_SDK_ASSERT(m_Owner->IsInitialized(), "SoundArchiveParameterEditor is not initialized.");

    if(soundName == NULL || soundName[0] == '\0')
    {
        return false;
    }

    ResDataType itemInfoDataType = ResDataType_Unknown;
    const ResSoundInfo* editInfo =
        m_Owner->m_SoundArchiveEditController->GetSoundInfo(soundName, NULL, &itemInfoDataType);

    if(editInfo == NULL)
    {
        return false;
    }

    editInfo->soundBasicParam.ApplyTo(*info, *m_Owner->m_SoundArchiveManager->GetMainSoundArchive());

    return true;
}

//----------------------------------------------------------
bool
SoundArchiveParameterEditor::Hook::ReadSound3DInfoImpl(
    SoundArchive::ItemId soundId,
    SoundArchive::Sound3DInfo* info) const NN_NOEXCEPT
{
    return ReadSound3DInfoImpl(GetItemName(soundId), info);
}

//----------------------------------------------------------
bool
SoundArchiveParameterEditor::Hook::ReadSound3DInfoImpl(
    const char* soundName,
    SoundArchive::Sound3DInfo* info) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(info);

    NN_SDK_ASSERT_NOT_NULL(m_Owner);
    NN_SDK_ASSERT(m_Owner->IsInitialized(), "SoundArchiveParameterEditor is not initialized.");

    if(soundName == NULL || soundName[0] == '\0')
    {
        return false;
    }

    ResDataType itemInfoDataType = ResDataType_Unknown;
    const ResSoundInfo* editInfo =
        m_Owner->m_SoundArchiveEditController->GetSoundInfo(soundName, NULL, &itemInfoDataType);

    if(editInfo == NULL)
    {
        return false;
    }

    editInfo->sound3DParam.ApplyTo(*info);

    return true;
}

//----------------------------------------------------------
bool
SoundArchiveParameterEditor::Hook::ReadSequenceSoundInfoImpl(
    SoundArchive::ItemId soundId,
    SoundArchive::SequenceSoundInfo* info) const NN_NOEXCEPT
{
    return ReadSequenceSoundInfoImpl(GetItemName(soundId), info);
}

//----------------------------------------------------------
bool
SoundArchiveParameterEditor::Hook::ReadSequenceSoundInfoImpl(
    const char* soundName,
    SoundArchive::SequenceSoundInfo* info) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(info);

    NN_SDK_ASSERT_NOT_NULL(m_Owner);
    NN_SDK_ASSERT(m_Owner->IsInitialized(), "SoundArchiveParameterEditor is not initialized.");

    if(soundName == NULL || soundName[0] == '\0')
    {
        return false;
    }

    // キャッシュがあれば、それを適用します。
    // キャッシュがなければ、HIO経由でサウンド情報を取得します。
    const ResSequenceSoundInfo* editInfo = reinterpret_cast<const ResSequenceSoundInfo*>(
        m_Owner->m_SoundArchiveEditController->GetSoundInfo(soundName, &ResItemInfoUtility::IsSequenceSoundInfoDataType)
        );

    if(editInfo == NULL)
    {
        return false;
    }

    editInfo->sequenceSoundParam.ApplyTo(*info, *m_Owner->m_SoundArchiveManager);

    return true;
}

//----------------------------------------------------------
bool
SoundArchiveParameterEditor::Hook::ReadStreamSoundInfoImpl(
    SoundArchive::ItemId soundId,
    SoundArchive::StreamSoundInfo* info ) const NN_NOEXCEPT
{
    return ReadStreamSoundInfoImpl(GetItemName(soundId), info);
}

//----------------------------------------------------------
bool
SoundArchiveParameterEditor::Hook::ReadStreamSoundInfoImpl(
    const char* soundName,
    SoundArchive::StreamSoundInfo* info ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(info);

    NN_SDK_ASSERT_NOT_NULL(m_Owner);
    NN_SDK_ASSERT(m_Owner->IsInitialized(), "SoundArchiveParameterEditor is not initialized.");

    if(soundName == NULL || soundName[0] == '\0')
    {
        return false;
    }

    const ResStreamSoundInfo* editInfo = reinterpret_cast<const ResStreamSoundInfo*>(
        m_Owner->m_SoundArchiveEditController->GetSoundInfo(soundName, &ResItemInfoUtility::IsStreamSoundInfoDataType)
        );

    if(editInfo == NULL)
    {
        return false;
    }

    editInfo->streamSoundParam.ApplyTo(*info);

    return true;
}

//----------------------------------------------------------
bool
SoundArchiveParameterEditor::Hook::ReadStreamSoundInfo2Impl(
    SoundArchive::ItemId soundId,
    SoundArchive::StreamSoundInfo2* info ) const NN_NOEXCEPT
{
    return ReadStreamSoundInfo2Impl(GetItemName(soundId), info);
}

//----------------------------------------------------------
bool
SoundArchiveParameterEditor::Hook::ReadStreamSoundInfo2Impl(
    const char* soundName,
    SoundArchive::StreamSoundInfo2* info ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(info);

    NN_SDK_ASSERT_NOT_NULL(m_Owner);
    NN_SDK_ASSERT(m_Owner->IsInitialized(), "SoundArchiveParameterEditor is not initialized.");

    if(soundName == NULL || soundName[0] == '\0')
    {
        return false;
    }

    const ResStreamSoundInfo* editInfo = reinterpret_cast<const ResStreamSoundInfo*>(
        m_Owner->m_SoundArchiveEditController->GetSoundInfo(soundName, &ResItemInfoUtility::IsStreamSoundInfoDataType)
        );

    if(editInfo == NULL)
    {
        return false;
    }

    editInfo->streamSoundParam2.ApplyTo(*info);

    return true;
}

//----------------------------------------------------------
bool
SoundArchiveParameterEditor::Hook::ReadWaveSoundInfoImpl(
    SoundArchive::ItemId soundId,
    SoundArchive::WaveSoundInfo* info) const NN_NOEXCEPT
{
    return ReadWaveSoundInfoImpl(GetItemName(soundId), info);
}

//----------------------------------------------------------
bool
SoundArchiveParameterEditor::Hook::ReadWaveSoundInfoImpl(
    const char* soundName,
    SoundArchive::WaveSoundInfo* info) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(info);

    NN_SDK_ASSERT_NOT_NULL(m_Owner);
    NN_SDK_ASSERT(m_Owner->IsInitialized(), "SoundArchiveParameterEditor is not initialized.");

    if(soundName == NULL || soundName[0] == '\0')
    {
        return false;
    }

    const ResWaveSoundInfo* editInfo = reinterpret_cast<const ResWaveSoundInfo*>(
        m_Owner->m_SoundArchiveEditController->GetSoundInfo(soundName, &ResItemInfoUtility::IsWaveSoundInfoDataType)
        );

    if(editInfo != NULL)
    {
        editInfo->waveSoundParam.ApplyTo(*info);
        return true;
    }

    return false;
}
//----------------------------------------------------------
bool
SoundArchiveParameterEditor::Hook::ReadSoundUserParamImpl(
    uint32_t* pOutValue,
    SoundArchive::ItemId soundId,
    int index) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);

    return ReadSoundUserParamImpl(pOutValue, GetItemName(soundId), index);
}

//----------------------------------------------------------
bool
SoundArchiveParameterEditor::Hook::ReadSoundUserParamImpl(
    uint32_t* pOutValue,
    const char* soundName,
    int index) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);
    NN_SDK_ASSERT_NOT_NULL(m_Owner);
    NN_SDK_ASSERT(m_Owner->IsInitialized(), "SoundArchiveParameterEditor is not initialized.");

    if(soundName == NULL || soundName[0] == '\0')
    {
        return false;
    }

    ResDataType itemInfoDataType = ResDataType_Unknown;
    const ResSoundInfo* editInfo =
        m_Owner->m_SoundArchiveEditController->GetSoundInfo(soundName, NULL, &itemInfoDataType);

    if(editInfo == NULL)
    {
        return false;
    }

    *pOutValue = editInfo->soundBasicParam.userParams[index];
    return true;
}

//----------------------------------------------------------
uint32_t
    SoundArchiveParameterEditor::Hook::GetSoundUserParamImpl(
    SoundArchive::ItemId soundId,
    uint32_t userParam ) const NN_NOEXCEPT
{
    return GetSoundUserParamImpl(GetItemName(soundId), userParam);
}

//----------------------------------------------------------
uint32_t
SoundArchiveParameterEditor::Hook::GetSoundUserParamImpl(
    const char* soundName,
    uint32_t userParam ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_Owner);
    NN_SDK_ASSERT(m_Owner->IsInitialized(), "SoundArchiveParameterEditor is not initialized.");

    if(soundName == NULL || soundName[0] == '\0')
    {
        return userParam;
    }

    ResDataType itemInfoDataType = ResDataType_Unknown;
    const ResSoundInfo* editInfo =
        m_Owner->m_SoundArchiveEditController->GetSoundInfo(soundName, NULL, &itemInfoDataType);

    if(editInfo == NULL)
    {
        return userParam;
    }

    return editInfo->soundBasicParam.userParams[0];
}

//----------------------------------------------------------
const char*
SoundArchiveParameterEditor::Hook::GetItemName(SoundArchive::ItemId itemID) const NN_NOEXCEPT
{
    if(itemID == SoundArchive::InvalidId)
    {
        return "";
    }

    NN_SDK_ASSERT_NOT_NULL(m_Owner);
    return !m_Owner->IsInitialized() ? "" : m_Owner->m_SoundArchiveManager->GetMainSoundArchive()->GetItemLabel(itemID);
}

bool
SoundArchiveParameterEditor::Hook::IsValidDataType(SoundArchive::SoundType soundType, ResDataType dataType) const NN_NOEXCEPT
{
    switch(dataType)
    {
    case ResDataType_StreamSound:
        return soundType == SoundArchive::SoundType_Stream;

    case ResDataType_WaveSound:
        return soundType == SoundArchive::SoundType_Wave;

    case ResDataType_SequenceSound:
        return soundType == SoundArchive::SoundType_Sequence;

    default:
        break;
    }

    return false;
}

SoundArchive::SoundType
SoundArchiveParameterEditor::Hook::ResDataTypeToSoundType(ResDataType dataType) const NN_NOEXCEPT
{
    switch(dataType)
    {
    case ResDataType_StreamSound:
        return SoundArchive::SoundType_Stream;

    case ResDataType_WaveSound:
        return SoundArchive::SoundType_Wave;

    case ResDataType_SequenceSound:
        return SoundArchive::SoundType_Sequence;

    default:
        break;
    }

    return SoundArchive::SoundType_Invalid;
}

} // namespace nn::atk::viewer::detail
} // namespace nn::atk::viewer
} // namespace nn::atk
} // namespace nn

#endif // NN_ATK_CONFIG_ENABLE_DEV
